/*
	HEHEADER.H

	contains definitions and prototypes
	for the Hugo Engine

	Copyright (c) 1995-1998 by Kent Tessman
*/


#define HEVERSION 2
#define HEREVISION 5
#define HEINTERIM ""


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <ctype.h>


/* The Hugo token set and their respective values are contained
   separately in htokens.h
*/
#include "htokens.h"

/* Define the compiler being used as one of:

	QUICKC
	AMIGA
	GCC_UNIX
    GCC_OS2
	ACORN

   (although this is typically done in the makefile)
*/


/*---------------------------------------------------------------------------
	MS-DOS specific definitions for Microsoft QuickC

	by Kent Tessman
---------------------------------------------------------------------------*/

#if defined (QUICKC)

#include <conio.h>

#define MATH_16BIT

#define PORT_NAME "16-bit"
#define PORTER_NAME "Kent Tessman"

#define HUGO_PTR        _huge           /* may be blank for most systems */
#define HUGO_FOPEN      fopen

#define MAXPATH         _MAX_PATH       /* maximum allowable lengths */
#define MAXFILENAME     _MAX_FNAME
#define MAXDRIVE        _MAX_DRIVE
#define MAXDIR          _MAX_DIR
#define MAXEXT          _MAX_EXT

/* Case-insensitive string compare */
#define STRICMP(a, b)           stricmp(a, b)

#define DEF_PRN         "LPT1"          /* default printer */

#define DEF_FCOLOR      7               /* default colors, fore and back */
#define DEF_BGCOLOR     0
#define DEF_SLFCOLOR	15		/* statusline */
#define DEF_SLBGCOLOR	1

#define MAXBUFFER 255                   /* max. input/output line length */

#endif /* defined (QUICKC) */


/*---------------------------------------------------------------------------
	Definitions for the djgpp/Allegro build(s)

	by Kent Tessman
---------------------------------------------------------------------------*/

#if defined (DJGPP)

#include <unistd.h>
#include <dir.h>

#define PORT_NAME "32-bit"
#define PORTER_NAME "Kent Tessman"

#define HUGO_PTR
#define HUGO_FOPEN      fopen

#define DEF_PRN "LPT1"

/* MAXPATH, MAXDRIVE, MAXDIR, MAXEXT defined in <dir.h> */
#define MAXFILENAME     MAXFILE

#define STRICMP(a, b)   strcasecmp(a, b)

#define DEF_FCOLOR      7
#define DEF_BGCOLOR     0
#define DEF_SLFCOLOR	15
#define DEF_SLBGCOLOR	1

#define MAXBUFFER 255

#if defined (ALLEGRO)
#define FRONT_END
#define GRAPHICS_SUPPORTED
#define SOUND_SUPPORTED
#undef PORT_NAME
#define PORT_NAME "32-bit Allegro"

#if !defined (PROGRAM_NAME) && !defined (DEBUGGER)
#define PROGRAM_NAME "hegr"
#elif defined (DEBUGGER)
#define PROGRAM_NAME "hdgr"
#endif
#endif	/* defined (ALLEGRO) */

#endif  /* defined (DJGPP) */


/*---------------------------------------------------------------------------
	Definitions for the LCC-Win32 build

	by Kent Tessman
---------------------------------------------------------------------------*/

#if defined (LCCWIN32)

#define PORT_NAME "Win32"
#define PORTER_NAME "Kent Tessman"

#define HUGO_PTR
#define HUGO_FOPEN      fopen

#define DEF_PRN ""

#define MAXPATH         FILENAME_MAX    /* Win32 allows long filenames */
#define MAXDRIVE        FILENAME_MAX
#define MAXDIR          FILENAME_MAX
#define MAXFILENAME     FILENAME_MAX
#define MAXEXT          FILENAME_MAX

#define STRICMP(a, b)   stricmp(a, b)

#define DEF_FCOLOR      7
#define DEF_BGCOLOR     1
#define DEF_SLFCOLOR	15
#define DEF_SLBGCOLOR	0

#define MAXBUFFER 255

#define FRONT_END
#define SCROLLBACK_DEFINED

void PrintFatalError(char *a);
#define PRINTFATALERROR(a) PrintFatalError(a);

#endif  /* defined (LCCWIN32) */


/*---------------------------------------------------------------------------
	Definitions for the Amiga port

	by David Kinder
---------------------------------------------------------------------------*/

#if defined (AMIGA)

#define PORT_NAME "Amiga"
#define PORTER_NAME "David Kinder"

#define HUGO_PTR
#define HUGO_FOPEN      fopen

#define MAXPATH         256
#define MAXFILENAME     256
#define MAXDRIVE        256
#define MAXDIR          256
#define MAXEXT          256

#define DEF_PRN         "PRT:"

#define DEF_FCOLOR      7
#define DEF_BGCOLOR     0
#define DEF_SLFCOLOR	15
#define DEF_SLBGCOLOR	1

#define MAXBUFFER 255

#endif  /* defined (AMIGA) */


/*---------------------------------------------------------------------------
	Definitions for the GCC Unix port

	by Bill Lash
---------------------------------------------------------------------------*/

#if defined (GCC_UNIX)

#include <unistd.h>

#define PORT_NAME "Unix"
#define PORTER_NAME "Bill Lash"

#define HUGO_PTR
#define HUGO_FOPEN      fopen

#define MAXPATH         256
#define MAXFILENAME     256
#define MAXDRIVE        256
#define MAXDIR          256
#define MAXEXT          256

#define DEF_PRN         "/dev/lp"       /* Bill suggests this is perhaps
					   a less-than-ideal choice for
					   the default printer */
#define DEF_FCOLOR      7
#define DEF_BGCOLOR     0
#define DEF_SLFCOLOR	15
#define DEF_SLBGCOLOR	1

#define MAXBUFFER 255

#endif /* defined (GCC_UNIX) */

/*---------------------------------------------------------------------------
	Definitions for the GCC OS/2 port

	by Bill Lash
---------------------------------------------------------------------------*/

#if defined (GCC_OS2)

#define PORT_NAME "OS/2"
#define PORTER_NAME "Gerald Bostock"

#define HUGO_PTR
#define HUGO_FOPEN      fopen

#define MAXPATH         256
#define MAXFILENAME     256
#define MAXDRIVE        256
#define MAXDIR          256
#define MAXEXT          256

#define DEF_PRN         "LPT1"       
#define DEF_FCOLOR      7
#define DEF_BGCOLOR     0
#define DEF_SLFCOLOR	15
#define DEF_SLBGCOLOR	1

#define MAXBUFFER 255

#endif /* defined (GCC_OS2) */
/*---------------------------------------------------------------------------
	Definitions for the Acorn Archimedes & RPC
	using Acorn C Compiler (v4)

	by Colin Turnbull
---------------------------------------------------------------------------*/

#if defined (ACORN)

#define PORT_NAME "Acorn"
#define PORTER_NAME "ct"

#define HUGO_PTR
#define HUGO_FOPEN      arc_fopen

#define MAXPATH         256
#define MAXFILENAME     64      /* We _could_ be using LongFilenames */
#define MAXDRIVE        32      /* arcfs#SCSIFS::harddisc4 ? */
#define MAXDIR          256
#define MAXEXT          32

#define isascii(c)      ((unsigned int)(c)<0x80)
#define toascii(c)      ((c)&0x7f)

#define DEF_PRN         ""

#define DEF_FCOLOR      7
#define DEF_BGCOLOR     0
#define DEF_SLFCOLOR	15
#define DEF_SLBGCOLOR	1

#define MAXBUFFER 255

/* Added these functions for the Acorn port: */
FILE *arc_fopen(const char *, const char *);
void arc_colour(unsigned char n, unsigned int c);

#endif /* defined (ACORN) */


/*-------------------------------------------------------------------------*/

#if !defined (PROGRAM_NAME) && !defined (DEBUGGER)
#define PROGRAM_NAME     "he"
#endif

#if !defined (STRICMP)
#define STRICMP(a, b)   stricmp(a, b)
#endif

#if !defined (PRINTFATALERROR)
#define PRINTFATALERROR(a)	fprintf(stderr, a)
#endif

/* address_scale refers to the factor by which addresses are multiplied to
   get the "real" address.  In this way, a 16-bit integer can reference
   64K * 4 = 256K of memory.
*/
#define address_scale 4

/* These static values are not changeable--they depend largely on internals
   of the Engine.
*/
#define MAXATTRIBUTES    128
#define MAXGLOBALS       240
#define MAXLOCALS         16
#define MAXPOBJECTS      256    /* contenders for disambiguation */
#define MAXWORDS          32    /* in an input line              */
#define MAXUNDO          256	/* number of undoable operations */
#define MAXSTACKDEPTH    256	/* for nesting {...}		 */

enum ERROR_TYPE                 /* fatal errors */
{
	MEMORY_E = 1,           /* out of memory                */
	OPEN_E,                 /* error opening file           */
	READ_E,                 /* error reading from file      */
	WRITE_E,                /* error writing to file        */
	EXPECT_VAL_E,           /* expecting value              */
	UNKNOWN_OP_E,           /* unknown operation            */
	ILLEGAL_OP_E,           /* illegal operation            */
	OVERFLOW_E              /* overflow                     */
};

/* The positions of various data in the header: */
#define H_GAMEVERSION	0x00
#define H_ID		0x01
#define H_SERIAL	0x03
#define H_CODESTART	0x0B

#define H_OBJTABLE	0x0D           /* data tables */
#define H_PROPTABLE	0x0F
#define H_EVENTTABLE	0x11
#define H_ARRAYTABLE	0x13
#define H_DICTTABLE	0x15
#define H_SYNTABLE	0x17

#define H_INIT		0x19           /* junction routines */
#define H_MAIN		0x1B
#define H_PARSE		0x1D
#define H_PARSEERROR	0x1F
#define H_FINDOBJECT	0x21
#define H_ENDGAME	0x23
#define H_SPEAKTO	0x25
#define H_PERFORM	0x27

#define H_TEXTBANK	0x29


/* Printing control codes--embedded in strings printed by AP(). */
#define FONT_CHANGE       1
#define COLOR_CHANGE      2
#define NO_CONTROLCHAR    3
#define FORCED_SPACE      31	/* Can't be <= # colors/font codes + 1
				   (See AP() for the reason) */				
#define NO_NEWLINE        255

/* Font control codes--these bitmasks follow FONT_CHANGE codes. */
#define NORMAL_FONT	  0
#define BOLD_FONT         1
#define ITALIC_FONT       2
#define UNDERLINE_FONT    4
#define PROP_FONT         8

/* CHAR_TRANSLATION is simply a value that is added to an ASCII character
   in order to encode the text, i.e., make it unreadable to casual
   browsing.
*/
#define CHAR_TRANSLATION  0x14

/* Passed to GetWord() */
#define PARSE_STRING_VAL  0xFFF0
#define SERIAL_STRING_VAL 0xFFF1

/* Returned by FindWord() */
#define UNKNOWN_WORD      0xFFFF

/* Bitmasks for certain qualities of properties */
#define ADDITIVE_FLAG   1
#define COMPLEX_FLAG    2

/* Property-table indicators */
#define PROP_END          255
#define PROP_ROUTINE      255

/* Reading C source code can be bad for your eyesight. */
#define and     &&
#define or      ||
#define not     !

/* To be used with caution; obviously, not all non-zero values are "true"
   in this usage.
*/
#define true 1
#define false 0


/*-------------------------------------------------------------------------*/


/* FUNCTION PROTOTYPES: */


/* First the porting-required functions (HEQUICKC.C): */

void HUGO_PTR *hugo_blockalloc(long num);
void hugo_blockfree(void HUGO_PTR *block);
int hugo_hasgraphics(void);
int hugo_charwidth(char a);
void hugo_cleanup_screen(void);
void hugo_clearfullscreen(void);
void hugo_clearwindow(void);
void hugo_closefiles(void);
void hugo_font(int f);
void hugo_getfilename(char *a, char *b);
#if !defined (GCC_OS2)
int hugo_getkey(void);
#else
int hugo_getkey(int);
#endif
void hugo_getline(char *p);
void hugo_init_screen(void);
void hugo_makepath(char *path, char *drive, char *dir, char *fname, char *ext);
int hugo_overwrite(char *f);
void hugo_print(char *a);
void hugo_scrollwindowup(void);
void hugo_setbackcolor(int c);
void hugo_settextcolor(int c);
void hugo_settextmode(void);
void hugo_settextpos(int x, int y);
void hugo_settextwindow(int left, int top, int right, int bottom);
char hugo_specialchar(char *a, int *i);
void hugo_splitpath(char *path, char *drive, char *dir, char *fname, char *ext);
int hugo_strlen(char *a);
int hugo_textwidth(char *a);
int hugo_waitforkey(void);
int hugo_iskeywaiting(void);
int hugo_timewait(int n);
#if defined (SCROLLBACK_DEFINED)
void hugo_sendtoscrollback(char *a);
#endif


/* Then the portable functions: */

/* HEEXPR.C */
int EvalExpr(int ptr);
int GetValue(void);
int Increment(int v, char inctype);
char IsIncrement(long addr);
void SetupExpr(void);

/* HEMISC.C */
void AP(char *a);
int CallRoutine(unsigned int addr);
unsigned int Dict(void);
void FatalError(int e);
void FileIO(void);
void Flushpbuffer(void);
void GetCommand(void);
char *GetString(long addr);
char *GetText(long textaddr);
char *GetWord(unsigned int a);
void InitGame(void);
void LoadGame(void);
void ParseCommandLine(int argc, char *argv[]);
void PassLocals(int n);
unsigned char Peek(long a);
unsigned int PeekWord(long a);
void Poke(unsigned int a, unsigned char v);
void PokeWord(unsigned int a, unsigned int v);
char *PrintHex(long a);
void Printout(char *a);
/* void PrintSetting(int t, int a, int b, int c, int d); */
void PromptMore(void);
int RecordCommands(void);
void SaveUndo(int t, int a, int b, int c, int d);
void SetStackFrame(int depth, int type, long brk, long returnaddr);
void SetupDisplay(void);
FILE *TrytoOpen(char *f, char *p, char *d);
int Undo(void);

/* HEOBJECT.C */
int Child(int obj);
int Children(int obj);
int Elder(int obj);
unsigned long GetAttributes(int obj, int attribute_set);
int GetProp(int obj, int p, int n, char s);
int GrandParent(int obj);
void MoveObj(int obj, int p);
char *Name(int obj);
int Parent(int obj);
unsigned int PropAddr(int obj, int p, unsigned int offset);
void PutAttributes(int obj, unsigned long a, int attribute_set);
void SetAttribute(int obj, int attr, int c);
int Sibling(int obj);
int TestAttribute(int obj, int attr, int nattr);
int Youngest(int obj);

/* HEPARSE.C */
int Available(int obj);
void FindObjProp(int obj);
unsigned int FindWord(char *a);
void KillWord(int a);
int MatchCommand(void);
int ObjWord(int obj, unsigned int w);
int Parse(void);
void ParseError(int e, int a);
void RemoveWord(int a);
void SeparateWords(void);
int ValidObj(int obj);

/* HERES.C */
void DisplayPicture(void);
void PlayMusic(void);
void PlaySample(void);

/* HERUN.C */
void RunDo(void);
void RunEvents(void);
void RunGame(void);
void RunIf(char override);
void RunInput(void);
void RunMove(void);
void RunPrint(void);
int RunRestart(void);
int RunRestore(void);
void RunRoutine(long addr);
int RunSave(void);
int RunScriptSet(void);
void RunSet(int gotvalue);
int RunString(void);
int RunSystem(void);
void RunWindow(void);

/* STRINGFN.C: */
char *Left(char *a, int l);
char *Ltrim(char *a);
char *Mid(char *a, int pos, int n);
char *Right(char *a, int l);
char *Rtrim(char *a);
#if !defined (QUICKC)
char *itoa(int a, char *buf, int base);
#if !defined (GCC_OS2)
char *strlwr(char *s);
char *strnset(char *s, int c, int l);
char *strupr(char *s);
#endif
#endif

/* A structure used for disambiguation in MatchObject(): */
struct pobject_structure
{
	int obj;                /* the actual object number */
	char type;              /* referred to by noun or adjective */
};

/* Structure used for navigating {...} blocks: */
struct CODE_BLOCK
{
	int type;     /* see #defines, below */
	long brk;		/* break address, or 0 to indicate NOP */
	long returnaddr;	/* used only for do-while loops */
#if defined (DEBUGGER)
	int dbnest;		/* for recovering from 'break' */
#endif
};

#define RESET_STACK_DEPTH (-1)

#define RUNROUTINE_BLOCK  1
#define CONDITIONAL_BLOCK 2
#define DOWHILE_BLOCK     3

/*-------------------------------------------------------------------------*/

#if defined (INIT_PASS)

/*      INIT_PASS is defined for the first inclusion of HEHEADER.H
	so that global variables may defined; the following section
	reproduces the same globals without definitions, only
	declarations.
*/

/* INITIALIZED GLOBAL VARIABLES: */

/* game_version is needed for backward compatibility.  It holds the
   version number of the game*10 + the revision number.  The object
   size of pre-v2.2 games was only 12 bytes.
*/
int game_version;
int object_size = 24;

/* General variables */
unsigned char HUGO_PTR *mem;            /* the memory buffer               */
unsigned int defseg;                    /* holds segment indicator         */
unsigned int gameseg;                   /* code segment                    */
long codeptr;                           /* code pointer                    */
long codeend;                           /* end of loaded code              */
long z = 0;                             /* universal send/return           */

struct CODE_BLOCK code_block[MAXSTACKDEPTH];
int stack_depth;

char buffer[MAXBUFFER+MAXWORDS*2];      /* input buffer                    */
char errbuf[MAXBUFFER+1];               /* last invalid input              */
char line[1025];                        /* line buffer                     */

char pbuffer[MAXBUFFER*2+1];            /* print buffer for line-wrapping  */
int currentpos = 0;                     /* column position (pixel or char) */
int currentline = 0;                    /* row number (line)               */
int full = 0;                           /* page counter for PromptMore     */
char fcolor = 16,			/* default fore/background colors  */
	bgcolor = 17,			/* (16 = default foreground,	   */
	icolor = 16;			/*  17 = default background)	   */
char default_bgcolor = 17;		/* default for screen background   */
int currentfont = NORMAL_FONT;		/* current font bitmasks           */
char capital = 0;                       /* if next letter is to be capital */
unsigned int textto = 0;                /* for printing to an array        */

int display_object = -1;                /* i.e., non-existent (yet)        */

int SCREENWIDTH, SCREENHEIGHT;		/* screen dimensions               */
					/*   (in pixels or characters)     */

int physical_windowwidth,		/* "physical_..." measurements	   */
	physical_windowheight,		/*   are in pixels (or characters) */
	physical_windowtop, physical_windowleft,
	physical_windowbottom, physical_windowright;
int inwindow = 0;
int charwidth, lineheight, FIXEDCHARWIDTH, FIXEDLINEHEIGHT;
int current_text_x, current_text_y;

int words = 0;                          /* parsed word count               */
char *word[MAXWORDS+1];                 /* breakdown into words            */
unsigned int wd[MAXWORDS+1];            /*     "      "   dict. entries    */
unsigned int parsed_number;		/* needed for numbers in input	   */

int var[MAXLOCALS+MAXGLOBALS];		/* variables                       */
int passlocal[MAXLOCALS];		/* locals passed to routine        */
int arguments_passed;                   /* when calling routine            */

int ret = 0; char retflag = 0;		/* return value and returning flag */

int incdec;                             /* value is being incremented/dec. */

char arrexpr = 0;                       /* true when assigning array       */
char multiprop = 0;                     /* true in multiple prop. assign.  */
char getaddress = 0;                    /* true when finding &routine      */
char inexpr = 0;                        /* true when in expression         */
char inobj = 0;                         /* true when in object compound    */

char oops[MAXBUFFER+1];                 /* illegal word                    */
int oopscount = 0;                      /* # of corrections in a row       */

/* See SaveUndo() and Undo() in hemisc.c for details on how the undo
   stack is built and dealt with.
*/
int undostack[MAXUNDO][5];              /* for saving undo information     */
int undoptr;                            /* number of operations undoable   */
int undoturn = 0;                       /* number of operations this turn  */
char undoinvalid = 0;                   /* for start of game, and restarts */
char undorecord = 0;                    /* true when recording             */

/* Grammar matching and parsing */
signed char remaining = 0;              /* multiple commands in input      */
char parseerr[MAXBUFFER+1];             /* for passing to RunPrint         */
char parsestr[MAXBUFFER+1];             /* for passing quoted string       */
char xverb;                             /* flag; 0 = regular verb          */
unsigned int grammaraddr;               /* address in grammar              */
int domain, odomain;                    /* of object(s)                    */
int objlist[32];                        /* for objects of verb             */
char objcount;                          /* of objlist                      */
struct pobject_structure
	pobjlist[MAXPOBJECTS];          /* for possible objects            */
int pobjcount;                          /* of pobjlist                     */
int pobj;                               /* last remaining suspect          */
int obj_match_state;			/* see MatchCommand() for details  */
char object_is_number;			/* number used in player command   */
unsigned int objgrammar;                /* for 2nd pass                    */
char objstart;                          /*  "   "   "                      */
char objfinish;                         /*  "   "   "                      */
char addflag;                           /* true if adding to objlist[]     */
int speaking;                           /* if command is addressed to obj. */

/* Library/engine globals */
const object = 0;
const xobject = 1;
const self = 2;
const wordcount = 3;
const player = 4;
const actor = 5;
const location = 6;
const verbroutine = 7;
const endflag = 8;
const prompt = 9;
const objectcount = 10;
const system_status = 11;

/* Library/engine properties */
const before = 1;
const after = 2;
const noun = 3;
const adjective = 4;
const article = 5;

/* "display" object properties */
const screenwidth = 1;
const screenheight = 2;
const linelength = 3;
const windowlines = 4;
const cursor_column = 5;
const cursor_row = 6;
const hasgraphics = 7;
const title_caption = 8;

/* Totals */
int objects;
int events;
int dictcount;                  /* dictionary entries */
int syncount;                   /* synonyms, etc.     */

/* File pointers, etc. */
FILE *game; char gamefile[MAXPATH];
FILE *script = 0; char scriptfile[MAXPATH];
FILE *save; char savefile[MAXPATH];
FILE *io = 0; char ioblock = 0; char ioerror = 0;
FILE *record = 0; char recordfile[MAXPATH];
FILE *playback = 0;

/* Header information */
char id[3];
char serial[9];
unsigned int codestart;                 /* start of executable code	*/
unsigned int objtable;                  /* object table			*/
unsigned int eventtable;                /* event table			*/
unsigned int proptable;                 /* property table		*/
unsigned int arraytable;                /* array data table		*/
unsigned int dicttable;                 /* dictionary			*/
unsigned int syntable;                  /* synonyms			*/
unsigned int initaddr;                  /* "init" routine		*/
unsigned int mainaddr;                  /* "main"			*/
unsigned int parseaddr;                 /* "parse"			*/
unsigned int parseerroraddr;            /* "parseerror"			*/
unsigned int findobjectaddr;            /* "findobject"			*/
unsigned int endgameaddr;               /* "endgame"			*/
unsigned int speaktoaddr;               /* "speakto"			*/
unsigned int performaddr;		/* "perform"			*/


/*-------------------------------------------------------------------------*/


#else   /* if !defined (INIT_PASS) */

extern int game_version;
extern int object_size;

extern char *token[];

extern unsigned char HUGO_PTR *mem;
extern unsigned int defseg;
extern unsigned int gameseg;
extern long codeptr;
extern long codeend;
extern long z;

extern struct CODE_BLOCK code_block[];
extern int stack_depth;

extern char buffer[];
extern char errbuf[];
extern char line[];

extern char pbuffer[];
extern int currentpos;
extern int currentline;
extern int full;
extern char fcolor, bgcolor, icolor, default_bgcolor;
extern int currentfont;
extern char capital;
extern unsigned int textto;

extern int display_object;

extern int SCREENWIDTH, SCREENHEIGHT;
extern int physical_windowwidth, physical_windowheight,
	physical_windowtop, physical_windowleft,
	physical_windowbottom, physical_windowright;
extern int inwindow;
extern int charwidth, lineheight, FIXEDCHARWIDTH, FIXEDLINEHEIGHT;
extern int current_text_x, current_text_y;

extern int words; extern char *word[];
extern unsigned int wd[], parsed_number;

extern int var[];
extern int passlocal[];
extern int arguments_passed;
extern int ret; extern char retflag;
extern int incdec;

extern char arrexpr;
extern char multiprop;
extern char getaddress;
extern char inexpr;
extern char inobj;

extern char oops[];
extern int oopscount;

extern int undostack[][5];
extern int undoptr;
extern int undoturn;
extern char undoinvalid;
extern char undorecord;

extern signed char remaining;
extern char parseerr[];
extern char parsestr[];
extern char xverb;
extern unsigned int grammaraddr;
extern int domain, odomain;
extern int objlist[];
extern char objcount;
extern struct pobject_structure pobjlist[];
extern int pobjcount;
extern int pobj;
extern int obj_match_state;
extern char object_is_number;
extern unsigned int objgrammar;
extern char objstart;
extern char objfinish;
extern char addflag;
extern int speaking;

extern const object;
extern const xobject;
extern const self;
extern const wordcount;
extern const player;
extern const actor;
extern const location;
extern const verbroutine;
extern const endflag;
extern const prompt;
extern const objectcount;
extern const system_status;

extern const screenwidth;
extern const screenheight;
extern const linelength;
extern const windowlines;
extern const cursor_column;
extern const cursor_row;
extern const hasgraphics;
extern const title_caption;

extern const before;
extern const after;
extern const noun;
extern const adjective;
extern const article;

extern int objects;
extern int events;
extern int dictcount;
extern int syncount;

extern FILE *game; extern char gamefile[];
extern FILE *script; extern char scriptfile[];
extern FILE *save; extern char savefile[];
extern FILE *io; extern char ioblock; extern char ioerror;
extern FILE *record; extern char recordfile[];
extern FILE *playback;

extern char id[];
extern char serial[];
extern unsigned int codestart;
extern unsigned int objtable;
extern unsigned int eventtable;
extern unsigned int proptable;
extern unsigned int arraytable;
extern unsigned int dicttable;
extern unsigned int syntable;
extern unsigned int initaddr;
extern unsigned int mainaddr;
extern unsigned int parseaddr;
extern unsigned int parseerroraddr;
extern unsigned int findobjectaddr;
extern unsigned int endgameaddr;
extern unsigned int speaktoaddr;
extern unsigned int performaddr;

#endif  /* defined (INIT_PASS) */


/* DEBUGGER should be defined (at the start of this file or in the makefile)
   to build the debugging version of the engine
*/
#if defined (DEBUGGER)
#include "hdinter.h"
#endif



