/*
	HEJPEG.C

	Image file (JPEG format) display routines, including the main
	function called from the engine:

		hugo_displaypicture

	Plus the non-portable djgpp/Allegro functions:

		hugo_initpalette
		hugo_initJPEGimage
		hugo_writescanline

	And adapted from the IJG library:

		read_JPEG_file

	for the Hugo Engine

	Copyright (c) 1995-2006 by Kent Tessman

	The routines in this file are based in part on the work of the
	Independent JPEG Group.  They are taken from release 6a (7-Feb-96).

	Basically, hugo_displaypicture(file) attempts to read and
	display the JPEG-compressed picture stored in <file>.  The file
	is passed hot, i.e., opened and positioned.  The image should fit
	the screen area described by the current text window.  (This
	calculation is performed by hugo_initJPEGimage().)

	This file assumes it will find a compiled IJG library and the
	necessary headers "jpeglib.h", "jconfig.h", and "jmorecfg.h".

	For what it's worth, if you're using not using some faster,
	OS-integrated JPEG-display mechanism, this should be easily
	portable to any operating system mainly by replacing
	hugo_initpalette() and hugo_writescanline().
*/

#include "heheader.h"

#if defined (ALLEGRO)
#define line line_unused
#include "allegro.h"
#undef line
#endif

int hugo_displaypicture(FILE *infile, long len);


#include <setjmp.h>

#include "jpeglib.h"


/* More function prototypes: */
int hugo_initpalette(void);
void hugo_initJPEGimage(j_decompress_ptr cinfo);
void hugo_writescanline(j_decompress_ptr cinfo, JSAMPLE *jbuffer);
int read_JPEG_file(FILE *infile);

/* from healleg.c */
void UpdateScreen(void);


int SCREENCOLORS;
int display_type = 0, color_space, mapped_color_space;
int reduce_quality;

extern int color_depth;			/* from healleg.c */
extern BITMAP *screen_buffer;
extern char screen_needs_repainting;
extern int display_width, display_height;

/* Display types */
extern const int SVGA_DT, VGA_DT, NONE_DT;  /* from healleg.c */

/* Since an int under djgpp is 32 bits, it can hold a reference to
   a color even in 32-bit truecolor mode
*/
int HUGO_PALETTE[16];                   /* for Hugo text colors */


#if defined (GRAPHICS_SUPPORTED) && !defined (IOTEST)

/* hugo_displaypicture

	(This is the only routine that gets called from the engine proper.)

	Assumes that "filename" is hot--i.e., opened and positioned--but
	closes it before returning.

	Loads and displays a JPEG picture, sizing it to the screen area
	described by the current text window.  hugo_displaypicture() could
	call any OS-specific loading mechanism to accomplish this; here,
	it depends on auxiliary functions such as hugo_initJPEGimage()
	to properly size and position the image.  If smaller than the
	current text window, the image should be centered, not stretched.

	Ultimately, graphic formats other than JPEG may be supported,
	hence the identification of the type flag and the switch
	at the end.
	
	Returns false if it fails because of an ERROR.
*/

int hugo_displaypicture(FILE *infile, long len)
{
	int type;

	if (display_type==NONE_DT)
	{
		fclose(infile);
		return true;		/* not an error */
	}

	/* Loading/display can fail at any of the following: */

	if ((type = fgetc(infile))==EOF) goto Failed;

	if (fseek(infile, -1, SEEK_CUR)) goto Failed;

	if (screen_needs_repainting)	/* also in hugo_scrollwindowup() */
	{
		UpdateScreen();
		screen_needs_repainting = false;
	}

	switch (type)
	{
		case 0xFF:      /* JPEG */
			if (!read_JPEG_file(infile)) return 0;
			break;

		default:        /* unrecognized */
#if defined (DEBUGGER)
			SwitchtoDebugger();
			DebugMessageBox("Picture Loading Error", "Unrecognized graphic format");
			SwitchtoGame();
#endif
			goto Failed;
	}

	return 1;

	/* read_JPEG_file() closes the file regardless of success or
	   failure; otherwise, it must be closed here
	*/
Failed:
	fclose(infile);
	return 0;
}

#endif	/* defined (GRAPHICS_SUPPORTED) && !defined (IOTEST) */


/* hugo_initpalette

	Does necessary initializations for number of colors and whether
	or not we're forcing to grayscale.  Basically, we have to
	generate a palette of 256 (actually 216 as far as the IJG library
	is concerned) possible colors.

	NEW MATERIAL:  The new version of Allegro supports 15/16/24/32-bit
	truecolor modes as well as 8-bit paletted color.  Truecolor modes
	are not paletted, so HUGO_PALETTE, instead of containing palette
	references, contains color values from makecol().
*/

int hugo_initpalette(void)
{
	int i, inc, red, green, blue;
	int rgb_brightness;
	PALETTE pal;


/* Truecolor (non-palette) modes: */

	if (bitmap_color_depth(screen)!=8)
	{
		generate_332_palette(pal);	/* just in case */
		set_palette(pal);

		inc = 127;
		rgb_brightness = 48;
		for (i=0; i<16; i++)
		{
			if (i>=8) rgb_brightness = 0, inc = 255;

			/* Match appropriate color values to Hugo's
			   16-bit palette
			*/
			HUGO_PALETTE[i] = makecol(((i&4)?inc:0)+rgb_brightness,
						  ((i&2)?inc:0)+rgb_brightness,
						  ((i&1)?inc:0)+rgb_brightness);
		}

		/* Color corrections (see below): */
		HUGO_PALETTE[0] = makecol(0, 0, 0);
		HUGO_PALETTE[6] = makecol(143, 110, 79);
		HUGO_PALETTE[7] = makecol(192, 192, 192);
		return true;
	}


/* Everything from here down is for palette modes only: */

	for (i=0; i<16; i++)
	{
		/* Re-establish the basic Hugo color set */
		HUGO_PALETTE[i] = i;
	}

	SCREENCOLORS = 256;

	/* First, make sure all palette numbers are valid */
	for (i=0; i<SCREENCOLORS; i++)
		pal[i].r = 0, pal[i].g = 0, pal[i].b = 0;

	/* Color... */
	if (color_space==JCS_RGB)
	{
	  inc = 12;

	  i = 0;
	  for (red   = 0;   red < 64;   red += inc)
	  for (green = 0; green < 64; green += inc)
	  for (blue  = 0;  blue < 64;  blue += inc)
	  {
		pal[i].r = red;
		pal[i].g = green;
		pal[i].b = blue;

		i++;
	  }

	  /* SCREENCOLORS will have changed to 216, which has a cube-
	     root and can therefore represent an RGB array
	  */
	  SCREENCOLORS = i;
	}

	/* ...or grayscale */
	else
	{
		for (i=0; i<SCREENCOLORS; i++)
		{
			pal[i].r = i/4;
			pal[i].g = i/4;
			pal[i].b = i/4;
		}
	}

	/* If we've changed from the original Hugo palette mapping,
	   remap HUGO_PALETTE[0..16]
	*/
	if (SCREENCOLORS>=216)
	{
		if (color_space==JCS_RGB)
		{
			i = 216;
			inc = 32;
			rgb_brightness = 12;
			for (red   = 0;   red < 64;   red += inc)
			for (green = 0; green < 64; green += inc)
			for (blue  = 0;  blue < 64;  blue += inc)
			{
				pal[i].r = red   + rgb_brightness;
				pal[i].g = green + rgb_brightness;
				pal[i].b = blue  + rgb_brightness;

				pal[i+8].r = pal[i].r|pal[i].r>>1;
				pal[i+8].g = pal[i].g|pal[i].g>>1;
				pal[i+8].b = pal[i].b|pal[i].b>>1;

				i++;
			}
			for (i=0; i<16; i++) HUGO_PALETTE[i] = 216+i;
		}

		else    /* grayscale */
		{
			for (i=0; i<16; i++) HUGO_PALETTE[i] = i*16;
		}
	}

	if (color_space==JCS_RGB)
	{
		/* Make black as black as possible */
		pal[216].r = pal[216].g = pal[216].b = 0;

		/* Make brown "brown" instead of "mustard"... */
		pal[216+6].r = 36;
		pal[216+6].g = 28;
		pal[216+6].b = 20;

		/* ...and make white a little whiter */
		pal[216+7].r = pal[216+7].g = pal[216+7].b = 46;
	}

	set_palette(pal);

	hugo_settextcolor(DEF_FCOLOR);
	hugo_setbackcolor(DEF_BGCOLOR);

	mapped_color_space = color_space;

	return 1;
}


#if defined (GRAPHICS_SUPPORTED) && !defined (IOTEST)

/* hugo_initJPEGimage

	Fits the JPEG image to the current text window, either scaling
	it down or centering it as necessary.
*/

float cscaling, rscaling;       /* column and row scaling */
int col_offset;                 /* column offset, if any  */
int row;                        /* last-drawn row         */
int row_stride;                 /* number bytes/row       */

void hugo_initJPEGimage(j_decompress_ptr cinfo)
{
	char wider = 0, taller = 0;
	float maxheight, maxwidth;              /* of screen */
	float height, width;                    /* of image */

	/* Default is no horizontal or vertical scaling */
	cscaling = rscaling = 1;
	row = 0;

	/* First of all, figure what the maximum allowable size of the image
	   is (i.e., the current text window), and coerce the drawing width
	   and height to fit it
	*/
	maxheight = physical_windowheight;
	maxwidth = physical_windowwidth;

	height = (int)(cinfo->output_height);
	if (height>maxheight)
	{
		rscaling = maxheight/height;
		height = maxheight;
	}

	width = (int)(cinfo->output_width);
	if (width>maxwidth)
	{
		cscaling = maxwidth/width;
		width = maxwidth;
	}

	if (cscaling > rscaling) cscaling = rscaling, taller = 1;
	if (rscaling > cscaling) rscaling = cscaling, wider = 1;


	/* Now, figure out both a screen row and a column offset--if
	   the image is smaller than the currently defined window,
	   we want to center it
	*/
	row = physical_windowtop;
	if (!taller)
		 row += (int)(maxheight/2) - (int)(cinfo->output_height*rscaling/2);
	col_offset = physical_windowleft;
	if (!wider)
		col_offset += (int)(maxwidth/2) - (int)(cinfo->output_width*cscaling/2);

	row++;
	row/=rscaling;

	/* cinfo->output_components is 3 if RGB, 1 if quantizing to a
	   color map or grayscale
	*/
	row_stride = cinfo->output_width * cinfo->output_components;
}


/* hugo_writescanline */

void hugo_writescanline(j_decompress_ptr cinfo, JSAMPLE *jbuffer)
{
	int j, x = 0;
	int color;
	static int last_row;
#ifdef COLOR_BLURRING
	int lastx = -1;
	int lastr = 0, lastg = 0, lastb = 0;
#endif

	/* Skim through the scanline, getting color indexes and firing
	   them out pixel-by-pixel--but only bother if we're not
	   simply drawing over a previous row (i.e., when the image is
	   larger than the display area).  However, if overplotting,
	   we have to draw every row.
	*/
	if ((int)(row*rscaling)!=(int)(last_row*rscaling))
	{
		/* If grayscale or paletted */
		if (cinfo->output_components==1)
		{
			/* Since we're quantizing to a color index (instead of
			   three RGB components), row_stride is equal to the
			   image width, and output_components is 1
			*/
			for (j=0; j<row_stride; j++)
			{
				color = *(jbuffer + j);
#ifdef COLOR_BLURRING
				/* Color-blurring code, intended to repair the effects
				   over over-reducing dithered pixels
				*/
				if (rscaling<0.67 or cscaling<0.67)
				{
				  if (lastx==(int)(x*cscaling))
				  {
				    color = makecol((lastr + getr(color)) / 2,
					    (lastg + getg(color)) / 2,
					    (lastb + getb(color)) / 2);
				  }
				}
				lastx = (int)(x*cscaling);
				lastr = getr(color);
				lastg = getg(color);
				lastb = getb(color);
#endif
                                /* Draw the pixel on both the screen and
                                   the screen_buffer in order to get the
                                   visible decompression effect
                                */
				putpixel(screen_buffer, col_offset+(int)(x*cscaling),
					(int)(row*rscaling), color);
                                putpixel(screen, col_offset+(int)(x++*cscaling),
					(int)(row*rscaling), color);
			}
		}
		else
		{
			for (j=0; j<row_stride; j+=cinfo->output_components)
			{
				color = makecol(*(jbuffer + j),
						*(jbuffer + j + 1),
						*(jbuffer + j + 2));

				putpixel(screen_buffer, col_offset+(int)(x*cscaling),
					(int)(row*rscaling), color);
                                putpixel(screen, col_offset+(int)(x++*cscaling),
					(int)(row*rscaling), color);
			}
		}
	}

	last_row = row++;
}


/*---------------------------------------------------------------------------
   JPEG Decompression Interface (from the IJG library):

   For further elaboration on the how and why of read_JPEG_file(), see
   "libjpeg.doc" with the IJG distribution.  For the sake of brevity
   and relevance to the Hugo implementation, some of the commenting has
   been trimmed or modified.
----------------------------------------------------------------------------*/

/* First of all, since a decompression error (memory or who knows what)
   may occur deep in the bowels of the IJG library, we need a way to
   harmlessly recover.  Hugo's default behavior is simply to abort loading
   the JPEG without reporting an error.

   The IJG library allows the user to build a hook into its normal
   error handler using setjmp() and longjmp(), which will instantly
   switch control to the blissfully ignorant return in read_JPEG_file(),
   below.
*/

struct hejpeg_error_mgr
{
	struct jpeg_error_mgr pub;      /* "public" fields */
	jmp_buf setjmp_buffer;          /* for return to caller */
};

typedef struct hejpeg_error_mgr *hejpeg_error_ptr;

void hejpeg_error_exit(j_common_ptr cinfo)
{
	/* cinfo->err really points to a hejpeg_error_mgr struct, so
	   coerce pointer
	*/
	hejpeg_error_ptr myerr = (hejpeg_error_ptr) cinfo->err;

	/* This is where the IJG library displays the error message--Hugo
	   doesn't, since it returns (even in mid-failed-decompression) to
	   the calling function

		(*cinfo->err->output_message) (cinfo);
	*/

	/* Return control to the setjmp point */
	longjmp(myerr->setjmp_buffer, 1);
}


/* read_JPEG_file

	Assumes the file passed as infile has been opened successfully.
*/


int read_JPEG_file(FILE *infile)
{
	/* This struct contains the JPEG decompression parameters and
	 * pointers to working space (which is allocated as needed by
	 * the JPEG library).
	 */
	struct jpeg_decompress_struct cinfo;

	/* We use our private extension JPEG error handler.
	 * Note that this struct must live as long as the main JPEG parameter
	 * struct, to avoid dangling-pointer problems.
	 */
	struct hejpeg_error_mgr jerr;

	JSAMPARRAY jbuffer;     /* output row buffer */
	int row_stride;         /* physical row width in output buffer */
#ifdef ADJUST_IMAGE_SIZE
	int height, width;
#endif


/* Step 1: Allocate and initialize JPEG decompression object */

	/* We set up the normal JPEG error routines, then override
	   error_exit.
	*/
	cinfo.err = jpeg_std_error(&jerr.pub);
	jerr.pub.error_exit = hejpeg_error_exit;

	/* Establish the setjmp return context for hejpeg_error_exit
	   to use.
	*/
	if (setjmp(jerr.setjmp_buffer))
	{
	    /* If we get here, the JPEG code has signaled an error.
	     * We need to clean up the JPEG object, close the input file,
	     * and return.
	     */
	    jpeg_destroy_decompress(&cinfo);
	    fclose(infile);
	    return 0;
	}

	/* Now we can initialize the JPEG decompression object. */
	jpeg_create_decompress(&cinfo);

/* Step 2: Specify the data source */

	/* Return values from most library initializations are ignored
	   since suspension is not possible from the stdio data source
	*/
	jpeg_stdio_src(&cinfo, infile);

/* Step 3: Read file parameters with jpeg_read_header() */

	jpeg_read_header(&cinfo, TRUE);

/* Step 4: set parameters for decompression */

	if (color_depth > 8)
	{
		/* Don't quantize if in a truecolor mode */
		cinfo.quantize_colors = FALSE;
	}
	else
	{
		/* Quantize down from 24-bit color to whatever we
	           can muster
		*/
		cinfo.quantize_colors = TRUE;
		if (cinfo.num_components > 1)
			cinfo.out_color_space = mapped_color_space;
	}

	/* Let the library build a suitable color map instead of supplying
	   it with one
	*/
	cinfo.colormap = NULL;
	cinfo.two_pass_quantize = FALSE;

	/* Choose Floyd-Steinberg dithering and give the library a color
	   count to quantize to
	*/
	cinfo.dither_mode = JDITHER_FS;
	cinfo.desired_number_of_colors = SCREENCOLORS;

	/* JDCT_DEFAULT (JDCT_ISLOW) is slower/higher quality; JDCT_FASTEST
	   is JDCT_IFAST by default
	*/
	if (!reduce_quality)
		cinfo.dct_method = JDCT_ISLOW;
	else
		cinfo.dct_method = JDCT_FASTEST;

	/* Little quality cost to forgo fancy upsampling */
	if (!reduce_quality)
		cinfo.do_fancy_upsampling = TRUE;
	else
		cinfo.do_fancy_upsampling = FALSE;

	/* Now, because if we try to scale down too much we're going to
	   badly mess up the dithering, so pre-shrink the image if
	   necessary--which it isn't unless we're in 8-bit color paletted
	   mode.
	*/
#ifdef ADJUST_IMAGE_SIZE
	width = cinfo.image_width;
	height = cinfo.image_height;
	if ((height>=physical_windowheight or width>=physical_windowwidth)
		and color_depth==8 and color_space==JCS_RGB)
	{
		/* No need to shrink if the dithering wouldn't be too badly
		   corrupted.  Arbitrarily, 75% is acceptable.
		*/
		if (height/2 > physical_windowheight*3/4 or
			width/2 > physical_windowwidth*3/4)
		{
			cinfo.scale_denom = 2;
		}
	}
#endif

/* Step 5: Start decompressor */

	jpeg_start_decompress(&cinfo);

	hugo_initJPEGimage(&cinfo);

	/* JSAMPLEs per row in output buffer */
	row_stride = cinfo.output_width * cinfo.output_components;

	/* Make a one-row-high sample array that will go away when
	   done with the image
	*/
	jbuffer = (*cinfo.mem->alloc_sarray)
		((j_common_ptr)&cinfo, JPOOL_IMAGE, row_stride, 1);

/* Step 6: while (scan lines remain to be read) */

	/* Here we use the library's state variable cinfo.output_scanline
	 * as the loop counter, so that we don't have to keep track
	 * ourselves.
	 */
	while (cinfo.output_scanline < cinfo.output_height)
	{
		/* jpeg_read_scanlines expects an array of pointers
		 * to scanlines.  Here the array is only one element
		 * long, but you could ask for more than one scanline
		 * at a time if that's more convenient.
		 */
		jpeg_read_scanlines(&cinfo, jbuffer, 1);
		hugo_writescanline(&cinfo, jbuffer[0]);

	}

/* Step 7: Finish decompression */

	jpeg_finish_decompress(&cinfo);

/* Step 8: Release JPEG decompression object */

	jpeg_destroy_decompress(&cinfo);

	fclose(infile);

	return 1;
}


#else   /* if !defined (GRAPHICS_SUPPORTED) || defined (IOTEST) */

int hugo_displaypicture(FILE *infile)
{
	fclose(infile);         /* since infile will be open */

	return 1;
}

#endif
