/*

Name:
DRV_DS.C

Description:
MikMod Software-mixing DirectSound driver

Method:
Single buffer, automatic refill using a high-resolution timer callback routine.

Portability:
Microsoft Win32 - MSVC++

	
NOTE:	Just for the record, I'm currently building libmik.lib as
	the debug build.  I've had to do this because drv_ds.c seems
	to demonstrate hissy output ONLY WHEN LIBMIK IS RELEASE BUILT.
	If anyone can figure out what the hell is going on, I'd love
	to hear it.
*/

#include <windows.h>
#include <windowsx.h>
#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <math.h>
#include <cguid.h>
#include <dsound.h>

#include "mikmod.h"


/* Globals: */
static HWND hwnd;
static LPDIRECTSOUND lpDirectSound = NULL;
static LPDIRECTSOUNDBUFFER lpDirectSoundBuffer = NULL;
#ifdef USE_DUP_DSB
static LPDIRECTSOUNDBUFFER lpDupDSB = NULL;
#endif
static LPDIRECTSOUNDBUFFER lpPrimaryDSB = NULL;
static UINT gwID;		/* timer handle */
static UWORD last;    		/* last recorded playing position */

#define WIN32BUFFERSIZE 30000

// Exportable:
HWND hwndMikMod = NULL;
CRITICAL_SECTION win32_mikmod_cs;	/* damned crash-proofing */


char *TranslateDSError(HRESULT hr)
{
	switch (hr)
	{
		case DSERR_ALLOCATED:
			return "DSERR_ALLOCATED";
		case DSERR_CONTROLUNAVAIL:
			return "DSERR_CONTROLUNAVAIL";
		case DSERR_INVALIDPARAM:
			return "DSERR_INVALIDPARAM";
		case DSERR_INVALIDCALL:
			return "DSERR_INVALIDCALL";
		case DSERR_GENERIC:
			return "DSERR_GENERIC";
		case DSERR_PRIOLEVELNEEDED:
			return "DSERR_PRIOLEVELNEEDED";
		case DSERR_OUTOFMEMORY:
			return "DSERR_OUTOFMEMORY";
		case DSERR_BADFORMAT:
			return "DSERR_BADFORMAT";
		case DSERR_UNSUPPORTED:
			return "DSERR_UNSUPPORTED";
		case DSERR_NODRIVER:
			return "DSERR_NODRIVER";
		case DSERR_ALREADYINITIALIZED:
			return "DSERR_ALREADYINITIALIZED";
		case DSERR_NOAGGREGATION:
			return "DSERR_NOAGGREGATION";
		case DSERR_BUFFERLOST:
			return "DSERR_BUFFERLOST";
		case DSERR_OTHERAPPHASPRIO:
			return "DSERR_OTHERAPPHASPRIO";
		case DSERR_UNINITIALIZED:
			return "DSERR_UNINITIALIZED";
		default:
			return "Unknown HRESULT";
	}
}


BOOL DS_IsThere(void)
{
	HRESULT hr;

	if (hr = DirectSoundCreate(NULL, &lpDirectSound, NULL)==DS_OK)
	{
		IDirectSound_Release(lpDirectSound);
		return TRUE;
	}
	else
	{
		myerr = TranslateDSError(hr);
		return FALSE;
	}
}


BOOL DS_Init(void)
{
	WAVEFORMATEX wf;
	DSCAPS dsc;
	DSBUFFERDESC dsbdesc;
	HRESULT hr;

	/* Init COM */
	if (hwndMikMod)
		hwnd = hwndMikMod;
	else
		hwnd = GetForegroundWindow();
	if (FAILED(CoInitialize(NULL))) goto Failed;

	/* Create DirectSound object */
	hr = CoCreateInstance(&CLSID_DirectSound, NULL, CLSCTX_INPROC_SERVER,
		  &IID_IDirectSound, (void**)&lpDirectSound);
	if (FAILED(hr)) goto Failed;

	/* Init DirectSound */
	hr = IDirectSound_Initialize(lpDirectSound, NULL);
	if (FAILED(hr)) goto Failed;

  	/* Get DS driver caps */
	IDirectSound_GetCaps(lpDirectSound, &dsc);

	if (!(dsc.dwFlags & DSCAPS_SECONDARYSTEREO) ||
		 !(dsc.dwFlags & DSCAPS_PRIMARYSTEREO))
	{
		/* Switch output to mono if device doesn't support stereo */
		md_mode &= ~DMODE_STEREO;
	}

	if (!(dsc.dwFlags & DSCAPS_SECONDARY16BIT) ||
		 !(dsc.dwFlags & DSCAPS_PRIMARY16BIT))
	{
		/* switch output to 8 bit if device doesn't support 16 bit */
		md_mode &= ~DMODE_16BITS;
	}

	/* Set up wave format structure */
	wf.wFormatTag = WAVE_FORMAT_PCM;
	wf.nChannels = (md_mode & DMODE_STEREO) ? 2 : 1;
	wf.nSamplesPerSec = md_mixfreq;

	wf.nBlockAlign = 1;
	if (md_mode & DMODE_STEREO) wf.nBlockAlign<<=1;
	if (md_mode & DMODE_16BITS) wf.nBlockAlign<<=1;
	wf.wBitsPerSample = (md_mode & DMODE_16BITS)?16:8;
	wf.nAvgBytesPerSec = wf.nSamplesPerSec * wf.nBlockAlign;
	wf.cbSize = 0;

	/* Set up DSBUFFERDESC structure */
	memset(&dsbdesc, 0, sizeof(DSBUFFERDESC));
	dsbdesc.dwSize = sizeof(DSBUFFERDESC);
	dsbdesc.dwFlags = DSBCAPS_GETCURRENTPOSITION2;
	dsbdesc.dwBufferBytes = WIN32BUFFERSIZE;
	dsbdesc.lpwfxFormat = &wf;

	/* Set cooperative level */
	hr = IDirectSound_SetCooperativeLevel(lpDirectSound, hwnd, DSSCL_PRIORITY);
	if (hr==DS_OK)
	{
		// Succeeded--try to create DirectSound buffer
		hr = IDirectSound_CreateSoundBuffer(lpDirectSound,
			&dsbdesc, &lpDirectSoundBuffer, NULL);
		if (hr==DS_OK)
		{
			// Get a primary sound buffer so we can
			// do some format tweaking
			// (Do I need to #@%$!! do this?)
			dsbdesc.dwFlags = DSBCAPS_PRIMARYBUFFER;
			dsbdesc.dwBufferBytes = 0;
			dsbdesc.lpwfxFormat = NULL;
			hr = IDirectSound_CreateSoundBuffer(lpDirectSound,
				&dsbdesc, &lpPrimaryDSB, NULL);
			if (hr==DS_OK)
			{
				hr = IDirectSoundBuffer_SetFormat(lpPrimaryDSB, &wf);
			}

			// Try to init software mixer
			if (VC_Init())
			{
				InitializeCriticalSection(&win32_mikmod_cs);
				return TRUE;
			}
		}
	}
	
	/* If we got here, then SetCooperativeLevel, CreateSoundBuffer,
	   or VC_Init failed
	*/
	if (lpDirectSoundBuffer) IDirectSoundBuffer_Release(lpDirectSoundBuffer);
	lpDirectSoundBuffer = NULL;
	if (lpDirectSound) IDirectSound_Release(lpDirectSound);
	lpDirectSound = NULL;
	if (lpPrimaryDSB) IDirectSound_Release(lpPrimaryDSB);
	lpPrimaryDSB = NULL;

Failed:
	myerr = TranslateDSError(hr);
	return FALSE;
}


void DS_Exit(void)
{
	/* Unload DirectSound buffer and object */
	if (lpDirectSoundBuffer)
	{
		IDirectSoundBuffer_Stop(lpDirectSoundBuffer);
		IDirectSoundBuffer_Release(lpDirectSoundBuffer);
		lpDirectSoundBuffer = NULL;
	}
	
	if (lpPrimaryDSB)
	{
		IDirectSound_Release(lpPrimaryDSB);
		lpPrimaryDSB = NULL;
	}

	if (lpDirectSound)
	{
		IDirectSound_Release(lpDirectSound);
		lpDirectSound = NULL;
	}
	
	/* Uninit COM */
	CoUninitialize();

	/* Cleanup memory, exit software mixer */
	VC_Exit();
	DeleteCriticalSection(&win32_mikmod_cs);
}


static ULONG GetPos(void)
{
	ULONG pos;

	IDirectSoundBuffer_GetCurrentPosition(lpDirectSoundBuffer, &pos, NULL);
//	return pos&0xfffffff0;
	return pos;
}


static void CALLBACK TimeProc(
	UINT IDEvent,		/* identifies timer event */
	UINT uReserved,		/* not used */
	DWORD dwUser,		/* application-defined instance data */
	DWORD dwReserved1,	/* not used */
	DWORD dwReserved2)	/* not used */
{
	UWORD todo, curr;
	static volatile int timersema = 0;
	
	/* Use a semaphore to prevent entering the mixing routines twice */
	if (++timersema==1)
	{
		EnterCriticalSection(&win32_mikmod_cs);

		curr = (UWORD)GetPos()%WIN32BUFFERSIZE;
		todo = (curr>last)?(curr-last):((WIN32BUFFERSIZE-last)+curr);

		/* Only if the number of bytes we have to fill isn't too
		   large--prevents lockups if the system can't handle
		   the load
		*/
		if (todo < (WIN32BUFFERSIZE/2))
		{
			LPVOID lpvPtr1;
			DWORD dwBytes1;
			LPVOID lpvPtr2;
			DWORD dwBytes2;
			HRESULT hr;

			hr = IDirectSoundBuffer_Lock(lpDirectSoundBuffer, last, todo,
				&lpvPtr1, &dwBytes1, &lpvPtr2, &dwBytes2, 0);

			// If we get DSERR_BUFFERLOST, restore and retry lock
			if (hr==DSERR_BUFFERLOST)
			{
				IDirectSoundBuffer_Restore(lpDirectSoundBuffer);
				hr = IDirectSoundBuffer_Lock(lpDirectSoundBuffer, curr, todo,
					&lpvPtr1, &dwBytes1, &lpvPtr2, &dwBytes2, 0);
			}
	
			if (hr==DS_OK)
			{
				if (curr > last)
				{
					VC_WriteBytes(lpvPtr1, (UWORD)(curr-last));
				}
				else
				{
					VC_WriteBytes(lpvPtr1, (UWORD)(WIN32BUFFERSIZE-last));
					VC_WriteBytes(lpvPtr2, curr);
				}

				hr = IDirectSoundBuffer_Unlock(lpDirectSoundBuffer,
					lpvPtr1, dwBytes1, lpvPtr2, dwBytes2);
			}
		}
		last = curr;

		LeaveCriticalSection(&win32_mikmod_cs);
	}
	timersema--;
}


void DS_PlayStart(void)
{
	long pos;

#ifdef USE_DUP_DSB
	/* Create the dupe buffer */
	IDirectSound_DuplicateSoundBuffer(lpDirectSound, lpDirectSoundBuffer, &lpDupDSB);

	/* Sync the dupe buffer */
	IDirectSoundBuffer_GetCurrentPosition(lpDirectSoundBuffer, &pos, NULL);
	IDirectSoundBuffer_SetCurrentPosition(lpDupDSB, pos);
#endif

	/* Begin the samples playing */
	IDirectSoundBuffer_Play(lpDirectSoundBuffer, 0, 0, DSBPLAY_LOOPING);
#ifdef USE_DUP_DSB
	IDirectSoundBuffer_Play(lpDupDSB, 0, 0, DSBPLAY_LOOPING);
#endif
	
	/* Start the software mixer */
	VC_PlayStart();

	timeBeginPeriod(20);      /* set the minimum resolution */

	gwID = timeSetEvent(40,		  /* how often */
			  40,		  /* timer resolution */
			  TimeProc,  	  /* callback function */
			  0,    	  /* info to pass to callback */
			  TIME_PERIODIC); /* oneshot or periodic */
}


void DS_PlayStop(void)
{
	LPVOID lpvPtr1;
	DWORD dwBytes1;
	LPVOID lpvPtr2;
	DWORD dwBytes2;
	HRESULT hr;

	/* Stop the timer */
	timeKillEvent(gwID);
	timeEndPeriod(20);

	/* Erase the leftover sample buffer */
	hr = IDirectSoundBuffer_Lock(lpDirectSoundBuffer, 0, 0, 
		&lpvPtr1, &dwBytes1, &lpvPtr2, &dwBytes2, DSBLOCK_ENTIREBUFFER);
	if (hr==DS_OK)
	{
		memset(lpvPtr1, 0, dwBytes1);
		hr = IDirectSoundBuffer_Unlock(lpDirectSoundBuffer,
			lpvPtr1, dwBytes1, lpvPtr2, dwBytes2);
	}

	/* Stop playing the wave */
	IDirectSoundBuffer_Stop(lpDirectSoundBuffer);

#ifdef USE_DUP_DSB
	/* Stop and release the dupe buffer */
	IDirectSoundBuffer_Stop(lpDupDSB);
	IDirectSoundBuffer_Release(lpDupDSB);
#endif
	/* Reset the software mixer */
	VC_PlayStop();
	VC_Exit();
	VC_Init();
}


void DS_Update(void)
{
	/* Not needed--buffers are automatically updated in the background */
}


DRIVER drv_ds = 
{
	NULL,
	"DirectSound",
	"MikMod Software-Mixing DirectSound Driver",
	DS_IsThere,
	VC_SampleLoad,
	VC_SampleUnload,
	DS_Init,
	DS_Exit,
	DS_PlayStart,
	DS_PlayStop,
	DS_Update,
	VC_VoiceSetVolume,
	VC_VoiceSetFrequency,
	VC_VoiceSetPanning,
	VC_VoicePlay
};

