/*
	HCWIN.C

	Win32 front-end for the Hugo Compiler

	Copyright (c) 1995-2006 by Kent Tessman
*/

#define HC_PROGRAM_NAME         "hc.exe"
#define APPNAME			"hcwin"

//#define USE_RICHEDIT

#include <windows.h>
#undef FAILED			// from windows.h

#include <shlobj.h>		// for folder-browse dialog

#include "hcheader.h"
#include "hcwin.h"

/* Function prototypes: */

LRESULT CALLBACK WndProc (HWND hwnd, UINT iMsg, WPARAM wParam, LPARAM lParam);
BOOL CALLBACK MainDialog(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam);
BOOL CALLBACK AboutDialog(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam);
void ResetCompileParams(HWND hwndDlg);
void ClearCompileParams(HWND hwndDlg);
int CallCompiler(char *, HWND);
BOOL CALLBACK SetupDialog(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam);
int ChooseNewFont(void);
int SetNewFont(char *face, int size, char *style);
void SetRunCommand(void);
void RunCompiledProgram(void);
void SetCurrentDirectories(void);
void LoadRegistryDefaults(void);
void SaveRegistryDefaults(void);

HINSTANCE AppInstance;
HWND wndMain = NULL;
HWND wndOutput = NULL;
HFONT fontOutput = NULL;
HMENU menuMain;
HACCEL accelMain;
PROCESS_INFORMATION processinfo;	// for compiler process

char *invocation_path;
char current_directory[MAXPATH];
char buffer[MAXPATH*2];
char *list_buffer = NULL;
char current_source_dir[MAXPATH];

#define DEFAULT_RUN_MENUITEM "Run Compiled Program"
char run_command[MAXPATH] = DEFAULT_RUN_MENUITEM;

/* Registry info: */
int reg_FullScreen = false;
int reg_xPos = CW_USEDEFAULT,
	reg_yPos = CW_USEDEFAULT,
	reg_Width = CW_USEDEFAULT,
	reg_Height = CW_USEDEFAULT;
int reg_LoadLast = true;
char reg_RunCommand[MAXPATH] = "hdwin";
char reg_Font[LF_FACESIZE] = "";
int reg_FontSize = 12;
char reg_FontStyle[32] = "";

/* Directories: */
char reg_SourceDir[MAXPATH] = "";
char reg_ObjectDir[MAXPATH] = "";	// reg_GamesDir
char reg_LibDir[MAXPATH] = "";
char reg_ResourceDir[MAXPATH] = "";
char reg_TempDir[MAXPATH] = "";

/* Menu constants: */
#define HC_FILE_COMPILE	1001
#define HC_FILE_EXIT	1002
#define HC_HELP_ABOUT	1101

/* Default "Compile" dialog settings (which also get
   saved into the registry): */
UINT default_abort = 0;
UINT default_hdx = 0;
UINT default_expanded = 0;
UINT default_full = 0;
UINT default_debuginfo = 0;
UINT default_hlb = 0;
UINT default_tree = 0;
UINT default_stats = 0;
UINT default_spellcheck = 0;
UINT default_mem = 0;
UINT default_override = 0;
//UINT default_compile_v25 =0;
UINT default_runafter = 0;
char switches[32];
char filename[MAXFILENAME] = "";
char additional_param[MAXPATH] = "";


/* WinMain

	The Windows entry point.  Set up windows, variables, menus,
	etc.  Create the output window--a child window of the edit
	control class.
*/

int WINAPI WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance,
		char *szCmdLine, int iCmdShow)
{
	static char szAppName[32];
	static char window_caption[32];
	WNDCLASS wndclass;
	MSG msg;


	/* Get global application info */
	AppInstance = hInstance;
	LoadRegistryDefaults();		/* i.e., reg_... variables */

#if defined (USE_RICHEDIT)
	LoadLibrary("RICHED32.DLL");	/* load RichEdit .DLL */
	InitCommonControls();
#endif
	
	// Needed for BIF_USENEWUI for SHBrowseForFolder
	CoInitialize(NULL);

	sprintf(szAppName, APPNAME);
	sprintf(window_caption, "Hugo Compiler v%d.%d%s", HCVERSION, HCREVISION, HCINTERIM);

	/* Create the main application window class and register it */
	wndclass.style          = CS_HREDRAW | CS_VREDRAW;
	wndclass.lpfnWndProc    = WndProc;
	wndclass.cbClsExtra     = 0;
	wndclass.cbWndExtra     = 0;
	wndclass.hInstance      = hInstance;
	wndclass.hIcon          = LoadIcon(hInstance, MAKEINTRESOURCE(PROGRAM_ICON));
	wndclass.hCursor        = LoadCursor(NULL, IDC_ARROW);
	wndclass.hbrBackground  = (HBRUSH)GetStockObject(GRAY_BRUSH);	//WHITE_BRUSH);
	wndclass.lpszMenuName   = NULL;
	wndclass.lpszClassName  = szAppName;
	RegisterClass(&wndclass);

	wndMain = CreateWindow( szAppName, 		// window class name
				window_caption, 	// window caption
				WS_OVERLAPPEDWINDOW |	// window style
					WS_CAPTION,
				reg_xPos,		// initial x position
				reg_yPos,		// initial y position
				reg_Width,		// initial x size
				reg_Height,		// initial y size
				NULL,                   // parent window handle
				NULL,                   // window menu handle
				hInstance,              // program instance handle
				NULL);                  // creation parameters


	/* Set up main menubar */
	menuMain = LoadMenu(AppInstance, MAKEINTRESOURCE(HCWIN_MENUBAR));
	SetMenu(wndMain, menuMain);
	accelMain = LoadAccelerators(AppInstance, MAKEINTRESOURCE(HCWIN_MENUBAR_ACCELERATORS));

	wndOutput = CreateWindowEx(WS_EX_CLIENTEDGE,
#if defined (USE_RICHEDIT)
				"RichEdit",		// window class name
#else
				"EDIT",
#endif
				"(No file compiled)",
				WS_CHILD |		// window style
					WS_VISIBLE | ES_LEFT |
					ES_READONLY | ES_MULTILINE |
					WS_HSCROLL | WS_VSCROLL,
				0, 0, 0, 0,		// initial position
				wndMain,		// parent window handle
				NULL,			// window menu handle
				hInstance,              // program instance handle
				NULL);                  // creation parameters


	if (strcmp(reg_Font, ""))
		SetNewFont(reg_Font, reg_FontSize, reg_FontStyle);
	else
		/* Default font is ANSI fixed font */
		SendMessage(wndOutput, WM_SETFONT, (WPARAM)GetStockObject(ANSI_FIXED_FONT), 0);

	/* Now actually bring the window up */
	ShowWindow(wndMain, reg_FullScreen?SW_MAXIMIZE:iCmdShow);
	if (reg_FullScreen)
		SendMessage(wndMain, WM_MOVE, 0, 0);

	invocation_path = GetCommandLine();

	/* Strip quotes from invocation_path */
	if (invocation_path[0]=='\"')
	{
		invocation_path[strlen(invocation_path)-2] = '\0';
		*invocation_path++;
	}

	GetCurrentDirectory(MAXPATH, current_directory);

	/* Set up SaveRegistryDefaults() to be called on normal termination */
	atexit(SaveRegistryDefaults);

	/* Launch the compile dialog off the top */
	SendMessage(wndMain, WM_COMMAND, HC_FILE_COMPILE, 0);

	while (GetMessage(&msg, NULL, 0, 0)>0)
	{
		if (!TranslateAccelerator(wndMain, accelMain, &msg))
		{
			TranslateMessage(&msg);
			DispatchMessage(&msg);
		}
	}

	CoUninitialize();

	return msg.wParam;
}


/* WndProc

	The callback for the main window.
*/

LRESULT CALLBACK WndProc(HWND hwnd, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
	RECT rect;

	switch (iMsg)
	{
		case WM_CREATE:
			return 0;

		case WM_SIZE:
			/* Resize/redraw the output window whenever
			   resizing the main window
			*/
			GetClientRect(wndMain, &rect);
			if (wndOutput)
			{
				MoveWindow(wndOutput, 0, 0, rect.right, rect.bottom, TRUE);
				ShowWindow(wndOutput, SW_SHOW);
			}

			if (wParam==SIZE_MAXIMIZED)
				reg_FullScreen = true;
			else
				reg_FullScreen = false;

			/* Only save values for normal move/size */
			if (wParam==SIZE_MINIMIZED || wParam==SIZE_MAXIMIZED)
				return 0;

			/* Otherwise fall through */

		case WM_MOVE:
		{
			RECT rect;

			GetWindowRect(hwnd, &rect);
			reg_xPos = rect.left;
			reg_yPos = rect.top;
			reg_Width = rect.right - rect.left;
			reg_Height = rect.bottom - rect.top;

			return 0;
		}

		case WM_COMMAND:
			switch (LOWORD(wParam))
			{
				case HC_FILE_COMPILE:
					DialogBox(AppInstance,
						MAKEINTRESOURCE(HC_MAIN_DIALOG),
						wndMain, MainDialog);
					break;

				case HC_FILE_COMPILEAGAIN:
					CallCompiler(filename, NULL);
					break;

				case HC_FILE_RUN:
					RunCompiledProgram();
					break;

				case HC_FILE_EXIT:
					PostQuitMessage(0);
					break;

				case HC_OPTIONS_FONT:
					ChooseNewFont();
					break;

				case HC_OPTIONS_RESETFONT:
					SendMessage(wndOutput, WM_SETFONT, (WPARAM)GetStockObject(ANSI_FIXED_FONT), 0);
					strcpy(reg_Font, "");
					RedrawWindow(wndOutput, NULL, NULL, RDW_ERASE | RDW_INVALIDATE);
					break;

				case HC_OPTIONS_SETUP:
					DialogBox(AppInstance,
						MAKEINTRESOURCE(HC_SETUP_DIALOG),
						wndMain, SetupDialog);
					break;

				case HC_HELP_ABOUT:
					DialogBox(AppInstance, MAKEINTRESOURCE(HC_ABOUT_DIALOG), wndMain, AboutDialog);
					break;
			}
			return 0;

		case WM_DESTROY:
			PostQuitMessage(0);
               		return 0;
	}
	return DefWindowProc(hwnd, iMsg, wParam, lParam);
}


/* MainDialog

	The callback for the "Select File to Compile" dialog.
*/


BOOL CALLBACK MainDialog(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	char *filetypes = "Hugo source files (*.hug)\0*.hug\0All files (*.*)\0*.*\0";
	int wNotifyCode, wID;
	HWND control;
	OPENFILENAME lpofn;

	switch (uMsg)
	{
		case WM_INITDIALOG:
			ResetCompileParams(hwndDlg);
			SetFocus(GetDlgItem(hwndDlg, HC_FILENAME_EDIT));
			SendMessage(GetDlgItem(hwndDlg, HC_FILENAME_EDIT), EM_SETSEL, 0, -1);

//#if defined (COMPILE_V25)
//			ShowWindow(GetDlgItem(hwndDlg, HC_COMPILE_V25_CHECK), SW_HIDE);
//#endif
			return 0;       /* since focus has been set */

		case WM_COMMAND:
			wNotifyCode = HIWORD(wParam);
			wID = LOWORD(wParam);
			control = (HWND)lParam;

			switch (wID)
			{
				case HC_BROWSE_BUTTON:
				{
					lpofn.lStructSize = sizeof(OPENFILENAME);
					lpofn.hwndOwner = hwndDlg;
					lpofn.hInstance = NULL;
					lpofn.lpstrFilter = filetypes;
					lpofn.lpstrCustomFilter = NULL;
					lpofn.nFilterIndex = 0;
					lpofn.lpstrFile = filename;
					lpofn.nMaxFile = MAXFILENAME;
					lpofn.lpstrFileTitle = NULL;
					lpofn.lpstrInitialDir = current_source_dir;
					lpofn.lpstrTitle = "Select File to Compile";
					lpofn.Flags = OFN_FILEMUSTEXIST | OFN_EXPLORER |
						OFN_HIDEREADONLY;
					lpofn.lpstrDefExt = "hug";
					lpofn.lpfnHook = NULL;
					lpofn.lpTemplateName = NULL;

					if (GetOpenFileName(&lpofn))
						SetDlgItemText(hwndDlg, HC_FILENAME_EDIT, filename);
					break;
				}

				case HC_CLEAR_BUTTON:
				{
					ClearCompileParams(hwndDlg);
					break;
				}

				case IDOK:
				{
					GetDlgItemText(hwndDlg, HC_FILENAME_EDIT, filename, MAXFILENAME);
					if (!strcmp(filename, ""))
					{
						MessageBox(hwndDlg,
							"No file selected for compilation.  Select a Hugo sourcefile using \"Filename to compile\" or \"Browse\".",
						       	"Compilation Error",
							MB_ICONEXCLAMATION);
					}
					else
					{
						// Erase the compile options dialog
						EndDialog(hwndDlg, 1);

						GetCurrentDirectory(MAXPATH, current_source_dir);
						CallCompiler(filename, hwndDlg);
						EnableMenuItem(menuMain, HC_FILE_COMPILEAGAIN, MF_ENABLED);
					}
					break;
				}

				case IDCANCEL:
					EndDialog(hwndDlg, 0);
					break;
			}
			return 0;

		case WM_CLOSE:
		case WM_DESTROY:
			EndDialog(hwndDlg, 0);
			return 0;
	}

	return 0;
}


/* AboutDialog

	Callback for the "About" dialog.
*/

BOOL CALLBACK AboutDialog(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	char about_text[256];

	switch (uMsg)
	{
		case WM_INITDIALOG:

		    	sprintf(about_text,
"Hugo Compiler v%d.%d%s - Win32 build (%s)\n\
Copyright  1995-2006 by Kent Tessman\n\
The General Coffee Company Film Productions\n\n\
All rights reserved.  Please see the Hugo License for details.",
			HCVERSION, HCREVISION, HCINTERIM, __DATE__);
			SetDlgItemText(hwndDlg, HC_ABOUT_TEXT, about_text);
			return TRUE;

		case WM_COMMAND:
			if (LOWORD(wParam)==IDCANCEL || LOWORD(wParam)==IDOK)
				EndDialog(hwndDlg, 0);
			return 0;

		case WM_DESTROY:
			break;
	}

	return 0;
}


/* ResetCompileParams

	Called to reset main dialog buttons, etc. to their defaults.
*/

void ResetCompileParams(HWND hwnd)
{
	CheckDlgButton(hwnd, HC_ABORT_CHECK, default_abort);
	CheckDlgButton(hwnd, HC_HDX_CHECK, default_hdx);
	CheckDlgButton(hwnd, HC_EXPANDED_CHECK, default_expanded);
	CheckDlgButton(hwnd, HC_FULL_CHECK, default_full);
	CheckDlgButton(hwnd, HC_DEBUGINFO_CHECK, default_debuginfo);
	CheckDlgButton(hwnd, HC_HLB_CHECK, default_hlb);
	CheckDlgButton(hwnd, HC_TREE_CHECK, default_tree);
	CheckDlgButton(hwnd, HC_STATS_CHECK, default_stats);
	CheckDlgButton(hwnd, HC_SPELLCHECK_CHECK, default_spellcheck);
	CheckDlgButton(hwnd, HC_MEM_CHECK, default_mem);
	CheckDlgButton(hwnd, HC_OVERRIDE_CHECK, default_override);
//	CheckDlgButton(hwnd, HC_COMPILE_V25_CHECK, default_compile_v25);
	CheckDlgButton(hwnd, HC_RUNAFTER_CHECK, default_runafter);

	SetDlgItemText(hwnd, HC_FILENAME_EDIT, filename);
	SetDlgItemText(hwnd, HC_PARAMETER_EDIT, additional_param);
}


/* ClearCompileParams */

void ClearCompileParams(HWND hwnd)
{
	default_abort = 0;
	default_hdx = 0;
	default_expanded = 0;
	default_full = 0;
	default_debuginfo = 0;
	default_hlb = 0;
	default_tree = 0;
	default_stats = 0;
	default_spellcheck = 0;
	default_mem = 0;
	default_override = 0;
//	default_compile_v25 = 0;
	default_runafter = 0;

	strcpy(filename, "");
	strcpy(additional_param, "");

	ResetCompileParams(hwnd);
}


/* ProgressDialog */

BOOL force_canceled = FALSE;

BOOL CALLBACK ProgressDialog(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg)
	{
		case WM_INITDIALOG:
			force_canceled = false;
			return TRUE;

		case WM_COMMAND:
			if (LOWORD(wParam)==IDCANCEL)
			{
				TerminateProcess(processinfo.hProcess, 1);
				EndDialog(hwndDlg, 0);
				force_canceled = true;
				break;
			}
	}

	return 0;
}


/* CallCompiler

	Creates a command line from the selected/entered parameters, then
	launches "hc.exe" (the command-line compiler) in a DOS box via
	CreateProcess().
*/

#define MAX_ARGS 64

int CallCompiler(char *filename, HWND hwnd)
{
	int i, r, argc = 0;
	char *argv[MAX_ARGS];
	char command_line[MAXPATH*2] = "";
	char compiler_path[MAXPATH];
	char ap[MAXPATH];	/* copy of additional_param */
	char listfile[MAXPATH], drive[MAXDRIVE], dir[MAXDIR], fname[MAXFILENAME], ext[MAXEXT];

	STARTUPINFO startupinfo;
	LPVOID lpEnvironment;
	DWORD exitcode;
	HANDLE listHandle;
	DWORD listfilesize, bytesread;


	/* Add program name */
	argv[argc++] = HC_PROGRAM_NAME;


	/* Add current directory as .LST file location, just to save the
	   headache of trying to figure out where the compiler put it
	*/
	argv[argc++] = "@list=.";


	/* Add the default switches, and figure out what additional
	   switches have been requested via checkboxes in the dialog
	*/
	if (hwnd)
	{
		strcpy(switches, "-ls");

		if (default_abort 	= IsDlgButtonChecked(hwnd, HC_ABORT_CHECK))
			strcat(switches, "a");
		if (default_hdx 	= IsDlgButtonChecked(hwnd, HC_HDX_CHECK))
			strcat(switches, "d");
		if (default_expanded	= IsDlgButtonChecked(hwnd, HC_EXPANDED_CHECK))
			strcat(switches, "e");
		if (default_full 	= IsDlgButtonChecked(hwnd, HC_FULL_CHECK))
			strcat(switches, "f");
		if (default_debuginfo 	= IsDlgButtonChecked(hwnd, HC_DEBUGINFO_CHECK))
			strcat(switches, "i");
		if (default_hlb		= IsDlgButtonChecked(hwnd, HC_HLB_CHECK))
			strcat(switches, "h");
		if (default_tree	= IsDlgButtonChecked(hwnd, HC_TREE_CHECK))
			strcat(switches, "o");
		if (default_stats	= IsDlgButtonChecked(hwnd, HC_STATS_CHECK))
			strcat(switches, "s");
		if (default_spellcheck	= IsDlgButtonChecked(hwnd, HC_SPELLCHECK_CHECK))
			strcat(switches, "t");
		if (default_mem		= IsDlgButtonChecked(hwnd, HC_MEM_CHECK))
			strcat(switches, "u");
		if (default_override	= IsDlgButtonChecked(hwnd, HC_OVERRIDE_CHECK))
			strcat(switches, "x");
//		if (default_compile_v25	= IsDlgButtonChecked(hwnd, HC_COMPILE_V25_CHECK))
//			strcat(switches, "25");

		default_runafter = IsDlgButtonChecked(hwnd, HC_RUNAFTER_CHECK);
	}

	/* Add switches */
	argv[argc++] = switches;


	/* Split the additional parameters line into argv parameters */

	if (hwnd) GetDlgItemText(hwnd, HC_PARAMETER_EDIT, additional_param, MAXPATH);
	/* Copy it because we're going to change it */
	strcpy(ap, additional_param);

	/* Strip out extra spaces from additional_param and change
	   remaining spaces to '\0' to mark end-of-string
	*/
	i = 0;
	while (ap[i]!='\0')
	{
		if (i==0) argv[argc++] = ap;

		if (ap[i]==' ')
		{
			i++;
			while (ap[i]==' ')
				strcpy(ap+i, ap+i+1);
			if (ap[i]=='\0')
			{
				ap[--i] = '\0';
				break;
			}
			ap[--i] = '\0';
			argv[argc++] = ap+i+1;
		}

		i++;
	}


	/* Finally, add the filename */
	argv[argc++] = filename;


	/* Build the command line */
	hugo_splitpath(invocation_path, drive, dir, fname, ext);
	if (drive[0]=='\0'&& dir[0]=='\0')
		hugo_makepath(command_line, "", current_directory, HC_PROGRAM_NAME, "");
	else
		hugo_makepath(command_line, drive, dir, HC_PROGRAM_NAME, "");

	strcpy(compiler_path, command_line);
#ifdef _DEBUG
// For debugging on Kent's machine
strcpy(compiler_path, "c:\\hugo\\hc.exe");
#endif

//	strcat(command_line, " ");
	// Have to quote the command_line executable name in case
	// it contains a space, etc.
	sprintf(command_line, "\"%s\" ", compiler_path);

	for (i=1; i<argc-1; i++)	// -1 is right before filename
	{
		strcat(command_line, argv[i]);
		strcat(command_line, " ");
	}
	sprintf(command_line+strlen(command_line), "\"%s\"", filename);

//	MessageBox(NULL, command_line, "Compiler Command Line", MB_OK);

	/* Figure out what the listfile will be called */
	hugo_splitpath(filename, drive, dir, fname, ext);
//	hugo_makepath(listfile, drive, dir, fname, ".lst");
	/* hc.exe will create the .lst file in the current directory */
	GetCurrentDirectory(MAXDIR, dir);
	hugo_makepath(listfile, "", dir, fname, ".lst");

	/* Remove the output file if it exists */
	remove(listfile);

	/* Call the compiler */
	GetStartupInfo(&startupinfo);
	startupinfo.wShowWindow = SW_SHOWNORMAL;
	lpEnvironment = GetEnvironmentStrings();

	r = CreateProcess(      compiler_path,
				command_line,
				NULL, NULL,     /* process/thread security attributes */
				TRUE,           /* handle inheritance */
				DETACHED_PROCESS | NORMAL_PRIORITY_CLASS,
				lpEnvironment,
				NULL,           /* current directory */
				&startupinfo,
				&processinfo );

	FreeEnvironmentStrings((LPTSTR)&lpEnvironment);

	if (r)
	{
		int count = 0;
		HWND wndProgress;
		DWORD status;

		wndProgress = CreateDialog(AppInstance, MAKEINTRESOURCE(HC_PROGRESS_DIALOG),
			wndOutput, ProgressDialog);
		ShowWindow(wndProgress, SW_SHOW);
				
		SendMessage(wndOutput, WM_SETTEXT, 0, (LPARAM)"Compiling...");
		UpdateWindow(wndOutput);

		/* Wait for the compiler to finish */
		do
		{
			MSG msg;

			status = WaitForSingleObject(processinfo.hProcess, 1000);
			if (++count > 9)
				count = 9;
			SendMessage(GetDlgItem(wndProgress, IDC_PROGRESS), PBM_SETPOS, count*10, 0);

			// Allow user to cancel
			if (PeekMessage(&msg, wndProgress, 0, 0, PM_REMOVE)>0)
			{
				TranslateMessage(&msg);
				DispatchMessage(&msg);
			}
		}
		while (status != WAIT_OBJECT_0);

		r = GetExitCodeProcess(processinfo.hProcess, &exitcode);

		// Show the completed status bar and pause for 1/2 second to display it
		if (r==TRUE && exitcode==0)
		{
			SendMessage(GetDlgItem(wndProgress, IDC_PROGRESS), PBM_SETPOS, 100, 0);
			Sleep(500);
		}
		DestroyWindow(wndProgress);
	}


	/* Load the list file (file.lst) */

	listHandle = CreateFile(listfile,
				GENERIC_READ, FILE_SHARE_READ,
				NULL,                   /* security */
				OPEN_EXISTING,
				FILE_ATTRIBUTE_NORMAL,
				NULL );                 /* no template */

	if (listHandle==INVALID_HANDLE_VALUE)
		goto ListFileError;

	if ((listfilesize = GetFileSize(listHandle, NULL))==0xFFFFFFFF)
		goto ListFileError;

	if (list_buffer!=NULL) free(list_buffer);

	/* Extra size in case we have to append any additional messages, like an error */
	if ((list_buffer = malloc((listfilesize+MAXBUFFER*2)*sizeof(char)))==NULL)
		goto ListFileError;
	strcpy(list_buffer, "");

	if (!ReadFile(listHandle, list_buffer, listfilesize, &bytesread, NULL ))
		goto ListFileError;

	list_buffer[listfilesize] = '\0';

	CloseHandle(listHandle);


	/* Add/print any appropriate error message, as necessary */
CompileError:
	if (force_canceled)
	{
		strcpy(list_buffer, "Compilation canceled by user.");
	}
	else if (r==FALSE || (r==TRUE && exitcode!=0))
	{
		if (r==TRUE && exitcode==2)
		{
			sprintf(buffer,	"Unable to compile \"%s\".", filename);
			MessageBox(hwnd, buffer, "Compilation Error", MB_ICONEXCLAMATION);
		}

//		sprintf(buffer, "Unable to compile using '%s' (%d)", command_line, r);
		sprintf(buffer, "Unable to compile using '%s' - Exit code = %d (%d)", command_line, exitcode, r);

		if (strcmp(list_buffer, "")) strcat(list_buffer, "\r\n");
		strcat(list_buffer, buffer);
	}

	/* Send list_buffer to the output window */
	SendMessage(wndOutput, WM_SETTEXT, 0, (LPARAM)list_buffer);

	/* The successfully compiled file can now be run from the
	   File menu
	*/
	SetRunCommand();

	/* If we compiled successfully and free of errors */
	if (r==TRUE && exitcode==0)
	{
		if (default_runafter) RunCompiledProgram();
	}
	/* Otherwise there was an error of some sort */
	else
		EnableMenuItem(menuMain, HC_FILE_RUN, MF_GRAYED);

	return TRUE;

ListFileError:
	/* A little bit of complexity here because an error loading
	   the list file may end up being due to a compilation error
	*/
	if (r==FALSE || (r==TRUE && exitcode!=0))
	{
		if (list_buffer = malloc(1024*sizeof(char)))
		{
			strcpy(list_buffer, "");
			goto CompileError;
		}
	}

	sprintf(buffer, "Error compiling \"%s\"\r\n\
(or unable to load compiler output \"%s\")", filename, listfile);
	SendMessage(wndOutput, WM_SETTEXT, 0, (LPARAM)buffer);

	EnableMenuItem(menuMain, HC_FILE_RUN, MF_GRAYED);

	return FALSE;
}


/*
 * Setup dialog:
 *
 */

int CALLBACK BrowseCallbackProc(HWND hwnd, UINT uMsg, LPARAM lParam, LPARAM lpData)
{
	switch (uMsg)
	{
		/* Select the default folder on init */
		case BFFM_INITIALIZED:
			SendMessage(hwnd, BFFM_SETSELECTION, TRUE, lpData);
			break;
	}
	return 0;
}


/* SetupDialog

	Callback for the "Setup" dialog.
*/

BOOL CALLBACK SetupDialog(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg)
	{
		case WM_INITDIALOG:
			SetDlgItemText(hwndDlg, HC_SETUP_SOURCE, reg_SourceDir);
			SetDlgItemText(hwndDlg, HC_SETUP_OBJECT, reg_ObjectDir);
			SetDlgItemText(hwndDlg, HC_SETUP_LIB, reg_LibDir);
			SetDlgItemText(hwndDlg, HC_SETUP_RESOURCE, reg_ResourceDir);
			SetDlgItemText(hwndDlg, HC_SETUP_TEMP, reg_TempDir);
			CheckDlgButton(hwndDlg, HC_SETUP_LOADLAST, reg_LoadLast);
			SetDlgItemText(hwndDlg, HC_SETUP_RUNCOMMAND, reg_RunCommand);

			return 1;	/* since focus wasn't set */

		case WM_COMMAND:
			switch (LOWORD(wParam))
			{
				case IDOK:
					GetDlgItemText(hwndDlg, HC_SETUP_SOURCE, reg_SourceDir, MAXPATH-1);
					GetDlgItemText(hwndDlg, HC_SETUP_OBJECT, reg_ObjectDir, MAXPATH-1);
					GetDlgItemText(hwndDlg, HC_SETUP_LIB, reg_LibDir, MAXPATH-1);
					GetDlgItemText(hwndDlg, HC_SETUP_RESOURCE, reg_ResourceDir, MAXPATH-1);
					GetDlgItemText(hwndDlg, HC_SETUP_TEMP, reg_TempDir, MAXPATH-1);
					reg_LoadLast = IsDlgButtonChecked(hwndDlg, HC_SETUP_LOADLAST);
					GetDlgItemText(hwndDlg, HC_SETUP_RUNCOMMAND, reg_RunCommand, MAXPATH-1);
					SetCurrentDirectories();

					EndDialog(hwndDlg, 1);
					return 0;

				case HC_BROWSE_BUTTON:
				case HC_BROWSE_BUTTON2:
				case HC_BROWSE_BUTTON3:
				case HC_BROWSE_BUTTON4:
				case HC_BROWSE_BUTTON5:
				{
					BROWSEINFO bi;
					LPITEMIDLIST pidlBrowse, pidlStart = NULL;
					LPMALLOC g_pMalloc;

					/* Get shell's global malloc interface */
					SHGetMalloc(&g_pMalloc);

					bi.hwndOwner = hwndDlg;
					// Must have called CoInitialize() to
					// use BIF_USENEWUI
					//bi.ulFlags = BIF_EDITBOX | BIF_USENEWUI;
					bi.ulFlags = BIF_EDITBOX | 0x40;
					bi.lpfn = BrowseCallbackProc;

					strcpy(buffer, "Select default folder for ");
					
					if (wParam==HC_BROWSE_BUTTON)
					{
						bi.pszDisplayName = reg_SourceDir;
						strcat(buffer, "Source Files");
					}
					else if (wParam==HC_BROWSE_BUTTON2)
					{
						bi.pszDisplayName = reg_ObjectDir;
						strcat(buffer, "Object Files");
					}
					else if (wParam==HC_BROWSE_BUTTON3)
					{
						bi.pszDisplayName =reg_LibDir;
						strcat(buffer, "Library Files");
					}
					else if (wParam==HC_BROWSE_BUTTON4)
					{
						bi.pszDisplayName =reg_ResourceDir;
						strcat(buffer, "Resource Files");
					}
					else
					{
						bi.pszDisplayName = reg_TempDir;
						strcat(buffer, "Temporary Files");
					}

					bi.lParam = (LPARAM)bi.pszDisplayName;
					bi.lpszTitle = &buffer[0];
					bi.pidlRoot = pidlStart;
					pidlBrowse = SHBrowseForFolder(&bi);

					if (pidlBrowse)
					{
						SHGetPathFromIDList(pidlBrowse, bi.pszDisplayName);

						/* Free the ID pointer */
						g_pMalloc->lpVtbl->Free(g_pMalloc, pidlBrowse); 

						/* Reset editbox text */
						SendMessage(hwndDlg, WM_INITDIALOG, 0, 0);
					}

					return 0;
				}

				case IDCANCEL:
					EndDialog(hwndDlg, 0);
					return 0;
			}
			return 0;

		case WM_CLOSE:
			EndDialog(hwndDlg, 0);
			return 0;
	}

	return 0;
}


/*
 * Utility functions for centering built-in dialog boxes:
 *
 */

/* CenterWindow

	Centers one window over another.
*/

void CenterWindow(HWND wndChild, HWND wndParent)
{
	RECT rect;
	POINT p;
	int width, height;

	/* Find center of parent window */
	GetWindowRect(wndParent, &rect);
	p.x = (rect.right + rect.left)/2;
	p.y = (rect.bottom + rect.top)/2;

	GetWindowRect(wndChild, &rect);
	width = rect.right - rect.left;
	height = rect.bottom - rect.top;

	SetWindowPos(wndChild, NULL,
		p.x - width/2, p.y - height/2,
		0, 0,
		SWP_NOSIZE | SWP_NOZORDER);
}


/* CenterHookProc

	Because the common dialog boxes supplied by Windows all
	position themselves at (0, 0), which looks dumb.
*/

UINT APIENTRY CenterHookProc(HWND hwndDlg, UINT uiMsg, WPARAM wParam, LPARAM lParam)
{
	if (uiMsg==WM_INITDIALOG)
	{
		if (!wParam) hwndDlg = GetParent(hwndDlg);

		CenterWindow(hwndDlg, GetDesktopWindow()); //GetParent(hwndDlg));
		return TRUE;
	}
	return FALSE;
}


/*
 * Font choosing and setting:
 *
 */

int ChooseNewFont(void)
{
	HDC dc;
	LOGFONT lf;
	CHOOSEFONT cf;

	/* First set up a LOGFONT structure with details on the 
	   current font (for defaults in the ChooseFont dialog)
	*/
	strcpy(lf.lfFaceName, reg_Font);

	/* Convert point to logical height */
	dc = GetDC(wndMain);
	lf.lfHeight = -MulDiv(reg_FontSize, GetDeviceCaps(dc, LOGPIXELSY), 72);
	DeleteDC(dc);

	/* Now initialize and launch the ChooseFont dialog: */

	cf.lStructSize = sizeof(CHOOSEFONT);
	cf.hwndOwner = wndMain;
	cf.lpLogFont = &lf;
	cf.lpszStyle = reg_FontStyle;
	cf.Flags = CF_SCREENFONTS | CF_FORCEFONTEXIST |
		CF_USESTYLE | CF_INITTOLOGFONTSTRUCT | CF_ENABLEHOOK;
	// hook function to center the dialog:
	cf.lpfnHook = (LPCFHOOKPROC)CenterHookProc;

	if (!ChooseFont(&cf)) return false;	/* return if failed */

	reg_FontSize = cf.iPointSize/10;
	strcpy(reg_Font, lf.lfFaceName);
	SetNewFont(reg_Font, reg_FontSize, reg_FontStyle);

	return true;
}

int SetNewFont(char *face, int size, char *style)
{
	HDC dc;
	LOGFONT lf;

	strcpy(lf.lfFaceName, face);

	lf.lfWidth = lf.lfEscapement = lf.lfOrientation = 0;

	if (!strncmp(reg_FontStyle, "Bold", 4))
		lf.lfWeight = FW_BOLD;
	else
		lf.lfWeight = FW_NORMAL;

	if ((strlen(reg_FontStyle)>=6) &&
		!strncmp(reg_FontStyle+strlen(reg_FontStyle)-6, "Italic", 5))
	{
		lf.lfItalic = 1;
	}
	else
		lf.lfItalic = 0;

	lf.lfUnderline = 0;
	lf.lfCharSet = DEFAULT_CHARSET;
	lf.lfStrikeOut = 0;
	lf.lfOutPrecision = OUT_DEFAULT_PRECIS;
	lf.lfClipPrecision = CLIP_DEFAULT_PRECIS;
	lf.lfQuality = DEFAULT_QUALITY;

	/* Convert point to logical height */
	dc = GetDC(wndOutput);
	lf.lfHeight = -MulDiv(reg_FontSize, GetDeviceCaps(dc, LOGPIXELSY), 72);
	DeleteDC(dc);

	if (fontOutput) DeleteObject(fontOutput);
	fontOutput = CreateFontIndirect(&lf);
	if (fontOutput==NULL)
	{
		MessageBox(wndMain, "Unable to load font", "Hugo Compiler", MB_ICONWARNING);
		return false;
	}

	SendMessage(wndOutput, WM_SETFONT, (WPARAM)fontOutput, 0);
	RedrawWindow(wndOutput, NULL, NULL, RDW_ERASE | RDW_INVALIDATE);

	return true;
}


/*
 * Run compiled program:
 *
 */

/* GetCompiledFilename

	Assumes that a successful compilation has taken place and
	that list_buffer holds the output (including the filename).
*/

char *GetCompiledFilename(void)
{
	static char compiled_filename[MAXPATH];
	char *filename, *endoffilename;

	filename = strstr(list_buffer, "Object file:  ");
	if (filename==NULL)
	{
NoFileError:
		EnableMenuItem(menuMain, HC_FILE_RUN, MF_GRAYED);
		return "(no file compiled)";
	}
	filename+=14;
	endoffilename = strstr(filename, "bytes)");
	while (*endoffilename!='(')
	{
		endoffilename--;
		if (endoffilename<=filename) goto NoFileError;
	}

	strncpy(compiled_filename, filename, endoffilename-filename-1);
	compiled_filename[endoffilename-filename+1] = '\0';

	return compiled_filename;
}


/* SetRunCommand

	Sets the File menu entry so that it contains the name
	of the compiled program (if applicable).
*/

void SetRunCommand(void)
{
	HMENU menuFile;
	MENUITEMINFO mii;

	EnableMenuItem(menuMain, HC_FILE_RUN, MF_ENABLED);

	menuFile = GetSubMenu(menuMain, 0);	
	mii.cbSize = sizeof(mii);
	mii.fMask = MIIM_TYPE;
	mii.fType = MFT_STRING;
	mii.dwTypeData = run_command;
	mii.cch = MAXPATH;
	GetMenuItemInfo(menuFile, HC_FILE_RUN, FALSE, &mii);
	sprintf(run_command, "&Run %s \tCtrl+R", GetCompiledFilename());
	SetMenuItemInfo(menuFile, HC_FILE_RUN, FALSE, &mii);
}


/* RunCompiledProgram */

void RunCompiledProgram(void)
{
	char command_line[MAXPATH];
	char line[MAXPATH+64];
	int r;

	STARTUPINFO startupinfo;
	PROCESS_INFORMATION processinfo;
	LPVOID lpEnvironment;

#ifdef _DEBUG
// For debugging on Kent's machine
strcpy(command_line, "c:\\hugo\\hdwin.exe");
#else
	strcpy(command_line, reg_RunCommand);
#endif
	strcat(command_line, " ");
	strncat(command_line, GetCompiledFilename(), MAXPATH-strlen(command_line));

	/* Call the run command */
	GetStartupInfo(&startupinfo);
	startupinfo.wShowWindow = SW_SHOWNORMAL;
	lpEnvironment = GetEnvironmentStrings();

	r = CreateProcess(
#ifdef _DEBUG
// For debugging on Kent's machine
				"c:\\hugo\\hdwin.exe",
#else
				NULL,//reg_RunCommand,
#endif
				command_line,
				NULL, NULL,     /* process/thread security attributes */
				TRUE,           /* handle inheritance */
				NORMAL_PRIORITY_CLASS,
				lpEnvironment,
				NULL,           /* current directory */
				&startupinfo,
				&processinfo );

	FreeEnvironmentStrings((LPTSTR)&lpEnvironment);

	if (!r)
	{
		sprintf(line, "Unable to execute command:  \"%s\"", command_line);
		MessageBox(wndMain, line, "Run Command Error", MB_ICONEXCLAMATION);
	}
}


/*
 * Hugo Engine registry management:
 *
 */

/* LoadRegistryDefaults */

void LoadRegistryDefaults(void)
{
	HKEY hkey;
	DWORD type;
	size_t size;

	if (RegOpenKey(HKEY_CURRENT_USER,
		"Software\\General Coffee Co.\\Hugo\\Compiler",
		&hkey)==ERROR_SUCCESS)
	{
		/* REG_DWORDs: */
		type = REG_DWORD;
		size = sizeof(int);
		RegQueryValueEx(hkey, "xPos", 0, &type, (LPBYTE)&reg_xPos, &size);
		RegQueryValueEx(hkey, "yPos", 0, &type, (LPBYTE)&reg_yPos, &size);
		RegQueryValueEx(hkey, "Width", 0, &type, (LPBYTE)&reg_Width, &size);
		RegQueryValueEx(hkey, "Height", 0, &type, (LPBYTE)&reg_Height, &size);
		RegQueryValueEx(hkey, "LoadLast", 0, &type, (LPBYTE)&reg_LoadLast, &size);

		RegQueryValueEx(hkey, "FontSize", 0, &type,
			(LPBYTE)&reg_FontSize, &size);

		/* REG_SZ strings: */
		type = REG_SZ;
		size = sizeof(reg_Font);
		RegQueryValueEx(hkey, "Font", 0, &type,
			(LPBYTE)&reg_Font, &size);
		size = sizeof(reg_FontStyle);
		RegQueryValueEx(hkey, "FontStyle", 0, &type,
			(LPBYTE)&reg_FontStyle, &size);

		size = sizeof(reg_RunCommand);
		RegQueryValueEx(hkey, "RunCommand", 0, &type,
			(LPBYTE)&reg_RunCommand, &size);

		size = sizeof(reg_SourceDir);
		RegQueryValueEx(hkey, "SourceDir", 0, &type,
			(LPBYTE)&reg_SourceDir, &size);
		size = sizeof(reg_ObjectDir);	/* have to reset this */
		RegQueryValueEx(hkey, "ObjectDir", 0, &type,
			(LPBYTE)&reg_ObjectDir, &size);
		size = sizeof(reg_LibDir);
		RegQueryValueEx(hkey, "LibDir", 0, &type,
			(LPBYTE)&reg_LibDir, &size);
		size = sizeof(reg_ResourceDir);
		RegQueryValueEx(hkey, "ResourceDir", 0, &type,
			(LPBYTE)&reg_ResourceDir, &size);
		size = sizeof(reg_TempDir);
		RegQueryValueEx(hkey, "TempDir", 0, &type,
			(LPBYTE)&reg_TempDir, &size);
		SetCurrentDirectories();

		RegCloseKey(hkey);
	}

	/* Loads data from most recent compile */

	if ((reg_LoadLast) && RegOpenKey(HKEY_CURRENT_USER,
		"Software\\General Coffee Co.\\Hugo\\Compiler\\Last Compile",
		&hkey)==ERROR_SUCCESS)
	{
		DWORD flags;

		/* REG_DWORDs: */
		type = REG_DWORD;
		size = sizeof(int);
		RegQueryValueEx(hkey, "Flags", 0, &type, (LPBYTE)&flags, &size);

		default_abort =		flags & 0x00000001;
		default_hdx =		flags & 0x00000002;
		default_expanded =	flags & 0x00000004;
		default_full =		flags & 0x00000008;
		default_debuginfo =	flags & 0x00000010;
		default_hlb =		flags & 0x00000020;
		default_tree =		flags & 0x00000040;
		default_stats =		flags & 0x00000080;
		default_spellcheck =	flags & 0x00000100;
		default_mem =		flags & 0x00000200;
		default_override =	flags & 0x00000400;
//		default_compile_v25 =	flags & 0x00000800;
		default_runafter =	flags & 0x00001000;

		type = REG_SZ;
		size = sizeof(filename);
		RegQueryValueEx(hkey, "Filename", 0, &type,
			(LPBYTE)filename, &size);
		size = sizeof(additional_param);
		RegQueryValueEx(hkey, "Parameters", 0, &type,
			(LPBYTE)additional_param, &size);

		RegCloseKey(hkey);
	}
}


/* SaveRegistryDefaults */

void SaveRegistryDefaults(void)
{
	HKEY hkey;
	int version;

	if (RegCreateKey(HKEY_CURRENT_USER,
		"Software\\General Coffee Co.\\Hugo\\Compiler",
		&hkey)==ERROR_SUCCESS)
	{
		/* Values are grouped by type */

		/* Save the version in case we ever need it */
		version = HCVERSION*10 + HCREVISION;
		RegSetValueEx(hkey, "Version", 0, REG_DWORD,
			(LPBYTE)&version, sizeof(version));
		RegSetValueEx(hkey, "LoadLast", 0, REG_DWORD,
			(LPBYTE)&reg_LoadLast, sizeof(reg_LoadLast));

		RegSetValueEx(hkey, "xPos", 0, REG_DWORD,
			(LPBYTE)&reg_xPos, sizeof(reg_xPos));
		RegSetValueEx(hkey, "yPos", 0, REG_DWORD,
			(LPBYTE)&reg_yPos, sizeof(reg_yPos));
		
		RegSetValueEx(hkey, "Width", 0, REG_DWORD,
			(LPBYTE)&reg_Width, sizeof(reg_Width));
		RegSetValueEx(hkey, "Height", 0, REG_DWORD,
			(LPBYTE)&reg_Height, sizeof(reg_Height));

		RegSetValueEx(hkey, "FontSize", 0, REG_DWORD,
			(LPBYTE)&reg_FontSize, sizeof(reg_FontSize));

		RegSetValueEx(hkey, "Font", 0, REG_SZ,
			(LPBYTE)reg_Font, strlen(reg_Font)+1);
		RegSetValueEx(hkey, "FontStyle", 0, REG_SZ,
			(LPBYTE)reg_FontStyle, strlen(reg_FontStyle)+1);

		RegSetValueEx(hkey, "RunCommand", 0, REG_SZ,
			(LPBYTE)reg_RunCommand, strlen(reg_RunCommand)+1);

		RegSetValueEx(hkey, "SourceDir", 0, REG_SZ,
			(LPBYTE)reg_SourceDir, strlen(reg_SourceDir)+1);
		RegSetValueEx(hkey, "ObjectDir", 0, REG_SZ,
			(LPBYTE)reg_ObjectDir, strlen(reg_ObjectDir)+1);
		RegSetValueEx(hkey, "LibDir", 0, REG_SZ,
			(LPBYTE)reg_LibDir, strlen(reg_LibDir)+1);
		RegSetValueEx(hkey, "ResourceDir", 0, REG_SZ,
			(LPBYTE)reg_ResourceDir, strlen(reg_ResourceDir)+1);
		RegSetValueEx(hkey, "TempDir", 0, REG_SZ,
			(LPBYTE)reg_TempDir, strlen(reg_TempDir)+1);

		RegCloseKey(hkey);
	}

	/* Save data from most recent compile */

	if (RegCreateKey(HKEY_CURRENT_USER,
		"Software\\General Coffee Co.\\Hugo\\Compiler\\Last Compile",
		&hkey)==ERROR_SUCCESS)
	{
		DWORD flags = 0;

		if (default_abort)	 flags |= 0x00000001;
		if (default_hdx)	 flags |= 0x00000002;
		if (default_expanded)	 flags |= 0x00000004;
		if (default_full)	 flags |= 0x00000008;
		if (default_debuginfo)	 flags |= 0x00000010;
		if (default_hlb)	 flags |= 0x00000020;
		if (default_tree)	 flags |= 0x00000040;
		if (default_stats)	 flags |= 0x00000080;
		if (default_spellcheck)	 flags |= 0x00000100;
		if (default_mem)	 flags |= 0x00000200;
		if (default_override)	 flags |= 0x00000400;
//		if (default_compile_v25) flags |= 0x00000800;
		if (default_runafter)	 flags |= 0x00001000;

		RegSetValueEx(hkey, "Flags", 0, REG_DWORD,
			(LPBYTE)&flags, sizeof(flags));

		RegSetValueEx(hkey, "Filename", 0, REG_SZ,
			(LPBYTE)filename, strlen(filename)+1);
		RegSetValueEx(hkey, "Parameters", 0, REG_SZ,
			(LPBYTE)additional_param, strlen(additional_param)+1);

		RegCloseKey(hkey);
	}
}


/* SetCurrentDirectories */

void SetCurrentDirectories(void)
{
	strcpy(current_source_dir, reg_SourceDir);
	SetEnvironmentVariable("HUGO_SOURCE", reg_SourceDir);

	SetEnvironmentVariable("HUGO_GAMES", reg_ObjectDir);
	SetEnvironmentVariable("HUGO_OBJECT", reg_ObjectDir);

	SetEnvironmentVariable("HUGO_LIB", reg_LibDir);

	SetEnvironmentVariable("HUGO_RESOURCE", reg_ResourceDir);

	SetEnvironmentVariable("HUGO_TEMP", reg_TempDir);
}
