/*
	HEVIDEO.CPP

	AVI and MPEG video player for Win32:

		hugo_playvideo

	for the Hugo Engine

	Copyright (c) 1999-2006 by Kent Tessman

	Remember that hugo_playvideo() is called with the file hot--
	i.e., opened and positioned to the start of the resource.
	It must be closed before returning.
*/

#ifndef NO_VIDEO

#pragma warning(disable : 4995)

#include <windows.h>
#undef GetProp		// from windows.h
#include <process.h>	// for _beginthreadex()
#include <mmsystem.h>
#include <dsound.h>
#include <control.h>
#include <strmif.h>
#include <uuids.h>
#include <evcode.h>
#include <math.h>

#ifndef USE_VIDEO_TEMPFILE
// Async file reader stuff, requires DirectX SDK with DirectShow extras:
#include <stdio.h>
#include <streams.h>
#include <include/asyncio.h>
#include <include/asyncrdr.h>
#include <memfile/memfile.h>
#endif

extern "C" {

#include "heheader.h"
#include "hewin32.h"

int PlayVideo_DirectShow(HUGO_FILE infile, long reslength,
	char loop_flag, int volume);

HWND wndVideo = 0;
WNDCLASS wcVideo;
char video_running;
char video_is_looping;
long initial_video_volume;
char video_safely_stopped = true;
char just_stopped_video = false;

// If we're calling the video player in a thread
#ifdef USE_VIDEO_TEMPFILE
char video_tempfile[MAX_PATH];
#endif
DWORD WINAPI VideoThread(LPVOID);
HANDLE video_thread = 0;
DWORD thread_id;
HUGO_FILE thread_infile;
long thread_reslength;
char thread_loop_flag;
int thread_volume;

// DirectShow interfaces:
static IFilterGraph *pifg = NULL;
static IGraphBuilder *pigb = NULL;
static IMediaControl *pimc = NULL;
static IMediaEvent *pime = NULL;
static IVideoWindow *pivw = NULL;
static IMediaPosition *pimp = NULL;
static IBasicAudio *piba = NULL;

struct VIDEO_WINDOW video_window;
char video_filename[MAX_PATH];
char video_resname[MAX_PATH];


/* hugo_hasvideo */

int hugo_hasvideo(void)
{
	if (!reg_DisplayGraphics) return false;
	return true;
}


/* hugo_stopvideo */

void hugo_stopvideo(void)
{
	if (video_thread)
	{
		DWORD exitcode;
		MSG msg;

		video_running = 0;
		GetExitCodeThread(video_thread, &exitcode);
		while (exitcode==STILL_ACTIVE || video_safely_stopped==false)
		{
			/* Check Windows messages while we're waiting,
			   since we're going to be closing the video
			   playback window
			*/
			if (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
			{
				if (msg.message==WM_QUIT)
				{
					exit(msg.wParam);
				}
				TranslateMessage(&msg);
				DispatchMessage(&msg);
			}
			GetExitCodeThread(video_thread, &exitcode);
		}
	}
	video_thread = NULL;
}


/* hugo_playvideo

	The actual call(s) to PlayVideo_*() get dispatched from here.
*/

int hugo_playvideo(HUGO_FILE infile, long reslength,
	char loop_flag, char background, int volume)
{
	if (!reg_DisplayGraphics)
	{
		fclose(infile);
		return false;
	}
	
	hugo_stopvideo();

	// Save all this stuff since we may be using it
	// in a thread (during which the main thread may
	// change it):
	video_window.left = physical_windowleft;
	video_window.top = physical_windowtop;
	video_window.right = physical_windowright;
	video_window.bottom = physical_windowbottom;
	video_window.width = physical_windowwidth;
	video_window.height = physical_windowheight;
	strcpy(video_filename, loaded_filename);
	strcpy(video_resname, loaded_resname);
	
	// Need to do this due to changes in UpdateClient() for
	// handling updating during video playback:
	post_picture_update = false;
	UpdateClient(true);

	if (!background)
	{
		video_thread = NULL;
		return PlayVideo_DirectShow(infile, reslength, loop_flag, volume);
	}

	/* If we're playing the video in the background, we have to
	   start the player as a thread
	*/
	else
	{
		thread_infile = infile;
		thread_reslength = reslength;
		thread_loop_flag = loop_flag;
		thread_volume = volume;
		video_running = -1;
//		video_thread = CreateThread(NULL, 1024, VideoThread, 0, 0, &thread_id);
		video_thread = (void *)_beginthreadex(NULL, 1024, (unsigned int (__stdcall *)(void *))VideoThread, 0, 0, (unsigned int *)&thread_id);
		// Wait for video playback to start
		while (video_running==-1 && video_thread)
		{
			MSG msg;
			
			if (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
			{
				if (msg.message==WM_QUIT)
					exit(msg.wParam);

				TranslateMessage(&msg);
				DispatchMessage(&msg);
			}
		}

		/* Needed for XP, it seems: */
		if (video_thread)
		{
			RECT rect;
			GetClientRect(wndMain, &rect);
			InvalidateRect(wndMain, &rect, FALSE);
		}

		return video_thread?1:0;
	}
}


/* VideoThread

	Called if we're playing the video in the background, in
	which case we need to launch another thread.
*/

DWORD WINAPI VideoThread(LPVOID param)
{
	int r;
	CoInitialize(NULL);
	r = PlayVideo_DirectShow(thread_infile, thread_reslength, thread_loop_flag, thread_volume);
	CoUninitialize();
	return r;
}


/* VideoWndProc

	The callback for wndVideo, used to monitor messages during
	DirectShow video playback, below.
*/

LRESULT CALLBACK VideoWndProc(HWND hwnd, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
	switch (iMsg)
	{
		case WM_CREATE:
			return 0;

		/* Double-clicking video window stops playback.
		   Pressing Esc does, too, but that's handled in
		   VideoControl
		*/
		case WM_LBUTTONDBLCLK:
			video_running = 0;
			return 0;
	}

	return DefWindowProc(hwnd, iMsg, wParam, lParam);
}


/* VideoControl

	Here, just for clarity, is the loop that runs while the
	video resource is playing.  Control is not returned to the
	engine until the video is done (unless, we call VideoControl
	from a thread).
*/

void VideoControl(void)
{
	MSG msg;
	long lEventCode;
	long lParam1;
	long lParam2;
	static char suspended = false;
	static int last_reg_PlaySounds = 1,
		last_reg_DisplayGraphics = 1;

	video_running = true;

	// Needed for XP:
	RECT rect;
	GetClientRect(wndMain, &rect);
	InvalidateRect(wndMain, &rect, false);

	while (video_running)
	{
		/* Silence this baby if the main window loses focus */
		if (GetForegroundWindow()!=wndMain)
		{
			pimc->Pause();
			suspended = true;
		}
		else if (suspended)
		{
			pimc->Run();
			suspended = false;
		}

		/* In case we change menu settings for audio/video */
		if (reg_PlaySounds!=last_reg_PlaySounds)
		{
			if (!reg_PlaySounds)
				piba->put_Volume(-10000);
			else
				piba->put_Volume(initial_video_volume);
			last_reg_PlaySounds = reg_PlaySounds;
		}
		if (reg_DisplayGraphics!=last_reg_DisplayGraphics)
		{
			if (!reg_DisplayGraphics) video_running = 0;
			last_reg_DisplayGraphics = reg_DisplayGraphics;
		}

		/* Check to see if the video resource is finished */
		pime->GetEvent(&lEventCode, &lParam1, &lParam2, 20);
		if (lEventCode==EC_COMPLETE)
		{
			if (video_is_looping)
			{
				pimp->put_CurrentPosition(0);
				pimc->Run();
			}
			else
			{
				video_running = 0;
				break;
			}
		}
		pime->FreeEventParams(lEventCode, lParam1, lParam2);

		// Only allow the user to break playback if we're
		// not running in the background (i.e., a thread)
		if ((video_thread==NULL) && PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
		{
			if (msg.message==WM_QUIT)
			{
#ifdef USE_VIDEO_TEMPFILE
				if (strcmp(video_tempfile, ""))
					DeleteFile(video_tempfile);
#endif
				exit(msg.wParam);
			}
				
			// Disallow keypress messages during video
			if (msg.message!=WM_KEYDOWN && msg.message!=WM_CHAR)
			{
				TranslateMessage(&msg);
				DispatchMessage(&msg);
			}
			else if (msg.wParam==VK_ESCAPE)
			{
				video_running = 0;
				break;
			}
		}

		SleepEx(10, FALSE);
	}

	pimc->Stop();	// in case it's still running
	just_stopped_video = true;
}


/* PlayVideo_DirectShow

	For playing video resources.  Returns true on success, 
	false on failure.  Uses the same calling architecture as 
	PlayMusic_DirectShow in hesound.c.
*/

int PlayVideo_DirectShow(HUGO_FILE infile, long length, char loop_flag, int volume)
{
	HRESULT hr = 0;
	long x, y, width, height;
	float ratio;
	long ws;

#ifdef USE_VIDEO_TEMPFILE
	WCHAR wFile[MAX_PATH];
#else
	// async reader stuff:
	long pos;
	CMediaType mt;
	CMemReader *rdr = NULL;
	PBYTE pbMem = NULL;
	CMemStream *Stream = NULL;
#endif

	/* If we're running on Windows NT 4.0 or earlier( which
	   don't support DirectShow), NT_sound_system will be true
	*/
	if (NT_sound_system)
	{
		fclose(infile);
		goto DirectShowError;
	}

	// As long as this is false, we can't start any new video
	// (see hugo_stopvideo())
	video_safely_stopped = false;

	video_is_looping = loop_flag;

#ifdef USE_VIDEO_TEMPFILE
	/* See first if resourcefile is a standalone resource, or
	   if we need to copy the resource into a standalone tempfile.
	   Either way, the file gets closed.
	*/
	if (ftell(infile)!=0)
	{
		/* Change resource_file to the temp cache file */
		if (!CreateResourceCache(video_resname, infile, length))
			return false;
		strcpy(video_filename, "");
		strcpy(video_tempfile, video_resname);
	}
	else
	{
		strcpy(video_tempfile, "");
		fclose(infile);
	}

	/* DirectShow speaks Unicode */
	MultiByteToWideChar(CP_ACP, 0, video_resname, -1, wFile, MAX_PATH);

#else	// otherwise we're using the async filter:

	pos = ftell(infile);
	fclose(infile);
	HANDLE hFile;
	if ((hFile = TryToOpenHandle(loaded_filename, "games"))==INVALID_HANDLE_VALUE)
	{
		if ((hFile = TryToOpenHandle(loaded_filename, "object"))==INVALID_HANDLE_VALUE)
		{
			goto DirectShowError;
		}
	}
	SetFilePointer(hFile, pos, NULL, FILE_BEGIN);

	pbMem = new BYTE[length];
	if (pbMem == NULL)
	{
//		printf("Could not allocate memory\n");
		CloseHandle(hFile);
		goto DirectShowError;
	}

	DWORD dwBytesRead;
	if (!ReadFile(hFile, (LPVOID)pbMem, length, &dwBytesRead, NULL) 
		|| dwBytesRead!=(unsigned)length)
	{
//		printf("Could not read file\n");
		CloseHandle(hFile);
		goto DirectShowError;
	}
	CloseHandle(hFile);

	mt.majortype = MEDIATYPE_Stream;
	if (resource_type==MPEG_R)	mt.subtype = MEDIASUBTYPE_MPEG1System;
	else if (resource_type==AVI_R)	mt.subtype = MEDIASUBTYPE_Avi;

	Stream = new CMemStream(pbMem, length);
	rdr = new CMemReader(Stream, &mt, &hr);
	if (FAILED(hr) || rdr == NULL)
	{
//		printf("Could not create filter HRESULT 0x%8.8X\n", hr);
		goto DirectShowError;
	}
	// MS's async filter sample says to do this so we don't
	// "accidentally go away"--not sure why, but:
	rdr->AddRef();
#endif

	/* Create the control window */
	ZeroMemory(&wcVideo, sizeof(wcVideo));
	wcVideo.style = CS_DBLCLKS | CS_PARENTDC;
	wcVideo.lpszClassName = "HugoVideoWindow";
	wcVideo.lpfnWndProc = VideoWndProc;
	wcVideo.hInstance = AppInstance;
	RegisterClass(&wcVideo);
	wndVideo = CreateWindow(wcVideo.lpszClassName, "",
		WS_CHILD, 0, 0, 10, 10,
		wndMain, NULL, AppInstance, NULL);
//		NULL, NULL, AppInstance, NULL);
	if (!wndVideo) goto DirectShowError;

	/* Instantiate the filter graph manager, asking for the 
	   IFilterGraph interface
	*/
	hr = CoCreateInstance(CLSID_FilterGraph, NULL, CLSCTX_INPROC_SERVER, IID_IFilterGraph, (void **)&pifg);
	if (FAILED(hr))
	{
		goto DirectShowError;
	}

#ifndef USE_VIDEO_TEMPFILE
	// Add the async filter
	pifg->AddFilter(rdr, NULL);
#endif
	/* Get the IGraphBuilder interface */
	hr = pifg->QueryInterface(IID_IGraphBuilder, (void **)&pigb);
	if (FAILED(hr))
	{
		goto ReleaseAll;
	}

	/* Get the IMediaControl interface */
	pigb->QueryInterface(IID_IMediaControl, (void **)&pimc);

	/* Get the IMediaEvent interface */
	pigb->QueryInterface(IID_IMediaEvent, (void **)&pime);

	/* Get the IMediaPosition interface */
	pigb->QueryInterface(IID_IMediaPosition, (void **)&pimp);

	/* Render the file */
#ifdef USE_VIDEO_TEMPFILE
	hr = pigb->RenderFile(wFile, NULL);
#else
	hr = pigb->Render(rdr->GetPin(0));
#endif
	if (FAILED(hr)) goto ReleaseAll;

	/* Get the IVideoWindow interface and set up the window */
	pigb->QueryInterface(IID_IVideoWindow, (void **)&pivw);
	pivw->put_Owner((OAHWND)wndMain);
	pivw->put_MessageDrain((OAHWND)wndVideo);
	pivw->get_WindowStyle(&ws);
	ws &= ~WS_CAPTION;	// remove the title bar
	ws &= ~WS_THICKFRAME;	// remove frame
	pivw->put_WindowStyle(ws);

	/* Get the IBasicAudio interface */
	pigb->QueryInterface(IID_IBasicAudio, (void **)&piba);
	piba->put_Volume(
		// This formula is from hesound.c:
		volume?(long)((log10((long double)volume)-2)*5000):-10000);
	piba->get_Volume(&initial_video_volume);
	/* Turn volume off if audio disabled */
	if (!reg_PlaySounds) piba->put_Volume(-10000);

	/* Position the video window */
	pivw->GetWindowPosition(&x, &y, &width, &height);

	// For some goofy reason, when we put_WindowStyle() it doesn't
	// adjust the window size to remove the title bar and frame
	width -= (GetSystemMetrics(SM_CXSIZEFRAME)*2);
	height -= (GetSystemMetrics(SM_CYSIZEFRAME)*2 +
		GetSystemMetrics(SM_CYCAPTION));

	ratio = (float)width/(float)height;
	if (width > video_window.width)
	{
		width = video_window.width;
		height = (int)((float)width / ratio);
	}
	if (height > video_window.height)
	{
		height = video_window.height;
		width = (int)((float)height * ratio);
	}
	x = video_window.left+video_window.width/2 - width/2;
	y = video_window.top+video_window.height/2 - height/2;
	pivw->SetWindowPosition(x, y, width-1, height-1);

	// Now resize (possibly shrink) the video window to fit around the actual
	// video window, so that UpdateClient() will update around it properly

	video_window.left = x;
	video_window.right = x+width-1;
	video_window.top = y;
	video_window.bottom = y+height-1;
	video_window.width = width;
	video_window.height = height;

	/* Play the video resource */
	hr = pimc->Run();
	if (FAILED(hr)) goto ReleaseAll;

	/* Let VideoControl() handle all Windows messaging, etc. while
	   the resource is playing
	*/
	VideoControl();

ReleaseAll:
	// Need to set main window background to black brush to avoid some video
	// window jitter when shutting down
	SetClassLong(wndMain, GCL_HBRBACKGROUND, (long)GetStockObject(BLACK_BRUSH));

	if (pivw)
	{
		HWND hasfocus;
		hasfocus = GetFocus();
		pivw->put_Visible(OAFALSE);
		pivw->put_Owner(NULL);
		// Do this to avoid the titlebar flicker when 
		// focus shifts due to the put_Owner() call:
		if (hasfocus==wndMain) SetFocus(wndMain);
	}
	if (pimp) { pimp->Release(); pimp = NULL; }
	if (pime) { pime->Release(); pime = NULL; }
	if (pimc) { pimc->Release(); pimc = NULL; }
	if (piba) { piba->Release(); piba = NULL; }
	if (pivw) { pivw->Release(); pivw = NULL; }
	if (pigb) { pigb->Release(); pigb = NULL; }
	if (pifg) { pifg->Release(); pifg = NULL; }
	if (wndVideo) DestroyWindow(wndVideo);

	// Reset window background to NULL brush
	SetClassLong(wndMain, GCL_HBRBACKGROUND, 0);

	wndVideo = 0;

DirectShowError:

#ifdef USE_VIDEO_TEMPFILE
	if (strcmp(video_tempfile, ""))
		DeleteFile(video_tempfile);
#else
	// Release async filter stuff:
	if (rdr) rdr->Release();
	if (Stream) delete Stream;
	if (pbMem) delete pbMem;
#endif

/*
LPVOID lpMsgBuf;
 
BOOL r = 

FormatMessage( 
    FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
    NULL,
    GetLastError(),
    MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
    (LPTSTR) &lpMsgBuf,
    0,
    NULL 
);

// Display the string.
if (r) MessageBox(wndMain, (char *)lpMsgBuf, "GetLastError", MB_OK|MB_ICONINFORMATION );

// Free the buffer.
LocalFree( lpMsgBuf );
*/
	video_running = false;

	video_safely_stopped = true;

	if (FAILED(hr))
	{
#if defined (DEBUGGER)
		DebugMessageBox("Video Playing Error",
			"Unable to play video via DirectShow.");
#endif
		return false;
	}
	else
		return true;
}

}	// extern "C"

#else	// NO_VIDEO

#include <stdio.h>

extern "C"
{

char video_running;
char just_stopped_video = false;

int hugo_hasvideo(void)
{
	return 0;
}
void hugo_stopvideo(void)
{}
int hugo_playvideo(FILE *infile, long reslength,
	char loop_flag, char background, int volume)
{
	return 0;
}

}	// extern "C"
#endif
