/* $Header$ */

/* Copyright (c) 2000, 2002 Michael J. Roberts.  All Rights Reserved. */
/*
Name
  vmarray.h - T3 array metaclass
Function
  
Notes
  
Modified
  03/07/00 MJRoberts  - Creation
*/

#ifndef VMARRAY_H
#define VMARRAY_H

#include <stdlib.h>
#include "vmtype.h"
#include "vmobj.h"
#include "vmcoll.h"
#include "vmglob.h"


/* ------------------------------------------------------------------------ */
/*
 *   Arrays are very similar to lists, but an array's contents can be
 *   modified during execution.  Because arrays are dynamic, an array's
 *   image file data are copied into the heap.
 *   
 *   An array's image file data is the same as a list's:
 *   
 *   UINT2 number_of_elements
 *.  DATAHOLDER element[1]
 *.  DATAHOLDER element[2]
 *.  etc
 *   
 *   The object extension for an array is almost identical, but adds a
 *   prefix giving the "allocated" size of the array, and a suffix with a
 *   pointer to our original image data (if applicable) plus one bit per
 *   element after the end of the element[] list for undo bits:
 *   
 *   UINT2 allocated_elements
 *.  UINT2 number_of_elements
 *.  DATAHOLDER element[1]
 *.  DATAHOLDER element[2]
 *.  etc
 *.  undo_bits
 *   
 *   The allocated_elements counter gives the total number of elements
 *   allocated in the array.  This is normally the same as the
 *   number_of_elements counter, but in some cases an array is allocated
 *   with more elements than are actually used; for example, when we
 *   construct an array as a filtered subset of another array, we allocate
 *   the new array at the full size of the original as an upper bound, then
 *   shrink the stored number of elements according to the filter condition.
 *   
 *   The undo_bits are packed 8 bits per byte.  Note that the undo_bits
 *   follow all allocated slots, so these do not change location when we
 *   change the number of elements actually in use.  
 */

/* ------------------------------------------------------------------------ */
/*
 *   Array Metaclass 
 */

class CVmObjArray: public CVmObjCollection
{
    friend class CVmMetaclassArray;
    friend class CVmQSortArray;

public:
    /* metaclass registration object */
    static class CVmMetaclass *metaclass_reg_;
    class CVmMetaclass *get_metaclass_reg() const { return metaclass_reg_; }

    /* am I of the given metaclass? */
    virtual int is_of_metaclass(class CVmMetaclass *meta) const
    {
        /* try my own metaclass and my base class */
        return (meta == metaclass_reg_
                || CVmObjCollection::is_of_metaclass(meta));
    }

    /* create dynamically using stack arguments */
    static vm_obj_id_t create_from_stack(VMG_ const uchar **pc_ptr,
                                         uint argc);

    /* 
     *   call a static property - we don't have any of our own, so simply
     *   "inherit" the base class handling 
     */
    static int call_stat_prop(VMG_ vm_val_t *result,
                              const uchar **pc_ptr, uint *argc,
                              vm_prop_id_t prop)
    {
        return CVmObjCollection::
            call_stat_prop(vmg_ result, pc_ptr, argc, prop);
    }

    /* reserve constant data */
    virtual void reserve_const_data(VMG_ class CVmConstMapper *mapper,
                                    vm_obj_id_t self);

    /* convert to constant data */
    virtual void convert_to_const_data(VMG_ class CVmConstMapper *mapper,
                                       vm_obj_id_t self);

    /* create an array with no initial contents */
    static vm_obj_id_t create(VMG_ int in_root_set);

    /* create an array with a given number of elements */
    static vm_obj_id_t create(VMG_ int in_root_set, size_t element_count);

    /* 
     *   determine if an object is an array - it is if the object's
     *   virtual metaclass registration index matches the class's static
     *   index 
     */
    static int is_array_obj(VMG_ vm_obj_id_t obj)
        { return vm_objp(vmg_ obj)->is_of_metaclass(metaclass_reg_); }

    /* notify of deletion */
    void notify_delete(VMG_ int in_root_set);

    /* set a property */
    void set_prop(VMG_ class CVmUndo *undo,
                  vm_obj_id_t self, vm_prop_id_t prop, const vm_val_t *val);

    /* get a property */
    int get_prop(VMG_ vm_prop_id_t prop, vm_val_t *val,
                 vm_obj_id_t self, vm_obj_id_t *source_obj, uint *argc);

    /* add a value to the array */
    void add_val(VMG_ vm_val_t *result,
                 vm_obj_id_t self, const vm_val_t *val);

    /* subtract a value from the array */
    void sub_val(VMG_ vm_val_t *result,
                 vm_obj_id_t self, const vm_val_t *val);

    /* undo operations */
    void notify_new_savept();
    void apply_undo(VMG_ struct CVmUndoRecord *rec);
    void mark_undo_ref(VMG_ struct CVmUndoRecord *rec);

    /* we keep only strong references */
    void remove_stale_undo_weak_ref(VMG_ struct CVmUndoRecord *) { }

    /* mark references */
    void mark_refs(VMG_ uint state);

    /* 
     *   remove weak references - we keep only normal (strong) references,
     *   so this routine doesn't need to do anything 
     */
    void remove_stale_weak_refs(VMG0_) { }

    /* load from an image file */
    void load_from_image(VMG_ vm_obj_id_t self, const char *ptr, size_t siz);

    /* rebuild for image file */
    virtual ulong rebuild_image(VMG_ char *buf, ulong buflen);

    /* reload from the image file */
    void reload_from_image(VMG_ vm_obj_id_t self,
                           const char *ptr, size_t siz);

    /* save to a file */
    void save_to_file(VMG_ class CVmFile *fp);

    /* restore from a file */
    void restore_from_file(VMG_ vm_obj_id_t self,
                           class CVmFile *fp, class CVmObjFixup *fixups);

    /*
     *   When we're the right-hand side of a '+' or '-' operation whose
     *   left-hand side is another collection type that treats these
     *   operators as concatenation/set subtraction, add/subtract our
     *   elements individually.  
     */
    virtual size_t get_coll_addsub_rhs_ele_cnt(VMG0_) const
        { return get_element_count(); }
    virtual void get_coll_addsub_rhs_ele(VMG_ vm_val_t *result,
                                         vm_obj_id_t self, size_t idx)
    {
        vm_val_t idx_val;

        /* set up the index value */
        idx_val.set_int((int)idx);

        /* index the array */
        index_val(vmg_ result, self, &idx_val);
    }

    /* index the array */
    void index_val(VMG_ vm_val_t *result, vm_obj_id_t self,
                   const vm_val_t *index_val);

    /* set an indexed element of the array */
    void set_index_val(VMG_ vm_val_t *new_container, vm_obj_id_t self,
                       const vm_val_t *index_val, const vm_val_t *new_val);

    /* 
     *   Check a value for equality.  We will match any list or array with
     *   the same number of elements and the same value for each element.  
     */
    int equals(VMG_ vm_obj_id_t self, const vm_val_t *val) const;

    /* calculate a hash value for the array */
    uint calc_hash(VMG_ vm_obj_id_t self) const;

    /* 
     *   assume that we've been changed since loading, if we came from the
     *   image file 
     */
    int is_changed_since_load() const { return TRUE; }

    /* 
     *   get the allocated number of elements of the array - this will be
     *   greater than or equal to get_element_count(), and reflects the
     *   actual allocated size 
     */
    size_t get_allocated_count() const
        { return vmb_get_len(get_array_ext_ptr()); }

    /* get the number of elements in the array */
    size_t get_element_count() const
        { return vmb_get_len(get_array_ext_ptr() + 2); }

protected:
    /* load image data */
    virtual void load_image_data(VMG_ const char *ptr, size_t siz);

    /* 
     *   Create a new instance of the same class for a getp method.  Each
     *   subclass must override this to create an instance of the subclass
     *   rather than an instance of the CVmObjArray base class.  
     */
    virtual vm_obj_id_t create_for_getp(VMG_ size_t cnt)
        { return create(vmg_ FALSE, cnt); }
    
    /* 
     *   Get an instance of the same class for the result of the append or
     *   appendUnique methods.  For an array, this creates a new object.
     *   For a Vector, however, this will return the original object,
     *   because appending to a vector simply adds to the existing vector.  
     */
    virtual vm_obj_id_t create_for_append(VMG_ vm_obj_id_t /*self*/,
                                          size_t cnt)
        { return create(vmg_ FALSE, cnt); }

    /* uniquify the value for an appendUnique operation */
    virtual void uniquify_for_append(VMG_ vm_obj_id_t /*self*/)
    {
        /* 
         *   for an array, we can simply use the construction uniquifier,
         *   since appendUnique creates a brand new object 
         */
        cons_uniquify(vmg0_);
    }
    
    /* 
     *   remove one element from the vector; does not update the element
     *   count, but returns the new element counter so that the caller can
     *   update the element count 
     */
    size_t sub_one_val(VMG_ vm_obj_id_t self,
                       const vm_val_t *val, size_t ele_cnt);

    /* 
     *   set the element count for a subtract operation - for an array,
     *   subtraction is a construction operation, so we simply set the
     *   element count as we would for construction; subclasses (such as
     *   Vector) might need to do something different, such as save undo
     *   for the size change 
     */
    virtual void set_element_count_for_sub(VMG_ vm_obj_id_t self,
                                           size_t ele_cnt)
        { set_element_count(ele_cnt); }

    /* create an iterator */
    virtual void new_iterator(VMG_ vm_val_t *retval,
                              const vm_val_t *self_val);

    /* create a live iterator */
    virtual void new_live_iterator(VMG_ vm_val_t *retval,
                                   const vm_val_t *self_val);

    /* create a list with no initial contents */
    CVmObjArray() { ext_ = 0; }

    /* 
     *   create a list with a given number of elements, for construction
     *   of the list element-by-element 
     */
    CVmObjArray(VMG_ size_t element_count);

    /* get a pointer to my array data */
    const char *get_array_ext_ptr() const { return ext_; }

    /* 
     *   get my extension data pointer for construction purposes - this is
     *   a writable pointer, so that a caller can fill our data buffer
     *   during construction 
     */
    char *cons_get_array_ext_ptr() const { return ext_; }

    /*
     *   Calculate the amount of space we need to store a list of a given
     *   length.  We require two bytes for the length prefix, plus the
     *   space for each element, plus the space for the undo bits (one bit
     *   per element, but we pack eight bits per byte).  
     */
    static size_t calc_alloc(size_t elecnt)
    {
        return (calc_alloc_prefix_and_ele(elecnt)
                + ((elecnt + 7) & ~7));
    }

    /* 
     *   calculate the allocation amount, including only the count prefix
     *   and element data 
     */
    static size_t calc_alloc_prefix_and_ele(size_t elecnt)
    {
        return (2*VMB_LEN + calc_alloc_ele(elecnt));
    }

    /* calculate the allocation amount, including only the element storage */
    static size_t calc_alloc_ele(size_t elecnt)
    {
        return elecnt * VMB_DATAHOLDER;
    }

    /* allocate space for the array, given the number of elements */
    void alloc_array(VMG_ size_t element_count);

    /* property evaluator - undefined function */
    int getp_undef(VMG_ vm_obj_id_t, vm_val_t *, uint *) { return FALSE; }

    /* property evaluator - convert to a list */
    int getp_to_list(VMG_ vm_obj_id_t self, vm_val_t *val, uint *argc);

    /* property evaluator - get the number of elements */
    int getp_get_size(VMG_ vm_obj_id_t self, vm_val_t *val, uint *argc);

    /* property evaluator - copy from another array or list */
    int getp_copy_from(VMG_ vm_obj_id_t self, vm_val_t *val, uint *argc);

    /* property evaluator - fill with a value */
    int getp_fill_val(VMG_ vm_obj_id_t self, vm_val_t *val, uint *argc);

    /* property evaluator - select a subset */
    int getp_subset(VMG_ vm_obj_id_t self, vm_val_t *val, uint *argc);

    /* property evaluator - apply a callback to each element */
    int getp_apply_all(VMG_ vm_obj_id_t self, vm_val_t *val, uint *argc);

    /* property evaluator - find the first element satisfying a condition */
    int getp_index_which(VMG_ vm_obj_id_t self, vm_val_t *val, uint *argc);

    /* property evaluator - call a callback on each element */
    int getp_for_each(VMG_ vm_obj_id_t self, vm_val_t *val, uint *argc);

    /* property evaluator - mapAll */
    int getp_map_all(VMG_ vm_obj_id_t self, vm_val_t *val, uint *argc);

    /* property evaluator - indexOf */
    int getp_index_of(VMG_ vm_obj_id_t self, vm_val_t *val, uint *argc);

    /* property evaluator - valWhich */
    int getp_val_which(VMG_ vm_obj_id_t self, vm_val_t *val, uint *argc);

    /* property evaluator - lastIndexOf */
    int getp_last_index_of(VMG_ vm_obj_id_t self, vm_val_t *val, uint *argc);

    /* property evaluator - lastIndexWhich */
    int getp_last_index_which(VMG_ vm_obj_id_t self,
                              vm_val_t *val, uint *argc);

    /* property evaluator - valWhich */
    int getp_last_val_which(VMG_ vm_obj_id_t self, vm_val_t *val, uint *argc);

    /* property evaluator - countOf */
    int getp_count_of(VMG_ vm_obj_id_t self, vm_val_t *val, uint *argc);

    /* property evaluator - countWhich */
    int getp_count_which(VMG_ vm_obj_id_t self, vm_val_t *val, uint *argc);

    /* property evaluator - getUnique */
    int getp_get_unique(VMG_ vm_obj_id_t self, vm_val_t *val, uint *argc);
    
    /* property evaluator - appendUnique */
    int getp_append_unique(VMG_ vm_obj_id_t self, vm_val_t *val, uint *argc);

    /* property evaluator - sort */
    int getp_sort(VMG_ vm_obj_id_t self, vm_val_t *val, uint *argc);

    /* general handler for indexOf, lastIndexOf, and countOf */
    int gen_index_of(VMG_ vm_obj_id_t self, vm_val_t *val, uint *argc,
                     int forward, int count_only);

    /* general handler for indexWhich, lastIndexWhich, and countWhich */
    int gen_index_which(VMG_ vm_obj_id_t self, vm_val_t *val, uint *argc,
                        int forward, int count_only);

    /* for construction, remove all redundant elements */
    void cons_uniquify(VMG0_);

    /* get an element */
    void get_element(size_t idx, vm_val_t *val) const
    {
        /* get the data from the data holder in our extension */
        vmb_get_dh(get_element_ptr(idx), val);
    }

    /* set an element, recording undo for the change */
    void set_element_undo(VMG_ vm_obj_id_t self,
                          size_t idx, const vm_val_t *val);

    /* set an element */
    void set_element(size_t idx, const vm_val_t *val)
    {
        /* set the element's data holder from the value */
        vmb_put_dh(get_element_ptr(idx), val);
    }

    /* set the allocated size */
    void set_allocated_count(size_t cnt)
        { vmb_put_len(cons_get_array_ext_ptr(), cnt); }

    /* set the number of elements in the array */
    void set_element_count(size_t cnt)
        { vmb_put_len(cons_get_array_ext_ptr() + 2, cnt); }

    /* given an index, get a pointer to the element's data in the list */
    char *get_element_ptr(size_t idx) const
    {
        /* 
         *   figure out where this element's data holder is by skipping
         *   the count prefix, then skipping past preceding data holders 
         */
        return cons_get_array_ext_ptr() + 2*VMB_LEN + (idx * VMB_DATAHOLDER);
    }

    /* get a constant pointer to an element */
    static const char *get_element_ptr_const(const char *ext, size_t idx)
    {
        return ext + 2*VMB_LEN + (idx * VMB_DATAHOLDER);
    }

    /* increment an element pointer */
    static void inc_element_ptr(char **ptr) { *ptr += VMB_DATAHOLDER; }

    /* get a given undo bit */
    int get_undo_bit(size_t idx) const
    {
        char *p;
        int bitnum;

        /* get the byte containing the bit for this index */
        p = get_undo_ptr() + (idx >> 3);

        /* get the bit number within the byte */
        bitnum = (idx & 7);

        /* get this bit */
        return ((*p & (1 << bitnum)) != 0);
    }

    /* set a given undo bit */
    void set_undo_bit(size_t idx, int val)
    {
        char *p;
        int bitnum;

        /* get the byte containing the bit for this index */
        p = get_undo_ptr() + (idx >> 3);

        /* get the bit number within the byte */
        bitnum = (idx & 7);

        /* set or clear the bit */
        if (val)
            *p |= (1 << bitnum);
        else
            *p &= ~(1 << bitnum);
    }

    /* clear all undo bits */
    void clear_undo_bits()
    {
        memset(get_undo_ptr(), 0, ((get_allocated_count() + 7) & ~7));
    }

    /* get a pointer to the first byte of the undo bits */
    char *get_undo_ptr() const
    {
        /* the undo bits follow the last element data */
        return (cons_get_array_ext_ptr()
                + 2*VMB_LEN
                + get_allocated_count()*VMB_DATAHOLDER);
    }

    /* property evaluation function table */
    static int (CVmObjArray::*func_table_[])(VMG_ vm_obj_id_t self,
                                             vm_val_t *retval, uint *argc);
};

/* ------------------------------------------------------------------------ */
/*
 *   Registration table object 
 */
class CVmMetaclassArray: public CVmMetaclass
{
public:
    /* get the global name */
    const char *get_meta_name() const { return "array/030006"; }

    /* create from image file */
    void create_for_image_load(VMG_ vm_obj_id_t id)
        { new (vmg_ id) CVmObjArray(); }

    /* create from restoring from saved state */
    void create_for_restore(VMG_ vm_obj_id_t id)
        { new (vmg_ id) CVmObjArray(); }

    /* create dynamically using stack arguments */
    vm_obj_id_t create_from_stack(VMG_ const uchar **pc_ptr, uint argc)
        { return CVmObjArray::create_from_stack(vmg_ pc_ptr, argc); }

    /* call a static property */
    int call_stat_prop(VMG_ vm_val_t *result,
                       const uchar **pc_ptr, uint *argc,
                       vm_prop_id_t prop)
    {
        return CVmObjArray::call_stat_prop(vmg_ result, pc_ptr, argc, prop);
    }

    /* I'm a Collection object */
    CVmMetaclass *get_supermeta_reg() const
        { return CVmObjCollection::metaclass_reg_; }
};

#endif /* VMARRAY_H */

/*
 *   Register the class 
 */
VM_REGISTER_METACLASS(CVmObjArray)
