!!!=====================================================================!!!
!!! holder.h, by Jim Aikin.
!!!=====================================================================!!!

! holder.h is a modest extension that implements "indirect holders" such as
! tweezers, pliers, and tongs in Inform 6. This extension relies on Peer Schaefer's
! recept.h (a slightly revised version of which is bundled with this file).
!
! In general, the purpose of an indirect holder is to allow the player to grasp
! objects that, for one reason or another, can't be grasped directly. A forked
! stick with which you can pick up a poisonous snake would naturally be implemented
! as an indirect holder.
! 
! Because indirect holding is a bit tricky, using this extension is also a bit
! tricky, or at least not quite as transparent as you might prefer. Here are
! the necessary steps:
! 
! 1. Include both recept.h and holder.h near the top of your .inf file, after
! 	VerbLib.
! 
! 2. At the end of your .inf file, after "Include Grammar", include holderverbs.h.
! 	This file implements some necessary Actions. If these conflict with your own
!	"Extend first" declarations, you may have to resort to hacking grammar.h to
!	get the grammar lines in an appropriate order for your game.
! 
! 3. All objects in your game that can be carried around should be declared as
! 	belonging to Class Portable. This class provides defaults (which you'll
! 	no doubt want to override) for the properties size, volume, and weight.
! 
! 4. All supporters should be declared as class aSupporter, and all containers
! 	as class aContainer. These classes provide defaults for the properties
! 	capacity_size, capacity_volume, and capacity_weight, which again should
! 	be overridden with your own values. Details on these properties can be
! 	found in recept.h.
! 
! 5. Any indirect holder in your game should be declared as belonging to
! 	Class IndHolder.
! 
! 6. For each IndHolder object, you need to create a corresponding IndHolderHolder
! 	object. This is the "container" in which objects carried by the IndHolder
! 	are carried. The IndHolderHolder should have no short_name or vocabulary
! 	words, so that the player can't refer to it directly (and indeed won't know
! 	that it exists).
! 
! 7. Each IndHolder must be given a my_holder property that contains the name of
! 	its associated IndHolderHolder. Likewise, each IndHolderHolder object must
! 	have a my_parent property that contains the name of its associated IndHolder.
! 
! 8. Give the IndHolderHolder its own capacity_size, capacity_volume, and
! 	capacity_weight values, just as you would with any other container.
! 
! Some assumptions are built into the code, but these can be changed if desired.
! 
! First, the use of an indirect holder is assumed to require some dexterity on
! the part of the player character. Thus the player can be holding, at most,
! one other object while using the indirect holder. (If the indirect holder is
! not in use, it can be carried like any other object.) If the holder is large
! enough to require two hands (for example, fire tongs) this could be changed to
! allow no other objects to be held. We'll allow one other object by default,
! so that you can pick up a stamp in a pair of tweezers while holding a glassine
! envelope in your other hand, and then put the stamp in the envelope.
! 
! Second, if the indirect holder is dropped or put somewhere else, whatever is in
! it will fall, landing in the same location as the indirect holder. If there isn't
! room in the intended location, the game will print an error message and the
! action will abort.
! 
! Third, an indirect holder can only hold one thing at a time. If you need ice
! tongs that can hold several ice cubes, you'll need to change the code.
!
! Here's an example of a simple indirect holder. Note how the print_whats_held
! routine is called.
! 
! IndHolder tweezers "tweezers"
! 	with name 'tweezers',
! 	description [;
! 		print "They're an ordinary pair of tweezers";
! 		self.print_whats_held();
! 		".";
! 	],
! 	my_holder tweezers_holder,
! 	add_to_scope tweezers_holder,
! 	size 4,
! 	weight 3,
! 	volume 3,
! 	has pluralname;
! 
! IndHolderHolder tweezers_holder
! 	with
! 	my_parent tweezers,
! 	capacity_volume 3,
! 	capacity_weight 3,
! 	capacity_size 3;
! 
! Within the game, the grammar available for the indirect holder includes
! 'pick up x with (or in) holder' and 'take x with (or in) holder'. However,
! 'put x in holder' is not allowed, on the assumption that you'd need to be holding
! x in order to do that, and if you can hold x in the normal way, why do you need
! an indirect holder in the game? To release the object from the holder, simply
! 'drop x'.
! 
! Note that the new verb grammar (which makes use of the new attribute heldindirect)
! interferes with the 'drop all' command. As a result, I've recoded 'drop all'
! (and also 'put all down' and 'put down all') manually using a new DropAll action.
! This generates a <Drop obj> action for each child of the player that isn't worn, so any
! object that doesn't want to be dropped can intercept the command in its own before
! routine, as usual. If you're giving orders to an NPC, 'butler, drop all' will now
! fail to work until you edit the DropAllSub.
! 
! Please report any bugs you find to me at midiguru23@sbcglobal.net. Suggestions
! for enhancements to the functionality are also welcome. Thanks!


!===============================================================!
! New Attributes:
!===============================================================!
!
! Note: If you're running short of Attributes, you could recode holdingindirect as
! a true/false property. It's included here purely for the sake of symmetry. The
! heldindirect attribute is used by the new verb grammar, so it can't easily be
! dispensed with.

Attribute heldindirect;
Attribute holdingindirect;

!===============================================================!
! New Classes:
!===============================================================!

! Other code for portable objects could be added to the Portable class. Mainly this
! class exists to insure that everything we might want to move in or out of any type
! of receptacle has a size, volume, and weight that can be tested. All of your
! portable objects should be derived from the Portable class.

Class Portable
	with
	size 10,
	volume 10,
	weight 10;

! The ContainerSupporter class provides default capacities for containers and supporters
! in case they don't provide their own. It also has an error-checking routine called
! receive_indirect, which returns true if an error message has been printed indicating
! that the ContainerSupporter's capacity would be exceeded. This routine is called by
! the new verbsubs in the file HolderVerbs.h. The indirect holder will call the
! check_volume/size/weight/number routines itself (silently) if it's about to be
! deposited on the container/supporter, because we need to catch the situation where
! the thing being held (or the combination of the thing and the holder) would exceed
! the capacity of the container/supporter if the holder were deposited there.

Class ContainerSupporter class Receptacle
	with
	capacity_weight 100,
	capacity_volume 100,
	capacity_size 100,
	capacity_number 7,
	! The check_volume routine is called with a known volume amount and a silent_flag
	! for suppressing the printout. It returns true if the volume exceeds the capacity
	! of the container, and false if there's no problem. Ditto for check_weight,
	! check_size, and check_number. These routines are based on Peer Schaefer's code:
	check_volume [vol silent_flag s i;
		if (CapacityVolumeOf(self) == INFINITE_CAPACITY) rfalse;
		if (vol > CapacityVolumeOf (self) )
		{
			if (silent_flag) rtrue;
			print_ret (The) noun, " does not fit into ", (the) self, ".";
		}
		s = 0;
		objectloop (i in self)			! calculate the
			s = s + VolumeOf(i);		! volume of all
										! contents
		if ( (s + vol) > CapacityVolumeOf (self) )
		{
			if (silent_flag) rtrue;
			else
			{
				print "There isn't room ";
				if (self has supporter) print "on"; else print "in";
				print_ret " ", (the) self, ".";
			}
		}
		rfalse;
	],
	check_size [siz silent_flag;
		if (CapacitySizeOf(self) == INFINITE_CAPACITY) rfalse;
		if (siz > CapacitySizeOf (self))
		{
			if (silent_flag) rtrue;
			print_ret (The) noun, " is too large for ", (the) self, ".";
		}
		rfalse;
	],
	check_weight [wgt silent_flag s i;
		if (CapacityWeightOf(self) == INFINITE_CAPACITY) rfalse;
		if (wgt > CapacityWeightOf (self) )
		{
			if (silent_flag) rtrue;
			print_ret (The) noun, " is too heavy for ", (the) self, ".";
		}
		s = 0;
		objectloop (i in self)			! calculate the
			s = s + WeightOf(i);		! weight of all
										! contents
		if ( (s + wgt) > CapacityWeightOf (self) )
		{
			if (silent_flag) rtrue;
			else
			{
				print_ret (The) self, " would break under the weight.";
			}
		}
		rfalse;
	],
	check_number [num silent_flag s i;
		if (CapacityNumberOf(self) == INFINITE_CAPACITY) rfalse;
		s = 0;
		objectloop (i in self) s++;
		if ( (s+num) > CapacityNumberOf(self) )
		{
			if (silent_flag) rtrue;
			print_ret (The) self, " already contains too many objects.";
		}
		rfalse;
	],
	! The receive_indirect routine, which is adapted from recept.h, is called by
	! InsertIndirectSub, and will respond true if an error message has been printed,
	! or false if the object can be deposited. Here, we pass 'false' to the check
	! routines, because we DO want them to print out error messages.
    receive_indirect [x;

		! Check volume:
		x = VolumeOf(noun);
		if (self.check_volume(x, false)) rtrue;

		! Check size -- but size is basically a measure of an object's longest
		! dimension, and long things can flop over the edges of supporters, so
		! we only want to check the size if we're a container:
		if (self has container)
		{
			x = SizeOf(noun);
			if (self.check_size(x, false)) rtrue;
		}

		! Check weight:
		x = WeightOf(noun);
		if (self.check_weight(x, false)) rtrue;

		! Check number:
		if (self.check_number(1, false)) rtrue;
		rfalse;
	],
;

Class aContainer class ContainerSupporter,
	has container;
Class aSupporter class ContainerSupporter,
	has supporter;

! The IndHolder is not actually a container or supporter, because containment is handled
! by its invisible "holderholder". However, it's declared as a container so that Inform's
! normal 'put in' action will be routed to the holder's before Receive code, where we'll
! put a more useful error message than the confusing, "Those can't hold things."
! In addition, we'll give the class the properties of a
! recept.h container, just on the off-chance that someone might want to use them. It's
! declared as a Receptacle class from recept.h. Note that a Receptacle is neither a
! container nor a supporter by default; all it is is an embedded before Receive routine.

Class IndHolder class Portable Receptacle
	with
	print_whats_held [h o;
		h = self.my_holder;
		if (child(h) ~= nothing)
		{
			o = child(h);
			print_ret "(in which you're holding ", (a) o, ")";
		}
		else rfalse;
	],
	capacity_weight [;
		if (self.my_holder)
		{
!			if ((self.my_holder provides capacity_weight) == false) return INFINITE_CAPACITY;
			if (metaclass (self.my_holder.capacity_weight) == Routine)
			return self.my_holder.capacity_weight();
			else return self.my_holder.capacity_weight;
		}
	],
	capacity_size [;
		if (self.my_holder)
		{
!			if ((self.my_holder provides capacity_size) == false) return INFINITE_CAPACITY;
			if (metaclass (self.my_holder.capacity_size) == Routine)
			return self.my_holder.capacity_size();
			else return self.my_holder.capacity_size;
		}
	],
	capacity_volume [;
		if (self.my_holder)
		{
!			if ((self.my_holder provides capacity_volume) == false) return INFINITE_CAPACITY;
			if (metaclass (self.my_holder.capacity_volume) == Routine)
			return self.my_holder.capacity_volume();
			else return self.my_holder.capacity_volume;
		}
	],
	! Every IndHolder has a property that points to its invisible holderholder:
	my_holder 0,
	! The invent routine insures that when an inventory list is printed, whatever
	! is being held is mentioned:
	invent [c h;
		if (self has pluralname) print "some"; else print "a";
		print " ", (name) self;
		h = self.my_holder;
		c = child(h);
		if (c)
		{
			print " (in which you're holding ";
			if (c has pluralname) print "some"; else print "a";
			print " ", (name) c, ")";
		}
		rtrue;
	],
	! If you try to pick up more than one other thing while you have something
	! in the holder, the react_before will object:
	react_before [c h w o;
		if (self hasnt holdingindirect) rfalse;
		h = self.my_holder;
		c = child(h);
		! This extra test shouldn't be needed, unless the code is buggy:
		if (c == nothing) rfalse;
		if (action == ##Take)
		{
			c = children (player);
			if (c > 1)
			{
				w = 0;
				objectloop (o in player)
				{
					if (o has worn) w++;
				}
				! If you're holding something else besides the indirect holder
				! and one other thing:
				if (c - w > 1) "While you're holding something in ", (the) self, " and
					also something else, your hands are full.";
			}
		}
	],
	! The before Insert, PutOn routine tests the possibility that the player might
	! be about to put the holder on or in something else while it's holding something,
	! in which case the held thing might exceed the capacity of the receiving container
	! or holder. The before Receive routine intercepts 'put x in holder' commands.
	before [s v w n obj;
		Receive: "To put things in ", (the) self, ", you need to pick them up
			with ", (the) self, ".";
		Insert, PutOn:
			if (self hasnt holdingindirect) rfalse;
			if ((second hasnt supporter) && (second hasnt container))
			"Other things can't be put on or in ", (the) second, ".";

			! Test whether the second can hold the combined weight and volume and
			! the size. There should always be an obj, but we're going to be safe
			! and test that as we go along.
			obj = child(self.my_holder);
			! Check silently, because the routine wants to print the name of the
			! noun, and here we're checking the combined weight of the holder (which
			! is, at the moment, the noun) PLUS what it's holding.
			! We'll start with the volume, since that's the problem that's likely
			! to be the most visible to the player character:

			if (obj) v = VolumeOf(obj) + VolumeOf(self);
				else v = VolumeOf(self);
			if (second.check_volume(v, true))
			{
				print "There isn't room ";
				if (second has container) print "in";
					else print "on";
				" the ", (name) second, ".";
			}
			! Next, the number of things the intended recipient can contain:
			if (obj) n = 2; else n = 1;
			if (second.check_number(n, true))
			{
				print "You can't put that many things ";
				if (second has container) print "in";
					else print "on";
				" the ", (name) second, ".";
			}
			! Next, the weight, because we don't want to accidentally allow a
			! cement block to be put on a spindy end table:
			if (obj) w = WeightOf(obj) + WeightOf(self);
				else w = WeightOf(self);
			if (second.check_weight(w, true)) "The ", (name) second, " won't support
				the combined weight.";
			! And finally, let's check the size (but only if the intended recipient
			! is a container, because long things can stick out over the edges of a
			! supporter). For this to work, we want to know the larger of the
			! two sizes, since size is basically a lengthwise measurement of the
			! longest thing, and is not additive:
			if (second has container)
			{
				if (obj) s = SizeOf(obj); else s = 0;
				if (SizeOf(self) > s) s = SizeOf(self);
				if (second.check_size(s, true))
					"There isn't room in ", (the) second, ".";
			}
			rfalse;
	],
	after [c h;
		Drop, Insert, PutOn:
			h = self.my_holder;
			c = child(h);
			if (c)
			{
				! The held object will land in the same place as the holder -- which
				! might be nowhere, so we have to test for that:
				if (parent(self)) move c to parent(self);
					else remove c;
				give c ~heldindirect;
				give self ~holdingindirect;
				print "As you let go of ", (the) self, ", ", (the) c;
				if (c has pluralname) print " fall"; else print " falls";
				" free.";
			}
	],
	has container
;

Class IndHolderHolder class Receptacle,
	! Every invisible holderholder has a property that points to its "parent" (not
	! a parent in the usual object-tree sense). It also has default capacities, in
	! case we forget to override them:
	with my_parent 0,
	capacity_weight 20,
	capacity_volume 20,
	capacity_size 20,
	capacity_number 1,		! Not actually needed, as the IndHolder only allows it to hold 1...
	has container open;

! End.