! ==============================================================================
! PARSER for Triform 1.2a
!
! Nearly all of this file is copyrighted by Graham Nelson, 1993-2004, or David
! Griffith, 2012-2014, and licensed under the traditional Inform license:

! Copyright on Inform, the program and its source code, its example
! games and documentation (including this book) is retained by Graham
! Nelson, who asserts the moral right to be identified as the author
! under the Copyrights, Designs and Patents Act 1988.  Having said
! this, I am happy for it to be freely distributed to anybody who wants
! a copy, provided that: (a) distributed copies are not substantially
! different from those archived by the author, (b) this and other
! copyright messages are always retained in full, and (c) no profit is
! involved.  (Exceptions to these rules must be negotiated directly
! with the author.)  However, a story file produced with the Inform
! compiler (and libraries) then belongs to its author, and may be sold
! for profit if desired, provided that its game banner contains the
! information that it was compiled by Inform, and the Inform version
! number.

! All code contributed by myself and not appearing in any of the standard Inform
! libraries is public domain.
! ==============================================================================

System_file;

#Ifndef VN_1633;
Message fatalerror "*** Triform 1.2a requires Inform v6.33 or later ***";
#Endif; ! VN_

#Ifdef TARGET_ZCODE; #IfV3;
#Message fatalerror "*** Triform 1.2a is not compatible with Z-code V3. ***";
#Endif; #Endif;

#IFDEF USE_LIBRARY_EXTENSIONS;
#Message " Compiling with library extensions... ";
#ENDIF;

#IFDEF TRACK_LOCATIONS;
#Message " Compiling with location tracking... ";
#ENDIF;

#IFDEF COMPLEXCLOTHING;
#Message " Compiling with complex clothing... ";
#ENDIF;

#IFDEF GRADUAL_LIGHT;
#Message " Compiling with gradual light options... ";
#ENDIF;

#IFDEF NODARKNESS;
#Message " Compiling with no darkness... ";
#ENDIF;

#IFDEF NOFIRSTLOOK;
#Message " Compiling with no initial #Look action... ";
#ENDIF;

#IFDEF SERIAL_COMMAS;
#Message " Compiling with serial (Oxford) commas... ";
#ENDIF;

#IFDEF NO_MENUS;
#Message " Compiling without default menu options... ";
#ENDIF;

Constant LibSerial      "150831";
Constant LibRelease     "v1.2a";
Constant Grammar__Version 2;

Default COMMENT_CHARACTER '*';

#Ifdef INFIX;
Default DEBUG 0;
#Endif; ! INFIX

#Ifndef WORDSIZE;                   ! compiling with Z-code only compiler
Constant TARGET_ZCODE;
Constant WORDSIZE 2;
#Endif; ! WORDSIZE

#Ifdef TARGET_ZCODE;                ! offsets into Z-machine header

Constant HDR_ZCODEVERSION  $00;     ! byte
Constant HDR_TERPFLAGS     $01;     ! byte
Constant HDR_GAMERELEASE   $02;     ! word
Constant HDR_HIGHMEMORY    $04;     ! word
Constant HDR_INITIALPC     $06;     ! word
Constant HDR_DICTIONARY    $08;     ! word
Constant HDR_OBJECTS       $0A;     ! word
Constant HDR_GLOBALS       $0C;     ! word
Constant HDR_STATICMEMORY  $0E;     ! word
Constant HDR_GAMEFLAGS     $10;     ! word
Constant HDR_GAMESERIAL    $12;     ! six ASCII characters
Constant HDR_ABBREVIATIONS $18;     ! word
Constant HDR_FILELENGTH    $1A;     ! word
Constant HDR_CHECKSUM      $1C;     ! word
Constant HDR_TERPNUMBER    $1E;     ! byte
Constant HDR_TERPVERSION   $1F;     ! byte
Constant HDR_SCREENHLINES  $20;     ! byte
Constant HDR_SCREENWCHARS  $21;     ! byte
Constant HDR_SCREENWUNITS  $22;     ! word
Constant HDR_SCREENHUNITS  $24;     ! word
Constant HDR_FONTWUNITS    $26;     ! byte
Constant HDR_FONTHUNITS    $27;     ! byte
Constant HDR_ROUTINEOFFSET $28;     ! word
Constant HDR_STRINGOFFSET  $2A;     ! word
Constant HDR_BGCOLOUR      $2C;     ! byte
Constant HDR_FGCOLOUR      $2D;     ! byte
Constant HDR_TERMCHARS     $2E;     ! word
Constant HDR_PIXELSTO3     $30;     ! word
Constant HDR_TERPSTANDARD  $32;     ! two bytes
Constant HDR_ALPHABET      $34;     ! word
Constant HDR_EXTENSION     $36;     ! word
Constant HDR_UNUSED        $38;     ! two words
Constant HDR_INFORMVERSION $3C;     ! four ASCII characters

#Ifnot; ! TARGET_GLULX              ! offsets into Glulx header and start of ROM

Constant HDR_MAGICNUMBER   $00;     ! long word
Constant HDR_GLULXVERSION  $04;     ! long word
Constant HDR_RAMSTART      $08;     ! long word
Constant HDR_EXTSTART      $0C;     ! long word
Constant HDR_ENDMEM        $10;     ! long word
Constant HDR_STACKSIZE     $14;     ! long word
Constant HDR_STARTFUNC     $18;     ! long word
Constant HDR_DECODINGTBL   $1C;     ! long word
Constant HDR_CHECKSUM      $20;     ! long word
Constant ROM_INFO          $24;     ! four ASCII characters
Constant ROM_MEMORYLAYOUT  $28;     ! long word
Constant ROM_INFORMVERSION $2C;     ! four ASCII characters
Constant ROM_COMPVERSION   $30;     ! four ASCII characters
Constant ROM_GAMERELEASE   $34;     ! short word
Constant ROM_GAMESERIAL    $36;     ! six ASCII characters

#Endif; ! TARGET_

! ------------------------------------------------------------------------------
!   Some VM-specific constants.
!   (WORDSIZE and TARGET_XXX are defined by the compiler.)
! ------------------------------------------------------------------------------

#Ifdef TARGET_ZCODE;
Constant NULL         = $ffff;
Constant WORD_HIGHBIT = $8000;
#Ifnot; ! TARGET_GLULX
Constant NULL         = $ffffffff;
Constant WORD_HIGHBIT = $80000000;
#Endif; ! TARGET_

! ------------------------------------------------------------------------------
!   The common attributes and properties.
! ------------------------------------------------------------------------------

Attribute animate;
#Ifndef COMPLEXCLOTHING;
Attribute clothing;
Attribute worn;
#Endif;
Attribute concealed;
Attribute edible;
Attribute moved;
Attribute on;
Attribute closed;
Attribute openable;
Attribute proper;
Attribute scenery;
Attribute switchable;
Attribute static;
Attribute talkable;
Attribute transparent;
Attribute workflag;

Attribute male;
Attribute female;
Attribute neuter;
Attribute pluralname;

#Ifndef NO_DARKNESS;
Attribute light;
#Ifdef GRADUAL_LIGHT;
Attribute quasilight;
Attribute demilight;
Attribute semilight;
Attribute semitransparent;
#Endif; ! gradual light
#Endif; ! darkness

Attribute anatomy;

Attribute activedaemon;		! Stolen from Platypus
Attribute activetimer;		!
Attribute topic;		!


! ------------------------------------------------------------------------------

Property additive before NULL;
Property additive middle NULL;
Property additive after NULL;
Property additive final NULL;
Property additive life NULL;

Property n_to;
Property s_to;
Property e_to;
Property w_to;
Property ne_to;
Property nw_to;
Property se_to;
Property sw_to;
Property u_to;
Property d_to;
Property in_to;
Property out_to;

Property switch_for;
Property switched_by;
Property switched_desc;
Property worn_desc;
Property invent;
Property plural;
Property add_to_scope;
Property list_together;
Property react_before;
Property react_middle;
Property react_after;
Property react_final;
Property grammar NULL;
Property additive orders;

Property description;
Property additive describe NULL;
Property article "a";

Property cant_go;

Property found_in;          ! For fiddly reasons this can't alias

Property time_left;
Property number;
Property additive time_out NULL;
Property daemon;
Property additive each_turn NULL;

Property short_name 0;
Property short_name_indef 0;
Property parse_name 0;

Property additive adjective 0;

Property articles;
Property inside_description;
Property entered_desc;
Property light_desc;
Property open_desc;
Property closed_desc;

Global failed_action;
Global original_noun;
Global grammar_line;		! Also an idea stolen from Platypus.

! ==============================================================================

Fake_Action Receive;
Fake_Action ThrownAt;
Fake_Action Order;
Fake_Action TheSame;
Fake_Action PluralFound;
Fake_Action ListMiscellany;
Fake_Action Miscellany;
Fake_Action Prompt;
Fake_Action NotUnderstood;

Fake_Action LetGo;
Fake_Action GoneFrom;

#Ifdef NO_PLACES;
Fake_Action Places;
Fake_Action Objects;
#Endif; ! NO_PLACES

! ------------------------------------------------------------------------------

[ Main; InformLibrary.play(); ];

! ------------------------------------------------------------------------------

! ------------------------------------------------------------------------------
!   Global variables and their associated Constant and Array declarations
! ------------------------------------------------------------------------------

Global sline1;                      ! Must be second
Global sline2;                      ! Must be third
                                    ! (for status line display)
! ------------------------------------------------------------------------------
!   Z-Machine and interpreter issues
! ------------------------------------------------------------------------------

#Ifdef TARGET_ZCODE;
Global top_object;                  ! Largest valid number of any tree object
! ### these globals are not meaningful... well, maybe standard_interpreter,
! but I'll decide that later (AP).
Constant INDIV_PROP_START 64;       ! Equivalent of a Glulx constant

#Endif; ! TARGET_ZCODE

Global standard_interpreter;        ! The version number of the Z-Machine Standard which the
                                    ! interpreter claims to support, in form (upper byte).(lower)

Global undo_flag;                   ! Can the interpreter provide "undo"?
Global just_undone;                 ! Can't have two successive UNDOs

Global transcript_mode;             ! true when game scripting is on

#Ifdef TARGET_ZCODE;
Global xcommsdir;                   ! true if command recording is on
#Endif; ! TARGET_ZCODE

#Ifdef TARGET_GLULX;
Constant GG_MAINWIN_ROCK     201;
Constant GG_STATUSWIN_ROCK   202;
Constant GG_QUOTEWIN_ROCK    203;
Constant GG_SAVESTR_ROCK     301;
Constant GG_SCRIPTSTR_ROCK   302;
Constant GG_COMMANDWSTR_ROCK 303;
Constant GG_COMMANDRSTR_ROCK 304;
Constant GG_SCRIPTFREF_ROCK  401;
Array gg_event --> 4;
#Ifdef VN_1630;
Array gg_arguments buffer 28;
#Ifnot;
Array gg_arguments --> 8;
#Endif; ! VN_
Global gg_mainwin = 0;
Global gg_statuswin = 0;
Global gg_quotewin = 0;
Global gg_scriptfref = 0;
Global gg_scriptstr = 0;
Global gg_savestr = 0;
Global gg_commandstr = 0;
Global gg_command_reading = 0;      ! true if gg_commandstr is being replayed
#Endif; ! TARGET_GLULX

Global gg_statuswin_cursize = 0;
Global gg_statuswin_size = 1;

! ------------------------------------------------------------------------------
!   Time and score
! ------------------------------------------------------------------------------

#Ifndef sys_statusline_flag;
Global sys_statusline_flag = 0;     ! non-zero if status line displays time
#Endif;
#Ifndef START_MOVE;
Constant START_MOVE 0;              ! Traditionally 0 for Infocom, 1 for Inform
#Endif;
Global turns = START_MOVE;          ! Number of turns of play so far
Global the_time = NULL;             ! Current time (in minutes since midnight)
Global time_rate = 1;               ! How often time is updated
Global time_step;                   ! By how much

#Ifndef MAX_TIMERS;
Constant MAX_TIMERS  32;            ! Max number timers/daemons active at once
#Endif; ! MAX_TIMERS
Array  the_timers  --> MAX_TIMERS;
Global active_timers;               ! Number of timers/daemons actives

#Ifndef NO_SCORE;
Global score;                       ! The current score
Global last_score;                  ! Score last turn (for testing for changes)
Global notify_mode = true;          ! Score notification
Global places_score;                ! Contribution to score made by visiting
Global things_score;                ! Contribution made by acquisition
#Endif;

! ------------------------------------------------------------------------------
!   The player
! ------------------------------------------------------------------------------

Global player;                      ! Which object the human is playing through
Global gameover;                    ! Normally 0, or false; 1 for dead; 2 for victorious, and
                                    ! higher numbers represent exotic forms of death

! ------------------------------------------------------------------------------
!   Light and room descriptions
! ------------------------------------------------------------------------------

Global lookmode = 2;                ! 1=standard, 2=verbose, 3=brief room descs
Global normalmode = 2;		    ! 1=standard, 2=verbose, 3=brief room descs
Global print_player_flag;           ! If set, print something like "(as Fred)" in room descriptions,
                                    ! to reveal whom the human is playing through
Global lastdesc;                    ! Value of location at the time of the most recent room description
                                    ! printed out

! ------------------------------------------------------------------------------
!   List writing  (style bits are defined as Constants in "3Verblib.h")
! ------------------------------------------------------------------------------

Global c_style;                     ! Current list-writer style
Global lt_value;                    ! Common value of list_together
Global listing_together;            ! Object number of one member of a group being listed together
Global listing_size;                ! Size of such a group
Global wlf_indent;                  ! Current level of indentation printed by WriteListFrom()

Global inventory_stage = 1;         ! 1 or 2 according to the context in which "invent" routines
                                    ! of objects are called
Global inventory_style;             ! List-writer style currently used while printing inventories

! ------------------------------------------------------------------------------
!   Menus and printing
! ------------------------------------------------------------------------------

#Ifndef NO_MENUS;
Global pretty_flag = true;          ! Use character graphics, or plain text?
Global menu_nesting;                ! Level of nesting (0 = root menu)
Global menu_item;                   ! These are used in communicating
Global item_width = 8;              ! with the menu-creating routines
Global item_name = "---";
#Endif;
Global lm_n;                        ! Parameters used by LibraryMessages
Global lm_o;                        ! mechanism

#Ifdef DEBUG;
Global debug_flag;                  ! Bitmap of flags for tracing actions,
                                    ! calls to object routines, etc.
Global x_scope_count;               ! Used in printing a list of everything
#Endif; ! DEBUG                     ! in scope

! five for colour control
! see http://www.inform-fiction.org/patches/L61007.html
Global clr_fg = 1;                  ! foreground colour
Global clr_bg = 1;                  ! background colour
Global clr_fgstatus = 1;            ! foreground colour of statusline
Global clr_bgstatus = 1;            ! background colour of statusline
Global clr_on;                      ! has colour been enabled by the player?
Global statuswin_current;           ! if writing to top window

Constant CLR_DEFAULT 1;
Constant CLR_BLACK   2;
Constant CLR_RED     3;
Constant CLR_GREEN   4;
Constant CLR_YELLOW  5;
Constant CLR_BLUE    6;
Constant CLR_MAGENTA 7;
Constant CLR_CYAN    8;
Constant CLR_WHITE   9;
Constant CLR_PURPLE  7;
Constant CLR_AZURE   8;

Constant WIN_ALL     0;
Constant WIN_STATUS  1;
Constant WIN_MAIN    2;

! ------------------------------------------------------------------------------
!   Action processing
! ------------------------------------------------------------------------------

Global action;			! Action currently being asked to perform
Global inp1;			! 0 (nothing), 1 (number) or first noun
Global inp2;			! 0 (nothing), 1 (number) or second noun
Global noun;			! First noun or numerical value
Global second;			! Second noun or numerical value

Global keep_silent;		! If true, attempt to perform the action silently (e.g. for
				! implicit takes, implicit opening of unlocked doors)

Global reason_code;		! Reason for calling a "life" rule
				! (an action or fake such as ##Kiss)

Global receive_action;		! Either ##PutOn, ##Insert, ##PutBehind, or ##PutUnder,
				! whichever action is being tried when an object's
				! "before" rule is checking "Receive"

Global action_mode;		! 1 = player performing an implicit action
				! 2 = NPC acting after being given an order
				! 3 = NPC performing an implicit action on orders
				! 4 = NPC independently performing an implicit action
				! 0 = player or NPC acting independently

Global lastspoken;		! Keeps track of the last person spoken to.

Global lastconsulted;		! Keeps track of the last thing consulted about something.

Global arrived;			! Did the player just arrive somewhere via a GO action?

Global reparsing;		! True when carrying out a reparse request.

Array cant_touch_reason --> 0 0;
				! The reason something can't be reached. The first entry
				! is the object in the way, and the second entry holds
				! one of the following values:
				! 1 = the object must be opened
				! 2 = the actor must exit the object
				! 3 = the actor must disrobe the object
				! 4 = it is being held by another actor

Global count;			! For keeping track of multiple-object actions.

! ------------------------------------------------------------------------------
!   Parser variables: first, for communication to the parser
! ------------------------------------------------------------------------------

Global parser_trace = 0;            ! Set this to 1 to make the parser trace tokens and lines
Global parser_action;               ! For the use of the parser when calling
Global parser_one;                  ! user-supplied routines
Global parser_two;                  !
Array  inputobjs       --> 16;      ! For parser to write its results in
Global parser_inflection;           ! A property (usually "name") to find object names in
Global parser_inflection_func;	    ! Programmer sets this to true when parser_inflection
				    ! is a function

! ------------------------------------------------------------------------------
!   Parser output
! ------------------------------------------------------------------------------

Global actor;                       ! Person asked to do something
Global meta;                        ! Verb is a meta-command (such as "save")
#Ifdef INFIX;
Global infix_verb;                  ! Verb is an Infix command
#Endif;

Array  multiple_object --> 64;      ! List of multiple parameters
Global multiflag = false;           ! Multiple-object flag passed to actions
                                    ! Also used to prevent misleading MULTI_PE
Global multiflag2 = false;
Global toomany_flag;                ! Flag for "multiple match too large"
                                    ! (e.g. if "take all" took over 100 things)
Global successful_action;

Global special_word;                ! Dictionary address for "special" token
Global special_number;              ! Number typed for "special" token
Global parsed_number;               ! For user-supplied parsing routines
Global consult_from;                ! Word that a "consult" topic starts on
Global consult_words;               ! ...and number of words in topic

Global tense = 1;                   ! = 1 Present ("You see...")
                                    ! = 2 Past ("You could see...")

Global caps_mode;
Global behindorbeneath;

! ------------------------------------------------------------------------------
!   Implicit taking
! ------------------------------------------------------------------------------

Global notheld_mode;                ! To do with implicit taking
Global onotheld_mode;               !     "old copy of notheld_mode", ditto
Global not_holding;                 ! Object to be automatically taken as an
                                    ! implicit command
Array  kept_results --> 16;         ! Delayed command (while the take happens)

! ------------------------------------------------------------------------------
!   Error numbers when parsing a grammar line
! ------------------------------------------------------------------------------

Global etype;                       ! Error number on current line
Global best_etype;                  ! Preferred error number so far
Global nextbest_etype;              ! Preferred one, if ASKSCOPE_PE disallowed

Constant STUCK_PE     = 1;
Constant UPTO_PE      = 2;
Constant NUMBER_PE    = 3;
Constant CANTSEE_PE   = 4;
Constant NOTHELD_PE   = 6;
Constant MULTI_PE     = 7;
Constant MMULTI_PE    = 8;
Constant VAGUE_PE     = 9;
Constant EXCEPT_PE    = 10;
Constant ANIMA_PE     = 11;
Constant VERB_PE      = 12;
Constant SCENERY_PE   = 13;
Constant ITGONE_PE    = 14;
Constant JUNKAFTER_PE = 15;
Constant TOOFEW_PE    = 16;
Constant NOTHING_PE   = 17;
Constant ASKSCOPE_PE  = 18;

! ------------------------------------------------------------------------------
!   Pattern-matching against a single grammar line
! ------------------------------------------------------------------------------

Array pattern --> 32;               ! For the current pattern match
Global pcount;                      ! and a marker within it
Array pattern2 --> 32;              ! And another, which stores the best match
Global pcount2;                     ! so far
Constant PATTERN_NULL = $ffff;      ! Entry for a token producing no text

Array  line_ttype-->32;             ! For storing an analysed grammar line
Array  line_tdata-->32;
Array  line_token-->32;

Global parameters;                  ! Parameters (objects) entered so far
Global nsns;                        ! Number of special_numbers entered so far
Global special_number1;             ! First number, if one was typed
Global special_number2;             ! Second number, if two were typed

! ------------------------------------------------------------------------------
!   Inferences and looking ahead
! ------------------------------------------------------------------------------

Global params_wanted;               ! Number of parameters needed (which may change in parsing)

Global inferfrom;                   ! The point from which the rest of the command must be inferred
Global inferword;                   ! And the preposition inferred
Global dont_infer;                  ! Another dull flag

Global action_to_be;                ! (If the current line were accepted.)
Global action_reversed;             ! (Parameters would be reversed in order.)
Global advance_warning;             ! What a later-named thing will be

! ------------------------------------------------------------------------------
!   At the level of individual tokens now
! ------------------------------------------------------------------------------

Global found_ttype;                 ! Used to break up tokens into type
Global found_tdata;                 ! and data (by AnalyseToken)
Global token_filter;                ! For noun filtering by user routines
Global token_allows_multiple;

Global length_of_noun;              ! Set by NounDomain to no. of words in noun

#Ifdef TARGET_ZCODE;
Constant REPARSE_CODE = 10000;      ! Signals "reparse the text" as a reply from NounDomain
#Ifnot; ! TARGET_GLULX
Constant REPARSE_CODE = $40000000;  ! The parser rather gunkily adds addresses to REPARSE_CODE for
                                    ! some purposes. And expects the result to be greater than
                                    ! REPARSE_CODE (signed comparison). So Glulx Inform is limited
                                    ! to a single gigabyte of storage, for the moment.
#Endif; ! TARGET_

Global lookahead;                   ! The token after the one now being matched

Global multi_mode;                  ! Multiple mode
Global multi_wanted;                ! Number of things needed in multitude
Global multi_had;                   ! Number of things actually found
Global multi_context;               ! What token the multi-obj was accepted for

Global indef_mode;                  ! "Indefinite" mode - ie, "take a brick"
                                    ! is in this mode
Global indef_specific;
Global indef_type;                  ! Bit-map holding types of specification
Global indef_wanted;                ! Number of items wanted (100 for all)
Global first_indef_wanted;
Global indef_guess_p;               ! Plural-guessing flag
Global indef_owner;                 ! Object which must hold these items
Global indef_cases;                 ! Possible gender and numbers of them
Global indef_possambig;             ! Has a possibly dangerous assumption
                                    ! been made about meaning of a descriptor?
Global indef_nspec_at;              ! Word at which a number like "two" was parsed
                                    ! (for backtracking)
Global allow_plurals;               ! Whether plurals presently allowed or not

Global take_all_rule;               ! Slightly different rules apply to "take all" than other uses
                                    ! of multiple objects, to make adjudication produce more
                                    ! useful results
                                    ! (Not a flag: possible values 0, 1, 2)

Global dict_flags_of_noun;          ! Of the noun currently being parsed
                                    ! (a bitmap in #dict_par1 format)

Constant DICT_VERB $01;
Constant DICT_META $02;
Constant DICT_PLUR $04;
Constant DICT_PREP $08;
Constant DICT_X654 $70;
Constant DICT_NOUN $80;

Global pronoun_word;                ! Records which pronoun ("it", "them", ...) caused an error
Global pronoun_obj;                 ! And what obj it was thought to refer to
Global pronoun__word;               ! Saved value
Global pronoun__obj;                ! Saved value

! ------------------------------------------------------------------------------
!   Searching through scope and parsing "scope=Routine" grammar tokens
! ------------------------------------------------------------------------------

Constant PARSING_REASON       = 0;  ! Possible reasons for searching scope
Constant TALKING_REASON       = 1;
Constant EACH_TURN_REASON     = 2;
Constant REACT_BEFORE_REASON  = 3;
Constant REACT_AFTER_REASON   = 4;
Constant LOOPOVERSCOPE_REASON = 5;
Constant TESTSCOPE_REASON     = 6;
Constant REACT_MIDDLE_REASON  = 7;
Constant REACT_FINAL_REASON   = 8;

Global scope_reason = PARSING_REASON; ! Current reason for searching scope

Global scope_token;                 ! For "scope=Routine" grammar tokens
Global scope_error;
Global scope_stage;                 ! 1, 2 then 3

Global ats_flag = 0;                ! For AddToScope routines
Global ats_hls;                     !

Global placed_in_flag;              ! To do with PlaceInScope

! ------------------------------------------------------------------------------
!   The match list of candidate objects for a given token
! ------------------------------------------------------------------------------

Constant MATCH_LIST_SIZE = 128;
Array  match_list    --> 64;        ! An array of matched objects so far
Array  match_classes --> 64;        ! An array of equivalence classes for them
Array  match_scores --> 64;         ! An array of match scores for them
Global number_matched;              ! How many items in it?  (0 means none)
Global number_of_classes;           ! How many equivalence classes?
Global match_length;                ! How many words long are these matches?
Global saved_ml;                    ! Used for performing OOPS
Global match_from;                  ! At what word of the input do they begin?
Global bestguess_score;             ! What did the best-guess object score?

! ------------------------------------------------------------------------------
!   Low level textual manipulation
! ------------------------------------------------------------------------------

#Ifdef TARGET_ZCODE;

! 'buffer' holds the input line as typed by the player
!
!       buffer->0                     INPUT_BUFFER_LEN - WORDSIZE
!       buffer->1                     Number of characters input by player
!       buffer->2 ... buffer->121     The actual characters
!       buffer->122                   Spare byte to allow for 'terp bugs
!
! 'parse' holds the result of parsing that line into dictionary words
!
!       parse->0                      MAX_BUFFER_WORDS
!       parse->1                      Number of words input by player
!
!       parse-->1                     Dictionary addr of first input word
!       parse->4                      Number of characters in the word
!       parse->5                      Start position in 'buffer' of the word
!
!       parse-->3  parse->8,9         Same data for second input word
!       ...
!       parse-->29 parse->60,61       Same data for MAX_BUFFER_WORDS input word
!       parse->62,63,64               Spare bytes (not sure why)


Constant INPUT_BUFFER_LEN = WORDSIZE + 120;  ! 120 is limit on input chars
Constant MAX_BUFFER_WORDS = 15;              ! Limit on input words

Array  buffer  -> INPUT_BUFFER_LEN + 1;      ! For main line of input
Array  buffer2 -> INPUT_BUFFER_LEN + 1;      ! For supplementary questions
Array  buffer3 -> INPUT_BUFFER_LEN + 1;      ! Retaining input for "AGAIN"

#Ifdef VN_1630;
Array  parse   buffer (MAX_BUFFER_WORDS * 4) + 3;   ! Parsed data from 'buffer'
Array  parse2  buffer (MAX_BUFFER_WORDS * 4) + 3;   ! Parsed data from 'buffer2'
#Ifnot;
Array  parse   -> 2 + (MAX_BUFFER_WORDS * 4) + 3;
Array  parse2  -> 2 + (MAX_BUFFER_WORDS * 4) + 3;
#Endif; ! VN_

#Ifnot; ! TARGET_GLULX

! 'buffer' holds the input line as typed by the player
!
!       buffer-->0                    Number of characters input by player
!       buffer->4 ... buffer->259     The actual characters
!
! 'parse' holds the result of parsing that line into dictionary words
!
!       parse-->0                     Number of words input by player
!
!       parse-->1                     Dictionary addr of first input word
!       parse-->2                     Number of characters in the word
!       parse-->3                     Start position in 'buffer' of the word
!
!       parse-->4,5,6                 Same data for second input word
!       ...
!       parse-->58,59,60              Same data for MAX_BUFFER_WORDS input word

Constant INPUT_BUFFER_LEN = WORDSIZE + 256;    ! 256 is limit on input chars
Constant MAX_BUFFER_WORDS = 20;                ! Limit on input words

#Ifdef VN_1630;
Array  buffer  buffer (INPUT_BUFFER_LEN-WORDSIZE);  ! For main line of input
Array  buffer2 buffer (INPUT_BUFFER_LEN-WORDSIZE);  ! For supplementary questions
Array  buffer3 buffer (INPUT_BUFFER_LEN-WORDSIZE);  ! Retaining input for "AGAIN"
#Ifnot;
Array  buffer  -> INPUT_BUFFER_LEN;
Array  buffer2 -> INPUT_BUFFER_LEN;
Array  buffer3 -> INPUT_BUFFER_LEN;
#Endif; ! VN_
Array  parse   --> 1 + (MAX_BUFFER_WORDS * 3);      ! Parsed data from 'buffer'
Array  parse2  --> 1 + (MAX_BUFFER_WORDS * 3);      ! Parsed data from 'buffer2'

#Endif; ! TARGET_

Constant comma_word = 'comma,';     ! An "untypeable word" used to substitute
                                    ! for commas in parse buffers

Global wn;                          ! Word number within "parse" (from 1)
Global num_words;                   ! Number of words typed
Global num_desc;                    ! Number of descriptors typed
Global verb_word;                   ! Verb word (eg, take in "take all" or
                                    ! "dwarf, take all") - address in dict
Global verb_wordnum;                ! its number in typing order (eg, 1 or 3)
Global usual_grammar_after;         ! Point from which usual grammar is parsed (it may vary from the
                                    ! above if user's routines match multi-word verbs)

Global oops_from;                   ! The "first mistake" word number
Global saved_oops;                  ! Used in working this out
Constant OOPS_WORKSPACE_LEN 64;     ! Used temporarily by "oops" routine
Array  oops_workspace -> OOPS_WORKSPACE_LEN;

Global held_back_mode;              ! Flag: is there some input from last time
Global hb_wn;                       ! left over?  (And a save value for wn.)
                                    ! (Used for full stops and "then".)

! ----------------------------------------------------------------------------

Array PowersOfTwo_TB                ! Used in converting case numbers to case bitmaps
  --> $$100000000000
      $$010000000000
      $$001000000000
      $$000100000000
      $$000010000000
      $$000001000000
      $$000000100000
      $$000000010000
      $$000000001000
      $$000000000100
      $$000000000010
      $$000000000001;

! ----------------------------------------------------------------------------
!  Constants, and one variable, needed for the language definition file
! ----------------------------------------------------------------------------

Constant POSSESS_PK  = $100;
Constant DEFART_PK   = $101;
Constant INDEFART_PK = $102;
Global short_name_case;

Global dict_start;
Global dict_entry_size;
Global dict_end;

! ----------------------------------------------------------------------------

Include "language__";               ! The natural language definition, whose filename is taken from
                                    ! the ICL language_name variable

! ----------------------------------------------------------------------------

#Ifndef LanguageCases;
Constant LanguageCases = 1;
#Endif; ! LanguageCases

! ------------------------------------------------------------------------------
! The building block classes of the library.
! ----------------------------------------------------------------------------

Class Thing
with	volume 1,
	weight 1,
	insideofparent false,
	ontopofparent false,
	behindparent false,
	beneathparent false,
	attachedtoparent false,
	location 0,
#IFNDEF NO_SCORE;
	points 0,
#ENDIF;
#IFDEF TRACK_LOCATIONS;
	last_seen 0 0 0 0,
	last_seen_npc 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0,
	last_known 0,
	last_known_npc 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0,
#ENDIF;
	visibility_levels,
	visibility_ceiling;

Class Room
with	travel_name NULL,
	visited false,
	initial NULL,
	visibility_ceiling,
#IFNDEF NO_SCORE;
	points 0,
#ENDIF;
	when_dark  [; return L__M(##Miscellany, 17); ],
has	scenery;

Class Person class Thing,
with	capacity 100,
	indirect_weight 0,
	last_action 0 0 0,
	last_direction 0,
	npc_number 0,
	old_loc 0,
	posture 1,
	short_name NULL,
	weight 150,
	before_implicit NULL,
has	animate transparent proper;

Class Container class Thing,
with	attached_capacity 0,
	behind_capacity 0,
	beneath_capacity 0,
	inside_capacity 0,
	ontop_capacity 0,
	hidden_contents 2 1 1,
	preferred_posture 2,
	sack_object 0,
	indirect_weight 0,
	smellproof true,
	soundproof false,
	weight 1,
	when_dark  [; return L__M(##Miscellany, 17); ];

Class Door
with	door_dir
	[ p x;
		if (~~p) x = actor.location; else x = p.location;
		if (x provides n_to && x.n_to == self) return n_to;
		if (x provides s_to && x.s_to == self) return s_to;
		if (x provides e_to && x.e_to == self) return e_to;
		if (x provides w_to && x.w_to == self) return w_to;
		if (x provides ne_to && x.ne_to == self) return ne_to;
		if (x provides se_to && x.se_to == self) return se_to;
		if (x provides nw_to && x.nw_to == self) return nw_to;
		if (x provides sw_to && x.sw_to == self) return sw_to;
		if (x provides u_to && x.u_to == self) return u_to;
		if (x provides d_to && x.d_to == self) return d_to;
		return self;
	],
	door_to
	[ p;
		if (~~p) p = actor;
		if ((self.&found_in-->0) == p.location) return self.&found_in-->1;
		return self.&found_in-->0;
	],
	visibility_ceiling,
	found_in 0 0,
 has	openable static scenery;

#Ifdef COMPLEXCLOTHING;
Class Wearable class Thing,
with	name "clothing/p" "clothes/p",
	area_covered 0,
	clothing_size 0,
	clothing_type 1,
	covered_by 0,
	covering 0,
	gendered false,
	bulky 0,
	manipulable_when_locked 0,
	layer 0,
	max_layer 5,
	min_layer 1,
	same_layer 0,
	worn_by 0,
	skirtlike 0,
	skirt_covering 0,
	skirt_layer 0,
	skirt_maxlayer 3;
#Endif; ! COMPLEXCLOTHING

Class Lockable class Thing,
with	lockstate 1,
	locked false,
	with_key 0;

! ------------------------------------------------------------------------------
! The building block routines.
! ------------------------------------------------------------------------------

[ Move obj target flag i j k;
  if (~~obj && target) return RunTimeError(24);
  if (~~target) return RunTimeError(25, obj);
  if (~~obj) return RunTimeError(26, target);

  if (obj == player && flag == 4) { <Go target>; <Look>; return; }

  j = parent(obj);
  if (CountChildren(obj, 1) || CountChildren(obj, 2))
	{ i = child(obj);
	  while (i) {
		if (i.beneathparent || i.behindparent)
		 { 
		   i.ontopofparent = obj.ontopofparent;
		   i.insideofparent = obj.insideofparent;
		   i.behindparent = obj.behindparent;
		   i.beneathparent = obj.beneathparent;
		   k = sibling(i);
		   Move(i, j); i = k;
		 }
		}
	}

  move obj to target;
  if (obj in player) give obj moved ~concealed;
  if (~~obj ofclass Thing) return;

  if (~~target provides location) obj.location = target;
  else obj.location = target.location;

  FindVisibilityLevels(obj);

  if (child(obj)) {
	objectloop (i ofclass Thing && IndirectlyContains(obj, i) ~= 0)
		{ i.location = obj.location; FindVisibilityLevels(i); }
  }

  if (obj == player) {
	switch (flag)
	{ 1: <Look>;
	#Ifndef NO_SCORE;
	  2: NoteArrival(); ScoreArrival();
	#Endif;
	  3: LookSub(1);
	  default: ;
	}
  }

  if (target ofclass Thing) CalcWeight(target);
  if (j && j ofclass Thing) CalcWeight(j);
];

[ Remove obj i j;
  if (~~obj) return RunTimeError(27);

  j = parent(obj);

  remove obj;  if (~~(obj ofclass Thing)) return;

  obj.location = 0; obj.visibility_levels = 0; obj.visibility_ceiling = 0;
  objectloop (i in obj && i.location) i.location = 0;

#Ifdef COMPLEXCLOTHING;
  if (obj provides worn_by && obj.worn_by)
	{ obj.worn_by = 0; obj.layer = 0; obj.covering = 0;
	  obj.covered_by = 0; obj.skirt_layer = 0;
	  obj.skirt_covering = 0;
	}
#Ifnot;
  if (obj has clothing) give obj ~worn;
#Endif;

  if (obj.insideofparent)	obj.insideofparent = 0;
  if (obj.ontopofparent)	obj.ontopofparent = 0;
  if (obj.behindparent)		obj.behindparent = 0;
  if (obj.beneathparent)	obj.beneathparent = 0;
  if (obj.attachedtoparent)	obj.attachedtoparent = 0;

  if (j) CalcWeight(j);
];

! If an object is not specified, the routine
! will loop over all objects, calculating
! their weights and (if appropriate) their
! indirect weights.
[ CalcWeight i j k;

  if (i) { if (~~i provides weight) rfalse;

	if (i provides indirect_weight) {
		i.indirect_weight = 0;
		objectloop ( k provides weight && parent(k) == i)
			{ i.indirect_weight = i.indirect_weight + k.weight;
		  	  if (k provides indirect_weight)
				{ CalcWeight(k); i.indirect_weight = i.indirect_weight + k.indirect_weight; }
			}
	}
	return;
  }

  objectloop (j provides indirect_weight) { j.indirect_weight = 0;
	objectloop ( k ofclass Thing && parent(k) == j ) {
  		j.indirect_weight = j.indirect_weight + k.weight;
		if (k provides indirect_weight)
			{ CalcWeight(k); j.indirect_weight = j.indirect_weight + k.indirect_weight; }
		}
	}
];

! Finds the Room in which the object is contained, or
! returns 0 if there is none.
[ WhereIs obj j;

  j = parent(obj);
  if (j == 0) rfalse;
  while (j ofclass Thing)
	j = parent(j);
  if (j ofclass Room) return j;
  return 0;

];

! Loops through every object and sets its location property.
! Only called before Initialise.
[ SetLocations obj;

  objectloop (obj ofclass Object) {

  	if (obj ofclass Room) obj.visibility_ceiling = obj;

  	if (obj provides location) obj.location = WhereIs(obj);

  	if (obj provides visibility_levels) FindVisibilityLevels(obj);
  }
];

[ SetNPCNumbers x obj;

  x = 0;

  objectloop (obj ofclass Person && obj ~= player) {
	obj.npc_number = x; x++;
  }
];

[ FindVisibilityLevels obj visibility i j k;

  if (~~obj provides visibility_levels) rfalse;

  if (obj ofclass Room) { obj.visibility_ceiling = obj; return 1; }

  if (obj in nothing) { obj.visibility_ceiling = 0; obj.visibility_levels = 0; return 0; }

  visibility = 1;

  j = parent(obj); if (j ofclass Class) rfalse;

#IFDEF COMPLEXCLOTHING;
  if (obj ofclass Wearable && obj.covered_by ~= 0) {
	obj.visibility_ceiling = obj.covered_by;
	jump ReturnVis;
  }
#ENDIF;

  i = parent(j);
  k = obj;

#Ifdef GRADUAL_LIGHT;
  while (i ~= 0 && (j ofclass Person || SimpleAttach(obj) ||
	(obj.insideofparent && (j hasnt closed || j has transparent or semitransparent)) ||
	obj.ontopofparent ||
	(obj.beneathparent && j.&hidden_contents-->1) ||
	(obj.behindparent && j.&hidden_contents-->2)))
		{ obj = j; j = i; i = parent(i); visibility++; }
#Ifnot;
  while (i ~= 0 && (j ofclass Person || SimpleAttach(obj) ||
	(obj.insideofparent && (j hasnt closed || j has transparent)) ||
	obj.ontopofparent ||
	(obj.beneathparent && j.&hidden_contents-->1) ||
	(obj.behindparent && j.&hidden_contents-->2)))
		{ obj = j; j = i; i = parent(i); visibility++; }
#Endif;

  k.visibility_ceiling = j;
  k.visibility_levels = visibility;

  #IFDEF TRACK_LOCATIONS;
  if (obj ofclass Person) objectloop (i ofclass Thing && IsVisible(obj, i))
	Recognize(obj, i);
  #IFNOT;
  if (obj == player) objectloop (i ofclass Thing && IsVisible(obj, i))
	give i topic;
  #ENDIF;

  .ReturnVis;
  return visibility;
];

[ SimpleAttach obj;

  if (~~obj.attachedtoparent) rfalse;
  if (obj.insideofparent || obj.ontopofparent || obj.beneathparent || obj.behindparent)
	rfalse;
  rtrue;
];

! ----------------------------------------------------------------------------
! The default player object; not compiled if CHARACTER is used to define
! an alternative.
! ----------------------------------------------------------------------------

#Ifndef CHARACTER;
Person  selfobj "(self object)"
with 	short_name  [;  return L__M(##Miscellany, 18); ],
	description [;  return L__M(##Miscellany, 19); ],
	before NULL,
	after NULL,
	life NULL,
	each_turn NULL,
	time_out NULL,
	describe NULL,
	add_to_scope 0,
	capacity 100,
	orders 0,
	number 0,
	location InformLibrary,
has	concealed topic proper;
#Endif;

! ----------------------------------------------------------------------------
!  The definition of the token-numbering system.
! ----------------------------------------------------------------------------

Constant ILLEGAL_TT         = 0;    ! Types of grammar token: illegal
Constant ELEMENTARY_TT      = 1;    !     (one of those below)
Constant PREPOSITION_TT     = 2;    !     e.g. 'into'
Constant ROUTINE_FILTER_TT  = 3;    !     e.g. noun=CagedCreature
Constant ATTR_FILTER_TT     = 4;    !     e.g. edible
Constant SCOPE_TT           = 5;    !     e.g. scope=Spells
Constant GPR_TT             = 6;    !     a general parsing routine

Constant NOUN_TOKEN         = 0;    ! The elementary grammar tokens, and
Constant HELD_TOKEN         = 1;    ! the numbers compiled by Inform to
Constant MULTI_TOKEN        = 2;    ! encode them
Constant MULTIHELD_TOKEN    = 3;
Constant MULTIEXCEPT_TOKEN  = 4;
Constant MULTIINSIDE_TOKEN  = 5;
Constant CREATURE_TOKEN     = 6;
Constant SPECIAL_TOKEN      = 7;
Constant NUMBER_TOKEN       = 8;
Constant TOPIC_TOKEN        = 9;


Constant GPR_FAIL           = -1;   ! Return values from General Parsing
Constant GPR_PREPOSITION    = 0;    ! Routines
Constant GPR_NUMBER         = 1;
Constant GPR_MULTIPLE       = 2;
Constant GPR_REPARSE        = REPARSE_CODE;
Constant GPR_NOUN           = $ff00;
Constant GPR_HELD           = $ff01;
Constant GPR_MULTI          = $ff02;
Constant GPR_MULTIHELD      = $ff03;
Constant GPR_MULTIEXCEPT    = $ff04;
Constant GPR_MULTIINSIDE    = $ff05;
Constant GPR_CREATURE       = $ff06;

Constant ENDIT_TOKEN        = 15;   ! Value used to mean "end of grammar line"

#Iftrue (Grammar__Version == 1);

[ AnalyseToken token m;
    found_tdata = token;
    if (token < 0)   { found_ttype = ILLEGAL_TT; return; }
    if (token <= 8)  { found_ttype = ELEMENTARY_TT; return; }
    if (token < 15)  { found_ttype = ILLEGAL_TT; return; }
    if (token == 15) { found_ttype = ELEMENTARY_TT; return; }
    if (token < 48)  { found_ttype = ROUTINE_FILTER_TT;
                       found_tdata = token - 16;
                       return;
    }
    if (token < 80)  { found_ttype = GPR_TT;
                       found_tdata = #preactions_table-->(token-48);
                       return;
    }
    if (token < 128) { found_ttype = SCOPE_TT;
                       found_tdata = #preactions_table-->(token-80);
                       return;
    }
    if (token < 180) { found_ttype = ATTR_FILTER_TT;
                       found_tdata = token - 128;
                       return;
    }

    found_ttype = PREPOSITION_TT;
    m = #adjectives_table;
    for (::) {
        if (token == m-->1) { found_tdata = m-->0; return; }
        m = m+4;
    }
    m = #adjectives_table; RunTimeError(1);
    found_tdata = m;
];

[ UnpackGrammarLine line_address i m;
    for (i=0 : i<32 : i++) {
        line_token-->i = ENDIT_TOKEN;
        line_ttype-->i = ELEMENTARY_TT;
        line_tdata-->i = ENDIT_TOKEN;
    }
    for (i=0 : i<=5 : i++) {
        line_token-->i = line_address->(i+1);
        AnalyseToken(line_token-->i);
        if ((found_ttype == ELEMENTARY_TT) && (found_tdata == NOUN_TOKEN)
           && (m == line_address->0)) {
            line_token-->i = ENDIT_TOKEN;
            break;
        }
        line_ttype-->i = found_ttype;
        line_tdata-->i = found_tdata;
        if (found_ttype ~= PREPOSITION_TT) m++;
    }
    action_to_be = line_address->7;
    action_reversed = false;
    params_wanted = line_address->0;
    return line_address + 8;
];

#Ifnot; ! Grammar__Version == 2

[ AnalyseToken token;
    if (token == ENDIT_TOKEN) {
        found_ttype = ELEMENTARY_TT;
        found_tdata = ENDIT_TOKEN;
        return;
    }
    found_ttype = (token->0) & $$1111;
    found_tdata = (token+1)-->0;
];

#Ifdef TARGET_ZCODE;

[ UnpackGrammarLine line_address i;
    for (i=0 : i<32 : i++) {
        line_token-->i = ENDIT_TOKEN;
        line_ttype-->i = ELEMENTARY_TT;
        line_tdata-->i = ENDIT_TOKEN;
    }
    action_to_be = 256*(line_address->0) + line_address->1;
    action_reversed = ((action_to_be & $400) ~= 0);
    action_to_be = action_to_be & $3ff;
    line_address--;
    params_wanted = 0;
    for (i=0 : : i++) {
        line_address = line_address + 3;
        if (line_address->0 == ENDIT_TOKEN) break;
        line_token-->i = line_address;
        AnalyseToken(line_address);
        if (found_ttype ~= PREPOSITION_TT) params_wanted++;
        line_ttype-->i = found_ttype;
        line_tdata-->i = found_tdata;
    }
    return line_address + 1;
];

#Ifnot; ! TARGET_GLULX

[ UnpackGrammarLine line_address i;
    for (i=0 : i<32 : i++) {
        line_token-->i = ENDIT_TOKEN;
        line_ttype-->i = ELEMENTARY_TT;
        line_tdata-->i = ENDIT_TOKEN;
    }
    @aloads line_address 0 action_to_be;
    action_reversed = (((line_address->2) & 1) ~= 0);
    line_address = line_address - 2;
    params_wanted = 0;
    for (i=0 : : i++) {
        line_address = line_address + 5;
        if (line_address->0 == ENDIT_TOKEN) break;
        line_token-->i = line_address;
        AnalyseToken(line_address);
        if (found_ttype ~= PREPOSITION_TT) params_wanted++;
        line_ttype-->i = found_ttype;
        line_tdata-->i = found_tdata;
    }
    return line_address + 1;
];

#Endif; ! TARGET_
#Endif; ! Grammar__Version

! To protect against a bug in early versions of the "Zip" interpreter:
! Of course, in Glulx, this routine actually performs work.

#Ifdef TARGET_ZCODE;

[ Tokenise__ b p; b->(2 + b->1) = 0; @tokenise b p; ];

#Ifnot; ! TARGET_GLULX

Array gg_tokenbuf -> DICT_WORD_SIZE;

[ GGWordCompare str1 str2 ix jx;
    for (ix=0 : ix<DICT_WORD_SIZE : ix++) {
        jx = (str1->ix) - (str2->ix);
        if (jx ~= 0) return jx;
    }
    return 0;
];

[ Tokenise__ buf tab
    cx numwords len bx ix wx wpos wlen val res dictlen entrylen;
    len = buf-->0;
    buf = buf+WORDSIZE;

    ! First, split the buffer up into words. We use the standard Infocom
    ! list of word separators (comma, period, double-quote).

    cx = 0;
    numwords = 0;
    while (cx < len) {
        while (cx < len && buf->cx == ' ') cx++;
        if (cx >= len) break;
        bx = cx;
        if (buf->cx == '.' or ',' or '"') cx++;
        else {
            while (cx < len && buf->cx ~= ' ' or '.' or ',' or '"') cx++;
        }
        tab-->(numwords*3+2) = (cx-bx);
        tab-->(numwords*3+3) = WORDSIZE+bx;
        numwords++;
        if (numwords >= MAX_BUFFER_WORDS) break;
    }
    tab-->0 = numwords;

    ! Now we look each word up in the dictionary.

    dictlen = #dictionary_table-->0;
    entrylen = DICT_WORD_SIZE + 7;

    for (wx=0 : wx<numwords : wx++) {
        wlen = tab-->(wx*3+2);
        wpos = tab-->(wx*3+3);

        ! Copy the word into the gg_tokenbuf array, clipping to DICT_WORD_SIZE
        ! characters and lower case.
        if (wlen > DICT_WORD_SIZE) wlen = DICT_WORD_SIZE;
        cx = wpos - WORDSIZE;
        for (ix=0 : ix<wlen : ix++) gg_tokenbuf->ix = glk($00A0, buf->(cx+ix));
        for (: ix<DICT_WORD_SIZE : ix++) gg_tokenbuf->ix = 0;

        val = #dictionary_table + WORDSIZE;
        @binarysearch gg_tokenbuf DICT_WORD_SIZE val entrylen dictlen 1 1 res;
        tab-->(wx*3+1) = res;
    }
];

#Endif; ! TARGET_

! ============================================================================
!   The InformParser object abstracts the front end of the parser.
!
!   InformParser.parse_input(results)
!   returns only when a sensible request has been made, and puts into the
!   "results" buffer:
!
!   --> 0 = The action number
!   --> 1 = Number of parameters
!   --> 2, 3, ... = The parameters (object numbers), but
!                   0 means "put the multiple object list here"
!                   1 means "put one of the special numbers here"
!
! ----------------------------------------------------------------------------

Object  InformParser "(Inform Parser)"
  with  parse_input [ results; Parser__parse(results); ],
  has   proper;

! ----------------------------------------------------------------------------
!   The Keyboard routine actually receives the player's words,
!   putting the words in "a_buffer" and their dictionary addresses in
!   "a_table".  It is assumed that the table is the same one on each
!   (standard) call.
!
!   It can also be used by miscellaneous routines in the game to ask
!   yes-no questions and the like, without invoking the rest of the parser.
!
!   Return the number of words typed
! ----------------------------------------------------------------------------

#Ifdef TARGET_ZCODE;

[ GetNthChar a_buffer n i;
    for (i = 0: a_buffer->(2+i) == ' ': i++) {
	if (i > a_buffer->(1)) return false;
    }
    return a_buffer->(2+i+n);
];

[ KeyboardPrimitive  a_buffer a_table;

    read a_buffer a_table;

    #Iftrue (#version_number == 6);
    @output_stream -1;
    @loadb a_buffer 1 -> sp;
    @add a_buffer 2 -> sp;
    @print_table sp sp;
    new_line;
    @output_stream 1;
    #Endif;
];

[ KeyCharPrimitive win  key;
    if (win) @set_window win;
    @read_char 1 -> key;
    return key;
];

[ KeyTimerInterrupt;
    rtrue;
];

[ KeyDelay tenths  key;
    @read_char 1 tenths KeyTimerInterrupt -> key;
    return key;
];

#Ifnot; ! TARGET_GLULX

[ GetNthChar a_buffer n i;
    for (i = 0: a_buffer->(4+i) == ' ': i++) {
	if (i > a_buffer->(1)) return false;
    }
    return a_buffer->(4+i+n);
];

[ KeyCharPrimitive win nostat done res ix jx ch;
    jx = ch; ! squash compiler warnings
    if (win == 0) win = gg_mainwin;
    if (gg_commandstr ~= 0 && gg_command_reading ~= false) {
        ! get_line_stream
        done = glk($0091, gg_commandstr, gg_arguments, 31);
        if (done == 0) {
            glk($0044, gg_commandstr, 0); ! stream_close
            gg_commandstr = 0;
            gg_command_reading = false;
            ! fall through to normal user input.
        }
        else {
            ! Trim the trailing newline
            if (gg_arguments->(done-1) == 10) done = done-1;
            res = gg_arguments->0;
            if (res == '\') {
                res = 0;
                for (ix=1 : ix<done : ix++) {
                    ch = gg_arguments->ix;
                    if (ch >= '0' && ch <= '9') {
                        @shiftl res 4 res;
                        res = res + (ch-'0');
                    }
                    else if (ch >= 'a' && ch <= 'f') {
                        @shiftl res 4 res;
                        res = res + (ch+10-'a');
                    }
                    else if (ch >= 'A' && ch <= 'F') {
                        @shiftl res 4 res;
                        res = res + (ch+10-'A');
                    }
                }
            }
        jump KCPContinue;
        }
    }
    done = false;
    glk($00D2, win); ! request_char_event
    while (~~done) {
        glk($00C0, gg_event); ! select
        switch (gg_event-->0) {
          5: ! evtype_Arrange
            if (nostat) {
                glk($00D3, win); ! cancel_char_event
                res = $80000000;
                done = true;
                break;
            }
            DrawStatusLine();
          2: ! evtype_CharInput
            if (gg_event-->1 == win) {
                res = gg_event-->2;
                done = true;
                }
        }
        ix = HandleGlkEvent(gg_event, 1, gg_arguments);
#IFDEF USE_LIBRARY_EXTENSIONS;
        if (ix == 0) ix = LibraryExtensions.RunWhile(ext_handleglkevent, 0, gg_event, 1, gg_arguments);
#ENDIF;
        if (ix == 2) {
            res = gg_arguments-->0;
            done = true;
        }
        else if (ix == -1) {
            done = false;
        }
    }
    if (gg_commandstr ~= 0 && gg_command_reading == false) {
        if (res < 32 || res >= 256 || (res == '\' or ' ')) {
            glk($0081, gg_commandstr, '\'); ! put_buffer_char
            done = 0;
            jx = res;
            for (ix=0 : ix<8 : ix++) {
                @ushiftr jx 28 ch;
                @shiftl jx 4 jx;
                ch = ch & $0F;
                if (ch ~= 0 || ix == 7) done = 1;
                if (done) {
                    if (ch >= 0 && ch <= 9) ch = ch + '0';
                    else                    ch = (ch - 10) + 'A';
                    glk($0081, gg_commandstr, ch); ! put_buffer_char
                }
            }
        }
        else {
            glk($0081, gg_commandstr, res); ! put_buffer_char
        }
        glk($0081, gg_commandstr, 10); ! put_char_stream (newline)
    }
  .KCPContinue;
    return res;
];

[ KeyDelay tenths  key done ix;
    glk($00D2, gg_mainwin); ! request_char_event
    glk($00D6, tenths*100); ! request_timer_events
    while (~~done) {
        glk($00C0, gg_event); ! select
        ix = HandleGlkEvent(gg_event, 1, gg_arguments);
#IFDEF USE_LIBRARY_EXTENSIONS;
        if (ix == 0) ix = LibraryExtensions.RunWhile(ext_handleglkevent, 0, gg_event, 1, gg_arguments);
#ENDIF;
        if (ix == 2) {
            key = gg_arguments-->0;
            done = true;
        }
        else if (ix >= 0 && gg_event-->0 == 1 or 2) {
            key = gg_event-->2;
            done = true;
        }
    }
    glk($00D3, gg_mainwin); ! cancel_char_event
    glk($00D6, 0); ! request_timer_events
    return key;
];

[ KeyboardPrimitive  a_buffer a_table done ix;
    if (gg_commandstr ~= 0 && gg_command_reading ~= false) {
        ! get_line_stream
        done = glk($0091, gg_commandstr, a_buffer+WORDSIZE, (INPUT_BUFFER_LEN-WORDSIZE)-1);
        if (done == 0) {
            glk($0044, gg_commandstr, 0); ! stream_close
            gg_commandstr = 0;
            gg_command_reading = false;
            ! L__M(##CommandsRead, 5); would come after prompt
            ! fall through to normal user input.
        }
        else {
            ! Trim the trailing newline
            if ((a_buffer+WORDSIZE)->(done-1) == 10) done = done-1;
            a_buffer-->0 = done;
            glk($0086, 8); ! set input style
            glk($0084, a_buffer+WORDSIZE, done); ! put_buffer
            glk($0086, 0); ! set normal style
            print "^";
            jump KPContinue;
        }
    }
    done = false;
    glk($00D0, gg_mainwin, a_buffer+WORDSIZE, INPUT_BUFFER_LEN-WORDSIZE, 0); ! request_line_event
    while (~~done) {
        glk($00C0, gg_event); ! select
        switch (gg_event-->0) {
          5: ! evtype_Arrange
            DrawStatusLine();
          3: ! evtype_LineInput
            if (gg_event-->1 == gg_mainwin) {
                a_buffer-->0 = gg_event-->2;
                done = true;
            }
        }
        ix = HandleGlkEvent(gg_event, 0, a_buffer);
#IFDEF USE_LIBRARY_EXTENSIONS;
        if (ix == 0) ix = LibraryExtensions.RunWhile(ext_handleglkevent, 0, gg_event, 0, a_buffer);
#ENDIF;
        if (ix == 2) done = true;
        else if (ix == -1) done = false;
    }
    if (gg_commandstr ~= 0 && gg_command_reading == false) {
        ! put_buffer_stream
        glk($0085, gg_commandstr, a_buffer+WORDSIZE, a_buffer-->0);
        glk($0081, gg_commandstr, 10); ! put_char_stream (newline)
    }
  .KPContinue;
    Tokenise__(a_buffer,a_table);
    ! It's time to close any quote window we've got going.
    if (gg_quotewin) {
        glk($0024, gg_quotewin, 0); ! close_window
        gg_quotewin = 0;
    }
];

#Endif; ! TARGET_


[ Keyboard  a_buffer a_table  nw i w w2 x1 x2;
    DisplayStatus();

  .FreshInput;

    ! Save the start of the buffer, in case "oops" needs to restore it
    ! to the previous time's buffer

    for (i=0 : i<OOPS_WORKSPACE_LEN : i++) oops_workspace->i = a_buffer->i;

    ! In case of an array entry corruption that shouldn't happen, but would be
    ! disastrous if it did:

    #Ifdef TARGET_ZCODE;
    a_buffer->0 = INPUT_BUFFER_LEN - WORDSIZE;
    a_table->0  = MAX_BUFFER_WORDS; ! Allow to split input into this many words
    #Endif; ! TARGET_

    ! Print the prompt, and read in the words and dictionary addresses

    L__M(##Prompt);
#IFDEF USE_LIBRARY_EXTENSIONS;
    if (AfterPrompt() == 0) LibraryExtensions.RunAll(ext_afterprompt);
#IFNOT;
    AfterPrompt();
#ENDIF;
    #IfV5;
    DrawStatusLine();
    #Endif; ! V5

    KeyboardPrimitive(a_buffer, a_table);
    nw = NumberWords(a_table);

    ! If the line was blank, get a fresh line
    if (nw == 0) {
        L__M(##Miscellany, 10);
        jump FreshInput;
    }

    ! Unless the opening word was "oops", return
    ! Conveniently, a_table-->1 is the first word in both ZCODE and GLULX.

    w = a_table-->1;
    if (w == OOPS1__WD or OOPS2__WD or OOPS3__WD) jump DoOops;

    if (a_buffer->WORDSIZE == COMMENT_CHARACTER) {
        #Ifdef TARGET_ZCODE;
        if ((HDR_GAMEFLAGS-->0) & $0001 || xcommsdir)
                                           L__M(##Miscellany, 54);
        else                               L__M(##Miscellany, 55);
        #Ifnot; ! TARGET_GLULX
        if (gg_scriptstr || gg_commandstr) L__M(##Miscellany, 54);
        else                               L__M(##Miscellany, 55);
        #Endif; ! TARGET_

        jump FreshInput;
    }

    #IfV5;
    ! Undo handling

    if ((w == UNDO1__WD or UNDO2__WD or UNDO3__WD) && (nw==1)) {
        if (turns == START_MOVE) {
            L__M(##Miscellany, 11);
            jump FreshInput;
        }
        if (undo_flag == 0) {
            L__M(##Miscellany, 6);
            jump FreshInput;
        }
        if (undo_flag == 1) jump UndoFailed;
        #Ifdef TARGET_ZCODE;
        @restore_undo i;
        #Ifnot; ! TARGET_GLULX
        @restoreundo i;
        i = (~~i);
        #Endif; ! TARGET_
        if (i == 0) {
          .UndoFailed;
            L__M(##Miscellany, 7);
        }
        jump FreshInput;
    }
    #Ifdef TARGET_ZCODE;
    @save_undo i;
    #Ifnot; ! TARGET_GLULX
    @saveundo i;
    if (i == -1) {
        GGRecoverObjects();
        i = 2;
    }
    else  i = (~~i);
    #Endif; ! TARGET_
    just_undone = 0;
    undo_flag = 2;
    if (i == -1) undo_flag = 0;
    if (i == 0) undo_flag = 1;
    if (i == 2) {
        RestoreColours();
        #Ifdef TARGET_ZCODE;
        style bold;
        #Ifnot; ! TARGET_GLULX
        glk($0086, 4); ! set subheader style
        #Endif; ! TARGET_
        print (name) player.location, "^";
        #Ifdef TARGET_ZCODE;
        style roman;
        #Ifnot; ! TARGET_GLULX
        glk($0086, 0); ! set normal style
        #Endif; ! TARGET_
        L__M(##Miscellany, 13);
        just_undone = 1;
        jump FreshInput;
    }
    #Endif; ! V5

    return nw;

  .DoOops;
    if (oops_from == 0) {
        L__M(##Miscellany, 14);
        jump FreshInput;
    }
    if (nw == 1) {
        L__M(##Miscellany, 15);
        jump FreshInput;
    }
    if (nw > 2) {
        L__M(##Miscellany, 16);
        jump FreshInput;
    }

    ! So now we know: there was a previous mistake, and the player has
    ! attempted to correct a single word of it.

    for (i=0 : i<INPUT_BUFFER_LEN : i++) buffer2->i = a_buffer->i;
    #Ifdef TARGET_ZCODE;
    x1 = a_table->9;  ! Start of word following "oops"
    x2 = a_table->8;  ! Length of word following "oops"
    #Ifnot; ! TARGET_GLULX
    x1 = a_table-->6; ! Start of word following "oops"
    x2 = a_table-->5; ! Length of word following "oops"
    #Endif; ! TARGET_

    ! Repair the buffer to the text that was in it before the "oops"
    ! was typed:

    for (i=0 : i < OOPS_WORKSPACE_LEN : i++) a_buffer->i = oops_workspace->i;
    Tokenise__(a_buffer, a_table);

    ! Work out the position in the buffer of the word to be corrected:

    #Ifdef TARGET_ZCODE;
    w = a_table->(4*oops_from + 1); ! Start of word to go
    w2 = a_table->(4*oops_from);    ! Length of word to go
    #Ifnot; ! TARGET_GLULX
    w = a_table-->(3*oops_from);      ! Start of word to go
    w2 = a_table-->(3*oops_from - 1); ! Length of word to go
    #Endif; ! TARGET_

#IfDef OOPS_CHECK;
    print "[~";
    for (i=0 : i<w2 : i++) for (i=0 : i<w2 : i++) print (char)a_buffer->(i+w);
    print "~ --> ~";
#Endif;

    ! Write spaces over the word to be corrected:

    for (i=0 : i<w2 : i++) a_buffer->(i+w) = ' ';

    if (w2 < x2) {
        ! If the replacement is longer than the original, move up...
        for (i=INPUT_BUFFER_LEN-1 : i>=w+x2 : i--)
            a_buffer->i = a_buffer->(i-x2+w2);

        ! ...increasing buffer size accordingly.
        SetKeyBufLength(GetKeyBufLength(a_buffer) + (x2-w2), a_buffer);
    }

    ! Write the correction in:

    for (i=0 : i<x2 : i++) {
        a_buffer->(i+w) = buffer2->(i+x1);
#IfDef OOPS_CHECK;
        print (char) buffer2->(i+x1);
#Endif;
    }

#IfDef OOPS_CHECK;
        print "~]^^";
#Endif;

    Tokenise__(a_buffer, a_table);
    nw=NumberWords(a_table);

!    saved_ml = 0;
    return nw;
]; ! end of Keyboard

! ----------------------------------------------------------------------------
!   To simplify the picture a little, a rough map of the main routine:
!
!   (A) Get the input, do "oops" and "again"
!   (B) Is it a direction, and so an implicit "go"?  If so go to (K)
!   (C) Is anyone being addressed?
!   (D) Get the verb: try all the syntax lines for that verb
!   (E) Break down a syntax line into analysed tokens
!   (F) Look ahead for advance warning for multiexcept/multiinside
!   (G) Parse each token in turn (calling ParseToken to do most of the work)
!   (H) Cheaply parse otherwise unrecognised conversation and return
!   (I) Print best possible error message
!   (J) Retry the whole lot
!   (K) Last thing: check for "then" and further instructions(s), return.
!
!   The strategic points (A) to (K) are marked in the commentary.
!
!   Note that there are three different places where a return can happen.
! ----------------------------------------------------------------------------

[ Parser__parse  results   syntax line num_lines line_address i j k
                           token l m;

    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    !
    ! A: Get the input, do "oops" and "again"
    !
    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

    ! Firstly, in "not held" mode, we still have a command left over from last
    ! time (eg, the user typed "eat biscuit", which was parsed as "take biscuit"
    ! last time, with "eat biscuit" tucked away until now).  So we return that.

    if (notheld_mode == 1) {
        for (i=0 : i<8 : i++) results-->i = kept_results-->i;
        notheld_mode = 0;
        rtrue;
    }

    if (held_back_mode == 1) {
        held_back_mode = 0;
        Tokenise__(buffer, parse);
        jump ReParse;
    }

  .ReType;

    ! Clear certain variables.

    failed_action = 0;
    keep_silent = 0;
    action_mode = 0;
    indef_wanted = 0; first_indef_wanted = 0;
    multiflag = 0; multi_had = 0; multi_wanted = 0;
    reparsing = false;
    keep_silent = 0;
    arrived = 0;
    indef_specific = 0;
    i = multiple_object-->0; while (multiple_object-->i~=0) { multiple_object-->i = 0; i++; }

    Keyboard(buffer, parse);

#Ifdef INFIX;
    ! An Infix verb is a special kind of meta verb.  We mark them here.
    if (GetNthChar(buffer, 0) == ';')
	infix_verb = true;
    else
	infix_verb = false;
#Endif;

  .ReParse;

    parser_inflection = name;
    parser_inflection_func = false;
    grammar_line = 0;

    ! Initially assume the command is aimed at the player, and the verb
    ! is the first word

    num_words = NumberWords();
    wn = 1;

    #Ifdef LanguageToInformese;
    LanguageToInformese();
    #IfV5;
    ! Re-tokenise:
    Tokenise__(buffer,parse);
    #Endif; ! V5
    #Endif; ! LanguageToInformese

#IFDEF USE_LIBRARY_EXTENSIONS;
    if (BeforeParsing() == false) {
        LibraryExtensions.ext_number_1 = wn;    ! Set "between calls" functionality to restore wn each pass
        LibraryExtensions.BetweenCalls = LibraryExtensions.RestoreWN;
        LibraryExtensions.RunWhile(ext_beforeparsing, false);
        LibraryExtensions.BetweenCalls = 0;     ! Turn off "between calls" functionality
     }
#ENDIF;
    num_words = NumberWords();

    k=0;
    #Ifdef DEBUG;
    if (parser_trace >= 2) {
        print "[ ";
        for (i=0 : i<num_words : i++) {

            #Ifdef TARGET_ZCODE;
            j = parse-->(i*2 + 1);
            #Ifnot; ! TARGET_GLULX
            j = parse-->(i*3 + 1);
            #Endif; ! TARGET_
            k = WordAddress(i+1);
            l = WordLength(i+1);
            print "~"; for (m=0 : m<l : m++) print (char) k->m; print "~ ";

            if (j == 0) print "?";
            else {
                #Ifdef TARGET_ZCODE;
                if (UnsignedCompare(j, HDR_DICTIONARY-->0) >= 0 &&
                    UnsignedCompare(j, HDR_HIGHMEMORY-->0) < 0)
                     print (address) j;
                else print j;
                #Ifnot; ! TARGET_GLULX
                if (j->0 == $60) print (address) j;
                else print j;
                #Endif; ! TARGET_
            }
            if (i ~= num_words-1) print " / ";
        }
        print " ]^";
    }
    #Endif; ! DEBUG
    verb_wordnum = 1;
    actor = player;
    usual_grammar_after = 0;

  .AlmostReParse;

    scope_token = 0;
    action_to_be = NULL;

    ! Begin from what we currently think is the verb word

  .BeginCommand;

    wn = verb_wordnum;
    verb_word = NextWordStopped();

    ! If there's no input here, we must have something like "person,".

    if (verb_word == -1) {
        best_etype = STUCK_PE;
        jump GiveError;
    }

    ! Now try for "again" or "g", which are special cases: don't allow "again" if nothing
    ! has previously been typed; simply copy the previous text across

    if (verb_word == AGAIN2__WD or AGAIN3__WD) verb_word = AGAIN1__WD;
    if (verb_word == AGAIN1__WD) {
        if (actor ~= player) {
            L__M(##Miscellany, 20);
            jump ReType;
        }
	if (turns == START_MOVE) {
	    L__M(##Miscellany, 64);
	    jump ReType;
	}
        #Ifdef TARGET_ZCODE;
        if (buffer3->1 == 0) {
            L__M(##Miscellany, 21);
            jump ReType;
        }
        #Ifnot; ! TARGET_GLULX
        if (buffer3-->0 == 0) {
            L__M(##Miscellany, 21);
            jump ReType;
        }
        #Endif; ! TARGET_

    if (WordAddress(verb_wordnum) == buffer + WORDSIZE) { ! not held back
    ! splice rest of buffer onto end of buffer3
    #Ifdef TARGET_ZCODE; i = buffer3->1;
    #Ifnot; ! TARGET_GLULX i = buffer3-->0;
    #Endif;
    while (buffer3->(i + WORDSIZE - 1) == ' ' or '.')
    i--;
    j = i - WordLength(verb_wordnum); ! amount to move buffer up by
    if (j > 0) {
    for (m=INPUT_BUFFER_LEN-1 : m>=WORDSIZE+j : m--)
    buffer->m = buffer->(m-j);
    #Ifdef TARGET_ZCODE; buffer->1 = buffer->1 + j;
    #Ifnot; ! TARGET_GLULX buffer-->0 = buffer-->0 + j;
    #Endif;
    }

    for (m=WORDSIZE : m<WORDSIZE+i : m++) buffer->m = buffer3->m;
     if (j < 0) for (:m<WORDSIZE+i-j : m++) buffer->m = ' '; }
    else
        for (i=0 : i<INPUT_BUFFER_LEN : i++) buffer->i = buffer3->i;
        jump ReParse;
    }

    ! Save the present input in case of an "again" next time

    if (verb_word ~= AGAIN1__WD)
        for (i=0 : i<INPUT_BUFFER_LEN : i++) buffer3->i = buffer->i;

    if (usual_grammar_after == 0) {
        j = verb_wordnum;
        i = RunRoutines(actor, grammar); 
        #Ifdef DEBUG;
        if (parser_trace >= 2 && actor.grammar ~= 0 or NULL)
            print " [Grammar property returned ", i, "]^";
        #Endif; ! DEBUG

        #Ifdef TARGET_ZCODE;
        if ((i ~= 0 or 1) &&
            (UnsignedCompare(i, dict_start) < 0 ||
             UnsignedCompare(i, dict_end) >= 0 ||
             (i - dict_start) % dict_entry_size ~= 0)) {
            usual_grammar_after = j;
            i=-i;
        }

        #Ifnot; ! TARGET_GLULX
        if (i < 0) { usual_grammar_after = verb_wordnum; i=-i; }
        #Endif;

        if (i == 1) {
            results-->0 = action;
            results-->1 = 2;
            results-->2 = noun;
            results-->3 = second;
            rtrue;
        }
        if (i ~= 0) { verb_word = i; wn--; verb_wordnum--; }
        else { wn = verb_wordnum; verb_word = NextWord(); }
    }
    else usual_grammar_after = 0;

    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    !
    ! B: Is it a direction, and so an implicit "go"?  If so go to (K)
    !
    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

    #Ifdef LanguageIsVerb;
    if (verb_word == 0) {
        i = wn; verb_word = LanguageIsVerb(buffer, parse, verb_wordnum);
        wn = i;
    }
    #Endif; ! LanguageIsVerb

    ! If the first word is not listed as a verb, it must be a direction,
    ! an object to be examined, or the name of someone to talk to.

    if (verb_word == 0 || ((verb_word->#dict_par1) & DICT_VERB) == 0) {

        ! First we need to identify the first word.  This needs use of NounDomain,
	! a routine which does the object matching, returning the object number,
	! or 0 if none found, or REPARSE_CODE if it has restructured the parse table
	! so the whole parse must be begun again.

        wn = verb_wordnum; indef_mode = false; token_filter = 0;
        l = NounDomain(actor);
        if (l == REPARSE_CODE) jump ReParse;

        ! Only check for a comma (a "someone, do something" command) if we are
        ! not already in the middle of one.  (This simplification stops us from
        ! worrying about "robot, wizard, you are an idiot", telling the robot to
        ! tell the wizard that she is an idiot.)

	  if (actor == player)
		for (j=2 : j<=num_words : j++) {
			i=NextWord();
			if (i == comma_word) jump Conversation;
            }

        if (l) {

        ! If it is a direction, send back the results:
        ! action=GoSub, no. of arguments=1, argument 1=the direction.

		if (l in compass) {
            		results-->0 = ##Go;
            		action_to_be = ##Go;
        	}

	  ! Otherwise, treat the text as an EXAMINE action.

	  	else {
			results-->0 = ##Examine;
			action_to_be = ##Examine;
			PronounNotice(l);
	  	}
		
            results-->1 = 1;
            results-->2 = l;
            multiple_object-->0 = 0;
            jump LookForMore;
	  }

	verb_word = UnknownVerb(verb_word);
#IFDEF USE_LIBRARY_EXTENSIONS;
        if (verb_word == false) verb_word = LibraryExtensions.RunWhile(ext_unknownverb, false, vw);
#ENDIF;
	if (verb_word ~= 0) jump VerbAccepted;
        best_etype = VERB_PE;
        jump GiveError;

    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    !
    ! C: Has the player named a person?
    !
    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

      .Conversation;

        j = wn - 1;
        if (j == 1) {
            L__M(##Miscellany, 22);
            jump ReType;
        }
	wn = 1;
        lookahead = HELD_TOKEN;
        scope_reason = TALKING_REASON;
	l = NounDomain(actor);
	scope_reason = PARSING_REASON;
        if (l == REPARSE_CODE) jump ReParse;
        if (l == 0) {
            L__M(##Miscellany, 23);
            jump ReType;
        }

        ! The object addressed must at least be "talkable" if not actually "animate"
        ! (the distinction allows, for instance, a microphone to be spoken to,
        ! without the parser thinking that the microphone is human).

        if (l hasnt animate && l hasnt talkable) {
            L__M(##Miscellany, 24, l);
            jump ReType;
        }

        ! Check that there aren't any mystery words between the end of the person's
        ! name and the comma (eg, throw out "dwarf sdfgsdgs, go north").

        if (wn ~= j) {
            L__M(##Miscellany, 25);
            jump ReType;
        }

      .Conversation2;

        ! The player has now successfully named someone.  Adjust "him", "her", "it":

        PronounNotice(l);

	#IFDEF TRACK_LOCATIONS;
	! If the person is absent, give an error.
	if (AbsentItem(l)) {
		L__M(##Examine, 5, l); jump Retype;
	}
	#ENDIF;

        ! Set the global variable "actor", adjust the number of the first word,
        ! and begin parsing again from there.

        verb_wordnum = j + 1;

        ! Stop things like "me, again":

        if (l == player) {
            wn = verb_wordnum;
            if (NextWordStopped() == AGAIN1__WD or AGAIN2__WD or AGAIN3__WD) {
                L__M(##Miscellany, 20);
                jump ReType;
            }
        }

        actor = l;
        action_mode = 2;
        #Ifdef DEBUG;
        if (parser_trace >= 1)
            print "[Actor is ", (the) actor, " in ", (name) actor.location, "]^";
        #Endif; ! DEBUG
        jump BeginCommand;

    } ! end of first-word-not-a-verb

    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    !
    ! D: Get the verb: try all the syntax lines for that verb
    !
    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

  .VerbAccepted;
    ! We now definitely have a verb, not a direction, whether we got here by the
    ! "take ..." or "person, take ..." method.  Get the meta flag for this verb:

    meta = ((verb_word->#dict_par1) & 2)/2;

    ! You can't order other people to "full score" for you, and so on...

    if (meta == 1 && actor ~= player) {
        best_etype = VERB_PE;
        meta = 0;
        jump GiveError;
    }

    ! Now let i be the corresponding verb number, stored in the dictionary entry
    ! (in a peculiar 255-n fashion for traditional Infocom reasons)...

    i = $ff-(verb_word->#dict_par2);

    ! ...then look up the i-th entry in the verb table, whose address is at word
    ! 7 in the Z-machine (in the header), so as to get the address of the syntax
    ! table for the given verb...

    #Ifdef TARGET_ZCODE;
    syntax = (HDR_STATICMEMORY-->0)-->i;
    #Ifnot; ! TARGET_GLULX
    syntax = (#grammar_table)-->(i+1);
    #Endif; ! TARGET_

    ! ...and then see how many lines (ie, different patterns corresponding to the
    ! same verb) are stored in the parse table...

    num_lines = (syntax->0) - 1;

    ! ...and now go through them all, one by one.
    ! To prevent pronoun_word 0 being misunderstood,

    pronoun_word = NULL; pronoun_obj = NULL;

    #Ifdef DEBUG;
    if (parser_trace >= 1) print "[Parsing for the verb '", (address) verb_word, "' (", num_lines+1, " lines)]^";
    #Endif; ! DEBUG

    best_etype = STUCK_PE; nextbest_etype = STUCK_PE;
    multiflag = false;

    ! "best_etype" is the current failure-to-match error - it is by default
    ! the least informative one, "don't understand that sentence".
    ! "nextbest_etype" remembers the best alternative to having to ask a
    ! scope token for an error message (i.e., the best not counting ASKSCOPE_PE).
    ! multiflag is used here to prevent inappropriate MULTI_PE errors
    ! in addition to its unrelated duties passing information to action routines

    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    !
    ! E: Break down a syntax line into analysed tokens
    !
    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

    line_address = syntax + 1;

    for (line=0 : line<=num_lines : line++) {

        for (i=0 : i<32 : i++) {
            line_token-->i = ENDIT_TOKEN;
            line_ttype-->i = ELEMENTARY_TT;
            line_tdata-->i = ENDIT_TOKEN;
        }

	grammar_line = line + 1;

        ! Unpack the syntax line from Inform format into three arrays; ensure that
        ! the sequence of tokens ends in an ENDIT_TOKEN.

        line_address = UnpackGrammarLine(line_address);

        #Ifdef DEBUG;
        if (parser_trace >= 1) {
            if (parser_trace >= 2) new_line;
            print "[line ", line; DebugGrammarLine();
            print "]^";
        }
        #Endif; ! DEBUG

        ! We aren't in "not holding" or inferring modes, and haven't entered
        ! any parameters on the line yet, or any special numbers; the multiple
        ! object is still empty.

        not_holding = 0;
        inferfrom = 0;
        parameters = 0;
        nsns = 0; special_word = 0; special_number = 0;
        multiple_object-->0 = 0;
        multi_context = 0;
        etype = STUCK_PE;

        ! Put the word marker back to just after the verb

        wn = verb_wordnum+1;

    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    !
    ! F: Look ahead for advance warning for multiexcept/multiinside
    !
    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

        ! There are two special cases where parsing a token now has to be
        ! affected by the result of parsing another token later, and these
        ! two cases (multiexcept and multiinside tokens) are helped by a quick
        ! look ahead, to work out the future token now.  We can only carry this
        ! out in the simple (but by far the most common) case:
        !
        !     multiexcept <one or more prepositions> noun
        !
        ! and similarly for multiinside.

        advance_warning = NULL; indef_mode = false;
        for (i=0,m=false,pcount=0 : line_token-->pcount ~= ENDIT_TOKEN : pcount++) {
            scope_token = 0;

            if (line_ttype-->pcount ~= PREPOSITION_TT) i++;

            if (line_ttype-->pcount == ELEMENTARY_TT) {
                if (line_tdata-->pcount == MULTI_TOKEN) m = true;
                if (line_tdata-->pcount == MULTIEXCEPT_TOKEN or MULTIINSIDE_TOKEN && i == 1) {
                    ! First non-preposition is "multiexcept" or
                    ! "multiinside", so look ahead.

                    ! We need this to be followed by one or more prepositions.

                    pcount++;

                    if (line_ttype-->pcount == PREPOSITION_TT) {

                    #Ifdef DEBUG;
                    if (parser_trace >= 2) print " [Trying look-ahead]^";
                    #Endif; ! DEBUG

                         ! skip ahead to the next preposition word in the input
                        do {
                            l = NextWord();
                        } until ((wn > num_words) ||
                                 (l && (l->#dict_par1) & 8 ~= 0));

                         if (wn > num_words) {				     
                             #Ifdef DEBUG;
                             if (parser_trace >= 2)
                                 print " [Look-ahead aborted: prepositions missing]^";
                             #Endif;
				     jump CheckNoun;
                         }

                         do {
                             if (PrepositionChain(l, pcount) ~= -1) {
                                 ! advance past the chain
                                 if ((line_token-->pcount)->0 & $20 ~= 0) {
                                     pcount++;
                                     while ((line_token-->pcount ~= ENDIT_TOKEN) &&
                                            ((line_token-->pcount)->0 & $10 ~= 0))
                                         pcount++;
                                 } else {
                                     pcount++;
                                 }
                             } else {
                                 ! try to find another preposition word
                                 do {
                                     l = NextWord();
                                 } until ((wn >= num_words) ||
                                          (l && (l->#dict_par1) & 8 ~= 0));

                                 if (l && (l->#dict_par1) & 8) continue;
					   if (wn >= num_words) jump CheckNoun;

                                 ! lookahead failed
                                 #Ifdef DEBUG;
                                 if (parser_trace >= 2)
                                     print " [Look-ahead aborted: prepositions don't match]^";
                                 #endif;
                                 jump LineFailed;
                             }
                             l = NextWord();
                         } until (line_ttype-->pcount ~= PREPOSITION_TT);

					.CheckNoun;

                         ! put back the non-preposition we just read
                         wn--;

                          if ((line_ttype-->pcount == ELEMENTARY_TT) && (line_tdata-->pcount == NOUN_TOKEN)) {
                             l = Descriptors();  ! skip past THE etc
                             if (l~=0) etype=l;  ! don't allow multiple objects
                             l = NounDomain(actor, NOUN_TOKEN);
                             #Ifdef DEBUG;
                             if (parser_trace >= 2) {
                                 print " [Advanced to ~noun~ token: ";
                                 if (l == REPARSE_CODE) print "re-parse request]^";
                                 if (l == 1) print "but multiple found]^";
                                 if (l == 0) print "error ", etype, "]^";
                                 if (l >= 2) print (the) l, "]^";
                              }
                             #Endif; ! DEBUG
                             if (l == REPARSE_CODE) jump ReParse;
                             if (l >= 2) advance_warning = l;
                          }
                      }
		      break;
                }
            }
        }

        ! Slightly different line-parsing rules will apply to "take multi", to
        ! prevent "take all" behaving correctly but misleadingly when there's
        ! nothing to take.

        take_all_rule = 0;
        if (m && params_wanted == 1 &&
		action_to_be == ##Take)
            take_all_rule = 1;

        ! And now start again, properly, forearmed or not as the case may be.
        ! As a precaution, we clear all the variables again (they may have been
        ! disturbed by the call to NounDomain, which may have called outside
        ! code, which may have done anything!).

        not_holding = 0;
        inferfrom = 0;
        parameters = 0;
        nsns = 0; special_word = 0; special_number = 0;
        multiple_object-->0 = 0;
        etype = STUCK_PE;
        wn = verb_wordnum+1;

    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    !
    ! G: Parse each token in turn (calling ParseToken to do most of the work)
    !
    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

        ! "Pattern" gradually accumulates what has been recognised so far,
        ! so that it may be reprinted by the parser later on

        for (pcount=1 : : pcount++) {
            pattern-->pcount = PATTERN_NULL; scope_token = 0;

            token = line_token-->(pcount-1);
            lookahead = line_token-->pcount;

            #Ifdef DEBUG;
            if (parser_trace >= 2)
                print " [line ", line, " token ", pcount, " word ", wn, " : ", (DebugToken) token,
                  "]^";
            #Endif; ! DEBUG

            if (token ~= ENDIT_TOKEN) {
                scope_reason = PARSING_REASON;
                parser_inflection = name;
		parser_inflection_func = false;
                AnalyseToken(token);

                if (action_to_be == ##AskTo && found_ttype == ELEMENTARY_TT &&
                    found_tdata == TOPIC_TOKEN)
                {
                    l=inputobjs-->2;
                    wn--;
                    j = wn;
                    jump Conversation2;
                }

                l = ParseToken__(found_ttype, found_tdata, pcount-1, token);
                while (l<-200) l = ParseToken__(ELEMENTARY_TT, l + 256);
                scope_reason = PARSING_REASON;

                if (l == GPR_PREPOSITION) {
                    if (found_ttype~=PREPOSITION_TT && (found_ttype~=ELEMENTARY_TT ||
                        found_tdata~=TOPIC_TOKEN)) params_wanted--;
                    l = true;
                }
                else
                    if (l < 0) l = false;
                    else
                        if (l ~= GPR_REPARSE) {
                            if (l == GPR_NUMBER) {
                                if (nsns == 0) special_number1 = parsed_number;
                                else special_number2 = parsed_number;
                                nsns++; l = 1;
                            }
                            if (l == GPR_MULTIPLE) l = 0;
                            results-->(parameters+2) = l;
                            parameters++;
                            pattern-->pcount = l;
                            l = true;
                        }

                #Ifdef DEBUG;
                if (parser_trace >= 3) {
                    print "  [token resulted in ";
                    if (l == REPARSE_CODE) print "re-parse request]^";
                    if (l == 0) print "failure with error type ", etype, "]^";
                    if (l == 1) print "success]^";
                }
                #Endif; ! DEBUG

                if (l == REPARSE_CODE) jump ReParse;
                if (l == false) break;
            }
            else {

                ! If the player has entered enough already but there's still
                ! text to wade through: store the pattern away so as to be able to produce
                ! a decent error message if this turns out to be the best we ever manage,
                ! and in the mean time give up on this line.

                ! However, if the superfluous text begins with a comma, "then", or "and then" then
                ! take that to be the start of another instruction.

                if (wn <= num_words) {
                    l = NextWord();
                    if (l == AND1__WD or AND2__WD or AND3__WD or THEN1__WD or THEN2__WD or THEN3__WD or comma_word) {
                        held_back_mode = 1;
			if (l == comma_word) { l = NextWord(); if (l ~= AND1__WD or AND2__WD or AND3__WD) wn--;
			}
			hb_wn = wn-1;
                    }
                    else {
			if (reparsing && WordInProperty(l, results-->2, name)) jump NotSuperfluous;
                        for (m=0 : m<32 : m++) pattern2-->m = pattern-->m;
                        pcount2 = pcount;
                        etype = UPTO_PE;
			saved_oops = wn-1;
                        break;
                    }
                }

		.NotSuperfluous;

                ! Now, we may need to revise the multiple object because of the single one
                ! we now know (but didn't when the list was drawn up).

                if (parameters >= 1 && results-->2 == 0) {
                    l = ReviseMulti(results-->3);
                    if (l ~= 0) { etype = l; results-->0 = action_to_be; break; }
                }
                if (parameters >= 2 && results-->3 == 0) {
                    l = ReviseMulti(results-->2);
                    if (l ~= 0) { etype = l; break; }
                }
                ! To trap the case of "take all" inferring only "yourself" when absolutely
                ! nothing else is in the vicinity.
                if (take_all_rule == 2 && results-->2 == actor) {
                    best_etype = NOTHING_PE;
                    jump GiveError;
                }

                #Ifdef DEBUG;
                if (parser_trace >= 1) print "[Line successfully parsed]^";
                #Endif; ! DEBUG

                ! The line has successfully matched the text.  Declare the input error-free...

                oops_from = 0;

                ! ...explain any inferences made (using the pattern)...

                if (inferfrom ~= 0) {
                    print "("; PrintCommand(inferfrom); print ")^";
                }
		    if (multi_had < multi_wanted && indef_wanted < 100) {
			 for (m=1 : multiple_object-->m : m++)
				{ if (multiple_object-->m in actor) l = true; }
			if (l == 0 && multi_context == MULTIHELD_TOKEN)
				{ if (original_noun == 0) original_noun = true; best_etype = NOTHELD_PE; jump GiveError; }
			else if (l && multi_context == MULTIHELD_TOKEN)
				L__M(##Miscellany, 44, multi_had);
			else L__M(##Miscellany, 44, multi_had, multi_had);
		    }

                ! ...copy the action number, and the number of parameters...

                results-->0 = action_to_be;
                results-->1 = parameters;

                ! ...reverse first and second parameters if need be...

                if (action_reversed && parameters == 2) {
                    i = results-->2; results-->2 = results-->3;
                    results-->3 = i;
                    if (nsns == 2) {
                        i = special_number1; special_number1 = special_number2;
                        special_number2 = i;
                    }
                }

                ! ...and to reset "it"-style objects to the first of these parameters, if
                ! there is one (and it really is an object)...

                if (parameters > 0 && results-->2 >= 2)
                    PronounNotice(results-->2);

                ! ...and worry about the case where an object was allowed as a parameter
                ! even though the player wasn't holding it and should have been: in this
                ! event, keep the results for next time round, go into "not holding" mode,
                ! and for now tell the player what's happening and return a "take" request
                ! instead. This happens here if there is only one object; in multiple mode
		! it is handled later.

                if (not_holding ~= 0 && ~~multiple_object-->0) {
                    action = ##Take; action_mode = 1;
                    i = RunRoutines(not_holding, before_implicit); keep_silent = 1;
                    ! i = 0: Take the object, tell the player (default)
                    ! i = 1: Take the object, don't tell the player
                    ! i = 2: Don't take the object, continue
                    ! i = 3: Don't take the object, don't continue
                    if (i > 2) { best_etype = NOTHELD_PE; jump GiveError; }
                    if (i < 2) {        ! perform the implicit Take
                        if (i ~= 1)     ! and tell the player
                            L__M(##Miscellany, 26, not_holding); new_line;
	      		
                        notheld_mode = 1;
                        for (i=0 : i<8 : i++) kept_results-->i = results-->i;
                        results-->0 = ##Take;
                        results-->1 = 1;
                        results-->2 = not_holding;
                    }
                }

                ! ...and return from the parser altogether, having successfully matched
                ! a line.

                if (held_back_mode == 1) {
                    wn=hb_wn;
                    jump LookForMore;
                }
                rtrue;
            } ! end of if(token ~= ENDIT_TOKEN) else
        } ! end of for(pcount++)

         .LineFailed;

        ! The line has failed to match.
        ! We continue the outer "for" loop, trying the next line in the grammar.

        if (etype > best_etype) best_etype = etype;
        if (etype ~= ASKSCOPE_PE && etype > nextbest_etype) nextbest_etype = etype;

        ! ...unless the line was something like "take all" which failed because
        ! nothing matched the "all", in which case we stop and give an error now.

        if (take_all_rule == 2 && etype == NOTHING_PE) break;
	  if (action_to_be == ##Drop && l)
		{ best_etype = NOTHELD_PE; original_noun = -1; break; }

    } ! end of for(line++)

    ! The grammar is exhausted: every line has failed to match.

    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    !
    ! H: Cheaply parse otherwise unrecognised conversation and return
    !
    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

  .GiveError;

    etype = best_etype;

    ! Errors are handled differently depending on who was talking.
    ! If the command was addressed to somebody else (eg, "dwarf, sfgh") then
    ! it is taken as conversation which the parser has no business in disallowing.

    if (actor ~= player) {
        if (usual_grammar_after ~= 0) {
            verb_wordnum = usual_grammar_after;
            jump AlmostReParse;
        }
        wn = verb_wordnum;
        special_word = NextWord();
        if (special_word == comma_word) {
            special_word = NextWord();
            verb_wordnum++;
        }
        special_number = TryNumber(verb_wordnum);
        results-->0 = ##NotUnderstood;
        results-->1 = 2;
        results-->2 = 1; special_number1 = special_word;
        results-->3 = actor;
        consult_from = verb_wordnum; consult_words = num_words-consult_from+1;
        rtrue;
    }

    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    !
    ! I: Print best possible error message
    !
    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

    ! If the player was the actor (eg, in "take dfghh") the error must be printed,
    ! and fresh input called for.  In four cases the oops word must be jiggled.

    if (ParserError(etype) ~= 0) jump ReType;
#IFDEF USE_LIBRARY_EXTENSIONS;
    if (LibraryExtensions.RunWhile(ext_parsererror, false, etype)) jump ReType;
#ENDIF;
    pronoun_word = pronoun__word; pronoun_obj = pronoun__obj;

    if (etype == STUCK_PE) {    L__M(##Miscellany, 27); oops_from = 1; }
    if (etype == UPTO_PE)  {    L__M(##Miscellany, 28);
        for (m=0 : m<32 : m++) pattern-->m = pattern2-->m;
        pcount = pcount2; PrintCommand(0); L__M(##Miscellany, 56);
	oops_from = saved_oops;
    }
    if (etype == NUMBER_PE)     L__M(##Miscellany, 29);
    if (etype == CANTSEE_PE)  { L__M(##Miscellany, 30); oops_from = saved_oops; }
    if (etype == NOTHELD_PE)  { L__M(##Drop, 2, noun); oops_from = saved_oops; }
    if (etype == MULTI_PE)      L__M(##Miscellany, 33);
    if (etype == MMULTI_PE)     L__M(##Miscellany, 34);
    if (etype == VAGUE_PE)      L__M(##Miscellany, 35);
    if (etype == EXCEPT_PE)     L__M(##Miscellany, 36);
    if (etype == ANIMA_PE)      L__M(##Miscellany, 37);
    if (etype == VERB_PE)       L__M(##Miscellany, 38);
    if (etype == SCENERY_PE)    L__M(##Miscellany, 39);
    if (etype == ITGONE_PE) {
        if (pronoun_obj == NULL)
                                L__M(##Miscellany, 35);
        else                    L__M(##Miscellany, 40);
    }
    if (etype == JUNKAFTER_PE)  L__M(##Miscellany, 41);
    if (etype == TOOFEW_PE)     L__M(##Miscellany, 42, multi_had);
    if (etype == NOTHING_PE) {
            if (indef_wanted) L__M(##Miscellany, 43);
            else { if (indef_specific < 0) L__M(##Miscellany, 43);
			else L__M(##Miscellany, 44);
		 }
   }
   if (etype == ASKSCOPE_PE) {
        scope_stage = 3;
        if (indirect(scope_error) == -1) {
            best_etype = nextbest_etype;
            jump GiveError;
        }
   }

    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    !
    ! J: Retry the whole lot
    !
    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

    ! And go (almost) right back to square one...

    jump ReType;

    ! ...being careful not to go all the way back, to avoid infinite repetition
    ! of a deferred command causing an error.


    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    !
    ! K: Last thing: check for "then" and further instructions(s), return.
    !
    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

    ! At this point, the return value is all prepared, and we are only looking
    ! to see if there is an "and", "then", or "and then" followed by subsequent instruction(s).

  .LookForMore;

    if (wn > num_words) rtrue;

    i = NextWord();
    if (i == AND1__WD or AND2__WD or AND3__WD or comma_word) {
	i = NextWord();
	if (i ~= THEN1__WD or THEN2__WD or THEN3__WD) { wn--; i = comma_word; }
    }
    if (i == THEN1__WD or THEN2__WD or THEN3__WD or comma_word) {
        if (wn > num_words) {
           held_back_mode = false;
           return;
        }
        i = WordAddress(verb_wordnum);
        j = WordAddress(wn);
        for (: i<j : i++) i->0 = ' ';
        i = NextWord();
        if (i == AGAIN1__WD or AGAIN2__WD or AGAIN3__WD) {
            ! Delete the words "then again" from the again buffer,
            ! in which we have just realised that it must occur:
            ! prevents an infinite loop on "i. again"

            i = WordAddress(wn-2)-buffer;
            if (wn > num_words) j = INPUT_BUFFER_LEN-1;
            else j = WordAddress(wn)-buffer;
            for (: i<j : i++) buffer3->i = ' ';
        }
        Tokenise__(buffer,parse);
        held_back_mode = true;
        return;
    }
    best_etype = UPTO_PE;
    jump GiveError;

]; ! end of Parser__parse

[ ScopeCeiling person;
  return person.visibility_ceiling;
];

! ----------------------------------------------------------------------------
!  Descriptors()
!
!  Handles descriptive words like "my", "his", "another" and so on.
!  Skips "the", and leaves wn pointing to the first misunderstood word.
!
!  Allowed to set up for a plural only if allow_p is set
!
!  Returns error number, or 0 if no error occurred
! ----------------------------------------------------------------------------

Constant OTHER_BIT  =   1;     !  These will be used in Adjudicate()
Constant MY_BIT     =   2;     !  to disambiguate choices
Constant THAT_BIT   =   4;
Constant PLURAL_BIT =   8;
#IFNDEF NO_DARKNESS;
Constant LIT_BIT    =  16;
Constant UNLIT_BIT  =  32;
#ENDIF;

[ ResetDescriptors;
    first_indef_wanted = indef_wanted;
    indef_mode = 0; indef_type = 0; indef_wanted = 0; indef_guess_p = 0;
    indef_possambig = false;
    indef_owner = nothing;
    indef_cases = $$111111111111;
    indef_nspec_at = 0;
];

[ Descriptors o x flag cto type n m;
    ResetDescriptors();
    if (wn > num_words) return 0;

    m = wn;

    for (flag=true : flag :) {
        o = NextWordStopped(); flag = false;

       for (x=1 : x<=LanguageDescriptors-->0 : x=x+4)
            if (o == LanguageDescriptors-->x) {
                flag = true;
                type = LanguageDescriptors-->(x+2);
                if (type ~= DEFART_PK) indef_mode = 1;
                indef_possambig = true;
                indef_cases = indef_cases & (LanguageDescriptors-->(x+1));

                if (type == POSSESS_PK) {
                    cto = LanguageDescriptors-->(x+3);
                    switch (cto) {
                      0: indef_type = indef_type | MY_BIT;
                      1: indef_type = indef_type | THAT_BIT;
                      default:
                        indef_owner = PronounValue(cto);
                        if (indef_owner == NULL) indef_owner = InformParser;
                    }
                }
#Ifndef NO_DARKNESS;
                if (type == light)  indef_type = indef_type | LIT_BIT;
                if (type == -light) indef_type = indef_type | UNLIT_BIT;
#Endif;
            }
        if (o == OTHER1__WD or OTHER2__WD or OTHER3__WD) {
            indef_mode = 1; flag = 1;
            indef_type = indef_type | OTHER_BIT;
        }
        if (o == ALL1__WD or ALL2__WD or ALL3__WD or ALL4__WD or ALL5__WD) {
            indef_mode = 1; flag = 1; indef_wanted = 100;
            if (take_all_rule == 1) take_all_rule = 2;
            indef_type = indef_type | PLURAL_BIT;
        }
        if (allow_plurals) {
            n = TryNumber(wn-1);
            if (n == 1) { indef_mode = 1; flag = 1; indef_wanted = 1; }
            if (n > 1) {
                indef_guess_p = 1;
                indef_mode = 1; flag = 1; indef_wanted = n;
                indef_nspec_at = wn-1;
                indef_type = indef_type | PLURAL_BIT;
            }
        }
        if (flag == 1 && NextWordStopped() ~= OF1__WD or OF2__WD or OF3__WD or OF4__WD)
            wn--;  ! Skip 'of' after these
    }
    wn--; num_desc = wn - m;
    return 0;
];

! ----------------------------------------------------------------------------
!  CreatureTest: Will this person do for a "creature" token?
! ----------------------------------------------------------------------------

[ CreatureTest obj;
    if (obj has animate) rtrue;
    if (obj hasnt talkable) rfalse;
    rfalse;
];

[ PrepositionChain wd index;
    if (line_tdata-->index == wd) return wd;
    if ((line_token-->index)->0 & $20 == 0) return -1;
    do {
        if (line_tdata-->index == wd) return wd;
        index++;
    } until ((line_token-->index == ENDIT_TOKEN) || (((line_token-->index)->0 & $10) == 0));
    return -1;
];

! ----------------------------------------------------------------------------
!  ParseToken(type, data):
!      Parses the given token, from the current word number wn, with exactly
!      the specification of a general parsing routine.
!      (Except that for "topic" tokens and prepositions, you need to supply
!      a position in a valid grammar line as third argument.)
!
!  Returns:
!    GPR_REPARSE  for "reconstructed input, please re-parse from scratch"
!    GPR_PREPOSITION  for "token accepted with no result"
!    $ff00 + x    for "please parse ParseToken(ELEMENTARY_TT, x) instead"
!    0            for "token accepted, result is the multiple object list"
!    1            for "token accepted, result is the number in parsed_number"
!    object num   for "token accepted with this object as result"
!    -1           for "token rejected"
!
!  (A)            Analyse the token; handle all tokens not involving
!                 object lists and break down others into elementary tokens
!  (B)            Begin parsing an object list
!  (C)            Parse descriptors (articles, pronouns, etc.) in the list
!  (D)            Parse an object name
!  (E)            Parse connectives ("and", "but", etc.) and go back to (C)
!  (F)            Return the conclusion of parsing an object list
! ----------------------------------------------------------------------------

[ ParseToken given_ttype given_tdata token_n x y;
    x = lookahead; lookahead = NOUN_TOKEN;
    y = ParseToken__(given_ttype,given_tdata,token_n);
    if (y == GPR_REPARSE) Tokenise__(buffer,parse);
    lookahead = x; return y;
];

[ ParseToken__ given_ttype given_tdata token_n
             token l o i j k and_parity single_object desc_wn many_flag
             prev_indef_wanted;

    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    !
    ! A: Analyse token; handle all not involving object lists, break down others
    !
    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

    token_filter = 0;

    switch (given_ttype) {
      ELEMENTARY_TT:
        switch (given_tdata) {
          SPECIAL_TOKEN:
            l = TryNumber(wn);
            special_word = NextWord();
            #Ifdef DEBUG;
            if (l ~= -1000)
                if (parser_trace >= 3) print "  [Read special as the number ", l, "]^";
            #Endif; ! DEBUG
            if (l == -1000) {
                #Ifdef DEBUG;
                if (parser_trace >= 3) print "  [Read special word at word number ", wn, "]^";
                #Endif; ! DEBUG
                l = special_word;
            }
            parsed_number = l; return GPR_NUMBER;

          NUMBER_TOKEN:
            l=TryNumber(wn++);
            if (l == -1000) { etype = NUMBER_PE; return GPR_FAIL; }
            #Ifdef DEBUG;
            if (parser_trace>=3) print "  [Read number as ", l, "]^";
            #Endif; ! DEBUG
            parsed_number = l; return GPR_NUMBER;

          CREATURE_TOKEN:
            if (action_to_be == ##Answer or ##Ask or ##AskFor or ##Tell)
                scope_reason = TALKING_REASON;

          TOPIC_TOKEN:
            consult_from = wn;
            if ((line_ttype-->(token_n+1) ~= PREPOSITION_TT) &&
               (line_token-->(token_n+1) ~= ENDIT_TOKEN))
                RunTimeError(13);
            do o = NextWordStopped();
            until (o == -1 || PrepositionChain(o, token_n+1) ~= -1);
            wn--;
            consult_words = wn-consult_from;
		if (consult_words == 0) return GPR_FAIL;
            if (action_to_be == ##Ask or ##Answer or ##Tell) {
                o = wn; wn = consult_from; parsed_number = NextWord();
                wn = o; return 1;
            }
            if (o==-1 && (line_ttype-->(token_n+1) == PREPOSITION_TT))
                return GPR_FAIL;    ! don't infer if required preposition is absent
            return GPR_PREPOSITION;
        }

      PREPOSITION_TT:
        #Iffalse (Grammar__Version == 1);
        ! Is it an unnecessary alternative preposition, when a previous choice
        ! has already been matched?
        if ((token->0) & $10) return GPR_PREPOSITION;
        #Endif; ! Grammar__Version

        ! If we've run out of the player's input, but still have parameters to
        ! specify, we go into "infer" mode, remembering where we are and the
        ! preposition we are inferring...

        if (wn > num_words) {
            if (inferfrom==0 && parameters<params_wanted) {
                inferfrom = pcount; inferword = token;
                pattern-->pcount = REPARSE_CODE + Dword__No(given_tdata);
            }

            ! If we are not inferring, then the line is wrong...

            if (inferfrom == 0) return -1;

            ! If not, then the line is right but we mark in the preposition...

            pattern-->pcount = REPARSE_CODE + Dword__No(given_tdata);
            return GPR_PREPOSITION;
        }

        o = NextWord();

        pattern-->pcount = REPARSE_CODE + Dword__No(o);

        ! Whereas, if the player has typed something here, see if it is the
        ! required preposition... if it's wrong, the line must be wrong,
        ! but if it's right, the token is passed (jump to finish this token).

        if (o == given_tdata) return GPR_PREPOSITION;
        #Iffalse (Grammar__Version == 1);
        if (PrepositionChain(o, token_n) ~= -1) return GPR_PREPOSITION;
        #Endif; ! Grammar__Version
        return -1;

      GPR_TT:
        l = indirect(given_tdata);
        #Ifdef DEBUG;
        if (parser_trace >= 3) print "  [Outside parsing routine returned ", l, "]^";
        #Endif; ! DEBUG
        return l;

      SCOPE_TT:
        scope_token = given_tdata;
        scope_stage = 1;
        l = indirect(scope_token);
        #Ifdef DEBUG;
        if (parser_trace >= 3) print "  [Scope routine returned multiple-flag of ", l, "]^";
        #Endif; ! DEBUG
        if (l == 1) given_tdata = MULTI_TOKEN; else given_tdata = NOUN_TOKEN;

      ATTR_FILTER_TT:
        token_filter = 1 + given_tdata;
        given_tdata = NOUN_TOKEN;

      ROUTINE_FILTER_TT:
        token_filter = given_tdata;
        given_tdata = NOUN_TOKEN;

    } ! end of switch(given_ttype)

    token = given_tdata;

    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    !
    ! B: Begin parsing an object list
    !
    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

    ! There are now three possible ways we can be here:
    !     parsing an elementary token other than "special" or "number";
    !     parsing a scope token;
    !     parsing a noun-filter token (either by routine or attribute).
    !
    ! In each case, token holds the type of elementary parse to
    ! perform in matching one or more objects, and
    ! token_filter is 0 (default), an attribute + 1 for an attribute filter
    ! or a routine address for a routine filter.

    token_allows_multiple = false;
    if (token == MULTI_TOKEN or MULTIHELD_TOKEN or MULTIEXCEPT_TOKEN or MULTIINSIDE_TOKEN)
        token_allows_multiple = token;

    many_flag = false; and_parity = true; dont_infer = false;

    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    !
    ! C: Parse descriptors (articles, pronouns, etc.) in the list
    !
    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

    ! We expect to find a list of objects next in what the player's typed.

  .ObjectList;

    #Ifdef DEBUG;
    if (parser_trace >= 3) print "  [Object list from word ", wn, "]^";
    #Endif; ! DEBUG

    ! Take an advance look at the next word: if it's "it" or "them", and these
    ! are unset, set the appropriate error number and give up on the line
    ! (if not, these are still parsed in the usual way - it is not assumed
    ! that they still refer to something in scope)

    o = NextWord(); wn--;

    pronoun_word = NULL; pronoun_obj = NULL;
    l = PronounValue(o);
    if (l ~= 0) {
        pronoun_word = o; pronoun_obj = l;
        if (l == NULL) {
            ! Don't assume this is a use of an unset pronoun until the
            ! descriptors have been checked, because it might be an
            ! article (or some such) instead

            for (l=1 : l<=LanguageDescriptors-->0 : l=l+4)
                if (o == LanguageDescriptors-->l) jump AssumeDescriptor;
            pronoun__word = pronoun_word; pronoun__obj = pronoun_obj;
            etype = VAGUE_PE; return GPR_FAIL;
        }
    }

  .AssumeDescriptor;

    if (o == ME1__WD or ME2__WD) { pronoun_word = o; pronoun_obj = player; }
    if (o == ME3__WD or SELF1__WD or SELF2__WD or SELF3__WD or YOU1__WD or YOU2__WD)
	{ pronoun_word = o; pronoun_obj = actor; }

    allow_plurals = true;
    desc_wn = wn;

  .TryAgain;

    ! First, we parse any descriptive words (like "the", "five", or "every"):
    l = Descriptors();
    if (l ~= 0) { etype = l; return GPR_FAIL; }

  .TryAgain2;

    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    !
    ! D: Parse an object name
    !
    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

    ! This is an actual specified object, and is therefore where a typing error
    ! is most likely to occur, so we set:

    oops_from = wn;

    ! So, two cases.  Case 1: token not equal to "held"
    ! and so we may well be dealing with multiple objects.

    ! In either case below we use NounDomain, giving it the token number as
    ! context, and instructing to examine everything in scope to the actor.

    if (token ~= HELD_TOKEN) {
        i = multiple_object-->0;
        #Ifdef DEBUG;
        if (parser_trace >= 3) print "  [Calling NounDomain on location and actor]^";
        #Endif; ! DEBUG
        l = NounDomain(actor, token);
        if (l == REPARSE_CODE) return l;                  ! Reparse after Q&A
        if (indef_wanted == 100 && l == 0 && number_matched == 0)
            l = 1; ! ReviseMulti if TAKE ALL FROM empty container

        if (token_allows_multiple && ~~multiflag) {
            if (best_etype==MULTI_PE) best_etype=STUCK_PE;
            multiflag = true; ! noun = multiple_object-->1;
        }
        if (l == 0) {

            if (indef_possambig) {
		saved_ml = match_length;
                ResetDescriptors();
                wn = desc_wn;
                jump TryAgain2;
            }
            if (etype == MULTI_PE or TOOFEW_PE && multiflag) etype = STUCK_PE;
            etype = CantSee();
            jump FailToken;
        } ! Choose best error

        #Ifdef DEBUG;
        if (parser_trace >= 3) {
            if (l > 1) print "  [ND returned ", (the) l, "]^";
            else {
                print "  [ND appended to the multiple object list:^";
                k = multiple_object-->0;
                for (j=i+1 : j<=k : j++)
                    print "  Entry ", j, ": ", (The) multiple_object-->j,
                          " (", multiple_object-->j, ")^";
                print "  List now has size ", k, "]^";
            }
        }
        #Endif; ! DEBUG

        if (l == 1) {
            if (~~many_flag) many_flag = true;
            else {                                ! Merge with earlier ones
                k = multiple_object-->0;          ! (with either parity)
                multiple_object-->0 = i;
                for (j=i+1 : j<=k : j++) {
                    if (and_parity) MultiAdd(multiple_object-->j);
                    else            MultiSub(multiple_object-->j);
                }
                #Ifdef DEBUG;
                if (parser_trace >= 3) print "  [Merging ", k-i, " new objects to the ", i, " old ones]^";
                #Endif; ! DEBUG
            }
        }
        else {
            ! A single object was indeed found

            if (match_length == 0 && indef_possambig) {
                ! So the answer had to be inferred from no textual data,
                ! and we know that there was an ambiguity in the descriptor
                ! stage (such as a word which could be a pronoun being
                ! parsed as an article or possessive).  It's worth having
                ! another go.

                ResetDescriptors();
                wn = desc_wn;
                jump TryAgain2;
            }

            if (token == CREATURE_TOKEN && CreatureTest(l) == 0) {
                 etype = ANIMA_PE;
                jump FailToken;
            } !  Animation is required

            if (~~many_flag) {
			single_object = l;
			if (token == MULTIHELD_TOKEN) jump JustOne;
		}
            else {
                if (and_parity) MultiAdd(l); else MultiSub(l);
                #Ifdef DEBUG;
                if (parser_trace >= 3) print "  [Combining ", (the) l, " with list]^";
                #Endif; ! DEBUG
            }
        }
    }

    else {

    ! Case 2: token is "held" (which cannot take multiple objects) and may generate an implicit take.

        l = NounDomain(actor, token);       ! Same as above...
        if (l == REPARSE_CODE) return GPR_REPARSE;
        if (l == 0) {
            if (indef_possambig) {
                ResetDescriptors();
                wn = desc_wn;
                jump TryAgain2;
            }
            etype = CantSee(); jump FailToken;            ! Choose best error
        }
	  else {
        ! ...until it produces something not held by the actor.  Then an implicit
        ! take must be tried.  If this is already happening anyway, things are too
        ! confused and we have to give up (but saving the oops marker so as to get
        ! it on the right word afterwards).
        ! The point of this last rule is that a sequence like
        !
        !     > read newspaper
        !     (taking the newspaper first)
        !     The dwarf unexpectedly prevents you from taking the newspaper!
        !
        ! should not be allowed to go into an infinite repeat - read becomes
        ! take then read, but take has no effect, so read becomes take then read...
        ! Anyway for now all we do is record the number of the object to take.

	.JustOne;
        o = parent(l);
        if (o ~= actor) {
            if (notheld_mode == 1) {
                saved_oops = oops_from;
                etype = NOTHELD_PE;
                jump FailToken;
            }
            if (action_to_be ~= ##Drop) not_holding = l;
            #Ifdef DEBUG;
            if (parser_trace >= 3) print "  [Allowing object ", (the) l, " for now]^";
            #Endif; ! DEBUG
        }
        single_object = l;
	}
    } ! end of if (token ~= HELD_TOKEN) else

    ! The following moves the word marker to just past the named object...

    wn = oops_from + match_length;

    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    !
    ! E: Parse connectives ("and", "but", etc.) and go back to (C)
    !
    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

    ! Object(s) specified now: is that the end of the list, or have we reached
    ! "and", "but" and so on?  If so, create a multiple-object list if we
    ! haven't already (and are allowed to).

  .NextInList;

    o = NextWord();

    if (o == AND1__WD or AND2__WD or AND3__WD or THEN1__WD or THEN2__WD or THEN3__WD or BUT1__WD or BUT2__WD or BUT3__WD or comma_word) {
        #Ifdef DEBUG;
        if (parser_trace >= 3) print "  [Read connective '", (address) o, "' at word ", wn, "]^";
        #Endif; ! DEBUG

	k = NextWord();

        if (k ~= 0 && (k->#dict_par1) & (DICT_VERB) == DICT_VERB) {
            wn = wn - 2; ! player meant 'THEN'
            jump PassToken;
        }
        if (o == comma_word or AND1__WD or THEN1__WD) {
		if (o == comma_word && k == AND1__WD) k = NextWord();
		if (k == THEN1__WD or THEN2__WD or THEN3__WD) {
			#Ifdef DEBUG;
			if (parser_trace >= 3) print "  [Read connective '", (address) k, "' at word ", wn, "]^";
			#Endif; ! DEBUG
			k = NextWord();
		}
		wn--;
	}
	else if (o == BUT1__WD or BUT2__WD or BUT3__WD) wn--;
 
        if (k ~= 0 && (k->#dict_par1) & (DICT_VERB) == DICT_VERB) {
            wn = wn - 2; ! player meant 'THEN'
            jump PassToken;
        }

        if (~~token_allows_multiple) {
            if (multiflag) jump PassToken; ! give UPTO_PE error
            etype=MULTI_PE;
            jump FailToken;
        }

        if (o == BUT1__WD or BUT2__WD or BUT3__WD) and_parity = 1-and_parity;

        if (~~many_flag) {
            multiple_object-->0 = 1;
            multiple_object-->1 = single_object;
            many_flag = true;
            #Ifdef DEBUG;
            if (parser_trace >= 3) print "  [Making new list from ", (the) single_object, "]^";
            #Endif; ! DEBUG
        }
        dont_infer = true; inferfrom=0;           ! Don't print (inferences)
        jump ObjectList;                          ! And back around
    }

    wn--;   ! Word marker back to first not-understood word

    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    !
    ! F: Return the conclusion of parsing an object list
    !
    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

    ! Happy or unhappy endings:

  .PassToken;

    if (many_flag) {
        single_object = GPR_MULTIPLE;
        multi_context = token;
    }
    else {
        if (indef_mode == 1 && indef_type & PLURAL_BIT ~= 0) {
            if (indef_wanted < 100 && indef_wanted > 1) {
                multi_had = 1; multi_wanted = indef_wanted;
                etype = TOOFEW_PE;
                jump FailToken;
            }
        }
    }
    return single_object;

  .FailToken;

    ! If we were only guessing about it being a plural, try again but only
    ! allowing singulars (so that words like "six" are not swallowed up as
    ! Descriptors)

    if (allow_plurals && indef_guess_p == 1) {
        #Ifdef DEBUG;
        if (parser_trace >= 4) print "   [Retrying singulars after failure ", etype, "]^";
        #Endif;
        prev_indef_wanted = indef_wanted;
        allow_plurals = false;
        wn = desc_wn;
        jump TryAgain;
    }

    if ((indef_wanted > 0 || prev_indef_wanted > 0) && (~~multiflag)) etype = MULTI_PE;

    return GPR_FAIL;

]; ! end of ParseToken__

! ----------------------------------------------------------------------------
!  NounDomain does the most substantial part of parsing an object name.
!
!  It is given a "domain" - usually the actor who is
!  looking - and a context (i.e. token type), and returns:
!
!   0    if no match at all could be made,
!   1    if a multiple object was made,
!   k    if object k was the one decided upon,
!   REPARSE_CODE if it asked a question of the player and consequently rewrote
!        the player's input, so that the whole parser should start again
!        on the rewritten input.
!
!   In the case when it returns 1<k<REPARSE_CODE, it also sets the variable
!   length_of_noun to the number of words in the input text matched to the
!   noun.
!   In the case k=1, the multiple objects are added to multiple_object by
!   hand (not by MultiAdd, because we want to allow duplicates).
! ----------------------------------------------------------------------------

[ NounDomain domain context first_word i j k l answer_words marker;
    #Ifdef DEBUG;
    if (parser_trace >= 4) {
        print "   [NounDomain called at word ", wn, "]^";
        print "   ";
        if (indef_mode) {
            print "seeking indefinite object: ";
            if (indef_type & OTHER_BIT)  print "other ";
            if (indef_type & MY_BIT)     print "my ";
            if (indef_type & THAT_BIT)   print "that ";
            if (indef_type & PLURAL_BIT) print "plural ";
#IFNDEF NO_DARKNESS;
            if (indef_type & LIT_BIT)    print "lit ";
            if (indef_type & UNLIT_BIT)  print "unlit ";
#ENDIF;
            if (indef_owner ~= 0) print "owner:", (name) indef_owner;
            new_line;
            print "   number wanted: ";
            if (indef_wanted == 100) print "all"; else print indef_wanted;
            new_line;
            print "   most likely GNAs of names: ", indef_cases, "^";
        }
        else print "seeking definite object^";
    }
    #Endif; ! DEBUG

    match_length = 0; number_matched = 0; match_from = wn; placed_in_flag = 0;

    SearchScope(domain, context);

    #Ifdef DEBUG;
    if (parser_trace >= 4) print "   [NounDomain made ", number_matched, " matches]^";
    #Endif; ! DEBUG

    wn = match_from + match_length;

    ! If nothing worked at all, leave with the word marker skipped past the
    ! first unmatched word...

    if (number_matched == 0) { wn++; rfalse; }

    ! Suppose that there really were some words being parsed (i.e., we did
    ! not just infer).  If so, and if there was only one match, it must be
    ! right and we return it...

    if (match_from <= num_words) {
        if (number_matched == 1) {
            i=match_list-->0;
#IFNDEF NO_DARKNESS;
            if (indef_mode) {
                if ((indef_type & LIT_BIT) && i hasnt light) rfalse;
                if ((indef_type & UNLIT_BIT) && i has light) rfalse;
            }
#ENDIF;
            return i;
        }

        ! ...now suppose that there was more typing to come, i.e. suppose that
        ! the user entered something beyond this noun.  If nothing ought to follow,
        ! then there must be a mistake, (unless what does follow is just a full
        ! stop, and/or comma)

        if (wn <= num_words) {
            i = NextWord(); wn--;
            if (i ~=  AND1__WD or AND2__WD or AND3__WD or comma_word
                   or THEN1__WD or THEN2__WD or THEN3__WD
                   or BUT1__WD or BUT2__WD or BUT3__WD) {
                if (lookahead == ENDIT_TOKEN) rfalse;
            }
        }
    }

    ! Now look for a good choice, if there's more than one choice...

    number_of_classes = 0;

    if (number_matched == 1) i = match_list-->0;
    if (number_matched > 1) {
        i = Adjudicate(context);
        if (i == -1) rfalse;
        if (i == 1) rtrue;  !  Adjudicate has made a multiple
                            !  object, and we pass it on
    }

    ! If i is non-zero here, one of two things is happening: either
    ! (a) an inference has been successfully made that object i is
    !     the intended one from the user's specification, or
    ! (b) the user finished typing some time ago, but we've decided
    !     on i because it's the only possible choice.
    ! In either case we have to keep the pattern up to date,
    ! note that an inference has been made and return.
    ! (Except, we don't note which of a pile of identical objects.)

    if (i ~= 0) {
        if (dont_infer) return i;
        if (inferfrom == 0) inferfrom=pcount;
        pattern-->pcount = i;
        return i;
    }

    ! If we get here, there was no obvious choice of object to make.  If in
    ! fact we've already gone past the end of the player's typing (which
    ! means the match list must contain every object in scope, regardless
    ! of its name), then it's foolish to give an enormous list to choose
    ! from - instead we go and ask a more suitable question...

    if (match_from > num_words) jump Incomplete;

    ! If token is held or multiheld but none of the possible matches are
    ! actually being held, then return a suitable error.

    marker = 0;

    if (context == HELD_TOKEN or MULTIHELD_TOKEN) {
		for (i=0 : i <= number_matched : i++) {
			k = match_list-->i;
			if (k && k in actor) marker++;
		}
		if (~~marker)
			{ etype = 6; if (i > 0) original_noun = true; }
    }

    ! Now we print up the question, using the equivalence classes as worked
    ! out by Adjudicate() so as not to repeat ourselves on plural objects...

    if (context == CREATURE_TOKEN) L__M(##Miscellany, 45);
    else                           L__M(##Miscellany, 46);

    j = number_of_classes; marker = 0;

    for (i=1 : i<=number_of_classes : i++) {
        while (((match_classes-->marker) ~= i) && ((match_classes-->marker) ~= -i)) marker++;
        k = match_list-->marker;

        if (match_classes-->marker > 0) print (the) k; else print (a) k;

        if (i < j-1)  print (string) COMMA__TX;
        if (i == j-1) {
		if (j > 2) print (string) OR__TX;
		else print (string) OR2__TX;
	  }
    }
    L__M(##Miscellany, 57);

    ! ...and get an answer:

  .WhichOne;
    #Ifdef TARGET_ZCODE;
    for (i=2 : i<INPUT_BUFFER_LEN : i++) buffer2->i = ' ';
    #Endif; ! TARGET_ZCODE
    answer_words=Keyboard(buffer2, parse2);

    ! Conveniently, parse2-->1 is the first word in both ZCODE and GLULX.
    first_word = WordValue(1, parse2);

    ! Take care of "all", because that does something too clever here to do
    ! later on:

    if (first_word == ALL1__WD or ALL2__WD or ALL3__WD or ALL4__WD or ALL5__WD) {
        if (context == MULTI_TOKEN or MULTIHELD_TOKEN or MULTIEXCEPT_TOKEN or MULTIINSIDE_TOKEN) {
            l = multiple_object-->0;
            for (i=0 : i<number_matched && l+i<63 : i++) {
                k = match_list-->i;
                multiple_object-->(i+1+l) = k;
            }
            multiple_object-->0 = i+l;
            rtrue;
        }
        L__M(##Miscellany, 47);
        jump WhichOne;
    }

    ! If the first word of the reply can be interpreted as a verb, then
    ! assume that the player has ignored the question and given a new
    ! command altogether.
    ! (This is one time when it's convenient that the directions are
    ! not themselves verbs - thus, "north" as a reply to "Which, the north
    ! or south door" is not treated as a fresh command but as an answer.)

    #Ifdef LanguageIsVerb;
    if (first_word == 0) {
        j = wn; first_word = LanguageIsVerb(buffer2, parse2, 1); wn = j;
    }
    #Endif; ! LanguageIsVerb
     if (first_word) {
         if (((first_word->#dict_par1) & DICT_VERB) && ~~LanguageVerbMayBeName(first_word)) {
             CopyBuffer(buffer, buffer2);
             return REPARSE_CODE;
         }
         if (NumberWords(parse2) > 2) {
             j = WordValue(2, parse2);
             k = WordValue(3, parse2);
             if (j == ',//' && k && (k->#dict_par1) & DICT_VERB) {
                 CopyBuffer(buffer, buffer2);
                 return REPARSE_CODE;
             }
         }
     }

    ! Now we insert the answer into the original typed command, as
    ! words additionally describing the same object
    ! (eg, > take red button
    !      Which one, ...
    !      > music
    ! becomes "take music red button".  The parser will thus have three
    ! words to work from next time, not two.)

    #Ifdef TARGET_ZCODE;
    k = WordAddress(match_from) - buffer; l=buffer2->1+1;
    for (j=buffer + buffer->0 - 1 : j>=buffer+k+l : j--) j->0 = 0->(j-l);
    for (i=0 : i<l : i++) buffer->(k+i) = buffer2->(2+i);
    buffer->(k+l-1) = ' ';
    buffer->1 = buffer->1 + l;
    if (buffer->1 >= (buffer->0 - 1)) buffer->1 = buffer->0;
    #Ifnot; ! TARGET_GLULX
    k = WordAddress(match_from) - buffer;
    l = (buffer2-->0) + 1;
    for (j=buffer+INPUT_BUFFER_LEN-1 : j>=buffer+k+l : j--) j->0 = j->(-l);
    for (i=0 : i<l : i++) buffer->(k+i) = buffer2->(WORDSIZE+i);
    buffer->(k+l-1) = ' ';
    buffer-->0 = buffer-->0 + l;
    if (buffer-->0 > (INPUT_BUFFER_LEN-WORDSIZE)) buffer-->0 = (INPUT_BUFFER_LEN-WORDSIZE);
    #Endif; ! TARGET_

    ! Having reconstructed the input, we warn the parser accordingly
    ! and get out.

    reparsing = true; return REPARSE_CODE;

    ! Now we come to the question asked when the input has run out
    ! and can't easily be guessed (eg, the player typed "take" and there
    ! were plenty of things which might have been meant).

  .Incomplete;

    if (context == CREATURE_TOKEN) L__M(##Miscellany, 48);
    else                           L__M(##Miscellany, 49);

    #Ifdef TARGET_ZCODE;
    for (i=2 : i<INPUT_BUFFER_LEN : i++) buffer2->i=' ';
    #Endif; ! TARGET_ZCODE
    answer_words = Keyboard(buffer2, parse2);
 
    first_word = WordValue(1, parse2);
    #Ifdef LanguageIsVerb;
    if (first_word==0) {
        j = wn; first_word=LanguageIsVerb(buffer2, parse2, 1); wn = j;
    }
    #Endif; ! LanguageIsVerb

    ! Once again, if the reply looks like a command, give it to the
    ! parser to get on with and forget about the question...

     if (first_word) {
         if ((first_word->#dict_par1) & DICT_VERB) {
             CopyBuffer(buffer, buffer2);
             return REPARSE_CODE;
         }
         if (NumberWords(parse2) > 2) {
             j = WordValue(2, parse2);
             k = WordValue(3, parse2);
             if (j == ',//' && k && (k->#dict_par1) & DICT_VERB) {
                 CopyBuffer(buffer, buffer2);
                 return REPARSE_CODE;
             }
         }
     }

    ! ...but if we have a genuine answer, then:
    !
    ! (1) we must glue in text suitable for anything that's been inferred.

    if (inferfrom ~= 0) {
        for (j=inferfrom : j<pcount : j++) {
            if (pattern-->j == PATTERN_NULL) continue;
            #Ifdef TARGET_ZCODE;
            i = 2+buffer->1; (buffer->1)++; buffer->(i++) = ' ';
            #Ifnot; ! TARGET_GLULX
            i = WORDSIZE + buffer-->0;
            (buffer-->0)++; buffer->(i++) = ' ';
            #Endif; ! TARGET_

            #Ifdef DEBUG;
            if (parser_trace >= 5) print "[Gluing in inference with pattern code ", pattern-->j, "]^";
            #Endif; ! DEBUG

            ! Conveniently, parse2-->1 is the first word in both ZCODE and GLULX.

            parse2-->1 = 0;

            ! An inferred object.  Best we can do is glue in a pronoun.
            ! (This is imperfect, but it's very seldom needed anyway.)

            if (pattern-->j >= 2 && pattern-->j < REPARSE_CODE) {
                ! was the inference made from some noun words?
                ! In which case, we can infer again.
                if ((WordValue(NumberWords())->#dict_par1) & DICT_NOUN) continue;
                PronounNotice(pattern-->j);
                for (k=1 : k<=LanguagePronouns-->0 : k=k+3)
                    if (pattern-->j == LanguagePronouns-->(k+2)) {
                        parse2-->1 = LanguagePronouns-->k;
                        #Ifdef DEBUG;
                        if (parser_trace >= 5) print "[Using pronoun '", (address) parse2-->1, "']^";
                        #Endif; ! DEBUG
                        break;
                    }
            }
            else {
                ! An inferred preposition.
                parse2-->1 = No__Dword(pattern-->j - REPARSE_CODE);
                #Ifdef DEBUG;
                if (parser_trace >= 5) print "[Using preposition '", (address) parse2-->1, "']^";
                #Endif; ! DEBUG
            }

            ! parse2-->1 now holds the dictionary address of the word to glue in.

            if (parse2-->1 ~= 0) {
                k = buffer + i;
                #Ifdef TARGET_ZCODE;
                @output_stream 3 k;
                 print (address) parse2-->1;
                @output_stream -3;
                k = k-->0;
                for (l=i : l<i+k : l++) buffer->l = buffer->(l+2);
                i = i + k; buffer->1 = i-2;
                #Ifnot; ! TARGET_GLULX
                k = PrintAnyToArray(buffer+i, INPUT_BUFFER_LEN-i, parse2-->1);
                i = i + k; buffer-->0 = i - WORDSIZE;
                #Endif; ! TARGET_
            }
        }
    }

    ! (2) we must glue the newly-typed text onto the end.

    #Ifdef TARGET_ZCODE;
    i = 2+buffer->1; (buffer->1)++; buffer->(i++) = ' ';
    for (j=0 : j<buffer2->1 : i++,j++) {
        buffer->i = buffer2->(j+2);
        (buffer->1)++;
        if (buffer->1 == INPUT_BUFFER_LEN) break;
    }
    #Ifnot; ! TARGET_GLULX
    i = WORDSIZE + buffer-->0;
    (buffer-->0)++; buffer->(i++) = ' ';
    for (j=0 : j<buffer2-->0 : i++,j++) {
        buffer->i = buffer2->(j+WORDSIZE);
        (buffer-->0)++;
        if (buffer-->0 == INPUT_BUFFER_LEN) break;
    }
    #Endif; ! TARGET_

    ! (3) we fill up the buffer with spaces, which is unnecessary, but may
    !     help incorrectly-written interpreters to cope.

    #Ifdef TARGET_ZCODE;
    for (: i<INPUT_BUFFER_LEN : i++) buffer->i = ' ';
    #Endif; ! TARGET_ZCODE

    reparsing = true; return REPARSE_CODE;

]; ! end of NounDomain

! ----------------------------------------------------------------------------
!  The Adjudicate routine tries to see if there is an obvious choice, when
!  faced with a list of objects (the match_list) each of which matches the
!  player's specification equally well.
!
!  To do this it makes use of the context (the token type being worked on).
!  It counts up the number of obvious choices for the given context
!  (all to do with where a candidate is, except for 6 (animate) which is to
!  do with whether it is animate or not);
!
!  if only one obvious choice is found, that is returned;
!
!  if we are in indefinite mode (don't care which) one of the obvious choices
!    is returned, or if there is no obvious choice then an unobvious one is
!    made;
!
!  at this stage, we work out whether the objects are distinguishable from
!    each other or not: if they are all indistinguishable from each other,
!    then choose one, it doesn't matter which;
!
!  otherwise, 0 (meaning, unable to decide) is returned (but remember that
!    the equivalence classes we've just worked out will be needed by other
!    routines to clear up this mess, so we can't economise on working them
!    out).
!
!  Returns -1 if an error occurred.
! ----------------------------------------------------------------------------

Constant SCORE__CHOOSEOBJ = 1000;
Constant SCORE__IFGOOD = 500;
Constant SCORE__UNCONCEALED = 100;
Constant SCORE__BESTLOC = 60;
Constant SCORE__NEXTBESTLOC = 40;
Constant SCORE__NOTCOMPASS = 20;
Constant SCORE__NOTSCENERY = 10;
Constant SCORE__NOTACTOR = 5;
Constant SCORE__GNA = 1;
Constant SCORE__DIVISOR = 20;

[ Adjudicate context i j k good_flag good_ones last n flag offset sovert;

    #Ifdef DEBUG;
    if (parser_trace >= 4) {
        print "   [Adjudicating match list of size ", number_matched, " in context ", context, "]^";
        print "   ";
        if (indef_mode) {
            print "indefinite type: ";
            if (indef_type & OTHER_BIT)  print "other ";
            if (indef_type & MY_BIT)     print "my ";
            if (indef_type & THAT_BIT)   print "that ";
            if (indef_type & PLURAL_BIT) print "plural ";
#IFNDEF NO_DARKNESS;
            if (indef_type & LIT_BIT)    print "lit ";
            if (indef_type & UNLIT_BIT)  print "unlit ";
#ENDIF;
            if (indef_owner ~= 0) print "owner:", (name) indef_owner;
            new_line;
            print "   number wanted: ";
            if (indef_wanted == 100) print "all"; else print indef_wanted;
            new_line;
            print "   most likely GNAs of names: ", indef_cases, "^";
        }
        else print "definite object^";
    }
    #Endif; ! DEBUG

    j = number_matched-1; good_ones = 0; last = match_list-->0;
    for (i=0 : i<=j : i++) {
        n = match_list-->i;
        match_scores-->i = 0;

        good_flag = false;

        switch (context) {
          HELD_TOKEN, MULTIHELD_TOKEN:
            if (parent(n) == actor) {
		if (~~(n has anatomy && n.attachedtoparent)) good_flag = true;
		}
          MULTIEXCEPT_TOKEN:
            if (advance_warning == -1) {
                good_flag = true;
            }
            else {
                if (n ~= advance_warning) good_flag = true;
            }
          MULTIINSIDE_TOKEN:
            if (advance_warning == -1) {
                if (parent(n) ~= actor) good_flag = true;
            }
            else {
                if (n in advance_warning) good_flag = true;
            }
          CREATURE_TOKEN:
            if (CreatureTest(n) == 1 && n ~= actor) good_flag = true;
          default:
            good_flag = true;
        }

        if (good_flag) {
            match_scores-->i = SCORE__IFGOOD;
            good_ones++; last = n;
        }
    }
    if (good_ones == 1 && ~~indef_wanted) return last;
    if (action_to_be == ##Drop) {
	if (j > 0 && good_ones == 0) original_noun = -1;
	original_noun = 1;
    }

    n = 0;

    ! If there is ambiguity about what was typed, but it definitely wasn't
    ! animate as required, then return anything; higher up in the parser
    ! a suitable error will be given.  (This prevents a question being asked.)

    ! if (context == CREATURE_TOKEN && good_ones == 0) return match_list-->0;

    if (indef_mode == 0) indef_type = 0;

    ScoreMatchL(context);
    if (number_matched == 0) return -1;

    if (indef_mode == 0) {
        !  Is there now a single highest-scoring object?
        i = SingleBestGuess();
        if (i >= 0) {

            #Ifdef DEBUG;
            if (parser_trace >= 4) print "   Single best-scoring object returned.]^";
            #Endif; ! DEBUG
            return i;
        }
    }

    if (indef_mode == 1 && indef_type & PLURAL_BIT ~= 0) {
        if (context ~= MULTI_TOKEN or MULTIHELD_TOKEN or MULTIEXCEPT_TOKEN
                     or MULTIINSIDE_TOKEN) {
            etype = MULTI_PE;
            return -1;
        }

        i = 0; offset = multiple_object-->0; sovert = -1;
        
        for (j=BestGuess() : j~=-1 && i<indef_wanted && i+offset<63 : j=BestGuess()) {
            flag = 0;

            if (sovert == -1) sovert = bestguess_score/SCORE__DIVISOR;
            else {
                if (indef_wanted == 100 && bestguess_score/SCORE__DIVISOR < sovert)
                   flag = 0;
            }

            #IFDEF TRACK_LOCATIONS;
	    if (AbsentItem(j)) { flag = 1; jump BasicRuleDone; }
            #ENDIF;
	    if (j hasnt concealed or scenery or static or animate) {

	    flag = 1;

	    if (~~parent(j)) flag = 0;
	    else {

		k = Owner(j, 1);

		if (context == MULTIEXCEPT_TOKEN) {
			if (IndirectlyContains(advance_warning, j) == true) flag = 0;
			if (parent(j) == actor) flag = 1;
		}

		if (context == MULTIHELD_TOKEN) {
			if (parent(j) == actor) flag = 1;
				else if (indef_wanted == 100 && action_to_be == ##Drop)
					{ best_etype = NOTHELD_PE; flag = 0; }
		}

	    #Ifdef COMPLEXCLOTHING;
		if (j ofclass Wearable && j.worn_by ~= 0) flag = 0;
	    #Ifnot;
		if (j has worn) flag = 0;
	    #Endif;

	    	if (k has animate && k ~= actor)
			flag = 0;
		if (k == actor && action_to_be == ##Take && indef_wanted)
			flag = 0;

! The following chunk makes sure that GET ALL IN/ON/etc. works correctly.

		if (indef_specific == 0 && action_to_be == ##Take && j in advance_warning) flag = 1;

!Get all from ground FIXME
!if (action_to_be == ##Take && grammar_line < 5 && parent(j) ~= actor.location)
!flag = 0;

		if (j.attachedtoparent == true)
			flag = 0;
		}
	    }
	    .BasicRuleDone;
            k = ChooseObjects(j, flag);
	#IFDEF USE_LIBRARY_EXTENSIONS;
            if (n == 0) n = LibraryExtensions.RunWhile(ext_chooseobjects, 0, j, flag);
	#ENDIF;
            switch (k) {
              2: flag = 0;  ! forcing rejection
              1: flag = 1;  ! forcing acceptance
             !0:            ! going with parser's decision
            }
            if (flag == 1) {
                if (~~(action_to_be == ##Drop && indef_wanted < 100 && j notin actor))
			n++;
		    i++;
		    multiple_object-->(i+offset) = j;
                #Ifdef DEBUG;
                if (parser_trace >= 4) print "   Accepting it^";
                #Endif; ! DEBUG
            }
            else {
                i = i;
                #Ifdef DEBUG;
                if (parser_trace >= 4) print "   Rejecting it^";
                #Endif; ! DEBUG
            }
        } ! End of for loop
        if (n < indef_wanted && indef_wanted < 100) {
		multi_wanted = indef_wanted; multi_had = n;
	  }
        multiple_object-->0 = i+offset;
        multi_context = context;
        #Ifdef DEBUG;
        if (parser_trace >= 4)
            print "   Made multiple object of size ", i, "]^";
        #Endif; ! DEBUG
        return 1;
    }

    for (i=0 : i<number_matched : i++) match_classes-->i = 0;

    n = 1;
    for (i=0 : i<number_matched : i++)
        if (match_classes-->i == 0) {
            match_classes-->i = n++; flag = 0;
            for (j=i+1 : j<number_matched : j++)
                if (match_classes-->j == 0 && Identical(match_list-->i, match_list-->j) == 1) {
                    flag=1;
                    match_classes-->j = match_classes-->i;
                }
            if (flag == 1) match_classes-->i = 1-n;
        }
     n--; number_of_classes = n;

    #Ifdef DEBUG;
    if (parser_trace >= 4) {
        print "   Grouped into ", n, " possibilities by name:^";
        for (i=0 : i<number_matched : i++)
            if (match_classes-->i > 0)
                print "   ", (The) match_list-->i, " (", match_list-->i, ")  ---  group ",
                  match_classes-->i, "^";
    }
    #Endif; ! DEBUG

    if (indef_mode == 0) {
        if (n > 1) {
            k = -1;
            for (i=0 : i<number_matched : i++) {
                if (match_scores-->i > k) {
                    k = match_scores-->i;
                    j = match_classes-->i; j = j*j;
                    flag = 0;
                }
                else
                    if (match_scores-->i == k) {
                        if ((match_classes-->i) * (match_classes-->i) ~= j)
                            flag = 1;
                    }
            }

        if (flag) {
            #Ifdef DEBUG;
            if (parser_trace >= 4) print "   Unable to choose best group, so ask player.]^";
            #Endif; ! DEBUG
            return 0;
        }
        #Ifdef DEBUG;
        if (parser_trace >= 4) print "   Best choices are all from the same group.^";
        #Endif; ! DEBUG
        }
    }

    !  When the player is really vague, or there's a single collection of
    !  indistinguishable objects to choose from, choose the one the player
    !  most recently acquired, or if the player has none of them, then
    !  the one most recently put where it is.

    if (n == 1) dont_infer = true;
    return BestGuess();

]; ! Adjudicate

! ----------------------------------------------------------------------------
!  ReviseMulti  revises the multiple object which already exists, in the
!    light of information which has come along since then (i.e., the second
!    parameter).  It returns a parser error number, or else 0 if all is well.
!    This only ever throws things out, never adds new ones.
! ----------------------------------------------------------------------------

[ ReviseMulti second_p  i j low;

    #Ifdef DEBUG;
    if (parser_trace >= 4) print "   Revising multiple object list of size ", multiple_object-->0,
      " with 2nd ", (name) second_p, "^";
    #Endif; ! DEBUG

    if (multi_context == MULTIEXCEPT_TOKEN or MULTIINSIDE_TOKEN) {
        for (i=1,low=0 : i<=multiple_object-->0 : i++) {
            if ( (multi_context==MULTIEXCEPT_TOKEN && multiple_object-->i ~= second_p) ||
                 (multi_context==MULTIINSIDE_TOKEN && multiple_object-->i in second_p)) {
                low++;
                multiple_object-->low = multiple_object-->i;
            }
        }
        multiple_object-->0 = low;
    }

    if (multi_context == MULTI_TOKEN && action_to_be == ##Take) {
        for (i=1,low=0 : i<=multiple_object-->0 : i++) {
            j = multiple_object-->i;
            #IFDEF TRACK_LOCATIONS;
            if (j.visibility_ceiling == actor.visibility_ceiling || AbsentItem(j)) low++;
            #IFNOT;
            if (j.visibility_ceiling == actor.visibility_ceiling) low++;
            #ENDIF;
        }
        #Ifdef DEBUG;
        if (parser_trace >= 4) print "   Token 2 plural case: number with actor ", low, "^";
        #Endif; ! DEBUG
        if (take_all_rule == 2 || low > 0) {
            for (i=1,low=0 : i<=multiple_object-->0 : i++) {
                j = multiple_object-->i;
                #IFDEF TRACK_LOCATIONS;
                if (j.visibility_ceiling == actor.visibility_ceiling || AbsentItem(j)) {
                #IFNOT;
                if (j.visibility_ceiling == actor.visibility_ceiling) {
                #ENDIF;
                    low++;
                    multiple_object-->low = multiple_object-->i;
                }
            }
            multiple_object-->0 = low;
        }
    }

    i = multiple_object-->0;
    #Ifdef DEBUG;
    if (parser_trace >= 4) print "   Done: new size ", i, "^";
    #Endif; ! DEBUG
    if (i == 0) return NOTHING_PE;
    return 0;
];

! ----------------------------------------------------------------------------
!  ScoreMatchL scores the match list for quality in terms of what the
!  player has vaguely asked for.  Points are awarded for conforming with
!  requirements like "my", and so on.  Remove from the match list any
!  entries which fail the basic requirements of the descriptors.
! ----------------------------------------------------------------------------

[ ScoreMatchL context its_owner its_score obj i j threshold met a_s l_s;
    if (indef_type & OTHER_BIT ~= 0) threshold++;
    if (indef_type & MY_BIT ~= 0)    threshold++;
    if (indef_type & THAT_BIT ~= 0)  threshold++;
#IFNDEF NO_DARKNESS;
    if (indef_type & LIT_BIT ~= 0)   threshold++;
    if (indef_type & UNLIT_BIT ~= 0) threshold++;
#ENDIF;
    if (indef_owner ~= nothing)      threshold++;

    #Ifdef DEBUG;
    if (parser_trace >= 4) print "   Scoring match list: indef mode ", indef_mode, " type ",
      indef_type, ", satisfying ", threshold, " requirements:^";
    #Endif; ! DEBUG

    if (action_to_be ~= ##Take) a_s = SCORE__NEXTBESTLOC;
    l_s = SCORE__BESTLOC;
    if (context == HELD_TOKEN or MULTIHELD_TOKEN or MULTIEXCEPT_TOKEN) {
        a_s = SCORE__BESTLOC; l_s = SCORE__NEXTBESTLOC;
    }

    for (i=0 : i<number_matched : i++) {
        obj = match_list-->i; its_owner = parent(obj); its_score=0; met=0;

        !      if (indef_type & OTHER_BIT ~= 0
        !          &&  obj ~= itobj or himobj or herobj) met++;
        if (indef_type & MY_BIT ~= 0 && its_owner == actor) met++;
        if (indef_type & THAT_BIT ~= 0 && its_owner == actor.location) met++;
#Ifndef NO_DARKNESS;
        if (indef_type & LIT_BIT ~= 0 && obj has light) met++;
        if (indef_type & UNLIT_BIT ~= 0 && obj hasnt light) met++;
#Endif;
        if (indef_owner ~= 0 && its_owner == indef_owner) met++;

        if (met < threshold) {
            #Ifdef DEBUG;
            if (parser_trace >= 4) print "   ", (The) match_list-->i, " (", match_list-->i, ") in ",
              (the) its_owner, " is rejected (doesn't match descriptors)^";
            #Endif; ! DEBUG
            match_list-->i = -1;
        }
        else {
            its_score = 0;
		if ((context == CREATURE_TOKEN && obj hasnt animate)
		    || (context == HELD_TOKEN or MULTIHELD_TOKEN && IndirectlyContains(player, obj, 8) == 0 or 8))
			jump Restart;

            if (obj hasnt concealed) its_score = SCORE__UNCONCEALED;

            if (its_owner == actor) its_score = its_score + a_s;
            else
                if (its_owner == actor.location) its_score = its_score + l_s;
                else
                    if (its_owner ~= compass) its_score = its_score + SCORE__NOTCOMPASS;

		if (action_to_be == ##Eat && obj has edible)
			its_score = its_score + SCORE__NOTSCENERY;

		if (action_to_be == ##Take && obj has scenery or static)
			its_score = its_score - SCORE__NOTSCENERY;

            j = ChooseObjects(obj, 2);
	#IFDEF USE_LIBRARY_EXTENSIONS;
            if (j == 0) j = LibraryExtensions.RunAll(ext_chooseobjects, obj, 2);
	#ENDIF;
            its_score = its_score + SCORE__CHOOSEOBJ * j;

            if (obj ~= actor) its_score = its_score + SCORE__NOTACTOR;

            !   A small bonus for having the correct GNA,
            !   for sorting out ambiguous articles and the like.

            if (indef_cases & (PowersOfTwo_TB-->(GetGNAOfObject(obj))))
                its_score = its_score + SCORE__GNA;

            match_scores-->i = match_scores-->i + its_score;
            #Ifdef DEBUG;
            if (parser_trace >= 4) print "     ", (The) match_list-->i, " (", match_list-->i,
              ") in ", (the) its_owner, " : ", match_scores-->i, " points^";
            #Endif; ! DEBUG
		.Restart;
        }
     }

    for (i=0 : i<number_matched : i++) {
        while (match_list-->i == -1) {
            if (i == number_matched-1) { number_matched--; break; }
            for (j=i : j<number_matched-1 : j++) {
                match_list-->j = match_list-->(j+1);
                match_scores-->j = match_scores-->(j+1);
            }
            number_matched--;
        }
    }
];

! ----------------------------------------------------------------------------
!  BestGuess makes the best guess it can out of the match list, assuming that
!  everything in the match list is textually as good as everything else;
!  however it ignores items marked as -1, and so marks anything it chooses.
!  It returns -1 if there are no possible choices.
! ----------------------------------------------------------------------------

[ BestGuess  earliest its_score best i;
    earliest = 0; best = -1;
    for (i=0 : i<number_matched : i++) {
        if (match_list-->i >= 0) {
            its_score = match_scores-->i;
            if (its_score > best) { best = its_score; earliest = i; }
        }
    }
    #Ifdef DEBUG;
    if (parser_trace >= 4)
      if (best < 0) print "   Best guess ran out of choices^";
      else print "   Best guess ", (the) match_list-->earliest, " (", match_list-->earliest, ")^";
    #Endif; ! DEBUG
    if (best < 0) return -1;
    i = match_list-->earliest;
    match_list-->earliest = -1;
    bestguess_score = best;
    return i;
];

! ----------------------------------------------------------------------------
!  SingleBestGuess returns the highest-scoring object in the match list
!  if it is the clear winner, or returns -1 if there is no clear winner
! ----------------------------------------------------------------------------

[ SingleBestGuess  earliest its_score best i;
    earliest = -1; best = -1000;
    for (i=0 : i<number_matched : i++) {
        its_score = match_scores-->i;
        if (its_score == best) earliest = -1;
        if (its_score > best) { best = its_score; earliest = match_list-->i; }
    }
    bestguess_score = best;
    return earliest;
];

! ----------------------------------------------------------------------------
!  Identical decides whether or not two objects can be distinguished from
!  each other by anything the player can type.  If not, it returns true.
! ----------------------------------------------------------------------------

[ Identical o1 o2 i j;
    if (o1 == o2) rtrue;  ! This should never happen, but to be on the safe side
    if (o1 == 0 || o2 == 0) rfalse;  ! Similarly
    if (parent(o1) ~= parent(o2)) rfalse;
    if (parent(o1) == compass || parent(o2) == compass) rfalse; ! Saves time

    !  What complicates things is that o1 or o2 might have a parsing routine,
    !  so the parser can't know from here whether they are or aren't the same.
    !  If they have different parsing routines, we simply assume they're
    !  different.  If they have the same routine (which they probably got from
    !  a class definition) then the decision process is as follows:
    !
    !     the routine is called (with self being o1, not that it matters)
    !       with noun and second being set to o1 and o2, and action being set
    !       to the fake action TheSame.  If it returns -1, they are found
    !       identical; if -2, different; and if >=0, then the usual method
    !       is used instead.

    if (o1.parse_name ~= 0 || o2.parse_name ~= 0) {
      if (o1.parse_name ~= o2.parse_name) rfalse;
      parser_action = ##TheSame; parser_one = o1; parser_two = o2;
      j = wn; i = RunRoutines(o1,parse_name); wn = j;
      if (i == -1) rtrue;
      if (i == -2) rfalse;
    }

    !  This is the default algorithm: do they have the same words in their
    !  "name" and "adjective" properties.  (Note that the following allows
    !  for repeated words and words in different orders.)

    if (TestPropsMatch(o1, o2, name)
        && TestPropsMatch(o1, o2, adjective)) rtrue;
    rfalse;
];

[ TestPropsMatch x y pr     ad1 ad2 l1 l2 c;

    ad1 = x.&pr;      ad2 = y.&pr;
    l1  = x.#pr / WORDSIZE;  l2  = y.#pr / WORDSIZE;

    for (c = 0:c < l1:c++)
        if (WordInProperty(ad1-->c, y, pr) == 0) rfalse;
    for (c = 0:c < l2:c++)
        if (WordInProperty(ad2-->c, x, pr) == 0) rfalse;
];

! ----------------------------------------------------------------------------
!  The CantSee routine returns a good error number for the situation where
!  the last word looked at didn't seem to refer to any object in context.
!
!  The idea is that: if the actor is in a location (but not inside something
!  like, for instance, a tank which is in that location) then an attempt to
!  refer to one of the words listed as meaningful-but-irrelevant there
!  will cause "you don't need to refer to that in this game" rather than
!  "no such thing" or "what's 'it'?".
!  (The advantage of not having looked at "irrelevant" local nouns until now
!  is that it stops them from clogging up the ambiguity-resolving process.
!  Thus game objects always triumph over scenery.)
! ----------------------------------------------------------------------------

[ CantSee i w e;

    if (scope_token ~= 0) {
        scope_error = scope_token;
        return ASKSCOPE_PE;
    }

    wn--;
    w = NextWord();
    e = CANTSEE_PE;
    if (w == pronoun_word) {
        pronoun__word = pronoun_word; pronoun__obj = pronoun_obj;
        e = ITGONE_PE;
    }
    i = actor.location;

    wn--;
    if (i) if (i.visited == true && Refers(i,wn) == 1) e = SCENERY_PE;
    else {
        Descriptors();  ! skip past THE etc
        if (i.visited == true && Refers(i,wn) == 1) e = SCENERY_PE;
    }

    if (saved_ml) saved_oops = num_desc + match_from + saved_ml;
	else saved_oops = num_desc + match_from + match_length;

    wn++;

    if (etype > e) return etype;
    return e;
];

! ----------------------------------------------------------------------------
!  The MultiAdd routine adds object "o" to the multiple-object-list.
!
!  This is only allowed to hold 63 objects at most, at which point it ignores
!  any new entries (and sets a global flag so that a warning may later be
!  printed if need be).
! ----------------------------------------------------------------------------

[ MultiAdd o i j;
    i = multiple_object-->0;
    if (i == 63) { toomany_flag = 1; rtrue; }
    for (j=1 : j<=i : j++)
        if (o == multiple_object-->j) rtrue;
    i++;
    multiple_object-->i = o;
    multiple_object-->0 = i;
];

! ----------------------------------------------------------------------------
!  The MultiSub routine deletes object "o" from the multiple-object-list.
!
!  It returns 0 if the object was there in the first place, and 9 (because
!  this is the appropriate error number in Parser()) if it wasn't.
! ----------------------------------------------------------------------------

[ MultiSub o i j k et;
    i = multiple_object-->0; et = 0;
    for (j=1 : j<=i : j++)
        if (o == multiple_object-->j) {
            for (k=j : k<=i : k++)
                multiple_object-->k = multiple_object-->(k+1);
            multiple_object-->0 = --i;
            return et;
        }
    et = 9; return et;
];

! ----------------------------------------------------------------------------
!  The MultiFilter routine goes through the multiple-object-list and throws
!  out anything without the given attribute "attr" set.
! ----------------------------------------------------------------------------

[ MultiFilter attr  i j o;

  .MFiltl;

    i = multiple_object-->0;
    for (j=1 : j<=i : j++) {
        o = multiple_object-->j;
        if (o hasnt attr) {
            MultiSub(o);
            jump Mfiltl;
        }
    }
];

! ----------------------------------------------------------------------------
!  The UserFilter routine consults the user's filter (or checks on attribute)
!  to see what already-accepted nouns are acceptable
! ----------------------------------------------------------------------------

[ UserFilter obj;
    if (token_filter > 0 && token_filter < 49) {
        if (obj has (token_filter-1)) rtrue;
        rfalse;
    }
    noun = obj;
    return indirect(token_filter);
];

! ----------------------------------------------------------------------------
!  MoveWord copies word at2 from parse buffer b2 to word at1 in "parse"
!  (the main parse buffer)
! ----------------------------------------------------------------------------

#Ifdef TARGET_ZCODE;

[ MoveWord at1 b2 at2 x y;
    x = at1*2-1; y = at2*2-1;
    parse-->x++ = b2-->y++;
    parse-->x = b2-->y;
];

#Ifnot; ! TARGET_GLULX

[ MoveWord at1 b2 at2 x y;
    x = at1*3-2; y = at2*3-2;
    parse-->x++ = b2-->y++;
    parse-->x++ = b2-->y++;
    parse-->x = b2-->y;
];

#Endif; ! TARGET_

! ----------------------------------------------------------------------------
!  SearchScope  domain context
!
!  Works out what objects are in scope (possibly asking an outside routine),
!  but does not look at anything the player has typed.
! ----------------------------------------------------------------------------

[ SearchScope domain context i;

    ! Everything is in scope to the debugging commands

    #Ifdef DEBUG;
    if (scope_reason == PARSING_REASON
        && LanguageVerbIsDebugging(verb_word)) {
        objectloop (i) {
            if (i ofclass Object)
                PlaceInScope(i);
	}
	rtrue;
    }
    #Endif; ! DEBUG

    ! First, a scope token gets priority here:

    if (scope_token ~= 0) {
        scope_stage = 2;
        if (indirect(scope_token) ~= 0) rtrue;
    }

    ! Next, call any user-supplied routine adding things to the scope,
    ! which may circumvent the usual routines altogether if they return true:

    if (actor == domain) {
	i = InScope(actor);
	#IFDEF USE_LIBRARY_EXTENSIONS;
	if ( == false) s = LibraryExtensions.RunWhile(ext_inscope, false, actor);
	#ENDIF;
	if (i) rtrue;
    }

    ! Otherwise, everything visible is in scope.

    objectloop (i && IsVisible(i, domain)) {
	ScopeWithin_O(i, context);
    }

    ! Players may refer to items in their (for example) closed rucksacks.

    if (player == domain) objectloop (i && HiddenInventory(i)) {
	PlaceInScope(i, context);
    }

    ! If nothing matched, and TRACK_LOCATIONS is enabled, try absent
    ! objects.

    #IFDEF TRACK_LOCATIONS;
    if (scope_reason == PARSING_REASON or TALKING_REASON && (number_matched == 0 || match_length < num_words && indef_wanted == 0)) {
    #IFDEF DEBUG;
    if (parser_trace >= 2) print "  [Trying absent items.]^";
    #ENDIF;
	objectloop (i provides last_seen && AbsentItem(i)) {
		PlaceInScope(i);
	}
    }
    #ENDIF;
];

! ----------------------------------------------------------------------------
!  PlaceInScope is provided for routines outside the library, and is not
!  called within the parser (except for debugging purposes).
! ----------------------------------------------------------------------------

[ PlaceInScope thing;
    if (scope_reason ~= PARSING_REASON or TALKING_REASON) {
        DoScopeAction(thing);
	#IFDEF TRACK_LOCATIONS; Recognize(actor, thing); #ENDIF;
	rtrue;
    }
    wn = match_from; TryGivenObject(thing); placed_in_flag = 1;
];

! ----------------------------------------------------------------------------
!  DoScopeAction
! ----------------------------------------------------------------------------

[ DoScopeAction thing s p1;
    s = scope_reason; p1 = parser_one;
    #Ifdef DEBUG;
    if (parser_trace >= 6)
        print "[DSA on ", (the) thing, " with reason = ", scope_reason,
            " p1 = ", parser_one, " p2 = ", parser_two, "]^";
    #Endif; ! DEBUG
    switch (scope_reason) {

      REACT_BEFORE_REASON:
        if (thing.react_before == 0 or NULL) return;
        #Ifdef DEBUG;
        if (parser_trace >= 2)
              print "[Considering react_before for ", (the) thing, "]^";
        #Endif; ! DEBUG
        if (parser_one == 0) parser_one = RunRoutines(thing, react_before);
	if (parser_one > 0) failed_action = 1;

      REACT_MIDDLE_REASON:
        if (thing.react_middle == 0 or NULL) return;
        #Ifdef DEBUG;
        if (parser_trace >= 2)
            print "[Considering react_middle for ", (the) thing, "]^";
        #Endif; ! DEBUG
        if (parser_one == 0) parser_one = RunRoutines(thing, react_middle);

      REACT_AFTER_REASON:
        if (thing.react_after == 0 or NULL) return;
        #Ifdef DEBUG;
        if (parser_trace >= 2)
            print "[Considering react_after for ", (the) thing, "]^";
        #Endif; ! DEBUG
        if (parser_one == 0) parser_one = RunRoutines(thing, react_after);

      REACT_FINAL_REASON:
        if (thing.react_final == 0 or NULL) return;
        #Ifdef DEBUG;
        if (parser_trace >= 2)
            print "[Considering react_final for ", (the) thing, "]^";
        #Endif; ! DEBUG
        if (parser_one == 0) parser_one = RunRoutines(thing, react_final);

      EACH_TURN_REASON:
        if (thing.each_turn == 0 or NULL) return;
        #Ifdef DEBUG;
        if (parser_trace >= 2)
              print "[Considering each_turn for ", (the) thing, "]^";
        #Endif; ! DEBUG
        PrintOrRun(thing, each_turn);

      TESTSCOPE_REASON:
	if (thing == parser_one) parser_two = 1;

      LOOPOVERSCOPE_REASON:
        indirect(parser_one, thing);
        parser_one = p1;
    }
    scope_reason = s;
];

[ ScopeWithin_O domain context i ad n;

    ! If the scope reason is unusual, don't parse.

    if (scope_reason ~= PARSING_REASON or TALKING_REASON) {
        DoScopeAction(domain);
        jump DontAccept;
    }

    ! "it" or "them" matches to the it-object only.  (Note that (1) this means
    ! that "it" will only be understood if the object in question is still
    ! in context, and (2) only one match can ever be made in this case.)

    if (match_from <= num_words) {  ! If there's any text to match, that is
        wn = match_from;
        i = NounWord();
        if (i == 1 && player == domain) MakeMatch(domain, 1);
        if (i == 2 && actor == domain) MakeMatch(domain, 1);
        if (i >= 2 && i < 128 && (LanguagePronouns-->i == domain)) MakeMatch(domain, 1);
    }

    ! Construing the current word as the start of a noun, can it refer to the
    ! object?

    wn = match_from;
    if (TryGivenObject(domain) > 0)
        if (indef_nspec_at > 0 && match_from ~= indef_nspec_at) {
            ! This case arises if the player has typed a number in
            ! which is hypothetically an indefinite descriptor:
            ! e.g. "take two clubs".  We have just checked the object
            ! against the word "clubs", in the hope of eventually finding
            ! two such objects.  But we also backtrack and check it
            ! against the words "two clubs", in case it turns out to
            ! be the 2 of Clubs from a pack of cards, say.  If it does
            ! match against "two clubs", we tear up our original
            ! assumption about the meaning of "two" and lapse back into
            ! definite mode.

            wn = indef_nspec_at;
            if (TryGivenObject(domain) > 0) {
                match_from = indef_nspec_at;
                ResetDescriptors();
            }
            wn = match_from;
        }

  .DontAccept;

    ! Drag any extras into context

    ad = domain.&add_to_scope;
    if (ad ~= 0) {

        ! Test if the property value is not an object.
        #Ifdef TARGET_ZCODE;
        i = (UnsignedCompare(ad-->0, top_object) > 0);
        #Ifnot; ! TARGET_GLULX
        i = (((ad-->0)->0) ~= $70);
        #Endif; ! TARGET_

        if (i) {
            ats_flag = 2+context;
            RunRoutines(domain, add_to_scope);
            ats_flag = 0;
        }
        else {
            n = domain.#add_to_scope;
            for (i=0 : (WORDSIZE*i)<n : i++)
                if (ad-->i) {
			#IFDEF TRACK_LOCATIONS; Recognize(actor, ad-->i); #ENDIF;
			ScopeWithin_O(ad-->i, 0, context);
		}
        }
    }
];

[ AddToScope obj;
    if (ats_flag >= 2)
        ScopeWithin_O(obj, 0, ats_flag-2);
    if (ats_flag == 1) {
        if  (LightToSeeBy(actor) > 0) ats_hls = 1;
    }
];

! ----------------------------------------------------------------------------
!  MakeMatch looks at how good a match is.  If it's the best so far, then
!  wipe out all the previous matches and start a new list with this one.
!  If it's only as good as the best so far, add it to the list.
!  If it's worse, ignore it altogether.
!
!  The idea is that "red panic button" is better than "red button" or "panic".
!
!  number_matched (the number of words matched) is set to the current level
!  of quality.
!
!  We never match anything twice, and keep at most 64 equally good items.
! ----------------------------------------------------------------------------

[ MakeMatch obj quality i;
    #Ifdef DEBUG;
    if (parser_trace >= 6) print "    Match with quality ", quality, "^";
    #Endif; ! DEBUG
    if (token_filter ~= 0 && UserFilter(obj) == 0) {
        #Ifdef DEBUG;
       if (parser_trace >= 6) print "    Match filtered out: token filter ", token_filter, "^";
        #Endif; ! DEBUG
       rtrue;
    }
    if (quality < match_length) rtrue;

!   if (action_to_be == ##Drop && IndirectlyContains(actor, obj, 8) == 0 or 8
!	 && indef_wanted == 100) rtrue;

    if (quality > match_length) { match_length = quality; number_matched = 0; }
    else {
        if (2*number_matched >= MATCH_LIST_SIZE) rtrue;
        for (i=0 : i<number_matched : i++)
            if (match_list-->i == obj) rtrue;
    }
    match_list-->number_matched++ = obj;
    #Ifdef DEBUG;
    if (parser_trace >= 6) print "    ", (The) obj, " added to list^";
    #Endif; ! DEBUG
];

! ----------------------------------------------------------------------------
!  TryGivenObject tries to match as many words as possible in what has been
!  typed to the given object, obj.  If it manages any words matched at all,
!  it calls MakeMatch to say so, then returns the number of words (or 1
!  if it was a match because of inadequate input).
! ----------------------------------------------------------------------------

[ TryGivenObject obj flag threshold k w j p;

    #Ifdef DEBUG;
    if (parser_trace >= 5 && ~~flag) print "    Trying ", (the) obj, " (", obj, ") at word ", wn, "^";
    #Endif; ! DEBUG

    dict_flags_of_noun = 0;

!  If input has run out then always match, with only quality 0 (this saves
!  time).

    if (wn > num_words) {
        if (indef_mode ~= 0)
            dict_flags_of_noun = $$01110000;  ! Reject "plural" bit
        if (flag) rfalse;
	  MakeMatch(obj, 0);
        #Ifdef DEBUG;
        if (parser_trace >= 5) print "    Matched (0)^";
        #Endif; ! DEBUG
        return 1;
    }

!  Ask the object to parse itself if necessary, sitting up and taking notice
!  if it says the plural was used:

    if (obj.parse_name~=0) {
        parser_action = NULL; j = wn;
        k = RunRoutines(obj, parse_name);
        if (k > 0) {
            wn = j + k;

          .MMbyPN;

            ! The next two lines take care of position parsing.

            j = dict_flags_of_noun;
            p = AppendPreps(obj);
            dict_flags_of_noun = j;
            if (p) k = k + p;

            if (parser_action == ##PluralFound)
                dict_flags_of_noun = dict_flags_of_noun | DICT_PLUR;

            if (dict_flags_of_noun & DICT_PLUR) {
                if (~~allow_plurals) k = 0;
                else {
                    if (indef_mode == 0) {
                        indef_mode = 1; indef_type = 0; indef_wanted = 0;
                    }
                    indef_type = indef_type | PLURAL_BIT;
                    if (indef_wanted == 0) indef_wanted = 100;
                }
            }

            if (flag) return k;
            #Ifdef DEBUG;
            if (parser_trace >= 5) print "    Matched (", k, ")^";
            #Endif; ! DEBUG
            MakeMatch(obj, k);
            return k;
        }
        if (k == 0) jump NoWordsMatch;
    }

    parser_action = NULL;

    w = NounWord();

    if (w == 1 && player == obj) { k=1; jump MMbyPN; }
    if (w == 2 && actor == obj)  { k=1; jump MMbyPN; }

    if (w >= 2 && w < 128 && (LanguagePronouns-->w == obj)) { k = 1; jump MMbyPN; }

    j=wn--;
    threshold = ParseNoun(obj);
    #Ifdef DEBUG;
    if (threshold >= 0 && parser_trace >= 5 && ~~flag) print "    ParseNoun returned ", threshold, "^";
    #Endif; ! DEBUG

    if (threshold < 0) {
	#IFDEF USE_LIBRARY_EXTENSIONS;
        LibraryExtensions.ext_number_1 = wn;    ! Set the "between calls" functionality to
        LibraryExtensions.BetweenCalls = LibraryExtensions.RestoreWN;
        threshold = LibraryExtensions.RunWhile(ext_parsenoun, -1, obj);
        LibraryExtensions.BetweenCalls = 0;     ! Turn off the "between calls" functionality
	#ENDIF;
	wn = j;
    }
    if (threshold > 0) { k = threshold; jump MMbyPN; }

    if (threshold <= 0) {
	threshold = ParseObj(obj);
        dict_flags_of_noun = (w->#dict_par1) & (DICT_X654+DICT_PLUR);!$$01110100;
	if (threshold > 0) {
		k = threshold;
		jump MMbyPN;
	}
      .NoWordsMatch;
        if (indef_mode ~= 0) {
            k = 0; parser_action = NULL;
            jump MMbyPN;
        }
        rfalse;
    }

    k = threshold;
    jump MMbyPN;
];

[ PostureMatchesInput obj wd;

	switch (obj.posture) {
		1: if (wd == 'standing') rtrue;
		2: if (wd == 'sitting') rtrue;
		3: if (wd == 'lying' or 'resting' or 'reclining') rtrue;
	}

	rfalse;
];

[ ParseObj obj wd n m;

	wn--; wd = NextWord();

	if (reparsing && wd == 'is') { n++; reparsing == 100; wd = NextWord(); }

	while (WordInProperty(wd, obj, adjective) || (obj ofclass Person && PostureMatchesInput(obj, wd)))
		{ m++; wd = NextWord(); }

	if (reparsing && wd == 'one') { n++; reparsing == 100; wd = NextWord(); }

	while (WordInProperty(wd, obj, name)) { n++; wd = NextWord(); }

	.End;

	if (n) return n + m; rfalse;
];

[ AppendPreps obj p wd i m ap x;

	! This routine handles the case of having to distinguish
	! PUT THE DAGGER IN (THE SHEATH ON MY BELT) from
	! PUT (THE DAGGER IN THE SHEATH) ON MY BELT.

	if ((~~obj ofclass Thing) || ~~parent(obj)) rfalse;

	p = parent(obj);

	if (wn > 2) wn = wn - 2;

	wd = NextWord();

	if (wd == ALL1__WD && indef_specific ~= -2) indef_specific = -1;

	if (p ofclass Person) {
		i = TryGivenObject(p, 1);
		wn--; wd = NextWord();
		if (wd == 'is' or 'are' or 'was' or 'were') {
			m = m + i + 1; wd = NextWord();
		}
	}
	else wd = NextWord();

	if (wd == 'being') { m++; wd = NextWord(); }

	if (obj ofclass Person) {
		if (wd == 'who') {
			m++; wd = NextWord();
			if (wd == 'is' or 'are' or 'was' or 'were') { m++; wd = NextWord(); }
		}
		if (PostureMatchesInput(obj, wd)) { i++; wd = NextWord(); }
		jump Position;
	}

	if (wd == 'which' or 'that') {
		m++; wd = NextWord();

		if (p ofclass Person) {
			if (wd == ME4__WD && p == player) i++;
			else { wn--; i = TryGivenObject(p, 1); }
			if (i) { m = m + i; wd = NextWord(); }
		}

		if (wd == 'is' or 'are' or 'was' or 'were' or 'am') {
			m++; wd = NextWord();
			if (p ofclass Person && wd == 'being') { m++; wd = NextWord(); }
		}
	}

	if (wd == 'resting' or 'lying') { m++; wd = NextWord(); }

	.Position;

	if ((obj.insideofparent && wd == 'in' or 'inside' or 'within' or 'from')
		|| (obj.ontopofparent && wd == 'on' or 'upon' or 'off' or 'from')
		|| (obj.behindparent && wd == 'behind' or 'from')
		|| (obj.beneathparent && wd == 'below' or 'beneath' or 'under' or 'underneath' or 'from')
		|| (obj.attachedtoparent && wd == 'attached')
		|| (p ofclass Person && wd == 'carrying' or 'carried' or 'holding' or 'holds'
			or 'held' or 'has' or 'had' or 'wearing' or 'worn')) {

		m++;

		if (wd == 'carried' or 'held' or 'worn'){
			if (NextWord() == 'by') m++; else wn--;
		}

		if (wd == 'off' or 'inside') { if (NextWord() == 'of') m++; else wn--; }

		if (wd == 'on') {
			if (NextWord() == 'top')
				{ m++; if (NextWord() == 'of') m++; else wn--; }
			else wn--;
		}

		if (wd == 'attached') {
			if (NextWord() == 'to') m++; else wn--;
		}

		if (wd == 'from') {
			if ((obj.insideofparent && NextWord() == 'in' or 'inside' or 'within')
				|| (obj.ontopofparent && NextWord() == 'on' or 'upon' or 'off')
				|| (obj.behindparent && NextWord() == 'behind')
				|| (obj.beneathparent && NextWord() == 'below' or 'beneath' or 'under'))
					{ m++; if (NextWord() == 'of') m++; else wn--; }
			else wn--;
		}

		for (x=1 : x<=LanguageDescriptors-->0 : x=x+4)
            	if (NextWord() == LanguageDescriptors-->x) {
				m++;
			}
			else wn--;

            .Prepositions2;
		
		ap = allow_plurals;
		if (~~i) i = TryGivenObject(p, 1);
		allow_plurals = ap;
		if (verb_word == 'put' or 'place' && NextWordStopped() == -1) rfalse;
	}

	.GiveResult;

	m = m + i;

	if (i == 0) {
		etype = CANTSEE_PE;
		rfalse;
	}

	if (indef_wanted == 100) indef_specific = 1;

	return m;
];

[ WordLeft w     own cw flag;
    own = wn;
    while ((cw = NextWordStopped()) ~= -1)
    {   if (flag)
        {   if (cw && (cw->#dict_par1) & 1) break;
            flag = 0;
        }
        if (cw == w) { wn = own; rtrue; }
        if (IsALinkWord(w)) flag = 1;
    }
    wn = own;
    rfalse;
];

[ IsALinkWord wd;
    if (wd == COMMA_WORD or AND1__WD or AND2__WD or AND3__WD
        or THEN1__WD or THEN2__WD or THEN3__WD)
        rtrue;
    rfalse;
];

! ----------------------------------------------------------------------------
!  Refers works out whether the word at number wnum can refer to the object
!  obj, returning true or false.  The standard method is to see if the
!  word is listed under "name" or "adjective" for the object, but this is more complex
!  in languages other than English.
! ----------------------------------------------------------------------------


[ Refers obj wnum   wd k l m;
    if (obj == 0) rfalse;

    #Ifdef LanguageRefers;
    k = LanguageRefers(obj,wnum); if (k >= 0) return k;
    #Endif; ! LanguageRefers

    k = wn; wn = wnum; wd = NextWordStopped(); wn = k;

    if (parser_inflection_func) {
        k = parser_inflection(obj, wd);
        if (k >= 0) return k;
        m = -k;
    }
    else
        m = parser_inflection;

    k = obj.&m; l = (obj.#m)/WORDSIZE-1;
    for (m=0 : m<=l : m++)
        if (wd == k-->m) rtrue;
    rfalse;
];

[ WordInProperty wd obj prop k l m;
    k = obj.&prop; l = (obj.#prop)/WORDSIZE-1;
    for (m=0 : m<=l : m++)
        if (wd == k-->m) rtrue;
    rfalse;
];

#Ifdef TARGET_ZCODE;

[ DictionaryLookup b l i;
    for (i=0 : i<l : i++) buffer2->(2+i) = b->i;
    buffer2->1 = l;
    Tokenise__(buffer2,parse2);
    return parse2-->1;
];

#Ifnot; ! TARGET_GLULX

[ DictionaryLookup b l i;
    for (i=0 : i<l : i++) buffer2->(WORDSIZE+i) = b->i;
    buffer2-->0 = l;
    Tokenise__(buffer2,parse2);
    return parse2-->1;
];

#Endif; ! TARGET_

! ----------------------------------------------------------------------------
!  NounWord (which takes no arguments) returns:
!
!   0  if the next word is unrecognised or does not carry the "noun" bit in
!      its dictionary entry,
!   1  if a word meaning "me",
!   the index in the pronoun table (plus 2) of the value field of a pronoun,
!      if the word is a pronoun,
!   the address in the dictionary if it is a recognised noun.
!
!  The "current word" marker moves on one.
! ----------------------------------------------------------------------------

[ NounWord i j s;
    i = NextWord();
    if (i == 0) rfalse;
    if (i == ME1__WD or ME2__WD) return 1;
    if (i == ME3__WD or SELF1__WD or SELF2__WD or SELF3__WD or YOU1__WD or YOU2__WD) return 2;
    s = LanguagePronouns-->0;
    for (j=1 : j<=s : j=j+3)
        if (i == LanguagePronouns-->j)
            return j+2;
    if ((i->#dict_par1) & DICT_NOUN == 0) rfalse;
    return i;
];

! ----------------------------------------------------------------------------
!  NextWord (which takes no arguments) returns:
!
!  0            if the next word is unrecognised,
!  comma_word   if a comma
!  THEN1__WD    if a full stop
!  or the dictionary address if it is recognised.
!  The "current word" marker is moved on.
!
!  NextWordStopped does the same, but returns -1 when input has run out
! ----------------------------------------------------------------------------

#Ifdef TARGET_ZCODE;

[ NextWord i j;
    if (wn > parse->1) { wn++; rfalse; }
    i = wn*2-1; wn++;
    j = parse-->i;
    if (j == ',//') j = comma_word;
    if (j == './/') j = THEN1__WD;
    return j;
];

[ NextWordStopped;
    if (wn > parse->1) { wn++; return -1; }
    return NextWord();
];

[ WordAddress wordnum p b;  ! Absolute addr of 'wordnum' string in buffer
    if (p==0) p=parse;
    if (b==0) b=buffer;
    return b + p->(wordnum*4+1);
];

[ WordLength wordnum p;     ! Length of 'wordnum' string in buffer
    if (p==0) p=parse;
    return p->(wordnum*4);
];

[ WordValue wordnum p;      ! Dictionary value of 'wordnum' string in buffer
    if (p==0) p=parse;
    return p-->(wordnum*2-1);
];

[ NumberWords p;            ! Number of parsed strings in buffer
    if (p==0) p=parse;
    return p->1;
];

[ GetKeyBufLength b;        ! Number of typed chars in buffer
    if (b==0) b=buffer;
    return b->1;
];

[ SetKeyBufLength n b;      ! Update number of typed chars in buffer
    if (b==0) b=buffer;
    if (n > INPUT_BUFFER_LEN-WORDSIZE) n=INPUT_BUFFER_LEN-WORDSIZE;
    b->1 = n;
];

#Ifnot; ! TARGET_GLULX

[ NextWord i j;
    if (wn > parse-->0) { wn++; rfalse; }
    i = wn*3-2; wn++;
    j = parse-->i;
    if (j == ',//') j=comma_word;
    if (j == './/') j=THEN1__WD;
    return j;
];

[ NextWordStopped;
    if (wn > parse-->0) {
        wn++;
        return -1;
    }
    return NextWord();
];

[ WordAddress wordnum p b;  ! Absolute addr of 'wordnum' string in buffer
    if (p==0) p=parse;
    if (b==0) b=buffer;
    return b + p-->(wordnum*3);
];

[ WordLength wordnum p;     ! Length of 'wordnum' string in buffer
    if (p==0) p=parse;
    return p-->(wordnum*3-1);
];

[ WordValue wordnum p;      ! Dictionary value of 'wordnum' string in buffer
    if (p==0) p=parse;
    return p-->(wordnum*3-2);
];

[ NumberWords p;            ! Number of parsed strings in buffer
    if (p==0) p=parse;
    return p-->0;
];

[ GetKeyBufLength b;        ! Number of typed chars in buffer
    if (b==0) b=buffer;
    return b-->0;
];

[ SetKeyBufLength n b;      ! Update number of typed chars in buffer
    if (b==0) b=buffer;
    if (n > INPUT_BUFFER_LEN-WORDSIZE) n=INPUT_BUFFER_LEN-WORDSIZE;
    b-->0 = n;
];

#Endif; ! TARGET_

! ----------------------------------------------------------------------------
!  TryNumber is the only routine which really does any character-level
!  parsing, since that's normally left to the Z-machine.
!  It takes word number "wordnum" and tries to parse it as an (unsigned)
!  decimal number, returning
!
!  -1000                if it is not a number
!  the number           if it has between 1 and 4 digits
!  10000                if it has 5 or more digits.
!
!  (The danger of allowing 5 digits is that Z-machine integers are only
!  16 bits long, and anyway this isn't meant to be perfect.)
!
!  Using NumberWord, it also catches "one" up to "twenty".
!
!  Note that a game can provide a ParseNumber routine which takes priority,
!  to enable parsing of odder numbers ("x45y12", say).
! ----------------------------------------------------------------------------

[ TryNumber wordnum   i j c num len mul tot d digit;

    i = wn; wn = wordnum; j = NextWord(); wn = i;
    j = NumberWord(j);
    !! if (j >= 1) return j;
    if (j >= 1) {
	if (j < 20) return j;
	else {
		i = wn; wn = wordnum; c = j;
		j = NextWord(); if ( j == 0) { wn = i; return c; }
		j = NumberWord(j);
		return j  + c;
	}
    }

    num = WordAddress(wordnum); len = WordLength(wordnum);
    tot = ParseNumber(num, len);

#IFDEF USE_LIBRARY_EXTENSIONS;
    if (tot == 0) tot = LibraryExtensions.RunWhile(ext_parsenumber, 0, num, len);
#ENDIF;

    if (tot ~= 0) return tot;

    if (len >= 4) mul=1000;
    if (len == 3) mul=100;
    if (len == 2) mul=10;
    if (len == 1) mul=1;

    tot = 0; c = 0;

    for (c = 0 : c < len : c++) {
        digit=num->c;
        if (digit == '0') { d = 0; jump digok; }
        if (digit == '1') { d = 1; jump digok; }
        if (digit == '2') { d = 2; jump digok; }
        if (digit == '3') { d = 3; jump digok; }
        if (digit == '4') { d = 4; jump digok; }
        if (digit == '5') { d = 5; jump digok; }
        if (digit == '6') { d = 6; jump digok; }
        if (digit == '7') { d = 7; jump digok; }
        if (digit == '8') { d = 8; jump digok; }
        if (digit == '9') { d = 9; jump digok; }
        return -1000;
     .digok;
        tot = tot+mul*d; mul = mul/10;
    }
    if (len > 4) tot=10000;
    return tot;
];

! ----------------------------------------------------------------------------
!  AnyNumber is a general parsing routine which accepts binary, hexadecimal
!  and decimal numbers up to the full Zcode/Glulx ranges.
! ----------------------------------------------------------------------------

#Ifdef TARGET_ZCODE;                ! decimal range is -32768 to 32767
Constant MAX_DECIMAL_SIZE 5;
Constant MAX_DECIMAL_BASE 3276;
#Ifnot; ! TARGET_GLULX              ! decimal range is -2147483648 to 2147483647
Constant MAX_DECIMAL_SIZE 10;
Constant MAX_DECIMAL_BASE 214748364;
#Endif; ! TARGET_

[ AnyNumber
    wa we sign base digit digit_count num;

    if (wa == 0) { wa = WordAddress(wn); we = WordLength(wn); }
    we = wa + we;
    sign = 1; base = 10;
    if     (wa->0 == '-') { sign = -1; wa++; }
    else {
        if (wa->0 == '$') { base = 16; wa++; }
        if (wa->0 == '$') { base = 2;  wa++; }
    }
    if (wa >= we) return GPR_FAIL;  ! no digits after -/$
    while (wa->0 == '0') wa++;      ! skip leading zeros
    for (num=0,digit_count=1 : wa<we : wa++,digit_count++) {
        switch (wa->0) {
          '0' to '9': digit = wa->0 - '0';
          'A' to 'F': digit = wa->0 - 'A' + 10;
          'a' to 'f': digit = wa->0 - 'a' + 10;
          default:    return GPR_FAIL;
        }
        if (digit >= base) return GPR_FAIL;
        !digit_count++;
        switch (base) {
          16:     if (digit_count > 2*WORDSIZE)  return GPR_FAIL;
          2:      if (digit_count > 8*WORDSIZE)  return GPR_FAIL;
          10:
            if (digit_count >  MAX_DECIMAL_SIZE) return GPR_FAIL;
            if (digit_count == MAX_DECIMAL_SIZE) {
                if (num >  MAX_DECIMAL_BASE)     return GPR_FAIL;
                if (num == MAX_DECIMAL_BASE) {
                    if (sign == 1  && digit > 7) return GPR_FAIL;
                    if (sign == -1 && digit > 8) return GPR_FAIL;
                }
            }
        }
        num = base*num + digit;
    }
    parsed_number = num * sign;
    wn++;
    return GPR_NUMBER;
];

[ HiddenInventory obj;
    if (metaclass(obj) ~= Object) rfalse;
    if (IndirectlyContains(player, obj)) {
	if (obj has topic && IsVisible(obj) == false) rtrue;
    }
    rfalse;
];

! ----------------------------------------------------------------------------
!  GetGender returns 0 if the given animate object is female, and 1 if male
!  (not all games will want such a simple decision function!)
! ----------------------------------------------------------------------------

[ GetGender person;
    if (person hasnt female) rtrue;
    rfalse;
];

[ GetGNAOfObject obj case gender;
    if (obj hasnt animate) case = 6;
    if (obj has male) gender = male;
    if (obj has female) gender = female;
    if (obj has neuter) gender = neuter;
    if (gender == 0) {
        if (case == 0) gender = LanguageAnimateGender;
        else gender = LanguageInanimateGender;
    }
    if (gender == female)   case = case + 1;
    if (gender == neuter)   case = case + 2;
    if (obj has pluralname) case = case + 3;
    return case;
];

! ----------------------------------------------------------------------------
!  Converting between dictionary addresses and entry numbers
! ----------------------------------------------------------------------------

#Ifdef TARGET_ZCODE;

[ Dword__No w; return (w-(HDR_DICTIONARY-->0 + 7))/9; ];
[ No__Dword n; return HDR_DICTIONARY-->0 + 7 + 9*n; ];

#Ifnot; ! TARGET_GLULX

! In Glulx, dictionary entries *are* addresses.
[ Dword__No w; return w; ];
[ No__Dword n; return n; ];

#Endif; ! TARGET_

! ----------------------------------------------------------------------------
!  For copying buffers
! ----------------------------------------------------------------------------

#Ifdef TARGET_ZCODE;

[ CopyBuffer bto bfrom i size;
    size = bto->0;
    for (i=1 : i<=size : i++) bto->i = bfrom->i;
];

#Ifnot; ! TARGET_GLULX

[ CopyBuffer bto bfrom i;
    for (i=0 : i<INPUT_BUFFER_LEN : i++) bto->i = bfrom->i;
];

#Endif; ! TARGET_

! ----------------------------------------------------------------------------
!  Provided for use by language definition files
! ----------------------------------------------------------------------------

#Ifdef TARGET_ZCODE;

[ LTI_Insert i ch  b y;

    ! Protect us from strict mode, as this isn't an array in quite the
    ! sense it expects
    b = buffer;

    ! Insert character ch into buffer at point i.
    ! Being careful not to let the buffer possibly overflow:
    y = b->1;
    if (y > b->0) y = b->0;

    ! Move the subsequent text along one character:
    for (y=y+2 : y>i : y--) b->y = b->(y-1);
    b->i = ch;

    ! And the text is now one character longer:
    if (b->1 < b->0) (b->1)++;
];

#Ifnot; ! TARGET_GLULX

[ LTI_Insert i ch  b y;

    ! Protect us from strict mode, as this isn't an array in quite the
    ! sense it expects
    b = buffer;

    ! Insert character ch into buffer at point i.
    ! Being careful not to let the buffer possibly overflow:
    y = b-->0;
    if (y > INPUT_BUFFER_LEN) y = INPUT_BUFFER_LEN;

    ! Move the subsequent text along one character:
    for (y=y+WORDSIZE : y>i : y--) b->y = b->(y-1);
    b->i = ch;

    ! And the text is now one character longer:
    if (b-->0 < INPUT_BUFFER_LEN) (b-->0)++;
];

#Endif; ! TARGET_

! ============================================================================

[ PronounsSub x y c d;
    L__M(##Pronouns, 1);

    c = (LanguagePronouns-->0)/3;
    #Ifdef CHARACTER; if (player ~= CHARACTER) c++;
    #Ifnot; if (player ~= selfobj) c++;
    #Endif;

    if (c == 0) return L__M(##Pronouns, 4);

    for (x=1,d=0 : x<=LanguagePronouns-->0 : x=x+3) {
        print "~", (address) LanguagePronouns-->x, "~ ";
        y = LanguagePronouns-->(x+2);
        if (y == NULL) L__M(##Pronouns, 3);
        else {
            L__M(##Pronouns, 2);
            print (the) y;
        }
        d++;
        if (d < c-1) print (string) COMMA__TX;
        if (d == c-1) {
		if (c > 2) print (string) AND__TX;
		else print (string) AND2__TX;
	  }
    }
    #Ifdef CHARACTER;
    print "~", (address) ME1__WD, "~ "; L__M(##Pronouns, 2);
    print (the) player;
    #Ifnot;
    if (player ~= selfobj) {
        print "~", (address) ME1__WD, "~ "; L__M(##Pronouns, 2);
        c = player; player = selfobj;
        print (the) c; player = c;
    }
    #Endif;
    L__M(##Pronouns, 5);
];

[ SetPronoun dword value x;
    for (x=1 : x<=LanguagePronouns-->0 : x=x+3)
        if (LanguagePronouns-->x == dword) {
            LanguagePronouns-->(x+2) = value; return;
        }
    RunTimeError(14);
];

[ PronounValue dword x;
    for (x=1 : x<=LanguagePronouns-->0 : x=x+3)
        if (LanguagePronouns-->x == dword)
            return LanguagePronouns-->(x+2);
    return 0;
];

[ ResetVagueWords obj; PronounNotice(obj); ];

[ PronounNotice obj x bm;
    if (obj == player) return;

    bm = PowersOfTwo_TB-->(GetGNAOfObject(obj));

    for (x=1 : x<=LanguagePronouns-->0 : x=x+3)
        if (bm & (LanguagePronouns-->(x+1)) ~= 0)
            LanguagePronouns-->(x+2) = obj;
];

! ----------------------------------------------------------------------------
!  End of the parser proper: the remaining routines are its front end.
! ----------------------------------------------------------------------------

Object  InformLibrary "(Inform Library)"
  with  play [ i j k l m n next_obj;

            #Ifdef TARGET_ZCODE;
            standard_interpreter = HDR_TERPSTANDARD-->0;
            transcript_mode = ((HDR_GAMEFLAGS-->0) & 1);
		#Ifndef V6;
            sys_statusline_flag = ( (HDR_TERPFLAGS->0) & 2 ) / 2;
		#Endif;
            #Ifnot; ! TARGET_GLULX
            GGInitialise();
            #Endif; ! TARGET_

            #Ifdef TARGET_ZCODE;
            dict_start = HDR_DICTIONARY-->0;
            dict_entry_size = dict_start->(dict_start->0 + 1);
            dict_start = dict_start + dict_start->0 + 4;
            dict_end = dict_start + (dict_start - 2)-->0 * dict_entry_size;

            #Ifdef DEBUG;
            if (dict_start > 0 && dict_end < 0 &&
              ((-dict_start) - dict_end) % dict_entry_size == 0)
                print "** Warning: grammar properties might not work correctly **^";
            #Endif; ! DEBUG

            buffer->0  = INPUT_BUFFER_LEN;
            buffer2->0 = INPUT_BUFFER_LEN;
            buffer3->0 = INPUT_BUFFER_LEN;
            parse->0   = 15;
            parse2->0  = 15;
            #Endif; ! TARGET_ZCODE

		#Ifdef CHARACTER;
		ChangePlayer(CHARACTER);
		#Ifnot;
		player = selfobj; actor = player;
		#Endif;

            #Ifdef TARGET_ZCODE;
            top_object = #largest_object-255;
            #Endif; ! TARGET_ZCODE

            #Ifdef LanguageInitialise;
            LanguageInitialise();
            #Endif; ! LanguageInitialise

		if (sys_statusline_flag ~= 0)
		the_time = 0;

	#Ifndef NO_SCORE;
            last_score = score;
	#Endif;

            new_line;
	#IFDEF USE_LIBRARY_EXTENSIONS;
            LibraryExtensions.RunAll(ext_initialise);
	#ENDIF;
            j = Initialise();

	    if (parent(player) == 0) Move(player, player.location);

	  SetLocations(); CalcWeight(); SetNPCNumbers();

	#IFDEF TRACK_LOCATIONS;
	  objectloop (i ofclass Person) {
		objectloop (m && IsVisible(i, m)) { Recognize(i, m); }
	  }
	#IFNOT;
	  objectloop (m && IsVisible(player, m)) give m topic;
	#ENDIF;

		#Ifndef NO_DARKNESS;
		switch (LightToSeeBy(player))
			{ 0: give self ~light;
			  1: give self light;
			#Ifdef GRADUAL_LIGHT;
			  2: give self demilight;
			  3: give self semilight;
			#Endif;
			}
		#Endif;

            objectloop (i in player) give i moved ~concealed;

            if (j ~= 2) Banner();

	    #Ifndef NOFIRSTLOOK; <Look>; #Endif;

            for (i=1 : i<=100 : i++) j = random(i);

            while (~~gameover) {    ! everything happens in this loop

              .very__late__error;
		#Ifndef NO_SCORE;
                if (score ~= last_score) {
                    if (notify_mode == 1 or 2) NotifyTheScore();
                    last_score = score;
                }
		#Endif;

              .late__error;

                inputobjs-->0 = 0; inputobjs-->1 = 0;
                inputobjs-->2 = 0; inputobjs-->3 = 0; meta = false;

                ! The Parser writes its results into inputobjs and meta,
                ! a flag indicating a "meta-verb".  This can only be set for
                ! commands by the player, not for orders to others.

                InformParser.parse_input(inputobjs);
                action = inputobjs-->0;

                ! --------------------------------------------------------------

                ! Reverse "give fred biscuit" into "give biscuit to fred"

                if (action == ##GiveR or ##ShowR) {
                    i = inputobjs-->2; inputobjs-->2 = inputobjs-->3; inputobjs-->3 = i;
                    if (action == ##GiveR) action = ##Give; else action = ##Show;
                }

                ! Convert "P, tell me about X" to "ask P about X"

                if (action == ##Tell && inputobjs-->2 == player && actor ~= player) {
                    inputobjs-->2 = actor; actor = player; action = ##Ask;
                }

                ! Convert "ask P for X" to "P, give X to me"

                if (action == ##AskFor) {
			  if (actor ~= inputobjs-->2) action_mode = 2;
                    actor = inputobjs-->2; inputobjs-->2 = inputobjs-->3;
                    inputobjs-->3 = player; action = ##Give;
                }

                ! For old, obsolete code: special_word contains the topic word
                ! in conversation

                if (action == ##Ask or ##Tell or ##Answer)
                    special_word = special_number1;

              .begin__action;

                multiflag = false; onotheld_mode = notheld_mode; notheld_mode = false;

                inp1 = 0; inp2 = 0; i = inputobjs-->1;
                if (i >= 1) inp1 = inputobjs-->2;
                if (i >= 2) inp2 = inputobjs-->3;

                ! inp1 and inp2 hold: object numbers, or 0 for "multiple object",
                ! or 1 for "a number or dictionary address".

                if (inp1 == 1) noun = special_number1; else noun = inp1;
                if (inp2 == 1) {
                    if (inp1 == 1) second = special_number2;
                    else           second = special_number1;
                }
                else second = inp2;

		if (multiple_object-->0 ~= 0) { multiflag = true; noun = multiple_object-->1; }

                !  --------------------------------------------------------------
		!  Possibly disallow the action if in darkness.

		#IFNDEF NO_DARKNESS;
		if (~~LighttoSeeBy(actor)) {
			if (InDarkness(actor)) jump turn__end;
			if (action == ##Examine or ##Search or ##LookInside or ##LookOnTop
				or ##LookBehind or ##LookUnder or ##LookAttached or ##Consult)
				{ if (actor == player) L__M(##Examine, 1); jump turn__end; }
		}
		#ENDIF;

                !  --------------------------------------------------------------
		!  Certain actions on hidden inventory items are disallowed.

		if (HiddenInventory(noun)) {
			if (action == ##Examine or ##Search or ##LookInside or ##LookOnTop
				or ##LookBehind or ##LookUnder or ##LookAttached or ##Consult)
				{  if (ObjectIsUntouchable(noun, actor, false, true, true))
				   	{ L__M(##Examine, 4, noun); jump turn__end; }
				}
		}

                ! --------------------------------------------------------------
                ! Generate the action...

		.execute__action;

		! Simply perform the action...

		if (~~multiflag) {
		#IFDEF TRACK_LOCATIONS;
		!  Actions on absent items are disallowed.
		m = 0;
		if (AbsentItem(noun)) m = noun;
		else if (AbsentItem(second)) m = second;

		if (meta == false && m) {
			L__M(##Examine, 5, m); failed_action = true; jump turn__end;
		}
		#ENDIF;

		if (self.actor_act(actor, action, noun, second) == 1) jump begin__action;
		jump turn__end;
                }

                ! ...unless a multiple object must be substituted.  First:
                ! (a) check the multiple list isn't empty;
                ! (b) warn the player if it has been cut short because too long;
                ! (c) generate a sequence of actions from the list
                !     (stopping in the event of death, movement away, a too-full
		!      container, or an author-supplied rule.)

                j = multiple_object-->0;
                if (j == 0) {
                    L__M(##Miscellany, 2);
                    return -1;
                }
                if (toomany_flag) {
                    toomany_flag = false;
                    L__M(##Miscellany, 1);
                }

		    ! This check allows the "(first opening...)" message to be printed first,
		    ! rather than in the middle of the action.

		    #IFDEF TRACK_LOCATIONS;
		    if (~~AbsentItem(second)) {
		    #ENDIF;
		    if ((action == ##Insert && second has openable && second has closed)
			   ||	(action == ##Transfer && second.inside_capacity()
				&& second has openable && second has closed)) {
			multiflag = false; multiflag2 = true;
			CheckBeforeImplicit(second, ##Open, ##Miscellany, 65, second, ##EmptyT, 2);
			if (failed_action) jump turn__end; multiflag = true; multiflag2 = false;
		    }
		    #IFDEF TRACK_LOCATIONS;
		    }
		    #ENDIF;

                i = actor.location; count = 0;
                for (k=1 : k<=j : k++) {
                    if (gameover) break;
                    if (actor.location ~= i) {
                        L__M(##Miscellany, 51);
                        break;
                    }
                    l = multiple_object-->k; next_obj = multiple_object-->k+1;

		    if (first_indef_wanted) indef_wanted = first_indef_wanted;
		    if (count == 0) {
				while (Identical(l, next_obj) && count < indef_wanted) {
					count++; n = k + count; next_obj = multiple_object-->n;
				}
                    }

		if (count == 0) {
			#IFDEF TRACK_LOCATIONS;
		!  Actions on absent items are disallowed.
			if (AbsentItem(l)) {
				L__M(##Examine, 5, l); failed_action = true; continue;
			}
			#ENDIF;
		! Sometimes, disallow actions on non-held items.
			if (multi_context == MULTIHELD_TOKEN && l notin actor) continue;
		}
		
                    if (actor == player) PronounNotice(l);
                    if (count > 0) {
			if (multi_context == MULTIHELD_TOKEN && l notin actor) continue;
			print count, " ";
			if (count > 1 && l.plural) print (string) l.plural;
			else print (name) l;
                        #IFDEF TRACK_LOCATIONS;
			if (AbsentItem(l) == false) L__M(##ListMiscellany, 11, l);
                        #ENDIF;
                    }
                    else print (name) l; print (string) COLON__TX;

                    if (inp1 == 0) {
                        inp1 = l;
                        if (count > 0) while (count ~= 1) {
				count--; if (indef_wanted < 100) indef_wanted--;
				#IFDEF TRACK_LOCATIONS;
				if (AbsentItem(l)) jump Increment1;
				#ENDIF;
				keep_silent = 2;
				switch (self.actor_act(actor, action, l, second)) {
					1: keep_silent = 0; jump begin__action;
					2: keep_silent = 0; jump turn__end;
				}
				.Increment1;
				l = multiple_object-->++k; if (failed_action == 0) successful_action++;
				keep_silent = 0;
                        }
			#IFDEF TRACK_LOCATIONS;
                        if (AbsentItem(l)) { L__M(##Examine, 5, l, true); failed_action = true; continue; }
			#ENDIF;
			switch (self.actor_act(actor, action, l, second)) {
				1: jump begin__action;
				2: jump turn__end;
			}
                        inp1 = 0; if (count == 1 && failed_action == 0) successful_action++;
			if (successful_action ~= 0 && successful_action < first_indef_wanted) {
				L__M(##Miscellany, 75, l, successful_action); successful_action = 0;
			}
                        if (count == 1) { count = 0; if (indef_wanted < 100) indef_wanted--; }
                    }
                    else {
                        inp2 = l;
                        if (count > 0) while (count ~= 1) {
				count--; if (indef_wanted < 100) indef_wanted--;
				#IFDEF TRACK_LOCATIONS;
				if (AbsentItem(l)) jump Increment2;
				#ENDIF;
				keep_silent = 2;
				switch (self.actor_act(actor, action, l, second)) {
					1: keep_silent = 0; jump begin__action;
					2: keep_silent = 0; jump turn__end;
				}
				.Increment2;
				l = multiple_object-->++k; if (failed_action == 0) successful_action++;
				keep_silent = 0;
                        }
			#IFDEF TRACK_LOCATIONS;
                        if (AbsentItem(l)) { L__M(##Examine, 5, l, true); failed_action = true; continue; }
			#ENDIF;
			switch (self.actor_act(actor, action, noun, l)) {
				1: jump begin__action;
				2: jump turn__end;
			}
                        inp2 = 0; if (count == 1 && failed_action == 0) successful_action++;
			if (successful_action ~= 0 && successful_action < first_indef_wanted) {
				L__M(##Miscellany, 75, l, successful_action); successful_action = 0;
			}
                        if (count == 1) { count = 0; if (indef_wanted < 100) indef_wanted--; }
                    }
		    if (failed_action == -1) break; m = 0;
                }

                ! --------------------------------------------------------------

              .turn__end;

                ! No time passes if either (i) the verb was meta, or
                ! (ii) we've only had the implicit take before the "real"
                ! action to follow.

                if (notheld_mode == 1) { NoteObjects(); continue; }
                if (meta) continue;
		#Ifndef NO_DARKNESS;
		switch (LightToSeeBy(player))
			{ 0: if (self has light) {
				give self ~light;

			     #Ifdef GRADUAL_LIGHT;
				give self ~demilight ~semilight;
			     #Endif;
				
				if (~~arrived) L__M(##Miscellany, 9);
				}

			  1: if (self hasnt light) {
				give self light;
				if (~~arrived) L__M(##Miscellany, 60);
			     #Ifdef GRADUAL_LIGHT;
				give self ~demilight ~semilight;
			     #Endif;
				}

			     #Ifdef GRADUAL_LIGHT;
			  2: 
			     if (~~arrived) {
				if (self hasnt light or semilight) L__M(##Miscellany, 60);
				else L__M(##Miscellany, 61);
			     }
			     give self demilight ~light ~semilight;

			  3: 
			     if (~~arrived) {
				if (self hasnt light or demilight) L__M(##Miscellany, 60);
				else L__M(##Miscellany, 61);
			     }
			     give self semilight ~light ~demilight;
			     #Endif;
			}
		#Endif;
                if (~~gameover) self.end_turn_sequence();
                else if (START_MOVE ~= 1) turns++;

            } ! end of while()

	#IFDEF USE_LIBRARY_EXTENSIONS;
            if (gameover ~= 2 && AfterLife() == false)
                 LibraryExtensions.RunAll(ext_afterlife);
	#IFNOT;
            if (gameover ~= 2) AfterLife();
	#ENDIF;
            if (gameover == 0) jump very__late__error;

            print "^^    ";
            #Ifdef TARGET_ZCODE;
            #IfV5; style bold; #Endif; ! V5
            #Ifnot; ! TARGET_GLULX
            glk($0086, 5); ! set alert style
            #Endif; ! TARGET_
            print "***";
            if (gameover == 1) L__M(##Miscellany, 3);
            if (gameover == 2) L__M(##Miscellany, 4);
            if (gameover > 2)  {
                print " ";
		#IFDEF USE_LIBRARY_EXTENSIONS;
                if (DeathMessage() == false)
                    LibraryExtensions.RunAll(ext_deathmessage);
		#IFNOT;
                DeathMessage();
		#ENDIF;
                print " ";
            }
	    
            print "***";
	    Epilogue();
            #Ifdef TARGET_ZCODE;
            #IfV5; style roman; #Endif; ! V5
            #Ifnot; ! TARGET_GLULX
            glk($0086, 0); ! set normal style
            #Endif; ! TARGET_
            #Ifdef NO_SCORE;
            print "^^";
            #Ifnot;
            print "^^^";
            ScoreSub();
            #Endif; ! NO_SCORE
            DisplayStatus();
            AfterGameOver();

        ], ! end of 'play' property

        end_turn_sequence [ i m;
		actor = player;
		AdvanceWorldClock();		if (gameover) return;
		RunTimersAndDaemons(); 		if (gameover) return;
		RunEachTurnProperties();	if (gameover) return;
		TimePasses();			if (gameover) return;
		NoteObjects();			if (gameover) return;

		objectloop (i hasnt topic && IsVisible(i, player)) give i topic;
		#IFDEF TRACK_LOCATIONS;
		objectloop (i ofclass Person && i.location ~= 0) {
			objectloop (m && IsVisible(i, m)) { Recognize(i, m); }
		}
		#IFNOT; m = 0;
		#ENDIF;
        ],

        begin_action [ a n s source sa sn ss;
		sa = action; sn = noun; ss = second;
		action = a; noun = n; inp1 = n; second = s;

		if (source == 0) actor = player;

		#Ifdef DEBUG;
		if (debug_flag & 2 ~= 0) TraceAction(source);
		#Endif; ! DEBUG

		actor.&last_action-->0 = action;
		actor.&last_action-->1 = noun;
		actor.&last_action-->2 = second;

            #Iftrue (Grammar__Version == 1);
            if ((meta || BeforeRoutines() == false) && action < 256) {
		#Ifdef INFIX;
		if (infix_verb) {
		    if (BeforeRoutines() == false)
			ActionPrimitive();
		} else ActionPrimitive();
		#Ifnot;
		ActionPrimitive();
		#Endif;
	    }
            #Ifnot;
            if ((meta || BeforeRoutines() == false) && action < 4096) {
		#Ifdef INFIX;
		if (infix_verb) {
		    if (BeforeRoutines() == false)
			ActionPrimitive();
		} else ActionPrimitive();
		#Ifnot;
		ActionPrimitive();
		#Endif;
	    }
            #Endif; ! Grammar__Version

		FinalRoutines();

                action = sa; noun = sn; inp1 = sn; second = ss;
        ],

	actor_act [ source a n s   sa sn ss si ssource j;
            sa = action; sn = noun; ss = second; si = inp1; ssource = actor;
            action = a; noun = n; inp1 = n; second = s; actor = source;

                if (actor ~= player && action_mode == 2) {

                ! The player's "orders" property can refuse to allow
                ! conversation here, by returning true.  If not, the order is
                ! sent to the other person's "orders" property.  If that also
                ! returns false, then: if it was a misunderstood command
                ! anyway, it is converted to an Answer action (thus
                ! "floyd, grrr" ends up as "say grrr to floyd").  If it was a
                ! good command, it is finally offered to the Order: part of
                ! the other person's "life" property, the old-fashioned
                ! way of dealing with conversation.

                        j = RunRoutines(player, orders);
			if (j == 0) j = RunRoutines(actor, orders);
                        if (j == 0) {
                            if (action == ##NotUnderstood) {
                                inputobjs-->3 = actor; actor = player; action = ##Answer;
				return 1; ! abort, not resetting action globals
                            }
                            if (RunLife(actor, ##Order) == 0) {
                                L__M(##Order, actor);
				return 2;
                            }
                        } ! if (j == 0)
			if (j ~= 0) jump end;
                } ! if (actor ~= player)
                self.begin_action(a, n, s, source);

		.end;
		action = sa; noun = sn; second = ss; actor = ssource;
		inp1 = si;
	],
  has   proper;

#IFDEF TRACK_LOCATIONS;
[ AbsentItem l;

  if (meta == false && l provides last_seen) {
	if (l.&last_seen-->0 ~= 0 && IsVisible(l) == false) rtrue;
  }
  rfalse;
];
#ENDIF;

[ AdvanceWorldClock;
    turns++;
    if (the_time ~= NULL) {
        if (time_rate >= 0) the_time = the_time + time_rate;
        else {
            time_step--;
            if (time_step == 0) {
                the_time++;
                time_step = -time_rate;
            }
        }
        the_time = the_time % 1440;
    }
];

[ RunTimersAndDaemons i j;
    #Ifdef DEBUG;
    if (debug_flag & 4 ~= 0) {
        for (i=0 : i<active_timers : i++) {
            j = the_timers-->i;
            if (j ~= 0) {
                print (name) (j&~WORD_HIGHBIT), ": ";
                if (j & WORD_HIGHBIT) print "daemon";
                else
                    print "timer with ", j.time_left, " turns to go";
                new_line;
            }
        }
    }
    #Endif; ! DEBUG

    for (i=0 : i<active_timers : i++) {
        if (gameover) return;
        j = the_timers-->i;
        if (j ~= 0) {
            if (j & WORD_HIGHBIT) RunRoutines(j&~WORD_HIGHBIT, daemon);
            else {
                if (j.time_left == 0) {
                    StopTimer(j);
                    RunRoutines(j, time_out);
                }
                else
                    j.time_left = j.time_left-1;
            }
        }
    }
];

[ RunEachTurnProperties i;
    scope_reason = EACH_TURN_REASON; verb_word = 0;
    objectloop (i provides visibility_ceiling && IsVisible(i, player) == true)
			DoScopeAction(i);
    scope_reason = PARSING_REASON;
];

#Ifdef TARGET_ZCODE;

[ ActionPrimitive; indirect(#actions_table-->action); ];

#Ifnot; ! TARGET_GLULX

[ ActionPrimitive; indirect(#actions_table-->(action+1)); ];

#Endif; ! TARGET_

[ AfterGameOver i amus_ret;

  .RRQPL;

    L__M(##Miscellany,5);

  .RRQL;

    print "> ";
    #Ifdef TARGET_ZCODE;
    #IfV3; read buffer parse; #Endif; ! V3
    temp_global=0;
    #Ifndef NO_STATUSLINE;
    #IfV5; read buffer parse DrawStatusLine; #Endif; ! V5
    #Endif; !NO_STATUSLINE;
    #Ifnot; ! TARGET_GLULX
    KeyboardPrimitive(buffer, parse);
    #Endif; ! TARGET_
    i = parse-->1;
    if (i == QUIT1__WD or QUIT2__WD) {
        #Ifdef TARGET_ZCODE;
        quit;
        #Ifnot; ! TARGET_GLULX
        quit;
        #Endif; ! TARGET_
    }
    if (i == RESTART__WD) {
        #Ifdef TARGET_ZCODE;
        @restart;
        #Ifnot; ! TARGET_GLULX
        @restart;
        #Endif; ! TARGET_
    }
    if (i == RESTORE__WD) {
        RestoreSub();
        jump RRQPL;
    }
    #Ifndef NO_SCORE;
    if (i == FULLSCORE1__WD or FULLSCORE2__WD && TASKS_PROVIDED == 0) {
        new_line; FullScoreSub();
        jump RRQPL;
    }
    #Endif;
    amus_ret = 0; ! suppress compiler warning
    #Ifdef AMUSING_PROVIDED;
    if (gameover == 2 && i == AMUSING__WD && AMUSING_PROVIDED == 0) {
        new_line; amus_ret = Amusing();
	#IFDEF USE_LIBRARY_EXTENSIONS;
        if (amus_ret == false) LibraryExtensions.RunAll(ext_amusing);
	#ENDIF;
        jump RRQPL;
    }
    #Endif;
    #IfV5;
    if (i == UNDO1__WD or UNDO2__WD or UNDO3__WD) {
        if (undo_flag == 0) {
            L__M(##Miscellany, 6);
            jump RRQPL;
        }
        if (undo_flag == 1) jump UndoFailed2;
        #Ifdef TARGET_ZCODE;
        @restore_undo i;
        #Ifnot; ! TARGET_GLULX
        @restoreundo i;
        i = (~~i);
        #Endif; ! TARGET_
        if (i == 0) {
          .UndoFailed2;
            L__M(##Miscellany, 7);
        }
        jump RRQPL;
    }
    #Endif; ! V5
    L__M(##Miscellany, 8);
    jump RRQL;
];

! ----------------------------------------------------------------------------
! R_Process is invoked by the <...> and <<...>> statements, whose syntax is:
!   <action [noun] [second]>                ! traditional
!   <action [noun] [second], actor>         ! introduced at compiler 6.33

[ R_Process a n s p
    s1 s2 s3;
    s1 = inp1; s2 = inp2; s3 = actor;
    inp1 = n; inp2 = s; if (p) actor = p; else actor = player;
    InformLibrary.begin_action(a, n, s, 1);
    inp1 = s1; inp2 = s2; actor = s3;
];

! ----------------------------------------------------------------------------

[ NoteObjects i;
    objectloop (i in player)
        if (i hasnt moved) {
            give i moved;
	#Ifndef NO_SCORE;
            if (i.points ~= 0) {
                score = score + i.points;
                things_score = things_score + i.points;
            }
	#Endif;
        }
];

#IFDEF TRACK_LOCATIONS;
[ Recognize i m;

	if (i == 0) i = actor;

	if (i == player) {
		give m topic;
		if (m ofclass Thing) {
			m.&last_seen-->0 = i.location;
			m.&last_known-->0 = i.location;
			if (parent(m)) {
				m.&last_seen-->1 = parent(m);
				m.&last_seen-->2 = IndirectlyContains(parent(m), m);
			}
			if (the_time ~= NULL) m.&last_seen-->3 = the_time;
			else m.&last_seen-->3 = turns;
		}
	}
	else {
		if (m ofclass Thing) {
			m.&last_seen_npc-->(i.npc_number) = m.location;
			m.&last_known_npc-->(i.npc_number) = m.location;
		}
	}
];
#ENDIF;

! ----------------------------------------------------------------------------

[ TestScope obj act a sr x y;
    x = parser_one; y = parser_two;
    parser_one = obj; parser_two = 0; a = actor;
    sr = scope_reason; scope_reason = TESTSCOPE_REASON;
    if (act == 0) actor = player; else actor = act;
    SearchScope(actor); scope_reason = sr; actor = a;
    parser_one = x; x = parser_two; parser_two = y;
    return x;
];

[ LoopOverScope routine act x y a;
    x = parser_one; y = scope_reason; a = actor;
    parser_one = routine;
    if (act == 0) actor = player; else actor = act;
    scope_reason = LOOPOVERSCOPE_REASON;
    SearchScope(actor);
    parser_one = x; scope_reason = y; actor = a;
];

[ BeforeRoutines rv;
    if (GamePreRoutine() ~= 0) rtrue;
    rv = 0; ! suppress compiler warning
#IFDEF USE_LIBRARY_EXTENSIONS;
    if (rv == false) rv=LibraryExtensions.RunWhile(ext_gamepreroutine, 0);
    if (rv) rtrue;
#ENDIF;
    if (actor == player && RunRoutines(player, orders) ~= 0) rtrue;
    scope_reason = REACT_BEFORE_REASON; parser_one = 0;
    SearchScope(actor, 0);
    scope_reason = PARSING_REASON;
    if (parser_one ~= 0) rtrue;
    if (actor.location ~= 0 && RunRoutines(actor.location, before) ~= 0)
	{ failed_action = 1; rtrue; }
    if (inp1 > 1 && RunRoutines(inp1, before) ~= 0) rtrue;
    rfalse;
];

[ MiddleRoutines;
    scope_reason = REACT_MIDDLE_REASON; parser_one = 0;
    SearchScope(actor, 0); scope_reason = PARSING_REASON;
    if (parser_one ~= 0) rtrue;
    if (action == ##GoneFrom) {
		if (RunRoutines(actor.old_loc, middle)) rtrue;
	}
    else if (actor.location && RunRoutines(actor.location, middle)) rtrue;
    if (inp1 > 1 && RunRoutines(inp1, middle)) { GamePostRoutine(); rtrue; }
    rfalse;
];

[ AfterRoutines rv;
    scope_reason = REACT_AFTER_REASON; parser_one = 0;
    SearchScope(actor, 0); scope_reason = PARSING_REASON;
    if (parser_one ~= 0) rtrue;
    if (action == ##GoneFrom) {
		if (RunRoutines(actor.old_loc, after)) rtrue;
	}
    else if (actor.location && RunRoutines(actor.location, after)) rtrue;
    if (inp1 > 1 && RunRoutines(inp1, after)) rtrue;
    rv = GamePostRoutine();
#IFDEF USE_LIBRARY_EXTENSIONS;
    if (rv == false) rv=LibraryExtensions.RunWhile(ext_gamepostroutine, false);
#ENDIF;
    return rv;
];

[ FinalRoutines;
    scope_reason = REACT_FINAL_REASON; parser_one = 0;
    SearchScope(actor, 0); scope_reason = PARSING_REASON;
    if (parser_one ~= 0) rtrue;
    if (action == ##GoneFrom) {
		if (RunRoutines(actor.old_loc, final)) rtrue;
	}
    else if (actor.location && RunRoutines(actor.location, final)) rtrue;
    if (inp1 > 1 && RunRoutines(inp1, final)) rtrue;
    GamePostRoutine();
];

[ RunLife a j;
    #Ifdef DEBUG;
    if (debug_flag & 2 ~= 0) TraceAction(2, j);
    #Endif; ! DEBUG
    reason_code = j; return RunRoutines(a,life);
];

[ ZRegion addr;
    switch (metaclass(addr)) {      ! Left over from Inform 5
      nothing:          return 0;
      Object, Class:    return 1;
      Routine:          return 2;
      String:           return 3;
    }
];

[ PrintOrRun obj prop flag;
   if ((~~obj provides prop) || obj.prop == NULL) rfalse;
   if (obj.#prop > WORDSIZE) return RunRoutines(obj, prop);
    switch (metaclass(obj.prop)) {
      Class, Object, nothing:
        return RunTimeError(2, obj, prop);
      String:
        print (string) obj.prop;
        if (flag == 0) new_line;
        rtrue;
      Routine:
        return RunRoutines(obj, prop);
    }
  rfalse;
];

[ PrintOrRunVar var flag;
    switch (metaclass(var)) {
      Object:
        print (name) var;
      String:
        print (string) var;
        if (flag == 0) new_line;
      Routine:
        return var();
      default:
        print (char) '(', var, (char) ')';
    }
    rtrue;
];

[ ValueOrRun obj prop arg;
    if (obj.prop < 256) return obj.prop;
    return RunRoutines(obj, prop, arg);
];

[ RunRoutines obj prop arg;
   if ((~~obj provides prop) || obj.&prop == NULL) rfalse;
   if (obj.&prop == 0 && prop >= INDIV_PROP_START) rfalse;
   return obj.prop(arg);
];

#Ifdef TARGET_ZCODE;

[ ChangeDefault prop val a b;
    ! Use assembly-language here because -S compilation won't allow this:
    @loadw 0 5 -> a;
    b = prop-1;
    @storew a b val;
];

#Ifnot; ! TARGET_GLULX

[ ChangeDefault prop val;
    ! Use assembly-language here because -S compilation won't allow this:
    ! #cpv__start-->prop = val;
    @astore #cpv__start prop val;
];

#Endif; ! TARGET_

! ----------------------------------------------------------------------------

[ StartTimer obj timer i;
    for (i=0 : i<active_timers : i++)
        if (the_timers-->i == obj) rfalse;
    for (i=0 : i<active_timers : i++)
        if (the_timers-->i == 0) jump FoundTSlot;
    i = active_timers++;
    if (i >= MAX_TIMERS) { RunTimeError(4); return; }
  .FoundTSlot;
    if (obj.&time_left == 0) { RunTimeError(5,obj); return; }
    the_timers-->i = obj; obj.time_left = timer;
  give obj activetimer;
];

[ StopTimer obj i;
    for (i=0 : i<active_timers : i++)
        if (the_timers-->i == obj) jump FoundTSlot2;
    rfalse;
  .FoundTSlot2;
    if (obj.&time_left == 0) { RunTimeError(5,obj); return; }
    the_timers-->i = 0; obj.time_left = 0;
  give obj ~activetimer;
];

[ StartDaemon obj i;
    for (i=0 : i<active_timers : i++)
        if (the_timers-->i == WORD_HIGHBIT + obj) rfalse;
    for (i=0 : i<active_timers : i++)
        if (the_timers-->i == 0) jump FoundTSlot3;
    i = active_timers++;
    if (i >= MAX_TIMERS) RunTimeError(4);
  .FoundTSlot3;
    the_timers-->i = WORD_HIGHBIT + obj;
  give obj activedaemon;
];

[ StopDaemon obj i;
    for (i=0 : i<active_timers : i++)
        if (the_timers-->i == WORD_HIGHBIT + obj) jump FoundTSlot4;
    rfalse;
  .FoundTSlot4;
    the_timers-->i = 0;
  give obj ~activedaemon;
];

! ----------------------------------------------------------------------------

[ DisplayStatus;
    if (sys_statusline_flag == 0) {
		#Ifndef NO_SCORE;
		sline1 = score;
		#Endif;
		sline2 = turns;
	}
    else { sline1 = the_time/60; sline2 = the_time%60; }
];

[ SetTime t s;
    the_time = t; time_rate = s; time_step = 0;
    sline1 = the_time/60; sline2 = the_time%60;
    if (s < 0) time_step = 0 - s;
];

#Ifndef NO_SCORE;
[ NotifyTheScore;
    #Ifdef TARGET_GLULX;
    glk($0086, 6); ! set note style
    #Endif; ! TARGET_GLULX
    print "^[";  L__M(##Miscellany, 50, score-last_score);  print ".]^";
    #Ifdef TARGET_GLULX;
    glk($0086, 0); ! set normal style
    #Endif; ! TARGET_GLULX
];
#Endif;

! ----------------------------------------------------------------------------
! Light and visibility routines.
! ----------------------------------------------------------------------------

! 0 = darkness
! 1 = full light available
! 2 = demilight	"
! 3 = semilight	"
! -semilight- is brighter than -demilight-.
#Ifdef NO_DARKNESS;
[ LightToSeeBy; rtrue; ];
#Ifnot; ! Darkness allowed
[ LightToSeeBy obj flag j;

  if (obj == 0) { RunTimeError(23); rfalse; }
  flag = 0;

  #Ifdef GRADUAL_LIGHT;
  objectloop (j has demilight && j ofclass Thing or Room)
	{ if (IsVisible(j, obj) == 2) flag = 0;
	  if (IsVisible(j, obj) == 1) flag = 2;
	  
	}
  objectloop (j has semilight && j ofclass Thing or Room)
	{ if (IsVisible(j, obj) == 2) flag = 2;
	  if (IsVisible(j, obj) == 1) flag = 3;
	  
	}
  #Endif; ! Gradual light allowed

  objectloop (j has light && IsVisible(j, obj) == 2)
	{ flag = 3; }

  objectloop (j has light && IsVisible(j, obj) == 1)
	{ flag = 1; }

  return flag;
];
#Endif; ! Darkness allowed

#Ifdef GRADUAL_LIGHT;
[ DescribeQuasilight obj i m;
  m = 0;
  objectloop (i ofclass Thing && i.location == obj)
	{ if (i has quasilight && IsVisible(i))
		{ m++; give i workflag; }
	  else give i ~workflag;
	}

  if (m == 0) return;
  else L__M(##Look, 10, obj);
];
#Endif;

[ RoomWithFloater room floater address j i;

	if (~~floater provides found_in) rfalse;

	address = floater.&found_in;
	if (floater.found_in == true) rtrue;
	if (metaclass(address-->0) == Routine) {
		if (floater.found_in(room) == true || floater.found_in() == room) rtrue;
	}

	j = floater.#found_in;
	i = floater.&found_in-->j;
	while (j > -1) {
		if (i == room) rtrue;
		j--;
		i = floater.&found_in-->j;
	}
	rfalse;
];

! flag = 1: 
[ Owner obj flag i;

  while (parent(obj)) {
	i = parent(obj);
	if (flag == 1) {
		if (i ofclass Person) jump Finish;
	}
	obj = i;
	}
  .Finish;
  return i;
];

! = 0 if invisible (including darkness).
! = 1 if visible.
! = 2 if semivisible (only under GRADUAL_LIGHT)
[ IsVisible obj1 obj2 i j o p address;
  if (obj1 == 0) { RunTimeError(21); rfalse; }
  if (obj2 == 0) obj2 = actor;
  if (scope_reason == EACH_TURN_REASON && obj1 == player) rfalse; ! prevents an each_turn bug
  if (obj1 == obj2 || obj1 in compass || obj2 in compass) rtrue;

  if (~~(obj1 provides visibility_ceiling && obj2 provides visibility_ceiling)) rfalse;
  if (obj1 == obj2.visibility_ceiling || obj2 == obj1.visibility_ceiling) rtrue;

  i = obj1.visibility_ceiling;
  if (i == 0 && ~~(obj1 provides found_in)) rfalse;
  if (i provides found_in && i.location == 0) obj1 = i;


  i = obj2.visibility_ceiling;
  if (i == 0 && ~~(obj2 provides found_in)) rfalse;
  if (i provides found_in && i.location == 0) obj2 = i;

  i = obj2;

  if (obj1 provides found_in) { obj2 = obj1; obj1 = i; }

  if (obj2 provides found_in && parent(obj2) == 0) {
	if (obj1 provides found_in && parent(obj1) == 0) {
		objectloop (j && j ofclass Room) {
			if (RoomWithFloater(j, i)) rtrue;
		}
		rfalse;
	}
	address = obj2.found_in;
	if (metaclass(address) == Routine) {
		i = obj2.found_in(obj1.visibility_ceiling);
		if (i == true) rtrue;
	}
	else {
		j = obj2.#found_in;
		i = obj2.&found_in-->j;
		while (j > -1) {
			if (i == obj1.visibility_ceiling || true) rtrue;
			j--;
			i = obj2.&found_in-->j;
		}
	}
	rfalse;
  }

#Ifndef GRADUAL_LIGHT;
  if (obj1.visibility_ceiling == obj2.visibility_ceiling) {
  i = CommonAncestor(obj1, obj2);

  if (i && i ofclass Container && i hasnt transparent) {
	if ((IndirectlyContains(i, obj1) == 4 && IndirectlyContains(i, obj2) == 2) ||
	(IndirectlyContains(i, obj1) == 2 && IndirectlyContains(i, obj2) == 4))
		rfalse;
  }

  if (o) obj1.visibility_ceiling = parent(obj1);
  if (p) obj1.visibility_ceiling = parent(obj2);
  rtrue;
  }
#Ifnot;

  if (obj1.visibility_ceiling == obj2.visibility_ceiling) {
  i = parent(obj1); j = parent(obj2);

  if (i == j && i ofclass Container && i hasnt transparent or semitransparent) {
	if ((IndirectlyContains(i, obj1) == 4 && IndirectlyContains(i, obj2) == 2) ||
	(IndirectlyContains(i, obj1) == 2 && IndirectlyContains(i, obj2) == 4))
		rfalse;
	}

	if ((i has semitransparent || j has semitransparent) && OtherSide(obj1, obj2)) return 2;

  if (o) obj1.visibility_ceiling = parent(obj1);
  if (p) obj1.visibility_ceiling = parent(obj2);
  rtrue;
  }
#Endif; ! Gradual light

  if (o) obj1.visibility_ceiling = parent(obj1);
  if (p) obj1.visibility_ceiling = parent(obj2);

  rfalse;
];

[ OtherSide obj1 obj2 i j m n;
  i = parent(obj1);

  while (i ofclass Container || i ofclass Room) {
	if (i hasnt closed || i ofclass Room) m = true; else m = false; i = parent(i);
  }

  j = parent(obj2);

  while (j ofclass Container || j ofclass Room) {
	if (j hasnt closed || j ofclass Room) n = true; else n = false; j = parent(j);
  }

  if (m == false || n == false) rtrue;
];

! ----------------------------------------------------------------------------

[ ChangePlayer obj flag i;
    if (turns == START_MOVE)
	{ actor = obj; player = obj; }
    else if (actor == player) actor = obj;
    give player ~transparent ~concealed;
    i = obj; while (parent(i) ~= 0) {
        if (i has animate) give i transparent;
        i = parent(i);
    }

#Ifndef CHARACTER;
    if (player == selfobj) player.short_name = FORMER__TX;
#Endif;

    player = obj;

#Ifndef CHARACTER;
    if (player == selfobj) player.short_name = NULL;
#Endif;

    give player transparent concealed animate proper topic;
    print_player_flag = flag;
    if (turns ~= START_MOVE) {
	if (parent(player) == 0) return RunTimeError(10);
	player.location = WhereIs(player);
    }
];

! ----------------------------------------------------------------------------

#Ifdef DEBUG;

#Ifdef TARGET_ZCODE;

[ DebugParameter w;
    print w;
    if (w >= 1 && w <= top_object) print " (", (name) w, ")";
    if (UnsignedCompare(w, dict_start) >= 0 &&
            UnsignedCompare(w, dict_end) < 0 &&
            (w - dict_start) % dict_entry_size == 0)
        print " ('", (address) w, "')";
];

[ DebugAction a anames;
    #Iftrue (Grammar__Version == 1);
    if (a >= 256) { print "<fake action ", a-256, ">"; return; }
    #Ifnot;
    if (a >= 4096) { print "<fake action ", a-4096, ">"; return; }
    #Endif; ! Grammar__Version
    anames = #identifiers_table;
    anames = anames + 2*(anames-->0) + 2*48;
    print (string) anames-->a;
];

[ DebugAttribute a anames;
    if (a < 0 || a >= 48) print "<invalid attribute ", a, ">";
    else {
        anames = #identifiers_table; anames = anames + 2*(anames-->0);
        print (string) anames-->a;
    }
];

#Ifnot; ! TARGET_GLULX

[ DebugParameter w endmem;
    print w;
    @getmemsize endmem;
    if (w >= 1 && w < endmem) {
        if (w->0 >= $70 && w->0 < $7F) print " (", (name) w, ")";
        if (w->0 >= $60 && w->0 < $6F) print " ('", (address) w, "')";
    }
];

[ DebugAction a str;
    if (a >= 4096) { print "<fake action ", a-4096, ">"; return; }
    if (a < 0 || a >= #identifiers_table-->7) print "<invalid action ", a, ">";
    else {
        str = #identifiers_table-->6;
        str = str-->a;
        if (str) print (string) str; else print "<unnamed action ", a, ">";
    }
];

[ DebugAttribute a str;
    if (a < 0 || a >= NUM_ATTR_BYTES*8) print "<invalid attribute ", a, ">";
    else {
        str = #identifiers_table-->4;
        str = str-->a;
        if (str) print (string) str; else print "<unnamed attribute ", a, ">";
    }
];

#Endif; ! TARGET_

[ TraceAction source ar;
    if (source ofclass Person) print "[ (", (The) actor, ") Action ", (DebugAction) action;
    else {
        if (ar == ##Order)
            print "[ Order to ", (name) actor, ": ", (DebugAction) action;
        else
            print "[ Life rule ", (DebugAction) ar;
    }
    if (noun ~= 0)   print " with noun ", (DebugParameter) noun;
    if (second ~= 0) print " and second ", (DebugParameter) second;
    if (source == 0) print " ";
    if (source == 1) print " (from < > statement) ";
    print "]^";
];

[ DebugToken token;
    AnalyseToken(token);
    switch (found_ttype) {
      ILLEGAL_TT:
        print "<illegal token number ", token, ">";
      ELEMENTARY_TT:
        switch (found_tdata) {
          NOUN_TOKEN:           print "noun";
          HELD_TOKEN:           print "held";
          MULTI_TOKEN:          print "multi";
          MULTIHELD_TOKEN:      print "multiheld";
          MULTIEXCEPT_TOKEN:    print "multiexcept";
          MULTIINSIDE_TOKEN:    print "multiinside";
          CREATURE_TOKEN:       print "creature";
          SPECIAL_TOKEN:        print "special";
          NUMBER_TOKEN:         print "number";
          TOPIC_TOKEN:          print "topic";
          ENDIT_TOKEN:          print "END";
        }
      PREPOSITION_TT:
        print "'", (address) found_tdata, "'";
      ROUTINE_FILTER_TT:
        #Ifdef INFIX;
        print "noun=", (InfixPrintPA) found_tdata;
        #Ifnot;
        print "noun=Routine(", found_tdata, ")";
        #Endif; ! INFIX
      ATTR_FILTER_TT:
        print (DebugAttribute) found_tdata;
      SCOPE_TT:
        #Ifdef INFIX;
        print "scope=", (InfixPrintPA) found_tdata;
        #Ifnot;
        print "scope=Routine(", found_tdata, ")";
        #Endif; ! INFIX
      GPR_TT:
        #Ifdef INFIX;
        print (InfixPrintPA) found_tdata;
        #Ifnot;
        print "Routine(", found_tdata, ")";
        #Endif; ! INFIX
    }
];

[ DebugGrammarLine pcount;
    print " * ";
    for (: line_token-->pcount ~= ENDIT_TOKEN : pcount++) {
        if ((line_token-->pcount)->0 & $10) print "/ ";
        print (DebugToken) line_token-->pcount, " ";
    }
    print "-> ", (DebugAction) action_to_be;
    if (action_reversed) print " reverse";
];

#Ifdef TARGET_ZCODE;

[ ShowVerbSub address lines da meta i j;
    if (noun == 0 || ((noun->#dict_par1) & 1) == 0)
        "Try typing ~showverb~ and then the name of a verb.";
    meta = ((noun->#dict_par1) & 2)/2;
    i = $ff-(noun->#dict_par2);
    address = (HDR_STATICMEMORY-->0)-->i;
    lines = address->0;
    address++;
    print "Verb ";
    if (meta) print "meta ";
    da = HDR_DICTIONARY-->0;
    for (j=0 : j<(da+5)-->0 : j++)
        if (da->(j*9 + 14) == $ff-i) print "'", (address) (da + 9*j + 7), "' ";
    new_line;
    if (lines == 0) "has no grammar lines.";
    for (: lines>0 : lines--) {
        address = UnpackGrammarLine(address);
        print "    "; DebugGrammarLine(); new_line;
    }
];

#Ifnot; ! TARGET_GLULX

[ ShowVerbSub address lines i j meta wd dictlen entrylen;
    if (noun == 0 || ((noun->#dict_par1) & 1) == 0)
        "Try typing ~showverb~ and then the name of a verb.";
    meta = ((noun->#dict_par1) & 2)/2;
    i = $ff-(noun->#dict_par2);
    address = (#grammar_table)-->(i+1);
    lines = address->0;
    address++;
    print "Verb ";
    if (meta) print "meta ";
    dictlen = #dictionary_table-->0;
    entrylen = DICT_WORD_SIZE + 7;
    for (j=0 : j<dictlen : j++) {
        wd = #dictionary_table + WORDSIZE + entrylen*j;
        if (wd->#dict_par2 == $ff-i)
            print "'", (address) wd, "' ";
    }
    new_line;
    if (lines == 0) "has no grammar lines.";
    for (: lines>0 : lines--) {
        address = UnpackGrammarLine(address);
        print "    "; DebugGrammarLine(); new_line;
    }
];

#Endif; ! TARGET_

[ ShowobjSub c f l a n x numattr;
    if (noun == 0) noun = player.location;
    objectloop (c ofclass Class) if (noun ofclass c) { f++; l=c; }
    if (f == 1) print (name) l, " ~"; else print "Object ~";
    print (name) noun, "~ (", noun, ") ";
    if (parent(noun) ~= 0)
        if (parent(noun) ofclass Container) {
		if (noun.insideofparent) print "(in ", (name) parent(noun), " ", parent(noun), ")";
		if (noun.ontopofparent) print "(on ", (name) parent(noun), " ", parent(noun), ")";
		if (noun.behindparent) print "(behind ", (name) parent(noun), " ", parent(noun), ")";
		if (noun.beneathparent) print "(under ", (name) parent(noun), " ", parent(noun), ")";
		if (noun.attachedtoparent) print "(attached to ", (name) parent(noun), " ", parent(noun), ")";
	}
    else print "(in ", (name) parent(noun), " ", parent(noun), ")";
    new_line;
    if (f > 1) {
        print "  class ";
        objectloop (c ofclass Class) if (noun ofclass c) print (name) c, " ";
        new_line;
    }
    #Ifdef TARGET_ZCODE;
    numattr = 48;
    #Ifnot; ! TARGET_GLULX
    numattr = NUM_ATTR_BYTES * 8;
    #Endif; ! TARGET_
    for (a=0,f=0 : a<numattr : a++) if (noun has a) f=1;
    if (f) {
        print "  has ";
        for (a=0 : a<numattr : a++) if (noun has a) print (DebugAttribute) a, " ";
        new_line;
    }
    if (noun ofclass Class) return;

    f=0;
    #Ifdef TARGET_ZCODE;
    l = #identifiers_table-->0;
    #Ifnot; ! TARGET_GLULX
    l = INDIV_PROP_START + #identifiers_table-->3;
    #Endif; ! TARGET_
    for (a=1 : a<=l : a++) {
        if ((a ~= 2 or 3) && noun.&a) {
            if (f == 0) { print "  with "; f=1; }
            print (property) a;
            n = noun.#a;
            for (c=0 : WORDSIZE*c<n : c++) {
                print " ";
                x = (noun.&a)-->c;
                if (a == name) print "'", (address) x, "'";
                else {
                    if (a == number or capacity or time_left) print x;
                    else {
                        switch (x) {
                          NULL: print "NULL";
                          0:    print "0";
                          1:    print "1";
                          default:
                            switch (metaclass(x)) {
                              Class, Object:
                                print (name) x;
                              String:
                                print "~", (string) x, "~";
                              Routine:
                                print "[...]";
                           }
                           print " (", x, ")";
                        }
                    }
                }
            }
            print ",^       ";
        }
    }
!   if (f==1) new_line;
];

#Endif; ! DEBUG

! ----------------------------------------------------------------------------
!  Miscellaneous display routines used by DrawStatusLine and available for
!  use.  Most of these vary according to which machine is being compiled to.
! ----------------------------------------------------------------------------

#Ifdef TARGET_ZCODE;

[ ClearScreen window;
    switch (window) {
      WIN_ALL:    @erase_window -1;
      WIN_STATUS: @erase_window 1;
      WIN_MAIN:   @erase_window 0;
    }
];

#Iftrue (#version_number == 6);
[ MoveCursorV6 line column  charw;  ! 1-based postion on text grid
    @get_wind_prop 1 13 -> charw; ! font size
    charw = charw & $FF;
    line = 1 + charw*(line-1);
    column = 1 + charw*(column-1);
    @set_cursor line column;
];
#Endif;

#Ifndef MoveCursor;
[ MoveCursor line column;  ! 1-based position on text grid
    if (~~statuswin_current) {
         @set_window 1;
         if (clr_on && clr_bgstatus > 1) @set_colour clr_fgstatus clr_bgstatus;
         else                            style reverse;
    }
    if (line == 0) {
        line = 1;
        column = 1;
    }
    #Iftrue (#version_number == 6);
    MoveCursorV6(line, column);
    #Ifnot;
    @set_cursor line column;
    #Endif;
statuswin_current = true;
];
#Endif;

[ MainWindow;
    if (statuswin_current) {
        if (clr_on && clr_bgstatus > 1) @set_colour clr_fg clr_bg;
        else                            style roman;
        @set_window 0;
        }
    statuswin_current = false;
];

#Iftrue (#version_number == 6);
[ ScreenWidth  width charw;
    @get_wind_prop 1 3 -> width;
    @get_wind_prop 1 13 -> charw;
    charw = charw & $FF;
    return (width+charw-1) / charw;
];
#Ifnot;
[ ScreenWidth;
    return (HDR_SCREENWCHARS->0);
];
#Endif;

[ ScreenHeight;
    return (HDR_SCREENHLINES->0);
];

#Iftrue (#version_number == 6);
[ StatusLineHeight height  wx wy x y charh;
    ! Split the window. Standard 1.0 interpreters should keep the window 0
    ! cursor in the same absolute position, but older interpreters,
    ! including Infocom's don't - they keep the window 0 cursor in the
    ! same position relative to its origin. We therefore compensate
    ! manually.
    @get_wind_prop 0 0 -> wy; @get_wind_prop 0 1 -> wx;
    @get_wind_prop 0 13 -> charh; @log_shift charh $FFF8 -> charh;
    @get_wind_prop 0 4 -> y; @get_wind_prop 0 5 -> x;
    height = height * charh;
    @split_window height;
    y = y - height + wy - 1;
    if (y < 1) y = 1;
    x = x + wx - 1;
    @set_cursor y x 0;
    gg_statuswin_cursize = height;
];
#Ifnot;
[ StatusLineHeight height;
    if (gg_statuswin_cursize ~= height)
        @split_window height;
    gg_statuswin_cursize = height;
];
#Endif;

[ SetColour f b window;
    if (clr_on && f && b) {
        if (window == 0) {  ! if setting both together, set reverse
            clr_fgstatus = b;
            clr_bgstatus = f;
            }
        if (window == 1) {
            clr_fgstatus = f;
            clr_bgstatus = b;
        }
        if (window == 0 or 2) {
            clr_fg = f;
            clr_bg = b;
        }
        if (statuswin_current)
            @set_colour clr_fgstatus clr_bgstatus;
        else
            @set_colour clr_fg clr_bg;
    }
];


#Ifnot; ! TARGET_GLULX

[ ClearScreen window;
    if (window == WIN_ALL or WIN_MAIN) {
        glk($002A, gg_mainwin);
        if (gg_quotewin) {
            glk($0024, gg_quotewin, 0); ! close_window
            gg_quotewin = 0;
        }
    }
    if (gg_statuswin && window == WIN_ALL or WIN_STATUS) glk($002A, gg_statuswin);
];

[ MoveCursor line column;  ! 0-based postion on text grid
    if (gg_statuswin) {
        glk($002F, gg_statuswin); ! set_window
    }
    if (line == 0) {
        line = 1;
        column = 1;
    }
    glk($002B, gg_statuswin, column-1, line-1); ! window_move_cursor
    statuswin_current=1;
];

[ MainWindow;
    glk($002F, gg_mainwin); ! set_window
    statuswin_current=0;
];

[ MakeColourWord c;
    if (c > 9) return c;
    c = c-2;
    return $ff0000*(c&1) + $ff00*(c&2 ~= 0) + $ff*(c&4 ~= 0);
];

[ ScreenWidth  id;
    id=gg_mainwin;
    if (gg_statuswin && statuswin_current) id = gg_statuswin;
    glk($0025, id, gg_arguments, 0); ! window_get_size
    return gg_arguments-->0;
];

[ ScreenHeight;
    glk($0025, gg_mainwin, 0, gg_arguments); ! window_get_size
    return gg_arguments-->0;
];

[ SetColour f b window doclear  i fwd bwd swin;
    if (clr_on && f && b) {
        if (window) swin = 5-window; ! 4 for TextGrid, 3 for TextBuffer

        fwd = MakeColourWord(f);
        bwd = MakeColourWord(b);
        for (i=0 : i<=10: i++) {
            if (f == CLR_DEFAULT || b == CLR_DEFAULT) {  ! remove style hints
                glk($00B1, swin, i, 7);
                glk($00B1, swin, i, 8);
            }
            else {
                glk($00B0, swin, i, 7, fwd);
                glk($00B0, swin, i, 8, bwd);
            }
        }

        ! Now re-open the windows to apply the hints
        if (gg_statuswin) glk($0024, gg_statuswin, 0); ! close_window

        if (doclear || ( window ~= 1 && (clr_fg ~= f || clr_bg ~= b) ) ) {
            glk($0024, gg_mainwin, 0);
            gg_mainwin = glk($0023, 0, 0, 0, 3, GG_MAINWIN_ROCK); ! window_open
            if (gg_scriptstr ~= 0)
                glk($002D, gg_mainwin, gg_scriptstr); ! window_set_echo_stream
        }

        gg_statuswin = glk($0023, gg_mainwin, $12, gg_statuswin_cursize,
           4, GG_STATUSWIN_ROCK); ! window_open
        if (statuswin_current && gg_statuswin) MoveCursor(); else MainWindow();

        if (window ~= 2) {
            clr_fgstatus = f;
            clr_bgstatus = b;
        }
        if (window ~= 1) {
            clr_fg = f;
            clr_bg = b;
        }
    }
];
#Endif;

[ SetClr f b w;
    SetColour (f, b, w);
];

[ RestoreColours;    ! L61007
    gg_statuswin_cursize = -1;
    if (clr_on) { ! check colour has been used
        SetColour(clr_fg, clr_bg, 2); ! make sure both sets of variables are restored
        SetColour(clr_fgstatus, clr_bgstatus, 1, true);
        ClearScreen();
    }
    #Ifdef TARGET_ZCODE;
    #Iftrue (#version_number == 6); ! request screen update
    (0-->8) = (0-->8) | $$00000100;
    #Endif;
    #Endif;
];

! ----------------------------------------------------------------------------
!  Except in Version 3, the DrawStatusLine routine does just that: this is
!  provided explicitly so that it can be Replace'd to change the style, and
!  as written it emulates the ordinary Standard game status line, which is
!  drawn in hardware.
! ----------------------------------------------------------------------------

#Ifndef NO_STATUSLINE;
#Ifdef TARGET_ZCODE;

#IfV5;

#Iftrue (#version_number == 6);
[ DrawStatusLine width x charw scw mvw i;
   (0-->8) = (0-->8) &~ $$00000100;

   StatusLineHeight(gg_statuswin_size);
   ! Now clear the window. This isn't totally trivial. Our approach is to select the
   ! fixed space font, measure its width, and print an appropriate
   ! number of spaces. We round up if the screen isn't a whole number
   ! of characters wide, and rely on window 1 being set to clip by default.
   MoveCursor(1, 1);
   @set_font 4 -> x;
   width = ScreenWidth();
   spaces width;
   ! Back to standard font for the display. We use output_stream 3 to
   ! measure the space required, the aim being to get 50 characters
   ! worth of space for the location name.
   MoveCursor(1, 2);
   @set_font 1 -> x;

   #Ifndef NO_DARKNESS;
   if (LightToSeeBy(player) == false)  print (string) DARKNESS__TX;
        else print (name) player.location;
   #Ifnot;
	print (name) player.location;
   #Endif;

    i = parent(player);
    if (i ~= player.location) {
	  if (i provides entered_desc) PrintorRun (i, entered_desc, true);
	  else {
        if (player.insideofparent == true)	L__M(##Look, 2, i);
	if (player.ontopofparent == true)		L__M(##Look, 1, i);
	if (player.behindparent == true)		L__M(##Look, 8, i);
        if (player.beneathparent == true)		L__M(##Look, 9, i); }
    }
    if (print_player_flag == 1) L__M(##Look, 3, player);
   @get_wind_prop 1 3 -> width;
   @get_wind_prop 1 13 -> charw;
   charw = charw & $FF;
   @output_stream 3 StorageForShortName;
   print (string) SCORE__TX, "00000";
   @output_stream -3; scw = HDR_PIXELSTO3-->0 + charw;
   @output_stream 3 StorageForShortName;
   print (string) MOVES__TX, "00000";
   @output_stream -3; mvw = HDR_PIXELSTO3-->0 + charw;
   if (width - scw - mvw >= 50*charw) {
       x = 1+width-scw-mvw;
       @set_cursor 1 x; print (string) SCORE__TX, sline1;
       x = x+scw;
       @set_cursor 1 x; print (string) MOVES__TX, sline2;
   }
   else {
       @output_stream 3 StorageForShortName;
       print "00000/00000";
       @output_stream -3; scw = HDR_PIXELSTO3-->0 + charw;
       if (width - scw >= 50*charw) {
           x = 1+width-scw;
           @set_cursor 1 x; print sline1, "/", sline2;
       }
   }
   ! Reselect roman, as Infocom's interpreters go funny
   ! if reverse is selected twice.
   MainWindow();
];

#Endif; ! #version_number == 6

#Endif; ! V5

#Endif; ! TARGET_ZCODE

#Ifndef DrawStatusLine;
[ DrawStatusLine width posa posb i;
    #Ifdef TARGET_GLULX;
    ! If we have no status window, we must not try to redraw it.
    if (gg_statuswin == 0)
        return;
    #Endif;

    ! If there is no player location, we shouldn't try to draw status window
    if (player.location == nothing || parent(player) == nothing)
        return;

    StatusLineHeight(gg_statuswin_size);
    MoveCursor(1, 1);

    width = ScreenWidth();
    posa = width-26; posb = width-13;
    spaces width;

    MoveCursor(1, 2);

    #Ifndef NO_DARKNESS;
    if (LightToSeeBy(player) == false)  print (string) DARKNESS__TX;
        else print (name) player.location;
    #Ifnot;
	print (name) player.location;
    #Endif;

    i = parent(player);
    if (i ~= player.location) {
	  if (i provides entered_desc) PrintorRun(i, entered_desc, true);
	  else {
        if (player.insideofparent == true)	L__M(##Look, 2, i);
	if (player.ontopofparent == true)		L__M(##Look, 1, i);
	if (player.behindparent == true)		L__M(##Look, 8, i);
        if (player.beneathparent == true)		L__M(##Look, 9, i); }
    }
    if (print_player_flag == 1) L__M(##Look, 3, player);

    if (sys_statusline_flag && width > 53) {
        MoveCursor(1, posa);
        print (string) TIME__TX;
        LanguageTimeOfDay(sline1, sline2);
    }
    else {
        if (width > 66) {
            #Ifndef NO_SCORE;
            MoveCursor(1, posa);
            print (string) SCORE__TX, sline1;
            #Endif;
            MoveCursor(1, posb);
            print (string) MOVES__TX, sline2;
        }
        #Ifndef NO_SCORE;
        if (width > 53 && width <= 66) {
            MoveCursor(1, posb);
            print sline1, "/", sline2;
        }
        #Endif;
    }

    MainWindow(); ! set_window
];
#Endif;

#Endif; ! NO_STATUSLINE;

#Ifdef TARGET_GLULX;

[ StatusLineHeight hgt parwin;
    if (gg_statuswin == 0) return;
    if (hgt == gg_statuswin_cursize) return;
    parwin = glk($0029, gg_statuswin); ! window_get_parent
    glk($0026, parwin, $12, hgt, 0); ! window_set_arrangement
    gg_statuswin_cursize = hgt;
];

[ Box__Routine maxwid arr ix lines lastnl parwin;
    maxwid = 0; ! squash compiler warning
    lines = arr-->0;

    if (gg_quotewin == 0) {
        gg_arguments-->0 = lines;
        ix = InitGlkWindow(GG_QUOTEWIN_ROCK);
	#IFDEF USE_LIBRARY_EXTENSIONS;
        if (ix == false) ix = LibraryExtensions.RunWhile(ext_InitGlkWindow, 0, GG_QUOTEWIN_ROCK);
	#ENDIF;
        if (ix == 0)
            gg_quotewin = glk($0023, gg_mainwin, $12, lines, 3,
                GG_QUOTEWIN_ROCK); ! window_open
    }
    else {
        parwin = glk($0029, gg_quotewin); ! window_get_parent
        glk($0026, parwin, $12, lines, 0); ! window_set_arrangement
    }

    lastnl = true;
    if (gg_quotewin) {
        glk($002A, gg_quotewin); ! window_clear
        glk($002F, gg_quotewin); ! set_window
        lastnl = false;
    }

    ! If gg_quotewin is zero here, the quote just appears in the story window.

    glk($0086, 7); ! set blockquote style
    for (ix=0 : ix<lines : ix++) {
        print (string) arr-->(ix+1);
        if (ix < lines-1 || lastnl) new_line;
    }
    glk($0086, 0); ! set normal style

    if (gg_quotewin) {
        glk($002F, gg_mainwin); ! set_window
    }
];

#Endif; ! TARGET_GLULX

#Ifdef TARGET_GLULX;

[ GGInitialise res;
    @gestalt 4 2 res; ! Test if this interpreter has Glk.
    if (res == 0) {
      ! Without Glk, we're entirely screwed.
      quit;
    }
    ! Set the VM's I/O system to be Glk.
    @setiosys 2 0;

    ! First, we must go through all the Glk objects that exist, and see
    ! if we created any of them. One might think this strange, since the
    ! program has just started running, but remember that the player might
    ! have just typed "restart".

    GGRecoverObjects();

    res = InitGlkWindow(0);
#IFDEF USE_LIBRARY_EXTENSIONS;
    if (res == false) res = LibraryExtensions.RunWhile(ext_InitGlkWindow, 0, 0);
#ENDIF;
    if (res ~= 0) return;

    ! Now, gg_mainwin and gg_storywin might already be set. If not, set them.

    if (gg_mainwin == 0) {
        ! Open the story window.
        res = InitGlkWindow(GG_MAINWIN_ROCK);
	#IFDEF USE_LIBRARY_EXTENSIONS;
        if (res == false) res = LibraryExtensions.RunWhile(ext_InitGlkWindow, 0, GG_MAINWIN_ROCK);
	#ENDIF;
        if (res == 0)
            gg_mainwin = glk($0023, 0, 0, 0, 3, GG_MAINWIN_ROCK); ! window_open
        if (gg_mainwin == 0) {
            ! If we can't even open one window, there's no point in going on.
            quit;
        }
    }
    else {
        ! There was already a story window. We should erase it.
        glk($002A, gg_mainwin); ! window_clear
    }

    if (gg_statuswin == 0) {
        res = InitGlkWindow(GG_STATUSWIN_ROCK);
	#IFDEF USE_LIBRARY_EXTENSIONS;
        if (res == false) res = LibraryExtensions.RunWhile(ext_InitGlkWindow, 0, GG_STATUSWIN_ROCK);
	#ENDIF;
        if (res == 0) {
            gg_statuswin_cursize = gg_statuswin_size;
            gg_statuswin = glk($0023, gg_mainwin, $12, gg_statuswin_cursize,
                4, GG_STATUSWIN_ROCK); ! window_open
        }
    }
    ! It's possible that the status window couldn't be opened, in which case
    ! gg_statuswin is now zero. We must allow for that later on.

    glk($002F, gg_mainwin); ! set_window

#IFDEF USE_LIBRARY_EXTENSIONS;
    if (InitGlkWindow(1) == false) LibraryExtensions.RunWhile(ext_InitGlkWindow, 0, 1);
#IFNOT;
    InitGlkWindow(1);
#ENDIF;
];

[ GGRecoverObjects id;
    ! If GGRecoverObjects() has been called, all these stored IDs are
    ! invalid, so we start by clearing them all out.
    ! (In fact, after a restoreundo, some of them may still be good.
    ! For simplicity, though, we assume the general case.)
    gg_mainwin = 0;
    gg_statuswin = 0;
    gg_quotewin = 0;
    gg_scriptfref = 0;
    gg_scriptstr = 0;
    gg_savestr = 0;
    gg_statuswin_cursize = 0;
    #Ifdef DEBUG;
    gg_commandstr = 0;
    gg_command_reading = false;
    #Endif; ! DEBUG
    ! Also tell the game to clear its object references.
#IFDEF USE_LIBRARY_EXTENSIONS;
    if (IdentifyGlkObject(0) == false) LibraryExtensions.RunWhile(ext_identifyglkobject, 0, 0);
#IFNOT;
    IdentifyGlkObject(0);
#ENDIF;

    id = glk($0040, 0, gg_arguments); ! stream_iterate
    while (id) {
        switch (gg_arguments-->0) {
            GG_SAVESTR_ROCK: gg_savestr = id;
            GG_SCRIPTSTR_ROCK: gg_scriptstr = id;
            #Ifdef DEBUG;
            GG_COMMANDWSTR_ROCK: gg_commandstr = id;
                                 gg_command_reading = false;
            GG_COMMANDRSTR_ROCK: gg_commandstr = id;
                                 gg_command_reading = true;
            #Endif; ! DEBUG
            default:
		#IFDEF USE_LIBRARY_EXTENSIONS;
		if (IdentifyGlkObject(1, 1, id, gg_arguments-->0) == false)
                         LibraryExtensions.RunWhile(ext_identifyglkobject, false, 1, 1, id, gg_arguments-->0);
		#IFNOT;
		IdentifyGlkObject(1, 1, id, gg_arguments-->0);
		#ENDIF;
        }
        id = glk($0040, id, gg_arguments); ! stream_iterate
    }

    id = glk($0020, 0, gg_arguments); ! window_iterate
    while (id) {
        switch (gg_arguments-->0) {
            GG_MAINWIN_ROCK: gg_mainwin = id;
            GG_STATUSWIN_ROCK: gg_statuswin = id;
            GG_QUOTEWIN_ROCK: gg_quotewin = id;
            default:
		#IFDEF USE_LIBRARY_EXTENSIONS;
		if (IdentifyGlkObject(1, 0, id, gg_arguments-->0) == false)
                        LibraryExtensions.RunWhile(ext_identifyglkobject, false, 1, 0, id, gg_arguments-->0);
		#IFNOT;
		IdentifyGlkObject(1, 0, id, gg_arguments-->0);
		#ENDIF;
        }
        id = glk($0020, id, gg_arguments); ! window_iterate
    }

    id = glk($0064, 0, gg_arguments); ! fileref_iterate
    while (id) {
        switch (gg_arguments-->0) {
            GG_SCRIPTFREF_ROCK: gg_scriptfref = id;
            default:
		#IFDEF USE_LIBRARY_EXTENSIONS;
		if (IdentifyGlkObject(1, 2, id, gg_arguments-->0) == false)
                        LibraryExtensions.RunWhile(ext_identifyglkobject, false, 1, 2, id, gg_arguments-->0);
		#IFNOT;
		IdentifyGlkObject(1, 2, id, gg_arguments-->0);
		#ENDIF;
        }
        id = glk($0064, id, gg_arguments); ! fileref_iterate
    }

    ! Tell the game to tie up any loose ends.
#IFDEF USE_LIBRARY_EXTENSIONS;
    if (IdentifyGlkObject(2) == false)
        LibraryExtensions.RunWhile(ext_identifyglkobject, 0, 2);
#IFNOT;
    IdentifyGlkObject(2);
#ENDIF;
];

! This somewhat obfuscated function will print anything.
! It handles strings, functions (with optional arguments), objects,
! object properties (with optional arguments), and dictionary words.
! It does *not* handle plain integers, but you can use
! DecimalNumber or EnglishNumber to handle that case.
!
! Calling:                           Is equivalent to:
! -------                            ----------------
! PrintAnything()                    <nothing printed>
! PrintAnything(0)                   <nothing printed>
! PrintAnything("string");           print (string) "string";
! PrintAnything('word')              print (address) 'word';
! PrintAnything(obj)                 print (name) obj;
! PrintAnything(obj, prop)           obj.prop();
! PrintAnything(obj, prop, args...)  obj.prop(args...);
! PrintAnything(func)                func();
! PrintAnything(func, args...)       func(args...);

[ PrintAnything _vararg_count obj mclass;
    if (_vararg_count == 0) return;
    @copy sp obj;
    _vararg_count--;
    if (obj == 0) return;

    if (obj->0 == $60) {
        ! Dictionary word. Metaclass() can't catch this case, so we do
        ! it manually.
        print (address) obj;
        return;
    }

    mclass = metaclass(obj);
    switch (mclass) {
      nothing:
        return;
      String:
        print (string) obj;
        return;
      Routine:
        ! Call the function with all the arguments which are already
        ! on the stack.
        @call obj _vararg_count 0;
        return;
      Object:
        if (_vararg_count == 0) {
            print (name) obj;
        }
        else {
            ! Push the object back onto the stack, and call the
            ! veneer routine that handles obj.prop() calls.
            @copy obj sp;
            _vararg_count++;
            @call CA__Pr _vararg_count 0;
        }
        return;
    }
];

! This does the same as PrintAnything, but the output is sent to a
! byte array in memory. The first two arguments must be the array
! address and length; the following arguments are interpreted as
! for PrintAnything. The return value is the number of characters
! output.
! If the output is longer than the array length given, the extra
! characters are discarded, so the array does not overflow.
! (However, the return value is the total length of the output,
! including discarded characters.)

[ PrintAnyToArray _vararg_count arr arrlen str oldstr len;
    @copy sp arr;
    @copy sp arrlen;
    _vararg_count = _vararg_count - 2;

    oldstr = glk($0048); ! stream_get_current
    str = glk($0043, arr, arrlen, 1, 0); ! stream_open_memory
    if (str == 0) return 0;

    glk($0047, str); ! stream_set_current

    @call PrintAnything _vararg_count 0;

    glk($0047, oldstr); ! stream_set_current
    @copy $ffffffff sp;
    @copy str sp;
    @glk $0044 2 0; ! stream_close
    @copy sp len;
    @copy sp 0;

    return len;
];

! And this calls PrintAnyToArray on a particular array, jiggering
! the result to be a Glulx C-style ($E0) string.

Constant GG_ANYTOSTRING_LEN 66;
Array AnyToStrArr -> GG_ANYTOSTRING_LEN+1;

[ ChangeAnyToCString _vararg_count ix len;
    ix = GG_ANYTOSTRING_LEN-2;
    @copy ix sp;
    ix = AnyToStrArr+1;
    @copy ix sp;
    ix = _vararg_count+2;
    @call PrintAnyToArray ix len;
    AnyToStrArr->0 = $E0;
    if (len >= GG_ANYTOSTRING_LEN)
        len = GG_ANYTOSTRING_LEN-1;
    AnyToStrArr->(len+1) = 0;
    return AnyToStrArr;
];

#Endif; ! TARGET_GLULX

! This is a trivial function which just prints a number, in decimal
! digits. It may be useful as a stub to pass to PrintAnything.

[ DecimalNumber num; print num; ];

#IfV5;

#Ifdef VN_1630;
Array StorageForShortName buffer 160;
#Ifnot;
Array StorageForShortName -> 160 + WORDSIZE;
#Endif; ! VN_1630

#Endif; ! V5

#Ifdef TARGET_ZCODE;

! Platform-independent way of printing strings and properties to a
! buffer (defined as length word followed by byte characters).

[ PrintToBuffer buf len a b c;
    @output_stream 3 buf;
    switch (metaclass(a)) {
      String:
        print (string) a;
      Routine:
        a(b, c);
      Object,Class:
        if (b)
            PrintOrRun(a, b, true);
        else
            print (name) a;
    }
    @output_stream -3;
    if (buf-->0 > len) print "Error: Overflow in PrintToBuffer.^";
    return buf-->0;
];

#Ifnot; ! TARGET_GLULX

[ PrintToBuffer buf len a b;
    if (b) {
        if (metaclass(a) == Object && a.#b == WORDSIZE
            && metaclass(a.b) == String)
            buf-->0 = PrintAnyToArray(buf+WORDSIZE, len, a.b);
        else
            buf-->0 = PrintAnyToArray(buf+WORDSIZE, len, a, b);
    }
    else
        buf-->0 = PrintAnyToArray(buf+WORDSIZE, len, a);
    if (buf-->0 > len) buf-->0 = len;
    return buf-->0;
];

#Endif; ! TARGET_

! None of the following functions should be called for zcode if the
! output exceeds the size of the buffer.

[ StringSize a b;
    PrintToBuffer(StorageForShortName, 160, a, b);
    return StorageForShortName-->0;
];

[ PrintCapitalised obj prop flag nocaps centred  length i width;
    ! a variation of PrintOrRun, capitalising the first letter and returning nothing

    if (obj ofclass String || prop == 0) {
        PrintToBuffer (StorageForShortName, 160, obj);
        flag = 1;
    }
    else
        if (obj.#prop > WORDSIZE || metaclass(obj.prop) == Routine or String) {
            PrintToBuffer(StorageForShortName, 160, obj, prop);
        }
        else
            if (obj.prop == NULL) rfalse;
            else                  return RunTimeError(2, obj, prop);

    length = StorageForShortName-->0;
    width = ScreenWidth();
    if (centred && length < width)
        spaces ( (width - length) / 2 );
    if (~~nocaps)
        StorageForShortName->WORDSIZE = UpperCase(StorageForShortName->WORDSIZE);

    for (i=WORDSIZE: i<length+WORDSIZE: i++) print (char) StorageForShortName->i;

    if (flag == 0 && obj.#prop == WORDSIZE && metaclass(obj.prop) == String)
        new_line;
    return;
];

[ Centre a b;
    PrintCapitalised(a, b, 0, 1, 1);
];

[ Cap str nocaps;
    if (nocaps) print (string) str;
    else        PrintCapitalised(str);
];

Global known_giving_flag = 0;

[ PSN__ o;
    if (o == 0) { print (string) NOTHING__TX; rtrue; }
    switch (metaclass(o)) {
      Routine:  print "<routine ", o, ">"; rtrue;
      String:   print "<string ~", (string) o, "~>"; rtrue;
      nothing:  print "<illegal object number ", o, ">"; rtrue;
    }
    if (o == player) { print (string) YOURSELF__TX; rtrue; }

    #Ifdef LanguagePrintShortName;
    if (LanguagePrintShortName(o)) rtrue;
    #Endif; ! LanguagePrintShortName
    if (indef_mode && o.&short_name_indef ~= 0 && PrintOrRun(o, short_name_indef, 1) ~= 0) rtrue;
    if (o.&short_name ~= 0 && PrintOrRun(o, short_name, 1) ~= 0) rtrue;
    print (object) o;
];

[ Indefart o i saveCaps;
    i = indef_mode; indef_mode = 1;
    saveCaps = caps_mode; caps_mode = false;
    if (o has proper) { indef_mode = NULL; print (PSN__) o; indef_mode = i; return; }
    if (o provides article) {
        PrintOrRun(o, article, 1); print " ", (PSN__) o; indef_mode = i;
        return;
    }
    PrefaceByArticle(o, 2); indef_mode = i; caps_mode = saveCaps;
];

[ CInDefArt o i;
    i = indef_mode; indef_mode = 1;
    if (o has proper) { indef_mode = NULL; print (PSN__) o; indef_mode = i; return; }
    if (o provides article) {
        PrintCapitalised(o, article, 1); print " ", (PSN__) o; indef_mode = i;
        return;
    }
    PrefaceByArticle(o, 2, 0, 1); indef_mode = i;
];

[ Defart o i saveCaps;
    i = indef_mode; indef_mode = false;
    saveCaps = caps_mode; caps_mode = false;
    if ((~~o ofclass Object) || o has proper) {
        indef_mode = NULL; print (PSN__) o; indef_mode = i;
        return;
    }
    PrefaceByArticle(o, 1); indef_mode = i; caps_mode = saveCaps;
];

[ CDefart o saveIndef saveCaps;
    #Ifdef YOU__TX;
    if (o == player) {
        print (string) YOU__TX;
        return;
    }
    #Endif;
    saveIndef = indef_mode; indef_mode = false;
    saveCaps = caps_mode; caps_mode = true;
    if ((~~o ofclass Object)) {
        indef_mode = NULL; print (PSN__) o;
    }
    else
    if (o has proper) {
    indef_mode = NULL;
    PrintToBuffer(StorageForShortName, 160, PSN__, o);
    if (caps_mode) StorageForShortName->WORDSIZE =
     UpperCase(StorageForShortName->WORDSIZE);
    for (o=0 : o<StorageForShortName-->0 : o++)
	print (char) StorageForShortName->(o+WORDSIZE);
    }
    else PrefaceByArticle(o, 0);
    indef_mode = saveIndef; caps_mode = saveCaps;
];

[ PrintShortName o i;
    i = indef_mode; indef_mode = NULL;
    PSN__(o); indef_mode = i;
];

[ EnglishNumber n; LanguageNumber(n); ];

[ NumberWord o i n;
    n = LanguageNumbers-->0;
    for (i=1 : i<=n : i=i+2)
        if (o == LanguageNumbers-->i) return LanguageNumbers-->(i+1);
    return 0;
];

[ RandomEntry tab;
    if (tab-->0 == 0) return RunTimeError(8);
    return tab-->(random(tab-->0));
];

! ----------------------------------------------------------------------------
!  Useful routine: unsigned comparison (for addresses in Z-machine)
!    Returns 1 if x>y, 0 if x=y, -1 if x<y
! ----------------------------------------------------------------------------

[ UnsignedCompare x y u v;
    if (x == y) return 0;
    if (x < 0 && y >= 0) return 1;
    if (x >= 0 && y < 0) return -1;
    u = x&~WORD_HIGHBIT; v= y&~WORD_HIGHBIT;
    if (u > v) return 1;
    return -1;
];

! ==============================================================================

#Ifdef NITFOL_HOOKS;          ! Code contributed by Evin Robertson
#Ifdef TARGET_GLULX;          ! Might be nice for Z-machine games too,
                              ! but I'm not going to try to make this work
                              ! given #Ifdef funniness.

Array magic_array -->         ! This is so nitfol can do typo correction /
                              ! automapping / debugging on Glulx games
    $6e66726d $4d616763 $ff0010 ! Goes to 'NfrmMagc'  10 refers to length
    Magic_Global_Dispatch__
    DI__check_word            ! DI__check_word(buf, length)
    PrintShortName
    WV__Pr RV__Pr CA__Pr      ! obj.prop = x; x = obj.prop; obj.prop(x)
    RA__Pr RL__Pr RA__Sc      ! obj.&prop; obj.#prop; class::prop
    OP__Pr OC__Cl             ! obj provides prop; obj ofclass class
    OB__Move OB__Remove
    OB__Parent__ OB__Child__ OB__Sibling__  ! No explicit veneer for these
    ;

[ OB__Parent__ obj; return parent(obj); ];

[ OB__Child__ obj; return child(obj); ];

[ OB__Sibling__ obj; return sibling(obj); ];

[ Magic_Global_Dispatch__ glbl;
    switch (glbl) {
      0:
        return player.location;
      1:
        return Player;
      -1:
        return CompassDirection::number; ! Silliness to make exist RA__Sc
                                         ! Should never be called.
    }
    return magic_array;       ! Silences a warning.
];

[ DI__check_word buf wlen  ix val res dictlen entrylen;
    ! Just like in Tokenise__.  In fact, Tokenise__ could call this if
    ! it wanted, instead of doing this itself.
    if (wlen > DICT_WORD_SIZE) wlen = DICT_WORD_SIZE;
    for (ix=0 : ix<wlen : ix++) {
        gg_tokenbuf->ix = glk($00A0, buf->ix);
    }
    for (: ix<DICT_WORD_SIZE : ix++) {
        gg_tokenbuf->ix = 0;
    }
    val = #dictionary_table + WORDSIZE;
    entrylen = DICT_WORD_SIZE + 7;
    @binarysearch gg_tokenbuf DICT_WORD_SIZE val entrylen dictlen 1 1 res;
    return res;
];

#Endif; ! TARGET_
#Endif; ! NITFOL_HOOKS

! ==============================================================================


#IFDEF USE_LIBRARY_EXTENSIONS;
Object  LibraryExtensions "(Library Extensions)"
  with  RunAll [ prop a1 a2 a3
            obj rval max;
            objectloop (obj in self)
                if (obj provides prop && obj.prop ofclass Routine) {
                    rval = obj.prop(a1, a2, a3);
                    if (rval > max) max = rval;
                    if (self.BetweenCalls) self.BetweenCalls();
                }
            return max;
        ],
        RunUntil [ prop exitval a1 a2 a3
            obj rval;
            objectloop (obj in self)
                if (obj provides prop && obj.prop ofclass Routine) {
                    rval = obj.prop(a1, a2, a3);
                    if (rval == exitval) return rval;
                    if (self.BetweenCalls) self.BetweenCalls();
                }
            return ~~exitval;
        ],
        RunWhile [ prop exitval a1 a2 a3
            obj rval;
            objectloop (obj in self)
                if (obj provides prop && obj.prop ofclass Routine) {
                    rval = obj.prop(a1, a2, a3);
                    if (rval ~= exitval) return rval;
                    if (self.BetweenCalls) self.BetweenCalls();
                }
            return exitval;
        ],

        ext_number_1 0, ! general temporary workspace

        ! can be set to a function (e.g. RestoreWN) meant for execution
        ! after non-terminating calls to extension objects
        ! (via RunUntil/While/All)
        BetweenCalls 0,
        RestoreWN [; wn = self.ext_number_1; ],

        ! Special interception points
        ext_messages            0,  ! Called if LibraryMessages.before()
                                    !    returns false
                                    ! Extensions run while they return false

        ! Cross-platform entry points
        !                             Called/Runs
        ext_afterlife           0,  ! [C2/R1]
        ext_afterprompt         0,  ! [C2/R1]
        ext_amusing             0,  ! [C2/R1]
        ext_beforeparsing       0,  ! [C2/R2]
        ext_chooseobjects       0,  ! [C2/R2]
        ext_darktodark          0,  ! [C2/R1]
        ext_deathmessage        0,  ! [C2/R1]
        ext_gamepostroutine     0,  ! [C2/R2]
        ext_gamepreroutine      0,  ! [C2/R2]
        ext_initialise          0,  ! [C1/R1]
        ext_inscope             0,  ! [C2/R2]
        ext_lookroutine         0,  ! [C2/R1]
        ext_newroom             0,  ! [C2/R1]
        ext_objectdoesnotfit    0,  ! [C2/R2]
        ext_parsenoun           0,  ! [C3/R3]
        ext_parsenumber         0,  ! [C2/R2]
        ext_parsererror         0,  ! [C2/R2]
        ext_printrank           0,  ! [C2/R1]
        ext_printtaskname       0,  ! [C2/R1]
        ext_printverb           0,  ! [C2/R2]
        ext_timepasses          0,  ! [C2/R1]
        ext_unknownverb         0,  ! [C2/R2]
        !                             [C1] = Called in all cases
        !                             [C2] = Called if EP is undefined, or returns false
        !                             [C3] = called if EP is undefined, or returns -1
        !                             [R1] = All extensions run
        !                             [R2] = Extensions run while they return false
        !                             [R3] = Extensions run while they return -1

        #Ifdef TARGET_GLULX;
        ! Glulx entry points
        !                             Called:           Runs:
        ext_handleglkevent      0,  ! if EP undefined   while extensions return false
        ext_identifyglkobject   0,  ! if EP undefined   while extensions return false
        ext_initglkwindow       0,  ! if EP undefined   while extensions return false
        #Endif; ! TARGET_GLULX;

  has   proper;
#ENDIF;

! ==============================================================================

! ==============================================================================

Constant LIBRARY_PARSER;        ! for dependency checking

! ==============================================================================