! NPC Goals for Triform
! beta release 4
!
!
! This extension makes it possible to have self-directed NPCs
! able to build up dynamic, complex plans for achieving specific
! goals. This is a beta release, so it lacks many planned features
! while advanced features are undocumented and there likely are
! unknown bugs, but for many purposes it is already quite useable.
!
! =====================================================================
! OVERVIEW
! =====================================================================
!
! Essentially, each NPC in your game will be assigned one or more
! "queues" that each contain a plan for achieving a goal. Each plan is
! broken down into a series of sub-goals, and each goal is stored in an
! object belonging to the class Goal. As an NPC attempts to carry out
! the goal it will likely encounter difficulties such as locked containers.
! The library will then add new sub-goals such as "find the key that unlocks
! the drawer."
!
! Each queue consists of a main Goal, which is placed in the first
! entry of the array, and necessary sub-Goals placed in
! the following entries. Since the queue property itself may be
! an array, an NPC can have as many parallel goals as you like.
! The focus property array's first entry contains the current
! track to pursue, and the second entry contains the last
! step generated so far. On each turn, the library will attempt
! to choose a track to pursue by balancing priority against
! estimated ease.
!
! By default there can be up to 32 Goals at any given time, and you can
! adjust this by defining the constant MAX_GOALS before including this file.
! Goals will be appropriately named when generated, and destroyed upon
! being accomplished or abandoned as being either impossible or moot.
!
! If you have more than a few NPCs pursuing different goals, you will quickly
! overrun the memory capacity of the Z-Machine and need to compile to Glulx.
!
! The following goal types are possible:
! 1. Go to a room (##Go)
! 2. Perform an action with no direct object (e.g. ##Jump)
! 3. Perform an action with a direct object (and possibly an indirect object)
! 4. Wait for the occurrence of a Condition (see below)
! 5. Find a generic object (e.g., a member of the class Key)
!
! Each Goal object has a 'type' property that corresponds to this list.
!
! =====================================================================
! INITIAL SETUP
! =====================================================================
!
! Include this file once after 3Parser.h, and once again after 3Grammar.h.
! Define the constant TRACK_LOCATIONS before 3Parser.h.
!
! Give every NPC the following properties:
!
!	queue
!	knowledge
!	focus (an array)
!
! Set the queue property to an array created for that NPC, like so:
!
! Array paul_goals --> MAX_GOALS;
!
! Person paul "Paul"
! with name "paul",
!      queue paul_goals,
!      knowledge 0,
!      focus 0 0,
! has animate male;
!
! In your Initialise() routine, include the line StartDaemon(goals_manager);.
!
! You must also include the file 3Pathmaker.h before this file. This is necessary
! for NPCs to be able to move around the game map.
!
! =====================================================================
! CREATING AND ASSIGNING GOALS
! =====================================================================
!
! For example, say Paul's goal at the opening of your game will be to
! find a particular gold coin. You might create this goal like so:
!
! Goal find_the_coin "Goal of finding the scratched coin"
! with what_to_do ##Take,
!      direct_object scratched_coin,
!      type 3;
!
! Then, in your Initialise routine, call AssignGoal(paul, find_the_coin, paul_queue_one).
! Paul will now search for the scratched coin until he finds and takes it.
! He will travel through the map, search containers, and so on. If he encounters a
! locked container he will start searching for its key.
!
! Certain conditions can be placed on the goal. You may want the NPC to rub
! a magic lamp while standing in the cave, at precisely midnight, while
! nobody is watching:
!
! Goal reveal_the_genie "Goal of revealing the genie"
! with what_to_do ##Rub,
!      direct_object magic_lamp,
!      where spooky_cave,
!      when 0 0,
!      secretly true,
!      type 3;
!
! The NPC will try to find and pick up the magic lamp, then go to the cave
! and wait until midnight. The NPC will then rub the lamp as long as no other
! person is in the cave.
!
! You can order an NPC's goals by changing the 'priority' property to a
! higher number. If a Goal's priority is set to -1, it will always take priority
! over every other Goal.
!
! The 'flag' property of a Goal allows you to have it perform some
! special tricks:
!
! 1: When this Goal is completed, pass its direct object to the
!    previous Goal to become that Goal's indirect object
! 2: Repeat this Goal before marking it completed (FIXME not implemented)
! 3: Do not destroy this Goal, and any added steps must be added to
!    a parallel queue -- this is a permanent Goal (FIXME not implemented)
!
! =====================================================================
! NPC KNOWLEDGE
! =====================================================================
!
! There are three ways to set up NPCs' knowledge of the game world. The first is
! to create a Common_Sense object with a number of routines for determining
! where to find an object or how to achieve a goal. For example, in the game world
! it may be common sense that the most likely place to find a particular valuable
! jewel is in the Queen's bedroom:
!
! Object Common_Sense "Common Sense"
! with location_of
!      [ obj;
!         switch (obj) {
!           precious_jewel: return queens_bedroom;
!           default: return 0;
!         }
!      ];
!
! The knowledge returned by Common_Sense will be available to all NPCs.
!
! Be sure to include such an object in your game or you will not be able
! to compile.
!
! The second way is to define the NPC's 'knowledge' property. For example,
! perhaps it is "common sense" that the precious jewel is most likely in
! the Queen's bedroom, but Paul happens to have particular knowledge that he
! stole it and placed it in his hideout:
!
! Person paul "Paul"
! with name "paul",
!      queue paul_queue_one,
!      knowledge 0,
!      focus 0 0,
!      knowledge
!      [ obj x;
!        switch (x) {
!         1: if (obj == precious_jewel) return hideout;
!         default: return 0;
!        }
!      ],
! has animate male;
!
! If paul.knowledge returns 0, the library will consult the
! Common_Sense object. If Common_Sense returns 0, the library will
! have the NPC begin searching for the object.
!
! The following values may be passed to knowledge:
!
! 1: seeking location of an object
! 100: seeking generic object (see below)
! 101: seeking the parent of an object
!
! Finally, with the use of TRACK_LOCATIONS mode, required to use this file,
! every object has a last_seen_npc property, tied to the npc_number property
! possessed by every NPC. For example, if Paul's npc_number is 0 and he last
! saw the precious jewel in his office, then precious_jewel.&last_seen_npc-->0 == office.
!
! =====================================================================
! GENERIC OBJECTS
! =====================================================================
!
! There will be times when an NPC needs to find an object that is a
! member of a class, such as Coin, Key, Battery, etc., and doesn't
! care which one, so long as one is obtained.
!
! For example, say that at the start of the game, Paul is in need of
! a torch, which is of the class Torch. Any Torch will do. So, you
! would set up the following:
!
! Goal find_a_torch "Goal of finding a torch"
! with what_to_do ##Take,
!      direct_object Torch,
!      type 3;
!
! In the Initialise routine, call AssignGoal(paul, find_a_torch, paul_queue_one);.
! Paul will look around for a Torch, with the library selecting likely
! places to look by consulting the sources of NPC knowledge described above.
!
! =====================================================================
! CONDITIONS
! =====================================================================
!
! Conditions can be specified using objects of the class Condition. By
! default there can be up to 16 such custom conditions, which can be
! increased by defining the constant MAX_CONDITIONS before including
! this file.
!
! =====================================================================
! MEMORIES
! =====================================================================
!
! As NPCs move around and carry out actions, memories will be created to
! document what they've done. Each memory is of the class Memory. By default
! there can be up to 128 of them, which is probably an optimistically low
! number and can be increased by defining the constant MAX_MEMORIES before
! including this file.
!
! Memories enable the library to keep track of what's already been done
! by use of the routine NPCHasAlready(). For example, to track whether
! Paul has already unlocked the Queen's jewelry box, call:
!
! if (NPCHasAlready(paul, ##Unlock, queens_jewelry_box) == true) { ... }
!
! =====================================================================
! CUSTOM RULES
! =====================================================================
!
! Finally, NPCIntervene() is an optional entry point called at certain times.
! The Goals Manager will pass the following values to it:
!
! 1: for writing a custom privacy condition:
!	return 0 to use the standard rule (full privacy)
!	return 1 to say privacy condition is satisfied
!	return -1 to say privacy condition is not satisfied
! 2: for writing a post-action routine for custom verbs or fail states
! 3: for handling custom Goal types
! 4: for handling custom Conditions
! 5: for printing custom verbs
!
! If wanted, NPCIntervene() must be defined before inclusion of this file.
!
! =====================================================================
! DEBUGGING
! =====================================================================
!
! The debugging verb "goals" will toggle messages showing the actions
! carried out by this extension.
!
! =====================================================================
! FUTURE PLANS
! =====================================================================
!
! Planned features for future releases include NPCs' plans being
! rewritten based on updated knowledge about the game state; setting
! up Goals of waiting for the occurrence of a Condition; NPCs having
! imperfect knowledge of the game map; NPCs using paths that
! avoid particular rooms; and NPC acquisitiveness (grabbing items
! that aren't needed now but may be useful later).

! bug: check all FIXME
! extend: ranges for time (allow when to be an array or routine)

message " Processing library extension 3Goals...";

#IFNDEF TRACK_LOCATIONS;
Constant TRACK_LOCATIONS;
#ENDIF;

#IFDEF GOALS;
#IFNDEF common_sense;
Object common_sense "Common Sense"
with	location_of
	[ o;
		switch (o) {
			default: return 0;
		}
	],
	container_of
	[ o;
		switch (o) {
			default: return 0;
		}

	],
	how_to
	[ action;

		switch (action) {
			default: return 0;
		}

	],
	output 0 0 0 0 0 0 0;
#ENDIF;
#IFNDEF NPCIntervene; #Stub NPCIntervene 0; #ENDIF;
#IFDEF DEBUG;
[ XGoalsSub;
	if (~~debug_goals) {
		debug_goals = true;
		"[Goals debugging messages ON.]";
	}

	debug_goals = false;
	"[Goals debugging messages OFF.]";
];
Verb meta 'goals'	*			-> XGoals;
#ENDIF;
#ENDIF;

#IFNDEF GOALS;
system_file;

Constant GOALS;

#IFDEF DEBUG;
Global debug_goals = false;
#ENDIF;

#IFNDEF MAX_GOALS;
Constant MAX_GOALS = 32;
#ENDIF;
#IFNDEF MAX_MEMORIES;
Constant MAX_MEMORIES = 128;
#ENDIF;
#IFNDEF MAX_CONDITIONS;
Constant MAX_CONDITIONS = 16;
#ENDIF;

Array match_list_g --> 64;
Array match_scores_g --> 64;

Property focus;
Property knowledge;
Property queue;

Class Key
 class Thing,
 with name "key";

Class LightSource
 class Thing;

Class Goal(MAX_GOALS)
with	type 1,
	what_to_do -1,
	direct_object -1,
	indirect_object -1,
	where -1,
	when (-1) (-1),
	secretly false,
	how_to_learn_where -2,
	how_to_learn_when -2,
	dynamically_generated false,
	flag 0,
	linked_to 0,
	priority 1,
	short_name
	[;
		if (self.dynamically_generated == false) rfalse;
		print "Goal of ";
		PrintActionName(self.what_to_do); print "ing ";
		if (self.what_to_do == ##Go) print "to ";
		if (self.direct_object) print (the) self.direct_object;
		if (self.indirect_object) switch (self.what_to_do) {
			##Ask, ##Consult, ##TalkTo, ##Tell:
							print " about ", (the) self.indirect_object;
			##AskFor:			print " for ", (the) self.indirect_object;
			##EmptyT, ##Give, ##Show:	print " to ", (the) self.indirect_object;
			##Insert:			print " into ", (the) self.indirect_object;
			##PutBehind:			print " on ", (the) self.indirect_object;
			##PutOn:			print " behind ", (the) self.indirect_object;
			##PutUnder:			print " under ", (the) self.indirect_object;
			##Swing:			print " at ", (the) self.indirect_object;
			default:			print " using ", (the) self.indirect_object;
		}
		return;
	],
	create
	[ x y z;
		self.what_to_do = x;
		self.direct_object = y;
		self.indirect_object = z;
		if (self.what_to_do == ##Go) self.type = 1;
		else if (self.direct_object ofclass Thing) self.type = 3;
		else if (self.direct_object ofclass Condition) self.type = 4;
		else if (self.direct_object ofclass Class) self.type = 5;
		else if (self.indirect_object ofclass Class) self.type = 5;
		else self.type = 2;
		self.dynamically_generated = true;

#IFDEF DEBUG;
if (debug_goals ~= 0) print "^[", (the) self, " created]^";
#ENDIF;

		move self to goals_manager;
	],
	destroy
	[ i j p q;

		objectloop (p provides queue) {
			for (i=0:i<p.#queue/WORDSIZE:i++) {
				q = p.&queue-->i;

				for (j=0:j<MAX_GOALS:j++) {
					if (q-->j == self) q-->j = 0;
				}
			}
			if (p.&focus-->1 == self) p.&focus-->1 = 0;
		}
#IFDEF DEBUG;
if (debug_goals) print "^[", (the) self, " destroyed]^";
#ENDIF;
	],
has	proper;

Class Memory(MAX_MEMORIES)
with	who_did_it,
	what_happened,
	direct_object,
	indirect_object,
	where,
	when,
	secretly,
	attempt,
	dynamically_generated false,
	short_name
	[;
		if (self.dynamically_generated == false) rfalse;
		print "Memory that ";
		print (name) self.who_did_it, " ";
		if (self.attempt) {
			print "attempted to ";
			PrintActionName(self.what_happened); print " ";
		}
		else { PrintActionName(self.what_happened); print "ed "; }
		if (self.direct_object) print (the) self.direct_object;
		if (self.indirect_object) switch (self.what_to_do) {
			##Ask, ##Consult, ##TalkTo, ##Tell:
							print " about ", (the) self.indirect_object;
			##AskFor:			print " for ", (the) self.indirect_object;
			##EmptyT, ##Give, ##Show:	print " to ", (the) self.indirect_object;
			##Insert:			print " into ", (the) self.indirect_object;
			##PutBehind:			print " on ", (the) self.indirect_object;
			##PutOn:			print " behind ", (the) self.indirect_object;
			##PutUnder:			print " under ", (the) self.indirect_object;
			##Swing:			print " at ", (the) self.indirect_object;
			default:			print " using ", (the) self.indirect_object;
		}
		return;
	],
	create
	[ p a;

		if (NPCHasAlready(p, p.&last_action-->0, p.&last_action-->1, p.&last_action-->2)) rfalse;

		a = p.&last_action-->0;
		self.who_did_it = p;
		self.what_happened = a;
		a = p.&last_action-->1;
		self.direct_object = a;
		a = p.&last_action-->2;
		self.indirect_object = a;
		self.where = p.location;
		self.when = turns;
		if (failed_action) self.attempt = true;
		self.dynamically_generated = true;

#IFDEF DEBUG;
if (debug_goals) print "^[", (the) self, " created]^";
#ENDIF;

		move self to memories_manager;
	],
has	proper;

Object memories_manager "Memories Manager";

Class Condition(MAX_CONDITIONS)
with	dynamically_generated false,
	met 0,
	type 0,
	conditions 0 0,
	short_name
	[;
		if (self.dynamically_generated == false) rfalse;
		print "Condition of ";

		return;
	],
	daemon
	[ a b;
		switch (self.type) {
			1:
				a = self.&conditions-->0;
				b = self.&conditions-->1;

				if (a has b) { self.met = true; StopDaemon(self); }

			2:
				a = self.&conditions-->0;
				b = self.&conditions-->1;

				if (a hasnt b) { self.met = true; StopDaemon(self); }

			3:
				a = self.&conditions-->0;
				b = self.&conditions-->1;

				if (a.b == true) { self.met = true; StopDaemon(self); }

			4:
				a = self.&conditions-->0;
				b = self.&conditions-->1;
 
				if (a.b == false) {self.met = true; StopDaemon(self); }

			5:
				a = self.&conditions-->0;
				b = self.&conditions-->1;

				if (a in b) { self.met = true; StopDaemon(self); }

			6:
				a = self.&conditions-->0;
				b = self.&conditions-->1;

				if (a.location == b) { self.met = true; StopDaemon(self); }

			7:
				a = self.&conditions-->0;
				b = self.&conditions-->1;

				if (a notin b) { self.met = true; StopDaemon(self); }

			8:
				a = self.&conditions-->0;
				b = self.&conditions-->1;

				if (a.location ~= b) { self.met = true; StopDaemon(self); }

			9:
				a = self.&conditions-->0;
				b = self.&conditions-->1;

				if (a == b) { self.met = true; StopDaemon(self); }

			10:
				a = self.&conditions-->0;
				b = self.&conditions-->1;

				if (a ~= b) { self.met = true; StopDaemon(self); }

			default: NPCIntervene(4, self);
		}
	],
	create
	[ a b c;

		self.&conditions-->0 = a;
		self.&conditions-->1 = b;
		self.type = c;

		move self to conditions_manager; StartDaemon(self);

#IFDEF DEBUG;
if (debug_goals) print "^[", (the) self, " created]^";
#ENDIF;
	],
has	proper;

Object conditions_manager "Conditions Manager";

Class Plan
with	steps 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0;

Object goals_manager "Goals Manager",
with	daemon
	[ g k o1 o2 p q s t1 t2 w x y;

		action_mode = 0;

! Loop over every active NPC

		objectloop (p provides queue && p ~= player && p has animate) {

! Identify which Goal this NPC should be working on

		g = ChooseGoal(p); if (~~g) jump FinishThisNPC;
		q = p.&focus-->0;

#IFDEF DEBUG;
if (debug_goals) print "^[Goals Manager thinks ", (the) p, " should be working on ", (the) g, "]^";
#ENDIF;

! Attempt to carry out the identified Goal, possibly generating new Goals
! along the way

		.Start;
		k = 0; t1 = 0; t2 = 0; s = 0; w = 0; x= 0; y = 0;

			if (g) switch (g.type) {
				1: ! A Goal of going to a Room or direction

					! goal of going to a direction

					if (g.direct_object in Compass) {
						o2 = g.direct_object; jump GoThere;
					}

					! goal of going to the location of an object

					else if (g.direct_object ofclass Thing) {
						o2 = (g.direct_object).&last_seen_npc-->(p.npc_number);
						if (o2 == 0) o2 = (g.direct_object).&last_known_npc-->(p.npc_number);
					}

					! if neither of those, must be goal of going to a Room

					else o2 = g.direct_object;

					#Ifdef USE_Pathmaker;
					o2 = PathMaker.determine_path(p, o2);
					#Endif;

					! Determine the direction to travel from the current location.

					o1 = p.location;

					objectloop (k in compass && k provides door_dir) {
						x = k.door_dir;
						if (o1 provides x) {
							if (o1.x == o2) { o2 = k; break; }
							if (o1.x ofclass Door) {
								y = o1.x;
								if (y.door_to(p) == o2) { o2 = k; break; }
							}
						}
					}

					.GoThere;

					informlibrary.actor_act(p, ##Go, o2); AfterAction(p, g, q);

				2: ! A noun-less Goal such as jumping in place

! For the (where) property of a goal:
!	-1 = anywhere is ok
!	-2 = the npc must figure out the correct location
!	a room = the place where it must be done

					if (ValueOrRun(g, where) == -2) {
						k = p.knowledge(g, 1); ! FIXME
						if (k == -1) {
							k = common_sense.location_of(g, p);
							if (k ofclass Room) {
								g = Goal.create(##Go, g.where);
								AssignGoal(p, g, q);
								jump Start;
							}
						}
						if (k ofclass Room) {
							g = Goal.create(##Go, g.where);
							AssignGoal(p, g, q);
							jump Start;
						}
					}

					else if (g.where ~= -1 or p.location) {
						g = Goal.create(##Go, g.where);
						AssignGoal(p, g, q);
						jump Start;
					}

! For the (when) property of a goal:
!	-1 = any time is ok
!	-2 = the npc must figure out the correct time (not implemented)
!	a positive number = the precise time when it must be done

					if (g.&when-->0 ~= -1) {
						if (~~(((g.&when-->0) < the_time) && (the_time < (g.&when-->1))))
							jump FinishThisNPC;
					}

! The NPC should now be in the correct time and place.

! Must the action be carried out in privacy?

					if (g.secretly == true) {
						switch (NPCIntervene(1)) {
							-1: jump FinishThisNPC;

							0:
							objectloop (x ofclass Person && x has animate && x ~= p && IsVisible(p, x))
								jump FinishThisNPC;

							1: ;
						}

					}

! Presumably all conditions are met.

					informlibrary.actor_act(p, g.what_to_do);
					AfterAction(p, g, q);

				3: ! A Goal of performing an action on an object, e.g. "take the jewel"

					o1 = g.direct_object; o2 = g.indirect_object;

! First, the easy case: the direct and indirect objects are immediately accessible

					if (o2) {
						if (IsVisible(o1, p) && IsVisible(o2, p) && g.what_to_do ~= ##Ask or ##Tell)
							jump StartType3;
							else jump MissingObj;
					}
					else if (IsVisible(o1, p) && g.what_to_do ~= ##Ask or ##Tell) jump StartType3;
						else jump MissingObj;

					.StartType3;

! For the (where) property of a goal:
!	-1 = anywhere is ok
!	-2 = the npc must figure out the correct location
!	a room = the place where it must be done

					if (ValueOrRun(g, where) == -2) {
						k = p.knowledge(g, 1);
						if (k ofclass Room) {
							g = Goal.create(##Go, g.where);
							AssignGoal(p, g, q);
							jump Start;
						}
						if (k == -1) {
							k = common_sense.location_of(g, p);
							if (k ofclass Room) { ! d
								g = Goal.create(##Go, g.where);
								AssignGoal(p, g, q);
								jump Start;
							}
						}
					}


					else if (ValueOrRun(g, where) ~= -1 or p.location) {
						g = Goal.create(##Go, g.where);
						AssignGoal(p, g, q);
						jump Start;
					}

! For the (when) property of a goal:
!	-1 = any time is ok
!	-2 = the npc must figure out the correct time (not implemented)
!	a positive number = the precise time when it must be done

						if (g.&when-->0 ~= -1) {
							if (~~((g.&when-->0) < turns && turns < (g.&when-->1)))
								jump FinishThisNPC;
						}

! Secrecy is not important:

						if (ValueOrRun(g, secretly) == false) {
							informlibrary.actor_act(p, g.what_to_do, o1, o2);
							AfterAction(p, g, q);
							jump FinishThisNPC;
						}

! Secrecy is important:

						switch (NPCIntervene(1)) {
							-1: jump FinishThisNPC;

							0:
							objectloop (x ofclass Person && x ~= p && IsVisible(x))
								jump FinishThisNPC;

							1: ;
						}
						informlibrary.actor_act(p, g.what_to_do, o1, o2);
						AfterAction(p, g, q);
						jump FinishThisNPC;

					.MissingObj;

					! If the indirect object is not visible, find it.
					! If it is, find the direct object.

					if (o2 && IsVisible(o2, p) == false && g.what_to_do ~= ##Ask or ##Tell)
						o1 = o2;

					! Avoid being caught in a loop.

					if (o1 == g.direct_object && g.indirect_object ~= 0 && g.what_to_do ~= ##Take) {
						g = Goal.create(##Take, o1);
						AssignGoal(p, g, q);
						jump Start;
					}

					! Has the NPC seen the object somewhere?

					if (o1 ofclass Thing) k = o1.&last_seen_npc-->(p.npc_number);

					! If not, consult the NPC's knowledge property.

					if (~~k) {
						if (o1 ofclass Thing) k = p.knowledge(o1, 1);
							else k = p.knowledge(o1, 100); ! generic class object sought
					}

					! If not, is there a common-sense location?

					if (~~k) k = common_sense.location_of(o1, p);

					! FIXME what if the NPC needs to search around?

					! If the NPC is not already there, create a Goal of going there.

					if (k && k ofclass Room && k ~= p.location) {
						g = Goal.create(##Go, k); AssignGoal(p, g, q);
						jump Start;
					}

					! Now, does the NPC have special knowledge of the object's
					! container?

					k = p.knowledge(o1, 101);

					! If not, is there a common-sense container that hasn't
					! already been searched?

					if (~~k) {

						k = common_sense.container_of(o1, p);

						! If the common-sense container has already been searched,
						! or if common sense came up with nothing, choose the first
						! closed, opaque container in the present location.

						if ((~~k) || NPCHasAlready(p, ##Search, k) == true) {
							objectloop (x ofclass Container && IsVisible(x, p)) {
								if (x hasnt transparent && x has closed) {
									if (NPCHasAlready(p, ##Search, k) == false) k = x;
								}
							}
						}

						! The NPC doesn't have a clue as to the location of the object.

						if (~~k) {

							.GoSomewhereElse;

							k = ChooseLocation(p);
							if (k) {
								g = Goal.create(##Go, k);
								AssignGoal(p, g, q);
								jump Start;
							}
							jump FinishThisNPC;
						}
					}

					! The NPC has (at least guessed) a container to try.
						
					if (k ofclass Container) {
						if (IsVisible(k, p)) {
							common_sense.how_to(##Search, k, p);
							x  = common_sense.&output->0; ! The action

							if (x) {
								k  = common_sense.&output->1; ! The direct object (if any)
								y  = common_sense.&output->2; ! The indirect object (if any)
								t1 = common_sense.&output->3; ! Start time (if important)
								t2 = common_sense.&output->4; ! End time (if important)
								w  = common_sense.&output->5; ! Where (if important)
								s  = common_sense.&output->6; ! Secretly (if important)
							}
							else x = ##Search;
							g = Goal.create(x, k, y);
							if (t1) g.&when-->0 = t1;
							if (t2) g.&when-->1 = t2;
							if (w) g.where = w;
							if (s) g.secretly = true;
							AssignGoal(p, g, q);
							informlibrary.actor_act(p, x, k, y);
							AfterAction(p, g, q);
							jump FinishThisNPC;
						}

						k = common_sense.location_of(k, p);

						if (k && k ofclass Room) {
							if (k ~= p.location) {
								g = Goal.create(##Go, k);
								AssignGoal(p, g, q);
								jump FinishThisNPC;
							}
						}
					}

					! The container is (presumably) in the NPC's location, but must be found.

					g = Goal.create(##Examine, k);
					AssignGoal(p, g, q);
					jump FinishThisNPC;

				4: ! Waiting for the occurrence of a Condition (not implemented)

					o1 = g.direct_object;

					if (o1.met == true) Goal.destroy(g);

				5: ! Goal involving a generic object, e.g. a Key or Coin

					o1 = g.direct_object; o2 = g.indirect_object;

					! If the indirect object is generic, look for that first, regardless
					! of whether the direct object is also.

					if (o2 ofclass Class) {
						x = SearchGeneric(o2, p, g, q);
						if (x == 0) jump FinishThisNPC;
						if (x notin p) {
							y = g;
							switch (g.what_to_do) {
								##Answer, ##Show: ;
								##Insert, ##PutOn, ##PutBehind, ##PutUnder, ##Transfer: ;
								##Attack, ##Unlock:
									g = Goal.create(##Take, x);
									g.flag++; g.linked_to = y;
									AssignGoal(p, g, q);
							}
							jump Start;
						}
						if (x in p) jump Changegoal;
					}

					! The direct object is generic, so look for that.

					x = SearchGeneric(o1, p, g, q);
					if (x == 0) jump FinishThisNPC;

					if (x in p) {
						Goal.destroy(g);
						g = ChooseGoal(p);
						jump Changegoal;
					}

					.ChangeGoal;
#IFDEF DEBUG;
if (debug_goals ~= 0) print "[", (the) g, " changed to ";
#ENDIF;
					g.direct_object = x;
					g.type = 3;
#IFDEF DEBUG;
if (debug_goals ~= 0) print "[", (the) g, ".]^";
#ENDIF;
					jump Start;

				default: ! Unknown type
					if (NPCIntervene(3) == false) {

#IFDEF DEBUG;
if (debug_goals ~= 0) print "^[", (the) g, " is of unknown type and was not executed]^";
#ENDIF;

					jump FinishThisNPC;
					}
			} ! switch (goal.type)

		.FinishThisNPC;
		failed_action = 0;
		x = 0; while (x < 6) { common_sense.&output->x = 0; x++; }
		} ! objectloop
	];


[ ChooseGoal p g i j q1 q2 d r;

	CheckUpdatedKnowledge(p);

! From among the active queues, choose the one with the highest-
! priority end-Goal, except under certain circumstances.

	for (i=0:i<p.#queue/WORDSIZE:i++) {
		q1 = p.&queue-->i; if (q1-->0 == 0) jump EmptyQueue;

		if (q2 == 0) q2 = q1;

		else if ((q1-->0).priority > (q2-->0).priority) {
			q2 = q1;
			j = i;
		}

		.EmptyQueue;
	}

	if (q2 == 0) rfalse;

	! Check the chosen queue against the other active queues,
	! to see if it may be worth pausing this one to take
	! advantage of an opportunity.

	if ((q2-->0).priority ~= -1) for (i=0:i<p.#queue/WORDSIZE:i++) {
		q1 = p.&queue-->i; if (q1 == q2) jump SkipQueue;

		g = q1-->0;

		if (g) switch (g.type) {
			1: ! not implemented

			2: ! not implemented

			3:
				d = g.direct_object;
				r = d.location;

				#Ifdef USE_Pathmaker;
				PathMaker.determine_path(p, r);
				d = PathMaker.size;
				#Endif;

				d = g.priority - d; if (d) jump SkipQueue;
				if (g.direct_object in p.location) q2 = q1;

			4: ! not implemented

			5: ! not implemented

			default: ! not implemented
		}

		.SkipQueue;
	}

	for (i=0:i<MAX_GOALS:i++) {
		if (q2-->i ~= 0) g = q2-->i; else break;
	}
	
	p.&focus-->0 = j; p.&focus-->1 = g; return g;
];

! p = the NPC
[ ChooseLocation p d l r;

	l = p.location;

	! Choose a random direction from the current direction.

	objectloop (d in compass && d provides door_dir) {
		if (l provides d.door_dir) {
			r = ValueorRun(l, d.door_dir);

			if (r && r ofclass Room)
				return r;

			if (r && r ofclass Door) {
				r = r.ValueOrRun(d, door_to);
				return r;
			}
		}
	}
];

[ CheckUpdatedKnowledge p g1 g2 q pos a b c;

	if (~~g1) return;

	pos = FindPosition(p, g1);

	q = p.&queue-->0; ! for now

	switch (g1.type) {

		1: ! going to a Room

			! Delete all the sub-goals up to the changed goal, and then modify the goal
			! to reflect the new Room.

		2: ! performing a noun-less action such as jumping in place

		3: ! performing an action on a specific object


			! first case: change in knowledge of object's location
			! e.g., NPC thought the diamond was in the vault, but it's actually in the office

			! possible sub-cases:
			! 1. new location is a different-enough part of the map that the plan for accessing the old
			! location should be abandoned.
			!
			! 2. new location is in a similar-enough part of the map that the plan just needs to be modified.

			a = p.knowledge(g1.direct_object, 1); ! a is the location where the item is now thought to be

			g2 = q-->(++pos); b = g2.where; ! b is the previous location it was thought to be

			if (a ~= b) {

				! if c is 0, we are in sub-case 1

				c = PathMaker.compare_paths(a, p.location, b, p.location, true);
			}

			if (c) {

				g2.where = c;
				return;
			}

			g2.where = c; pos++;
			for (:pos<32:pos++) {
				Goal.destroy(q-->(pos)); q-->pos = 0;
			}
			return;

		4: ! waiting for the occurrence of an action

		5: ! performing an action involving a generic object, e.g. a Key or Coin
	}
];

[ AssignGoal person goal q i x y;

	if (goal == 0 || person == 0) rfalse;

	x = person.&queue-->q;
	for (y=0:y<MAX_GOALS:y++) {
		i = x-->y;
		if (i == 0) break;
	}

	x-->y = goal;

#IFDEF DEBUG;
if (debug_goals ~= 0) {
print "^[", (the) goal, " assigned to ", (the) person, " at position ", (number) y,
" and queue ", (number) q, "]"; if (goal.linked_to) print " [Linked to ", (the) goal.linked_to, "]";
new_line; }
#ENDIF;
];

[ AssignPlan pl p q g i;

	if (~~pl ofclass Plan) rfalse;
	if (~~p ofclass Person) rfalse;

	for (i=0:i<pl.#steps:i++) {
		g = pl.&steps-->i; AssignGoal(person, g, q);
	}

#IFDEF DEBUG;
if (debug_goals ~= 0) 
print "^[", (the) plan, " assigned to ", (the) person, "]^";
#ENDIF;
];

[ DeleteGoal person i q;

	q = person.&queue-->i;

	for (i=0:i<MAX_GOALS:i++) { q-->i = 0; }
];

[ WipeGoals person i;

	for (i=0:i<person.#queue:i++) {
		person.&queue-->i = 0;
	}
];

[ WipeEveryGoal p;

	objectloop (p provides queue) WipeGoals(p);
];

[ FindPosition p g1 g2 i q;

	q = p.&queue-->0; ! FIXME extend to include other queues

	for (i=0:i<MAX_GOALS:i++) {
		g2 = q-->i;
		if (g2 == g1) return i;
	}

	return -1;
];

[ NPCHasAlready p a o1 o2 m;

	if (a == ##Search)
		a = common_sense.how_to(##Search, o1, p); ! FIXME: hmmm

	objectloop (m in memories_manager) {
		if (m.who_did_it == p && m.what_happened == a
		    && m.direct_object == o1 && m.indirect_object == o2) rtrue;
	}
	rfalse;
];

[ AfterAction p G q G2 i o1 o2;

	! First, remember the action that just occurred.

	Memory.create(p);

	! Next, take appropriate steps depending on whether that action
	! was successful or unsuccessful.

	! If successful, wipe that Goal and choose a new one, possibly
	! making use of information from it.

	if (failed_action == 0) {
		if (g.flag == 1) i = g.direct_object;
		Goal.destroy(G);
		if (i) {
#IFDEF DEBUG;
if (debug_goals ~= 0) print "[", (the) G, " changed to ";
#ENDIF;
			G = ChooseGoal(p);
			G.indirect_object = i;
			G.type = 3;
#IFDEF DEBUG;
if (debug_goals ~= 0) print "[", (the) G, ".]^";
#ENDIF;
		}
	}

	! Now, the more complicated situation of what to do if the
	! NPC's action was unsuccessful.

	else switch (p.&last_action-->0) {

		! Determine why the attempt failed, and generate a
		! corresponding Goal

		##Disrobe:
			switch (failed_action) {
				3: ! locked: find key
					o1 = p.&last_action-->1;
					o2 = ValueOrRun(o1, with_key);
					if (o2 == -1) {
						G = Goal.create(##Take, Key);
					}
					else G = Goal.create(##Unlock, o1, o2);
					AssignGoal(p, G, q);

				default: NPCIntervene(2);
			}
		##EmptyT:
			switch (failed_action) {
				2: ! closed: open
					o1 = p.&last_action-->1;

					G = Goal.create(##Open, o1);
					AssignGoal(p, G, q);

				default: NPCIntervene(2);
			}
		##Enter:
			switch (failed_action) {
				8: ! not enough space: take something out
					o2 = parent(p.&last_action-->2);

					objectloop (i in o2 && i.insideofparent) {
						if (i.volume >= p.volume) {
							G = Goal.create(##Take, i);
							AssignGoal (p, G, q);
							break;
						}
					}

				default: NPCIntervene(2);
			}
		##Examine:
			switch (failed_action) {
				1: ! too dark: find a light source
				! FIXME: this won't really work well

					o1 = p.&last_action-->1;

					common_sense.how_to(##Examine, o1);
					G  = Goal.create(common_sense.&output->0, common_sense.&output->1, common_sense.&output->2);
					AssignGoal(p, G, q);

				default: NPCIntervene(2);
			}
		##Exit:
			switch (failed_action) {
				2: ! closed: open
					o1 = p.&last_action-->1;

					G = Goal.create(##Open, o1);
					AssignGoal(p, G, q);

				default: NPCIntervene(2);
			}
		##Go:
			switch (failed_action) {
				1: ! must exit container
					o1 = parent(p);

					G = Goal.create(##Exit, o1);
					AssignGoal(p, G, q);

				5: ! something blocking path: move it

				default: NPCIntervene(2);
			}
		##Insert:
			switch (failed_action) {
				2: ! closed: open
					o1 = p.&last_action-->1;

					G = Goal.create(##Open, o1);
					AssignGoal(p, G, q);

				4: ! must disrobe: disrobe
					o1 = p.&last_action-->1;

					G = Goal.create(##Disrobe, o1);
					AssignGoal(p, G, q);

				10: ! not enough space: take something out
					o2 = parent(p.&last_action-->2);

					objectloop (i in o2 && i.insideofparent) {
						if (i.volume >= o1.volume) {
							G = Goal.create(##Take, i);
							AssignGoal (p, G, q);
							break;
						}
					}

				default: NPCIntervene(2);
			}
		##Listen:
			switch (failed_action) {
				2: ! container in the way: open it
					o1 = p.&last_action-->1;

					G = Goal.create(##Open, parent(o1));
					AssignGoal(p, G, q);

				default: NPCIntervene(2);
			}
		##Lock:
			switch (failed_action) {
				2: ! must be closed: close
					o1 = p.&last_action-->1;

					G = Goal.create(##Close, o1);
					AssignGoal(p, G, q);

				6: ! must be opened: open
					o1 = p.&last_action-->1;

					G = Goal.create(##Open, o1);
					AssignGoal(p, G, q);

				default: NPCIntervene(2);
			}
		##Open:
			switch (failed_action) {
				2: ! locked: find key
					o1 = p.&last_action-->1;
					o2 = ValueOrRun(o1, with_key);

					if (o2 == -1) {
						G2 = Goal.create(##Unlock, o1, Key);
						AssignGoal(p, G2, q);
						G = Goal.create(##Take, Key);
						G.flag++; G.linked_to = G2;
					}
					else G = Goal.create(##Unlock, o1, o2);

					AssignGoal(p, G, q);

				default: NPCIntervene(2);
			}
		##PutBehind:
			switch (failed_action) {
				5: ! insufficient space: take something off
					o2 = parent(p.&last_action-->2);

					objectloop (i in o2 && i.behindparent) {
						if (i.volume >= o1.volume) {
							G = Goal.create(##Take, i);
							AssignGoal (p, G, q);
							break;
						}
					}

				default: NPCIntervene(2);
			}
		##PutOn:
			switch (failed_action) {
				5: ! insufficient space: take something off
					o2 = parent(p.&last_action-->2);

					objectloop (i in o2 && i.ontopofparent) {
						if (i.volume >= o1.volume) {
							G = Goal.create(##Take, i);
							AssignGoal (p, G, q);
							break;
						}
					}

				default: NPCIntervene(2);
			}
		##PutUnder:
			switch (failed_action) {
				5: ! insufficient space: take something off
					o2 = parent(p.&last_action-->2);

					objectloop (i in o2 && i.beneathparent) {
						if (i.volume >= o1.volume) {
							G = Goal.create(##Take, i);
							AssignGoal (p, G, q);
							break;
						}
					}

				default: NPCIntervene(2);
			}
		##Smell:
			switch (failed_action) {
				2: ! container in the way: open it
					o1 = parent(p.&last_action-->1);

					G = Goal.create(##Open, o1);
					AssignGoal(p, G, q);

				default: NPCIntervene(2);
			}
		##Take:
			switch (failed_action) {
				9: ! container in the way: open it
					o1 = p.&last_action-->1;

					G = Goal.create(##Open, o1);
					AssignGoal(p, G, q);

				12: ! carrying too much already: drop something
					o1 = p.&last_action-->1;

					objectloop (o2 in p) {
						if (o2.size >= o1.size) {
							informlibrary.actor_act(actor, ##Drop, o2);
							break;
						}
					}

				default: NPCIntervene(2);
			}
		##Touch:
			switch (failed_action) {
				4: ! can't reach

					o1 = cant_touch_reason-->0;

					switch (cant_touch_reason-->1) {
						1: ! the object must be opened
							G = Goal.create(##Open, o1);
							AssignGoal(p, G, q);

						2: ! the actor must exit the object
							G = Goal.create(##Exit, o1);
							AssignGoal(p, G, q);

						3: ! the actor must disrobe the object
							G = Goal.create(##Disrobe, o1);
							AssignGoal(p, G, q);

						4: ! it is being held by another actor
							G = Goal.create(##Take, p.&last_action-->1);
							AssignGoal(p, G, q);
					}
			}
		##Unlock:
			switch (failed_action) {
				7: ! wrong key: find new key

					if (G.indirect_object == Key) return;

					if (G.what_to_do == ##Unlock) {
#IFDEF DEBUG;
if (debug_goals ~= 0) print "[", (the) G, " changed to ";
#ENDIF;
						G.indirect_object = Key;
						G.type = 5;
#IFDEF DEBUG;
if (debug_goals ~= 0) print (the) G, ".]^";
#ENDIF;
					}
					else {
						o1 = p.&last_action-->1;
						G = Goal.create(##Unlock, o1, Key);
						G.type = 5;
						AssignGoal(p, G, q);
					}

				default: NPCIntervene(2);
			}
		#IFDEF COMPLEXCLOTHING;
		##Wear:
			switch (failed_action) {
				9:  ! wearing too-bulky clothing: disrobe
					o1 = cant_touch_reason-->0;

					G = Goal.create(##Disrobe, o1);
					AssignGoal(p, G, q);

				10: ! hasn't put on an essential item yet: wear
				! FIXME: this won't work
					o1 = cant_touch_reason-->0;

					G = Goal.create(##Wear, o1);
					AssignGoal(p, G, q);

				11: ! wearing too much underneath: disrobe
					o1 = cant_touch_reason-->0;

					G = Goal.create(##Disrobe, o1);
					AssignGoal(p, G, q);

				default: NPCIntervene(2);
			}
		#ENDIF;
		default: NPCIntervene(2);
	}
];

[ PrintActionName a;

	switch (a) {
		##Disrobe: print "doff";
		##Drop: print "dropp";
		##Eat: print "eat";
		##EmptyT: print "empty";
		##Enter: print "enter";
		##Examine: print "examin";
		##Exit: print "exit";
		##Give: print "giv";
		##Go: print "go";
		##Insert: print "insert";
		##Jump: print "jump";
		##Lie: print "ly";
		##Listen: print "listen";
		##Lock: print "lock";
		##Open: print "open";
		##Pull: print "pull";
		##Push: print "push";
		##PutOn: print "put";
		##Search: print "search";
		##Show: print "show";
		##Smell: print "smell";
		##Squeeze: print "squeez";
		##SwitchOff: print "switch";
		##SwitchOn: print "switch";
		##Take: print "obtain";
		##Taste: print "taste";
		##Touch: print "touch";
		##Turn: print "turn";
		##Tell: print "tell";
		##Unlock: print "unlock";
		##Wait: print "wait";
		##Wave: print "wav";
		##Wear: print "wear";
		default: if (NPCIntervene(5) == 0) print "[...]";
	}
];

[ SearchGeneric o p g1 q g2 a o1 i k;

	if (metaclass(o) ~= Class) rfalse; ! Just in case.

	ClearMatchList();

	! FIXME: Document this section. It is for determining the
	! action to be performed with the generic object, which may
	! aid in choosing where to look... I guess. Maybe junk this.

	g2 = g1.linked_to;

	if (g2) {
		a = g2.what_to_do;
		o1 = g2.direct_object;
	}

	else {
		a = g1.what_to_do;
		o1 = g1.direct_object;
		if (o1 == o) { o1 = g1.indirect_object; k++; }
	}

	! Find a list of suitable generic objects, then select one of them.

	! First, does the NPC know the locations of any of the items sought?
	! Check the NPC's knowledge, then Common Sense.

	if (k) {
		i = p.knowledge(o, 102);
		if (~~i) i = common_sense.location_of(o, p);
		if (~~i) {
			k = 0;
			objectloop (i ofclass o && i notin o) {
				if (i.&last_seen_npc-->(p.npc_number) ~= 0 && NPCHasAlready(p, a, i, o1) == 0) {
					match_list_g-->k = i; k++;
				}
			}
		}
	}

	else {
		i = p.knowledge(o, 102);
		if (~~i) i = common_sense.location_of(o, p);
		k = 0;
		objectloop (i ofclass o && i notin o) {
			if (i.&last_seen_npc-->(p.npc_number) ~= 0 && NPCHasAlready(p, a, o1, i) == 0) {
				match_list_g-->k = i; k++;
			}
		}
	}

	! Score the match list, and get the highest-scoring item.

	i = ScoreMatchList(k, p, g1);

	! Or, if nothing was chosen, then select a Container to search.
	! First, consult the NPC's knowledge. Then, ask Common Sense.
	! If they give no answer, just start searching random
	! Containers.

	if (i == 0) {
		i = p.knowledge(o, 102);
		if (~~i) i = common_sense.location_of(o, p);
		if (~~i) {
			k = 0; o = Container;
			objectloop (i ofclass o && i notin o) {
				if (i.&last_seen_npc-->(p.npc_number) ~= 0 && NPCHasAlready(p, a, i, o1) == 0) {
					match_list_g-->k = i; k++;
				}
			}
		}
		i = ScoreMatchList(k, p, g1);
		g1 = Goal.create(##Search, i);
		AssignGoal(p, g1, q);
		rfalse;
	}

#IFDEF DEBUG;
if (debug_goals ~= 0) print "[Choosing ", (the) i, ".]^";
#ENDIF;
	return i;
];

[ ScoreMatchList k p g its_score obj i j;

	for (i=0:i<k:i++) {
		obj = match_list_g-->i; its_score = 0;

		! Impose penalty for distance.
		! First, where does the NPC think the object is?

		j = obj.&last_seen_npc-->(p.npc_number);
		if (j == 0) j = p.knowledge(obj, 1);
		if (j == 0) j = common_sense.location_of(obj, p);

		#Ifdef USE_Pathmaker;
		PathMaker.determine_path(p, j); j = 0;
		while (j ~= PathMaker.size) { j++; its_score = its_score - SCORE__BESTLOC; }
		#Ifnot;
		if (j ~= p.location) its_score = its_score - SCORE__BESTLOC;
		#Endif;

		! Impose penalty for being untakeable.

		if (g.what_to_do == ##Take && obj has scenery or static)
			its_score = its_score - SCORE__NOTSCENERY;

		! Impose penalty for being inaccessible.

		if (ObjectIsUntouchable(obj, p, true))
			its_score = its_score - SCORE__NOTSCENERY;

		! Bonus if within the NPC's possession.

		if (obj in p)
			its_score = its_score + SCORE__BESTLOC;
		else if (IndirectlyContains(p, obj) == true)
			its_score = its_score + SCORE__NEXTBESTLOC;

		! Bonus if not the NPC.

		if (obj ~= p)
			its_score = its_score + SCORE__NOTACTOR;


		! Record the score.

		match_scores_g-->i = its_score;
	}

	obj = 0; j = 0;

	for (i=0:i<k:i++) {

		if (match_scores_g-->i > j) { j = match_scores_g-->i; obj = match_list_g-->i; }
		else if (i == 0) { j = match_scores_g-->0; obj = match_list_g-->i; }
	
	}

	return obj;
];

[ ClearMatchList i;

	for (i=0:i<64:i++) { match_list_g-->i = 0; match_scores_g-->i = 0; }
];
#ENDIF;