/****************************************************************************
				   MENU.C

	Biblioteca de funciones para gestionar mens.

			     (c)1995 JSJ Soft Ltd.

	Las siguientes funciones son pblicas:
		- m_crea: crea un men de opciones
		- m_abre: dibuja un men en pantalla
		- m_cierra: borra un men de pantalla
		- m_elimina: elimina un men y libera memoria
		- m_cambia_color: actualiza los colores de un men
		- m_resalta_opcion: resalta la opcin actual de un men
		- m_accion: enva una accin a un men
		- m_elige_opcion: elige una opcin de un men
		- m_color: cambia paleta de colores

	Las siguientes estructuras estn definidas en MENU.H:
		STC_OPCION: opcin de un men
		STC_MENU: men desplegable
		STC_ACCION: accin para un men
****************************************************************************/

#include <stddef.h>
#include <alloc.h>
#include <bios.h>
#include "ventana.h"
#include "raton.h"
#include "menu.h"

/*** Variables globales internas ***/
static BYTE Menu_Clr[]={
	0x70,	/* principal */
	0x7f,	/* sombra 1 */
	0x70,	/* sombra 2 */
	0x74,	/* tecla activacin */
	0x07,	/* opcin seleccionada */
};

/*** Protipos de funciones internas ***/
static char mayuscula(char c);
static void coge_pos_cursor(int *fil, int *col);
static void pon_cursor(int fil, int col);
static char saca_tecla_opcion(char *(*menu), int *longitud);
static STC_MENU *crea_menu(char *opciones);
static void imprime_opcion(STC_MENU *menu, int opcion, int modo);

/****************************************************************************
	MAYUSCULA: convierte una letra en mayscula.
	  Entrada:      'c' carcter a convertir
	  Salida:       mayscula del carcter
****************************************************************************/
char mayuscula(char c)
{

if((c>='a') && (c<='z')) return(c-(char)'a'+(char)'A');

switch(c) {
	case (char)'' :
		c=(char)'';
		break;
	case (char)'' :
		c='A';
		break;
	case (char)'' :
		c='E';
		break;
	case (char)'' :
		c='I';
		break;
	case (char)'' :
		c='O';
		break;
	case (char)'' :
	case (char)'' :
		c='U';
		break;
}

return(c);
}

/****************************************************************************
	COGE_POS_CURSOR: devuelve la posicin del cursor.
	  Entrada:      'fil', 'col' punteros a variables donde se
			devolver la posicin del cursor
	  Salida:       posicin del cursor en las variables a las que
			apuntan 'fil' y 'col'
****************************************************************************/
void coge_pos_cursor(int *fil, int *col)
{
int fila=0, columna=0;

asm {
	mov ah,03h              // funcin buscar posicin del cursor
	mov bh,0                // supone pgina 0
	int 10h
	mov byte ptr fila,dh
	mov byte ptr columna,dl
}

*fil=fila;
*col=columna;

}

/****************************************************************************
	PON_CURSOR: coloca el cursor en una posicin de pantalla.
	  Entrada:      'fil', 'col' fila y columna del cursor
****************************************************************************/
void pon_cursor(int fil, int col)
{

asm {
	mov ah,02h              // funcin definir posicin del cursor
	mov bh,0                // supone pgina 0
	mov dh,byte ptr fil     // DH = fila del cursor
	mov dl,byte ptr col     // DL = columna del cursor
	int 10h
}

}

/****************************************************************************
	SACA_TECLA_OPCION: extrae informacin de la tecla de activacin de
	  una opcin desde una cadena de definicin de men.
	  Entrada:      'menu' puntero a cadena de definicin de las opciones
			del men con el formato:
			  "Opcin1^1:Opcin1^2:Opcin1^3|Opcin1^4"
			el carcter '^' indica la tecla de activacin y
			el carcter '|' coloca una lnea de separacin
			'longitud' puntero a variable donde se dejar la
			longitud del texto de la opcin
	  Salida:       devuelve el cdigo ASCII de la tecla de activacin o
			0 si no la encontr; el puntero 'menu' queda
			actualizado para apuntar a la siguiente opcin y la
			variable 'columna' contiene la ltima columna de la
			opcin actual
****************************************************************************/
char saca_tecla_opcion(char *(*menu), int *longitud)
{
char tecla=0;

*longitud=0;

do {
	switch(*(*menu)) {
		case CAR_SEPAR :
			(*menu)++;
			(*longitud)++;
			return(0);
		case CAR_TECLA :
			(*menu)++;
			tecla=mayuscula(*(*menu));
			break;
	}

	/* siguiente carcter */
	(*menu)++;
	(*longitud)++;
} while(*(*menu) && (*(*menu)!=CAR_FINOPC) && (*(*menu)!=CAR_SEPAR));

/* salta delimitador */
if(*(*menu)==CAR_FINOPC) (*menu)++;

return(tecla);
}

/****************************************************************************
	CREA_MENU: crea un men.
	  Entrada:      'menu' cadena con la lista de opciones del men
			de la forma:
			  "Opcin1^1:Opcin1^2:Opcin1^3|Opcin1^4"
			el carcter '^' indica la tecla de activacin y
			el carcter '|' coloca una lnea de separacin
	  Salida:       puntero a estructura con informacin del men a usar
			por las otras funciones de manejo de mens o NULL si
			hubo algn error
	  NOTA:         el mximo nmero de opciones de un men viene dado
			por la constante MAX_NUMOPCIONES
****************************************************************************/
STC_MENU *crea_menu(char *opciones)
{
STC_MENU *menu;
int i, longitud;

menu=(STC_MENU *)malloc(sizeof(STC_MENU));
if(menu==NULL) return(NULL);

i=0;
while((*opciones) && (i<MAX_NUMOPCIONES)) {
	/* puntero a texto de opcin */
	menu->opc[i].opcion=opciones;
	menu->opc[i].tecla=saca_tecla_opcion(&opciones,&longitud);
	menu->opc[i].lng_opcion=longitud;

	/* siguiente opcin */
	i++;
}

menu->num_opciones=i;

return(menu);
}

/****************************************************************************
	IMPRIME_OPCION: imprime una opcin de un men.
	  Entrada:      'menu' puntero a estructura con los datos del men
			'opcion' nmero de opcin a imprimir
			'modo' modo de impresin, 0 normal o 1 resaltada
****************************************************************************/
void imprime_opcion(STC_MENU *menu, int opcion, int modo)
{
int i, fil, col;
BYTE clr;
char *txt_opcion;

r_puntero(R_OCULTA);

/* selecciona posicin de impresin */
if((menu->tipo & MENU_NFIJO)==MENU_HORZ) {
	fil=menu->v.fil+menu->opc[opcion].fil;
	col=menu->v.col+menu->opc[opcion].col;
}
else {
	fil=menu->v.fil+menu->opc[opcion].fil+1;
	col=menu->v.col+menu->opc[opcion].col+1;
}

/* selecciona color correspondiente */
if(modo==0) clr=Menu_Clr[MCLR_PRINCIPAL];
else clr=Menu_Clr[MCLR_SELECC];

txt_opcion=menu->opc[opcion].opcion;

/* si es separador, lo imprime y sale */
if(*txt_opcion==CAR_SEPAR) {
	for(i=0; i<menu->v.ancho-2; i++) v_impcar(fil,col+i,SEPARADOR,clr);
	r_puntero(R_MUESTRA);
	return;
}

while(*txt_opcion && (*txt_opcion!=CAR_FINOPC) && (*txt_opcion!=CAR_SEPAR)) {
	/* si es carcter de tecla de activacin lo imprime en otro color */
	if(*txt_opcion==CAR_TECLA) {
		txt_opcion++;
		/* lo imprime en otro color slo si no resaltada */
		if(modo==0) v_impcar(fil,col,*txt_opcion,Menu_Clr[MCLR_TECLA]);
		else v_impcar(fil,col,*txt_opcion,clr);
	}
	else {
		/* si es men horizontal, no resalta los espacios */
		if(((menu->tipo & MENU_NFIJO)==MENU_HORZ) &&
		  (*txt_opcion==' ') && (modo!=0)) v_impcar(fil,col,
		  *txt_opcion,Menu_Clr[MCLR_PRINCIPAL]);
		else v_impcar(fil,col,*txt_opcion,clr);
	}

	txt_opcion++;
	col++;
}

/* si es men vertical resalta todo el ancho de la ventana */
if((menu->tipo & MENU_NFIJO)==MENU_VERT) {
	for(i=0; i<(menu->v.ancho-menu->opc[opcion].lng_opcion-2); i++)
	  v_impcar(fil,col++,' ',clr);
}

r_puntero(R_MUESTRA);

}

/****************************************************************************
	M_CREA: crea un men de opciones.
	  Entrada:      'tipo' tipo de men; MENU_HORZ barra de men
			horizontal, MENU_VERT men vertical; se puede aadir
			mediante OR lgico el valor MENU_FIJO para que el
			men no se pueda cerrar
			'titulo' texto de encabezamiento del men (NULL si
			ninguno)
			'opciones' cadena con las opciones del men
			de la forma:
			  "Opcin1^1:Opcin1^2:Opcin1^3|Opcin1^4"
			el carcter '^' indica la tecla de activacin y
			el carcter '|' coloca una lnea de separacin, la
			ltima cadena ser nula ("")
			'fil', 'col' posicin donde se colocar el men
			'separ_opciones' espacio de separacin entre las
			opciones de men de barra
	  Salida:       puntero a estructura con informacin del men a usar
			por las otras funciones de manejo de mens o NULL si
			hubo algn error
	  NOTA:         el mximo nmero de opciones de un men viene dado
			por la constante MAX_NUMOPCIONES
****************************************************************************/
STC_MENU *m_crea(BYTE tipo, char *titulo, char *opciones, int fil,
  int col, int separ_opciones)
{
STC_MENU *menu;
int i, ancho=0, alto, columna=0;

menu=crea_menu(opciones);
if(menu==NULL) return(NULL);

/* calcula dimensiones de la ventana del men y la posicin de */
/* los textos de cada opcin */
if((tipo & MENU_NFIJO)==MENU_HORZ) {
	for(i=0; i<menu->num_opciones; i++) {
		ancho+=menu->opc[i].lng_opcion;
		menu->opc[i].fil=0;
		menu->opc[i].col=columna;
		columna+=(menu->opc[i].lng_opcion+separ_opciones);
	}
	ancho+=(separ_opciones*(menu->num_opciones-1));
	alto=1;
}
else {
	for(i=0; i<menu->num_opciones; i++) {
		if(menu->opc[i].lng_opcion>ancho) {
			ancho=menu->opc[i].lng_opcion;
		}
		menu->opc[i].col=0;
		menu->opc[i].fil=i;
	}
	ancho+=2;
	alto=menu->num_opciones+2;
}

v_crea(&menu->v,fil,col,ancho,alto,Menu_Clr[MCLR_PRINCIPAL],
  Menu_Clr[MCLR_S1],Menu_Clr[MCLR_S2],titulo,1);
menu->tipo=tipo;
if((tipo & MENU_NFIJO)==MENU_HORZ) menu->separ_opc=separ_opciones;
else menu->separ_opc=0;
menu->opcion=0;

return(menu);
}

/****************************************************************************
	M_ABRE: dibuja un men en pantalla.
	  Entrada:      'menu' puntero a estructura de men
****************************************************************************/
void m_abre(STC_MENU *menu)
{
int i;

if((menu->tipo & MENU_NFIJO)==MENU_HORZ) {
	r_puntero(R_OCULTA);
	for(i=menu->v.col; i<(menu->v.col+menu->v.ancho); i++)
	  v_impcar(menu->v.fil,i,' ',Menu_Clr[MCLR_PRINCIPAL]);
	r_puntero(R_MUESTRA);
}
else v_abre(&menu->v);

for(i=0; i<menu->num_opciones; i++) imprime_opcion(menu,i,0);

}

/****************************************************************************
	M_CIERRA: borra un men de pantalla.
	  Entrada:      'menu' puntero a estructura de men
****************************************************************************/
void m_cierra(STC_MENU *menu)
{

if((menu->tipo & MENU_NFIJO)!=MENU_HORZ) v_cierra(&menu->v);

}

/****************************************************************************
	M_ELIMINA: elimina la definicin de un men y libera la memoria
	  ocupada.
	  Entrada:      'menu' puntero a estructura de men
****************************************************************************/
void m_elimina(STC_MENU *menu)
{

m_cierra(menu);
free(menu);

}

/****************************************************************************
	M_CAMBIA_COLOR: actualiza los colores de un men de opciones.
	  Entrada:      'menu' puntero a estructura de men
****************************************************************************/
void m_cambia_color(STC_MENU *menu)
{

menu->v.clr_princ=Menu_Clr[MCLR_PRINCIPAL];
menu->v.clr_s1=Menu_Clr[MCLR_S1];
menu->v.clr_s2=Menu_Clr[MCLR_S2];
menu->v.clr_texto=Menu_Clr[MCLR_PRINCIPAL];

}

/****************************************************************************
	M_RESALTA_OPCION: resalta la opcin actual de un men.
	  Entrada:      'menu' puntero a estructura con los datos del men
			'resalta' 0 quita resalte, 1 pone resalte
****************************************************************************/
void m_resalta_opcion(STC_MENU *menu, int resalta)
{

if(resalta==0) imprime_opcion(menu,menu->opcion,0);
else imprime_opcion(menu,menu->opcion,1);

}

/****************************************************************************
	M_ACCION: enva una accin a un men.
	  Entrada:      'menu' puntero a estructura de men
			'acc' puntero a estructura con la accin a enviar
	  Salida:       respuesta del men que puede ser el nmero de una
			opcin o un cdigo de accin (si es <0)
****************************************************************************/
int m_accion(STC_MENU *menu, STC_ACCION *acc)
{
int i, minfil, mincol, maxfil, maxcol;
char tecla;

switch(acc->accion) {
	case MENU_NULA :        /* accin nula */
		return(MENU_NULA);
	case MENU_SALIDA :      /* salir del men (ESCAPE) */
		m_resalta_opcion(menu,0);
		if(!(menu->tipo & MENU_FIJO)) m_cierra(menu);
		break;
	case MENU_SELECCIONA :  /* selecciona opcin */
		m_resalta_opcion(menu,0);
		return(menu->opcion);
	case MENU_ARRIBA :      /* cursor arriba */
		if((menu->tipo & MENU_NFIJO)!=MENU_HORZ) {
			m_resalta_opcion(menu,0);
			menu->opcion--;
			if(menu->opcion<0) menu->opcion=menu->num_opciones-1;
			if(*menu->opc[menu->opcion].opcion==CAR_SEPAR)
			  menu->opcion--;
			m_resalta_opcion(menu,1);
			return(MENU_NULA);
		}
		break;
	case MENU_ABAJO :       /* cursor abajo */
		if((menu->tipo & MENU_NFIJO)!=MENU_HORZ) {
			m_resalta_opcion(menu,0);
			menu->opcion++;
			if(menu->opcion>(menu->num_opciones-1)) menu->opcion=0;
			if(*menu->opc[menu->opcion].opcion==CAR_SEPAR)
			  menu->opcion++;
			m_resalta_opcion(menu,1);
			return(MENU_NULA);
		}
		break;
	case MENU_IZQUIERDA :   /* cursor izquierda */
		if((menu->tipo & MENU_NFIJO)==MENU_HORZ) {
			m_resalta_opcion(menu,0);
			menu->opcion--;
			if(menu->opcion<0) menu->opcion=menu->num_opciones-1;
			if(*menu->opc[menu->opcion].opcion==CAR_SEPAR)
			  menu->opcion--;
			m_resalta_opcion(menu,1);
			return(MENU_NULA);
		}
		else {
			if(!(menu->tipo & MENU_FIJO)) m_cierra(menu);
			return(MENU_SALIDA);
		}
	case MENU_DERECHA :     /* cursor derecha */
		if((menu->tipo & MENU_NFIJO)==MENU_HORZ) {
			m_resalta_opcion(menu,0);
			menu->opcion++;
			if(menu->opcion>(menu->num_opciones-1)) menu->opcion=0;
			if(*menu->opc[menu->opcion].opcion==CAR_SEPAR)
			  menu->opcion++;
			m_resalta_opcion(menu,1);
			return(MENU_NULA);
		}
		else {
			if(!(menu->tipo & MENU_FIJO)) m_cierra(menu);
			return(MENU_SALIDA);
		}
	case MENU_TECLA :       /* tecla de opcin */
		if(!acc->tecla) return(MENU_NULA);
		/* comprueba si es una tecla de activacin vlida */
		for(i=0; i<menu->num_opciones; i++) {
			tecla=mayuscula(acc->tecla);
			if(tecla==menu->opc[i].tecla) {
				m_resalta_opcion(menu,0);
				menu->opcion=i;
				return(i);
			}
		}
		return(MENU_NULA);
	case MENU_RATON :       /* accin del ratn */
		/* comprueba si coordenadas de ratn estn dentro de */
		/* ventana de men */
		if((menu->tipo & MENU_NFIJO)==MENU_HORZ) {
			minfil=menu->v.fil;
			maxfil=minfil;
		}
		else {
			minfil=menu->v.fil+1;
			maxfil=minfil+menu->v.alto-3;
		}
		mincol=menu->v.col;
		maxcol=mincol+menu->v.ancho-1;
		if((acc->fil>=minfil) && (acc->fil<=maxfil) &&
		  (acc->col>=mincol) && (acc->col<=maxcol)) {
			/* comprueba a que opcin corresponden las */
			/* coordenadas del puntero del ratn */
			if((menu->tipo & MENU_NFIJO)==MENU_VERT)
			  i=acc->fil-minfil;
			else {
				for(i=0; i<menu->num_opciones; i++) {
					mincol=menu->opc[i].col;
					maxcol=menu->opc[i].col+
					  menu->opc[i].lng_opcion+
					  menu->separ_opc-1;
					if((acc->col>=mincol) &&
					  (acc->col<=maxcol)) break;
				}
			}
			/* selecciona opcin si no es separador */
			if(*menu->opc[i].opcion!=CAR_SEPAR) {
				if(menu->opcion!=i) {
					m_resalta_opcion(menu,0);
					menu->opcion=i;
					m_resalta_opcion(menu,1);
				}
			}
		return(MENU_NULA);
		}
		/* si coordenadas del cursor estn fuera, y el botn no */
		/* est pulsado, cierra men */
		else {
			m_resalta_opcion(menu,0);
			if(!(menu->tipo & MENU_FIJO)) m_cierra(menu);
			return(MENU_SALIDA);
		}
}

return(acc->accion);
}

/****************************************************************************
	M_ELIGE_OPCION: elige una opcin de un men.
	  Entrada:      'menu' puntero a estructura de men
	  Salida:       nmero de la opcin seleccionada, -1 si se sali
			del men sin elegir ninguna opcin
****************************************************************************/
int m_elige_opcion(STC_MENU *menu)
{
static int boton1=0;
STC_ACCION acc;
STC_RATON r;
int accion, opcion, tecla, minfil, mincol, maxfil, maxcol, filcur, colcur;

/* si ratn no est inicializado, lo inicializa */
if(!r_puntero(R_MUESTRA)) r_inicializa();

/* oculta cursor */
coge_pos_cursor(&filcur,&colcur);
pon_cursor(25,0);

m_resalta_opcion(menu,1);

acc.accion=MENU_NULA;

while(1) {
	do {
		accion=0;
		r_estado(&r);
		/* si pulsado el botn 1 del ratn, indica accin del ratn */
		if(r.boton1) {
			accion=1;
			boton1=1;
		}
		/* si anteriormente estaba pulsado el botn 1 y ahora est */
		/* suelto, y dentro del men, indica seleccin con ratn */
		else if(boton1) {
			/* comprueba si puntero de ratn est dentro de */
			/* ventana de men */
			if((menu->tipo & MENU_NFIJO)==MENU_HORZ) {
				minfil=menu->v.fil;
				maxfil=minfil;
			}
			else  {
				minfil=menu->v.fil+1;
				maxfil=minfil+menu->v.alto-3;
			}
			mincol=menu->v.col;
			maxcol=mincol+menu->v.ancho-1;
			if((r.fil>=minfil) && (r.fil<=maxfil) &&
			  (r.col>=mincol) && (r.col<=maxcol)) accion=2;
			boton1=0;
		}
		else {
			/* si hay tecla esperando, indica accin de teclado */
			tecla=bioskey(1);
			if(tecla) {
				bioskey(0);
				accion=3;
			}
		}
	} while(!accion);

	/* si es accin de ratn */
	if(accion==1) {
		acc.accion=MENU_RATON;
		acc.fil=r.fil;
		acc.col=r.col;
	}
	/* seleccin con el ratn */
	else if(accion==2) acc.accion=MENU_SELECCIONA;
	else switch((tecla >> 8) & 0x00ff) {
		case 0x01 :     /* ESCAPE */
			acc.accion=MENU_SALIDA;
			break;
		case 0x1c :     /* RETURN */
			acc.accion=MENU_SELECCIONA;
			break;
		case 0x48 :     /* Cursor arriba */
			acc.accion=MENU_ARRIBA;
			break;
		case 0x50 :     /* Cursor abajo */
			acc.accion=MENU_ABAJO;
			break;
		case 0x4b :     /* Cursor izquierda */
			acc.accion=MENU_IZQUIERDA;
			break;
		case 0x4d :     /* Cursor derecha */
			acc.accion=MENU_DERECHA;
			break;
		default :
			acc.accion=MENU_TECLA;
			acc.tecla=(char)(tecla & 0x00ff);
			break;
	}

	opcion=m_accion(menu,&acc);

	if(opcion>=0) {
		pon_cursor(filcur,colcur);
		return(opcion);
	}
	else if(opcion==MENU_SALIDA) {
		pon_cursor(filcur,colcur);
		return(-1);
	}
}

}

/****************************************************************************
	M_COLOR: cambia paleta de colores para mens.
	  Entrada:      'clr_princ' color principal
			'clr_s1'      "    sombra 1
			'clr_s2'      "    sombra 2
			'clr_tecla'   "    tecla activacin
			'clr_sel'     "    opcin seleccionada
****************************************************************************/
void m_color(BYTE clr_princ, BYTE clr_s1, BYTE clr_s2, BYTE clr_tecla,
  BYTE clr_sel)
{

Menu_Clr[MCLR_PRINCIPAL]=clr_princ;
Menu_Clr[MCLR_S1]=clr_s1;
Menu_Clr[MCLR_S2]=clr_s2;
Menu_Clr[MCLR_TECLA]=clr_tecla;
Menu_Clr[MCLR_SELECC]=clr_sel;

}
