/*
  Derived from std.t 1.2, copyright (c) 1989, 1991 by Michael J. Roberts.
  All Rights Reserved.  Thanks Mike.

  Some code courtesy Stephen Granade (& Neil deMause) and Scott Starkey.

  carlstd.t   - Carl Klutzke's default adventure definitions
  Version 1.0
 
  This file provides some simple definitions for objects and functions
  that are required by TADS, but not defined in the file "adv.t".
  The definitions in std.t are suitable for use while a game is
  being written, but you will probably find that you will want to
  customize the definitions in this file for your game when the
  game is nearing completion.  This file is intended to help you
  get started more quickly by providing basic definitions for these
  functions and objects.
*/

/* Use C-style operators. */
#pragma C+

/* fix inconsistent capitalization in adv.t */
#define chairItem    chairitem
#define bedItem      beditem
#define fixedItem    fixeditem
#define foodItem     fooditem
#define buttonItem   buttonitem
#define nestedRoom   nestedroom
#define lightSource  lightsource

/*
 *   Pre-declare all functions, so the compiler knows they are functions.
 *   (This is only really necessary when a function will be referenced
 *   as a daemon or fuse before it is defined; however, it doesn't hurt
 *   anything to pre-declare all of them.)
 */
die: function;
init: function;
pardon: function;
darkTravel: function;
uberloc: function;

/*
 *   The die() function is called when the player dies.  It tells the
 *   player how well he has done (with his score), and asks if he'd
 *   like to start over (the alternative being quitting the game).

 	This isn't strictly not explicitly for ALMANAC, but so what.
 	I'll fix that later.
 */
die: function(cause)
{
    "\b*** AN END ***\b";
    scoreRank();
    "\bYou may restore a saved position, start over at the beginning, quit, undo
    the current command, or request a hint.\n";
    while ( 1 )
    {
        local resp;

	"\nPlease enter RESTORE, RESTART, QUIT, UNDO, or HINT: >";
        resp = upper(input());
        if ( resp == 'RESTORE' )
	{
	    resp = askfile( 'File to restore' );
	    if ( resp == nil ) "Restore failed. ";
	    else if ( restore( resp )) "Restore failed. ";
	    else
	    {
	        Me.location.lookAround(true);
	        scoreStatus( global.score, global.turnsofar );
		abort;
	    }
	}
        else if ( resp == 'RESTART' )
	{
	    scoreStatus( 0, 0 );
            restart();
	}
	else if ( resp == 'QUIT' )
        {
	    terminate();
            quit();
	    abort;
        }
	else if (resp == 'UNDO')
	{
	    if (undo())
	    {
		"(Undoing one command)\b";
		Me.location.lookAround(true);
	        scoreStatus(global.score, global.turnsofar);
		abort;
	    }
	    else
		"Sorry, no undo information is available. ";
	}
	else if (resp == 'HINT')
	{
		"\b";
		if (nil == cause) {
			hintVerb.action(Me);
		} else {
			cause.doHint(Me);
		}
		"\b";
	}
    }
}

/*
 *   The init() function is run at the very beginning of the game.
 *   It should display the introductory text for the game, start
 *   any needed daemons and fuses, and move the player's actor ("Me")
 *   to the initial room, which defaults here to "startroom".
 */
init: function
{
    version.intro;			// display the intro text
    version.sdesc;                // display the game's name and version number

    setdaemon( turncount, nil );               // start the turn counter daemon
    randomize();			// randomize the random number generator
    Me.location = startroom;                // move player to initial location
    startroom.lookAround( true );                    // show player where he is
    startroom.isseen = true;                  // note that we've seen the room
    scoreStatus(0, 0);                          // initialize the score display
}

/*
 *   The pardon() function is called any time the player enters a blank
 *   line.  The function generally just prints a message ("Speak up" or
 *   some such).  This default version just says "I beg your pardon?"
 */
pardon: function
{
    "I beg your pardon? ";
}

/*
 *   The numObj object is used to convey a number to the game whenever
 *   the player uses a number in his command.  For example, "turn dial
 *   to 621" results in an indirect object of numObj, with its "value"
 *   property set to 621.
 */
numObj: basicNumObj  // use default definition from adv.t
;

/*
 *   strObj works like numObj, but for strings.  So, a player command of
 *     type "hello" on the keyboard
 *   will result in a direct object of strObj, with its "value" property
 *   set to the string 'hello'.
 *
 *   Note that, because a string direct object is used in the save, restore,
 *   and script commands, this object must handle those commands.
 */
strObj: basicStrObj     // use default definition from adv.t
;

/*
 *   The "global" object is the dumping ground for any data items that
 *   don't fit very well into any other objects.  The properties of this
 *   object that are particularly important to the objects and functions
 *   are defined here; if you replace this object, but keep other parts
 *   of this file, be sure to include the properties defined here.
 *
 *   Note that awakeTime is set to zero; if you wish the player to start
 *   out tired, just move it up around the sleepTime value (which specifies
 *   the interval between sleeping).  The same goes for lastMealTime; move
 *   it up to around eatTime if you want the player to start out hungry.
 *   With both of these values, the player only starts getting warnings
 *   when the elapsed time (awakeTime, lastMealTime) reaches the interval
 *   (sleepTime, eatTime); the player isn't actually required to eat or
 *   sleep until several warnings have been issued.  Look at the eatDaemon
 *   and sleepDaemon functions for details of the timing.
 */
global: object
    turnsofar = 0                            // no turns have transpired so far
    score = 0                            // no points have been accumulated yet
    maxscore = 100                                    // maximum possible score
    verbose = nil                             // we are currently in TERSE mode
    awakeTime = 0               // time that has elapsed since the player slept
    sleepTime = 400     // interval between sleeping times (longest time awake)
    lastMealTime = 0              // time that has elapsed since the player ate
    eatTime = 200         // interval between meals (longest time without food)
    lamplist = []              // list of all known light providers in the game
    status = ""			// score ranking
;

/*
 *   "Me" is the player's actor.  Pick up the default definition, basicMe,
 *   from "adv.t".
 */
Me: basicMe
	smelldesc = "You smell acceptable by most standards."
	touchdesc = "This isn't that kind of story!"
	listendesc = "Not unless you say something interesting."
	bulk = 10
	weight = 10
;

/*
 *   darkTravel() is called whenever the player attempts to move from a dark
 *   location into another dark location.  By default, it just says "You
 *   stumble around in the dark," but it could certainly cast the player into
 *   the jaws of a grue, whatever that is...
 */
darkTravel: function
{
    "You stumble around in the dark, and don't get anywhere. ";
}

/*
 *   goToSleep - carries out the task of falling asleep.  We just display
 *   a message to this effect.
 */
goToSleep: function
{
    "***\bYou wake up some time later, feeling refreshed. ";
    global.awakeTime = 0;
}

/*
** uberloc returns the room an object is in, no matter how deeply nested.
** Returns the outermost room if inside a nested room.
** Based on code by SRG.
*/
uberloc: function(ob)
{
    local loc;

    loc = ob;
    while (loc.location) {
        if (isclass(loc, room)  && !isclass(loc, nestedroom)) return loc;
        loc = loc.location;
    }
    return loc;
}

modify readable
	doSynonym('Read') = 'Lookin'
;

knockOnVerb: deepverb
	verb = 'knock on' 'rap on' 'tap on' 'knock' 'rap' 'tap'
	sdesc = "knock on"
	prepDefault = onPrep
	doAction = 'KnockOn'
;

modify outVerb
	doAction = 'Exit'
	sdesc = "exit"
;

modify unfastenVerb
	verb = 'untie'
;

modify doorway
	verDoKnockOn(actor) = {}
	doKnockOn(actor) = {"\"Knock knock.\"  You hear no response. ";}
	verDoEnter(actor) = {}
	doEnter(actor) = {actor.travelTo( self.destination);}
	verDoExit(actor) = {}
	doExit(actor) = {actor.travelTo(self.destination);}
	verDoKick(actor) = {
		feet.verIoAttackWith(actor);
		self.verDoAttackWith(actor, feet);
	}
	doKick(actor) = {
		self.doAttackWith(actor, feet);
	}
;

// A null key for doors which cannot be unlocked.
nokey: keyItem
	location = nil
	sdesc = "missing key"
;

/*
** Used to bar and unbar doubleDoorways.
*/
barWithVerb: deepverb
	verb = 'bar'
	sdesc = "bar"
	ioAction(withPrep) = 'BarWith'
	prepDefault = withPrep
;
unbarVerb: deepverb
	verb = 'unbar'
	sdesc = "unbar"
	doAction = 'Unbar'
;

/*
** To make doors that can only be barred on one side, set "inside" to nil for the
** other side.  The key is actually the bar for the door.
*/
class doubleDoorway: lockableDoorway
    isqcontainer = true
    inside = true
    doSynonym('BarWith') = 'LockWith'
    doSynonym('Unbar') = 'Unlock'
    verDoOpen( actor ) =
    {
        if ( self.isopen ) "They're already open. ";
	else if ( self.islocked ) "They're barred. ";
    }
    doOpen( actor ) =
    {
        "Opened. ";
	self.isopen = true;
	if ( self.otherside ) self.otherside.isopen = true;
    }
    verDoClose( actor ) =
    {
        if ( !self.isopen ) "They're already closed. ";
    }
    doClose( actor ) =
    {
        "Closed. ";
	self.isopen = nil;
	if ( self.otherside ) self.otherside.isopen = nil;
    }
    verDoBarWith( actor, io ) =
    {
        if ( self.islocked ) "They're already barred! ";
	else if ( !self.islockable ) "They can't be barred. ";
	else if ( self.isopen ) "%You%'ll have to close them first. ";
	else if ( !self.inside ) "%You% can't bar them from this side. ";
    }
    doBarWith( actor, io ) = {
    	self.ioPutIn(actor, io);
    }
    verIoPutIn(actor) = {}
    ioPutIn(actor, dobj) = {
    	if (mykey != dobj) {
    		"%You% can't bar << thedesc >> with << dobj.thedesc >>.";
    	} else {
    		dobj.moveInto(self);
    		"%You% bar%s% << thedesc >> with << dobj.thedesc >>.";
    		islocked = true;
    		otherside.islocked = true;
    	}
    }
    verDoUnbar( actor ) =
    {
        if (!self.islocked) "They're not barred! ";
	else if ( !self.inside ) "%You% can't unbar them from this side. ";
    }
    doUnbar( actor ) = {
	    "%You% remove%s% the bar and set it aside. ";
	    self.islocked = nil;
	    if ( self.otherside ) self.otherside.islocked = nil;
	    self.mykey.moveInto(self.location);
    }
    ldesc = {
	if ( self.isopen ) "They're open. ";
	else
	{
	    if ( self.islocked ) "They're closed and barred. ";
	    else "They're closed. ";
	}
    }
;

/*
** A class for portable things you can both sit and lie on.  Defaults to something
** very bulky, but you can make it smaller.
*/
class cushionItem: nestedroom, surface
    bulk = 5
    weight = 5
    statusPrep = "on"
    outOfPrep = "off of"
    reachable = ([] + self) // list of all containers reachable from here;
                            //  normally, you can only reach carried items
                            //  from where you are, but this makes special allowances
    ischair = true
    isbed = true
    sitting = nil	// Keep track of whether player is sitting or lying

    isListed = {return (self != Me.location);}	// Don't list when player's on it.
    
    roomAction( actor, v, dobj, prep, io ) =
    {
        if ( dobj<>nil and v<>inspectVerb )
            checkReach( self, actor, v, dobj );
        if ( io<>nil and v<>askVerb and v<>tellVerb )
            checkReach( self, actor, v, io );
	pass roomAction;
    }

    // Assume the player will sit.
    enterRoom( actor ) = {
	self.sitting = true;
	inherited.enterRoom(actor);
    }
    noexit = {
        "%You're% not going anywhere until %you%
        get%s% off of <<thedesc>>. ";
        return( nil );
    }
    verDoBoard( actor ) = { self.verDoSiton( actor ); }
    doBoard( actor ) = { self.doSiton( actor ); }
    verDoSiton( actor ) = {
        if ( actor.location == self && self.sitting )
        {
            "%You're% already sitting on "; self.thedesc; "! ";
        }
        if (actor == self.location) {
        	"%You% will have to set it down first.";
        }
    }
    doSiton( actor ) = {
        "Okay, %you're% now sitting on "; self.thedesc; ".\b";
        actor.travelTo( self );
    }
    verDoLieon( actor ) = {
        if ( actor.location == self && !self.sitting )
        {
            "%You're% already lying on "; self.thedesc; "! ";
        }
        if (actor == self.location) {
        	"%You% will have to set it down first.";
        }
    }
    doLieon(actor) = {
        "Okay, %you're% now lying on <<self.thedesc>>.\b";
	actor.travelTo(self);
	self.sitting = nil;
    }
    verDoTake(actor) = {
    	if (self == actor.location) {
    		"%You% can't do that while %you're% on top of it!";
    	} else {
    		inherited.verDoTake(actor);
    	}
    }
;
    
/*
** Fixes how surfaces handle "look on [surface]"
*/
modify surface
	verDoLookin(actor) = {}
	doLookin(actor) = {
		inherited.ldesc;
	}
;

/*
** The following senses code is courtesy Neil deMause and Stephen Granade, with
** some modifications by myself.
*/

// New additions to "thing" to handle the new senses
modify thing
    verDoSmell(actor) = {}
    doSmell(actor) = {self.smelldesc;}
    smelldesc = "\^<<self.thedesc>> smells just like <<self.adesc>>. "
    verDoTouch(actor) = {}
    doTouch(actor) = {self.touchdesc;}
    touchdesc = "\^<<self.thedesc>> feels just like <<self.adesc>>. "
    verDoListenTo(actor) = {}
    doListenTo(actor) = {self.listendesc;}
    listendesc = "\^<<self.thedesc>> sounds just like <<self.adesc>>. "

// Fix adv.t bug: use statusPrep rather than an explicit "in"
    doUnboard( actor ) =
    {
        if ( self.fastenitem )
	{
	    "%You%'ll have to unfasten "; actor.location.fastenitem.thedesc;
	    " first. ";
	}
	else
	{
            "Okay, %you're% no longer << self.statusPrep >> "; self.thedesc; ". ";
            self.leaveRoom( actor );
	    actor.moveInto( self.location );
	}
    }
;

// Fix adv.t bug: if you wear something that's in a container, it
// stays in the container!
modify clothingItem
    doWear( actor ) =
    {
        "Okay, %you're% now wearing "; self.thedesc; ". ";
	self.moveInto(actor);
        self.isworn = true;
    }
;

modify room
    smelldesc = "%You% %do%n't smell anything unusual. "
    listendesc = "%You% %do%n't hear anything unusual. "
;

smellVerb: deepverb
    verb = 'smell' 'sniff'
    sdesc = "smell"
    action(actor) = {Me.location.smelldesc;} // global room smell
    doAction = 'Smell'
;

// Listen in a room.
listenVerb: deepverb
    verb = 'listen'
    sdesc = "listen"
    action(actor) = {Me.location.listendesc;}
;

// Listen to an object.
listentoVerb: deepverb
    verb = 'listen to' 'hear'
    sdesc = "listen to"
    doAction = 'ListenTo'
;

// Modification to touchVerb to add "feel" to it.  SRG
modify touchVerb
    verb = 'feel'
;

// Intangible, for smells and sounds and whatnot.  It can't be felt, looked
//  at, &c.
class intangible : fixeditem
    ldesc = "\^<<self.thedesc>> is not visible."
    verDoTouch(actor) = {"\^<<self.thedesc>> can't be touched.";}
    verDoLookbehind(actor) = "\^<<self.thedesc>> is not visible."
    verDoAttack(actor) = {"\^<<self.thedesc>> can't be attacked.";}
    verDoAttackWith(actor, iobj) = {"\^<<self.thedesc>> can't be attacked.";}
    verIoPutOn(actor) = {"%You% can't put anything on <<self.thedesc>>.";}
    verDoShowTo(actor, io) = {"\^<<self.thedesc>> can't be shown.";}
    verDoClean(actor) = {"\^<<self.thedesc>> can't be cleaned.";}
    isVisible(vantage) = {return (nil);}
;

// SRG: I have marked every verb which obviously requires the use of sight
//      with "sight=true".  This is useful if you want a room with a light
//      so bright that you cannot "look", "look at", &c.
modify inspectVerb
    sight=true;
modify lookBehindVerb
    sight=true;
modify lookInVerb
    sight=true;
modify lookThruVerb
    sight=true;
modify lookUnderVerb
    sight=true;
modify lookVerb
    sight=true;
modify readVerb
    sight=true;

// General verb modifications.

modify turnVerb
	verb = 'spin'
;

/*
** Add body parts and allow swatting and kicking.
*/

class bodypart: floatingItem, fixedItem
	adjective = 'my'
	isplural = nil
	ldesc = {
		if (isplural) {
			"They look like << adesc >>.";
		} else {
			"It looks like << adesc >>.";
		}
	}
	adesc = "your << self.sdesc >>"
	thedesc = "your << self.sdesc >>"
	location = {return Me.location;}
	verDoTake(actor) = {
		"There's no sensible way to do that.";
	}
	verDoMove(actor) = {}
	doMove(actor) = {
		"Okay, you shake << thedesc >> about.  Feel better now?";
	}
;

hands: bodypart
	adjective = 'bare'
	noun = 'hands' 'hand'
	sdesc = "hands"
	isplural = true
	verIoAttackWith(actor) = {}
	ioAttackWith(actor, dobj) = {
		dobj.doAttackWith(actor, self);
	}
	doMove(actor) = {
		"You wave at no one in particular.";
	}
;

swatVerb: deepverb
	verb = 'swat' 'slap' 'smack' 'swat at' 'punch'
	sdesc = "swat"
	doAction = 'Swat'
;

feet: bodypart
	noun = 'feet' 'foot'
	sdesc = "feet"
	isplural = true
	verIoAttackWith(actor) = {}
	ioAttackWith(actor, dobj) = {
		dobj.doAttackWith(actor, self);
	}
	doMove(actor) = {
		"You do a little shuffle.";
	}
;

kickVerb: deepverb
	verb = 'kick'
	sdesc = "kick"
	doAction = 'Kick'
;

modify Actor
	touchdesc = "\^<<self.thedesc>> shies away, avoiding %your% touch. "

	verDoSwat(actor) = {
		if (hands.location == actor.location) {
			hands.verIoAttackWith(actor);
			self.verDoAttackWith(actor, hands);
		} else {
			"%You% %do%n't have hands!";
		}
	}
	doSwat(actor) = {
		hands.ioAttackWith(actor, self);
	}
	verDoKick(actor) = {
		feet.verIoAttackWith(actor);
		self.verDoAttackWith(actor, feet);
	}
	doKick(actor) = {
		feet.ioAttackWith(actor, self);
	}
;
