////////////////////////////////////////////////////////////////////////
//  
//  ALT Library File: FloatingDestination 010123
//
//  Copyright (c) 2000, 2001 Kevin Forchione. All rights reserved.
//  Based on ADV.T (c) and STD.T (c) Michael Roberts.
//
//  This file is part of the ALT replacement library for ADV.T and 
//  STD.T and requires TADS 2.5.1 or later.
//
////////////////////////////////////////////////////////////////////////

#ifndef _FLOATING_DESTINATION_H_
#define _FLOATING_DESTINATION_H_

#include <floating.h>

#pragma C+

/*----------------------------------------------------------------------
 *  MIX-IN CLASS:   This class must always preced any non-mix-in classes
 *                  in a superclass list - this is required because a
 *                  mix-in can't override default methods in the root of
 *                  the hierarchy.
 *--------------------------------------------------------------------*/

/*
 *  FloatingDestination: Floating, Destination
 *
 *  To make a simple two-sided FloatingDestination you need only 
 *  code both room objects in the foundIn list of the
 *  FloatingDestination: 
 *
 *      foundIn = [ hallway, kitchen ]
 *  
 *  To make a one-sided FloatingDestination, code the foundIn list 
 *  with the room object in which the FloatingDestination is to be 
 *  found, and the leadsTo list with the room object to which the 
 *  FloatingDestination leads:
 *
 *      foundIn = hallway 
 *      leadsTo  = kitchen 
 *
 *  If you want a FloatingDestination that returns rooms different 
 *  from those it is found in, simply code both the foundIn list 
 *  and leadsTo list accordingly, each element of the foundIn list 
 *  must have a corresponding leadsTo element:
 *
 *      foundIn = [ hallway, kitchen ]
 *      leadsTo  = [ foyer, hallway ]
 *
 *  From the hallway the FloatingDestination leads to the foyer, 
 *  but from the kitchen it leads to the hallway.
 */
class FloatingDestination: Floating, Destination      
    findDest = {
        local f, loc, foundList, destList; 
        local actor = parserGetObj(PO_ACTOR);
        
        if (actor == nil) actor = parserGetMe();
            
        loc = self.location; 
        foundList = self.getMethReturn(&foundIn);        
        destList = self.getMethReturn(&leadsTo);
                        
        //
        //  door is linked to room directions, but no foundIn
        //  list exists. Simply return the noExit message.
        //
        if (length(foundList) == 0) {
            return self.noExit;
        }
          
        //
        //  We don't have a leadsTo list. Use the foundIn list,
        //  which must have 2 elements.
        //
        if (length(destList) == 0) {  
            if (length(foundList) != 2) {
                "\n[TADS-FloatingDestination-001] When omitting          
                the leadsTo list, the foundIn list should 
                contain 2 elements.\n";
                return nil;
            }
            if (loc == foundList[1])
                return foundList[2];
            else
                return foundList[1];
        }
        
        //
        //  We have a leadsTo list, each element should correspond to
        //  an element in the foundIn list.
        //
        if (length(foundList) != length(destList)) {
            "\n[TADS-FloatingDestination-002] <<self.theDesc>> 
            requires a corresponding leadsTo element for each 
            foundIn element.\n";
            return nil;
        }
        
        //
        //  Search for the actor location in the foundIn list.
        //
        f = find(foundList, loc);
        
        //
        //  Destination isn't in the actor location. return noExit.
        //
        if (f == nil) {
            return self.noExit;
        }
       
        //
        //  return the corresponding leadsTo element
        //
        return destList[f];
    }
; 

#pragma C-

#endif  /* _FLOATING_DESTINATION_H_ */
