////////////////////////////////////////////////////////////////////////
//  
//  ALT Library File: PreparseObj 010123
//
//  Copyright (c) 2000, 2001 Kevin Forchione. All rights reserved.
//  Based on ADV.T (c) and STD.T (c) Michael Roberts.
//
//  This file is part of the ALT replacement library for ADV.T and 
//  STD.T and requires TADS 2.5.1 or later.
//
////////////////////////////////////////////////////////////////////////

#ifndef _PREPARSE_OBJ_H_
#define _PREPARSE_OBJ_H_

#include <composite.h>
#include <gcmdstring.t>
#include <gcmdwordlist.t>

#pragma C+

/*
 *  PreparseObj: object
 *
 *  The preparse object is designed to allow for multiple occurrences
 *  of preparse(). Each instance of a preparse object should perform a
 *  unique function within your source. The ppoMethod() should return
 *  either true, nil, or a new command string, as required by
 *  preparse().
 */
class PreparseObj: object
	isPreparseObj = true
	ppoMethod(cmd) = {
		return true;
	}
	ppcoMethod(cmd) = {
	    return true;
	}
	/*
	 *  Removes leading and trailing spaces from a string.
	 */
    trim(val) = {
        val = self.postTrim(val);
        val = self.preTrim(val);
        return val;
    }
    /*
     *  Removes trailing spaces from a string.
     */
    postTrim(val) = {
        local i, char, len;
        
        len = length(val);
        for (i = len; i > 0; --i) {
            char = substr(val, i, 1);
            if (char == ' ') continue;
            break;
        }
        val = substr(val, 1, i);
        return val;
    }
    /*
     *  Remove leading spaces from a string.
     */
    preTrim(val) = {
        local i, char, len;
        
        len = length(val);
        for (i = 1; i <= len; ++i) {
            char = substr(val, i, 1);
            if (char == ' ') continue;
            break;
        }
        val = substr(val, i, len-i+1);
        return val;
    }
;

/*
 *	againPPO: preparseObj
 *
 *  This preparse object's preparseCmd() method either stores the
 *  command word list or replaces the command (if <<again>>) with
 *  the previously stored global command string.
 */
againPPO: PreparseObj
    ppcoMethod( cmd ) = {
        if (length(cmd) == 1 
        && (cmd[1] == 'g' || cmd[1] == 'again')) {
            self.correctForWaitCmd(cmd);
            parserReplaceCommand(gCmdString());
        } else {
            gCmdWordList(cmd);
        }
        return true;
    }
    /* 
     *  Because the <<wait>> command actually removes some of the 
     *  information of the command before passing it on to preparseCmd()
     *  we have to restore it to the original command first. This 
     *  requires a bit of string manipulating.
     */
    correctForWaitCmd(cmd) = {
        if (reSearch('.%<wait$', gCmdString())) {
            gCmdString(gCmdString() + ' ' + gameClock.timeString);
        } else if (reSearch('.%<wait%> %<for$', gCmdString())) {
            /*
            local str = replaceWith(gCmdString(), 'waitfor', 'wait
                for');
            gCmdString(str + ' ' + gameClock.timeString);
            */
            gCmdString(gCmdString() + ' ' + gameClock.timeString);
        } else if (reSearch('.%<wait%> %<until$', gCmdString())) {
            /*
            local str = replaceWith(gCmdString(), 'waituntil', 'wait
                until');
            gCmdString(str + ' ' + gameClock.timeString);
            */
            gCmdString(gCmdString() + ' ' + gameClock.timeString);
        }
    }
;

/*
 *	reparsePronounPPO: preparseObj
 *
 * 	This preparse object is required in order to allow for composite
 *	disambiguation in those cases where pronouns are used to refer to a 
 *	Composite class object. Because pronouns aren't parsed, but simply
 *	referenced for their corresponding objects we need to provide the
 *	original command words that a previous command used in order to 
 *	process through the deepverb disambigXobj() methods.
 */
reparsePronounPPO: PreparseObj
    ppcoMethod( cmd ) = {
    local i, j, objList, newCmdList = [];
    local reparseThisCmd = nil;
    
    for ( i = 1; i <= length( cmd ); ++i ) {
        switch( cmd[ i ] ) {
            case 'I':
                objList = parserGetObj( PO_IT );
                break;
            case 'M':
                objList = parserGetObj( PO_HIM );
                break;
            case 'R':
                objList = parserGetObj( PO_HER );
                break;
            case 'T':
                objList = parserGetObj( PO_THEM );
                break;
            default:
                objList = nil;
        }
        
        if ( objList ) {
            if ( datatype( objList ) == DTY_OBJECT )
                objList = [ objList ];
           	
          	/*
          	 *	Check to see if the object list returned for the given
          	 *	pronoun contains a Composite class object. Only
          	 *	Composite class objects need to be reparsed.
          	 */
            for ( j = 1; j <= length( objList ); ++j )
                if ( isclass( objList[ j ], Composite ) )
                    reparseThisCmd = true;
                
            if ( length( intersect( objList, global.dList[ 2 ] ) ) )
                newCmdList += global.dList[ 1 ];
            else if ( length( intersect( objList, 
            						global.iList[ 2 ] ) ) ) 
                newCmdList += global.iList[ 1 ];
            else newCmdList += cmd[ i ];
        }
        else newCmdList += cmd[ i ];
    }
    
    if ( reparseThisCmd )
        return newCmdList;
    else return true;	
    }
;


#pragma C-

#endif /* _PREPARSE_OBJ_H_ */
