////////////////////////////////////////////////////////////////////////
//  
//  ALT Library File: replaceWith() 010123
//
//  Copyright (c) 2000, 2001 Kevin Forchione. All rights reserved.
//  Based on ADV.T (c) and STD.T (c) Michael Roberts.
//
//  This file is part of the ALT replacement library for ADV.T and 
//  STD.T and requires TADS 2.5.1 or later.
//
////////////////////////////////////////////////////////////////////////

#ifndef __REPLACE_WITH_T_
#define __REPLACE_WITH_T_
 
#pragma C+

#define RW_REPLACE_ONCE     1
#define RW_MATCH_WORD       2
#define RW_MATCH_CASE       4
#define RW_RET_NIL          8

/*
 *  replaceWith(value, target, replacement, flags)
 *
 *  The function searches the value string, replacing the target string
 *  with the replacement string. 
 *
 *  Bit-flags can be passed to control the search. 
 *
 *      RW_REPLACE_ONCE replace only one occurrence of the target in the 
 *           value string.
 *
 *          The default for replaceWith() is to replace all occurrences 
 *          of the target in the value string.
 *
 *      RW_MATCH_WORD target must match whole words. For example:
 *
 *          target 'get in' will search for '%<get>% *%<in%>'
 *          which will match 'get    in the chair', but not 
 *          'get into the chair'
 *
 *      RW_MATCH_CASE target must match the case in value. 
 *
 *          The default for replaceWith() is case-insensitive. A target
 *          string of 'get into' will match on 'GET INTO THE CAR' as
 *          well as 'Get into the car' unless RW_MATCH_CASE is used. 
 *
 *      RW_RET_NIL function returns nil if no match for the target found
 *
 *          The default for replaceWith() returns the value unchanged.
 *          Using RW_RET_NIL will return the value only if replacement
 *          occurred; otherwise the function will return nil.
 */
replaceWith: function(value, target, replacement, ...) {
    local ret, newValue = '';
    local valueSave, targetSave, replacementSave;
    local flags = 0;
    
    if (argcount > 3) flags = getarg(4);
    
    if ((flags & RW_MATCH_WORD)!= 0) {
        local tmpTarget = '%<';
        tmpTarget += replaceWith(target, ' ', '%> *%<');
        tmpTarget += '%>';
        target = tmpTarget;
    }
    
    do {
        if ((flags & RW_MATCH_CASE) == 0) {
            valueSave = value;
            targetSave = target;
            replacementSave = replacement;
            value = lower(value);
            target = lower(target);
            replacement = lower(replacement);
        }
        
        ret = reSearch(target, value);
        
        if ((flags & RW_MATCH_CASE) == 0) {
            value = valueSave;
            target = targetSave;
            replacement = replacementSave;
        }
        if (ret) {
            local len, tmp = '';
            
            len = length(value) + 1;
            
            if (ret[1] - 1)
                tmp += substr(value, 1, ret[1]-1);
                
            tmp += replacement;
            newValue += tmp;
            
            if (len - (ret[1]+ret[2]))
                value = substr(value, ret[1]+ret[2], len - (ret[1]+ret[2]));
            else
                value = '';
                
            if ((flags & RW_REPLACE_ONCE) != 0) 
                break;
                
        } else if ((flags & RW_RET_NIL)!= 0 && newValue == '') 
                return nil;
                
    } while(ret != nil);
    
    newValue += value;
    return newValue;
}

#pragma C-

#endif  /* __REPLACE_WITH_T_ */
