#ifndef GOTO
#define GOTO
#pragma C-

gotoVersion: versionTag
  id = "Goto: v 0.9 (beta) September 1999.\n"
  author = 'David Haire'
  func = 'go to room or object implementation'
;

/*
 * GoTo: implementation of the 'go to' command that allows the player to 
 *   travel to a room or object. Only those rooms or objects that the 
 *   player already knows about are valid. This is checked using 
 *   room.isseen and object.isknownto. If it is an object, it must be 
 *   directly contained within a room (i.e. not inside a container, hidden, 
 *   or being carried by another actor.
 *
 * Note that in order to use 'go to' with rooms, you will need to define a 
 *   noun (and optionally an adjective) for each room.
 */

/*
 * findPath: function that looks for a path from the starting location to 
 *   the destination. The parameters are the 'actor' that is going to move 
 *   and the 'obj' that he/she is moving to (the destination). The 
 *   destination can be a room or an object. The function will return the 
 *   most direct path from the actor's current location to the destination 
 *   object, or nil if no path could be found. Only those rooms that the 
 *   player has already visited will be checked.
 */
findPath: function(actor, obj)
{
  local start, dest, path := [], verlen := length(global.verdirlist);
  local i, j, okdir;
  local rm, rm2, rms, paths, dirs;
  start := actor.location;
  if (firstsc(obj) = Room) dest := obj;
  else dest := obj.location;
  rms := [] + start;
  paths := [] + nil;
  dirs := [] + nil;
  j := 1;
  while (j <= length(rms)) {
    rm := rms[j];
    actor.movein(rm);
    for (i := 1; i <= verlen; i++) {
      local r, d := global.dirstringlist[i];
      if (not actor.movevalid(d)) continue;
      rm2 := actor.getdestination(d);
      if (rm2 = nil) continue;
      else if (not rm2.isseen) continue;
      else if (not rm2.islit) continue;
      else if (not rm2.cantravel) continue;
      else if (find(rms, rm2)) continue;
      else {
        rms += rm2;
        paths += rm;
        dirs += i;
      }
      if (rm2 = dest) break;
    }
    if (rm2 = dest) break;
    j++;
  }
  actor.movein(start);
  if (rm2 = dest) {
    path := [];
    rm := rm2;
    while (rm <> nil) {
      i := find(rms, rm);
      if (i = nil) return nil;
      j := dirs[i];
      path := [j] + path;
      rm := paths[i];
    }
    if (rms[i] = start) return path;
    else return nil;
  }
  else return nil;
}

/*
 * travelPath: function that moves the 'actor' along the 'path'. Each room 
 *   is described as it is entered, and each move consumes one turn. It is 
 *   assumed that the standard daemons have not been modified or cancelled, 
 *   and that the incturns() function is used to increment the turn 
 *   counter.
 */
travelPath: function(actor, path)
{
  local i := length(path), j, k, n := true;
  local rm;
  for (j := 1; j <= i; j++) {
    k := path[j];
    if (k = nil or k = 0) continue;
    if (n)
      n := nil;
    else
      "\b";
    rm := actor.getdestination(global.dirstringlist[k]);
    if (actor = parserGetMe()) {
      "\n(Going <<global.dirstringlist[k]>>)\n";
      actor.travelto(rm);
      "\n";
    }
    else {
      local seen := parserGetMe().cansee(actor.location, 
              actor.locationtype, true);
      if (seen) actor.leavemessage(global.dirstringlist[k]);
      Outhide(true);
      actor.travelto(rm);
      Outhide(nil);
      seen := parserGetMe().cansee(actor.location, actor.locationtype, true);
      if (seen) actor.entermessage(global.dirstringlist[9 - k]);
    }
    if (j < i and actor = parserGetMe()) {
      rundaemons;
    }
  }
}

/*
 * Modify Thing to accept the 'go to' command. The property 'cantravel' 
 *   dictates (for rooms) whether the actor can travel through the room. It 
 *   is included on Thing to allow for custom rooms. See 'modify Room' 
 *   (below) for more information.
 *
 * verDoGoto must remain empty otherwise the 'go to' command will fail with 
 *   'You can't see any xxx here.'
 */
modify Thing
  cantravel = nil
  verDoGoto(actor) = { }
  doGoto(actor) = {
    local pt;
    global.justTesting := true;
    pt := findPath(actor, self);
    global.justTesting := nil;
    if (actor.cansee(self, nil, true)) {
      "\^<<actor.subjthedesc>> <<actor.is>> already here! ";
      exit;
    }
    else if (pt = nil) {
      if (isclass(self, Room))
        "\^<<actor.subjthedesc>> can't remember how to get there. ";
      else
        "\^<<actor.subjthedesc>> can't remember where <<self.isplural ? "they are" : "that is">>. ";
      exit;
    }
    else travelPath(actor, pt);
  }
  cantravelto(obj, loctype, silent) = {
    if (obj.cantravel) return true;
    else {
      if (not silent) "<<self.subjprodesc>> can't travel to <<obj.objthedesc(self)>>.";
      return nil;
    }
  }
;

/*
 * Modify Room to allow travel during 'go to'. The 'cantravel' property is 
 *   checked within findPath to see whether the room is valid to be 
 *   traversed. This enabled the implementation of a series of rooms that 
 *   cannot be traversed within 'go to', such as a maze.
 */
modify Room
  cantravel = true
;

/*
 * Modify Item to allow travel during 'go to'.
 */
modify Item
  cantravel = true
;

/*
 * Modify Fixture to allow travel during 'go to'.
 */
modify Fixture
  cantravel = true
;

/*
 * The gotoVerb itself. The verb checks that the direct object referred to 
 *   is a valid destination, and that the player has seen it and knows 
 *   about it.
 */
gotoVerb: Verb
  verb = 'go to'
  sdesc = "go to"
  desc(obj) = {
    if (obj.isplural) "go to";
    else "goes to";
  }
  doAction = 'Goto'
  longexplanation = nil
  requires = [[&cantravelto]]
;

#endif
