#ifndef LOOK
#define LOOK
#pragma C-

/*
 * Look: This module mimics the existing "look" methods provided in TADS but
 *   allows another variation. At present, the default method will print ...
 *
 *  > LOOK
 *	Play Pen.
 *		You are currently standing in the play pen.
 *		There is a dinosaur, an armchair and a pound of butter here.
 *
 * In order to add to the "fiction" part of "interactive-fiction", we would 
 *   prefer to have
 *
 *	> LOOK
 *	Play Pen.
 *		You are currently standing in the play pen.
 *		Over to one side is a small plastic dinosaur which has been
 *		gnawed on by the child who seems to be absent.
 *		There is a large armchair which looks quite comfortable and
 *		yet the bloodstains fail to make it look welcoming.
 *		On the ground is a pound of butter which seems to be melting.
 *
 * This module allows objects to have an extra method called 'hdesc'. The code 
 *   in look.t will automatically detect this method and will use it instead of 
 *   just constructing the list of objects.
 *
 * However, in some circumstances, an object will decide that it doesn't want to 
 *   be listed specially.  For example, the first time you see this room, ...
 *
 *	> LOOK
 *	The Stone Room
 *		....
 *		Lying on the ground is the most magnificent sword you have ever seen.
 *
 * However, once you have taken the sword, played with it a while, carried it 
 *   somewhere else, etc, the description should become
 *
 *	> LOOK
 *	Play Pen
 *		....
 *		There is a sword here.
 *
 * In order to provide this flexibility, the method 'hasHdesc' is used. By 
 *   default, this will return true if the method 'hdesc' exists and the item in 
 *   question has not been manipulated in any way ...
 *
 *	sword: Item
 *		hdesc = "Lying on the ground <blah, blah>"
 *    hasHdesc = { return (defined(self, &hdesc) and not self.istouched); }
 *	;
 *
 * so that the sword will maintain its long description until its istouched 
 *   property is set true.
 *
 * The istouched property is manipulated by the postAction function, and will be 
 *   set true for any object that is referred to in a succesfully executed 
 *   command. So 'take sword' or 'examine sword' will set istouched true, but 
 *   'eat sword' will not (unless, of course, the sword is edible ...)
 */

/*
 * This code was based on an implementation for 'normal' TADS written by Jeff 
 *   Laing, so I have credited him here.
 */
lookVersion: versionTag
	id = "Look: v 1.9 May 1994\n"
  author = 'Jeff Laing'
  func = 'extended item descriptions'
;

/*
 * Modify listcontentsX to display an item's hdesc in room descriptions.
 */
replace listcontentsX: function(cont, actor, silent, leadingspace, ird, loctype, listfixtures, listactors, cansense, here)
{
  local i, j, k, len, l, clen;
  local see := [], hear := [], smell := [], touch := [];
  local ltlist;
  local o;
  local container, lt;
  local unlisted := [];
  local tolist := [], tlen;
  local indislist := [], indiscount := [], indislen, indistot;
  local olist := [];
  local listedhere := [], tot := 0;
  local Silent := silent;
  local indisfound := nil;

  /*
   * List things the player sees.
   *
   * Always see what the player can see so we get a valid list of
   *   things that are listedhere. We need this to decide later if we
   *   should say "something is ticking" or "the bomb is ticking".
   */
  if (not (cansense = nil or cansense = &cansee))
    silent := true;

  /*
   * If the actor can't see this container, explain why and exit.
   */
  caps();
  if (not actor.cansee(cont, nil, silent)) return;

  /*
   * Construct the list of location types to look for. If loctype is
   *   non-nil, verify that it works. Otherwise, try each locationtype.
   */
  if (loctype = nil)
    ltlist := actor.canseecontents(cont, true, nil);
  else
    ltlist := [loctype];

  /*
   * Obtain a list of the items that the actor can potentially see.
   */
  if (ltlist = nil)
    see := [];
  else
    see := cont.seecont(actor, ltlist);

  /*
   * Read through the list of items that the actor can potentially see,
   *   and check if the item should be listed.
   */
  for (i := 1; i <= length(see); i++) {
    indisfound := nil;
    l := see[i];
    container := l[1];
    lt := l[2];
    clen := l[3];

    /*
     * If this container doesn't want its contents listed in room
     *   descriptions, don't list.
     */
    if (ird and not container.(global.loccont[lt])(nil)) {
      unlisted += container;
      continue;
    }

    /*
     * If this is a room description we don't want to list the
     *   contents of any container whose container doesn't want
     *   things listed. So we need to keep a list of containers
     *   that don't want things listed. Since our list of objects
     *   is guaranteed to go down the containment hierarchy as we
     *   progress from 1 to n, we'll always see containers before
     *   we see their contents, so this works.
     *
     * The exception to this is that if the container is this Thing
     *   (cont), then it doesn't have to be listable for its
     *   contents to be listed.
     */
    if (ird) {
      if (find(unlisted, container) <> nil)
        continue;
      if (find(unlisted, container.location) <> nil) {
        unlisted += container;
        continue;
      }

    }

    /*
     * If this is a room description, remove any items that don't
     *   want to be listed in room descriptions.
     */
    if (ird) for (j := 4; j <= clen; j++) {
      if (not l[j].islistable(actor))
        l[j] := nil;
    }

    /*
     * Remove any items that aren't noticeable.
     */
    for (j := 4; j <= clen; j++) {
      if (l[j] <> nil)
        if (not l[j].isnoticeable(actor))
          l[j] := nil;
    }

    /*
     * Now list all the fixtures and actors, if we're listing them.
     *   If we're not listing them, just delete them.
     */
    tolist := [];
    tlen := 0;
    for (j := 4; j <= clen; j++) {
      o := l[j];
      if (o = nil)
        continue;
      if (not o.isfixture and not o.isactor)
        continue;
      l[j] := nil;
      if (not listfixtures and not listactors)
        continue;
      if (o.isactor) {
        if (not listactors)
          continue;
        if (o = actor)
          continue;
      }
      tolist += o;
      tlen++;
    }
    if (not silent and listfixtures) {
      for (j := 1; j <= tlen; j++) {
        o := tolist[j];
        if (leadingspace) {
          leadingspace := nil;
          P(); I();
        }
        if (not o.isactor) {
          "<<o.heredesc>> ";
          tot++;
          listedhere += o;
        }
      }
    }
    if (not silent and listactors) {
      for (j := 1; j <= tlen; j++) {
        o := tolist[j];
        if (leadingspace) {
          leadingspace := nil;
          P(); I();
        }
        if (o.isactor) {
          "<<o.actordesc>> ";
          tot++;
          listedhere += o;
        }
      }
    }

    /*
     * If this is a room description, list all items that are present *in that 
     *   location* that have a hdesc, and delete them from the list.
     */
    if (ird) {
      for (j := 4; j <= clen; j++) {
        o := l[j];
        if (o = nil)
          continue;
        if (find(here, o.location) and o.hasHdesc) {
          if (leadingspace) {
            leadingspace := nil;
            P(); I();
          }
          "<<o.hdesc>> ";
          l[j] := nil;
        }
      }
    }

    /*
     * Now list everything else. We separate out
     *   indistinguishables. We move plurals to the front to make 
     *   the listing more readable.
     */
    tolist := [];
    tlen := 0;
    indislist := [];
    indiscount := [];
    indislen := 0;
    for (j := 4; j <= clen; j++) {
      o := l[j];
      if (o = nil)
        continue;
      if (o.isequivalent) {
        indislist += o;
        indiscount += 1;
        indislen++;
      }
      else {
        if (o.isplural)
          tolist := [o] + tolist;
        else
          tolist += o;
        tlen++;
      }
    }

    /*
     * Now merge indistinguishable objects.
     */
    for (j := 1; j <= indislen; j++) {
      if (indislist[j] = nil)
        continue;
      for (k := j + 1; k <= indislen; k++) {
        if (indislist[k] = nil)
          continue;
        if (indistinguishable(indislist[j], indislist[k])) {
          indiscount[j]++;
          indislist[k] := nil;
        }
      }
    }

    /*
     * Put indistinguishable objects back in tolist.
     */
    indistot := 0;
    olist := [];
    for (j := 1; j <= indislen; j++) {
      local o := indislist[j];
      if (o = nil)
        continue;
      olist += o;
      tlen++;
      indistot++;
      indiscount[indistot] := indiscount[j];
    }
    if (indistot > 0) tolist := olist + tolist;

    /*
     * Now list the items.
     */
    if (tlen = 0)
      continue;
    tot += tlen;
    listedhere += tolist;
    if (silent)
      continue;
    if (leadingspace) {
      leadingspace := nil;
      P(); I();
    }
    "\^";
    if (container.isactor)
      "<<container.subjthedesc>> <<container.has>>\ ";
    for (j := 1; j <= tlen; j++) {
      o := tolist[j];
      if (o = nil)
        continue;
      if (j = 1) "";
      else if (j = tlen) {
        if (tlen > 2) ",";
        " and ";
      }
      else
        ", ";
      if (j <= indistot and indiscount[j] > 1) {
        "<<saynum(indiscount[j])>> \v<<o.listpluraldesc>>";
        indisfound := true;
      }
      else
        o.listdesc;
    }
    if (container.isactor) ". ";
    else {
      if (tlen > 1 or o.isplural or indisfound)
        " are ";
      else
        " is ";
      if (find(here, container) and global.loctypes[lt] = 'in')
        " here. ";
      else
        " <<global.loctypes[lt]>> <<container.objthedesc(nil)>>. ";
    }
  }
  global.listed += listedhere;

  /*
   * If we only did the "see" code to get a list of things that the
   *   player can see, don't count these items in the total.
   */
  silent := Silent;
  if (not (cansense = nil or cansense = &canseee))
    tot := 0;

  /*
   * Now list things the player hears.
   */
  if (cansense = nil or cansense = &canhear) {

    local h, o, contained, tolist := [], tlen := 0;

    for (i := length(global.listablesoundlist); i > 0; i--) {
      o := global.listablesoundlist[i];
      contained := nil;
      if (not o.islistablesound(actor))
        continue;
      while (o.location <> nil) {
        if (ird and not o.location.(global.locconthear[o.locationtypenum])(nil))
          break;
        if (o.location = cont and (loctype = nil or o.locationtype = loctype)) {
          contained := true;
          break;
        }
        o := o.location;
      }
      if (contained) {
        o := global.listablesoundlist[i];
        if (actor.canhear(o, nil, true)) {
          tolist += o;
          tlen++;
        }
      }
    }

    if (tlen = 0) goto EndHear;
    tot += tlen;
    if (silent) goto EndHear;
    if (leadingspace) {
      leadingspace := nil;
      P(); I();
    }
    for (j := 1; j <= tlen; j++) {
      o := tolist[j];

      /*
       * Print either the name of this thing (if we've
       *   listed it) or "something ..."
       */
      if (o.alwaysname(actor) and o.isknownto(actor) or actor.cansee(o, nil, true) or find(listedhere, o) <> nil)
          "\^<<o.subjthedesc>> ";
      else {
        if (o.isplural)
          "Some things ";
        else
          "Something ";
        if (cont.alwaysname(actor) and cont.isknownto(actor) or find(listedhere, cont) <> nil) {
          if (cont.isactor)
             "<<cont.subjdesc>> <<cont.has>> ";
          else
             "<<o.locationtype>> <<cont.objthedesc(actor)>> ";
        }
      }
      "<<o.listlistendesc>>. ";
    }
EndHear:;
  }

  /*
   * Now list things the player smells.
   */
  if (cansense = nil or cansense = &cansmell) {

    local o, contained, tolist := [], tlen := 0;

    for (i := length(global.listablesmelllist); i > 0; i--) {
      o := global.listablesmelllist[i];
      contained := nil;
      if (not o.islistablesmell(actor))
        continue;
      while (o.location <> nil) {
        if (ird and not o.location.(global.loccontsmell[o.locationtypenum])(nil))
          break;
        if (o.location = cont and (loctype = nil or o.locationtype = loctype)) {
          contained := true;
          break;
        }
        o := o.location;
      }
      if (contained) {
        o := global.listablesmelllist[i];
        if (actor.cansmell(o, nil, true)) {
          tolist += o;
          tlen++;
        }
      }
    }

    if (tlen = 0) goto EndSmell;
    tot += tlen;
    if (silent) goto EndSmell;
    if (leadingspace) {
      leadingspace := nil;
      P(); I();
    }
    for (j := 1; j <= tlen; j++) {
      o := tolist[j];
          
      /*
       * Print either the name of this thing (if we've
       *   listed it) or "something ..."
       */
      if (o.alwaysname(actor) and o.isknownto(actor) or actor.cansee(o, nil, true) or find(listedhere, o) <> nil)
          "\^<<o.subjthedesc>> ";
      else {
        if (o.isplural)
          "Some things ";
        else
          "Something ";
        if (cont.alwaysname(actor) and cont.isknownto(actor) or find(listedhere, cont) <> nil) {
          if (cont.isactor)
             "<<cont.subjdesc>> <<cont.has>> ";
          else
             "<<o.locationtype>> <<cont.objthedesc(actor)>> ";
        }
      }
      "<<o.listsmelldesc>>. ";
    }
EndSmell:;
  }

  /*
   * Now list things the player feels.
   */
  if (cansense = nil or cansense = &cantouch) {

    local o, contained, tolist := [], tlen := 0;

    for (i := length(global.listabletouchlist); i > 0; i--) {
      o := global.listabletouchlist[i];
      contained := nil;
      if (not o.islistabletouch(actor))
        continue;
      while (o.location <> nil) {
        if (ird and not o.location.(global.locconttouch[o.locationtypenum])(nil))
          break;
        if (o.location = cont and (loctype = nil or o.locationtype = loctype)) {
          contained := true;
          break;
        }
        o := o.location;
      }
      if (contained) {
        o := global.listabletouchlist[i];
        if (actor.cantouch(o, nil, true)) {
          tolist += o;
          tlen++;
        }
      }
    }

    if (tlen = 0) goto EndTouch;
    tot += tlen;
    if (silent) goto EndTouch;
    if (leadingspace) {
      leadingspace := nil;
      P(); I();
    }
    for (j := 1; j <= tlen; j++) {
      o := tolist[j];
        
      /*
       * Print either the name of this thing (if we've
       *   listed it) or "something ..."
       */
      if (o.alwaysname(actor) and o.isknownto(actor) or actor.cansee(o, nil, true) or find(listedhere, o) <> nil)
          "\^<<o.subjthedesc>> ";
      else {
        if (o.isplural)
          "Some things ";
        else
          "Something ";
        if (cont.alwaysname(actor) and cont.isknownto(actor) or find(listedhere, cont) <> nil) {
          if (cont.isactor)
             "<<cont.subjdesc>> <<cont.has>> ";
          else
             "<<o.locationtype>> <<cont.objthedesc(actor)>> ";
        }
      }
      "<<o.listtouchdesc>>. ";
    }
EndTouch:;
  }

  return tot;
}

/*
 * Modify Thing to include the 'istouched' property and the hasHdesc check. Note 
 *   that hdesc itself is not defaulted, as its presence is checked for.
 */
modify Thing
  istouched = nil
  hasHdesc = { return (defined(self, &hdesc) and not self.istouched); }
;

/*
 * Replace the standard implementation of postAction. This is executed for each 
 *   direct object in a command. If the command was successful, the 'istouched' 
 *   property is set true.
 */
replace postAction: function(a, v, d, p, i, s)
{
  if (s = EC_SUCCESS) {
    if (d) {
      d.istouched := true;
    }
  }
}

#endif
