#charset "us-ascii"

#include <adv3.h>
#include <en_us.h>


versionInfo: GameID
    name = 'Find the Ring (ConSpace Test Game)'
    byline = 'by Eric Eve'
    htmlByline = 'by <a href="mailto:eric.eve@hmc.ox.ac.uk">
                  ERIC EVE</a>'
    version = '2.0'
    authorEmail = 'ERIC EVE <eric.eve@hmc.ox.ac.uk>'
    desc = 'Find the ring - Test game for ConSpace and ECContainer.'
    htmlDesc = 'Find the ring - Test game for ConSpace and ECContainer.'

    showCredit()
    {
        /* show our credits */
        "The ConSpace extension is the joint work of Eric Eve and Steve
         Breslin. Steve worked mainly on the EnterableComplexContainer
         part, while Eric focused mainly on ConSpace. Steve has also
         done most of the work on upgrading ConSpace to version 2.\b
         The TADS 3 language and library is by Michael J. Roberts. ";

        /* 
         *   The game credits are displayed first, but the library will
         *   display additional credits for library modules.  It's a good
         *   idea to show a blank line after the game credits to separate
         *   them visually from the (usually one-liner) library credits
         *   that follow.  
         */
        "\b";
    }
    showAbout()
    {
        "Although this game does have a minimal plot, and it is possible
        to win it (pretty easily), the main point of this game 
        is to test and demonstrate the ConSpace and ECContainer library extensions.\b
        You might want to start by trying to move things around in the
        hall. Alternatively, (or additionally) you can try ordering Bob
        to do so. Also, you'll want to experiment with going near, behind, under,
        in and on things.<.p>";
    }
;


me: Person
    /* the initial location */
    location = southHall
    desc = "By George old Georgie! You look ready to cope with any situation! "
    noun = 'george'
;



gameMain: GameMainDef
    /* the initial player character is 'me' */
    initialPlayerChar = me

    /* 
     *   Show our introductory message.  This is displayed just before the
     *   game starts.  Most games will want to show a prologue here,
     *   setting up the situation for the player, and show the title of the
     *   game.  
     */
    showIntro()
    {
        "Welcome to the ConSpace Test Game!\b
        First time players may wish to type <a href='ABOUT'>ABOUT</a><.p>
        <i>Find the Ring</i>\n
        <<versionInfo.byline>>\n
        A game of suspense, intrigue, and testing!\b
        You are visiting your friends Bob and Susan in their home,
        when you are about to hear some alarming news....\b";
    }

    /* 
     *   Show the "goodbye" message.  This is displayed on our way out,
     *   after the user quits the game.  You don't have to display anything
     *   here, but many games display something here to acknowledge that
     *   the player is ending the session.  
     */
    showGoodbye()
    {
        "<.p>Thanks for playing!\b";
    }
    
    setAboutBox()
    {
      "<ABOUTBOX>
       <CENTER><H1>FIND THE RING</H1></CENTER>\b
       <CENTER><H2>Test Game for ConSpace</H2>\b
       Version 2.00\b
       by Eric Eve with Steve Breslin</CENTER></ABOUTBOX>";       
    }
;

  // THE HALL

 /* 
  *  The hall consists of two rooms, southHall and northHall, joined
  *  in a single connectionGroup. We set these up to allow movement
  *  between the two halves of the hall with the compass movement commands
  *  NORTH and SOUTH.
  */
  
southHall: Room 'Hall (south)' 'the south end of the hall' 'southern end of the hall'
    "Here at the southern end of the hall a cabinet runs along one wall
     and a door leads out to the west. At the far end of the hall a
     clock hangs on the wall, and a plain white door leads into the kitchen,
     next to a grand staircase leading up.
     The wooden front door is directly to the south. <<frontDoor.isOpen &&
     gPlayerChar.isIn(frontDoor.nearLoc) ? 'Through the open front door
     you can see out into the drive.' : ''>>"
     north = northHall
     south = frontDoor
     out asExit(south)
     
     /* Note that we can define the up direction here even though the hallStairs
      * are located in northHall. An UP command issued in southHall will still
      * work, first taking the player character into northHall
      */
     up = hallStairs
     
     /* The connection group is the SpaceConnector joining the southHall to the
      * northHall
      */
     connectionGroup = [hallGroup]
     
     
;

  /*
   *  If we made frontDoor a SpaceConnectorDoor, the drive and the southern
   *  end of the hall would appear to be a single space when the front door
   *  was opened. Since this is unrealistic, we'll use a simple Door here.
   *  If a SpaceConnectorDoor was really needed here the best approach would
   *  be to create a third hall room to put it in, a third room representing
   *  the small space immediately inside the door. For an example of this
   *  approach see the kitchen below.
   */


+ frontDoor : Door 'stout wood wooden front drive door' 'front door'
    "It's a stout door made of wood. "
    sightSize = large // the door is easily visible from northHall
;

+ Platform 'portable platform' 'portable platform'
;


cabinet : ComplexContainer, Heavy 'cabinet' 'cabinet' @southHall
  "The cabinet runs along one wall. It looks like there's space on top of it,
   underneath it, and inside it. "
     
  subContainer: ComplexComponent, OpenableContainer { }
  subSurface: ComplexComponent, Surface { }
  subUnderside: ComplexComponent, Underside { }
;

+ box : Container 'blue cardboard box' 'blue cardboard box' @southHall
  subLocation = &subSurface
;

+Thing 'blue ball' 'blue ball' "It's a blue ball. ";


northHall : Room 'Hall (north)' 'the north end of the hall' 'northern end of the hall'
   "A clock is fixed high up on the wall at this end of the hall,
    while a cabinet can be seen running along the wall at the far end,
    near the front door. Immediately to the north a white door
    leads into the kitchen; next to the kitchen door a grand wooden staircase
    leads up to the floor above. "
   south = southHall
   north = whiteDoor
   up = hallStairs
   connectionGroup = [hallGroup]   
;

 /*
  *  We experiment with making the next door a SpaceConnectorDoor. This allows the player
  *  character to talk to Bob while the player character is just inside the kitchen
  *  and Bob is in the northern end of the hall. Since the other side of door is
  *  in a kitchen room that represents the area of the kitchen immediately next to
  *  the door this works reasonably well without the need for defining a NestedRoom
  *  to represent this space as with the front door.
  */


+ whiteDoor: SpaceConnectorDoor 'white kitchen door' 'white kitchen door'    
;

+ hallStairs : StairwayUp 'grand (hall) wooden staircase/stairs' 'staircase'
  "For some reason the staircase is between the kitchen door and the clock. "
  sightSize = large
;

redBall : Thing 'red ball*balls' 'red ball' @northHall  
;

greenBall : Thing 'green ball*balls' 'green ball' @northHall
;


clock : Fixture 'clock' 'clock' @northHall
  desc()
  {
    local t = getTime(GetTimeDateAndTime);
    "According to the clock the time is <<t[6]>>:<<t[7]>>:<<t[8]>>. ";
  }
  sightSize = large
  
  /* Defining underLoc allows the clock to be the object of commands like
   * STAND UNDER THE CLOCK.
   * The result of such a command will be to move the actor into the location
   * defined here - in this case the clock's NestedSpace
   */
  underLoc: NestedSpaceUnder { }
;

/* This is all we need to do to define the connectionGroup object; every room that
 * defines hallGroup in its connectionGroup property will automatically be added
 * to hallGroup's locationList.
 */

hallGroup : SpaceConnector
;



drive : OutdoorRoom 'Drive'
  "This large gravel drive stands in front of a grand house just to the
   north. To the east is a large garden, in which stands a large
   apple tree, while the way out to the main road lies south. "
   north = frontDoorOutside
//   in asExit(north)
   east = gardenByTree
   south : FakeConnector { "You don't really want to go down to the
    main road, you'd much rather wander around the house and garden. " }   
;


+ frontDoorOutside: Door ->frontDoor 'front hall door' 'front door'
;

+ house:  Enterable -> frontDoorOutside 'grand tudor house' 'house'
  "It's a grand tudor house with mullioned windows. "  
;

++ Component 'mullioned windows' 'windows'
  isPlural = true 
;



// THE KITCHEN

/* 
 * The Kitchen will be made out of four Rooms. In this case motion via compass
 * direction commands (NORTH etc.) will not move the player around within the
 * room - such commands, where legal, will take the player character out of the
 * kitchen altogether into another location. Rather than define this on each
 * individual room in the kitchenGroup it is convenient to define them on a
 * class we'll use for all the rooms making up the kitchen.
 */

class KitchenRoom : Room
  south = hallDoor
  east = backDoor
  connectionGroup = [kitchenGroup]
;

kitchenGroup : SpaceConnector
;

kitchen : KitchenRoom 'Kitchen' 'the kitchen, by the hall door' 'hall door'
  "The entrance to this large square kitchen is through the white
   door immediately to the south. Over to the east the back door
   leads out into the garden. At the far end of the kitchen there's
   a ventillation grille high up on the wall. A large wooden table
   stands near the west wall, opposite the stove. "
   actorInPrep = 'in the kitchen, near'
   actorIntoName = 'into the kitchen'
   connectionGroup = [kitchenGroup]
;

+ hallDoor: Door ->whiteDoor 'white hall door' 'white hall door'
;


kitchenGrille : KitchenRoom 'Kitchen' 'the kitchen, by the back door' 'back door'
  "There's a ventilation grille high up on the wall here, while
   the back door out into the garden is immediately to the east. "  
  actorInPrep = 'just inside'
  actorIntoPrep() {
    if(!gActor)
      return actorInPrep;
    if(gActor.locationBefore.getOutermostRoom.ofKind(KitchenRoom))
      return 'over to';
    else
      return 'through';
  }
;

 /* To reach the grille an actor must be standing on a chair directly
  * under it.
  */

+ grille: OutOfReach, OpenableContainer, Fixture 'metal ventilation grille' 
   'ventilation grille'
   "It's a metal grille high up on the wall. "   
   
   /* We use a NestedSpace (rather than a Room) to represent the space
    * directly beneath the grille, and make NEAR GRILLE a synonym of
    * UNDER GRILLE.
    */
    
   underLoc : NestedSpace { objInPrep = 'beneath' }
   nearLoc = (underLoc)
   canObjReachContents(obj) 
   { 
     return obj.isIn(kitchenChair) && obj.posture == standing
       && kitchenChair.isIn(underLoc);      
   }
   
   /* We apply the nearObj precondition here so that commands like PUT CHAIR
    * UNDER GRILLE will automatically move the actor (and the chair) into the
    * grille's location (from elsewhere in the kitchen) to allow the command
    * to proceed, without enforcing touchability (since the grille is out
    * of reach.
    */
   
   iobjFor(PutUnder) { preCond = [nearObj] }
   iobjFor(PutNear) { preCond = [nearObj] }
   
   sightSize = large
   cannotReachFromOutsideMsg(dest) 
   {
     return 'It\'s too high up in the wall to reach. ';
   }
;

++ rubyRing: Thing 'susan\'s sue\'s ruby ring' 'ruby ring'
   "It's a beautiful silver band with an elegantly mounted ruby
    surrounded by tiny sparking diamonds. "
    owner = susan
    isKnown = true
    actionDobjTake()
    {
      inherited;
      achievement.awardPointsOnce();
    }
    achievement : Achievement { +5 "finding the ring" }
    returned = nil
;

+ backDoor: SpaceConnectorDoor 'back garden door' 'back door'
;

kitchenBehindTable: KitchenRoom 'Kitchen (behind table)' 'the kitchen, behind the table' 'kitchen table'
  "From behind the kitchen table you can see a white door leading
   out to the south, the back door out to the east, the stove
   opposite, and a grille high up on the north wall. "
   actorInPrep = 'behind'
;

+ kitchenChair : Chair 'plain wooden kitchen chair' 'wooden chair'
  "It's nothing fancy, just a plain old-fashioned straight-backed wooden kitchen chair. "
;

/* We locate the kitchen table in both the space behind it and the space in
 * front of it, since it can clearly be reached directly from both.
 */

kitchenTable : MultiLoc, ECContainer, Heavy
  'heavy wooden table' 'wooden table'
   
   /* There's not much headroom under a table, so if you want to get under it,
    * you must lie down.
    */
    
   /* Note that since the kitchenTable is a multiLoc, it has no
    * defined location, so we need to redefine the exitDestination
    * of its subXXX components to be something other than
    * the default (lexicalParent.location)
    */
    
   subUnderside: ESubUnderside 
   { 
     allowedPostures = [lying]
     defaultPosture = lying
     obviousPostures = [lying]
     exitDestination = nearLoc
   } 
     
    
   subSurface: ESubSurface { exitDestination = nearLoc }
   nearLoc = kitchenNearStove
   behindLoc = kitchenBehindTable
   frontLoc = (nearLoc)
   locationList = [nearLoc, behindLoc]
;

+ knife : Thing 'sharp stainless steel (kitchen) knife/blade' 'knife'
 "It's a sharp kitchen knife with a wooden handle and a stainless
  steel blade. "
 iobjFor(CutWith)  
 { 
    verify() {} 
    preCond = [objHeld]
 }
 iobjFor(MoveWith) { verify()  { } }
 subLocation = &subSurface
;

kitchenNearStove: KitchenRoom 'Kitchen' 'the kitchen, by the stove' 'stove'
  "This part of the kitchen lies between the stove (immediately to the east)
   and a large table (just over to the west). To the south a door leads out
   into the hall, while another door, at the opposite end of the kitchen,
  leads east out to the garden. High on the north wall is a ventillation
  grille. "
  actorInPrep = 'near' 
   
;

+ kitchenStove: ComplexContainer, Fixture 'large old black stove' 'stove'
  "It's a large, black old stove. "
  subSurface: ComplexComponent, Surface { }
  subContainer: ComplexComponent, OpenableContainer { } 
;

++ loaf: Food 'freash crusty loaf/bread' 'loaf of bread'
  "It's a fresh, crusty loaf of bread. "
  subLocation = &subContainer
  dobjFor(Eat)
  {
    action()
    {
      "It turns out to be delicious, so you tear off one chunk after
       another, and keep eating until it's all gone. Right at the
       end, you find a small piece of paper inside. ";
       smallPaper.moveInto(gActor);
    }
  }
  dobjFor(CutWith) 
  { 
    verify() { } 
    action() 
    {
      "You cut the loaf in two, revealing a small piece of paper within. ";
      smallPaper.moveInto(location);
      piece1.moveInto(location);
      piece2.moveInto(location);
      moveInto(nil);
    } 
  }
;


smallPaper: Readable 'small white piece/paper*notes' 'small piece of paper'
  "It's a small white piece of paper, with something written on it. "
  readDesc = "It reads: <q>I ran out of my normal baking powder
   on the last batch, so I finished this one off with the powder from the jar
   marked <q>Arsenic</q>. Hope that's okay.</q>";
;

class HalfLoaf : Food 'fresh crusty half/piece/bread/loaf*loaves*halves*pieces' 'piece of bread'
  "It's half a fresh, crusty loaf of bread. "
  pluralName = 'pieces of bread'
  isEquivalent = true
;

piece1: HalfLoaf;
piece2: HalfLoaf;


/*
 * We define a GardenRoom for all the rooms that will make up the garden.
 * We shall allow movement around within the garden using the compass
 * direction commands (NORTH etc.), so we don't define the compass
 * properties on the class. Since, however, a garden is a large outdoor
 * space we don't allow conversation between actors who are not in the same
 * room. If the player character is in another part of the garden when he
 * tries to strike up a conversation with Susan (who's on the bench), he
 * will be moved to Susan's location to allow the conversation to proceed.
 */

class GardenRoom: OutdoorRoom
  connectionGroup = [gardenGroup]

  /* Enforce the condition that actors must be in the same room to converse */
  allowRemoteConversation = nil
  
  /* But allow this condition to be fulfilled by an implicit action if the 
   * actors are within this connectionGroup.
   */
  autoApproachForConversation = true
;

gardenGroup: SpaceConnector;

gardenByBackDoor : GardenRoom 'Garden (by back door)' 'the piece of garden by the back door'
  'piece of the garden by the back door'
  "This is the section of the garden by the back door of the house.
   The main lawn stretches south towards a large apple tree, 
   while over to the west is a white
   paling fence with a small gate in it. "
  actorInPrep = 'by'
  actorIntoPrep() {
    if(!gActor)
      return actorInPrep;
    if(gActor.locationBefore.getOutermostRoom.ofKind(KitchenRoom))
      return 'through';
    else
      return 'over to';
  }
  west = backDoorOutside
  south = gardenLawn
  east = gardenByFence
  in asExit(west)
;

+ backDoorOutside : Door ->backDoor 'back kitchen door' 'back door'
;

gardenLawn : GardenRoom 'Garden (lawn)' 'the lawn'
  "This lawn forms the centre of the garden. Just to the south
   of here grows a tall apple tree, while over to the east a small
    gate leads through a white fence. "
  name = 'lawn'
  actorInPrep = 'on'
  north = gardenByBackDoor
  south = gardenByTree
  east = gardenByFence
;

+ bench : Chair, Heavy 'large old wooden bench' 'wooden bench' 
  
  specialDesc = "An old wooden bench sits in the middle of the lawn. "
  bulkCapacity = 30
;

herbGarden: MultiLoc, Fixture 'herb garden' 'herb garden' "It's a herb garden."
  locationList = [gardenLawn, gardenByTree]
  isListed = true
;

gardenByTree: GardenRoom 'Garden (by the tree)' 'the tree end of the garden' 
  "A tall apple tree grows here. To the north stretches the lawn,
   while to the west is the front drive. In the distance to the
   east runs a white paling fence, through which is a small gate. "  
  
  north = gardenLawn
  west = drive
  east = gardenByFence
  connectionGroup = [gardenGroup]
;

+ appleTree: Fixture 'tall spreading fine old apple tree' 'tall apple tree' // add 'fine old' to vocabWords
  "It's a tall, spreading apple tree, just coming into blossom. "
  remoteDesc(pov)
  {
    if(pov.isIn(bench))
      "The apple tree is behind you, so without turning round and
       getting a crick in your neck, you can't see it too well. ";
    else if(pov.isIn(gardenLawn))
      "The apple tree stands tall just behind the bench to the south. ";
    else if(pov.isIn(gardenByBackDoor))
      "It looks quite tall, but it's hard to make out much detail
       since it's some way away, on the far side of the lawn. ";
    else if(pov.isIn(drive))
      "The fine old apple tree stands in the garden to the east. ";
    else
      "It looks like a fine old apple tree. ";    
  }
  underLoc = (nearLoc)
  
  /* 
   * The following allows APPROACH APPLE TREE or STAND UNDER TREE to work
   * from the drive, even though the drive is not part of the 
   * garden connectionGroup.
   *
   * Note that it does NOT cause an implicit approach for actions that require
   * touching the apple tree.
   */
  
  extraApproachRooms = [drive] 
  
;

++ Component 'apple blossom' 'apple blossom'
  "It's white. "
;

+++ SimpleOdor
  desc = "The scent of apple blossom wafts on the air. "
;

+ football : Thing 'old leather ball/football*balls' 'football'
  "It's an old leather football, still reasonably well inflated. "
  
  /* We define this to allow KICK BALL */
  dobjFor(Attack)
  {
    verify()
    {
      if(!getOutermostRoom.ofKind(OutdoorRoom))
        illogicalNow('You really don\'t want to kick that ball around
         indoors -- you might break something. ');
    }
    action()
    {
      local loc = rand(gardenLawn, gardenByTree, gardenByBackDoor,
        gardenByFence, drive);
      moveIntoForTravel(loc);
      "You give the ball a hearty kick, sending it flying; it lands
       <<loc.actorInPrep>> <<loc.theName>>. ";  
        
    }
  }
;

 /*
  *  Use a standard DistanceConnector between these locations because
  *  we are using the tree to illustrate extraApproachRooms. If this was
  *  a real game it would probably be better to use a SpaceConnector here
  *  to present a more consistent interface to the player. If you try
  *  changing the DistanceConnector to a SpaceConnector as an experiment
  *  you'll probably find you also want to set its lookAroundOnMove property
  *  to true to avoid the sensation of moving around in a thick fog.
  */

DistanceConnector [gardenByTree, drive];


gardenByFence: GardenRoom 'Garden (by gate)' 'the gate' 'gate'
  "A white paling fence runs along the east side of the garden here, at
   the edge of the lawn. The main part of the lawn is directly to the
   west running south towards a tall apple tree. To the northwest
   the garden comes to the end outside the back door of the house. "
  west = gardenLawn
  northwest = gardenByBackDoor
  southwest = gardenByTree
  east = gardenGate
  actorInPrep = 'by'
;

+ gardenGate : Door 'small white gate' 'small white gate'
;

+ Fixture 'white wooden paling fence' 'white fence'
  "The white-painted wooden paling fence runs along the east side
   of the garden. Roughly in the middle of it is a small white gate. "
  sightSize = large
;

gardenPath: OutdoorRoom 'Long Path' 'the long path'
  "This is the start of a long, long path running east. Just to the
  west is a small white gate leading into a big garden. "
  west = pathGate
  east : TravelMessage { -> riverBank "You wander off down the path, and after a while
   arrive at the bank of a wide river. " }
;

+ pathGate: Door ->gardenGate 'small white gate' 'small white gate'
;

riverBank : OutdoorRoom 'Riverbank' 'the riverbank'
  "The path from the west comes to the end at the bank of a wide river, on the
   far side of which stretches a large open meadow. Another path leads off
   through a forest to the south."
  west = gardenPath  
  east : NoTravelMessage{ "You can't walk on water, and you're not up to
   swimming today. " }
  north : FakeConnector { "You walk northwards along the bank of the river
   for quite a while, but finding nothing of interest, you eventually turn
   back. " }
  south = forestPath1
;

+ Fixture 'wide river' 'river'
  "The river is too wide and deep to cross here. "
;

+ Distant 'large open meadow' 'meadow'
  "The meadow on the opposite bank stretches almost as far as you can see. "
  
  tooDistantMsg = '{You/he} can\'t get to the meadow from here; the river\'s
   in the way. '
;

+ Enterable -> forestPath1 'dense forest/trees/path' 'forest'
  "The dense forest stretches off to the south. A path runs into the forest. "
;


farBank : OutdoorRoom 'Meadow' 'the meadow'
  
;

+ pileOfGold : Thing 'pile/gold' 'pile of gold'
  "It's a large pile -- must be worth a fortune. Too bad
   there seems to be no way you can reach it. "
  sightSize = large  
;

/* We use a standard DistanceConnector to make each side of the river visible
 * to the other, without making them mutually accessible.
 *
 * This should show that the standard library behaviour of a DistanceConnector
 * is as normal when it's not used with a connectionGroup: the player character
 * is not taken into the meadow if, for example, the player types TAKE GOLD.
 */

DistanceConnector [riverBank, farBank]
;


/*--------------------------------------------------------------------
 * FOREST - We expand the game at this point to experiment with using
 * SpaceConnectors in the kind of forest path scenario envisaged in the
 * docs.
 *
 * The paths through the forest are implemented as a number of locations
 * connected by chains of SpaceConnectors, so that each pair of neighbouring 
 * locations is in the same connectionGroup. The idea is to blur the boundaries
 * between neighbouring locations so that the space feels more continuous, and
 * less like a series of discreet rooms. 
 *
 * Note, however, that we set lookAroundOnMove = true on these SpaceConnectors,
 * since players will want to be made aware that the player character has
 * moved to a different part of the forest on each move.
 * --------------------------------------------------------------------
 */

forestPath1: OutdoorRoom 'Narrow Forest Path' 'the narrow forest path'
  "Trees press in to east and west, and the thick foliage obscures the
   view to the north, although you know the path leads out of the forest
   that way. The path continues southwards towards a huge oak tree standing
   in a large clearing. "
   north = riverBank
   south = clearing
   connectionGroup = [forestGroup1]
;

 /* This should strictly be a MultiInstance, but a MultiLoc is harmless here */

MultiLoc, Decoration 'trees/foliage' 'trees'
  "Trees heavy with foliage are all around you. "
  isPlural = true
  locationList = [forestPath1, clearing, forestPath2]
;

forestGroup1: SpaceConnector lookAroundOnMove = true ;

clearing: OutdoorRoom 'Clearing' 
  "A tall oak tree stands at the centre of this clearing. A narrow
   path runs off to the north, while a broader one straggles off to
   the southeast. A third, windy, path disappears off to the west. "
   north = forestPath1
   southeast = forestPath2
   west = forestPath3
   connectionGroup = [forestGroup1, forestGroup2, forestGroup4]
;

+ Fixture 'tall huge oak tree/oak' 'oak tree'
  "The oak tree stands proud in the middle of the clearing, looking like
   it's been growing there since the time of the Stuarts. "
   cannotClimbMsg = 'You aren\'t agile enough. '
   sightSize = large
;

+ Underside 'small brown round acorn' 'acorn'
  "It's small, round and brown. "
  sightSize = small
;

++ Hidden, Readable 'tiny yellow note/writing*notes' 'yellow note'
  "It's a tiny yellow note with some writing on it. "
  readDesc = "It reads: <q>I didn't hide the ring here either!</q>"
  sightSize = small
;


forestGroup2: SpaceConnector lookAroundOnMove = true;
  
forestPath2: OutdoorRoom 'Broad Forest Path' 'the broad forest path'
  "Trees are packed closely on either side of this broad forest path which runs
   northwest towards a tall oak tree in a clearing and southeast towards
   the river. "
   northwest = clearing
   southeast = forestPathEnd
   connectionGroup = [forestGroup2, forestGroup3]
;

+ Thing 'long wooden gnarled stick' 'long wooden stick'
   "It's just an ordinary long wooden stick, slightly gnarled. "
   initSpecialDesc = "A long wooden stick lies by the side of the path. "
   remoteInitSpecialDesc(actor)
   {
     if(actor.isIn(clearing))
       "Some sort of stick lies by the side of the broad path
        running off to the southeast. ";
   }
   iobjFor(MoveWith) { verify()  { } }
;

forestGroup3: SpaceConnector lookAroundOnMove = true;

forestPathEnd: OutdoorRoom 'End of Path - By River' 'the end of the path, by the river'
  "The broad path from the northwest comes to an end here on the bank of the river. "
  northwest = forestPath2
  east : NoTravelMessage{ "You can't walk on water, and you don't feel like
   swimming today. " }
  north: NoTravelMessage { "The forest blocks your path. " }
  south: FakeConnector { "You walk along the riverbank for some distance, but finding
     nothing of interest you eventually turn back. " }
  connectionGroup = [forestGroup3]
;

+ Fixture 'broad fast flowing fast-flowing river/water/patches' 'river'
  "The river is broad and fast-flowing here; there is no way you could cross it. "
  remoteDesc(pov)
  {
    "All you can make out from here are some patches of water through the trees. ";
  }  
  sightSize = large
;

forestGroup4: SpaceConnector lookAroundOnMove = true;

forestPath3: OutdoorRoom 'Windy Forest Path' 'the windy forest path'
  "Here the forest path winds through the trees a little confusingly,
   but although it's a bit twisty the path basically runs east
   in one direction and northwest in the other. Through the foliage
   to the east you can just make out a tall oak tree standing
   in a clearing. "
   east = clearing
   northwest = roadside
   connectionGroup = [forestGroup4]
;

+ Fixture, Underside 'thick dark green bush' 'thick bush'
  "The bush is thick and green, a darker green than most of the surrounding
   foliage. <<envelope.discovered() ? '' : 'A closer look suggests
   that something might have fallen under it. '>>"
  distantDesc = "From here all you can make out is a dense mass of something
   dark green. "
  sightSize = large
  initSpecialDesc = "A thick green bush grows at the side of the path. "
  distantInitSpecialDesc = "To the side of the path leading west you can just
   make out a thick green bush. "
;

++ envelope: Hidden, OpenableContainer 'plain pale blue envelope' 'pale blue envelope'
   "It's just a pale, plain blue envelope. "
   achievement : Achievement { +5 "finding the envelope" }
   discover() { inherited; achievement.awardPointsOnce(); }
   dobjFor(Take)
   {
     check()
     {
        if(!moved)
          failCheck('The envelope is too far under the bush; you can\'t quite reach it. ');
     }
   }   
   dobjFor(MoveWith)
   {
     verify()
     {
        if(moved)
          illogicalAlready('There\'s no need to attempt that now. ');
     }
     action()
     {
        moved = true;
        "{You/he} manage{s} to move the envelope forward with {the iobj/him}. ";
     }
   }
;

+++ letter: Readable 'pale blue letter/paper' 'letter'
  "The letter is written on pale blue paper. "
  readDesc = "The letter reads, <q>Dear Bob, bad news I'm afraid. I don't know how
   to break this gently, so I'll put it to you straight: I've fallen madly in love
   with George and I'm going to leave you. George doesn't know this yet, and I'm
   not sure whether he even suspects, but I think he's really <i>fantastic</i>, and
   I'm pretty sure he's more than half in love with me already. Anyway, please don't
   blame George, he's done nothing wrong -- indeed, he's been a perfect gentleman,
   which may be part of why I care for him so much. Plus, he's just amazing with those
   guess-the-verb puzzles.\b
   I know I'm probably putting
   this very badly and just making things worse -- but there's no nice way to
   say this, is there? Anyway, I still think well of you, Bob, and hope we can
   always be friends, but now I know that George is the only man for me. I hope
   you can forgive me for hurting you.\b
   Love, Susan.</q>"
   returned = nil
   achievement : Achievement { +5 "returning the letter" }
   dobjFor(Read) { preCond = [objVisible, objHeld] }
   
;

roadside: OutdoorRoom 'Roadside Meadow' 'the roadside meadow'
  "To east and south this small scrap of meadow is bounded by thick forest, into
   which a path winds off to the southeast. To west and north the meadow is
   enclosed by a sharp bend in the road. "
   southeast = forestPath3
   north: NoTravelMessage { "It's too dangerous to walk onto the road from here:
     it's a blind bend with nowhere for pedestrians to walk; you could easily
     be hit by a car racing round that bend. " }
   northwest asExit(north)
   west asExit(north)   
;

+ Enterable ->forestPath3 'thick windy forest/path' 'thick forest'
  "The thick forest lies to south and east; a windy path penetrates
  it to the southeast. "
;

+ Fixture 'road road/bend' 'road'
  "The road bends sharply round, enclosing both the north and west sides of the
   small meadow. "
  cannotEnterMsg = 'It\'s too dangerous to step into the road at that bend. ' 
;
//--------------------------------------------------------------------
// Upstairs 
//--------------------------------------------------------------------

landing : Room 'Landing'
  "Most of this long landing lies to the south. Immediately to the north
   is a white-painted door, while a flight of stairs leads down. "
   down = landingStairs
   south = landingCorner
   north = landingDoor
   connectionGroup = [landingGroup]
   roomParts = static inherited - defaultSouthWall
;

+ landingStairs: StairwayDown ->hallStairs 'flight/stairs/staircase'
  'flight of stairs'
  "The flight of stairs at the northern end of the landing leads
   down to the hall below. "
   sightSize = large
;

+ landingDoor : Door 'white white-painted door' 'white door'
  "It looks like it has been repainted quite recently. "
  sightSize = large
;

/* 
 * Define the bedroom like the Kitchen - travel within is not by compass
 * directions, but by going near the various objects within the bedroom
 *
 * We'll use the bedroom to illustrate some more ECContainers.
 */


class Bedroom : Room
  out asExit(south)
  south = bedroomDoor
  connectionGroup = bedroomGroup
;

bedroomGroup : SpaceConnector
;


bedroom : Bedroom 'Susan\'s Bedroom'
  "This bedroom is so large you wonder what Susan can possibly do here.
   A large bed stands next to one wall near the middle of the room, 
   while a huge wardrobe occupies the large part of another wall at the far end. "
   name = 'door end of the bedroom'
;

+ bedroomDoor : Door ->landingDoor 'door' 'door'
;

bedroomMiddle : Bedroom 'Susan\'s Bedroom'
  "The large bed takes up most of the space here in the middle of this huge
   bedroom. At one end of the room is the door leading out, while at the
   other end a huge wardrobe stands against the wall. "
   name = 'middle of the bedroom'
;

+ bed : ECContainer, Heavy 'large bed' 'bed'
  "It's very big. "
  subSurface : ESubSurface { }
  subUnderside : ESubUnderside 
  { 
     allowedPostures = [lying]
     defaultPosture = lying
     obviousPostures = [lying]
  }  
  subRear : ESubRear { }
;

++ OpenableContainer, Hidden 'small green box' 'small green box'
   bulk = 1
   bulkCapacity = 1
   subLocation = &subUnderside
;

+++ Thing 'small tissue piece/paper' 'small piece of tissue paper'
  "It's all crinkled up, as if it's been used to wrap something small. "
;

bedroom2 : Bedroom 'Susan\'s Bedroom'
  "A huge wardrobe against the wall dominates this end of the bedroom.
   A large bedroom occupies the middle of the room, and the door out
   is at the far end. "
   name = 'wardrobe end of the bedroom'
;


+ wardrobe: ECContainer, Heavy 'huge white wardrobe' 'wardrobe'
  "The huge white wardrobe takes up most of the space against
   the wall at this end of the room. "
  frontLoc = (nearLoc)
  remoteDesc(pov)
  {
    if(pov.isIn(bedroom))
     "The huge white wardrobe takes up most of the space against
      the wall at the far end end of the room. ";
    else
     "The huge white wardrobe takes up most of the space against
      the wall at the end of the room. ";
  }
  subSurface : ComplexComponent, Surface { }
  subContainer : Openable, ESubContainer { objInPrep = 'inside' }
  subUnderside : ComplexComponent, Underside { }
;

++ Wearable 'smart brown winter coat' 'coat'
  "It's a smart brown winter coat; you feel sure Susan would look very good in it,
    except that it isn't winter. "
  subLocation = &subContainer
;

++ Hidden 'small brass 1p coin/piece' 'small brass coin'
  "It's a 1p piece, dated 2004. "
  subLocation = &subContainer
;

++ Thing 'wire coathanger/hanger' 'wire coathanger'
  "It's just an ordinary plain wire coathanger. "
  iobjFor(MoveWith) { verify()  { } }
;

 /* We add the landing corner to illustrate a location belonging to two
  * different connectionGroups. It is probably a good idea to define 
  * lookAroundOnMove = true on SpaceConnectors used in this way, otherwise
  * player navigation is likely to become very confusing.
  *
  * This is similar to the forest path example above, but illustrates a
  * slightly different topography. 
  */
  
landingGroup: SpaceConnector lookAroundOnMove = true;
passageGroup: SpaceConnector lookAroundOnMove = true;

landingCorner: Room 'Corner'
  "At this point the long landing leading from the white door by the stairs to the 
  north takes a sharp turning towards a large red door to the east. "
  north = landing
  east = passage
  connectionGroup = [landingGroup, passageGroup]
  roomParts = static inherited - defaultNorthWall - defaultEastWall
;  

+ CustomFixture 'large gilt framed frame gilt-framed mirror' 'gilt-framed mirror'
  "It's a large gilt-framed mirror in which you can see your reflection. "
  sightSize = large
  initSpecialDesc = "A large mirror hangs on the south wall. "
  remoteInitSpecialDesc(actor)
  {   
       "A large mirror hangs on the south wall at the corner of the landing. ";   
  }
  initNominalRoomPartLocation = defaultSouthWall
  remoteDesc(pov)
  {
    if(pov.isIn(landing))
      "It appears to be a large gilt-framed mirror; from here you can just
       make out your reflection in it. ";
    else
      "From here all you can see is the edge of the frame, since you're
       looking at the mirror edge-on. ";
  }
  
  cannotTakeMsg = 'The mirror is firmly fixed to the wall. '
;



passage : Room 'Passage'
  "The passage from the west comes to the end outside a
   large red door to the east. "
  west = landingCorner
  east = redDoor
  connectionGroup = [passageGroup]
  roomParts = static inherited - defaultWestWall
;

+ redDoor : Door 'large red door' 'large red door'
  sightSize = large
;

 /* We'll divide the Ballroom in two with a concertina partition */


ballRoomWest : Room 'Ballroom (west)' 'west end of the Ballroom'
  "A large red door leads out of the Ballroom to the west. <<partitionDoor.isOpen
   ? 'The Ballroom continues to the east, beyond the open partition' :
   'Immediately to the east, a closed partition cuts the Ballroom in two'>>. "
  west = redDoorInside
  out asExit(west)
  east = partitionDoor
  roomParts = static inherited - defaultEastWall
;

+ redDoorInside : Door -> redDoor 'large red door' 'large red door'

  sightSize = large
;

  /*
   *  This illustrates a third possible way of using a SpaceConnectorDoor. When the partition
   *  is open the two halves of the ballroom are effectively joined into a single room. Although
   *  this type of situation may be rare in IF, the SpaceConnectorDoor caters for it perfectly,
   *  and it illustrates what is probably the simplest use of SpaceConnectorDoor.
   */

+ partitionDoor : PushOpen, SpaceConnectorDoor 'sliding concertina beige door/partition' 'partition'
  desc = (partitionDoorEast.desc)
  sightSize = large
  lookAroundOnTraverse = nil // in this special case we probably don't want the lookAround()
;

ballRoomEast: Room 'Ballroom (east)' 'East end of the Ballroom'

  west = partitionDoorEast
;

+ partitionDoorEast: PushOpen, Door ->partitionDoor  'sliding concertina beige door/partition' 'partition'
  "It's a beige concertina partition, of the sort that can be pushed and pulled open and closed. "
  sightSize = large
;


+ piano: OpenableContainer, Heavy 'large black grand piano' 'piano'
  "It's a large, black grand piano. "
  sightSize = large
  initSpecialDesc = "A large black grand piano stands in the northeastern
   corner of the Ballroom. "
  dobjFor(Play)
  {
    verify() { }
    action() 
    {
       "You thump out a stirring rendition of <q><<rand('Chopsticks', 'By the Beautiful Blue Danube',
        'The Maple Leaf Rag', 'Rule Britannia!', 'Stars and Stripes Forever', 'The Military Polka',
        'Colonel Bogey', 'Liberty Bell', 'The Emperor Waltz')>></q>.";
    }
  }
;

++ Readable 'small blue note/writing*notes' 'small blue note'
  "It's a small blue note with some writing on it. "
  readDesc = "It reads, <q>I didn't hide the ring in here!</q>" 

;

class PushOpen: object
  dobjFor(Push) asDobjFor(Move)
  dobjFor(Pull) asDobjFor(Move)
  dobjFor(Move)
  {
     remap()
     {
        if(isOpen)
          return [CloseAction, self];
        else
          return [OpenAction, self];
     }
  }

;


/* 
 *  This avoids a potentially ugly response when a RoomPart is viewed from
 *  a remote location. Note that this may become standard in the adv3 library
 *  subsequent to version 3.0.9
 */

modify RoomPart
  sightSize = large
;


//-----------------------------------------------------------------------
// NPCs - First Bob
//-----------------------------------------------------------------------


bob: Person 'bob/man' 'Bob' @northHall
  "He's looking exceptionally agreeable today. "
  isProperName = true
  isHim = true
  sightSize = large
  isKnown = true
  initSpecialDesc() {
    "Bob is standing in the northern end of the hall.
    He smiles as you approach. ";
    initSpecialDesc = nil;
  }
;

+ bobTalking : InConversationState
  obeyCommand(issuingActor, action) 
  {   
    handleConversation(issuingActor, action, commandConvType);
    return !action.ofKind(TopicActionBase); 
  }
;

++ ConversationReadyState
   obeyCommand(issuingActor, action) 
   {   
     handleConversation(issuingActor, action, commandConvType);
     return !action.ofKind(TopicActionBase); 
   }
   isInitState = true
;

+++ HelloTopic
  "<q>Hello there!</q> Bob greets you. "
;

+++ ByeTopic
  "<q>Cheerio, then,</q> says Bob. "
;


++ AskTopic @susan
 "<q>Do you know where I can find Susan?</q> you inquire.\b
  <q>I believe she's out in the garden,</q> he tells you. "
;

++ AskTellTopic, StopEventList @rubyRing
  [
    '<q>What happened to Susan\'s ring?</q> you ask.\b
     <q>Some wretched burglar tried to steal it!</q> he tells you,
     <q>I nearly caught the blighter in the act -- but, well, he
        got away. We phoned the police straight away, and luckily they
        caught him a few minutes later, over on Lincoln Street, but he
        didn\'t have the ring on him -- but it\'s definitely gone from
        Susan\'s room.... We reckon he must have hidden it
        somewhere.</q>',
      
    '<q>Is the missing ring very valuable?</q> you ask.\b
     <q>Fairly - it\'s an antique ruby ring,</q> Bob tells you, <q>But so far
      as Susan\'s concerned it\'s more sentimental value; it was her mother\'s
      before her, and her grandmother\'s before that.</q>',
      
    '<q>So Susan\'s really keen to get her ring back,</q> you surmise.\b
     <q>Well, she\'ll certainly be pretty pleased with whoever finds it
     for her.</q> he agrees. '
  ]
;

+++ AltTopic
   "<q>I think I may have found Susan's ring,</q> you announce.\b
    <q>You'd better give it to her then,</q> he advises."
    isActive = rubyRing.seen
;

++ GiveShowTopic @rubyRing
   "You show Bob the ruby ring.\b
   <q>That certainly looks like Susan's ring,</q> Bob avers, <q>You'd
    better give it to her.</q> "
;


+ DefaultAskTopic
  "<q>What do you think about <<gTopic.getTopicText.toLower>>?</q> you ask.\b
   <q>Anything you say,</q> he agrees with a nod and a smile. "
;

+ DefaultTellTopic
  "<q>You know what I think about <<gTopic.getTopicText.toLower>>,</q> you say.\b
   <q>Naturally,</q> he agrees, <q>And of course, I agree absolutely.</q> "
;

+ DefaultGiveShowTopic
  "You offer Bob {the dobj/him}.\b
  <q>Very nice,</q> he beams, <q>so I'm so happy for you to have {it dobj/him}.</q> "
;

+ CommandTopic @TopicActionBase
  "It would make more sense for you to do it yourself. "
  isConversational = nil

;

+ DefaultCommandTopic
  handleTopic(fromActor, action)
  {
    inherited(fromActor, action);
    if(action.ofKind(TAction))
       action.dobjCur_ = action.dobjList_[1].obj_;
    if(action.ofKind(TIAction))
       action.iobjCur_ = action.iobjList_[1].obj_;


    "<q>Bob, would you <<youToMe(action.getInfPhrase)>>, please?</q> you ask.\b
     <q>Certainly -- anything you say.</q> he agrees.<.p>";
  }
;

+ ConvAgendaItem
  isReady { return inherited && getActor.canTalkTo(gPlayerChar); }
  initiallyActive = true
  invokeItem()
  {
    isDone = true;
    getActor.initiateConversation(bobTalking, 'rubyNode');
  }
;

+ AgendaItem
  isReady = (getActor.canSee(susan))
  initiallyActive = true
  invokeItem()
  {
     "<q>Ah,</q> says Bob, <q>There's Susan. Well, of course she's an absolute
      poppet, and I'm really fond of her and all that, but -- well, to tell you
      the truth I don't quite seem to be flavour of the month with her right
      now, if you take your meaning, so I think I'll make myself scarce.</q>\b
      So saying, Bob hurries off and disappears back inside the house.";
      getActor.moveIntoForTravel(northHall);
  }
;

+ rubyNode : ConvNode 'rubyNode'
  npcGreetingMsg = "<q>Have you heard about Susan's ring?</q> Bob asks,
    noticably agitated. <q>She's terribly upset about it!</q>"
;

/* Service Routine for DefaultCommandTopic */

function youToMe(str)
{
  if(str.ofKind(String))
  {       
   str = str.findReplace(' you ', ' me ', ReplaceAll);
    if(str.endsWith(' you'))
      str = str.findReplace(' you', ' me', ReplaceOnce,
        str.length-5);
  }
  return str.toLower();      
}


//----------------------------------------------------------------
// Susan
//---------------------------------------------------------------

susan: Person 'susan/sue/woman/girl' 'Susan' @bench
  "She looks very nice. "
  isHer = true
  isProperName = true
  posture = sitting
  sightSize = large
  isKnown = true
  cannotKissActorMsg = 'Susan likes that. '  
;

+ susanTalking: InConversationState
;

++ AskTopic @bob
  "<q>Do you know where I can find Bob?</q> you ask.\b
   <q>I think he's inside.</q> she replies. "
;

++ DefaultAskTopic
   "<q>Susan, I'd like to ask you about <<gTopicText>>,</q> you say.\b
    <q>Really? What fun!</q> she laughs. "
;

++ DefaultTellTopic
   "<q>I need to talk to you about <<gTopicText>>,</q> you announce.\b
    <q>You kniw, I just love it when people say that!</q> she smiles brightly. "
;


++ GiveShowTopic @rubyRing
   topicResponse
   {
     "<q>I think you might want this,</q> you say, handing her the ring.\b
     Her eyes light up with surprise and delight, <q>Hey! That's wonderful!
     Wherever did you find it!</q> she cries, <q>That's brilliant! Thanks
     ever so much!</q><.p>
     She's so grateful she even gives you a great big kiss before slipping the
     ring back on her finger.<.p>";
     achievement.awardPointsOnce();
     rubyRing.returned = true;
     if(letter.returned)
     {
       libGlobal.totalTurns++; // otherwise the winning turn isn't counted 
       finishGameMsg(ftVictory, [finishOptionUndo, finishOptionFullScore]);
     }
     else
      "<q>Now all I need is my letter back!</q> she sighs. <<gSetKnown(letter)>>
       <.reveal letter-missing>"; 
   }
   achievement : Achievement { +5 "giving Susan her ring back" }
; 

++ DefaultGiveShowTopic
   "{You/he} show{s} her {the dobj/him}.\b
    <q>That's nice!</q> she beams. "
;

++ AskTellTopic @rubyRing
   "<q>What's this I hear about your ring?</q> you ask.\b
    <q>It's been stolen, and I'd really like it back,</q> she tells you.
    <q>We think the thief may have hidden it in the house somewhere.</q> "
;

++ AskTellTopic @pileOfGold
   "<q>Susan, there's a great big pile of gold just sitting in the meadow
    across the river!</q> you declare.\b
    <q>Really?</q> she says, <q>What a strange place to leave it! But never
     mind that, I just want my ring back!</q>"
;

++ AskTopic, StopEventList @letter
  [
   '<q>What\'s this about a letter?</q> you ask.\b
    <q>It went missing at the same time as my ring,</q> Susan explains,
    <q>Only, it would be rather awkward if the wrong person found it, so
     I\'d be very grateful if you\'d look out for it.</q>',
   
   '<q>What does the missing letter look like?</q> you ask.\b
    <q>It\'s in a pale blue envelope,</q> she tells you, <q>Please
    bring it straight back to me if you find it!</q>'
  ]
  isActive = gRevealed('letter-missing') && !letter.returned
;

++ GiveShowTopic @envelope
  topicResponse()
  {
    letter.returned = true;
    envelope.moveInto(susan);   
    envelope.makeOpen(true);
    letter.achievement.awardPointsOnce(); 
    "Susan takes the envelope and finds the letter inside, breathing a huge sigh
     of relief. <q>Thank goodness!</q> she declares, <q>Have you read this?</q> she
     asks.<.convnode read-letter>";
  }
;

+++ AltTopic
  "Susan takes the envelope and looks inside, a worried frown creasing her face,
   <q>This looks like the envelope, but it's empty!</q> she cries. She hands
   the envelope back to you, adding, <q>I'm afraid it's the letter I need, George;
   without it the envelope isn't much use to me.</q>"
  isActive = !(letter.isIn(envelope))
;

++ GiveShowTopic @letter
  topicResponse()
  {
     letter.returned = true;
     letter.moveInto(susan);
     letter.achievement.awardPointsOnce();
     "Susan takes the letter, glances at it quickly, then looks up at you with
      some consternation, <q>Have you read this?</q> she demands. <.convnode read-letter>";      
  }
;

++ DefaultAskForTopic
   "<q>Susan, I'd really like <<gTopic.getTopicText>>,</q> you tell her.\b
   <q>You can have anything you like!</q> she smiles. "
;


++ ConversationReadyState
  isInitState = true
;

+++ HelloTopic
  "<q>Hello!</q> she exclaims, <q>How lovely to see you again!</q> "
;

+++ ByeTopic
  "<q>Take care now - and come back soon!</q> Susan smiles at you.<.p>"
;

+ ConvAgendaItem
  isReady { return inherited && getActor.isIn(gPlayerChar.getOutermostRoom) && !getActor.canSee(bob); }
  initiallyActive = true
  invokeItem()
  {
     getActor.initiateConversation(susanTalking, 'letterNode');
  }
  isDone = (letter.returned || gRevealed('letter-missing'))  
;

+ ConvNode 'letterNode'
  npcGreetingMsg = "<q>You've heard about my ring?</q> Susan asks, <q>Well, I'm terribly
   afraid it's not the only thing the thief took. There was a letter -- only -- well, it's
   all rather embarrassing. Anyway, it went missing at the same time, so I think the thief
   took it too. Only I hadn't made up my mind to send it -- and I'd really like it back,
   even more than my ring almost! So if you come across it I'd be <i>really</i> grateful
   if you could return it to me.</q>
   <<gSetKnown(letter)>><.reveal letter-missing>"
;

+ ConvNode 'read-letter'
  npcContinueMsg = "<q>Have you read the letter?</q> Susan insists on knowing. "
  isSticky = true
  canEndConversation(actor, reason)
  {
    "You'd better answer Susan's question. ";
    return nil;
  }
;

++ YesTopic
   "<q>Yes, I have,</q> you confess.\b
    <q>Then you know what's in it,</q> she remarks, <q>So -- now you know how
     I feel about you -- how do you feel about me? Do you love me?</q><.convnode love-me>";
;

++ NoTopic
  topicResponse()
  {
    "Susan hesitates a moment, then hands you the letter, <q>Perhaps you'd better
     read it,</q> she suggests. ";
     letter.moveInto(gPlayerChar);
  }
;

+++ AltTopic
  "<q>No, not yet,</q> you say.\b
   <q>Well, hurry up then -- it's not that long!</q> she complains. "
  isActive = letter.isIn(gPlayerChar)
;

++ DefaultAnyTopic
  "<q>This letter -- have you read it?</q> Susan demands. "
;

+ ConvNode 'love-me'
  npcContinueMsg = "<q>Do you love me?</q> Susan wants to know. "
  isSticky = true
  canEndConversation(actor, reason)
  {
    "You'd better answer Susan's question. ";
    return nil;
  }
;

++ YesTopic
  topicResponse()
  {
    "<q>Yes, Susan, I do!</q> you declare warmly, <q>To be honest, I've always been attracted
     to you, and reading this letter -- well, it was amazing! I never realized you felt that
     way too!</q>\b
     <q>Well, now you know I do,</q> she smiles.<.p>";
     if(rubyRing.returned)
     {
        "She leaps up from the bench and throws her arms round you. Some time later, after
         you've temporarily finished kissing and cuddling, you start to plan your new life
         together. Only one thing mars your happiness: you know that Bob will never forgive you. ";
        finishGameMsg(ftVictory, [finishOptionFullScore, finishOptionUndo]);
     }
     else
       "She continues to smile up at you, and then remarks, <q>Well, the one thing needed to
        make my day perfect is for you to find my ring!</q>";
  }
;

++ NoTopic
 topicResponse()
 {
  "You have always been attracted to Susan, and you are very tempted to say yes, because
   you really like her, and perhaps, if you were really honest with yourself, you are
   in love with her. But you remember that Bob is also your friend, so you decide to
   do the honourable thing.\b
   <q>No, Susan,</q> you lie, <q>I really like you a lot, but I'm not in love with
   you.</q>\b
   She looks momentarily shocked, then a little sad, before saying, <q><<letter.isIn(gPlayerChar)
   ? 'Then I think you\'d better give me that letter back.</q>\b You duly oblige.' : 'Hey, ho!</q>.'>><.p>";
   letter.moveInto(susan);
   if(rubyRing.returned)
   {
     "<q>Perhaps you'd better leave me now,</q> Susan suggests.\b
     You hesitate for a moment, then turn and walk slowly away.<.p>";
     finishGameMsg(ftVictory, [finishOptionFullScore, finishOptionUndo]);
   }
   else
     "<q>Well!</q> Susan declares with a brave attempt at brightness, <q>Now all you need
      to do is to find my ring!</q>";
 }
;

++ AskTellTopic @bob
  "<q>But what about Bob?</q> you ask.\b
   <q>You've read the letter,</q> she points out, <q>I don't want to 
    hurt Bob, but it's you I love -- so what I need to know is, do
    you love me?</q>"
;


++ DefaultAnyTopic
   "<q>Never mind <<gTopicText>>,</q> she replies, <q>What I need to
    know right now is whether you love me -- yes or no?</q>"
;



/* Custom Verbs */

DefineTAction(Play)
;

VerbRule(Play)
  'play' singleDobj
  : PlayAction
  verbPhrase = 'play/playing (what)'
;

modify Thing
  dobjFor(Play)
  {
    preCond = [touchObj]
    verify() { illogical('{That dobj/he} {is} not something {you/he} can play. '); } 
  }
;


modify VerbRule(MoveWith)
     ('move' | 'push' | 'pull' | 'shove') singleDobj 'with' singleIobj
     :
;


