#charset "us-ascii"

/* 
 *  Copyright (c) 2006 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of the TADS 3 Data-structures Library Extension
 *
 *  LinkedList.t
 *
 *  Defining the LinkList class.
 */

#include <tads.h>
#include <t3.h>

/*
 *  A base class for linked lists.
 */
class LinkedListBase: Node
{
    /*
     *  Appends the value val to this list, returning 
     *  the resulting list.
     */
    append(val)
    {
        local c, t, n, hRef, tRef;

        /* create a copy of our list */
        c = createCopy();
       
        if (c.length() == 0)
        {
            c.data  = val;
            c.type  = dataType(val);
            return c;
        }

        /* find the "tail" */
        t = c.nodeLast();

        /* save the tail "next" reference */
        tRef = t.next;

        /* save the head "prev" reference */
        hRef = c.prev;

        /* create a new node */
        n = createNewNode(val);

        /* append the new node to the "tail" */
        t.next = n;

        /*
         *  If this is a double linked list then we
         *  set the new node's previous to point to 
         *  the tail.
         */
        setNewNodePrev(n, t);

        /*
         *  If we have a "tail" ref then this is 
         *  a circular linked list and we need to
         *  set the new node's next to the tail
         *  reference.
         */
        if (tRef)
        {
            n.next = tRef;
        }

        /*
         *  If we have a "head" ref then this is 
         *  a circular linked list (specifically a
         *  double circular linked list) and we need
         *  to se the head prev to the new node.
         */
        if (hRef)
        {
            c.prev = n;
        }

        return c;
    }

    /*
     *  This works like append(val), except that if val is a List
     *  each element of val is individually appended to the new list.
     */
    appendAll(val)
    {
        local c;

        if (dataType(val) == TypeObject
            && val.ofKind(LinkedListBase))
        {
            c = self;
            val.forEach(new function(v)
            {
                if (v.type)
                    c = c.append(v.data);
            });
        }
        else
        {
            c = append(val);
        }

        return c;
    }

    /*
     *  Appends the elements of the list lst to this list, 
     *  returning a new list consisting only of the unique 
     *  elements of the combination.  Each value appearing 
     *  in the result list will appear in the result list 
     *  only once. 
     */
    appendUnique(node)
    {
        local c;

        c = createCopy();

        node.forEach(new function(n)
        {
            if (n.type)
            {
                if (c.indexOf(n.data) == nil)
                    c = c.append(n.data);
            }
        });

        return c;
    }

    /*
     *  Returns the first element of the list.  If the list 
     *  has no elements, returns nil.
     */
    car()
    {
        return data;
    }

    /*
     *  Returns the "tail" of the list; that is, the rest 
     *  of the list after removing the first node.  If 
     *  the list has an empty node, returns nil.   
     */
    cdr()
    {
        if (length())
            return next;
        else return nil;
    }

    /*
     *  Returns the number of nodes whose values equal val.
     */
    countOf(val)
    {
        local cnt = 0;

        forEach(new function(n)
        {
            if (n.type && n.data == val)
                cnt++;
        });

        return cnt;
    }

    /*
     *  Returns the number of nodes for which the callback 
     *  function cond returns a non-false value (anything 
     *  but nil or 0).  For each node, this method calls the 
     *  callback, passing the current node as the argument.  
     *  If the callback returns anything but nil or 0, the 
     *  method counts the node.  After invoking the callback 
     *  for each node, the method returns the resulting count.
     */
    countWhich(cond)
    {
        local cnt = 0;

        forEach(new function(n)
        {
            if (n.type && cond(n))
                cnt++;
        });

        return cnt;
    }

    /*
     *  Returns a copy of the list. 
     */
    createCopy()
    {
        local c, h, p, hp, len;

        len = length();
        forEachAssoc(new function(i, v)
        {
            c = v.createClone();
            c.setSuperclassList(getSuperclassList());
            if (i == 1)
            {
                h = c;
                if (v.prev)
                    hp = true;

                if (len < 2)
                {
                    if (hp)
                        h.prev = c;
                    if (v.next)
                        h.next = c;
                }
            }
            else
            {
                p.next = c;
                if (v.prev)
                    c.prev = p;

                if (i == len)
                {
                    if (v.next)
                        c.next = h;
                    if (hp)
                        h.prev = c;
                }
            }
            p = c;    
        });

        return h;
    }

    /*
     *  Converts one LinkList class into another. This
     *  is a class level method that only needs the 
     *  "head" node of a LinkedList in order to convert
     *  the entire list to another LinkedList class.
     */
    cvtLinkedList(node)
    {
        return createInstance(node.cvtToTadsList());
    }

    /*
     *  Converts this list to a native TADS 3 list.
     */
    cvtToTadsList()
    {
        local list = [];
        forEach(new function(node)
        {
            if (node.type)
            {
                if (node.type == TypeList)
                    list = list.append(node.data.cvtToTadsList());
                else
                    list += node.data;
            }
        });

        return list;
    }

    /*
     *  Returns true if for each node of this list the type
     *  and data are equal to the associated sequential node
     *  type and data for list node, and if the two lists are
     *  of equal length(). Otherwise returns nil.
     */
    equals(node)
    {
        local ret;

        if (length() != node.length())
            return ret;

        ret = true;
        forEachAssoc(new function(i, n)
        {
            local v;

            v   = node.nodeAt(i);

            if (v.type != n.type)
            {
                ret = nil;
                throw new LinkedListSignal();
            }

            if (v.data != n.data)
            {
                if (v.type == TypeObject
                    && v.data.ofKind(LinkedListBase)
                    && n.data.ofKind(LinkedListBase))
                {
                    if (!v.data.equals(n.data))
                    {
                        ret = nil;
                        throw new LinkedListSignal();
                    }
                }
                else
                {
                    ret = nil;
                    throw new LinkedListSignal();
                }
            }
        });
        
        return ret;
    }

    /*
     *  Invokes the callback function func(node) for each node, 
     *  in order from first to last, passing the value of one 
     *  node as value to the callback on each invocation.  The 
     *  callback function takes one argument, which is the current 
     *  node, and returns no value.  This method returns no value. 
     */
    forEach(func)
    {
        local node;

        try
        {
            node    = self;
            do
            {
                func(node);
                node = node.next;
            }
            while (node != nil && node != self);
        }
        catch(LinkedListSignal s)
        {
            // do nothing
        }
    }

    /*
     *  Invokes the callback function func(index, node) for 
     *  each node, in order from first to last, passing each 
     *  node's index and the node itself to the function func.  
     *  The callback function returns no value.  This method 
     *  returns no value.
     */
    forEachAssoc(func)
    {
        local node, index;

        try
        {
            node    = self;
            index   = 0;
            do
            {
                index++;
                func(index, node);
                node = node.next;
            }
            while (node != nil && node != self);
        }
        catch(LinkedListSignal s)
        {
            // do nothing
        }
    }

    /*
     *  Returns a new list consisting of the unique nodes 
     *  of the original list.  For each value in the original 
     *  list, the value will appear in the new list only once.  
     *  The order of the nodes in the new list is that of the 
     *  first appearances of the unique nodes of the original 
     *  list.  
     */
    getUnique()
    {
        local c;
        
        c = createNewNode(data);
        forEach(new function(n)
        {
            if (n.type)
            {
                if (c.indexOf(n.data) == nil)
                    c = c.append(n.data);
            }
        });

        return c;
    }

    /*
     *  Returns the index of the first node of the list equal 
     *  to the given value.  If val does not appear anywhere in 
     *  the list, the method returns nil.  The first node is at 
     *  index 1.
     */
    indexOf(val)
    {
        local index;

        forEachAssoc(new function(i, n)
        {
            if (dataType(val) == TypeObject
                && val.ofKind(LinkedListBase)
                && n.type == TypeObject
                && n.data.ofKind(LinkedListBase))
            {
                if (n.data.equals(val))
                {
                    index = i;
                    throw new LinkedListSignal();
                }
            }
            else if (n.type && n.data == val)
            {
                index = i;
                throw new LinkedListSignal();
            }
        });
        
        return index;
    }

    /*
     *  Finds the first node for which the given condition 
     *  is true.  The method iterates through the nodes of 
     *  the list, starting at the first element and proceeding 
     *  in order, and applies the callback function cond to 
     *  each node.  The callback takes one argument, which 
     *  is the node of the list itself, and returns a condition 
     *  result value.  For each node, if the callback function 
     *  returns a non-false value (i.e., any value except nil 
     *  or zero), the method immediately stops the iteration 
     *  and returns the index of that node.  If the callback 
     *  returns a false value (nil or zero) for every element 
     *  of the list, the method returns nil.
     */
    indexWhich(cond)
    {
        local index;

        forEachAssoc(new function(i, n)
        {
            if (n.type && cond(n))
            {
                index = i;
                throw new LinkedListSignal();
            }
        });
        
        return index;
    }

    /*
     *  Returns a new list which results from inserting the given 
     *  values (val1, val2, and so on) into the existing list 
     *  before the node at the position given by index.  At 
     *  least one value argument is required; if multiple value 
     *  arguments are provided, the values are inserted into 
     *  the list in the same order in which they appear in the 
     *  arguments.
     *
     *  If index is 1, the values are inserted before the first 
     *  node of the existing list (this means that insertAt(1, x) 
     *  is equivalent to prepend(x)), and if index is equal to 
     *  the number of nodes in the original list plus 1, the 
     *  values are inserted after the last node of the existing 
     *  list (so node.insertAt(node.length() + 1, x) is equivalent 
     *  to node.append(x)).  An index value less than 1 or greater 
     *  than the number of nodes in the list plus 1 is illegal and 
     *  results in an index out of range" exception.
     */
    insertAt(indx, [vals])
    {
        local c, t, n1, n2, v, len;

        if (vals.length() == 0)
            throw new LinkedListError('no insert values. ');

        if (indx < 1 || indx > length() + 1)
            throw new LinkedListError('index out of range. ');

        c = createCopy();
        len = c.length();
        
        if (indx == 1)
        {
            for (local i = vals.length(); i > 0; --i)
                c = c.prepend(vals[i]);
        }
        else if (indx == len + 1)
        {
            for (local i = 1; i <= vals.length(); ++i)
                c = c.append(vals[i]);
        }
        else
        {
            n1  = c.nodeAt(indx - 1);
            n2  = n1.next;
            v   = createNewNode(vals[1]);
            for (local i = 2; i <= vals.length(); ++i)
                v = v.append(vals[i]);

            t = v.nodeLast();

            n1.next = v;
            if (n2.prev)
                v.prev = n1;

            t.next = n2;
            if (n2.prev)
                n2.prev = t;
        }

        return c;
    }

    /*
     *  Returns a new list consisting of the intersection of this 
     *  list and lst2; that is, a list consisting of the nodes 
     *  common to both this list and lst2.  lst2 must also be a 
     *  linked list.  If the two lists have no nodes in common, 
     *  the result is an empty list.  If a node value of the 
     *  shorter list (or, if the lists are of equal length, 
     *  this list) appears more than once in the shorter list, 
     *  and that node value also appears in the longer list, 
     *  then the element will be in the result list the same 
     *  number of times that it is in the shorter list.  
     *  A node value repeated in the longer list will not be 
     *  repeated in the result list.
     */
    intersect(node)
    {
        local a, b, c;

        if (length() < node.length())
        {
            a = self;
            b = node;
        }
        else
        {
            a = node;
            b = self;
        }

        c = createNewNode();
        a.forEach(new function(n)
        {
            if (n.type)
            {
                if (b.indexOf(n.data) != nil)
                    c = c.append(n.data);
            }
        });

        return c;
    }

    /*
     *  Returns true if the list is empty; otherwise
     *  returns nil if the list is non-empty.
     */
    isEmpty()
    {
        return length() == 0;
    }

    /*
     *  Returns the index of the last node in the list whose 
     *  value equals val.  If no node in the list equals 
     *  val, this method returns nil.
    */
    lastIndexOf(val)
    {
        local index;

        revForEachAssoc(new function(i, n)
        {
            if (dataType(val) == TypeObject
                && val.ofKind(LinkedListBase)
                && n.type == TypeObject
                && n.data.ofKind(LinkedListBase))
            {
                if (n.data.equals(val))
                {
                    index = i;
                    throw new LinkedListSignal();
                }
            }
            else if (n.type && n.data == val)
            {
                index = i;
                throw new LinkedListSignal();
            }
        });
       
        return index;
    }

    /*
     *  Returns the index of the last node in the list for 
     *  which the callback function cond returns a non-false 
     *  value (anything except nil or 0).  This method is 
     *  similar to indexWhich(cond), but scans the list 
     *  in reverse order, starting at the last node and 
     *  working towards the first.  Returns nil if the 
     *  callback returns nil or 0 for every node.
     */
    lastIndexWhich(cond)
    {
        local index;

        revForEachAssoc(new function(i, n)
        {
            if (n.type && cond(n))
            {
                index = i;
                throw new LinkedListSignal();
            }
        });
       
        return index;
    }

    /*
     *  Returns the value of the last node in the list 
     *  for which the callback function cond returns a 
     *  non-false value.  This method is similar to 
     *  lastIndexWhich(cond), but returns the value of 
     *  the matching node rather than its index.  Returns 
     *  nil if the callback returns nil or 0 for every node.
     */
    lastValWhich(cond)
    {
        local val;

        revForEach(new function(n)
        {
            if (n.type && cond(n))
            {
                val = n.data;
                throw new LinkedListSignal();
            }
        });
       
        return val;
    }

    /*
     *  Returns the number of nodes in the list or zero if 
     *  this is an empty list.
     */
    length()
    {
        local cnt = 0;

        forEach(new function(n)
        {
            if (n.type)
            {
                cnt++;
            }
        });

        return cnt;
    }

    /*
     *  For each node of the list, this method invokes the 
     *  callback function func, passing the current node 
     *  as the single argument, then adds the return value 
     *  to a new list.  The method returns the resulting 
     *  new list, which has the same number of nodes as the 
     *  original list.  Each node of the returned list 
     *  contains the result returned by func for the 
     *  corresponding node of the original list.  This 
     *  method does not modify the original list, but 
     *  creates a new list.
     */
    mapAll(func)
    {
        local c;

        c = createNewNode();
        forEach(new function(n)
        {
            c = c.append(func(n));
        });

        return c;
    }

    /*
     *  Returns the node at the specified index of the list
     */
    nodeAt(index)
    {
        local node;

        forEachAssoc(new function(i, n)
        {
            if (i == index)
            {
                node = n;
                throw new LinkedListSignal();
            }
        });

        return node;
    }

    /*
     *  Returns this node
     */
    nodeFirst()
    {
        return nodeAt(1);
    }

    /*
     *  Returns the last node in the list, or returns this
     *  node if the node has no length.
     */
    nodeLast()
    {
        local len;

        len = length();
        if (len)
            return nodeAt(len);
        else return nodeAt(1);
    }

    /*
     *  Returns the next node in the list.
     */
    nodeNext()
    {
        if (next == nil)
            throw new LinkedListError('No next node in list. ');

        return next;
    }

    /*
     *  Returns the previous node in the list.
     */
    nodePrev()
    {
        if (prev == nil)
            throw new LinkedListError('No prev node in list. ');

        return prev;
    }

    /*
     *  Returns a new list which results from inserting the value 
     *  val before the first element of the existing list.  This 
     *  method is similar to append(val), but this method inserts 
     *  the new node at the beginning of the list rather than at 
     *  the end.  Note that if val is a list, it is prepended as 
     *  a single list-valued node.
     */
    prepend(val)
    {
         local c, t, n, hRef, tRef;

        /* create a copy of our list */
        c = createCopy();
       
        if (c.length() == 0)
        {
            c.data  = val;
            c.type  = dataType(val);
            return c;
        }

        /* find the "head" */
        t = c.nodeLast();

        /* save the tail "next" reference */
        tRef = t.next;

        /* save the head "prev" reference */
        hRef = c.prev;

        /* create a new node */
        n = createNewNode(val);

        /* set the new node's next to the "head" */
        n.next = c;

        /*
         *  If this is a double linked list then we
         *  set the head's previous to point to the 
         *  new node.
         */
        if (c.ofKind(DoubleLinkedList))
            c.prev = n;

        /*
         *  If we have a "tail" ref then this is 
         *  a circular linked list and we need to
         *  set the tail's next to the new node.
         */
        if (tRef)
        {
            t.next = n;
        }

        /*
         *  If we have a "head" ref then this is 
         *  a circular linked list (specifically a
         *  double circular linked list) and we need
         *  to set the new "head" prev to the head
         *  reference.
         */
        if (hRef)
        {
            n.prev = hRef;
        }

        return n;
   }

    prependAll(val)
    {
        local c;

        if (dataType(val) == TypeObject
            && val.ofKind(LinkedListBase))
        {
            c = val;
            forEach(new function(n)
            {
                if (n.type)
                    c = c.append(n.data);
            });
        }
        else
        {
            c = prepend(val);
        }

        return c;
    }

    /*
     *  Returns a new list which results from deleting the node 
     *  at the given index.  
     */
    removeElementAt(index)
    {
        local c, r, n, p;

        if (index < 1 || index > length())
            throw new LinkedListError('index out of range. ');

        c = createCopy();
        if (c.length() == 1)
        {
            c.data  = nil;
            c.type  = nil;
        }
        else
        {
            r = c.nodeAt(index);
            p = r.prev;
            n = r.next;
            if (p)
                p.next = n;
            if (n)
                n.prev = p;
            if (index == 1)
                c = n;
        }

        return c;
    }

    /*
     *  Returns a new list which results from deleting the nodes 
     *  starting at startIndex and ending with (and including) 
     *  endIndex.  If endIndex equals startIndex, only one node 
     *  is deleted, so removeRange(x, x) is equivalent to 
     *  removeElementAt(x). If no endIndex value is supplied,
     *  then the range extends to the end of the list.
     *  
     *  Both index values startIndex and endIndex must be at 
     *  least 1 and at most the number of nodes in the list, 
     *  and endIndex must be greater than or equal to startIndex; 
     *  the method throws an index out of range exception if 
     *  these conditions are not met.
     */
    removeRange(startIndex, [endIndex])
    {
        local c, cnt;

        endIndex = endIndex.car();
        if (endIndex == nil)
            endIndex = length();

        if (startIndex < 1 || endIndex > length())
            throw new LinkedListError('index out of range. ');

        if (endIndex < startIndex)
            throw new LinkedListError('index out of range. ');

        c = createCopy();
        cnt = endIndex - startIndex + 1;
        for (local i = 1; i <= cnt; ++i)
            c = c.removeElementAt(startIndex);

        return c;
    }

    /* 
     *  Similar to forEach(), except that the list is converted
     *  to a DoubleLinkedList first and then processed in reverse.
     */
    revForEach(func)
    {
        local c, node;

        try
        {
            c       = DoubleLinkedList.cvtToDoubleLinkedList(self);

            node    = c.nodeLast();
            do
            {
                func(node);
                node = node.prev;
            }
            while (node != nil && node != self);
        }
        catch(LinkedListSignal s)
        {
            // do nothing
        }
    }

    /* 
     *  Similar to forEachAssoc(), except that the list is converted
     *  to a DoubleLinkedList first and then processed in reverse.
     */
    revForEachAssoc(func)
    {
        local c, node, index;

        try
        {
            c       = DoubleLinkedList.cvtToDoubleLinkedList(self);
            node    = c.nodeLast();
            index   = c.length();
            do
            {
                index--;
                func(index, node);
                node = node.next;
            }
            while (node != nil && node != self);
        }
        catch(LinkedListSignal s)
        {
            // do nothing
        }
    }

    /*
     *  Set the data value of node at index in the list.
     */
    setValAt(index, [value])
    {
        nodeAt(index).setValThis(value...);
    }

    /*
     *  Set the data value of the first node of the list.
     */
    setValFirst([value])
    {
        nodeFirst().setValThis(value...);
    }

    /*
     *  Set the data value of the last node of the list.
     */
    setValLast([value])
    {
        nodeLast().setValThis(value...);
    }

    /*
     *  Set the data value of the prev node of the list.
     */
    setValPrev([value])
    {
        nodePrev().setValThis(value...);
    }

    /*
     *  Set the data value of the next node of the list.
     */
    setValNext([value])
    {
        nodeNext().setValThis(value...);
    }

    /*
     *  Returns a new list consisting of the elements of 
     *  this list rearranged into a sorted order. By default,
     *  this method sorts the elements of the list into
     *  ascending order, but you can reverse this ordering 
     *  by specifying true for the descending parameter. 
     *
     *  The optional comparisonFunction can be used to specify 
     *  the ordering of the result. If this argument is not 
     *  specified (or is nil), the method will sort the elements 
     *  according to the standard system ordering of values; 
     *  hence, the elements must be of comparable types 
     *  (such as all integers or all strings). By specifying 
     *  a comparison function, you can provide your own 
     *  special ordering, and you can also sort values that 
     *  have no system-defined order, such as object values. 
     *
     *  If provided, the comparisonFunction value is a pointer 
     *  to a function taking two arguments, which are two 
     *  values from the list to be compared. The function 
     *  returns an integer less than zero if the first value 
     *  is less than the first, zero if the two values are 
     *  equal, or an integer greater than zero if the first 
     *  value is greater than the second. 
     */
    sort([parms])
    {
        local list;

        list = cvtToTadsList();

        list = list.sort(parms...);

        return createInstance(list...);
    }

    /*
     *  Creates and returns a new list consisting of
     *  a sublist of this list starting at the node of 
     *  this list at index start, and continuing for 
     *  the number of elements given by length, if present, 
     *  or to the end of this list if not.
     */
    sublist(start, [len])
    {
        local c;

        if (start < 1)
            throw new LinkedListError('index out of range. ');

        len = len.car();    
        if (len == nil || len > length() - start + 1)
            len = length() - start + 1;
      
        c = createCopy();

        if (start > 1)
            c = c.removeRange(1, start - 1);

        if (c.length() > len)
            c = c.removeRange(len + 1);

        return c;                
    }

    /*
     *  Creates and returns a new list containing the nodes 
     *  of this list for which the callback function func 
     *  returns true (i.e., any value other than nil or the 
     *  integer value 0).  For each node of the source list, 
     *  this method invokes the callback function, passing 
     *  the node as the callback function's single argument.  
     *  If the callback returns nil or the integer value 0, 
     *  the method omits the node from the result; otherwise, 
     *  the method includes the node in the result list.  
     *  The new list's nodes will be in the same order as 
     *  the selected nodes from the source list.
     *  
     *  This method does not modify the original list.
     */
    subset(func)
    {
        local c;

        c = createNewNode();
        forEach(new function(n)
        {
            if (n.type && func(n))
                c = c.append(n.data);
        });

        return c;
    }

    /*
     *  Returns the data value of the list node at location index.
     */
    valAt(index)
    {
        return nodeAt(index).valThis();
    }

    /*
     *  Returns the data value of the first node
     *  in the list.
     */
    valFirst()
    {
        return nodeFirst().valThis();
    }

    /*
     *  Returns the data value of the next node
     *  in the list.
     */
    valNext()
    {
        try
        {
            return nodeNext().valThis();
        }
        catch(LinkedListError e)
        {
            throw new LinkedListError('No next value in list. ');
        }
    }

    /*
     *  Returns the data value of the prev node
     *  in the list.
     */
    valPrev()
    {
        try
        {
            return nodePrev().valThis();
        }
        catch(LinkedListError e)
        {
            throw new LinkedListError('No prev value in list. ');
        }
    }

    /*
     *  Returns the data value of the last node
     *  in the list.
     */
    valLast()
    {
        return nodeLast().valThis();
    }

    /*
     *  Returns the value of the first node for which the 
     *  callback function cond returns non-false (i.e., 
     *  anything except nil or 0).  This method is similar 
     *  to indexWhich(cond), but returns the value of the 
     *  first matching node rather than its index.  If 
     *  cond returns nil or 0 for every node of the list, 
     *  this method returns nil.
     */
    valWhich(cond)
    {
        local val;

        forEach(new function(n)
        {
            if (n.type && cond(n))
            {
                val = n.data;
                throw new LinkedListSignal();
            }
        });
       
        return val;
    }

    setDataAsList(list)
    {
        data    = createInstance(list...);
    }

    construct([elms])
    {
        prev    = nil;
        next    = nil;

        data    = elms.car();
        if (elms.length() == 0)
            type    = nil;
        else
        {
            type    = dataType(data);
        
            if (type == TypeList)
                setDataAsList(data);

            elms    = elms.cdr();
            
            if (elms.length() > 0)
                next    = createInstance(elms...);
        }
    }
    
    createNewNode([val])
    {
        local n;

        n = createClone();
        n.setSuperclassList(getSuperclassList());
        if (val.length())
        {
            val     = val.car();
            n.data  = val;
            n.type  = dataType(val);
        }
        else
        {
            n.data  = nil;
            n.type  = nil;
        }

        setNewNodeRefs(n);

        return n;
    }
    setNewNodeRefs(node) { node.next = node.prev = nil; }
    setNewNodePrev(node, val) {}
}


/*--------------------------------------------------------------------
 *
 *  A list which has one link per node. This link points to the next 
 *  node in the list, or to a nil value/type if it is the final node. 
 *  A singly linked list's node is divided into two parts. The first 
 *  part holds or points to information about the node, and second 
 *  part holds the address of next node. A singly linked list travels 
 *  one way.
 */
class SingleLinkedList: LinkedListBase
{
    cvtToSingleLinkedList(node)
    {
        local c, t;

        c   = node.createCopy();

        c.forEach(new function(n)
        {
            n.setSuperclassList([self]);
            n.prev  = nil;
        });

        t       = c.nodeLast();
        t.next  = nil;
        c.prev  = nil;

        return c;
    }
}

/*--------------------------------------------------------------------
 *  
 *  A list in which each node has two links: one points
 *  to the previous node, or points to a nil value/type,
 *  if it is the first node; and one points to the next,
 *  or points to a nil value if it is the final node.
 */
class DoubleLinkedList: LinkedListBase
{
    construct([elms])
    {
        inherited(elms...);
        if (next)
            next.prev = self;
    }

    setNewNodePrev(node, val)
    {
        node.prev = val;
    }

    cvtToDoubleLinkedList(node)
    {
        local c, p, t;

        c   = node.createCopy();

        c.forEach(new function(n)
        {
            n.setSuperclassList([self]);
            n.prev  = p;
            p       = n;
        });

        t       = c.nodeLast();
        t.next  = nil;
        c.prev  = nil;

        return c;
    }
}

/*--------------------------------------------------------------------
 *  
 *  A list in which each node has one link, similar to an ordinary 
 *  singly-linked list, except that the next link of the last node 
 *  points back to the first node. 
 */
class CircularSingleLinkedList: SingleLinkedList
{
    __tmpFirst      = nil

    setDataAsList(data)
    {
        local sav;

        sav     = CircularSingleLinkedList.__tmpFirst;
        CircularSingleLinkedList.__tmpFirst = nil;
        inherited(data);
        CircularSingleLinkedList.__tmpFirst = sav;
    }

    construct([elms])
    {
        if (CircularSingleLinkedList.__tmpFirst == nil)
            CircularSingleLinkedList.__tmpFirst = self;

        inherited(elms...);
        
        if (CircularSingleLinkedList.__tmpFirst == self)
        {
            nodeLast().next = self;
            CircularSingleLinkedList.__tmpFirst = nil;
        }
    }

    setNewNodeRefs(node)
    {
        inherited(node);
        node.next = node;
    }
}

/*--------------------------------------------------------------------
 *  
 *  A list in which each node has two links, similar to a doubly-linked 
 *  list, except that the previous link of the first node points to the 
 *  last node and the next link of the last node points to the first node. 
 */
class CircularDoubleLinkedList: DoubleLinkedList
{
    __tmpFirst      = nil

    setDataAsList(data)
    {
        local sav;

        sav     = CircularDoubleLinkedList.__tmpFirst;
        CircularDoubleLinkedList.__tmpFirst = nil;
        inherited(data);
        CircularDoubleLinkedList.__tmpFirst = sav;
    }

    construct([elms])
    {
        if (CircularDoubleLinkedList.__tmpFirst == nil)
            CircularDoubleLinkedList.__tmpFirst = self;

        inherited(elms...);
        
        if (CircularDoubleLinkedList.__tmpFirst == self)
        {
            prev = nodeLast();
            nodeLast().next = self;
            CircularDoubleLinkedList.__tmpFirst = nil;
        }
    }


    setNewNodeRefs(node)
    {
        inherited(node);
        node.next = node.prev = node;
    }

    cvtToCircularDoubleLinkedList(node)
    {
        local c, p, t;

        c   = node.createCopy();

        c.forEach(new function(n)
        {
            n.setSuperclassList([self]);
            n.prev  = p;
            p       = n;
        });

        t       = c.nodeLast();
        t.next  = c;
        c.prev  = t;

        return c;
    }
}

class LinkedListSignal: RuntimeError
{
    construct()
    {
        inherited(10);
    }
}

class LinkedListError: RuntimeError
{
    construct(msg)
    {
        exceptionMessage = msg;
        inherited(10);
    }
}