#charset "us-ascii"

/* Copyright (c) 2004 M.D. Dollahite.  All Rights Reserved. */
/*
 *   Library Extention: window
 *
 *   Create things that let you look from one room into another.
 */
 
#include <adv3.h>

/*--------------------------------------------------------------------------*/
/*
 *   Messages, up here at the top so they can be changed/translated easily.
 */
modify libMessages
    sayOpenWindowRemotely(window, stat)
    {
        "Someone <<stat ? 'open' : 'close'>>s <<window.theNameObj>> from
        the other side. ";
    }
;

modify playerActionMessages
    cannotSeeThroughWindowMsg(obj)
    {
        gMessageParams(obj);
        return '{You/he} can see nothing through {the obj/him}. ';
    }

    cannotSeeThroughClosedWindowMsg(obj)
    {
        gMessageParams(obj);
        return '{You/he} cannot see through {the obj/him} because it is closed. ';
    }
    
    windowObstructedMsg(obj, trans)
    {
        gMessageParams(obj);
        return '{You/he} can\'t see through {the obj/him} from here. ';
    }
;

/*--------------------------------------------------------------------------*/
/*
 *   The Window class creates objects that link two rooms together through
 *   a pair of one-way sense connectors. The objects themselves are facets, 
 *   so this can create visible fixtures on each side.
 *
 *   Although this was designed to model windows, it can in principle be used
 *   for other two-way sense connections, such as telephones or video 
 *   teleconference devices.
 *
 *   The BasicWindow class does not handle LOOK THROUGH commands. The Window
 *   class adds that capability.
 */
class BasicWindow: OneWaySenseConnector, BasicOpenable, Thing
    /*
     *   Description of what we see on the other side of the window.
     *   This is displayed at the end of an EXAMINE description, and
     *   subclasses use it to handle LOOK THROUGH, etc.
     *
     *   By default we use the standard room description generator and
     *   simply omit the room name.
     */
    lookThroughDesc()
    {
        /*
         *   If we have a linked object, tell it to do a look around.
         *   Otherwise, just use our "can't see anything" message.
         */
        if(otherSide != nil)
        {
            otherSide.lookAround(gActor, LookRoomDesc|LookListSpecials|LookListPortables);
        }
        else
            mainReport(&cannotSeeThroughWindowMsg);
    }

    /*
     *   Our corresponding window object on the other side of the
     *   wall.  This will be set automatically during initialization
     *   based on the masterObject property of the slave - it is not
     *   generally necessary to set this manually.
     */
    otherSide = nil

    /* our other side is our other "facet" */
    getFacets() { return otherSide != nil ? [otherSide] : inherited(); }
    
    /*
     *   This is the material of the window itself.  This is combined with
     *   the connectorMaterial (see OneWaySenseConnector) when the window
     *   is closed. Note that this has to be set on both sides of the window.
     */
    material = adventium

    /*
     *   Sensing through a window accounts for both the connectorMaterial and
     *   the window material.
     */
    transSensingThru(sense)
    {
        return (isOpen) ? inherited(sense) :
            transparencyAdd((material.senseThru(sense)), inherited(sense));
    }
    
    /*
     *   Our otherSide is our only connected location
     */
    connectionList = [otherSide]

    /*
     *   Initialize the master object.  The other side of a two-sided window
     *   will call this on the master object to let the master object know
     *   about the other side.  'other' is the other-side object.  By
     *   default, we'll simply remember the other object in our own
     *   'otherSide' property.
     */
    initMasterObject(other)
    {
        otherSide = other;
    }

    /*
     *   Initialize.  If we're a slave, we'll set up the otherSide
     *   relationship between this window and our master window.
     */
    initializeThing()
    {
        /* inherit default handling */
        inherited();

        /* 
         *   if we have a master side, initialize our relationship with
         *   the master side
         */
        if (masterObject != self)
        {
            /* set our otherSide to point to the master */
            otherSide = masterObject;

            /* set the master's otherSide to point to us */
            masterObject.initMasterObject(self);
        }
    }
    
    /*
     *   Generate a look through description
     */
    lookThrough(actor)
    {
        local trans = actor.senseObj(sight, self).trans;

        /*
         *   If the actor has an unobstructed view of us, do the look through.
         *   Otherwise, report that we can't see through the window very well.
         */
        if(trans == transparent)
        {
            /*
             *   If the actor can see our other side, look through us.
             *   Otherwise, display our cannot see through message.
             */
            //if(actor.canSee(otherSide)) lookThroughDesc;
            if(transSensingThru(sight) != opaque) lookThroughDesc;
            else mainReport(&cannotSeeThroughWindowMsg, self);
        }
        else
        {
            /* Display our obstructed view message. */
            mainReport(&windowObstructedMsg, self, trans);
        }
    }

    /*
     *   When examined, we'll also display what can be seen through us.
     */
    examineStatus()
    {
        /* inherit the default handling */
        inherited;
        
        /*
         *   Call on our look through handler
         */
        lookThrough(gActor);
    }

    /*
     *   If we're being used as the POV for a look through of some kind,
     *   artificially include the sense path through the window at the
     *   beginning of the calculation.
     */
    sensePathToLoc(sense, trans, obs, fill)
    {
        local transThru = trans;

        /* Fake a passage through the window at the start of the path */
        if(getPOV() == self)
        {
            /* Adjust the transparency */
            transThru = transparencyAdd(trans, otherSide.transSensingThru(sense));

            /* if we changed the transparency, we're the obstructor */
            if(transThru != trans) obs = self;
        }

        /* inherit the default handling */
        inherited(sense, transThru, obs, fill);
    }

    /*
     *   It would be more correct to override sensePathFromWithout
     *   on the side being used as the pov, but that would require
     *   a lot of duplicated code since only the tail end of the
     *   method should be dropped. It's easier and more efficient
     *   therefore to stop the sense path on the other side instead.
     */
    sensePathFromWithin(fromChild, sense, trans, obs, fill)
    {
        if(fromChild == otherSide && otherSide == getPOV()) return;
        else inherited(fromChild, sense, trans, obs, fill);
    }
;

/*--------------------------------------------------------------------------*/
/*
 *   The Window class adds action handling to BasicWindow.
 *   When mixing Window with Openable, make sure to put Window on the left,
 *   or some of the methods won't work properly.
 */
class Window: BasicWindow
    /*
     *   We can be looked through when we're open, we'll display our
     *   lookThroughDesc by itself.
     */
    dobjFor(LookThrough)
    {
        preCond = [objVisible, windowOpen]
        verify() { logicalRank(150, 'is a window'); }
        action() { lookThrough(gActor); }
    }

    /*
     *   We'll do an implicit look through in response to look in
     */
    dobjFor(LookIn) { action() { lookThrough(gActor); } }
;

/*--------------------------------------------------------------------------*/
/*
 *   OpenableWindow, a window that can be opened and closed. BasicWindow
 *   can be opened and closed programmatically, this class is for windows
 *   that open and close in response to player commands.
 */
class OpenableWindow: Window, Openable
    /*
     *   Open/close the window.  If we have a complementary window object
     *   representing the other side, we'll remark in the sensory context
     *   of its location that it's also opening or closing.
     */
    makeOpen(stat)
    {
        /* inherit the default behavior */
        inherited(stat);

        /*
         *   if our new status is in effect, notify the other side so
         *   that it can generate a message in its location
         */
        if (isOpen == stat && otherSide != nil)
            otherSide.noteRemoteOpen(stat);
    }

    /*
     *   Note a "remote" change to our open/close status.  This is an
     *   open/close operation performed on our complementary object
     *   representing the other side of the window.  We'll remark on the
     *   change in the sensory context of this side, but only if we're
     *   suppressing output in the current context - if we're not, then
     *   the player will see the message generated by the side that we
     *   directly acted upon, so we don't need a separate report for the
     *   other side.  
     */
    noteRemoteOpen(stat)
    {
        /* 
         *   If I'm not visible to the player character in the current
         *   sense context, where the action is actually taking place,
         *   switch to my own sensory context and display a report.  This
         *   way, if the player can see this window but not the other side,
         *   and the action is taking place on the other side, we'll still
         *   see a note about the change.  We only need to do this if
         *   we're not already visible to the player, because if we are,
         *   we'll generate an ordinary report of the window opening in the
         *   action's context.  
         */
        if (senseContext.isBlocking)
        {
            /* show a message in my own sensory context */
            callWithSenseContext(self, sight,
                {: libMessages.sayOpenWindowRemotely(self, stat) });
        }
    }

    /*
     *   When the window is opened, implicitly look through it
     */
    dobjFor(Open)
    {
        action()
        {
            /* 
             *   note the effect we have currently, while still closed, on
             *   sensing through the window
             */
            local trans = transSensingThru(sight);

            /* make it open */
            inherited();

            /*
             *   If the actor is outside me, and our sight transparency is
             *   now better than it was before we were open, look through.
             *   Otherwise, just show our default 'opened' message.
             *   
             *   As a special case, if we're running as an implied command
             *   within a LookIn or Search action on this same object,
             *   don't bother showing this result.  Doing so would be
             *   redundant with the explicit examination of the contents
             *   that we'll be doing anyway with the main action.  
             */
            if (!gActor.isIn(self)
                && transparencyCompare(transSensingThru(sight), trans) > 0
                && !(gAction.isImplicit
                     && (gAction.parentAction.ofKind(LookInAction)
                         || gAction.parentAction.ofKind(LookThroughAction)
                         || gAction.parentAction.ofKind(SearchAction))
                     && gAction.parentAction.getDobj() == self))
            {
                lookThrough(gActor);
            }
        }
    }
;

/*--------------------------------------------------------------------------*/
/*
 *   The window open precondition. We use a special precondition so that we
 *   can change the response on failure.
 */
windowOpen: ObjOpenCondition
    conditionFailed(obj)
    {
        /* can't open it implicitly - report failure and give up */
        reportFailure(&cannotSeeThroughClosedWindowMsg, obj);

        /* make it the pronoun */
        gActor.setPronounObj(obj);
    }
;


