#charset "us-ascii"

/* 
 *  Copyright (c) 2001-2004 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of the PROTEUS, the TADS 3 Utility Classes Package
 *
 *  Diagnostic.t
 *
 *  Provides meta-commands for diagnostic purposes.
 */

#include "adv3.h"
#include "tok.h"
#include "en_us.h"
#include "proteus.h"

/*
 *-------------------------------------------------------------------------
 *  DIAGNOSTIC ACTION CLASSES
 *-------------------------------------------------------------------------
 */

 modify Action
 {
     getHref([args])
     {
        local str, pattern;

        pattern = new RexPattern('(@.*)/@<alphanum|-|squote>+(.*)');
        rexMatch(pattern, verbPhrase);
        str = rexGroup(1)[3] + rexGroup(2)[3];

        pattern = new RexPattern('<lparen>(<^rparen>*)<rparen>');
        
        foreach (local repStr in args)
            str = rexReplace(pattern, str, repStr, ReplaceOnce);

        return str;
     }
}

class SystemTAction: SystemAction, TopicAction
{
    execAction()
    {
        sym_ = getDobjFromTopicText();
        if (sym_ == nil)
        {
            "<i><<gTopic.getTopicText()>></i> not found. ";
            exit;
        }
        if (enforceSymType)
        {
            if (dataType(sym_) != enforceSymType)
            {
                "<i><<gTopic.getTopicText()>></i> must be of datatype ";
                switch(enforceSymType)
                {
                    case TypeObject:
                        "TypeObject ";
                        break;

                    case TypeProp:
                        "TypeProp ";
                        break;
                }
                exit;
            }
            if (enforceSymType == TypeObject
            && enforceOfKind
            && !sym_.ofKindOrUU(enforceOfKind))
            {
                "Object <i><<gTopic.getTopicText()>></i> must be  
                ofKindOrUU() <<gSymbols.getSString(enforceOfKind)>>. ";
                exit;
            }
        }

        inherited();
    }

    getDobjFromTopicText()
    {
        local val;
        local str = gTopic.getTopicText();
        
        /* match on symbolic first */
        if (mode_ == nil || mode_ == 'sym')
        {
            val = gSymbols.getSymbol(str);
            if (val != nil)
                return val;
        }

        /* match on vocabulary next */
        if (mode_ == nil || mode_ == 'voc')
        {
            val = gTopic.getBestMatch();
            if (val && !val.ofKindOrUU(ResolvedTopic))
                return val;
        }

        /* match on tag last */
        if (mode_ == nil || mode_ == 'tag')
        {
            val = str.getRefTagObj();
            if (val != nil)
                return val;
        }

        return nil;
    }

    /*
     *  We return an empty list for 'all' so that the XxxxAll version
     *  of the command is selected instead of the XxxxObj version. This
     *  should NOT be overriden by subclasses.
     */
    getAllDobj(actor, scope) { return []; }

    // subclasses must define this method
    execSystemAction() {}

    sym_ = nil

    /* these actions consume no game time */
    actionTime = 0

    /* this is a diagnostic action */
    isDiagnosticAction = true

    /* default enforcement of sym_ type to TypeObject */
    enforceSymType  = TypeObject
    enforceOfKind   = Object
}

/*
 *-------------------------------------------------------------------------
 *  DIAGNOSTIC ACTIONS
 *-------------------------------------------------------------------------
 */

/* 
 *  An Action class for going to the location for the object that was 
 *  specified.
 */
DefineSystemTAction(ZgotoObj)
    execSystemAction()
    {
        gAHrefMetaCmd = self;

        local newLocOutermostRoom;

        /*
         *  If the destination isn't an object derived from Thing we
         *  don't want to travel to it.
         */
        if (sym_.isClass() 
        || !sym_.ofKindOrUU(Thing))
        {
            "This is not something {you/he} can travel to. ";
            return;
        }

        local loc = sym_;

        while (loc != nil && !loc.ofKindOrUU(BasicLocation))
        {
            loc = loc.location;
        }

        if (loc == nil)
        {
            "This object has no location. ";
            return;
        }

        newLocOutermostRoom = loc.getOutermostRoom();

        /*
         *  If the actor is already in the destination's outmost
         *  room then inform the player.
         */
        if (gActor.location == sym_)
        {
            "{You/he} {is} already there. ";
            return;
        }

        if (!sym_.ofKindOrUU(BasicLocation) 
            && gActor.location.getOutermostRoom == newLocOutermostRoom)
        {
            "{You/he} {is} already in the location containing this
            object. ";
            return;
        }

        /*
         *  Get the actor out of any nested locations and put him
         *  in the outermost room of his location. This attempts to
         *  sort out the actor's posture.
         */
        while (gActor.location != gActor.location.getOutermostRoom())
            nestedAction(GetOutOf, gActor.location...);      

        /* 
         *  Move the actor to the outermost room of the destination.
         */
        gActor.travelTo(newLocOutermostRoom, 
            newLocOutermostRoom, gActor.location);

        if (sym_.ofKindOrUU(BasicLocation) && sym_ != newLocOutermostRoom)
            nestedAction(Board, sym_...);      

        return;
    }
    enforceOfKind = Thing
;

/* 
 *  An Action class for listing the ancestors for the object that was 
 *  specified.
 */
DefineSystemTAction(ZlistAncestorObj)
    execSystemAction()
    {
        local lst;

        gAHrefMetaCmd = self;

        lst = sym_.getAncestorList(ObjAll);

        "\nPlease wait... processing <<lst.length()>> entries ";
        flushOutput();

        if (lst.length() == 0)
            "No ancestor for <<sym_.getObjTagAndSym()>>. ";
        else
        {
            "<ul>";
            for (local i = 1; i <= lst.length(); ++i)
            {
                "<li><<lst[i].isClass() ? "class" : "object">> 
                <<lst[i].mapObjRefAHrefToSString()>></li>";
            }
            "</ul>";
        }
    }
;

/* 
 *  An Action class for listing the children for the object that was 
 *  specified using the object's vocabulary.
 */
DefineSystemTAction(ZlistChildObj)
    listLimit   = 300
    execSystemAction()
    {
        local lst;

        gAHrefMetaCmd = self;

        lst = sym_.getChildList(ObjAll);

        "\nPlease wait... processing <<lst.length()>> entries ";
        flushOutput();

        if (lst.length() == 0)
            "\nNo child for <<sym_.getObjTagAndSym()>>. ";
        else
        {
            "<ul>";
            for (local i = 1; i <= lst.length() && i <= listLimit; ++i)
            {
                if (i%50 == 0)
                    flushOutput();

                "<li><<lst[i].isClass() ? "class" : "object">> 
                <<lst[i].mapToSString(MapObjReference, MapObjRefAHref)>></li>";
                // <<lst[i].mapObjRefAHrefToSString()>></li>";
            }
            "</ul>";
        }
    }
;

/* 
 *  An Action class for listing the descendants for the object that was 
 *  specified.
 */
DefineSystemTAction(ZlistDescendantObj)
    execSystemAction()
    {
        local lst;

        gAHrefMetaCmd = self;

        lst = sym_.getDescendantList(ObjAll);

        "\nPlease wait... processing <<lst.length()>> entries ";
        flushOutput();

        if (lst.length() == 0)
            "\nNo descendant for <<sym_.getObjTagAndSym()>>. ";
        else
        {
            "<ul>";
            for (local i = 1; i <= lst.length(); ++i)
            {
                "<li><<lst[i].isClass() ? "class" : "object">> 
                <<lst[i].mapObjRefAHrefToSString()>></li>";
            }
            "</ul>";
        }
    }
;

/* 
 *  An Action class for listing the first named ancestors for 
 *  the object that was specified.
 */
DefineSystemTAction(ZlistFirstNamedAncestorObj)
    execSystemAction()
    {
        local lst;

        gAHrefMetaCmd = self;

        lst = sym_.getFirstNamedAncestorList();

        "\nPlease wait... processing <<lst.length()>> entries ";
        flushOutput();

        if (lst.length() == 0)
            "No first named ancestor for <<sym_.getObjTagAndSym()>>. ";
        else
        {
            "<ul>";
            for (local i = 1; i <= lst.length(); ++i)
            {
                "<li><<lst[i].isClass() ? "class" : "object">> 
                <<lst[i].mapObjRefAHrefToSString()>></li>";
            }
            "</ul>";
        }
    }
;

/* 
 *  An Action class for listing the first named descendants for 
 *  the object that was specified.
 */
DefineSystemTAction(ZlistFirstNamedDescendantObj)
    execSystemAction()
    {
        local lst;

        gAHrefMetaCmd = 'ZshowDirObj'.getPredicate();

        lst = sym_.getFirstNamedDescendantList();

        "\nPlease wait... processing <<lst.length()>> entries ";
        flushOutput();

        if (lst.length() == 0)
            "\nNo first named descendant for <<sym_.getObjTagAndSym()>>. ";
        else
        {
            "<ul>";
            for (local i = 1; i <= lst.length(); ++i)
            {
                "<li><<lst[i].isClass() ? "class" : "object">> 
                <<lst[i].mapObjRefAHrefToSString()>></li>";
            }
            "</ul>";
        }
    }
;

/* 
 *  An Action class for listing the modification objects for 
 *  the object that was specified.
 */
DefineSystemTAction(ZlistModObj)
    execSystemAction()
    {
        local lst;

        gAHrefMetaCmd = 'ZshowDirObj'.getPredicate();

        lst = sym_.getObjModList();

        "\nPlease wait... processing <<lst.length()>> entries ";
        flushOutput();

        if (lst.length() == 0)
            "\nNo object found for <<sym_.getObjTagAndSym()>>. ";
        else
        {
            "<ul>";
            for (local i = 1; i <= lst.length(); ++i)
            {
                "<li><<lst[i].isClass() ? "class" : "object">> 
                <<lst[i].mapObjRefAHrefToSString()>></li>";
            }
            "</ul>";
        }
    }
;

/* 
 *  An Action class for listing the parents for the object that was 
 *  specified.
 */
DefineSystemTAction(ZlistParentObj)
    execSystemAction()
    {
        local lst;

        gAHrefMetaCmd = self;

        lst = sym_.getParentList(ObjAll);

        "\nPlease wait... processing <<lst.length()>> entries ";
        flushOutput();

        if (lst.length() == 0)
            "\nNo parent for <<sym_.getObjTagAndSym()>>. ";
        else
        {
            "<ul>";
            for (local i = 1; i <= lst.length(); ++i)
            {
                "<li><<lst[i].isClass() ? "class" : "object">> 
                <<lst[i].mapObjRefAHrefToSString()>></li>";
            }
            "</ul>";
        }
    }
;

/* 
 *  An Action class for purloining the object that was 
 *  specified using the object's vocabulary.
 */
DefineSystemTAction(ZpurloinObj)
    execSystemAction()
    {
        gAHrefMetaCmd = self;

        /*
         *  If the destination isn't an object derived from Thing we
         *  don't want to travel to it.
         */
        if (sym_.isClass() 
        || !sym_.ofKindOrUU(Thing)
        || sym_.ofKindOrUU(NonPortable))
        {
            "This is not something {you/he} can purloin. ";
        }
        else if (sym_ == gActor)
        {
            "This is not something {you/he} can purloin. ";
        }
        else if (sym_.ofKindOrUU(BasicLocation) && gActor.isIn(sym_))
        {
            "This is not something {you/he} can purloin. ";
        }
        else if (sym_.isIn(gActor))
        {
            "{You/he} already {have} <<sym_.theNameObj>>. ";
        }
        else
        {
            "{You/he} purloin{s} <<sym_.theNameObj>>. ";
            sym_.moveInto(gActor);
        }
    }
;

/* 
 *  An Action class for restoring the state of all 
 *  Thing class instances
 */
class ZrestoreAllAction: SystemAction
{
    execSystemAction()
    {
        "Please wait... restoring object states...\n";
        flushOutput();

        restoreStateAll(Thing, ObjInstances);
    }
}

/* 
 *  An Action class for saving the state of an object that was 
 *  specified using the object's vocabulary.
 */
DefineSystemTAction(ZrestoreObj)
    execSystemAction()
    {
        sym_.restoreState();
        "State restored for object <<sym_.getObjTagAndSym()>>. ";
    }
;

/* 
 *  An Action class for saving the state of all 
 *  Thing class instances
 */
class ZsaveAllAction: SystemAction
{
    execSystemAction()
    {
        "Please wait... saving object states...\n";
        flushOutput();

        saveStateAll(Thing, ObjInstances);
    }
}

/* 
 *  An Action class for saving the state of an object that was 
 *  specified using the object's vocabulary.
 */
DefineSystemTAction(ZsaveObj)
    execSystemAction()
    {
            sym_.saveState();
            "State saved for object <<sym_.getObjTagAndSym()>>. ";
    }
;

/* 
 *  An Action class for showing the action definition of an object 
 *  that was specified using the object's vocabulary.
 */
DefineSystemTAction(ZshowActObj)
    execSystemAction()
    {
        gAHrefMetaCmd = self;

        "\nPlease wait... processing object action methods...\b";
        flushOutput();

        sym_.displayStructure(new function(obj, prop) {
            local str;

            str = String.toSString(prop);

            if (obj.propType(prop) not in (TypeDString, TypeCode,
                    TypeNativeCode))
                return nil;
            else if (!str.startsWith('&preCondDobj')
                && !str.startsWith('&preCondIobj')
                && !str.startsWith('&verifyDobj')
                && !str.startsWith('&verifyIobj')
                && !str.startsWith('&remapDobj')
                && !str.startsWith('&remapIobj')
                && !str.startsWith('&checkDobj')
                && !str.startsWith('&checkIobj')
                && !str.startsWith('&actionDobj')
                && !str.startsWith('&actionIobj'))
                return nil;
            else return true;
        }, true);
    }
;

/* 
 *  An Action class for showing the object definition of an object 
 *  that was specified using the object's vocabulary.
 */
DefineSystemTAction(ZshowDefObj)
    execSystemAction()
    {
        gAHrefMetaCmd = self;

        "\nPlease wait... processing: defined properties\b ";
        flushOutput();

        sym_.displayStructure(new function(obj, prop) {
            if (obj == sym_.propDefined(prop, PropDefGetClass))
                return true;
            else return nil;
        }, true);
    }
;

/* 
 *  An Action class for showing the state deltas of all 
 *  Thing class instances
 */
class ZshowDeltaAllAction: SystemAction
{
    execSystemAction()
    {
        local obj, oldState, snapshotsFound;

        gAHrefMetaCmd = 'ZshowDirObj'.getPredicate();

        "\nPlease wait... processing object state deltas\b ";
        flushOutput();

        obj = firstObj(Thing, ObjInstances);
        while (obj != nil)
        {
            if (obj.hasMoreSnapshots())
            {
                snapshotsFound = true;
                oldState = obj.getSnapshot();
                oldState.displayStateDelta(obj, true);
            }

            obj = nextObj(obj, Thing, ObjInstances);
        }

        if (!snapshotsFound)
            "No snapshots found for object state comparisons. ";
    }
}

/* 
 *  An Action class for showing the state delta of an object 
 *  that was specified using the object's vocabulary.
 */
DefineSystemTAction(ZshowDeltaObj)
    execSystemAction()
    {
        local oldState;

        gAHrefMetaCmd = 'ZshowDirObj'.getPredicate();

        "\nPlease wait... processing object state delta\b ";
        flushOutput();

        if (sym_.hasMoreSnapshots())
        {
            oldState = sym_.getSnapshot();
            oldState.displayStateDelta(sym_, true);
        }
        else
            "No snapshots found for object <<sym_.getObjTagAndSym()>> 
            state comparisons. ";
    }
;

/* 
 *  An Action class for showing the object definition of an object 
 *  that was specified using the object's vocabulary.
 */
DefineSystemTAction(ZshowDirObj)
    execSystemAction()
    {
        gAHrefMetaCmd = self;

        "\nPlease wait... processing: directly-defined properties\b ";
        flushOutput();

        sym_.displayStructure(new function(obj, prop) {
            if (sym_ == obj 
                && sym_.propDesc(prop, PropDescDefined) == PropDefDirectly)
                return true;
            else return nil;
        }, true);
    }
;

/* 
 *  An Action class for showing the object property of an object 
 *  that was specified using the object's vocabulary.
 */
DefineSystemTAction(ZshowPropObj)
    execSystemAction()
    {
        local prop, val;

        gAHrefMetaCmd = 'ZshowDirObj'.getPredicate();

        prop = gSymbols.getSymbol(gLiteral);
        if (prop == nil || dataType(prop) != TypeProp)
        {
            "\n<i><<gLiteral>></i> is not a property. ";
            return;
        }

        if (sym_.propType(prop) is in (TypeCode, TypeNativeCode))
        {
            "\n<i><<gLiteral>></i> is code and not executable
            by this command. ";
            return;
        }

        val = sym_.(prop)();

        "\nPlease wait... processing: object 
        <i><<gTopic.getOrigText()>></i> 
        property <i>&<<text_>></i> ";
        flushOutput();

        if (dataType(val) == TypeObject)
            "\n<<val.displayStructure(new function(obj, prop) {
                if (sym_ == obj && sym_.propDefined(prop, PropDefDirectly))
                    return true;
                else return nil;
            }, true)>> ";
        "\n<b>Value</b>\n{\n<<String.toSString(val, MapObjValue, MapObjRefAHref)>>\n} ";
    }

    /* get the current literal text */
    getLiteral() { return text_; }

    resolveNouns(issuingActor, targetActor, results)
    {
        inherited(issuingActor, targetActor, results);

        /* 
         *   "Resolve" our literal phrase.  The literal phrase doesn't
         *   resolve to an object list the way a regular noun phrase would,
         *   but rather just resolves to a text string giving the original
         *   literal contents of the phrase.  
         */
        literalMatch.resolveLiteral(results);

        /* retrieve the text of the phrase, exactly as the player typed it */
        text_ = literalMatch.getLiteralText(results, self, DirectObject);    
    }
;

/* 
 *  An Action class for showing the object state of an object 
 *  that was specified using the object's vocabulary.
 */
DefineSystemTAction(ZshowStateObj)
    execSystemAction()
    {
        gAHrefMetaCmd = self;

        "\nPlease wait... processing: state properties ";
        flushOutput();

        sym_.displayStructure(new function(obj, prop) {
            if (obj.propType(prop) is in (TypeDString, TypeCode,
                    TypeNativeCode))
                return nil;
            else if (obj.propType(prop) is in (TypeNil, nil)
                && dataType(prop) == TypeProp)
                return nil;
            else if (obj != sym_.propDefined(prop, PropDefGetClass))
                return nil;
            else return true;
        }, true);
    }
;

/* 
 *  An Action class for showing the object non-static definition of 
 *  an object that was specified using the object's vocabulary.
 */
DefineSystemTAction(ZshowNonStaticObj)
    execSystemAction()
    {
        gAHrefMetaCmd = self;

        "\nPlease wait... processing: non-statically defined properties\b ";
        flushOutput();

        sym_.displayStructure(new function(obj, prop) {
            if (sym_ == obj 
            && sym_.propDesc(prop, PropDescDefined) == PropDefNonStatic)
                return true;
            else return nil;
        }, true);
    }
;

/* 
 *  An Action class for treeing the ancestor of an object 
 *  that was specified using the object's vocabulary.
 */
DefineSystemTAction(ZtreeAncestorObj)
    lst_ = []
    execSystemAction()
    {
        gAHrefMetaCmd = 'ZshowDirObj'.getPredicate();

        if (dataType(sym_) != TypeObject)
        {
            "This entity does not extend Object. ";
            return;
        }

        lst_ = sym_.getAncestorList();
        lst_ += sym_;

        "\nPlease wait... processing <<lst_.length()>> entries ";
        flushOutput();

        "\b<<treeIter(Object)>>";
    }
    treeIter(obj)
    {
        local dList;

        "<ul>";

        "<li><<obj.mapObjRefAHrefToSString()>>";

        dList = obj.getChildList(ObjAll);

        dList = screenForObjKind(dList);

        foreach (local o in dList)
            treeIter(o);

        "</ul>";
    }
    screenForObjKind(oList)
    {
        foreach (local o in oList)
        {
            if (!sym_.ofKindOrUU(o))
                oList -= o;
        }
        
        return oList;
    }
;

class ZtreeContAllAction: SystemAction
{
    execSystemAction()
    {
        gAHrefMetaCmd = self;

        "\nPlease wait... processing containment hierarchy for all ";
        flushOutput();

        "\bnil";
        for (local obj = firstObj(Thing); obj != nil; obj = nextObj(obj, Thing))
        {
            if (obj.location == nil)
                "<<treeIter(obj)>>";
        }
    }
    treeIter(obj)
    {
        "<ul>";

        "<li><<obj.mapObjRefAHrefToSString()>>";

        foreach (local o in obj.contents)
            treeIter(o);

        "</ul>";
    }
}

/* 
 *  An Action class for showing the containment tree of an object 
 *  that was specified using the object's vocabulary.
 */
DefineSystemTAction(ZtreeContObj)
    execSystemAction()
    {
        local loc;

        gAHrefMetaCmd = self;

        "\nPlease wait... processing containment hierarchy for object ";
        flushOutput();

        loc = sym_.location;
        if (loc)
            "\b<<loc.mapObjRefAHrefToSString()>>";
        else
            "\bnil";

        "<<treeIter(sym_)>>";
    }
    treeIter(obj)
    {
        "<ul>";

        "<li><<obj.mapObjRefAHrefToSString()>>";

        foreach (local o in obj.contents)
            treeIter(o);

        "</ul>";
    }
    enforceOfKind = Thing
;

/* 
 *  An Action class for treeing the definer of an object method 
 *  that was specified.
 */
DefineSystemTAction(ZtreeDefinerObj)
    defList = nil

    execSystemAction()
    {
        local stripList;
        
        defList = new Vector(1000);

        gAHrefMetaCmd = 'ZshowDirObj'.getPredicate();

        if (Object.propDesc(sym_, PropDescDefined) is in (PropDefDirectly, PropDefNonStatic))
            defList += Object;

        for (local o = firstObj(Object, ObjAll); o != nil; o = nextObj(o, Object, ObjAll))
            if (o.propDesc(sym_, PropDescDefined) is in (PropDefDirectly, PropDefNonStatic))            
                defList += o;

        "\nPlease wait... processing <<defList.length()>> entries ";
        flushOutput();

        stripList = childStripper(defList);

        foreach (local obj in stripList)
            "<<treeIter(obj)>>";
    }
    childStripper(lst)
    {
        foreach (local obj1 in lst)
        {
            foreach (local obj2 in lst)
            {
                if (obj2.ofKindOrUU(obj1) && obj2 != obj1)
                    lst -= obj2;
            }
        }

        return lst;
    }
    treeIter(obj)
    {
        local childList;

        "<ul>";

        if (obj.propDesc(sym_, PropDescDefined) is in (PropDefDirectly, PropDefNonStatic)
        && defList.indexOf(obj) != nil)
        {
            "<li><<obj.mapObjRefAHrefToSString()>>";
            defList -= obj;
        }

        childList = obj.getChildList(ObjAll);

        foreach (local o in childList)
            treeIter(o);

        "</ul>";
    }
    enforceSymType = TypeProp
;

/* 
 *  An Action class for treeing the descendant of an object 
 *  that was specified using the object's vocabulary.
 */
DefineSystemTAction(ZtreeDescendantObj)
    execSystemAction()
    {
        gAHrefMetaCmd = self;

        "\nPlease wait... processing descendants for object ";
        flushOutput();

        "<<treeIter(sym_)>>";
    }
    treeIter(obj)
    {
        local dList;

        "<ul>";

        "<li><<obj.mapObjRefAHrefToSString()>>";

        dList = obj.getChildList(ObjAll);

        foreach (local o in dList)
            treeIter(o);

        "</ul>";
    }
    enforceOfKind = TadsObject
;

/* 
 *  An Action class for treeing the modifications of an object 
 *  that was specified using the object's vocabulary.
 */
DefineSystemTAction(ZtreeModObj)
    lst_ = []
    execSystemAction()
    {
        local obj; 

        gAHrefMetaCmd = 'ZshowDirObj'.getPredicate();

        lst_ = sym_.getObjModList();

        if (lst_.length() == 1)
        {
            "This object has no modifications ";
            return;
        }

        "\nPlease wait... processing <<lst_.length()>> entries ";
        flushOutput();

        obj = lst_[lst_.length()];

        "\b<<treeIter(obj)>>";
    }
    treeIter(obj)
    {
        local dList;

        "<ul>";

        "<li><<obj.mapObjRefAHrefToSString()>>";

        dList = obj.getChildList(ObjAll);

        dList = dList.intersect(lst_);

        foreach (local o in dList)
            treeIter(o);

        "</ul>";
    }
;