#charset "us-ascii"

/*
 *  Copyright (c) 2001-2004 by Kevin Forchione. All rights reserved.
 *  Based on code provided by Michael J. Roberts.
 *
 *  This file is part of the PROTEUS, the TADS 3 Utility Classes Package
 *
 *  StackTraceInfo.t
 *
 *  Provides a basic implementation for retrieving T3 Stack Info with
 *  regards to command, predicate, and action information.
 */

/* include the Proteus header */
#include "proteus.h"

class StackTraceInfo: object
{
    stack_          = nil
    commandList     = []
    predicateList   = []
    actionClassList = []

    hasDiagnosticAction()
    {
        foreach (local action in actionClassList)
            if (action.isDiagnosticAction)
                return true;

        return nil;
    }

    getStackCommandList()
    {
        local lst = [];

        /* We work from the end of the stack trace list */
        for (local i = stack_.length(); i > 0; i--)
        {
            if (stack_[i].hasCommand())
                lst = lst.appendUnique([stack_[i].self_]);
        }

        return lst;
    }

    getStackPredicateList()
    {
        local lst = [];

        /* We work from the end of the stack trace list */
        for (local i = stack_.length(); i > 0; i--)
        {
            if (stack_[i].hasPredicate())
                lst = lst.appendUnique([stack_[i].self_]);
        }

        return lst;
    }

    getStackActionClassList()
    {
        local lst = [];

        /* We work from the end of the stack trace list */
        for (local i = stack_.length(); i > 0; i--)
        {
            if (stack_[i].hasPredicate())
                lst = lst.appendUnique([stack_[i].getActionClass()]);
        }

        return lst;
    }

    construct()
    {
        stack_          = t3GetStackTrace();
        commandList     = getStackCommandList();
        predicateList   = getStackPredicateList();
        actionClassList = getStackActionClassList();
    }
}

modify T3StackInfo
{
    hasCommand()
    {
        local obj, str;

        if (self_ == nil || !self_.ofKindOrUU(Object))
            return nil;

        obj = self_.getFirstNamedAncestorList().car();

        if (obj == nil)
            return nil;

        str = gSymbols.getSString(obj);

        if (str == nil || !str.find('ommandPhrase'))
            return nil;

        return true;
    }

    hasPredicate()
    {
        local obj, str;

        if (self_ == nil || !self_.ofKindOrUU(Object))
            return nil;

        obj = self_.getFirstNamedAncestorList().car();

        if (obj == nil)
            return nil;

        str = gSymbols.getSString(obj);

        if (str == nil || !str.startsWith('predicate'))
            return nil;

        return true;
    }

    getActionClass()
    {
        local cls;

        if (!hasPredicate())
            return nil;

        cls = self_.getFirstNamedAncestorList().car();

        return cls.getSuperclassList().car();
    }
}