#charset "us-ascii"

/* 
 *  Copyright (c) 2001-2004 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of PROTEUS, the TADS 3 Utility Classes Package
 *
 *  Proxy.t
 *
 *  Implements a basic ProxyService / SeviceClass pattern.
 */

/* include the proteus lookup table header */
#include "proteus.h"
#include "adv3.h"

/*
 *  Define the propNotDefined property and export it so that the VM can
 *  call this property when a call is made to an undefined property.
 */
property propNotDefined;
export propNotDefined;

class ServiceLU: ProteusLU
{
    /* lookup table custom props for Service */
    luCustomProps(Service)
    
    instantiate()
    {
        return new ServiceLU();
    }

    construct()
    {
        inherited();
        gServices = self;
    }

    addService(proxy, service)
    {
        storeValue(proxy, service);
    }
    
    getServices(proxy)
    {
        return getValues(proxy);
    }

    /*
     *  Modified to call the ProxySerivce's initializeForService()
     *  method.
     */
    servicePropertyIter(obj, objFunc)
    {
        local val, propList = [];

        /* If there's no service suffix, there's no nested services */
        if (myServiceSuffix == nil)
            return;

        /* get a list of properties for this object */
        propList = obj.getInherPropList();

        /* 
          *   Strip out only properties that are suffixed with
          *   myServicesSuffix.
          */
        propList = propList.subset(new function(prop) {
            local str;
            
            str = String.toSString(prop);
            return str.endsWith(myServiceSuffix);
        });
        
        /*
          *   Iterate over each property in the list. For those 
          *   properties that have object property types execute
          *   the objFunc callback function. 
          */
        foreach (local prop in propList)
        {
            if (obj.propType(prop) != TypeObject)
                continue;
            
            val = obj.(prop);

            obj.initializeForService(val);
            
            objFunc(obj,val);
        }
    }
}

/*
 *  This class acts as a proxy for service classes. When a message is
 *  sent to a property that is not defined by the object inheriting this
 *  class the message is redirected to the appropriate service class.
 */
class ProxyService: Morphic
{
    remapPropStr = 'remap([ID])obj(<Alpha>+)'

    /*
     *  A list of this object's services that are not directly defined
     *  as nested classes of this object.
     */
    myServices = []
    
    /*
     *  A ProxyService implements an interface when any of its services
     *  implements the interface.
     */
    implementsIF(interface)
    {
        local svcList;
        
        if (inherited(interface))
            return true;
            
        svcList = getServices();
        foreach (local service in svcList)
            if (service.implementsIF(interface))
                return true;
                
        return nil;
    }
    
    /*
     *  Add a service to this object.
     */
    addService(service) 
    {
        gServices.addService(self, service);
    }
    
    /*
     *  Return the services for this object.
     */
    getServices()
    {
        return gServices.getServices(self);
    }

    /*
     *  This method is called when any undefined properties are
     *  called on this object. We first determines if a service
     *  object capable of completing the action exists for this 
     *  ProxyService. If none are found, we delegate to the default
     *  handler. Otherwise we remap the action to the new target 
     *  object.
     */
    propNotDefined(prop, [args])
    {
        local remapToList;

        /* build the remap list for this property */
        remapToList = buildRemapToList(prop);

        /* 
         *  If the remapList is empty we delegate the property
         *  call to the default handler. Otherwise we return 
         *  the appropriate element of the remap list so that
         *  the parser can remap the action to the new target
         *  object.
         */
        if (remapToList.length() == 0)
            return delegated (getHandler()).(prop)(args...);
        else
            return remapToList;
    }

    /*
     *  Builds a remapTo list consisting of 2 elements, the first element
     *  is the new target object reference, the second element is the 
     *  remapTo replacement list. 
     *
     *  If neither of these elements can be constructed, an empty list
     *  is returned.
     */
    buildRemapToList(prop)
    {
        local remapList = [], actionInfoList;
        local actionClass, actionProp, objType, newTargetObjRef; 

        /* 
         *  Get the action information elements available for 
         *  this property. If an empty list was returned we 
         *  return an empty list to the caller.
         */
        actionInfoList = getActionInfoList(prop);
        if (actionInfoList.length() == 0)
            return remapList;

        actionClass = actionInfoList[1];
        actionProp  = actionInfoList[2];
        objType     = actionInfoList[3];

        /*
         *  Get the new target object reference for the action
         *  property provided by getActionInfoList(). The new
         *  target object is the one best suited to complete the
         *  action. If no target object was deteremined, return 
         *  an empty list.
         */
        newTargetObjRef = getService(actionProp);
        if (newTargetObjRef == nil)
            return remapList;

        return formatRemapToList(actionClass, 
            objType, newTargetObjRef);
    }

    /*
     *  Method returns a list consisting of [actionClass, actionProp, 
     *  objType]. If these cannot be determined then an empty list
     *  is returned.
     */
    getActionInfoList(prop)
    {
        local actionInfoList = [];
        local propStr, ret, actionClass;

        if (gSymbols == nil)
            return actionInfoList;

        propStr = String.toSString(prop).substr(2);
        if (propStr == '')
            return actionInfoList;
        
        ret = rexSearch(remapPropStr, propStr);
        if (ret == nil)
            return actionInfoList;
        
        actionClass = gSymbols.getSymbol(rexGroup(2)[3] + 'Action');
        if (actionClass == nil)
            return actionInfoList;

        actionInfoList += actionClass;

        if (rexGroup(1)[3] == 'D')
        {
            actionInfoList += gAction.actionDobjProp;
            actionInfoList += DirectObject;
        }
        else
        {
            actionInfoList += gAction.actionIobjProp;
            actionInfoList += IndirectObject;
        }

        return actionInfoList;
    }

    /*
     *  We retrieve the services for this object, and determine
     *  if any one of them is appropriate as a redirect for this 
     *  action property. If none are found to be appropriate, 
     *  we return nil.
     */
    getService(actionProp)
    {
        local svcList = [];

        /* 
          *   If the serviceLU hasn't been created then this
          *   object hasn't any services registered to handle
          *   the message, so we return nil.
          */
        if (gServices == nil)
            return nil;

        /*
         *  If there are no services for this object we return nil.
         */
        if (!gServices.hasMoreValues(self))
            return nil;

        svcList = getServices();

        /*
         *  Build a list of those services that define the property.
         */
        //svcList = svcList.subset({x : x.propDefined(actionProp)});
        foreach (local service in svcList)
        {
            if (service.propDefined(actionProp) == nil)
                svcList -= service;
        }
        
        /*
         *  If no services exist for this property, or more than
         *  one service satisfies the action, we return nil.
         */
        if (svcList.length() != 1)
            return nil;
        
        /* return the first service in the list */
        return svcList.car();
    }

    /*
     *  Returns the formatted remapTo List replacement for the 
     *  actionClass / newTargetObjRef based on the class of the 
     *  action (TIAction, TAction, etc) and the object type 
     *  of the newTargetObjRef (DirectObject or IndirectObject).
     */
    formatRemapToList(actionClass, objType, newTargetObjRef)
    {
        local remapToList = [];

        remapToList += actionClass;

        if (actionClass.ofKindOrUU(TIAction))

            if (objType == DirectObject)
                remapToList += [newTargetObjRef, IndirectObject];
            else
                remapToList += [DirectObject, newTargetObjRef];

        else if (actionClass.ofKindOrUU(TopicAction))

            if (objType == DirectObject)
                remapToList += [newTargetObjRef, IndirectObject];
            else
                remapToList += [DirectObject, newTargetObjRef];

        else if (actionClass.ofKindOrUU(LiteralAction))

            if (objType == DirectObject)
                remapToList += [newTargetObjRef, IndirectObject];
            else
                remapToList += [DirectObject, newTargetObjRef];

        else if (actionClass.ofKindOrUU(TopicAction))

            if (objType == DirectObject)
                remapToList += [newTargetObjRef, IndirectObject];
            else
                remapToList += [DirectObject, newTargetObjRef];

        else if (actionClass.ofKindOrUU(TAction))

            remapToList += newTargetObjRef;

        return remapToList;
    }

    /* 
     *  Perform any necessary initializations for this service. By 
     *  default we simply move the service object into the contents
     *  of the proxyService, making it a component.
     */
    initializeForService(serviceObj)
    {
        serviceObj.moveInto(self);
    }
}