#charset "us-ascii"
/* 
 *  Copyright (c) 2005 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of the TADS 3 Services Pack
 *   
 *  tsp_collection.t
 *
 *  The modifications to the collection subclasses in this module
 *  attempt to provide convenient and common functions.
 */

#include "tads.h"
#include "t3.h"
#include "vector.h"
#include "lookup.h"
#include "tsp_mod_registry.h"

RegisterModule

modify List
{
    /*
     *  This works like append(val), except that if val is a List or 
     *  Vector, each element of val is individually appended to the 
     *  the list value.  This method returns a new list.
     */
    appendAll(val)
    {
        local vec; 
        vec = new Vector(length(), self);
        vec.appendAll(val);
        return vec.toList();
    }

    /*
     *  Deletes one or more elements from the list; each list element 
     *  whose value equals val is removed from the list.  This reduces 
     *  the length of the list by the number of elements removed.  
     *  If there is no element of the list whose value equals val, 
     *  the list is unchanged.  Returns a new list.
     */
    removeElement(val)
    {
        return self - val;
    }

    /*
     *  Returns the first element of the list.  If the list 
     *  has no elements, returns nil.
     */
    firstElm()
    {
        if (length())
            return self[1];
        else return nil;
    }

    /*
     *  Returns the last element of the list.  If the list 
     *  has no elements, returns nil.
     */
    lastElm()
    {
        if (length())
            return self[length()];
        else return nil;
    }

    /*
     *  Returns the "tail" of the list; that is, the rest of the list 
     *  after removing the first element.  If the list has no elements, 
     *  returns nil.  This function is almost the same as sublist(2), 
     *  except that sublist() would return an empty list if given an 
     *  empty list, whereas cdr() returns nil in this case.
     */
    rest()
    {
        if (length())
            return sublist(2);
        else return nil;
    }

    /*
     *  Returns a new list consisting of the intersection 
     *  of this list and coll; that is, a list consisting 
     *  of the elements common to both this list and coll.  
     *  coll must also be either a vector or a list.  If the 
     *  two collections have no elements in common, the result 
     *  is an empty list.  If an element of the shorter collection
     *  (or, if the collections are of equal length, this collection) 
     *  appears more than once in the shorter collection, and that 
     *  element value also appears in the longer collection, then 
     *  the element will be in the result collection the same number 
     *  of times that it is in the shorter collection.  An element 
     *  repeated in the longer collection will not be repeated in 
     *  the result collection.
     *
     *  Provided for polymorphism with Vector class objects.
     */
    intersectColl(coll)
    {
        local lst2;
        if (dataType(coll) != TypeList)
            lst2 = coll.toList();
        else lst2 = coll;
        return intersect(lst2);
    }

    /*
     *  Identical to sublist. 
     *  Provided for polymorphism with Vector class objects.
     */
    subcollection(start, [len]) { return sublist(start, len...); }

    /*
     *  Provided for polymorphism with Vector class objects.
     */
    toList([args])
    {
        local start, count;

        switch (args.length())
        {
            case 0:
                return self;

            case 1:
                start = args[1];
                count = length() - start + 1;
                break;

            default:
                start = args[1];
                count = args[2];
        }
        return subcollection(start, count);
    }

    /*
     *  Creates and returns a new vector based on the list.  
     *  With no arguments, the new vector has the same number 
     *  of elements as the original lst, and each element of 
     *  the vector is a copy of the corresponding element of 
     *  the list.  If start is specified, it gives the starting 
     *  index in the list for the vector; elements of the list 
     *  before start are not included in the vector.  If count 
     *  is specified, it indicates the number of elements of 
     *  the list, starting at start, to copy into the vector.
     */
    toVector([args])
    {
        local start, count, coll;

        switch (args.length())
        {
            case 0:
                return new Vector(length(), self);

            case 1:
                start = args[1];
                count = length() - start + 1;
                break;

            default:
                start = args[1];
                count = args[2];
        }
        coll = subcollection(start, count);
        return new Vector(coll.length(), coll);
    }
}

modify Vector
{
    /* 
     *  Copies values from a list or from another list or vector into this vector.  
     *  This function doesn't create a new vector, but simply modifies entries 
     *  in the 'self' vector.  source is the source of the values; it must be 
     *  either a vector or a list.  sourceStart is an index into source, and 
     *  specifies the first element of source that is to be copied.  destStart 
     *  is an index into the 'self' vector, and specifies the first element of 
     *  the vector that is to be modified.  count is the number of elements to 
     *  modify.  The method copies elements from source into the 'self' vector, 
     *  one at a time, until it reaches the last element of source, or has copied 
     *  the number of elements specified by count.
     */
    copyElmFrom(source, [args])
    {
        local len, sourceStart, destStart, count;

        len = source.length();

        if (args.length() > 0)
            sourceStart = args[1];
        else sourceStart = 1;

        if (args.length() > 1)
            destStart = args[2];
        else destStart = 1;

        if (args.length() > 2)
            count = args[3];
        else count = len;
        
        if (sourceStart < 1)
        {
            sourceStart = 1;
            count       = 0;
        }

        if (destStart < 1)
            destStart = 1;

        if (count < 0)
            count = 0;
        if (count + sourceStart > len)
            count = len - sourceStart + 1;

        for (local i = 0 ; i < count ; ++i)
            self[destStart + i] = source[sourceStart + i];
    }

    /*
     *  Copies the elements of this vector
     *  to a new instance of Vector.
     */
    createCopy([args])
    {
        local vec; 
        vec = new Vector(self.length());
        vec.copyElmFrom(self, args...);
        return vec;
    }

    /*
     *  Returns a new vector consisting of the intersection 
     *  of this list and coll; that is, a list consisting 
     *  of the elements common to both this list and coll.  
     *  coll must also be either a vector or a list.  If the 
     *  two collections have no elements in common, the result 
     *  is an empty list.  If an element of the shorter collection
     *  (or, if the collections are of equal length, this collection) 
     *  appears more than once in the shorter collection, and that 
     *  element value also appears in the longer collection, then 
     *  the element will be in the result collection the same number 
     *  of times that it is in the shorter collection.  An element 
     *  repeated in the longer collection will not be repeated in 
     *  the result collection.
     *
     *  Provides the same method name as for List.
     */
    intersectColl(coll)
    {
        local lst1, lst2, iLst, iVec;
        lst1 = toList();

        if (dataType(coll) != TypeList)
            lst2 = coll.toList();
        else lst2 = coll;

        iLst = lst1.intersect(lst2);
        iVec = new Vector(iLst.length(), iLst);
        return iVec;
    }

    /*
     *  Returns the first element of the vector.  If the vector 
     *  has no elements, returns nil.
     */
    firstElm()
    {
        if (length())
            return self[1];
        else return nil;
    }

    /*
     *  Returns the last element of the vector.  If the vector 
     *  has no elements, returns nil.
     */
    lastElm()
    {
        if (length())
            return self[length()];
        else return nil;
    }

    /*
     *  Returns the "tail" of the vector; that is, the rest of the vector 
     *  after removing the first element.  If the vector has no elements, 
     *  returns nil.  This function is almost the same as sublist(2), 
     *  except that sublist() would return an empty vector if given an 
     *  empty vector, whereas cdr() returns nil in this case.
     */
    rest()
    {
        if (length())
            return subcollection(2);
        else return nil;
    }

    /*
     *  Creates and returns a new vector consisting of a subvector 
     *  of this vector starting at the element of this vector 
     *  at index start, and continuing for the number of elements 
     *  given by length, if present, or to the end of this vector 
     *  if not.
     */
    subvector(start, [len]) 
    {
        return createCopy(start, 1, len...);
    }

    /* 
     *  Identical to subvector. 
     *  Provided for polymorphism with List class objects.
     */
    subcollection(start, [len]) { return subvector(start, len...); }

    /*
     *  Provided for completeness. This method is equivalent to 
     *  subvector and subcollection in that it will produce a 
     *  new vector value. The method can take 2 optional arguments.
     *  The first indicates the position in which to begin the
     *  subvectoring, and the second indicates the number of elements
     *  from the starting position to add to the new vector.
     */
    toVector([args])
    {
        local start, count, coll;

        switch (args.length())
        {
            case 0:
                return new Vector(length(), self);

            case 1:
                start = args[1];
                count = length() - start + 1;
                break;

            default:
                start = args[1];
                count = args[2];
        }
        coll = subcollection(start, count);
        return new Vector(coll.length(), coll);
    }
}

modify LookupTable
{
    /*
     *  Creates and returns a new lookup table containing 
     *  the elements of this lookup table for which the 
     *  callback function func returns true (i.e., any 
     *  value other than nil or the integer value 0).  
     *  For each element of the source lookup table, this 
     *  method invokes the callback function, passing the 
     *  key and value of the current element as the callback 
     *  function's arguments.  If the callback returns nil or 
     *  the integer value 0, the method omits the element from 
     *  the result; otherwise, the method includes the element in 
     *  the result lookup table.  
     */
    subset(func)
    {
        local tbl, ret;

        tbl = new LookupTable(getBucketCount(), getEntryCount());
        forEachAssoc(new function(key, value)
        {
            ret = (func)(key, value);
            if (ret)
                tbl[key] = value;
        });
        return tbl;
    }

    /*
     *  Returns a new lookup table whose key-value relationships
     *  are the transposition of this lookup table. In other words,
     *  the keys of this lookup table become the values of the new
     *  lookup table, while the values of this lookup table become
     *  the keys of the new lookup table.
     *
     *  isomorph indicates whether the keys and values of this 
     *  lookup table are to be considered isomorphic, having 
     *  a one-to-one relationship: one and only one key for 
     *  each value of this lookup table. If two keys in the 
     *  table can return the same value, then we would lose
     *  one of the keys by simple transposition and should 
     *  set isomorph to nil so that both keys in this table
     *  will be accumulated in a list value.
     *
     *  If func is provided then the transpose will be performed 
     *  on the subset() of this lookuptable as derived from the
     *  function.
     */
    transpose(isomorph, [func])
    {
        local tbl1, tbl2;
        if (func.length())
            tbl1 = subset(func.car());
        else tbl1 = self;

        tbl2 = new LookupTable(tbl1.getBucketCount(), getEntryCount());
        tbl1.forEachAssoc(new function(key, value)
        {
            if (isomorph)
            {
                tbl2[value] = key;
            }
            else
            {
                if (!tbl2.isKeyPresent(value))
                    tbl2[value] = new Vector(50);

                tbl2[value].append(key);
            }
        });

        if (!isomorph)
        {
            tbl2.forEachAssoc(new function(key, value)
            {
                tbl2[key] = value.toList();
            });
        }

        return tbl2;
    }
}
