#charset "us-ascii"
/* 
 *  Copyright (c) 2005 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of the TADS 3 Services Pack
 *
 *  tsp_method_overload.t
 *
 *  Routes to a method built from the targetprop 
 *  and argument count. This lets the author define
 *  a simple form of "method overloading" on a given 
 *  method. For instance, just #include method_overload.h
 *  in your source, and include method_overload.t in the
 *  compilation. 
 *
 *  Then you can define an object like this:
 *
 *      myObject: object
 *      {
 *          MethodOverload(construct)
 *          {
 *              args0() {...}
 *              args1(arg1) {...}
 *              args2(arg1, arg2) {...}
 *          }
 *
 *          MethodOverload(sum)
 *          {
 *              args2(arg1, arg2) {...}
 *              args3(arg1, arg2, arg3) {...}
 *          }
 *      }
 *
 *  This lets you code the methods without having to code
 *  for variable argument lists and the usual switch statement
 *  required to handle each situation.
 *
 *---------------------------------------------------------------
 *  CAVEATS
 *---------------------------------------------------------------
 *  Note that no type determination is involved, overloading
 *  is done based solely on the *number* of arguments, not their
 *  type.
 *
 *  If a method hasn't been defined for the corresponding
 *  number of arguments, then a runtime error is returned
 *  indicating "wrong number of arguments".
 *---------------------------------------------------------------
 */

#include "tsp_method_overload.h"
#include "tsp_mod_registry.h"

RegisterModule

modify Object
{
    /* 
     *  Calls the method built from the targetprop
     *  and number of arguments passed. If the property
     *  isn't defined in the symbol table we return nil. 
     */
    callMethodOverload(tprop, [args])
    {
        local nStr, tStr, prop;

        nStr = toString(args.length());
        tStr = moGlobal.getSString(tprop);
        prop = moGlobal.getSymbol(tStr + '__args' + nStr);

        if (prop == nil)
            throw new MethodOverloadArgCntError();
        else 
            return self.(prop)(args...); 
    }
}

/*
 *  Stores a copy of the global symbol table
 *  and its reverse in order to build the 
 *  method overload property.
 */
moGlobal: PreinitObject
{
    symtab          = nil
    reverseSymtab   = nil

    getSString(key) { return reverseSymtab[key]; }
    getSymbol(key) { return symtab[key]; }

    execute()
    {
        /*  store a reference to the global symbol table */
        symtab = t3GetGlobalSymbols();
        
        /*
         *  building a reverse mapping - in other words, building a
         *  LookupTable that uses symbol values as the keys and symbol
         *  strings as the associated values.
         */
        reverseSymtab = new LookupTable();
        symtab.forEachAssoc({str, sym: reverseSymtab[sym] = str});    
    }
}

class MethodOverloadArgCntError: RuntimeError
{
    exceptionMessage = 'wrong number of arguments'
}