/*
Copyright (C) 2003 Hotsprings Inc.
For conditions of distribution and use, see copyright notice

Location: 
	www.HotspringsInc.com 

History:
	2003Aug27-GiuseppeG: code write
*/

#include "XSP_Core.h"
#include "XSP_File.h"
#include "XSP_GUI.h"

namespace XSP
{
/*
	Toolbars and menus can be designed and created by one side of the app
	and the actions taken by these views created by another side of the app.
	
	Example: TextView control can react to Edit/Paste and Edit/SelectAll
	The designer of the app may create a permanent menu in the app called Edit/SelectAll
	The menu is there whether a textview is focused or not.
	When a textview is not focused, the menu should be disabled to show it 
	can do nothing.
	When a textview is focused the menu triggers the appropriate command into 
	the text.
	When another textview is focused the menu triggers the appropriate command 
	into the later text.

	So the designer creates and populates the menu and toolbar.
	The focused control fills what it knows of with proper directed actions and 
	enables the entry.

But! The focused control knows nothing about the type of view the designer used 
	to create the entry. So it can only operate at the view level with 
	enable/disable.
	
	How do we deal with buttons, dialogs like combo, or dialogs like font select, 
	since they all can be part of the toolbar.
	
	1)  the designer only provides the tree of ports, no content, the focused 
		view fills in the appropriate controls. But the user will see empty rects 
		when the focus is not there to fill the stuff. What we intend is to allow 
		the user to see a disabled control in that place.
	2) 	the designer does the filling and the view has blind faith that there 
		was no mistake.
	3)  the designer does the filling and the communication between the control 
		and the	focused view is untyped. Neither side needs to know to whom it 
		is talking, all that is needed is to validate the communication protocol.

To find eachother the views need to export their action names, the publicActionRepo 
must be created, the menu and the toolbar must use that public repo, also the toolbar 
and menu must allow design insertions from the main app init.
*/


ToolSocket::ToolSocket()
{
}
ToolSocket::~ToolSocket()
{
}
void ToolSocket::SetToolName(const String& tnm)
{
	toolName = tnm;
}

void ToolSocket::Draw(Graphics& g, const Rect2D& area)
{
	/* only draw our own looks if there is no child, 
	else rely on the draw of the child */
	const Children& children = GetChildren();
	if (children.empty())
	{
		ViewSkin::owner skin = GetSkin();
		if (skin != 0)
		{
			StateSkin::owner stateSkin = skin->GetStateSkin(View::StateDisabled);
			if (stateSkin != 0)
			{
				Rect2D bounds = GetLocalBounds();
				stateSkin->Draw(g, bounds,area);
			}
		}
	}
}
void ToolSocket::SetBounds(const Rect2D& bounds)
{
	if (bounds == GetBounds())
		return;
	View::SetBounds(bounds);

	// children at 0,0, w,h
	Rect2D cb(bounds.GetSize());
	const Children& children = GetChildren();
	for (Children::const_iterator b=children.begin(), e=children.end(); b!=e; ++b)
		(*b)->SetBounds(cb);
}
Size2D ToolSocket::GetPreferredSize()
{
	const Children& children = GetChildren();
	if (children.empty())
		return Size2D(0,0);
	return children.front()->GetPreferredSize();
}

#if 0
#pragma mark -
#endif

ToolMgr* ToolMgr::_instance = 0;

ToolMgr* ToolMgr::Instance()
{
	if (ToolMgr::_instance == 0)
		ToolMgr::_instance = new ToolMgr();
	return ToolMgr::_instance;
}

ToolMgr::ToolMgr()
{
}
ToolMgr::~ToolMgr()
{
	if (ToolMgr::_instance == this)
		ToolMgr::_instance = 0;
}
void ToolMgr::RegisterTool(ToolSocket* ts)
{
	if (ts == 0)
		return;
	const String& tn = ts->GetToolName();

	ToolRepo::iterator tr = toolRepo.find(tn);
	if (tr == toolRepo.end())
	{
		ToolRepo::value_type tv(tn, ToolSockets());
		toolRepo.insert(tv);
		tr = toolRepo.find(tn);
	}
	ts->SetToolName(tr->first); // no duplicate strings

	ToolSockets& vc = tr->second;
	vc.push_back(ts); // on more tool for the central repo
	
	NotifyChange_ToolAddRemove(ts, true);
}
void ToolMgr::UnregisterTool(ToolSocket* ts)
{
	if (ts == 0)
		return;
	const String& tn = ts->GetToolName();
	ToolRepo::iterator tr = toolRepo.find(tn);
	if (tr == toolRepo.end())
		return;
	ToolSockets& vc = tr->second;
	vc.remove(ts);
	NotifyChange_ToolAddRemove(ts, false);
}
void ToolMgr::_ToolChange::ToolAddRemove(ToolSocket* /*ts*/, bool /*added*/)
{
	// by default do nothing
}
void ToolMgr::NotifyChange_ToolAddRemove(ToolSocket* ts, bool added)
{
	toolListeners.withinNotify = true;
	for (ToolListener::each e(toolListeners); !e.empty(); e.pop()) 
		if (!e->IsCanceled())
			e->ToolAddRemove(ts, added);
	toolListeners.withinNotify = false;
}
void ToolMgr::AddListener(const ToolListener::owner& l)
{
	toolListeners.Add(l);
}




#if 0
#pragma mark -
#endif

ToolBar::ToolBar()
:cellSize(20)
{
	toolMgr = ToolMgr::Instance();
}
ToolBar::~ToolBar()
{
	toolMgr = 0;
}

void ToolBar::AddTool(const String& toolname)
{
	ToolSocket* toolSock = new ToolSocket();
	AddView(toolSock); // one more child of this toolbar
	toolSock->SetToolName(toolname);
	toolSock->SetSkin(GetSkinPartOrDefault(String::From_c_str("Tool")));

	// register the tool socket into the global repo 
	toolMgr->RegisterTool(toolSock);
}

void ToolBar::AddTool(ToolSocket* toolSock)
{
	AddView(toolSock); // one more child of this toolbar
	// register the tool socket into the global repo 
	toolMgr->RegisterTool(toolSock);
}

void ToolBar::RemoveTool(const String& toolname)
{
	ToolSocket* toolSock = GetTool(toolname);
	if (toolSock != 0)
		RemoveTool(toolSock);
}
void ToolBar::RemoveTool(ToolSocket* toolSock)
{
	RemoveView(toolSock);
	toolMgr->UnregisterTool(toolSock);
}
ToolSocket* ToolBar::GetTool(const String& toolname)
{
	const View::Children& vc = GetChildren();
	for (Children::const_iterator b=vc.begin(), e=vc.end(); b!=e; ++b)
	{ 
		const View::owner& child = *b;
		ToolSocket* toolSock = static_cast<ToolSocket*> (child.operator->());
		if (toolSock == 0)
			continue;
		if (toolname == toolSock->GetToolName())
			return toolSock;
	}
	return 0;
}


void ToolBar::SetCellSize(uint32 z)
{
	cellSize = z;
	DoLayout();
}
void ToolBar::SetBounds(const Rect2D& bounds)
{
	View::SetBounds(bounds);
	DoLayout();
}
Size2D ToolBar::GetPreferredSize()
{
	Size2D z(0,cellSize);
	const Children& children = GetChildren();
	for (Children::const_iterator b=children.begin(), e=children.end(); b!=e; ++b)
	{
		Size2D cz((*b)->GetPreferredSize());
		z.w += std::max(cellSize, cz.w);
		z.h  = std::max(z.h, cz.h);
	}
	{
		ViewSkin::owner skin = GetSkin();
		if (skin != 0)
		{
			StateSkin::owner stateSkin = skin->GetStateSkin(0);
			if (stateSkin != 0)
			{
				Rect2D border = stateSkin->GetContent();
				z.w += border.w;
				z.h += border.h;
			}
		}
	}
	return z;
}

void ToolBar::DoLayout()
{
	bool dirty = false;

	Rect2D lb = GetLocalBounds();
	{
		ViewSkin::owner skin = GetSkin();
		if (skin != 0)
		{
			StateSkin::owner stateSkin = skin->GetStateSkin(0);
			if (stateSkin != 0)
				lb = stateSkin->GetContent(lb);
		}
	}

	const Children& children = GetChildren();
	Children::const_iterator b = children.begin();
	Children::const_iterator e = children.end();
	Children::const_iterator p;
	for(p = b; p!=e; ++p)
	{
		bool adjust = false;
		Rect2D r = (*p)->GetBounds();
		if (r.w < cellSize)	
		{
			r.w = cellSize;
			adjust = true;
		}
		if (r.h < cellSize)	
		{
			r.h = cellSize;
			adjust = true;
		}
			
		if (r.y != lb.y)
		{
			r.y = lb.y;
			adjust = true;
		}
		if (r.h != lb.h)
		{
			r.h = lb.h;
			adjust = true;
		}
		if (r.x != lb.x)
		{
			r.x = lb.x;
			adjust = true;
		}
		lb.x += r.w;
		if (adjust)
		{
			(*p)->SetBounds(r);
			dirty = true;
		}
	}
	if (dirty)
		Refresh();
}

void ToolBar::DemoPopulate()
{
	for(sint32 k=0; k<10; ++k)
		AddTool( String::From_c_str("Tool-Button-")+String::From_sint32(k+100) );
	DoLayout();
}

} // namespace XSP
