/* RESOLV.C - This file is part of
Resolver for Windows, copyright (C) Fabrizio Gennari, 2000-2003

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.  */

#include <winsock2.h>
#include <stdio.h>
#include <memory.h>
#include "resolv.h"

int dn_expand(const u_char *msg, const u_char *eom, const u_char *src,
          char *dst, int dstsiz){
	unsigned char length;
	const unsigned char *initial_src=src;
	int result;
	unsigned short new_pos;

	for(;;){
		if (src < msg || src >= eom)
			return -2;
		length=*src;
		src++;
		if ((length&192)==192){
			if (src >= eom)
				return -2;
			new_pos=(length&63)*256+*src;
			src++;
			if (msg+new_pos >= initial_src)/* force pointers to go backwards to avoid recursion loops */
				return -2;
			result=dn_expand(msg,eom,msg+new_pos,dst,dstsiz);
			if (result<0)
				return result;
			return src-initial_src;
		}
		if (length==0){
			*(dst-1)=0;
			return src-initial_src;
		}
		if (src+length > eom)
			return -2;
		if (length+1 > dstsiz)
		return -1;
		memcpy(dst,src,length);
		dst+=length;
		src+=length;
		*dst='.';
		dst++;
		dstsiz-=(length+1);
	}
}

/*Simplified version of the above: do not store names */
int dn_skipname(const u_char *src, const u_char *eom) {
	unsigned char length;
	const unsigned char *initial_src=src;

	for(;;){
		if (src >= eom)
			return -1;
		length=*src;
		src++;
		if ((length&192)==192){
			if (src >= eom)
				return -1;
			src++;
			return src-initial_src;
		}
		if (length==0){
			return src-initial_src;
		}
		if (src+length > eom)
			return -1;
		src+=length;
	}
}

int res_mkquery(int query_type, char* domain_name, int class, int type,
				char* data, int data_length, int inutile, char* buffer,
				int buf_length){
	int lunghezza,i,l;
	unsigned char count;
	struct query query;
	unsigned short qtype[2];
	lunghezza=sizeof(query)+sizeof(qtype)+strlen(domain_name)+2;
	if (buf_length<lunghezza) return -1;
	query.id[0]=query.id[1]='F';
	query.control[0]=1+((query_type&15)<<3);//recursion desired
	query.control[1]=0;//no error
	query.qdcount=htons(1);
	query.ancount=0;
	query.nscount=0;
	query.arcount=0;
	qtype[0]=htons((unsigned short)type);
	qtype[1]=htons((unsigned short)class);
	memcpy(buffer,&query,sizeof(query));
	l=strlen(domain_name);
	count=0;
	for(i=0;i<=l;i++){
		if ((domain_name[i]!='.')&&(domain_name[i]!=0))
			count++;
		else{
			buffer[sizeof(query)+i-count]=count;
			memcpy(buffer+sizeof(query)+i-count+1,domain_name+i-count,count);
			count=0;
		}
	}
	buffer[sizeof(query)+l+1]=0;
	memcpy(buffer+sizeof(query)+l+2,(char*)&qtype,sizeof(qtype));
	return lunghezza;
}

int process_errors(int errors,char* message, char* host){
	switch(errors&15){
	case 0:
		break;
	case 1:
		strcpy(message,"Format error");
		break;
	case 2:
		strcpy(message,"Server failure");
		break;
	case 3:
		sprintf(message,"The host or domain %s does not exist",host);
		break;
	case 4:
	strcpy(message,"Not implemented");
		break;
	case 5:
		strcpy(message,"Refused");
		break;
	default:
		strcpy(message,"Unknown error");
	}
	return (errors&15);
}

int res_nsend(SOCKADDR *dns_addr, const unsigned char *query, int querylen,
			  unsigned char *query_rep, int anslength){
	int timeout=20000;
	unsigned short int network_querylen,query_rep_len;
	SOCKET sock;

	network_querylen=htons((unsigned short)querylen);
	sock=socket(AF_INET,SOCK_STREAM,IPPROTO_TCP);
	setsockopt(sock,SOL_SOCKET, SO_SNDTIMEO, (char *)&timeout, sizeof(timeout));
	setsockopt(sock,SOL_SOCKET, SO_RCVTIMEO, (char *)&timeout, sizeof(timeout));
	if (connect(sock,dns_addr,16) == SOCKET_ERROR){
		return -4;
	}
	if (send(sock,(char*)&network_querylen,2,0) == SOCKET_ERROR){
		closesocket(sock);
		return -2;
	};
	if (send(sock,query,querylen,0) == SOCKET_ERROR){
		closesocket(sock);
		return -2;
	};
	/*MessageBox(0,"Inutile","Error receiving",MB_ICONERROR);*/
	if (recv(sock,(char*)&query_rep_len,2,0) == SOCKET_ERROR){
		char message[1000];
		_snprintf(message,1000,"Error is %u",WSAGetLastError());
		MessageBox(0,message,"Error receiving",MB_ICONERROR);
		closesocket(sock);
		return -3;
	};
	query_rep_len=ntohs(query_rep_len);
	if (anslength < query_rep_len){
		closesocket(sock);
		return -1;
	}
	if (recv(sock, (char*)query_rep, query_rep_len, 0) == SOCKET_ERROR){
		char message[1000];
		_snprintf(message,1000,"Error is %u",WSAGetLastError());
		MessageBox(0,message,"Error receiving",MB_ICONERROR);
		closesocket(sock);
		free(query_rep);
		return -3;
	};
	closesocket(sock);
	return query_rep_len;
}

int ns_skiprr(const u_char *msg, const u_char *eom,
			  ns_sect section,
			  int count){
	int i;
	int namelen;
	struct answer *ans;
	const u_char *initial_msg=msg;

	for(i=0;i<count;i++){
		if (msg >= eom)
			return -1;
		namelen=dn_skipname(msg,eom);
		if (namelen<0)
			return namelen;
		msg += namelen;
		if (section == ns_s_qd){
			if (msg+4 > eom) /* type (2 bytes) and class (2 bytes) */
				return -1;
			msg += 4;
			continue;
		}
		if (msg + sizeof(struct answer) > eom)
			return -1;
		ans=(struct answer*)msg;
		msg += sizeof(struct answer)+ntohs(ans->rdlength);
	}
	return msg-initial_msg;
}

int GetIPAddress(const u_char *src, const u_char *eom,
				 char *string, int len){
	if (src+4>eom)
		return -1;
	_snprintf(string,len,"%u.%u.%u.%u",src[0],src[1],src[2],src[3]);
	return 4;
}

int GetMailExchanger(unsigned char *msg, unsigned char *eom, unsigned char *src,
			   		   char *name, int namelen, unsigned short *preference){
	int result;

	if (src+2>eom)
		return -1;
	*preference=ntohs(*(unsigned short*)src);
	src+=2;
	if ( (result=dn_expand(msg,eom,src,name,namelen)) < 0)
		return result;
	return result+2;
}

int GetSOA(unsigned char *msg, unsigned char *eom, unsigned char *src,
			   char *name, int namelen,
			   char *rname, int rnamelen,
			   unsigned long *serial,
			   unsigned long *refresh,
			   unsigned long *retry,
			   unsigned long *expire){
	int result,	result2;

	if ( (result=dn_expand(msg,eom,src,name,namelen)) < 0)
		return result;
	src+=result;
	if (src>eom)
		return -1;
	if ( (result2=dn_expand(msg,eom,src,rname,rnamelen)) < 0)
		return result2;
	src+=result2;
	if (src+32>eom)
		return -1;
	*serial=ntohl(*(unsigned long*)src);
	src+=4;
	*refresh=ntohl(*(unsigned long*)src);
	src+=4;
	*retry=ntohl(*(unsigned long*)src);
	src+=4;
	*expire=ntohl(*(unsigned long*)src);
	src+=4;

	return result+result2+32;
};

int GetCPUOS(const u_char *src, const u_char *eom,
				 char *cpu, int cpulen,
				 char *os, int oslen){
	unsigned char realcpulen,realoslen;

	if (src>=eom)
		return -1;
	realcpulen=*src;
	if (cpulen<realcpulen)
		return -2;
	src+=1;
	if (src+realcpulen>eom)
		return -1;
	memcpy(cpu,src,realcpulen);

	src+=realcpulen;

	if (src>=eom)
		return -1;
	realoslen=*src;
	if (oslen<realoslen)
		return -2;

	src+=1;
	if (src+realoslen>eom)
		return -1;
	memcpy(os,src,realoslen);

	return realcpulen+realoslen+2;
}
