/*
Copyright (C) 2003 Hotsprings Inc.
For conditions of distribution and use, see copyright notice

Location: 
	www.HotspringsInc.com 

History:
	2003Apr10-GiuseppeG: code write
*/

#include "XSP_Core.h"

namespace XSP
{
// the preallocated sdata for the empty string
// the reference count is initialized to more than 0
// so this string can never be deleted
static char  _localEmpty_sdata_buf[String::_xsize] = "1111\0\0\0\0\0\0\0\0\0";
static char* const _localEmpty_sdata = _localEmpty_sdata_buf + String::_xsize - 1;

// the empty string  
const String String::kEmpty;

String::String()
: sdata(_localEmpty_sdata)
{
}

String::String(const String& s)
: sdata(s.sdata)
{
	if (sdata != _localEmpty_sdata)
    	++_refcount(); 
}

String::String(const_iterator b, const_iterator e)
{
	if (b == e)
	{
		sdata = _localEmpty_sdata;
		return;
	}
    size_type len = (size_type)(e - b);
    ASSERT(len <= 1000000000); // 1GB string is probably too large 
	
	sdata = (new char[_xsize + len]) + (_xsize - 1);
    _refcount() = 1;
    _capacity() = _length() = len;	 	
    std::string::traits_type::copy(sdata, b, len);
}

String::String(iterator b, iterator e)
{
	if (b == e)
	{
		sdata = _localEmpty_sdata;
		return;
	}
    size_type len = (size_type)(e - b);
    ASSERT(len <= 1000000000); // 1GB string is probably too large 
	
	sdata = (new char[_xsize + len]) + (_xsize - 1);
    _refcount() = 1;
    _capacity() = _length() = len;	 	
    std::string::traits_type::copy(sdata, b, len);
}

String::String(const_iterator b, size_type len)
{
	if (len == 0)
	{
		sdata = _localEmpty_sdata;
		return;
	}
    ASSERT(len <= 1000000000); // 1GB string is probably too large 
	sdata = (new char[_xsize + len]) + (_xsize - 1);
    _refcount() = 1;
    _capacity() = _length() = len;	 	
    std::string::traits_type::copy(sdata, b, len);
}

String String::From_c_str(const char* s)
{
    uint32 len = 0;
    if (s != 0) 
    	len = std::string::traits_type::length(s);
    return String(s, len);	
}

String String::From_p_str(const uint8* s)
{
	if (s == 0 || *s == 0)
		return kEmpty;
    
    uint32 len = *s;
    s++;
    return String(reinterpret_cast<const char*>(s), len);	
}

String& String::assign(const String& s)
{
	if (sdata == s.sdata)
		return *this;

    char* tmp = sdata;
	bool toDelete = (sdata != _localEmpty_sdata) && (0 == --_refcount());

	sdata = s.sdata;	
	if (sdata != _localEmpty_sdata)
    	++_refcount();

    if (toDelete)
		delete (tmp - (_xsize - 1));
    	
    return *this; 
}

String String::substr(size_type pos, size_type len) const
{
	size_type l = _length();
	if (len == -1)
		len = l;
	if (pos >= len)
		len = 0;
	if (pos + len > l)
		len = l-pos; 
	if (pos == 0 && len == l)
		return *this;
	if (len == 0)
		return kEmpty;
	return String(sdata+pos, sdata+pos+len);
}

String::~String()
{
	clear();
}									  

void String::clear()
{
	if (sdata != _localEmpty_sdata)
	{
	    ASSERT(_length()<=_capacity());
    	if (0 == --_refcount())
    	{ 
		    char* tmp = sdata;
		    sdata = _localEmpty_sdata;
			delete (tmp - (_xsize - 1));
		}
		else
		    sdata = _localEmpty_sdata;
	}
}

const char* String::c_str() const
{
	if (sdata != _localEmpty_sdata)
		sdata[_length()] = 0;
	return sdata;
}

const uint8* String::p_str() const
{
	uint8* ret = _p_str();
	if (sdata == _localEmpty_sdata)
	   return ret;

	uint32 len = _length();
	*ret = (uint8)len;

	if (*ret != len)
	{   // the string is longer than 255 chars
		Exception(XSPMSG(6,"The length of the string is longer than 255 characters '$1;'"))
			.Param(*this)
			.Raise();
	}
	return ret;
}

String::size_type String::capacity() const
{
	return _capacity();
}

void String::reserve(size_type z)
{
	size_type c = _capacity();
	if (z <= c)
		return;
    ASSERT(z <= 1000000000); // 1GB string is probably too large 
	
	// must grow the allocated size
	bool toDelete = (sdata != _localEmpty_sdata) && (0 == --_refcount());
	size_type len = _length();
	char* tmp = sdata;

	sdata = (new char[_xsize + z]) + (_xsize - 1);
    _refcount() = 1;
    _capacity() = z;
    _length() = len;
    std::string::traits_type::copy(sdata, tmp, len);
	if (toDelete)
		delete (tmp - (_xsize - 1));
}

void String::resize(size_type z)
{
	size_type c = _capacity();
	size_type len = _length();
    if (z == len && z == c)
    	return;
    ASSERT(z <= 1000000000); // 1GB string is probably too large 
    	
	bool toDelete = (sdata != _localEmpty_sdata) && (0 == --_refcount());
	char* tmp = sdata;

	sdata = (new char[_xsize + z]) + (_xsize - 1);
    _refcount() = 1;
    _capacity() = z;
    _length()   = z;
	std::char_traits<char>::copy(sdata, tmp, std::min(z,len));
	if (toDelete)
		delete (tmp - (_xsize - 1));
}
void String::compact()
{
	resize(_length());
}

void String::setsize(size_type z)
{
	size_type c = _capacity();
	if (z <= c)
		_length() = z;
	else
		resize(z);
}


void String::swap(String& s)
{
	if (sdata == s.sdata)
		return;
	char* tmp = sdata;
	sdata = s.sdata;
	s.sdata = tmp;
}

int String::cmp(const String& s) const
{
	if (sdata == s.sdata)
		return 0;
	uint32 l1 = _length();
	uint32 l2 = s._length();
	int r = std::string::traits_type::compare(sdata, s.sdata, std::min(l1,l2));		
    if (r != 0)
    	return r;
    return ((l1<l2) ? 1 : ((l1>l2) ? -1 : 0));
}

void String::replace(iterator dstb, iterator dste, const_iterator srcb, const_iterator srce)
{
	size_type len = _length();
    size_type nlen = len + (srce-srcb) - (dste-dstb);
    if (nlen == 0)
    {
    	clear();
    	return;
    }
    ASSERT(nlen <= 1000000000); // 1GB string is probably too large
    // the destination should point inside the string
    VERIFY(sdata<=dstb && dstb<=dste && dste<=sdata+len); 
  	size_type len2 = (size_type)(sdata+len - dste);
  	size_type len3 = (size_type)(srce-srcb);

    if ((nlen <= _capacity()) && (1 == _refcount()))
    {   // in place edit
    	std::string::traits_type::move(sdata+nlen-len2, dste, len2);
    	std::string::traits_type::copy(dstb, srcb, len3);
    }
    else
    {
		bool toDelete = (sdata != _localEmpty_sdata) && (0 == --_refcount());
		char* tmp = sdata;

		sdata = (new char[_xsize + nlen]) + (_xsize - 1);
	    _refcount() = 1;
	    _capacity() = nlen;
	
	  	size_type len1 = (size_type)(dstb-tmp);
    	std::string::traits_type::copy(sdata, tmp, len1);
    	std::string::traits_type::copy(sdata+nlen-len2, dste, len2);
    	std::string::traits_type::copy(sdata+len1, srcb, len3);

		if (toDelete)
			delete (tmp - (_xsize - 1));
    }
    _length() = nlen;
    
}

String::value_type String::operator[] (uint32 n) const
{
	VERIFY(n < _length());
	return sdata[n];
}

String::value_type& String::operator[] (uint32 n)
{
	VERIFY((n < _length()) && (_refcount() == 1));
	return sdata[n];
}

String String::operator+ (const String& s) const
{
	String rez(*this);
	rez.append(s);
	return rez;
}


static const char digits[] = "0123456789ABCDEFGHIJKLMNOPQRSTUVWZXY";
String String::From_uint64(uint64 n, uint32 baseN)
{
	VERIFY((baseN >= 2) && (baseN <= sizeof(digits)-1));
	
    char buf[256];
    char* e = buf+255;
    char* b = e;
    *b = 0;
    
    for(;;)
    {
	    *--b = digits[n % baseN];
	    n /= baseN;
	    if (n == 0)
	    	break;
	}
	
	return String(b,e);
} 
String String::From_sint64(sint64 n, uint32 baseN)
{
	VERIFY((baseN >= 2) && (baseN <= sizeof(digits)-1));
	
    char buf[256];
    char* e = buf+255;
    char* b = e;
    *b = 0;
    bool sgn = n < 0;
    if (sgn) n = -n;
    
    for(;;)
    {
	    * --b = digits[n % baseN];
	    n /= baseN;
	    if (n == 0)
	    	break;
	}
	if (sgn)
		* --b = '-';
	return String(b,e);
} 
uint64 String::To_uint64() const
{
	const_iterator b = begin();
	const_iterator e = end();
	uint64 n = 0;
	for (; b!=e && *b ==' '; ++b)
	{}
	for (; b!=e && *b >= '0' && *b <= '9'; ++b)
		n = n*10 + (*b - '0');
	return n;
}   
sint64 String::To_sint64() const
{
	const_iterator b = begin();
	const_iterator e = end();
	sint64 n = 0;
	bool sgn = false;
	for (; b!=e && *b ==' '; ++b)
	{}
	if (*b == '-')
		++b, sgn=true;
	for (; b!=e && *b >= '0' && *b <= '9'; ++b)
		n = n*10 + (*b - '0');
	return sgn ? -n : n;
}

#if 0
#pragma mark -
#endif

StringMessage::StringMessage(uint32 mid)
: msgID(mid)
{
}
StringMessage::~StringMessage()
{
	parms.clear();
}
StringMessage& StringMessage::Param(const String& p)
{
	parms.push_back(p);
	return *this;
}
StringMessage& StringMessage::Param(uint32 p)
{
	return Param(String::From_uint32(p,10));
}
StringMessage& StringMessage::Param(sint32 p)
{
	return Param(String::From_sint32(p,10));
}
StringMessage& StringMessage::Param(uint64 p)
{
	return Param(String::From_uint64(p,10));
}
StringMessage& StringMessage::Param(sint64 p)
{
	return Param(String::From_sint64(p,10));
}

String StringMessage::ToString() const
{
	// load the msgID
	// replace the param markers with the param values
	String msg(String::From_c_str("StringMessage::ToString(")); 
	msg.append(String::From_uint32(msgID));
	msg.append(String::From_c_str(")"));
	
	Params::const_iterator b = parms.begin();
	Params::const_iterator e = parms.end();
	for(uint32 n=0; b!=e; ++b,++n)
	{
		msg.append(String::From_c_str("  ["));
		msg.append(String::From_uint32(n));
		msg.append(String::From_c_str("]="));
		msg.append(*b); 
	}	
	return msg;
}

} // namespace XSP


