/*
Copyright (C) 2003 Hotsprings Inc.
For conditions of distribution and use, see copyright notice

Location: 
	www.HotspringsInc.com 

History:
	2003Apr14-GiuseppeG: code write
*/

#include "XSP_Core.h"
#include "XSP_File.h"

namespace XSP
{

#if TARGET_API_Linux

//#if !defined(_DIRENT_HAVE_D_NAMELEN)
//#error "The dirent structure does not seem to have the namelen defined"
//#endif
#if !defined(__USE_BSD)
#error "The BSD constants are not included"
#endif

static void _LinuxFS_Fail(int err, const FileName& fn)
{
	if (err == 0)
		return;

	uint32 ern = 0;
	switch(err)
	{
	case EPERM: ern=XSPMSG(271,"Operation not permitted"); break;
	case ENOENT: ern=XSPMSG(272,"No such file or directory"); break;
	case ESRCH: ern=XSPMSG(273,"No such process"); break;
	case EINTR: ern=XSPMSG(274,"Interrupted system call"); break;
	case EIO: ern=XSPMSG(275,"Input/output error"); break;
	case ENXIO: ern=XSPMSG(276,"Device not configured"); break;
	case E2BIG: ern=XSPMSG(277,"Argument list too long"); break;
	case ENOEXEC: ern=XSPMSG(278,"Exec format error"); break;
	case EBADF: ern=XSPMSG(279,"Bad file descriptor"); break;
	case ECHILD: ern=XSPMSG(280,"No child processes"); break;
	case EDEADLK: ern=XSPMSG(281,"Resource deadlock avoided"); break;
	case ENOMEM: ern=XSPMSG(282,"Cannot allocate memory"); break;
	case EACCES: ern=XSPMSG(283,"Permission denied"); break;
	case EFAULT: ern=XSPMSG(284,"Bad address"); break;
	case EBUSY: ern=XSPMSG(285,"Device busy"); break;
	case EEXIST: ern=XSPMSG(286,"File exists"); break;
	case EXDEV: ern=XSPMSG(287,"Cross-device link"); break;
	case ENODEV: ern=XSPMSG(288,"Operation not supported by device"); break;
	case ENOTDIR: ern=XSPMSG(289,"Not a directory"); break;
	case EISDIR: ern=XSPMSG(290,"Is a directory"); break;
	case EINVAL: ern=XSPMSG(291,"Invalid argument"); break;
	case ENFILE: ern=XSPMSG(292,"Too many open files in system"); break;
	case EMFILE: ern=XSPMSG(293,"Too many open files"); break;
	case ENOTTY: ern=XSPMSG(294,"Inappropriate ioctl for device"); break;
	case ETXTBSY: ern=XSPMSG(295,"Text file busy"); break;
	case EFBIG: ern=XSPMSG(296,"File too large"); break;
	case ENOSPC: ern=XSPMSG(297,"No space left on device"); break;
	case ESPIPE: ern=XSPMSG(298,"Illegal seek"); break;
	case EROFS: ern=XSPMSG(299,"Read-only file system"); break;
	case EMLINK: ern=XSPMSG(300,"Too many links"); break;
	case EPIPE: ern=XSPMSG(301,"Broken pipe"); break;
	case EAGAIN: ern=XSPMSG(302,"Resource temporarily unavailable, operation would block, try again later"); break;
	case EINPROGRESS: ern=XSPMSG(303,"Operation now in progress"); break;
	case EALREADY: ern=XSPMSG(304,"Operation already in progress"); break;
	case ENOTSOCK: ern=XSPMSG(305,"Socket operation on non-socket"); break;
	case EDESTADDRREQ: ern=XSPMSG(306,"Destination address required"); break;
	case EMSGSIZE: ern=XSPMSG(307,"Message too long"); break;
	case EPROTOTYPE: ern=XSPMSG(308,"Protocol wrong type for socket"); break;
	case ENOPROTOOPT: ern=XSPMSG(309,"Protocol not available"); break;
	case EPROTONOSUPPORT: ern=XSPMSG(310,"Protocol not supported"); break;
	case ESOCKTNOSUPPORT: ern=XSPMSG(311,"Socket type not supported"); break;
	case ENOTSUP: ern=XSPMSG(312,"Operation not supported"); break;
	case EPFNOSUPPORT: ern=XSPMSG(313,"Protocol family not supported"); break;
	case EAFNOSUPPORT: ern=XSPMSG(314,"Address family not supported by protocol family"); break;
	case EADDRINUSE: ern=XSPMSG(315,"Address already in use"); break;
	case EADDRNOTAVAIL: ern=XSPMSG(316,"Can't assign requested address"); break;
	case ENETDOWN: ern=XSPMSG(317,"Network is down"); break;
	case ENETUNREACH: ern=XSPMSG(318,"Network is unreachable"); break;
	case ENETRESET: ern=XSPMSG(319,"Network dropped connection on reset"); break;
	case ECONNABORTED: ern=XSPMSG(320,"Software caused connection abort"); break;
	case ECONNRESET: ern=XSPMSG(321,"Connection reset by peer"); break;
	case ENOBUFS: ern=XSPMSG(322,"No buffer space available"); break;
	case EISCONN: ern=XSPMSG(323,"Socket is already connected"); break;
	case ENOTCONN: ern=XSPMSG(324,"Socket is not connected"); break;
	case ESHUTDOWN: ern=XSPMSG(325,"Can't send after socket shutdown"); break;
	case ETOOMANYREFS: ern=XSPMSG(326,"Too many references: can't splice"); break;
	case ETIMEDOUT: ern=XSPMSG(327,"Operation timed out"); break;
	case ECONNREFUSED: ern=XSPMSG(328,"Connection refused"); break;
	case ELOOP: ern=XSPMSG(329,"Too many levels of symbolic links"); break;
	case ENAMETOOLONG: ern=XSPMSG(330,"File name too long"); break;
	case EHOSTDOWN: ern=XSPMSG(331,"Host is down"); break;
	case EHOSTUNREACH: ern=XSPMSG(332,"No route to host"); break;
	case ENOTEMPTY: ern=XSPMSG(333,"Directory not empty"); break;
	case EUSERS: ern=XSPMSG(334,"Too many users"); break;
	case EDQUOT: ern=XSPMSG(335,"Disc quota exceeded"); break;
	case ENOLCK: ern=XSPMSG(336,"No locks available"); break;
	case ENOSYS: ern=XSPMSG(337,"Function not implemented"); break;
	case EOVERFLOW: ern=XSPMSG(338,"Value too large to be stored in data type"); break;
	case ECANCELED: ern=XSPMSG(339,"Operation canceled"); break;
	default: ern=XSPMSG(340,"OS filesystem error ($2;) while accessing file '$1;'"); break;
    };

	Exception(ern)
		.Param(fn.GetFullPathNameLinux())
		.Param((sint32)err)
		.Raise();
    
}

SpecialFileNames::SpecialFileNames()
: ref(new _Data())
{
	{ SetFileName(TempDir, FileName::FromPathNameLinux(String::From_c_str("/tmp"))); }
	{ SetFileName(MachineRoot, FileName::FromPathNameLinux(String::From_c_str("/"))); }
	{ SetFileName(SystemDir, FileName::FromPathNameLinux(String::From_c_str("/usr"))); }
	{ 
		char* home = ::getenv("HOME");
		FileName h(FileName::FromPathNameLinux(String::From_c_str((home != NULL) ? home : "/home"))); 
		SetFileName(UserHomeDir, h);
		SetFileName(UserPreferencesDir, h); 
		h = FileName::From(h, String::From_c_str("Desktop"));
		SetFileName(UserDesktopDir, h); 
	}

	{ SetFileName(MachinePreferencesDir, FileName::FromPathNameLinux(String::From_c_str("/etc"))); }
    {
    	char cwdbuf[MAXPATHLEN];
    	if (NULL != ::getcwd(cwdbuf, sizeof(cwdbuf)))
			SetFileName(ProcessWorkingDir, FileName::FromPathNameLinux(String::From_c_str(cwdbuf)));
    }

	if (XSP::AppCmdLine::GetARGC() > 0)
	{
		FileName fn(FileName::FromPathNameLinux(String::From_c_str(XSP::AppCmdLine::GetARGV(0))));
		SetFileName(AppName, fn);
		if (fn.HasParent())
		{
			fn = fn.GetParent();
			SetFileName(AppDir, fn);
		}
	}	
//	{ SetFileName(RecycleBinDir, FileName::FromPathNameMac(_GetFolderName(kTrashFolderType))); }
//	{ SetFileName(StartupDir, FileName::FromPathNameMac(_GetFolderName(kStartupFolderType))); }
//	{ SetFileName(ShutdownDir, FileName::FromPathNameMac(_GetFolderName(kShutdownFolderType))); }
//	{ SetFileName(MenuDir, FileName::FromPathNameMac(_GetFolderName(kAppleMenuFolderType))); }
//	{ SetFileName(ControlPanelDir, FileName::FromPathNameMac(_GetFolderName(kControlPanelFolderType))); }
}


FileForkList::FileForkList(const FileName& fname)
: ref(new _Data())
{
	ref->filename = fname;
	if (fname.Exists())
	{
		ref->items.push_back(String::kEmpty);   // data fork is all that linux has
	}
}
   

FileList::FileList(const FileName& fname)
: ref(new _Data())
{
    String fnOS(fname.GetFullPathNameLinux(true));
    DIR* dir = ::opendir(fnOS.c_str());
    if (dir == NULL)
    	_LinuxFS_Fail(errno, fname);  // ENOENT
	for(;;)
	{
		struct dirent dirEntry;
		struct dirent* r;
		int err = readdir_r(dir, &dirEntry, &r);
		if (r == NULL)
			break;
		if (err != 0)
			_LinuxFS_Fail(err, fname);

		// ignore . and ..
		bool fIsDir = (r->d_type == DT_DIR);
		uint32 len = _D_EXACT_NAMLEN(r);
		if (fIsDir && (len <= 2) && (len >= 1) && (r->d_name[0] == '.'))
		{
		   if (len == 1)
		   		continue;
		   else if ((len == 2) && (r->d_name[1] == '.'))
		   		continue; 
		}

		FileProperties props(FileName::From(fname, String(r->d_name, len)));
//		props.SetFileName(FileName::From(fname, String(r->d_name, r->d_namlen)));	
	    ref->items.push_back(props);
	}
    ::closedir(dir);
}


FileProperties::FileProperties(const FileName& fname)
: ref(new _Data())
{
    String fnOS(fname.GetFullPathNameLinux());
    struct stat sb;
    if (-1 == ::lstat(fnOS.c_str(), &sb))
    	_LinuxFS_Fail(errno, fname);  
    // with lstat we don't get links automatically followed
    // for the file the link points to use stat,symlink    
    
   	SetFileName(fname);
	switch (sb.st_mode & S_IFMT)
	{
	case S_IFDIR:	
    	SetProperty(FileProperties::IsDir, String::kEmpty);
    	break;
	case S_IFLNK:	// put the real file here
    	SetProperty(FileProperties::IsLink, String::kEmpty);
    	break;
    case S_IFREG:
    	break;
    default:  // not a regular file ... 
    	break;
    };
//	SetProperty(FileProperties::IsHidden, String::kEmpty);
    SetProperty(FileProperties::Size, String::From_uint64(sb.st_size));
    SetProperty(FileProperties::UID, String::From_uint32(sb.st_uid));
    SetProperty(FileProperties::GID, String::From_uint32(sb.st_gid));
    SetProperty(FileProperties::Mode, String::From_uint32(sb.st_mode & ALLPERMS));
									 
    LocalTime dt(LocalTime::FromSecondsSince_1970Jan1(sb.st_ctime));
    SetProperty( FileProperties::CreationTime, dt.ToString());
    dt = LocalTime::FromSecondsSince_1970Jan1(sb.st_mtime);
    SetProperty( FileProperties::ModifyTime, dt.ToString());
    dt = LocalTime::FromSecondsSince_1970Jan1(sb.st_atime);
    SetProperty( FileProperties::AccessTime, dt.ToString());
}

#pragma mark -

bool FileName::Exists() const
{
    String fnOS(GetFullPathNameLinux(false));
    struct stat sb;
    if (-1 == ::stat(fnOS.c_str(), &sb))
    	return false;
    	// _LinuxFS_Fail(errno, fname);   // ENOENT
    return true;
}

bool FileName::IsDir() const
{
    String fnOS(GetFullPathNameLinux(true));
    struct stat sb;
    if (-1 == ::stat(fnOS.c_str(), &sb))
    	return false;
    	// _LinuxFS_Fail(errno, fname);   // ENOENT
    return (S_ISDIR(sb.st_mode) != 0);
}

bool FileName::IsLink() const
{
    String fnOS(GetFullPathNameLinux(false));
    struct stat sb;
    if (-1 == ::lstat(fnOS.c_str(), &sb))
    	return false;
    	// _LinuxFS_Fail(errno, fname);   // ENOENT
    return (S_ISLNK(sb.st_mode) != 0);
}

void FileName::Delete() const
{
    String fnOS(GetFullPathNameLinux(false));
    if (-1 == ::unlink(fnOS.c_str()))
    {
    	int err = errno;
    	if (err != ENOENT)  // if does not exist ... we just move on
    	{
    		if (-1 == ::rmdir(fnOS.c_str()))
    			_LinuxFS_Fail(err, *this);
    	}
    }  
}

void FileName::CreateDir(bool deep) const
{
    String fnOS(GetFullPathNameLinux(true));
    int err = 0;
    if (-1 == ::mkdir(fnOS.c_str(), S_IRWXU|S_IRWXG|S_IRWXO))
	{
    	err = errno;
		if (deep && HasParent())
		{   // recursively deal with the parent
			GetParent().CreateDir(true); 
			CreateDir(false); // try again
		}
		else
		{
			_LinuxFS_Fail(err, *this); 
	    }
	}
}


#pragma mark -

class OpenedFile::_Data : public ref_obj
{
public:
	 int handle;
	 FileName filename;
	 
	 ~_Data()
	 {
	    ::close( handle );
	 }
};

OpenedFile::OpenedFile( const FileName& fn, 
						const AccessType& acc, 
						const String& /*fork*/  )
{
    int perm = (&acc==&ReadOnlyShared) ? (O_RDONLY) 
    		   : (&acc==&ReadWriteExclusive) ? (O_RDWR|O_CREAT|O_EXCL)
    		   : (&acc==&ReadWriteShared) ? (O_RDWR |O_CREAT)
    		   : 0;

    String fnOS(fn.GetFullPathNameLinux(false));
    int h = ::open(fnOS.c_str(), perm, DEFFILEMODE);
	if (h == -1)
		_LinuxFS_Fail(errno, fn); 

    ref = new _Data();
    ref->handle = h;
    ref->filename = fn;
}

OpenedFile::~OpenedFile()
{
}

OpenedFile::file_size OpenedFile::GetSize() const
{
	off_t z = ::lseek(ref->handle, 0, SEEK_END);
	if (z == -1)
		_LinuxFS_Fail(errno, ref->filename); 
    return z;
}

void OpenedFile::SetSize(OpenedFile::file_size fsz) const
{
    off_t pz = static_cast<sint32>(fsz);
	VERIFY((pz == fsz) && (pz >= 0));
	off_t z = ::lseek(ref->handle, 0, SEEK_END);
	if (z == -1)
		_LinuxFS_Fail(errno, ref->filename); 
	if (z != pz)
	{
		if (pz > z)
		{
		  z = ::lseek(ref->handle, pz-1, SEEK_SET);	   // fill with 0
		  if (z == -1)
			_LinuxFS_Fail(errno, ref->filename); 
		  z = ::write(ref->handle, "", 1);			   // write the last 0
		  if (z == -1)
			_LinuxFS_Fail(errno, ref->filename); 
		}
		else if (pz < z)
		{
//		  z = ::fcntl(ref->handle, F_SETSIZE, pz);	   // truncate
			z = ::ftruncate(ref->handle, pz); 
		    if (z == -1)
			  _LinuxFS_Fail(errno, ref->filename);
		}
	}
}

OpenedFile::block_size 
	OpenedFile::Read(OpenedFile::file_size pos, 
					  void* bf, 
					  OpenedFile::block_size z) const
{
    off_t fp = static_cast<sint32>(pos);
    ssize_t bz = static_cast<sint32>(z);
	VERIFY((fp == pos) && (fp >= 0) && (bz >= 0));
	if (bz == 0)
		return 0;
    ssize_t r = ::pread(ref->handle, bf, bz, fp);
    if (r == -1)
		_LinuxFS_Fail(errno, ref->filename); 
    return r;
}

OpenedFile::block_size 
	OpenedFile::Write( OpenedFile::file_size pos, 
					   const void* bf, 
					   OpenedFile::block_size z) const
{
    off_t fp = static_cast<sint32>(pos);
    ssize_t bz = static_cast<sint32>(z);
	VERIFY((fp == pos) && (fp >= 0) && (bz >= 0));
	if (bz == 0)
		return 0;
    ssize_t r = ::pwrite(ref->handle, bf, bz, fp);
    if (r == -1)
		_LinuxFS_Fail(errno, ref->filename); 
    return r;
}

#endif
} // namespace XSP
