@Make(manual)
@Section[Synopsis]
@begin(verbatim)
include <ntcp.h>		/* include files for programs using TCP */
include <ntcpblk.h>
@end(verbatim)
@Section[@i{tcp@ux( )init()}]

@begin(verbatim)
int tcp_ init(stacksize)
        int stacksize;		/* stack size passed to Netinit */
@end(verbatim)

This routine initializes TCP, IP, UDP, tasking, timers, name resolution
procedures, e.g. all the lower layers of the network user interface.
Stacksize is the size of the stack parameter passed down to Netinit; it
should be a value like 800.  Tcp@ux( )init returns TRUE if initialization 
worked, or FALSE if initialization failed.

@Section[@i{tcp@ux( )open()}]

@begin(verbatim)
TcpCon tcp_ open(fh,fs,ls,win,lowwin,us_procs)
	in_ name fh;		/* foreign host to open to */
	unsigned fs;		/* foreign socket to open on */
	unsigned ls;		/* local socket to open on, see below */
	unsigned win;		/* window size, see below */
	unsigned lowwin;	/* window low water mark */
	int (*us_ procs[])();	/* 6 upcall procedures, see below */
@end(verbatim)

Immediately tries to open a connection to the foreign host on the foreign
socket, from the local socket.  If the local socket is zero, then 
tcp@ux( )open picks an unused socket number, greater than 1000.  
Note that this routine returns immediately, and you must wait 
until the @i{us@ux( )open()} upcall (described later) happens
to be sure the connection is open.  

A pointer to the connection block is
returned, or NULL is returned when one of these errors occur:  (1)
the foreign host/socket pair is already connected to the local socket (2),
there is already a listening TCP on the local socket, or (3) if a 
memory allocation fails.

The advertised window size is a number less than or equal to 2048.  Currently,
the size of the local window is fixed at 2048 bytes, so specifying a smaller 
window doesn't save memory.

The window's low water mark is tested when the local
input buffer empties.  When the amount of data drops below the low water 
mark, the larger window is advertised to the foreign host.
Until then, small emptyings 
of a nearly-full local window aren't advertised.  If these small emptyings
were advertised, tiny, inefficient packets might be sent by the foreign host,
causing the "silly window syndrome".  As long as "lowwin" is
significantly less than "win", like 1/2, this syndrome should never occur.

The array of six procedures that TCP upcalls is defined as follows:

@subsection(Opening a Connection)
@begin(verbatim)
us_ open(con)
	TcpCon con;
@end(verbatim)
This is upcalled once the TCP reaches the ESTABLISHED state, i.e. once
both the local and foreign hosts are able to send data.  Most applications
should use this routine to wake a task that writes data to the network.

@subsection(Timeouts)
@begin(verbatim)
us_ timeout(con,tnumber)
	TcpCon con;
	int tnumber;		/* timeout number -1,-2,-3.. or 1,2,3.. */
@end(verbatim)
This procedure is called when a timeout occurs.  
If the connection is not yet established, each timeout upcalls the
negated timeout count to us@ux( )timout().
If the connection is
established, the 9th, 10th, 11th ... timeout passes 1, 2, 3,... to
the us@ux( )timeout() procedure.

@subsection(Reading Network Data)
@begin(verbatim)
int us_ dispose(con,str,lth)
	TcpCon con;
	char *str;		/* pointer to string to dispose of */
	int lth;		/* length of string -- currently 100 max */
@end(verbatim)
Called to send a chunk of data to the user's program.  This
TCP does not does not 
acknowledge data until the @i{us@ux( )dispose()} upcall returns.  This
procedure MUST return the size of the new local window, which should
be the computed as the number of empty bytes in the local buffer.
Once the dispose
procedure returns zero, no more upcalls will occur.  In that case, the
@i{Tcp_New_Window()} macro can be used to reopen the window, causing more
data to be upcalled.

@subsection(When to Send More Data)
@begin(verbatim)
us_ buff(con)
	TcpCon con;
@end(verbatim)
This procedure is called when the output buffer has filled and buffer space
subsequently becomes available.  Before a connection is open, it is legal
to write to the connection, because the writing 
process is immediately blocked by TCP.
Also, when TCP determines the foreign window
is filled, or when the current output packet fills, any attempt to
write to the network blocks the writing process.  In all 3 cases, this
upcall signals that the blocking reason has disappeared.  This user procedure
should always awaken those blocked processes.  That is the main use of
this upcall.

@subsection(The Foreign Host's Close)
@begin(verbatim)
us_ forclose(con)
	TcpCon con;
@end(verbatim)
The @i(f)oreign @i(close) procedure is called @i(if) the foreign host sends
a FIN first, i.e. once the foreign host has closed its half of the connection.
Then, from this procedure, it is easy to immediately close the local side.

@subsection(Both Sides Closed)
@begin(verbatim)
us_ close(con,status)
	TcpCon con;
	int status;
@end(verbatim)
This upcall signals that both sides have closed, and
the connection block is erased.  Status is TRUE if the connection closed
normally, FALSE if the connection closed abnormally because of a TCP RESET.

@Section[@i{tcp@ux( )listen()}]
@begin(verbatim)
TcpCon tcp_ listen(ls,win,lowwin,pblock)
	unsigned ls;		/* local socket to listen to */
	unsigned win;		/* window to advertise */
	unsigned lowwin;	/* low water mark on window */
	int (*pblock[@i{}])();	/* pointers to 6 upcallable user procedures */
@end(verbatim)

This causes a TCP to listen to the specified local socket.
Once called,
it is illegal to open active connections on that socket.  Unlike some
listening TCP's that fork an active TCP for each connection request, 
a connection request causes this listening TCP
to become active.  If the user wants to continue listening
for more connections, it is a good idea to call tcp@ux( )listen from the 
us@ux( )open function to immediately re-listen to the socket.
If that is not done, further foreign opens will be reset.

The upcallable procedures are exactly the same as for tcp@ux( )open,
except that connection attempt timeouts (@i{us@ux[ ]timeout()} with a negative
value) are impossible.

This function returns a pointer to the connection block.  It returns NULL if
some TCP is already listening to the local socket, or if a memory allocation
fails.

@Section[@i{tprintf()}]

@begin(verbatim)
tprintf(con,string,..)
	TcpCon con;
	char *string;
@end(verbatim)

This function is analogous to UNIX's sprintf.  This procedure may block
after sending a portion of the string, if the foreign window fills, if
the single output packet fills, or if the connection is not yet established.
It must be awakened during a us@ux( )buff() upcall.

@Section[@i{tputs()}]

@begin(verbatim)
tputs(con,string)
	TcpCon con;
	char *string;
@end(verbatim)

This function is analogous to UNIX's puts function, except that you must
specify a network connection.  Again, this procedure will block and should
be waked during a us@ux( )buff() upcall.

@Section[@i{tputc()}]

@begin(verbatim)
tputc(con,c)
	TcpCon con;
	char c;
@end(verbatim)

This MACRO tries to write the character to the network, and blocks
if it cannot.  It must be awakened within a us@ux( )buff() upcall,
and will repeat its attempt to write to the network.  NOTE:  this
macro DOES NOT awaken the send task (to send the character), unless
the buffer fills (i.e. this macro always buffers data).  To be safe,
use tcp_newdata(con) after putting characters onto the network.

@Section[@i{tcp@ux( )close()}]

@begin(verbatim)
tcp_ close(con)
	TcpCon con;
@end(verbatim)

This procedure closes the local side of the current connection, by placing
a FIN in the outgoing packet.  Note that this routine returns immediately
and the user must wait for a close upcall to inform him when the connection is
fully closed.  If this routine is called more than once per connection, an
error message is printed.

@section[@i{tcp@ux( )urgent()}]

@begin(verbatim)
tcp_ urgent(con)
	TcpCon con;
@end(verbatim)

This procedure specifies that all the data in the outgoing packet is urgent.
Subsequent tcp@ux( )urgent calls are always legal, and can extend the length 
of the urgent portion of data.

@section[@i{tcp@ux( )clean()}]
@begin(verbatim)
tcp_clean(con,reason)
	TcpCon con;
	int reason;
@end(verbatim)
Totally erases a connection and upcalls the us@ux( )close procedure,
passing the reason as a value.  As a user function, this is only
useful for destroying listening TCP's, and for destroying active
TCP's when the foreign host is down, or never responds.

@section[@i{Macros}]
@begin(verbatim)
int *tcp_getu(con)
	TcpCon con;

tcp_ setu(con,ptr)
	TcpCon con;
	int *intptr;
@end(verbatim)

Each connection has an integer poinrter user defined field that can be 
accessed and set using the above macros.  One good value to put in this 
field is a pointer to the task that will be sending data through the 
network.  This pointer is supplied to tk_wake() when the us@ux( )buff() 
upcall occurs.

@begin(verbatim)
tcp_new_wind(con,value)
	TcpCon con;
	int value;		/* can't be less than zero */
@end(verbatim)

This call changes the value of the local window.  It should ONLY be used
to open a local window that has shrunk to zero, and is the only way to
do so.  If this macro is used at other times, poor performance and confusion 
from the foreign host may result.

@Section(Legal Recursive Downcalls)
@begin(verbatim)
You can				From These Procedures
Downcall
               us_ open us_ forclose us_ timeout us_ dispose us_ buff us_ close
tcp_ init       impossible - - - - - - - - - - - - - - - - - - - - - - -
tcp_ open       ok       ok        ok           ok            ok      ok
tcp_ listen     ok       ok        ok           ok            ok      ok
tprintf        *        *         *            *             *       illegal
tputs          *        *         *            *             *       illegal
tputc          *        *         *            *             *       illegal
tcp_ close      ok       ok        ok           ok            ok      illegal
tcp_ urg        ok       ok        ok           ok            ok      illegal
tcp_ clean      illegal  illegal   ok           illegal       illegal illegal

* = illegal since tprintf, tputs, and tputc can block, deadlocking
    the packet handler.  This would be fixed if TCP forked before
    each upcall, or (more efficiently) if a fork occurred only when
    TCP detected potential deadlock in tprintf, tputs, and tputc.
@end(verbatim)
