/*

        Specify the solution to a maze and obtain the maze.  A CGA (or better)
   graphics adapter is needed.  Mazes up to 39 columns wide may be printed
   on 80 column printers supporting the IBM graphics character set.

        Written (in Microsoft C 6.0) by James L. Dean on July 11, 1994.

*/

#include <stdio.h>
#include <stdlib.h>
#include <graph.h>
#include <conio.h>

#define TRUE -1
#define FALSE 0

static void generate_maze(int,int,int,int,char **,int *);
static void get_solution(int,int,int,int,char **,int *);
static void initialize(int,int,int *,int *,int *,char ***);
       void main(int,char **);
static void write_hex(FILE **,char *,int *);
static void write_maze(int,int,char **,int,char **,int *);

/* Graphics Constants */
#define VIDEOMODE            _HRESBW
#define MAX_X                    639
#define MAX_Y                    199

typedef struct stack_node
                 {
                   int               delta_index_1;       
                   int               delta_index_2;
                   struct stack_node *next;
                 } *stack_node_ptr;

typedef struct task_node
                 {
                   int               delta_index_1;       
                   int               delta_index_2;
                   int               done;
                   int               recurse;
                   struct stack_node *stack_head;
                   int               x;
                   int               y;
                   struct task_node  *next;
                 } *task_node_ptr;

void main(
  int  argc,
  char *argv[])
    {
      static int  aborted;
      static int  fatal_error;
      static int  num_columns;
      static int  num_rows;
      static char **page;
      static int  response;
      static int  x;
  
      fatal_error=FALSE;
      initialize(MAX_X,MAX_Y,&fatal_error,&num_columns,&num_rows,&page);
      if (! fatal_error)
        {
          if (_setvideomode(VIDEOMODE))
            {
              aborted=FALSE;
              get_solution(MAX_X,MAX_Y,num_columns,num_rows,page,&aborted);
              if (! aborted)
                {
                  generate_maze(MAX_X,MAX_Y,num_columns,num_rows,page,
                   &fatal_error);
                  if (! fatal_error)
                    {
                      response=getch();
                      if (response == 0)
                        getch();
                    }
                }
              _setvideomode(_DEFAULTMODE);
              if (fatal_error)
                printf("     Fatal error:  out of memory.\n");
            }
          else
            {
              fatal_error=TRUE;
              printf(
        "     Fatal error:  high resolution CGA graphics are not available.\n");
            }
          if (! fatal_error)
            write_maze(num_columns,num_rows,page,argc,argv,&fatal_error);
          for (x=2*num_columns; x >= 0; x--)
            free((void *) (page[x]));
          free((void *) page);
        }
      return;
    }

static void initialize(
  int  max_x,
  int  max_y,
  int  *fatal_error,
  int  *num_columns,
  int  *num_rows,
  char ***page)
    {
      static int max_num_columns;
      static int max_num_rows;
      static int twice_num_columns_plus_1;
      static int twice_num_rows_plus_1;
      static int x;

      _clearscreen(_GCLEARSCREEN);
      printf("                                 Design a Maze\n");
      printf("\n");
      printf("\n");
      printf("\n");
      printf(
"     You will be prompted for the number of columns and the number of\n");
      printf(
"rows.  Then you may use the arrow keys to specify the solution to the\n");
      printf(
"maze.  The solution must run from from the upper left hand corner to the\n");
      printf(                    
"lower right hand corner.  You may press Escape to abort specifying the\n");
      printf(
"solution.  After the solution is specified, the maze will be displayed.\n");
      printf(
"Press the space bar to exit; a copy of the maze will be saved as\n");
      printf(
"MAKEMAZE.OUT.\n");
      printf("\n");
      max_num_columns=max_x/2;
      do
        {
          printf("     Number of columns (2 to ");
          printf("%d",max_num_columns);
          printf(")? ");
          fflush(stdin);
          scanf("%d",num_columns);
          if ((*num_columns < 2)
          ||  (*num_columns > max_num_columns))
            {
              printf(
               "? The number of columns must be between 2 and ");
              printf("%d",max_num_columns);
              printf(", inclusively\n");
            }
        }
      while ((*num_columns < 2)
      ||     (*num_columns > max_num_columns));
      printf("\n");
      twice_num_columns_plus_1=2*(*num_columns)+1;
      if ((*page=(char **) malloc(twice_num_columns_plus_1*sizeof(char *)))
       == NULL)
        {
          *fatal_error=TRUE;
          printf("     Fatal error:  out of memory.\n");
        }
      if (! *fatal_error)
        {
          max_num_rows=max_y/2;
          do
            {
              printf("     Number of rows (2 to ");
              printf("%d",max_num_rows);
              printf(")? ");
              fflush(stdin);
              scanf("%d",num_rows);
              if ((*num_rows < 2) || (*num_rows > max_num_rows))
                {
                  printf(
                   "? The number of rows must be between 2 and ");
                  printf("%d",max_num_rows);
                  printf(", inclusively\n");
                }
            }
          while ((*num_rows < 2) || (*num_rows > max_num_rows));
          printf("\n");
          twice_num_rows_plus_1=2*(*num_rows)+1;
          for (x=0; ((! *fatal_error) && (x < twice_num_columns_plus_1)); x++)
            if (((*page)[x]=(char *) malloc(twice_num_rows_plus_1*sizeof(char)))
             == NULL)
              {
                *fatal_error=TRUE;
                printf("     Fatal error:  out of memory.\n");
                while (x > 0)
                  free((void *) ((*page)[--x]));
                free((void *) *page);
              }
        }
      return;
    }

static void get_solution(
  int  max_x,
  int  max_y,
  int  num_columns,
  int  num_rows,
  char **page,
  int  *aborted)
    {
      static int delta_index;
      static int delta_x [4];
      static int delta_y [4];
      static int key_pressed;
      static int passage_found;
      static int twice_num_columns;
      static int twice_num_columns_plus_1;
      static int twice_num_rows;
      static int twice_num_rows_plus_1;
      static int x;
      static int x_increment;
      static int x_next;
      static int x_out;
      static int y;
      static int y_increment;
      static int y_next;
      static int y_out;
      
      *aborted=FALSE;
      twice_num_columns=2*num_columns;
      twice_num_columns_plus_1=twice_num_columns+1;
      twice_num_rows=2*num_rows;
      twice_num_rows_plus_1=twice_num_rows+1;
      for (x=0; x < twice_num_columns_plus_1; x++)
        for (y=0; y < twice_num_rows_plus_1; y++)
          page[x][y]=' ';
      _setcolor((short) 1);
      x_increment=max_x/num_columns;
      y_increment=max_y/num_rows;
      x_out=num_columns*x_increment;
      y_out=0;
      for (y=0; y < twice_num_rows_plus_1; y+=2)
        { 
          for (x=0; x < twice_num_columns_plus_1; x++)
            page[x][y]='W';
          _moveto((short) 0,(short) y_out);
          _lineto((short) x_out,(short) y_out);
          y_out+=y_increment;
        }
      y_out=num_rows*y_increment;
      x_out=0;
      for (x=0; x < twice_num_columns_plus_1; x+=2)
        { 
          for (y=0; y < twice_num_rows_plus_1; y++)
            page[x][y]='W';
          _moveto((short) x_out,(short) 0);
          _lineto((short) x_out,(short) y_out);
          x_out+=x_increment;
        }
      _setcolor((short) 0);
      _moveto((short) 1,(short) 0);
      _lineto((short) (x_increment-1),(short) 0);
      page[1][0]='S';
      x_out=num_columns*x_increment-x_increment+1;
      _moveto((short) x_out,(short) y_out);
      x_out=num_columns*x_increment-1;
      _lineto((short) x_out,(short) y_out);
      page[twice_num_columns-1][twice_num_rows]='S';
      delta_x[0]=1;
      delta_y[0]=0;
      delta_x[1]=0;
      delta_y[1]=1;
      delta_x[2]=-1;
      delta_y[2]=0;
      delta_x[3]=0;
      delta_y[3]=-1;
      x=1;
      y=1;
      page[x][y]='S';
      do
        {
          do
            {
              passage_found=TRUE;
              key_pressed=getch();
              if (key_pressed == 0)
                {
                  key_pressed=getch();
                  switch (key_pressed)
                    {
                      case 72:
                         delta_index=3;
                         break;
                      case 77:
                         delta_index=0;
                         break;
                      case 80:
                         delta_index=1;
                         break;
                      case 75:
                         delta_index=2;
                         break;
                      default:
                         passage_found=FALSE;
                         break;
                    }
                }
              else
                {
                  switch (key_pressed)
                    {
                      case 27:
                        passage_found=FALSE;
                        *aborted=TRUE;
                        break;
                      case 56:
                        delta_index=3;
                        break;
                      case 54:
                        delta_index=0;
                        break;
                      case 50:
                        delta_index=1;
                        break;
                      case 52:
                        delta_index=2;
                        break;
                      default:
                        passage_found=FALSE;
                        break;
                    }
                }
              if (passage_found)
                {
                  x_next=x+delta_x[delta_index];
                  if (x_next <= 0)
                    passage_found=FALSE;
                  else
                    if (x_next >= twice_num_columns)
                      passage_found=FALSE;
                    else
                      {
                        y_next=y+delta_y[delta_index];
                        if (y_next <= 0)
                          passage_found=FALSE;
                        else
                          if (y_next >= twice_num_rows)
                            passage_found=FALSE;
                          else
                            if (page[x_next][y_next] == 'S')
                              { /* back out part of solution */
                                _setcolor((short) 1);
                                page[x][y]=' ';
                                page[x_next][y_next]='W';
                                if (x == x_next)
                                  {
                                    y_out=y_increment*y_next/2;
                                    x_out=x_increment*(x-1)/2+1;
                                    _moveto((short) x_out,(short) y_out);
                                    x_out=x_increment*(x+1)/2-1;
                                    _lineto((short) x_out,(short) y_out);
                                  }
                                else
                                  {
                                    x_out=x_increment*x_next/2;
                                    y_out=y_increment*(y-1)/2+1;
                                    _moveto((short) x_out,(short) y_out);
                                    y_out=y_increment*(y+1)/2-1;
                                    _lineto((short) x_out,(short) y_out);
                                  }
                                x=x_next+delta_x[delta_index];
                                y=y_next+delta_y[delta_index];
                              }
                            else
                              if (page[x_next+delta_x[delta_index]][
                               y_next+delta_y[delta_index]] == 'S')
                                passage_found=FALSE;
                              else
                                { /* add part of solution */
                                  _setcolor((short) 0);
                                  page[x_next][y_next]='S';
                                  if (x == x_next)
                                    {
                                      y_out=y_increment*y_next/2;
                                      x_out=x_increment*(x-1)/2+1;
                                      _moveto((short) x_out,(short) y_out);
                                      x_out=x_increment*(x+1)/2-1;
                                      _lineto((short) x_out,(short) y_out);
                                    }
                                  else
                                    {
                                      x_out=x_increment*x_next/2;
                                      y_out=y_increment*(y-1)/2+1;
                                      _moveto((short) x_out,(short) y_out);
                                      y_out=y_increment*(y+1)/2-1;
                                      _lineto((short) x_out,(short) y_out);
                                    }
                                  x=x_next+delta_x[delta_index];
                                  y=y_next+delta_y[delta_index];
                                  page[x][y]='S';
                                }
                      }
                }
            }
          while ((! passage_found) && (! *aborted));
        }
      while (((x != twice_num_columns-1)
           || (y != twice_num_rows-1))
      &&     (! *aborted));
    }

static void generate_maze(
  int  max_x,
  int  max_y,
  int  num_columns,
  int  num_rows,
  char **page,
  int  *fatal_error)
    {
      static int            delta_index;
      static int            delta_index_2a;
      static int            delta_index_2b;
      static int            delta_index_2c;
      static int            delta_index_2d;
      static int            delta_x [24] [4];
      static int            delta_y [24] [4];
      static int            some_task;
      static stack_node_ptr stack_ptr;
      static task_node_ptr  task_head;
      static int            task_possible;
      static task_node_ptr  task_ptr;
      static int            tasks_done;
      static int            twice_num_columns;
      static int            twice_num_columns_plus_1;
      static int            twice_num_rows;
      static int            twice_num_rows_plus_1;
      static int            x;
      static int            x_increment;
      static int            x_next;
      static int            x_out;
      static int            y;
      static int            y_increment;
      static int            y_next;
      static int            y_out;
  
      twice_num_columns=2*num_columns;
      twice_num_columns_plus_1=twice_num_columns+1;
      twice_num_rows=2*num_rows;
      twice_num_rows_plus_1=twice_num_rows+1;
      x_increment=max_x/num_columns;
      y_increment=max_y/num_rows;
      delta_x[0][0]=1;
      delta_y[0][0]=0;
      delta_x[0][1]=0;
      delta_y[0][1]=1;
      delta_x[0][2]=-1;
      delta_y[0][2]=0;
      delta_x[0][3]=0;
      delta_y[0][3]=-1;
      delta_index=0;
      for (delta_index_2a=0; delta_index_2a < 4; delta_index_2a++)
        for (delta_index_2b=0; delta_index_2b < 4; delta_index_2b++)
          if (delta_index_2a != delta_index_2b)
            for (delta_index_2c=0; delta_index_2c < 4;
             delta_index_2c++)
              if ((delta_index_2a != delta_index_2c)
              &&  (delta_index_2b != delta_index_2c))
                for (delta_index_2d=0; delta_index_2d < 4;
                 delta_index_2d++)
                  if ((delta_index_2a != delta_index_2d)
                  &&  (delta_index_2b != delta_index_2d)
                  &&  (delta_index_2c != delta_index_2d))
                    {
                      delta_x[delta_index][delta_index_2a]
                       =delta_x[0][0];
                      delta_y[delta_index][delta_index_2a]
                       =delta_y[0][0];
                      delta_x[delta_index][delta_index_2b]
                       =delta_x[0][1];
                      delta_y[delta_index][delta_index_2b]
                       =delta_y[0][1];
                      delta_x[delta_index][delta_index_2c]
                       =delta_x[0][2];
                      delta_y[delta_index][delta_index_2c]
                       =delta_y[0][2];
                      delta_x[delta_index][delta_index_2d]
                       =delta_x[0][3];
                      delta_y[delta_index++][delta_index_2d]
                       =delta_y[0][3];
                    }
      task_head=NULL;
      some_task=FALSE;
      task_possible=TRUE;
      while ((! *fatal_error) && (! some_task) && (task_possible))
        {
          task_possible=FALSE;
          y=0;
          for (x=2; ((! *fatal_error) && (x < twice_num_columns)); x+=2)
            if (page[x][1] != 'S')
              {
                task_possible=TRUE;
                if ((page[x-1][1] == 'S')
                ||  (page[x+1][1] == 'S')
                ||  (rand()%4 == 0))
                  {
                    if ((task_ptr=(struct task_node *)
                     malloc(sizeof(struct task_node)))
                     == NULL)
                      *fatal_error=TRUE;
                    else
                      {
                        task_ptr->delta_index_1=0;
                        task_ptr->delta_index_2=0;
                        task_ptr->done=FALSE;
                        task_ptr->recurse=TRUE;
                        task_ptr->stack_head=NULL;
                        task_ptr->x=x;
                        task_ptr->y=y;
                        task_ptr->next=task_head;
                        task_head=task_ptr;
                        some_task=TRUE;
                      }
                  }
              }
          x=twice_num_columns;
          for (y=2; ((! *fatal_error) && (y < twice_num_rows)); y+=2)
            if (page[x-1][y] != 'S')
              {
                task_possible=TRUE;
                if ((page[x-1][y-1] == 'S')
                ||  (page[x-1][y+1] == 'S')
                ||  (rand()%4 == 0))
                  {
                    if ((task_ptr=(struct task_node *)
                     malloc(sizeof(struct task_node)))
                     == NULL)
                      *fatal_error=TRUE;
                    else
                      {
                        task_ptr->delta_index_1=0;
                        task_ptr->delta_index_2=0;
                        task_ptr->done=FALSE;
                        task_ptr->recurse=TRUE;
                        task_ptr->stack_head=NULL;
                        task_ptr->x=x;
                        task_ptr->y=y;
                        task_ptr->next=task_head;
                        task_head=task_ptr;
                        some_task=TRUE;
                      }
                  }
              }
        }
      some_task=FALSE;
      task_possible=TRUE;
      while ((! *fatal_error) && (! some_task) && (task_possible))
        {
          task_possible=FALSE;
          x=0;
          for (y=2; ((! *fatal_error) && (y < twice_num_rows)); y+=2)
            if (page[1][y] != 'S')
              {
                task_possible=TRUE;
                if ((page[1][y-1] == 'S')
                ||  (page[1][y+1] == 'S')
                ||  (rand()%4 == 0))
                  {
                    if ((task_ptr=(struct task_node *)
                     malloc(sizeof(struct task_node)))
                     == NULL)
                      *fatal_error=TRUE;
                    else
                      {
                        task_ptr->delta_index_1=0;
                        task_ptr->delta_index_2=0;
                        task_ptr->done=FALSE;
                        task_ptr->recurse=TRUE;
                        task_ptr->stack_head=NULL;
                        task_ptr->x=x;
                        task_ptr->y=y;
                        task_ptr->next=task_head;
                        task_head=task_ptr;
                        some_task=TRUE;
                      }
                  }
              }
          y=twice_num_rows;
          for (x=2; ((! *fatal_error) && (x < twice_num_columns)); x+=2)
            if (page[x][y-1] != 'S')
              {
                task_possible=TRUE;
                if ((page[x-1][y-1] == 'S')
                ||  (page[x+1][y-1] == 'S')
                ||  (rand()%4 == 0))
                  {
                    if ((task_ptr=(struct task_node *) 
                     malloc(sizeof(struct task_node)))
                     == NULL)
                      *fatal_error=TRUE;
                    else
                      {
                        task_ptr->delta_index_1=0;
                        task_ptr->delta_index_2=0;
                        task_ptr->done=FALSE;
                        task_ptr->recurse=TRUE;
                        task_ptr->stack_head=NULL;
                        task_ptr->x=x;
                        task_ptr->y=y;
                        task_ptr->next=task_head;
                        task_head=task_ptr;
                        some_task=TRUE;
                      }
                  }
              }
        }
      _clearscreen(_GCLEARSCREEN);
      for (x=0; x < twice_num_columns_plus_1; x++)
        for (y=0; y < twice_num_rows_plus_1; y++)
          if (page[x][y] == 'W')
            page[x][y]=' ';
      _setcolor((short) 1);
      _moveto((short) 0,(short) 0);
      x_out=num_columns*x_increment;
      _lineto((short) x_out,(short) 0);
      y_out=num_rows*y_increment;
      _lineto((short) x_out,(short) y_out);
      _lineto((short) 0,(short) y_out);
      _lineto((short) 0,(short) 0);
      _setcolor((short) 0);
      _moveto((short) 1,(short) 0);
      _lineto((short) (x_increment-1),(short) 0);
      x_out=num_columns*x_increment-x_increment+1;
      _moveto((short) x_out,(short) y_out);
      x_out=num_columns*x_increment-1;
      _lineto((short) x_out,(short) y_out);
      for (x=0; x < twice_num_columns_plus_1; x++)
        {
          page[x][0]='W';
          page[x][twice_num_rows]='W';
        }
      for (y=0; y < twice_num_rows_plus_1; y++)
        {
          page[0][y]='W';
          page[twice_num_columns][y]='W';
        }
      tasks_done=FALSE;
      while ((! *fatal_error) && (! tasks_done))
        {
          tasks_done=TRUE;
          task_ptr=task_head;
          while ((! *fatal_error) && (task_ptr != NULL))
            {
              if (! (task_ptr->done))
                {
                  if (task_ptr->recurse)
                    {
                      task_ptr->delta_index_1=rand()%24;
                      task_ptr->delta_index_2=0;
                      task_ptr->recurse=FALSE;
                    }
                  while ((task_ptr->delta_index_2 < 4)
                  &&     (! (task_ptr->recurse))
                  &&     (! *fatal_error))
                    {
                      x_next=task_ptr->x
                       +delta_x[task_ptr->delta_index_1][
                       task_ptr->delta_index_2];
                      if ((x_next <= 0) || (x_next >= twice_num_columns))
                        (task_ptr->delta_index_2)++;
                      else
                        {
                          y_next=task_ptr->y
                           +delta_y[task_ptr->delta_index_1][
                           task_ptr->delta_index_2];
                          if ((y_next <= 0) || (y_next >= twice_num_rows))
                            (task_ptr->delta_index_2)++;
                          else
                            if (page[x_next][y_next] == ' ')
                              {
                                x_next
                                 +=delta_x[task_ptr->delta_index_1][
                                 task_ptr->delta_index_2];
                                if ((x_next <= 0) 
                                ||  (x_next >= twice_num_columns))
                                  (task_ptr->delta_index_2)++;
                                else
                                  {
                                    y_next
                                     +=delta_y[task_ptr->delta_index_1][
                                     task_ptr->delta_index_2];
                                    if ((y_next <= 0)
                                    ||  (y_next >= twice_num_rows))
                                      (task_ptr->delta_index_2)++;
                                    else
                                      if (page[x_next][y_next] == ' ')
                                        {
                                          page[(task_ptr->x+x_next)/2][
                                           (task_ptr->y+y_next)/2]='W';
                                          page[x_next][y_next]='W';
                                          _setcolor((short) 1);
                                          x_out=task_ptr->x*x_increment/2;
                                          y_out=task_ptr->y*y_increment/2;
                                          _moveto((short) x_out,(short) y_out);
                                          x_out=x_next*x_increment/2;
                                          y_out=y_next*y_increment/2;
                                          _lineto((short) x_out,(short) y_out);
                                          task_ptr->x=x_next;
                                          task_ptr->y=y_next;
                                          if ((stack_ptr=(struct stack_node *)
                                           malloc((unsigned)
                                           sizeof(struct stack_node))) == NULL)
                                            *fatal_error=TRUE;
                                          else
                                            {
                                              stack_ptr->next
                                               =task_ptr->stack_head;
                                              task_ptr->stack_head=stack_ptr;
                                              (task_ptr->stack_head)
                                               ->delta_index_1
                                               =task_ptr->delta_index_1;
                                              (task_ptr->stack_head)
                                               ->delta_index_2
                                               =task_ptr->delta_index_2;
                                              task_ptr->recurse=TRUE;
                                            }
                                        }
                                      else
                                        (task_ptr->delta_index_2)++;
                                  }
                              }
                            else
                              (task_ptr->delta_index_2)++;
                        }
                    }
                  if ((! task_ptr->recurse) && (! *fatal_error))
                    {
                      stack_ptr=task_ptr->stack_head;
                      if (stack_ptr == NULL)
                        task_ptr->done=TRUE;
                      else
                        {
                          task_ptr->delta_index_1
                           =(task_ptr->stack_head)->delta_index_1;
                          task_ptr->delta_index_2
                           =(task_ptr->stack_head)->delta_index_2;
                          task_ptr->stack_head=stack_ptr->next;
                          free((void *) stack_ptr);
                          if (task_ptr->stack_head == NULL)
                            task_ptr->done=TRUE;
                          else
                            {
                              task_ptr->x-=2*delta_x[task_ptr->delta_index_1][
                               task_ptr->delta_index_2];
                              task_ptr->y-=2*delta_y[task_ptr->delta_index_1][
                               task_ptr->delta_index_2];
                            }
                        }
                    }
                  tasks_done&=(task_ptr->done);
                }
              task_ptr=task_ptr->next;
            }
        }
      page[1][0]='S';
      page[twice_num_columns-1][twice_num_rows]='S';
      while (task_head != NULL)
        {
          task_ptr=task_head->next;
          free((void *) task_head);
          task_head=task_ptr;
        }
      return;
    }

static void write_hex(
  FILE **maze,
  char *hex,
  int  *fatal_error)
   {
     static int  byte;
     static char byte_in_hex [3];

     while ((*hex) && (! *fatal_error))
       {
         byte_in_hex[0]=*hex++;
         if (*hex)
           byte_in_hex[1]=*hex++;
         else
           byte_in_hex[1]='0';
         byte_in_hex[2]='\0';
         sscanf(&byte_in_hex[0],"%x",&byte);
         *fatal_error=(fputc(byte,*maze) != byte);
       }
     return;
   }

static void write_maze(
  int  num_columns,
  int  num_rows,
  char **page,
  int  argc,
  char *argv[],
  int  *fatal_error)
    {
             FILE *maze;
      static int  twice_num_columns;
      static int  twice_num_rows;
      static int  x;
      static int  x_minus_1;
      static int  x_plus_1;
      static int  y;
      static int  y_minus_1;
      static int  y_next;
      static int  y_plus_1;

      if ((maze=fopen("MAKEMAZE.OUT","w")) == NULL)
        {
          *fatal_error=TRUE;
          printf("     Fatal error:  cannot open MAKEMAZE.OUT for output.\n");
        }
      else
        {
          if (argc >= 2)
            write_hex(&maze,argv[1],fatal_error);
          twice_num_columns=2*num_columns;
          twice_num_rows=2*num_rows;
          if (! *fatal_error)
            *fatal_error=(fputc(0xb3,maze) != 0xb3);
          x=1;
          while ((! *fatal_error) && (x < twice_num_columns))
            {
              if (page[x][0] == 'W')
                *fatal_error=(fputc(0xc4,maze) != 0xc4);
              else
                *fatal_error=(fputc((int) ' ',maze) != ((int) ' '));
              x++;
              if ((! *fatal_error) && (x < twice_num_columns))
                {
                  if (page[x][1] == 'W')
                    *fatal_error=(fputc(0xc2,maze) != 0xc2);
                  else
                    *fatal_error=(fputc(0xc4,maze) != 0xc4);
                  x++;
                }
            }
          if (! *fatal_error)
            *fatal_error=(fputc(0xbf,maze) != 0xbf);
          if (! *fatal_error)
            *fatal_error=(fputc((int) '\n',maze) != ((int) '\n'));
          y=2;
          while ((! *fatal_error) && (y < twice_num_rows))
            {
              if (page[1][y] == 'W')
                *fatal_error=(fputc(0xc3,maze) != 0xc3);
              else
                *fatal_error=(fputc(0xb3,maze) != 0xb3);
              x=1;
              while ((! *fatal_error) && (x < twice_num_columns))
                {
                  if (page[x][y] == 'W')
                    *fatal_error=(fputc(0xc4,maze) != 0xc4);
                  else
                    *fatal_error=(fputc((int) ' ',maze) != ((int) ' '));
                  x++;
                  if ((! *fatal_error) && (x < twice_num_columns))
                    {
                      x_minus_1=x-1;
                      x_plus_1=x+1;
                      y_minus_1=y-1;
                      y_plus_1=y+1;
                      if (page[x_minus_1][y] == 'W')
                        if (page[x][y_minus_1] == 'W')
                          if (page[x][y_plus_1] == 'W')
                            if (page[x_plus_1][y] == 'W')
                              *fatal_error=(fputc(0xc5,maze) != 0xc5);
                            else
                              *fatal_error=(fputc(0xb4,maze) != 0xb4);
                          else
                            if (page[x_plus_1][y] == 'W')
                              *fatal_error=(fputc(0xc1,maze) != 0xc1);
                            else
                              *fatal_error=(fputc(0xd9,maze) != 0xd9);
                        else
                          if (page[x][y_plus_1] == 'W')
                            if (page[x_plus_1][y] == 'W')
                              *fatal_error=(fputc(0xc2,maze) != 0xc2);
                            else
                              *fatal_error=(fputc(0xbf,maze) != 0xbf);
                          else
                            *fatal_error=(fputc(0xc4,maze) != 0xc4);
                      else
                        if (page[x][y_minus_1] == 'W')
                          if (page[x][y_plus_1] == 'W')
                            if (page[x_plus_1][y] == 'W')
                              *fatal_error=(fputc(0xc3,maze) != 0xc3);
                            else
                              *fatal_error=(fputc(0xb3,maze) != 0xb3);
                          else
                            if (page[x_plus_1][y] == 'W')
                              *fatal_error=(fputc(0xc0,maze) != 0xc0);
                            else
                              {
                                y_next=y+2;
                                if (y_next < twice_num_rows)
                                  if ((page[x_minus_1][y_next] != 'W')
                                  &&  (page[x][y_next-1] != 'W')
                                  &&  (page[x_plus_1][y_next] != 'W')
                                  &&  (page[x][y_next+1] == 'W'))
                                   *fatal_error
                                    =(fputc((int) ' ',maze) != ((int) ' '));
                                  else
                                   *fatal_error=(fputc(0xb3,maze) != 0xb3);
                                else
                                  if (page[x][y_next-1] == 'W')
                                   *fatal_error
                                    =(fputc((int) ' ',maze) != ((int) ' '));
                                  else
                                   *fatal_error=(fputc(0xb3,maze) != 0xb3);
                              }
                        else
                          if (page[x][y_plus_1] == 'W')
                            if (page[x_plus_1][y] == 'W')
                              *fatal_error=(fputc(0xda,maze) != 0xda);
                            else
                              *fatal_error=(fputc(0xb3,maze) != 0xb3); 
                          else
                            if (page[x_plus_1][y] == 'W')
                              *fatal_error=(fputc(0xc4,maze) != 0xc4);
                            else
                              *fatal_error
                               =(fputc((int) ' ',maze) != ((int) ' '));
                      x=x_plus_1;
                    }
                }
              x_minus_1=twice_num_columns-1;
              if (! *fatal_error)
                {
                  if (page[x_minus_1][y] == 'W')
                    *fatal_error=(fputc(0xb4,maze) != 0xb4);
                  else
                    *fatal_error=(fputc(0xb3,maze) != 0xb3);
                }
              y+=2;
              if (! *fatal_error)
                *fatal_error=(fputc((int) '\n',maze) != ((int) '\n'));
            }
          if (! *fatal_error)
            *fatal_error=(fputc(0xc0,maze) != 0xc0);
          x=1;
          while ((! *fatal_error) && (x < twice_num_columns))
            {
              if (page[x][twice_num_rows] == 'W')
                *fatal_error=(fputc(0xc4,maze) != 0xc4);
              else
                *fatal_error=(fputc((int) ' ',maze) != ((int) ' '));
              x++;
              if ((! *fatal_error) && (x < twice_num_columns))
                {
                  y_minus_1=twice_num_rows-1;
                  if (page[x][y_minus_1] == 'W')
                    *fatal_error=(fputc(0xc1,maze) != 0xc1);
                  else
                    *fatal_error=(fputc(0xc4,maze) != 0xc4);
                  x++;
                }
            }
          if (! *fatal_error)
            *fatal_error=(fputc(0xb3,maze) != 0xb3);
          if (! *fatal_error)
            *fatal_error=(fputc((int) '\n',maze) != ((int) '\n'));
          if ((! *fatal_error) && (argc >= 3))
            write_hex(&maze,argv[2],fatal_error);
          if (*fatal_error)
            printf("     Fatal error:  failure to write MAKEMAZE.OUT.\n");
          fclose(maze);
        }
      return;
    }
