/* --------------------------------- stick.c -------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Handler for the joy-stick as a pointing device.
*/

#include "fly.h"

#include <pc.h>


#define inp(p) 		inportb(p)
#define outp(p,b)	outportb(p,b)

#define USE_TIMER

#define PO		p->opt
#define	BPORT		0x0001
#define JS_PORT		0x201
#define JS_TIMEOUT	20000
#define JS_READ		inp (JS_PORT)
#define WAITFORMASK	while (--i && !(~JS_READ & m))

#ifdef USE_TIMER
#define READING Tm->Hires ()
#else
#define READING i
#endif

static int FAR
JSread (POINTER *p)
/* Read joy-stick. Values are in the range 0...200.
*/
{
	register int	i, m;
	unsigned int	x, y, minx, miny;
	int		px, py, ntimes, mask;
#ifdef USE_TIMER
	unsigned int	t;
#endif

	minx = miny = 0;	/* avoid compiler warning */

	for (ntimes = 0;;) {
		i = JS_TIMEOUT;
		mask = (p->control->flags & BPORT) ? 0x04 : 0x01;
		m = mask | (mask<<1);
#ifdef USE_TIMER
		t = Tm->Hires ();
#endif
		outp (JS_PORT, 0);		/* set trigger */
		WAITFORMASK;			/* wait for x or y */
		if (!i++)
			return (1);
		if (JS_READ & mask) {	
			y = READING;
			m = mask;
			WAITFORMASK;		/* wait for x */
			if (!i++)
				return (1);
			x = READING;
		} else {
			x = READING;
			m = mask << 1;
			WAITFORMASK;		/* wait for y */
			if (!i++)
				return (1);
			y = READING;
		}
#ifdef USE_TIMER
		x -= t;
		y -= t;
#else
		x = JS_TIMEOUT - x;
		y = JS_TIMEOUT - y;
#endif
		if (ntimes) {
			if (minx > x)
				minx = x;
			if (miny > y)
				miny = y;
		} else {
			minx = x;
			miny = y;
		}

		if (++ntimes >= PO[8])		/* read more? */
			break;

		if (T(i = PO[7])) {		/* delay? */
			x = 1234;
			for (i *= 10; i-- > 0;)
				x *= 19;
		}
	}

	px = PO[1];
	py = PO[3];

	p->a[px] = minx;
	p->a[py] = miny;

	i = JS_READ;				/* read buttons */
	p->b[PO[4]] += !(i & (mask<<4));	/* button 1 */
	p->b[PO[5]] += !(i & (mask<<5));	/* button 2 */

	if (p->c[px] == 0 || p->c[py] == 0)
		return (0);		/* calibrating */

#define	REF	100		/* expected full range */
#define	DULL	20		/* dull center */
#define	EDGE	10		/* dull edge */

#define	MDULL	(REF+DULL)
#define	FULL	(MDULL+EDGE)

	p->a[px] -= p->c[px];		/* center */
	p->a[px] *= -PO[0];		/* orientation */
	if (p->a[px] > p->c[px])	/* scale */
		p->a[px] = REF;
	else if (p->a[px] < -p->c[px])
		p->a[px] = -REF;
	else {
		p->a[px] = muldiv (p->a[px], FULL, p->c[px]);
		if      (p->a[px] > MDULL)
			p->a[px] = REF;
		else if (p->a[px] >= DULL)
			p->a[px] -= DULL;
		else if (p->a[px] < -MDULL)
			p->a[px] = -REF;
		else if (p->a[px] <= -DULL)
			p->a[px] += DULL;
		else
			p->a[px] = 0;
	}

	p->a[py] -= p->c[py];		/* center */
	p->a[py] *= PO[2];		/* orientation */
	if (p->a[py] > p->c[py])	/* scale */
		p->a[py] = REF;
	else if (p->a[py] < -p->c[py])
		p->a[py] = -REF;
	else {
		p->a[py] = muldiv (p->a[py], FULL, p->c[py]);
		if      (p->a[py] > MDULL)
			p->a[py] = REF;
		else if (p->a[py] >= DULL)
			p->a[py] -= DULL;
		else if (p->a[py] < -MDULL)
			p->a[py] = -REF;
		else if (p->a[py] <= -DULL)
			p->a[py] += DULL;
		else
			p->a[py] = 0;
	}
	return (0);
}

static int FAR
JScal (POINTER *p)
/*
 * Calibrate joy-stick. Paddle must be at center!
*/
{
	p->c[PO[1]] = p->c[PO[3]] = 0;	/* indicate 'calibrating' */
	if (JSread (p))
		return (1);
	p->c[PO[1]] = p->a[PO[1]];
	p->c[PO[3]] = p->a[PO[3]];

	p->a[PO[1]] = p->a[PO[3]] = 0;
	p->l[PO[1]] = p->l[PO[3]] = 0;
	if (p->c[PO[1]] == 0 || p->c[PO[3]] == 0)
		return (1);
	return (0);
}

static int FAR
JSinit (POINTER *p, char *options)
{
	if (0 == p->opt[8])
		p->opt[8] = 1;		/* read joystick once */

	return (JScal (p));
}

static void FAR
JSterm (POINTER *p)
{}

static int FAR
JScenter (POINTER *p)
{
	p->a[PO[1]] = p->a[PO[3]] = 0;
	p->l[PO[1]] = p->l[PO[3]] = 0;

	return (0);
}

#undef inp
#undef outp

struct PtrDriver NEAR PtrAstick = {
	"ASTICK",
	0,
	JSinit,
	JSterm,
	JScal,
	JScenter,
	JSread,
	std_key
};

struct PtrDriver NEAR PtrBstick = {
	"BSTICK",
	BPORT,
	JSinit,
	JSterm,
	JScal,
	JScenter,
	JSread,
	std_key
};

