/* --------------------------------- timer.c -------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Time services for UNIX. The rsolution depends on your ftime().
*/

#include "fly.h"

#include <time.h>
#include <pc.h>


#define inp(p) 		inportb(p)
#define outp(p,b)	outportb(p,b)

#define COMMAND_REG	0x43
#define READ_SPECIAL	0xC2
#define CHANNEL_0	0x40
#define XTAL		1193000L

static Ushort	counter = 0;
static Ulong	seconds = 0, ticks = 0;

static int FAR
timer_hires (void)		/* get fastest timer available */
{
	Ushort		t, status;
	Ulong		flags;

	do {
		flags = Sys->Disable ();
		outp (COMMAND_REG, READ_SPECIAL);
		status = (Ushort) inp (CHANNEL_0);	/* get status */
		t  = (Ushort)inp (CHANNEL_0);		/* low byte */
		t += (Ushort)(inp (CHANNEL_0) << 8);	/* high byte */
		Sys->Enable (flags);
	} while (0 == t);

	return ((int)(Ushort)~((t>>1) + (Ushort)((status&0x0080)<<8)));
}

static Ulong FAR
timer_milli (void)
{
	Ushort		t;

	t = timer_hires ();

	if (t > counter)
		ticks += t - counter;
	else
		ticks += ((Ulong)t+0x00ffffL) - counter;
	counter = t;
	while (ticks >= XTAL) {
		ticks -= XTAL;
		++seconds;
	}
	return (seconds * 1000L + ticks * 1000L / XTAL);
}

static int FAR
timer_init (void)
{
	Uint		status;
	Ulong		flags;

	flags = Sys->Disable ();

	outp (COMMAND_REG, READ_SPECIAL);
	status = (Uint) inp (CHANNEL_0);		/* get status */
	(void)inp (CHANNEL_0);
	(void)inp (CHANNEL_0);

	Sys->Enable (flags);
	seconds = time (NULL);
	counter = 0;
	ticks = 0;

	return ((status&0x0e) != 0x06);
}

static void FAR
timer_term (void)
{}

static char * FAR
timer_ctime (void)
{
	time_t	tm;
	char	*t;

	tm = time (0);
	t = ctime (&tm);
	t[strlen (t) - 1] = '\0';	/* kill NewLine */
	return (t);
}

#undef inp
#undef outp
#undef COMMAND_REG
#undef READ_SPECIAL
#undef CHANNEL_0
#undef XTAL

struct TmDriver TmDriver = {
	"PC8254",
	0,
	timer_init,
	timer_term,
	timer_milli,
	timer_hires,
	timer_ctime
};
