/* --------------------------------- fly8srv.c ------------------------------ */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* This server is needed when using the udp driver.
*/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/timeb.h>
#include <netinet/in.h>
#include <netdb.h>
#include <time.h>


/* These are from fly.h.
*/
#define LADDRESS	6

/* These internal header definitions are from udp.c.
*/
#define PHTO		0
#define PHFROM		(PHTO+LADDRESS)
#define PHLEN		(PHFROM+LADDRESS)
#define PHDATA		(PHLEN+2)
#define SOCK_FLY8	SOCK_DGRAM
#define IPPORT_FLY8	0xf8f8

/* This is common to udpcli.c.
*/
static char	admin[LADDRESS] = {0, 0, 0, 0, 0x12, 0x34};

#define FLY8_TTL	10		/* 10 seconds idle time allowed */

struct client {
	struct client		*next;
	struct sockaddr_in	addr;
	time_t			timeout;
};

static struct client	*clients = 0;	/* active clients list */
static int		nclients = 0;	/* active clients count */
static char		*pname = 0;	/* program name */

static long		nin = 0, ninb = 0, nout = 0, noutb = 0;
static long		errs[10] = {0};

extern char *
my_ctime (void)
{
	time_t	tm;
	char	*t;

	tm = time (0);
	t = ctime (&tm);
	t[strlen (t) - 1] = '\0';	/* kill NewLine */
	return (t);
}

/* If this packet is from a new client then register it.
*/
static void
add_client (struct sockaddr_in *sin, int sinlen)
{
	struct client	*p;

	for (p = clients; p; p = p->next) {
		if (p->addr.sin_addr.s_addr == sin->sin_addr.s_addr &&
		    p->addr.sin_port        == sin->sin_port)
			break;
	}
	if (!p) {
		p = (struct client *)malloc (sizeof (*p));
		if (!p) {
			++errs[0];
			return;			/* no mem */
		}
		memcpy (&p->addr, sin, sinlen);
		p->next = clients;
		clients = p;
		printf ("%s %s added %08lx:%04x\n", 
			pname, my_ctime (), ntohl (p->addr.sin_addr.s_addr),
			ntohs (p->addr.sin_port));
		fflush (stdout);
		++nclients;
	}
	p->timeout = time (0) + FLY8_TTL;
}

/* Echo an incomming packet to all other registered clients.
*/
static void
echo_clients (int fd, struct sockaddr_in *sin, char *msg, int len)
{
	struct client	*p, *pp, *del;
	time_t		now;

	now = time (0);

	for (pp = 0, p = clients; p;) {
		if (p->addr.sin_addr.s_addr == sin->sin_addr.s_addr &&
		    p->addr.sin_port        == sin->sin_port) {
			pp = p;
			p = pp->next;
			continue;
		}
		if (p->timeout < now) {
			printf ("%s %s timed %08lx:%04x\n", 
				pname, my_ctime (),
				ntohl (p->addr.sin_addr.s_addr),
				ntohs (p->addr.sin_port));
			fflush (stdout);
			--nclients;
			del = p;
			p = p->next;
			if (pp)
				pp->next = p;
			else
				clients = p;
			free (del);
			continue;
		}
#ifdef FLY8_DEBUG
		printf ("%s %3u bytes echo %08lx:%04x\n", 
			pname, len, ntohl (p->addr.sin_addr.s_addr),
			ntohs (p->addr.sin_port));
		fflush (stdout);
#endif
		if (sendto (fd, msg, len, 0, (struct sockaddr *)&p->addr, 
						sizeof (p->addr)) != len){
#ifdef FLY8_DEBUG
			printf ("%s sendto() failed to %08lx:%04x\n", 
				pname, ntohl (p->addr.sin_addr.s_addr),
				ntohs (p->addr.sin_port));
			fflush (stdout);
#endif
			++errs[1];
		}
		++nout;
		noutb += len;
		pp = p;
		p = pp->next;
	}
}

static void
show_stats ()
{
	printf ("%s %s stats:\n",
		pname, my_ctime ());
	printf ("clients:      %d\n",  nclients);
	printf ("messages in:  %ld\n", nin);
	printf ("bytes    in:  %ld\n", ninb);
	printf ("messages out: %ld\n", nout);
	printf ("bytes    out: %ld\n", noutb);
	printf ("add no mem:   %ld\n", errs[0]);
	printf ("send failed:  %ld\n", errs[1]);
}

int
main (int argc, char *argv[])
{
	int			fd;
	struct sockaddr_in	sin;
	char			*protoname;
	struct protoent		*proto;
	int			sinlen;		/* address length */
	char			msg[256];	/* message buffer */
	int			n;		/* incomming message length */

	pname = argv[0];

	protoname = "udp";
	if ((proto = getprotobyname (protoname)) == NULL) {
		printf ("%s %s getprotobyname(%s) failed: %s\n", 
			pname, my_ctime (), protoname, strerror (errno));
		exit (1);
	}

	if ((fd = socket (AF_INET, SOCK_FLY8, proto->p_proto)) < 0) {
		printf ("%s %s socket() failed: %s\n",
			pname, my_ctime (), strerror (errno));
		exit (1);
	}

	memset (&sin, 0, sizeof (sin));
	sin.sin_family      = AF_INET;
	sin.sin_addr.s_addr = htonl (INADDR_ANY);
	sin.sin_port        = htons (IPPORT_FLY8);

	if (bind (fd, (struct sockaddr *) &sin, sizeof (sin)) < 0) {
		printf ("%s %s bind() failed: %s\n",
			pname, my_ctime (), strerror (errno));
		exit (1);
	}

	for (;;) {
		sinlen = sizeof (sin);
		if ((n = recvfrom (fd, msg, sizeof(msg), 0, 
				(struct sockaddr *)&sin, &sinlen)) < 0) {
			printf ("%s %s recvfrom() failed: %s\n", 
				pname, my_ctime (), strerror (errno));
			break;
		}
#ifdef FLY8_DEBUG
		printf ("%s %ld: %3u bytes from %08lx:%04x\n", 
			pname, nin, n, ntohl (sin.sin_addr.s_addr),
			ntohs (sin.sin_port));
		fflush (stdout);
#endif

/* Handle administrative requests first.
*/
		if (!memcmp (msg+PHFROM, admin, LADDRESS)) {
			if (!strcmp ("shutdown\n", msg+PHDATA))
				break;

			if (!strcmp ("stats\n", msg+PHDATA)) {
				show_stats ();
				continue;
			}
		}

/* This is a client packet, pass it on.
*/
		++nin;
		ninb += n;

/* Fill in the internal 'from address' in the packet since Fly8 has some
 * trouble getting it right. If it is already filled in then this msg came
 * from another server.
*/
		if (!memcmp (msg+PHFROM, "\0\0\0\0\0\0", LADDRESS)) {
			memcpy (msg+PHFROM, (char *)&sin.sin_addr.s_addr, 4);
			memcpy (msg+PHFROM+4, (char *)&sin.sin_port, 2);
		}

		add_client (&sin, sinlen);
		echo_clients (fd, &sin, msg, n);
	}

	close (fd);

	show_stats ();
	exit (0);
	return (0);
}
