/* --------------------------------- stick.c -------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Handler for the joystick as a pointing device for UNIX/X11.
 *
 * The joystick driver should be re-written to get around some problems. It
 * also has some straight bugs.
*/

#ifdef HAVE_JOYSTICK

#include "fly.h"

#include <unistd.h>
#include <fcntl.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#define _ASM_SEGMENT_H			/* avoid the garbage errors */
#include <linux/joystick.h>


#define USELOG		0x0001		/* log scale on x/y (default) */

#define PO		p->opt
#define	FA1D		PO[0]
#define	FA1F		PO[1]
#define	FA2D		PO[2]
#define	FA2F		PO[3]
#define	FNREAD		PO[4]
#define	FDELAY		PO[5]
#define	FA1BASE		PO[6]
#define	FA2BASE		PO[7]
#define FA1IDLE		PO[8]
#define FA2IDLE		PO[9]
#define FOPTS		PO[10]

#define	REF		100		/* expected full range */

#define BSTICK		0x0001

static int	js[2] = {-1, -1};

/* Interpret the joystick reading as symm_etric -100...+100.
 * The 'base' parameter is interpreted as a fraction of the center reading.
*/
static void FAR
symm_input (POINTER *p, int channel, int reading, int sign, int base, int idle,
	int transfer)
{
	int	center;

	center = p->c[channel];
	reading -= center;				/* center */
	reading *= sign;				/* orientation */
	center -= muldiv (center, base, 100);		/* clip range */
	if (reading > center)				/* clip + */
		reading = REF;
	else if (reading < -center)			/* clip - */
		reading = -REF;
	else {
		reading = muldiv (REF+idle, reading, center);
		if (reading > idle)
			reading -= idle;		/* active + */
		else if (reading > -idle)
			reading = 0;			/* idle center */
		else
			reading += idle;		/* active - */
	}
	p->a[channel] = reading;
	if (transfer)
		p->l[channel] = (FOPTS & USELOG)
				? lin2log (reading) : reading;
}

static int FAR
jread (POINTER *p, int transfer)
{
	struct JS_DATA_TYPE jsdata;
	int	which, x, y, ntimes, nlimit;
	char	btn[2];

	which = T(p->control->flags & BSTICK);
	if (js[which] < 0)
		return (1);

	x = y = 0;	/* avoid compiler warning */
	nlimit = (0 == p->c[0]) ? 10 : FNREAD;
	for (ntimes = 0; ntimes < nlimit; ++ntimes) {
		if (read (js[which], (void *)&jsdata, JS_RETURN) != JS_RETURN)
			return (1);
		if (ntimes) {
			if (x > jsdata.x)
				x = jsdata.x;
			if (y > jsdata.y)
				y = jsdata.y;
		} else {
			x = jsdata.x;
			y = jsdata.y;
		}
	}

	if (0 == p->c[0]) {			/* calibrating */
		p->a[FA1F] = x;
		p->a[FA2F] = y;
		return (0);
	}

	memset (btn, 0, sizeof (btn));

	symm_input (p, FA1F, x, -FA1D, FA1BASE, FA1IDLE, transfer);
	symm_input (p, FA2F, y,  FA2D, FA2BASE, FA2IDLE, transfer);

	btn[0] = T(jsdata.buttons & 1);
	btn[1] = T(jsdata.buttons & 2);
	do_btns (p, btn, rangeof (btn));

	return (0);
}

static int FAR
cal (POINTER *p)
/* Calibrate joy-stick. Paddle must be at center!
*/
{
	p->c[0] = 0;			/* indicate 'calibrating' */
	if (jread (p, 0))
		return (1);
	p->c[FA1F] = p->a[FA1F];
	p->c[FA2F] = p->a[FA2F];

	p->a[FA1F] = p->a[FA2F] = 0;
	p->l[FA1F] = p->l[FA2F] = 0;
	if (p->c[FA1F] == 0 || p->c[FA2F] == 0)
		return (2);
	return (0);
}

static int FAR
init (POINTER *p, char *options)
{
	int	which;
	long	l;
	struct JS_DATA_TYPE jsdata;

	which = T(p->control->flags & BSTICK);
	if ((js[which] = open (which ? "/dev/js1" : "/dev/js0", O_RDONLY)) < 0)
		return (1);

	l = -1L;	/* get around joystick driver bug */
	if (ioctl (js[which], JS_SET_TIMELIMIT, &l) < 0)
		goto badret;

	l = 5000L;
	if (ioctl (js[which], JS_SET_TIMEOUT, &l) < 0)
		goto badret;

	jsdata.x = jsdata.y = 0;
	if (ioctl (js[which], JS_SET_CAL, &jsdata) < 0)
		goto badret;

	if (get_narg (options, "rd=", &l))
		FNREAD = 1;
	else
		FNREAD = (int)l;

	if (get_narg (options, "bx1=", &l))
		FA1BASE = 20;
	else
		FA1BASE = (int)l;
	if (get_narg (options, "by1=", &l))
		FA2BASE = 20;
	else
		FA2BASE = (int)l;

	if (get_narg (options, "ix1=", &l))
		FA1IDLE = 20;
	else
		FA1IDLE = (int)l;
	if (get_narg (options, "iy1=", &l))
		FA2IDLE = 20;
	else
		FA2IDLE = (int)l;

	if (get_arg (options, "linear"))
		FOPTS &= ~USELOG;
	else
		FOPTS |= USELOG;

	get_btn (p, options);

	if (cal (p))
#if 0
		goto badret;
#else
		;	/* joystick driver is broken on early reads */
#endif
	return (0);

badret:
	close (js[which]);
	js[which] = -1;
	return (1);
}

static void FAR
term (POINTER *p)
{
	int	which;

	which = T(p->control->flags & BSTICK);
	if (js[which] >= 0) {
		close (js[which]);
		js[which] = -1;
	}
}

struct PtrDriver PtrAstick = {
	"ASTICK",
	0,
	init,
	term,
	cal,
	cal,			/* center */
	jread,
	std_key
};

struct PtrDriver PtrBstick = {
	"BSTICK",
	BSTICK,
	init,
	term,
	cal,
	cal,			/* center */
	jread,
	std_key
};
#undef USELOG
#undef PO
#undef FA1D
#undef FA1F
#undef FA2D
#undef FA2F
#undef FNREAD
#undef FDELAY
#undef FA1BASE
#undef FA2BASE
#undef FA1IDLE
#undef FA2IDLE
#undef REF
#undef BSTICK
#undef FOPTS

#endif
