/* --------------------------------- colors.c ------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* color handling.
*/

#include "fly.h"
#include "colors.h"


#define COLOR_NAME_LENGTH	7

static long NEAR
ShowColor (Ulong color)
{
	return ((0x00ff0000LU & (color << 16)) |
		(0x0000ff00LU & (color)) |
		(0x000000ffLU & (color >> 16)));
}

#if 0
static long NEAR
grey_color (long color)			/* get intensity */
{
	int	r, g, b, w;

	r = 0x0ff & (int)(color);
	g = 0x0ff & (int)(color >> 8);
	b = 0x0ff & (int)(color >> 16);

	w = (r*67+g*21+b*15)/(67+21+15);

	return (w | (w<<8) |((long)w<<16));
}
#endif

static long NEAR
hi_color (long color)			/* highlight a color */
{
	Uint	r, g, b, m;

	r = 0x0ff & (Uint)(color);
	g = 0x0ff & (Uint)(color >> 8);
	b = 0x0ff & (Uint)(color >> 16);

	m = (r > g) ? r : g;
	if (b > m)
		m = b;
	if (!m)
		return (color);
	
	r = (Uint)muldiv (r, 0x0ff, m);
	g = (Uint)muldiv (g, 0x0ff, m);
	b = (Uint)muldiv (b, 0x0ff, m);

	return (r | (g<<8) | ((long)b<<16));
}

static MENU MenuCoOp[] = {
	{'+', "Brighter"},	/*  0 */
	{'-', "Darker"},	/*  1 */
	{'R', "+red"},		/*  2 */
	{'r', "-red"},		/*  3 */
	{'G', "+green"},	/*  4 */
	{'g', "-green"},	/*  5 */
	{'B', "+blue"},		/*  6 */
	{'b', "-blue"},		/*  7 */
	{'=', "New"},		/*  8 */
	{'*', "Restore"},	/*  9 */
{'\0', 0}};

#define	COLOR_STEP	8

static void NEAR
get_color (char *name, int index, Ulong *color)
{
	Uint	r, g, b;
	int	sel;
	long	l, newc;
	char	msg[9];
	HMSG	*m;

	newc = (long)*color;

	m = MsgWPrintf (0, name);

	sel = 8;	/* new */
	do {
		if (m)
			sprintf (m->text+COLOR_NAME_LENGTH+1, "%06lx",
				ShowColor (newc));
		Gr->SetPalette (index, newc);

		r = 0x0ff & (Uint)(newc);
		g = 0x0ff & (Uint)(newc >> 8);
		b = 0x0ff & (Uint)(newc >> 16);

		sel = menu_open (MenuCoOp, sel);

		switch (sel) {
		default:
		case MENU_ABORTED:
		case MENU_FAILED:
			break;
		case 0:
			if ((r += (r+COLOR_STEP-1)/COLOR_STEP) > 0x0ff)
				r = 0x0ff;
			if ((g += (g+COLOR_STEP-1)/COLOR_STEP) > 0x0ff)
				g = 0x0ff;
			if ((b += (b+COLOR_STEP-1)/COLOR_STEP) > 0x0ff)
				b = 0x0ff;
			newc = r | (g<<8) | ((long)b<<16);
			break;
		case 1:
			r -= (r+COLOR_STEP-1)/COLOR_STEP;
			g -= (g+COLOR_STEP-1)/COLOR_STEP;
			b -= (b+COLOR_STEP-1)/COLOR_STEP;
			newc = r | (g<<8) | ((long)b<<16);
			break;
		case 2:
			if ((r += (r+COLOR_STEP-1)/COLOR_STEP) > 0x0ff)
				r = 0x0ff;
			newc = r | (g<<8) | ((long)b<<16);
			break;
		case 3:
			r -= (r+COLOR_STEP-1)/COLOR_STEP;
			newc = r | (g<<8) | ((long)b<<16);
			break;
		case 4:
			if ((g += (g+COLOR_STEP-1)/COLOR_STEP) > 0x0ff)
				g = 0x0ff;
			newc = r | (g<<8) | ((long)b<<16);
			break;
		case 5:
			g -= (g+COLOR_STEP-1)/COLOR_STEP;
			newc = r | (g<<8) | ((long)b<<16);
			break;
		case 6:
			if ((b += (b+COLOR_STEP-1)/COLOR_STEP) > 0x0ff)
				b = 0x0ff;
			newc = r | (g<<8) | ((long)b<<16);
			break;
		case 7:
			b -= (b+COLOR_STEP-1)/COLOR_STEP;
			newc = r | (g<<8) | ((long)b<<16);
			break;
		case 8:
			getstr ("color RRGGBB", msg, sizeof (msg));
			if (1 == sscanf (msg, "%lx", &l)) {
				b = (Uchar)(0x0ff & (l      ));
				g = (Uchar)(0x0ff & (l >> 8 ));
				r = (Uchar)(0x0ff & (l >> 16));
				if (r > 0x0ff || g > 0x0ff || b > 0x0ff)
					MsgEPrintf (50, "*** Bad primary");
				else
					newc = r | (g<<8) | ((long)b<<16);
			} else
				MsgEPrintf (50, "*** Bad color");
			break;
		case 9:
			newc = *color;
			break;
		}
		if (MENU_FAILED != sel)
			menu_close ();
	} while (sel >= 0);
	*color = newc;

	msg_del (m);
	m = 0;
}

static struct color {
	char	*name;
	int	letter;
	int	*index;
	Ulong	value;
} colors[] = {
	{"HudLow",  'h', &st.hudlow,  C_LYELLOW},
	{"HudHigh", 'H', &st.hudhigh, C_YELLOW},
	{"Friend",  'f', &st.lblue,   C_LIGHTBLUE},
	{"Foe",     'o', &st.lred,    C_LIGHTRED},
	{"Sky",     's', &st.skyblue, C_SKYBLUE},
	{"Ground",  'G', &st.ground,  C_GRAY},
	{"Border",  'B', &st.lgray,   C_LIGHTGRAY},
	{"Red",     'r', &st.red,     C_RED},
	{"Blue",    'b', &st.blue,    C_BLUE},
	{"Magenta", 'm', &st.magenta, C_MAGENTA},
	{"Green",   'g', &st.green,   C_GREEN},
	{"Brown",   'n', &st.brown,   C_BROWN},
	{"Gray",    'a', &st.gray,    C_GRAY},
	{"Black",   'k', &st.black,   C_BLACK},
	{"White",   'w', &st.white,   C_WHITE}
};

extern int FAR
set_rrggbb (int letter, Ulong value)
{
	int	i;

	value = RGB (0x0ff&(Uint)(value>>16), 0x0ff&(Uint)(value>>8),
			0x0ff&(Uint)value);
	for (i = 0; i < rangeof (colors); ++i) {
		if (letter == colors[i].letter) {
			colors[i].value = value;
			if (0 == i)
				colors[1].value = hi_color (colors[0].value);
			return (0);
		}
	}
	return (1);
}

extern void FAR
set_palette (void)
{
	int	i;

	for (i = 0; i < rangeof (colors); ++i)
		Gr->SetPalette (*colors[i].index, colors[i].value);
}

extern int FAR
menu_colors (void)
{
	MENU	*MenuColors;
	int	sel, i, n, nEntries, EntrySize;

	nEntries = rangeof (colors);
	EntrySize = COLOR_NAME_LENGTH+1+6+1;

	n = (nEntries+1) * sizeof (*MenuColors);
	if (!(MenuColors = (MENU *) memory_alloc (n)))
		return (1);

	sel = MENU_FAILED;
	for (i = 0; i < nEntries; ++i) {
		if (!(MenuColors[i].text = (char *) memory_alloc (EntrySize)))
			goto ret;
	}

	for (sel = 0;;) {
		for (i = 0; i < nEntries; ++i) {
			MenuColors[i].letter = colors[i].letter;
			sprintf (MenuColors[i].text, "%-*s %06lx",
				COLOR_NAME_LENGTH, colors[i].name,
				ShowColor (colors[i].value));
		}

		sel = menu_open (MenuColors, sel);
		if (sel >= 0) {
			get_color (MenuColors[sel].text, *colors[sel].index,
				&colors[sel].value);
			if (0 == sel) {		/* HudLow */
				colors[1].value = hi_color (colors[0].value);
				Gr->SetPalette (*colors[1].index,
							colors[1].value);
			}
			menu_close ();
		} else
			break;
	}

ret:
	for (i = 0; i < nEntries; ++i)
		if (MenuColors[i].text)
			memory_free (MenuColors[i].text, EntrySize);

	memory_free (MenuColors, n);

	if (MENU_FAILED == sel)
		return (1);
	menu_close ();
	return (0);
}

#undef COLOR_NAME_LENGTH
#undef COLOR_STEP
