/* ------------------------- line.c ----------------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Draw lines on SVGA. Can be very fast when usage of low level assembly
 * support is requested.
 *
 * It originated from the DJgpp package but completely re-written since, the
 * only surviving clue being the WriteMode specification in the color field.
 *
 * This version replaces DJgpp line.c but uses asm support.
 *
 * MoveTo (x, y)
 * DrawTo (x, y, color)
 * Line (x0, y0, x1, y1, color)
 * SetActiveBase (Ulong b)
 * SetVisualBase (Ulong b)
 * SetWriteMode (int mode)
 * GrClear (int x, int y, int sizex, int sizey, Uint color)
*/

#include "fly.h"
#include "gr.h"

#include <graphics.h>
#include <pc.h>


int		_GrWriteMode = 0;
char		*_ActiveBase = VGA_PAGE;
static int	VisualBase = 0;
static Uint	x1 = 0, y1 = 0, xydone = 0, width = 0, height = 0;
static char	*pva = VGA_PAGE;

extern void FAR
MoveTo (int x, int y)
{
#if 0
	if (x < 0)
		x = 0;
	else if (x >= width)
		x = width-1;

	if (y < 0)
		y = 0;
	else if (y >= height)
		y = height-1;
#endif
	x1 = x;
	y1 = y;
	xydone = 0;
}

extern void FAR
DrawTo (int x2, int y2, Uint c)
{
	register int	dx, dy, svx, svy;
#if 0
	if (x2 < 0)
		x2 = 0;
	else if (x2 >= width)
		x2 = width-1;

	if (y2 < 0)
		y2 = 0;
	else if (y2 >= height)
		y2 = height-1;
#endif
	if (!xydone) {
		xydone = 1;
		pva = _ActiveBase + x1 + y1*(long)width;
		if (_GrWriteMode == 0)
			*pva = c;
		else if (_GrWriteMode & GrXOR)
			*pva ^= c;
		else if (_GrWriteMode & GrOR)
			*pva |= c;
	}

	if ((dx = x2 - x1) < 0) {
		dx = -dx;
		svx = -1;
	} else
		svx = 1;

	if ((dy = y2 - y1) < 0) {
		dy = -dy;
		svy = -width;
	} else
		svy = width;

	c |= _GrWriteMode;
	if (dx > dy) {
		if (dy)
			pva = ILoop (pva, dy, dx, svx, svy, c);
		else {
			if (_GrWriteMode)
				pva = SLoop (pva, dx, svx, c);
			else {
				char	*pvb;

				pvb = pva + x2 - x1;		/* end point */
				vgaset ((svx>0 ? (pva+1) : pvb), c, dx);
				pva = pvb;
			}
		}
	} else if (dx < dy) {
		if (dx)
			pva = ILoop (pva, dx, dy, svy, svx,  c);
		else
			pva = SLoop (pva, dy, svy, c);
	} else if (dx)
			pva = SLoop (pva, dy, svx+svy, c);

	x1 = x2;
	y1 = y2;
}

extern void FAR
SetActiveBase (Ulong b)
{
	_ActiveBase = VGA_PAGE + b;
}

extern void FAR
SetVisualBase (Ulong base)		/* t4k specific!!!!!!!!!!!!!!!!! */
{
	int	i;
	Ulong	flags;

	if (base == VisualBase)
		return;

	flags = Sys->Disable ();
	outportb (0x3d4, 0x0d);	/* start base low */
	outportb (0x3d5, (int)((base>>2)&0x00ff));

	outportb (0x3d4, 0x0c);	/* start base high */
	outportb (0x3d5, (int)((base>>10)&0x00ff));

	outportb (0x3d4, 0x33);	/* start base very high */
	i = inportb (0x3d5);
	outportb (0x3d5, (i&~3)|(int)((base>>18)&3));
	Sys->Enable (flags);

	VisualBase = base;
}

extern void FAR
SetWriteMode (int mode)
{
	_GrWriteMode = mode & ~0xff;
}

extern void FAR
DrawLine (int xa, int ya, int xb, int yb, Uint c)
{
	if (xa != x1 || ya != y1 || !xydone)
		MoveTo (xa, ya);

	if ((c&0xff00) != _GrWriteMode)
		SetWriteMode (c);

	DrawTo (xb, yb, c);
}

extern void FAR
GrClear (int x, int y, int sizex, int sizey, Uint color)
{
	for (sizey += y, sizex += x-1;y < sizey; ++y)
		DrawLine (x, y, sizex, y, color);
}

extern void FAR
InitGr (int mode, int sizex, int sizey)
{
	GrSetMode (mode, sizex, sizey);
	width  = GrMaxX () + 1;
	height = GrMaxY () + 1;
}
