/* --------------------------------- fly.h ---------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* structures used by the system.
*/

#ifndef FLY8_FLY
#define FLY8_FLY

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>

#include "config.h"
#include "keymap.h"
#include "shape.h"
#include "objects.h" 

#define rangeof(array)	(sizeof(array)/sizeof(array[0]))

#define	BUFLEN	100		/* number of points in a display list block */

typedef unsigned char	Uchar;
typedef unsigned int	Uint;
typedef unsigned short	Ushort;
typedef unsigned long	Ulong;

typedef short	ANGLE;
typedef short	VECT[3];
typedef long	LVECT[3];
typedef ANGLE	AVECT[3];
typedef short	MAT[3][3];

#define X	0
#define Y	1
#define Z	2

#define	HEAD	0
#define	TAIL	1

#define QUEUE(q,m) \
	if ((q)[HEAD])			\
		(q)[TAIL]->next = (m);	\
	else				\
		(q)[HEAD] = (m);	\
	(q)[TAIL] = (m)

typedef struct vertex VERTEX;
typedef struct shape SHAPE;
typedef struct object OBJECT;
typedef struct bufline BUFLINE;
typedef struct buffer BUFFER;
typedef struct device DEVICE;
typedef struct pointer POINTER;
typedef struct screen SCREEN;
typedef struct view VIEW;
typedef struct window WINDOW;
typedef struct viewport VIEWPORT;
typedef struct player PLAYER;
typedef struct packet PACKET;
typedef struct netport NETPORT;
typedef struct msg HMSG;
typedef struct hdd HDD;
typedef struct pid PID;
typedef struct macro MACRO;


struct vertex {
	VECT	V;
	Ushort	flags;
};

struct shape {
	VERTEX	*v;
	Ushort	extent;
	Ushort	flags;
#define SH_G		0x0001
#define SH_HIT		0x0002
#define SH_BEHIT	0x0004
#define SH_LOCALSIM	0x0008
#define SH_FINE		0x0010
	long	weight;		/* grams */
	short	drag;		/* drag factor */
};

#define	SH(p)		(st.bodies[p->name]->shape)
#define TITLE(p)	(st.bodies[p->name]->title)

struct pid {
	long	Kp;
	long	Iband;
	long	Ki;
	long	Dband;
	long	Kd;
	long	factor;
	long	range;
	long	Pprev;
	long	I;
};

struct macro {
	Ushort	name;
#define KEYUNUSED		0
	Ushort	len;
	Ushort	*def;
};

typedef struct e_imported E_IMPORT;
struct e_imported {
	Ulong	lasttime;
	Ulong	timeout;
	short	misc[5];
};

#define EIM(p)	(p->extra.e_imported)


struct history {
	struct history	*next;
	LVECT	R;
	VECT	V;
	Uint	ms;
};
typedef struct history	HISTORY;

typedef struct e_plane E_PLANE;
struct e_plane {
	struct e_parms	*parms;		/* plane parameters */
	void		*private;	/* private model data */
	Ushort	flags;
#define	PF_AUTO		0x0001
#define	PF_ONGROUND	0x0002
#define PF_CHASE	0x0004
#define PF_KILL		0x0008
#define PF_FLAMEOUT	0x0010
#define PF_GLIMIT	0x0020
#define PF_STALL	0x0040
#define PF_AUTOFLAP	0x0080
#define PF_AUTORUDDER	0x0100
#define PF_AUTOELEVATOR	0x0200
#define PF_NOSTALL	0x0400
#define PF_LIMITED	0x0800
	Ushort	hud;
#define	HUD_ON		0x0001
#define	HUD_LADDER	0x0002
#define	HUD_BIG		0x0004
#define	HUD_FINE	0x0008
#define	HUD_XFINE	0x0010
#define	HUD_PLUS	0x0020
#define	HUD_CURSOR	0x0040
#define	HUD_VV		0x0080
#define	HUD_TARGET	0x0100
#define	HUD_DATA	0x0200
#define	HUD_RETICLE	0x0400
#define	HUD_ROSS	0x0800
#define	HUD_INFO	0x1000
#define	HUD_INFOM	0x7000
#define	HUD_DEFAULT	(HUD_TARGET|HUD_RETICLE)
#define	HUD_FULLHEADING	0x8000
	Ushort	hud1;
#define	HUD_TOP		0x0001
#define	HUD_LIMIT	0x0002
#define	HUD_CORNER	0x0004
#define	HUD_AALARM	0x0008
#define	HUD_VALARM	0x0010
#define	HUD_KNOTS	0x0020
#define	HUD_PANEL	0x0040
#define	HUD_ACCVECT	0x0080
#define	HUD_TYPE	0x0100
#define	HUD_TYPES	0x0700
#define	HUD_CLASSIC	(0*HUD_TYPE)
#define	HUD_FA18	(1*HUD_TYPE)
#define	HUD_F16		(2*HUD_TYPE)
#define	HUD_F15		(3*HUD_TYPE)
#define	HUD_ETHER	(4*HUD_TYPE)
#define	HUD_FLIR	(5*HUD_TYPE)
#define	HUD_PENDULUM	0x0800
#define	HUD_INAME	0x1000
#define	HUD_IDIST	0x2000
#define	HUD_THICK	0x4000
#define	HUD_BORDER	0x8000
	Ushort	hud2;
#define	HUD_ILS		0x0001
#define	HUD_VW 		0x0002
#define	HUD_ALTITUDE	0x0004
#define	HUD_SPEED	0x0008
#define	HUD_HEADING	0x0010
#define HUD_REALHEADING	0x0020		/* show real heding scale */
#define	HUD_HIDETGT	0x0040
#define	HUD_BETA	0x0080
#define HUD_XBREAK	0x0100
#define HUD_XGRID	0x0200
#define HUD_DIRECTOR	0x0400		/* show flight director */
#define HUD_WAYPOINT	0x0800
#define HUD_HISTORY	0x1000		/* bullet trail history */
#define HUD_CALIBRATED	0x2000		/* calibrated airspeed */
#define HUD_TPOINTER	0x4000		/* target pointer */
#define HUD_VPOINTER	0x8000		/* variable size target pointer */
	Ushort	hudmode;
#define	HM_LAND		0x0001
#define	HM_NAV		0x0002
#define	HM_DGFT		0x0004
#define	HM_CDIP		0x0008
#define	HM_DECLUTTER	0x0100
	Ushort	hdd;
#define HDD_INSTRUMENTS	0x0001
#define	HDD_NAV		0x0002		/* show nav data */
#define	HDD_COMPASS	0x0004
#define	HDD_SQRCOMPASS	0x0008
#define	HDD_ORTCOMPASS	0x0010
	Ushort	ladder;
#define	LD_FIXED	0x0001		/* ladder fixed on waterline */
#define	LD_RIGHT	0x0002		/* numerals on the right only */
#define	LD_ERECT	0x0004		/* numerals do not roll */
#define	LD_COLOR	0x0008		/* steps are colored */
#define	LD_FUNNEL	0x0010		/* tips are inside */
#define	LD_SLANT	0x0020		/* steps are slanted */
#define	LD_ZENITH	0x0040		/* zenith/nadir symbols */
#define	LD_UNDER	0x0080		/* numerals under the step */
#define	LD_TIP0		0x0100		/* show tip on horizon */
#define	LD_HOLD		0x0200		/* hold ladder  */
#define	LD_HOLDROLL	0x0400		/* hold on heading */
#define	LD_SUN		0x0800		/* show sun symbol */
#define	LD_NEGTIP	0x1000		/* -ve pitch tip inverted */
	Ushort	radar;
#define	R_ON		0x0001
#define	R_LOCK		0x0002
#define	R_INTEL		0x0004
#define	R_INTELCC	0x0008
#define	R_MODE		0x0010
#define	R_MODES		0x0070
#define	R_SHOOT		0x0080
#define	R_SELECT3	0x0100
#define	R_SELECT20	0x0200
#define	R_SELECT5	0x0400
	short	equip;
#define	EQ_GEAR		0x0001		/* gear deployed */
#define	EQ_GEAR1	0x0002		/* true if gear1 has ground contact */
#define	EQ_GEAR2	0x0004
#define	EQ_GEAR3	0x0008
#define	EQ_GEAR4	0x0010
#define	EQ_GEAR5	0x0020
	short	weapon;			/* weapon type */
#define	WE_M61		0x0001
#define	WE_MK82		0x0002
#define	WE_LAST		0x0002
#define	N_WE		5
	short	stores[N_WE];		/* weapons available */
	short	hudarea;		/* degrees from center to edge */
	short	hudFontSize;
	short	ils;			/* ils beacon id */
	Ushort	ilsRange;
	ANGLE	ilsHeading;
	short	tapelen;		/* speed/alt scale range */
	short	hudshift;		/* hud y shift */
	short	ldgap;			/* ladder gap size */
	short	ldstep;			/* ladder step size */
	short	ldstep0;		/* ladder step 00 size */
	short	ldstepg;		/* ladder step 00 (gear down) size */
	short	ldtip;			/* ladder tip size */
	short	ldndash;		/* ladder -ve step n-dashes */
	long	fuel;			/* fuel left [*100] */
	short	fuelRate;		/* [*100] */
	short	throttle;		/* position [-100..100]*/
	short	afterburner;		/* position [0..100] */
	short	elevators;		/* position [0..100] */
	short	ailerons;		/* position [-100..100] */
	short	rudder;			/* position [-70..70] */
	short	flaps;			/* position [0..100] */
	short	leFlaps;		/* position [0..100] */
	short	spoilers;		/* position [0..100] */
	short	tElevators;		/* trim position [0..100] */
	short	tRudder;		/* trim position [0..100] */
	short	airbrake;		/* position [0..100] */
	short	brake;			/* position [0..100] */
	short	gear[5];		/* position [0..100] */
	short	power;			/* percent*100 [0..10000] */
	short	thrust;			/* lb_thrust/10 */
	short	Gforce;			/* for display */
	short	maxG;			/* for display */
	short	aoa;			/* for display */
	short	mach;			/* for display */
	VECT	a;			/* acceleration [body] */
	VECT	v;			/* velocity [body] */
	short	StFont;
	Uchar	NEAR*  NEAR* StFontPtr;	/* StFonts[StFont] */
	short	StFontSize;
	OBJECT	*target;		/* aqcuired target */
	long	tid;			/* target id */
	PID	*PIDthrottle;
	PID	*PIDpitch;
	PID	*PIDroll;
	VECT	taccel;			/* target average acceleration */
	VECT	tspeed;			/* target average velocity */
	HISTORY	*history;		/* bullet trail history */
	short	misc[20];		/* autopilot, history etc. */
#define EE(p)	((p)->extra.e_plane)
#define	LIFETIME(p)	EE(p)->misc[0]	/* millisecs to next randomization */
#define	SPEED(p)	EE(p)->misc[1]	/* desired speed */
#define	HEADING(p)	EE(p)->misc[2]	/* desired heading */
#define	ALTITUDE(p)	EE(p)->misc[3]	/* desired altitude */
#define	NEWTGT(p)	EE(p)->misc[4]
};


typedef struct e_bomb E_BOMB;
struct e_bomb {
	LVECT	IP;
	long	timpact;
};

#define EBM(p)	(p->extra.e_bomb)


struct object {
	OBJECT		*next;
	OBJECT		*prev;
	union {
			E_IMPORT	*e_imported;
			E_PLANE		*e_plane;
			E_BOMB		*e_bomb;
	}		extra;
	OBJECT		*owner;
	PLAYER		*rplayer;
	POINTER		*pointer;
	VIEWPORT	*viewport;
	long		id;		/* local object id */
	long		rid;		/* remote object id */
	long		ownerid;	/* local object id of owner-object */
	Ulong		rtime;		/* remote time */
	LVECT		R;		/* position [earth] */
	VECT		V;		/* velocity [earth] */
	AVECT		dda;		/* angular accels [body] */
	AVECT		da;		/* angular rates [body] */
	AVECT		dae;		/* Euler rates [earth] */
	AVECT		a;		/* Euler angles [earth] */
	short		sinx;
	short		cosx;
	short		siny;
	short		cosy;
	short		sinz;
	short		cosz;
	MAT		T;		/* body->earth xform matrix */
	short		home;		/* home ils beacon id */
	short		longitude;
	short		longmin;
	short		latitude;
	short		latmin;
	short		name;
	short		color;
#define	FOREVER		0x7fff
	short		time;
#define	TIMEPSEC	100
	Ushort		flags;
#define F_VISIBLE	0x0001		/* permanent: */
#define F_IMPORTED	0x0002
#define F_EXPORTED	0x0004
#define F_MAINT		0x0008
#define F_STEALTH	0x0010
#define F_CC		0x0020
#define F_LAND		0x0040		/* part of landscape */
#define F_FRIEND	0x0080
#define F_ALIVE		0x0200		/* transitional: */
#define F_NEW		0x0400
#define F_DEL		0x0800
#define F_HIT		0x1000
#define F_KEEPNAV	0x2000
#define F_DONE		0x4000
#define F_MOD		0x8000
	Ushort		gpflags;
#define GPF_PILOT	0x0001
	Ushort		shflags;	/* copy of SH flags */
	short		speed;
	short		e_type;
#define ET_IMPORTED	0x0001
#define ET_PLANE	0x0002
#define ET_BOMB		0x0003
#define IS_PLANE(p)	(ET_PLANE == (p)->e_type)
	short		score;
	short		damage;
	short		damaging;
	short		misc[5];
};


struct bufline {
	short	x;
	short	y;
	short	t;	/* color or (if -ve) type */
#define	T_MOVE		-1
#define	T_MOR		-2
#define	T_MXOR		-3
#define	T_MSET		-4
#define	T_ELLIPSE	-5
#define	T_NOP		-99
};

struct buffer {
	BUFFER		*next;
	BUFLINE		*p;		/* first free */
	BUFLINE		*end;		/* &first[size] */
	int		size;		/* allocated size */
	BUFLINE		first[1];	/* start of data array */
};

struct device {
	DEVICE	*next;
	char	*name;
	void	*pdevice;	/* private device data */
	int	mode;		/* vga bios mode etc. */
	int	colors;		/* number of colors */
	int	minx;		/* physical */
	int	miny;		/* physical */
	int	sizex;		/* physical */
	int	sizey; 		/* physical */
	int	npages;		/* how many display pages */
	int	lengx;		/* millimeter */
	int	lengy;		/* millimeter */
	int	FontWidth;	/* pixels */	/* belongs in SCREEN! */
	int	FontHeight;	/* pixels */	/* ditto */
	int	flags;		/* driver dpivate */
};

struct pointer {
	char	*name;
	Ushort	flags;
#define PF_PRESENT	0x0001
#define PF_INITED	0x0002
	struct PtrDriver NEAR* control;
#define	NANALOG	10
/* Channel allocation:
 * 0	roll clockwise
 * 1	pitch up
 * 2	rudder right
 * 3	throttle
 * 4	trim pitch up
 * 5	trim rudder right
 * 6	flaps
 * 7	spoilers
 * 8	speed brakes
 * 9	ground brakes
*/
	short	a[NANALOG];	/* analog -100...+100 */
	short	l[NANALOG];	/* last value of a[] */
	short	c[NANALOG];	/* calibration */
	short	b[10];		/* digital commands */
	char	d[36];		/* button debounce */
#define	NOPTS	20
	int	opt[NOPTS];
};

struct screen {
	DEVICE	*device;
	void	*pscreen;	/* private screen data */
	int	minx;		/* physical */
	int	miny;		/* physical */
	int	sizex;		/* physical */
	int	sizey;		/* physical */
	Uint	FgColor;
	Uint	BgColor;
	Uint	BoColor;	/* border */
	int	FontWidth;	/* pixels */
	int	FontHeight;	/* pixels */
};

struct window {
	int	orgx;	/* all in normalized coordinates [0...1] */
	int	orgy;
	int	maxx;	/* left=orgx-maxx right =orgx+maxx */
	int	maxy;	/* top =orgy-maxy bottom=orgy+maxy */
};

struct viewport {
	int	x;	/* NDC normally 0.0 */
	int	y;	/* NDC normally 0.0 */
	int	z;	/* NDC zoom, normally 0.5, must >= maxx and maxy! */
	int	maxx;	/* NDC vp width normally 0.5 */
	int	maxy;	/* NDC vp height normally 0.5 */
			/* distz and shift in units of VONE*VONE */
	int	distz;	/* eye to vp in viewers coords */
	int	shift;	/* eye x shift (v coords) in stereo */
	int	eyex;	/* viewer coords: */
	int	eyey;	/* eye position relative to viewer object origin */
	int	eyez;
	ANGLE	rotx;	/* eye direction relative to viewer origin */
	ANGLE	roty;
	ANGLE	rotz;
	int	zoom;	/* zoom count */
};

struct	view {
	VIEWPORT	*viewport;	/* camera: world -> 2D */
	WINDOW		*window;	/* camera -> NDC window */
	SCREEN		*screen;	/* window -> physical screen */
};

typedef struct body BODY;
struct body {
	int	name;
	Ushort	flags;
	char	*title;
	SHAPE	*shape;
	int	(FAR* init) (BODY *b);
	void	(FAR* term) (BODY *b);
	int	(FAR* create) (OBJECT *object);
	void	(FAR* delete) (OBJECT *object);
	void	(FAR* dynamics) (OBJECT *object, int interval);
	void	(FAR* hit) (OBJECT *object, int speed, int extent,
			int damaging);
};

#define	LADDRESS	6
#define	LNAME		20

struct player {
	PLAYER	*next;
	Ushort	flags;
#define	PL_ACTIVE	0x0001
#define	PL_PENDREPLY	0x0002
#define	PL_PENDBOSS	0x0004
#define	PL_PENDCONFIRM	0x0008
#define	PL_PEND	(PL_PENDREPLY|PL_PENDCONFIRM|PL_PENDBOSS)
#define	PL_PLAYING	0x0010
#define	PL_RECEIVE	(PL_PENDCONFIRM|PL_PLAYING)
#define	PL_SEND	(PL_PLAYING)
#define	PL_NOTIDLE	(PL_ACTIVE|PL_PEND|PL_PLAYING)
#define	PL_FRIEND	0x0020
	short	netport;
	Uchar	address[LADDRESS];
	char	name[LNAME];
	char	team[LNAME];
	short	ComVersion;			/* net version */
	PACKET	*incoming;
	PACKET	*tail;
	Ulong	timeout;
	long	rtime;				/* RemoteTime-LocalTime */
	short	rtimeErr;			/* intergrated error */
};

struct netport {
	Ushort			flags;
#define	NP_ON		0x0001
#define	NP_BCAST	0x0002
#define	NP_PACKING	0x0004
	short			netport;
	char			unit;
	int			nplayers;	/* playing users */
	struct NetDriver	*NetDriver;
	PACKET			*incoming[2];
	PACKET			*outgoing;	/* async send */
	PACKET			*outpak;	/* packed packet */
};

#define	PACKHEADLEN	(2*LADDRESS+2+2)	/* to, from, type, len */
#define	ONEPACKLEN	64
#define	PAKPACKLEN	512
#define	PAKGRAIN	64			/* must be power of 2 */

struct packet {
	PACKET	*next;		/* must be first! */
	Ushort	flags;		/* running length if incomming */
	short	netport;
	short	length;		/* message length */
	Ushort	size;		/* data size */
	Uchar	*address;
	Uchar	*data;
	Ulong	arrived;	/* arrival time */
};

struct SysDriver {
	char	*name;
	Ushort	flags;
	int	(FAR* Init) (void);
	void	(FAR* Term) (void);
	void	(FAR* Poll) (void);
	Ulong	(FAR* Disable) (void);
	void	(FAR* Enable) (Ulong i);
	void	(FAR* Shell) (void);
	void	(FAR* BuildFileName) (char *FullName, char *path, char *name,
					char *ext);
};

struct TmDriver {
	char	*name;
	Ushort	flags;
	int	(FAR* Init) (char *options);
	void	(FAR* Term) (void);
	Ulong	(FAR* Milli) (void);
	int	(FAR* Hires) (void);
	char *	(FAR* Ctime) (void);
	Ulong	(FAR* Interval) (int mode, Ulong res);
#define TMR_PUSH	0x0001
#define TMR_POP		0x0002
#define TMR_READ	0x0004
#define TMR_SET		0x0008
#define	TMR_START	(TMR_PUSH|TMR_SET)
#define TMR_STOP	(TMR_READ|TMR_POP)
#define TMR_RESTART	(TMR_READ|TMR_SET)
};

struct KbdDriver {
	char	*name;
	Ushort	flags;
	int	(FAR* Init) (char *options);
	void	(FAR* Term) (void);
	int	(FAR* Read) (void);
	int	(FAR* Getch) (void);
	int	(FAR* Wait) (void);
};

struct SndDriver {
	char	*name;
	Ushort	flags;
	int	(FAR* Init) (char *options);
	void	(FAR* Term) (void);
	void	(FAR* Poll) (int force);
	int 	(FAR* Beep) (int f, int milli);
	int 	(FAR* Effect) (int eff);
	int 	(FAR* List) (int *list, int id);
};

struct GrDriver {
	char	*name;
	Ushort	flags;
	DEVICE	*devices;
	int	(FAR* Init) (DEVICE *dev, char *options);
	void	(FAR* Term) (DEVICE *dev);
	void	(FAR* OpenTextWindow) (SCREEN *scr);
	void	(FAR* CloseTextWindow) (SCREEN *scr);
	void 	(FAR* FontSet) (DEVICE *dev, char *text);
	void	(FAR* TextPut) (int c);
	void 	(FAR* TextChar) (int c);
	void	(FAR* TextColor) (Uint fg, Uint bg);
	int	(FAR* CursorMode) (int mode);
	void	(FAR* MoveTo) (int x1, int y1);
	void	(FAR* DrawTo) (int x2, int y2, Uint c);
	void	(FAR* SetVisual) (int page);
	void	(FAR* SetActive) (int page);
	void	(FAR* Clear) (SCREEN *scr);
	void	(FAR* SetTextPos) (int row, int col);
	void	(FAR* PushTextPos) (void);
	void	(FAR* PopTextPos) (void);
	void	(FAR* TextClear) (void);
	void	(FAR* WriteMode) (int mode);
	void	(FAR* SetPalette) (int index, long color);
	void	(FAR* Ellipse) (int x, int y, int rx, int ry, Uint c);
	void	(FAR* Flush) (void);
	void	(FAR* Shutters) (int eye);
};

struct PtrDriver {
	char	*name;
	Ushort	flags;
	int	(FAR* Init) (POINTER *p, char *options);
	void	(FAR* Term) (POINTER *p);
	int	(FAR* Cal) (POINTER *p);
	int	(FAR* Center) (POINTER *p);
	int	(FAR* Read) (POINTER *p, int transfer);
	void	(FAR* Key) (POINTER *p, int key);
};

struct NetDriver {
	char	*name;
	Ushort	flags;
	int	(FAR* Init) (NETPORT *port, char *options);
	void	(FAR* Term) (NETPORT *port);
	int	(FAR* Send) (NETPORT *port, PACKET *pack);
	int	(FAR* Receive) (NETPORT *port);
};

struct netname {
	struct netname	*next;
	char		*name;
};

struct msg {
	HMSG		*next;
	char		*text;
	Ulong		timeout;	/* time to delete */
	Uint		flags;
#define MSG_WARN	0x0001
#define MSG_ERR		0x0002
};

#define NBUFS		2		/* must be 2 for now */

struct hdd {
	short		type;
#define	HDT_FRONT	 0
#define	HDT_NONE	 1
#define	HDT_REAR	 2
#define	HDT_MAP		 3
#define	HDT_RADAR	 4
#define	HDT_TARGET	 5
#define	HDT_PAN		 6
#define	HDT_GAZE	 7
#define	HDT_CHASE	 8
#define	HDT_FOLLOW	 9
#define	HDT_HUD		10
#define	HDT_UPFRONT	11
#define	HDT_PANEL	12
#define	HDT_RIGHT	13
#define	HDT_LEFT	14
#define	HDT_STORES	15
	Ushort		flags;
#define HDF_ON	0x0001
	VIEW		view;
	BUFFER		*bufs[NBUFS];	/* HDD window */
};
#define NVIEWERS	10

struct status {
	struct SysDriver	NEAR* system;
	struct TmDriver		NEAR* time;
	struct GrDriver		NEAR* graphics;
	struct SndDriver	NEAR* sound;
	struct KbdDriver	NEAR* keyboard;
	BODY		**bodies;
	char		*iname;			/* init file */
	char		*mname;			/* macros file */
	char		*fname;			/* font file */
	char		*vmdname;		/* video modes file */
	char		*lname;			/* log file name */
	char		*navname;		/* nav file name */
	char		*lndname;		/* landscape file name */
	char		*fdir;			/* support files directory */
	char		*ptrname;		/* pointer device name */
	char		*grname;		/* video device name */
	char		*grmname;		/*    and mode */
	char		*sndname;		/* sound device name */
	char		*kbdname;		/* keyboard device name */
	struct netname	*netnames;		/* net drivers */
	char		*ptype;			/* plane type */
	char		*dtype;			/* drone type */
	char		*options;		/* options for 'create_obj' */
	char		*nikname;		/* my handle */
	char		*teamname;		/* my team name */
	char		*homename;		/* my nav home name */
	char		*timeropts;
	Ulong		big_bang;		/* keep relative time */
	Ulong		present;
	Ulong		lasttime;
#define	TO_OBJECT	(4*1000L)
	Ulong		ObjectTimeout;
#define	TO_PLAYER	(8*1000L)
	Ulong		PlayerTimeout;
#define	TO_DRONE	(8*1000L)
	Ulong		DroneTime;		/* drone takeoff delay */
#define	TO_REFRESH	(TO_OBJECT/4)
	Ulong		RefreshTime;
	Ulong		ShutdownTime;
	Ulong		test_start;
	long		mscore;
	Ulong		nbullets;
	Ushort		ComVersion;
	Ushort		nbuffers;
	Ushort		maxbuffers;
	Ushort		flags;
#define	SF_BLANKER	0x0001
#define	SF_LANDSCAPE	0x0002
#define	SF_SKY		0x0004
#define	SF_VERBOSE	0x0008
#define	SF_INTERACTIVE	0x0010
#define	SF_SIMULATING	0x0020
#define	SF_INITED	0x0040
#define	SF_PAUSED	0x0080
#define	SF_MODES	0x0100
#define	SF_HUD		0x0200
#define	SF_HELP		0x0400
#define	SF_NET 		0x0800
#define	SF_LISTS	(SF_MODES|SF_HUD|SF_HELP|SF_NET)
#define SF_DEBUG	0x1000
#define SF_FONT		0x2000
#define SF_CLEARED	0x4000
#define	SF_MAIN		0x8000
	Ushort		flags1;
#define	SF_USEG		0x0001
#define SF_FLUSHLOG	0x0002
#define	SF_DBUFFERING	0x0004
#define	SF_TESTING	0x0008
#define	SF_STEREOREV	0x0010
#define	SF_EXTVIEW	0x0020
#define	SF_IDENT	0x0040
#define	SF_INFO		0x0080
#define	SF_SMOKE	0x0100
#define SF_TERM		0x0200
#define	SF_HUDINFRONT	0x0400
	Ushort		debug;
#define DF_GPW		0x1000
#define DF_GPX		0x2000
#define DF_GPY		0x4000
#define DF_GPZ		0x8000
	Ushort		network;
#define	NET_ON		0x0001
#define	NET_NOBCAST	0x0002
#define	NET_INITED	0x0004
#define	NET_AUTOACCEPT	0x0010
#define	NET_AUTODECLINE	0x0020
#define	NET_AUTOREPLY	(NET_AUTOACCEPT|NET_AUTODECLINE)
	short		ntargets;
	short		extview;
	short		info;
	short		stereo;		/* vision type */
#define VIS_MONO		0
#define VIS_STEREOSCOPIC	1
#define VIS_REDBLUE		2
#define VIS_ALTERNATING		3
	short		windows;	/* windows configuration */
#define WIN_TEXT	0
#define WIN_FULL	1
#define WIN_LANDSCAPE	2
#define WIN_PORTRATE	3
#define WIN_SQUARE	4
#define WIN_WIDE	5
#define WIN_PANORAMA	6
#define WIN_ETHER	7
	short		paralax;	/* stereo eye shift. MAX 100*VONE! */
	short		focus;		/* stereo eye focus distance */
	short		gap;		/* gap between stereo frames: 1/gap */
	short		quiet;
	short		tfg;		/* text fg */
	short		tbg;		/* text bg */
	short		wfg;		/* watch fg */
	short		cfg;		/* cursor color */
	short		hfg;		/* hud color */
	short		hfgi;		/* hud color (intensified)*/
	OBJECT		*owner;
	BUFFER		*bufs[NBUFS];	/* main window */
	BUFFER		*buf[2];	/* current writable buffer */
	short		which_buffer;
	short		nobjects;
	long		object_id;
	OBJECT		*world[2];	/* all objects in world */
#define CO	st.world[0]
#define COT	st.world[1]
	OBJECT		*land[2];	/* all objects in landscape */
#define CL	st.land[0]
#define CLT	st.land[1]
	OBJECT		*viewer;	/* origin of view */
#define CV	st.viewer
	OBJECT		*control;	/* object under user control */
#define CC	st.control
	VIEW		view[1];	/* active view */
#define CVIEW	st.view
#define CP	st.view->viewport
#define CW	st.view->window
#define CS	st.view->screen
	SCREEN		*textscreen;	/* text screen info */
#define CT	st.textscreen
#define NHDD	10
	HDD		hdd[NHDD];	/* HDD windows */
	int		interval;	/* milliseconds of last frame */
	int		dither;		/* rand () % 1000 */
	POINTER		*pointer;	/* kludge for oplane.c */
	int		StFont;
	Uchar	NEAR* NEAR* StFontPtr;	/* StFonts[StFont] */
	int		StFontSize;
	int		landx;		/* land square: x and y */
	int		landy;
	int		gravity;	/* meters/sec/sec */
	HMSG		*msg;		/* message queue */
	int		drones;		/* number of drones */
	int		killers;	/* how many drones are killers */
	int		SkyLines;	/* number of lines in sky */
	short		home;		/* home base nav point */
	Ushort		btnmode;	/* buttons shift mode */
	short		nMacros;	/* max number of keyboard macros */
	long		lineno;

	int		black;
	int		white;
	int		red;
	int		blue;
	int		green;
	int		magenta;
	int		brown;
	int		gray;
	int		lblue;
	int		lred;
	int		hudlow;
	int		hudhigh;
	int		lgray;
	int		skyblue;
	int		ground;

	PLAYER NEAR	*all_known;	/* for remote_* () usage */
	PLAYER NEAR	*all_active;	/* -"- */
	PLAYER NEAR	*all_team;	/* -"- */
	PLAYER NEAR	*all_ports;	/* -"- */
	PLAYER NEAR	*all_pports;	/* -"- */
	PLAYER NEAR	*all_players;	/* internal to menus.c */
	PLAYER NEAR	*no_players;	/* -"- */
	int		misc[20];
	long		stats[100];
	char		filename[1024];	/* file name work area */
};

#define Sys	st.system
#define Tm	st.time
#define Gr	st.graphics
#define Snd	st.sound
#define Kbd	st.keyboard

#define GACC	st.gravity

#define VP	view->viewport
#define VW	view->window
#define VS	view->screen

struct FldTab {
	int	type;
	void	*p;
};

struct hud {
	short	flags;
#define HF_ETHER	0x0001	/* this is an ether display/HUD */
#define HF_ETHERFRAME	0x0002	/* use ether style frame indicators */
	short	orgx;		/* window center */
	short	orgy;
	short	maxx;		/* window size */
	short	maxy;
	short	shifty;		/* hud down shift */
	short	cx;		/* hud center */
	short	cy;
	short	sx;		/* hud size */
	short	sy;
	short	clipx;
	short	clipy;
	short	clipr;		/* bounding rectangle (relative) */
	short	clipl;
	short	clipt;
	short	clipb;
	short	right;		/* bounding rectangle (absolute) */
	short	left;
	short	top;
	short	bottom;
	short	tx;		/* tick size */
	short	ty;
	short	ttx;		/* 'big' adjusted tick size */
	short	tty;
	short	ss;		/* stroke size */
	short	dd;		/* digit width */
	ANGLE	width;		/* horizontal aperture */
	ANGLE	height;		/* vertical aperture */
	short	fg;		/* color (normal) */
	short	fgi;		/* color (intensified) */
	int	VV[2];		/* vv position */
	short	etherx;		/* ether frame x centerline */
	short	ethery;		/* ether frame y centerline */
	short	ethertx;	/* ether frame x half-width */
	short	etherty;	/* ether frame y half-width */
	short	misc[10];
};
typedef struct hud HUD;

struct ils {
	char	*name;
	LVECT	R;		/* runway position */
	short	longitude;
	short	latitude;
	short	l[2];		/* localizer relative x/y */
	short	g[2];		/* glide-path relative x/y */
	ANGLE	localizer;	/* heading of forward beam */
	ANGLE	glidepath;	/* pitch of beam */
};

#include "extern.h"

#endif
