/* --------------------------------- hud.c ---------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* paint the Head Up Display.
*/

#include "fly.h"
#include "plane.h"

#define DOGRAPH		(st.debug & DF_GPX)

static void NEAR show_vv (HUD *h, VIEW *view, OBJECT *p, int color);
static void NEAR show_wl (HUD *h, OBJECT *p, int color);
static void NEAR alarm_off (void);
static void NEAR show_xbreak (HUD *h, OBJECT *p, int color);
static void NEAR hud_alarm (HUD *h, OBJECT *p, int color, int mode, int hon);
static int  NEAR is_in (HUD *h, int x, int y, int dx, int dy);
static void NEAR show_waypoint (HUD *h, VIEW *view, OBJECT *p);
static void NEAR show_history (HUD *h, VIEW *view, OBJECT *p);


extern void FAR
show_hud (VIEW *view, OBJECT *pov, OBJECT *p, int orgx, int orgy,
	int maxx, int maxy, int mode)
{
	int	hud, hud1, hud2, big, htype, hudarea, limit, front, hon;
	int	sx, sy, tx, ty, ttx, tty;
	int	x, y, ss, clipx, clipy, shifty;
	ANGLE	a;
	HUD	h[1];

/* hud only for planes, in front view
*/
	if (!IS_PLANE(CV) || (HDT_HUD != mode && !scenery (mode))) {
		alarm_off ();
		return;
	}

	hud = EX->hud;
	hud1 = EX->hud1;
	hud2 = EX->hud2;
	big = hud & HUD_BIG;
	htype = hud1 & HUD_TYPES;
	front = !view->viewport->rotz && !view->viewport->rotx;
	x = HDT_FRONT == mode && (hud & HUD_ON) && front;
	hon = x || HDT_HUD == mode;
	limit = (x && (hud1 & HUD_LIMIT)) || HDT_HUD == mode;

	get_square (view, maxx, maxy, &sx, &sy);

	if (0 == (tx = (sx+32)/64))
		tx = 1;
	if (0 == (ty = (sy+32)/64))
		ty = 1;

	shifty = EX->hudshift;
	if (HDT_HUD == mode) {
		orgy -= fmul (sy, shifty);
	} else {
		y = sy + fmul (sy, shifty);
		hudarea = fdiv (maxy, y);		/* largest allowed */
		a = DEG2ANG (EX->hudarea);
		y = muldiv (VP->z, SIN (a), COS (a));
		if (y < VP->maxy) {			/* fits in window */
			y = muldiv (y, maxy, VP->maxy);	/* y pixels */
			y = fdiv (y, sy);		/* ratio */
			if (y < hudarea)		/* fits in square */
				hudarea = y;
		}
		sx = fmul (sx, hudarea);
		sy = fmul (sy, hudarea);
	}

	ss = fmul (sy, EX->hudFontSize);
	if (ss < 8)
		ss = 8;

	if (!big && HUD_CLASSIC == htype) {
		x = 3*tx;
		y = 2+tx + num_size (99L, ss);
		if (y > x)
			x = y;
		if (maxx-x < sx) {
			x = maxx-x;
			sy = muldiv (sy, x, sx);
			sx = x;
		}

		y = 2+3*ty+ss;
		if (maxy-y < sy) {
			y = maxy-y;
			sx = muldiv (sx, y, sy);
			sy = y;
		}
	}
	if (0 == (tx = (sx+32)/64))
		tx = 1;
	if (0 == (ty = (sy+32)/64))
		ty = 1;

	shifty = fmul (sy, shifty);

	if (limit) {
		clipx = sx;
		clipy = sy;
	} else {
		clipx = maxx;
		clipy = maxy;
	}

	h->flags = 0;

	h->orgx = orgx;		/* window sizes */
	h->orgy = orgy;
	h->maxx = maxx;
	h->maxy = maxy;

	h->shifty = shifty;		/* hud sizes */
	h->cx = orgx;
	h->cy = orgy+shifty;
	h->sx = sx;
	h->sy = sy;

	h->clipx =  clipx;		/* clip sizes */
	h->clipy =  clipy;
	h->clipr =  clipx;		/* clip rectangle, relative */
	h->clipl = -clipx;
	h->clipt =  clipy - shifty;
	h->clipb = -clipy - shifty;

	h->right  = orgx + h->clipr;	/* hud border, absolute */
	h->left   = orgx + h->clipl;
	h->top    = orgy - h->clipt;
	h->bottom = orgy - h->clipb;

	h->tx = h->ttx = tx;
	h->ty = h->tty = ty;
	h->ss = ss;
	h->dd = num_size (9L, ss);
	h->width = 0;
	h->height = 0;
	h->fg = st.hfg;
	h->fgi = st.hfgi;
	h->VV[X] = h->VV[Y] = 0;

	if (WIN_ETHER == st.windows && HUD_ETHER == (EX->hud1 & HUD_TYPES)) {
	    	h->flags |= HF_ETHER;
		if (st.flags & SF_MAIN)
		    	h->flags |= HF_ETHERFRAME;
	}

	if (h->flags & HF_ETHERFRAME) {
		get_area (&st.hdd[4].view, 0, 0, &x, &y);
		h->etherx  = (h->maxx + x) / 2 + 1;
		h->ethery  = (h->maxy + y) / 2 + 1;
		x = h->etherx - h->maxx - 2;
		y = h->ethery - h->maxy - 2;
		get_square (&st.hdd[4].view, x*8, y*8, &x, &y);
		h->ethertx = x/8;
		h->etherty = y/8;
	}

	hud_alarm (h, p, st.hfgi, mode, hon);

	if (hon) {
		if (big) {
			h->ttx = -h->ttx;
			h->tty = -h->tty;
		}
		ttx = h->ttx;
		tty = h->tty;
/* hud border
*/
		if ((hud1 & HUD_BORDER) && HDT_FRONT == mode)
			show_rect (h->cx, h->cy, h->sx, h->sy, st.gray, 0);

/* velocity vector
*/
		show_vv (h, view, p, st.hfg);

/* waterline mark.
*/
		show_wl (h, p, st.hfg);

/* pitch ladder is centered on the vv or the waterline mark
*/
		show_pitch (h, view, p, sx, sy, maxx, maxy, orgx, orgy,
			ttx, tty, tx, ty, ss, shifty, mode, h->VV);

/* heading on upper/lower edge
*/
		show_heading (h, view, p, sx, sy, maxx, maxy, orgx, orgy,
			ttx, tty, tx, ty, ss, shifty, h->VV);

/* waypoint (experimental).
*/
		show_waypoint (h, view, p);

/* altitude on right edge
*/
		show_altitude (h, p, sx, sy, maxx, maxy, orgx, orgy, ttx, tty,
			tx, ty, ss, shifty, h->VV);
/* speed on left edge
*/
		show_speed (h, p, sx, sy, maxx, maxy, orgx, orgy,
			ttx, tty, tx, ty, ss, shifty, h->VV);
/* ILS
*/
		show_ils (h, p, sx, sy, orgx, orgy, ss, shifty);

/* Bullets trail history.
*/
		show_history (h, view, p);
	}

/* Show radar stuff
*/
	if (HDT_HUD == mode || scenery (mode))
		show_radar (h, view, p, pov, orgx, orgy, maxx, maxy, tx, ty,
			ss, clipx, clipy, sx, sy, limit ? shifty : 0, h->VV,
			mode, hon);

/* ailerons/elevators cursor (helps keypad/mouse mode)
*/
	if ((hud & HUD_CURSOR) && p->pointer) {
		x = orgx + muldiv (-p->pointer->a[0], sx-tx, 100);
		y = orgy + muldiv (-p->pointer->a[1], sy-ty, 100) + shifty;

		show_rect (x, y, tx, ty, st.cfg, 0);
	}

/* cross hair
*/
	if ((hud & HUD_PLUS) || (HDT_MAP == mode || HDT_RADAR == mode))
		show_bplus (h->orgx, h->orgy, h->tx*3, h->ty*3, FCON(0.125),
			st.hfg);

/* debug: show Cm/alpha graph.
*/
	if (DOGRAPH) {
		show_plus (h->cx, h->cy, h->sx, h->sy, st.cfg);
		x = fmul (EX->aoa, sx*3);
		if (x > sx)
			x = sx;
		else if (x < -sx)
			x = -sx;
		y = EX->misc[5];
		if (y > 10000)
			y = 10000;
		else if (y < -10000)
			y = -10000;
		y = muldiv (y , sy-ty, 10000);
		x = h->cx + x;
		y = h->cy - y;
		show_rect (x, y, tx, ty, st.cfg, 0);	/* show point */
	}
}

/* Show the flight path marker and (optionally) the flight director.
*/
static void NEAR
show_vv (HUD *h, VIEW *view, OBJECT *p, int color)
{
	int	x, y, tx, ty, rx, ry, type, ret, blink;
	int	sa, ca;
	VECT	RR;

	if (!(EX->hud & HUD_VV) && !(EX->hud2 & HUD_DIRECTOR)) {
		VVDELAY = 0;
		return;
	}

	rx = fmul (h->sx, RVV);
	ry = fmul (h->sy, RVV);

	type = EX->hud1 & HUD_TYPES;
	if (type == HUD_F16) {
		tx = fmul (h->sx, EX->ldgap);
		ty = ry*2;
	} else if (type == HUD_F15) {
		tx = rx*2;
		ty = ry*2;
	} else {
		tx = fmul (h->sx, SVV);
		ty = fmul (h->sy, SVV*4/5);
	}

	if (p->speed > 4*VONE) {
		Vcopy (RR, EX->v);
		screen_coords (view, RR);
		ret = clip_to_screen (h->VV, RR, h->maxx, h->maxy,
				h->clipx-tx, h->clipy-ry, h->shifty);
		if (2 == ret)
			goto no_fpm;
	} else
		ret = 0;
	if (h->VV[Y] > h->clipt-ty) {
		h->VV[Y] = h->clipt-ty;
		ret = 1;
	}
	if (1 == ret) {
		blink = ((int)st.present)&0x080;	/* blink rate: 4/sec */
		if (blink)
			goto no_fpm;
	}

	x = h->orgx + h->VV[X];
	y = h->orgy - h->VV[Y];

	show_fpm (x, y, rx, ry, tx, ty, color, type != HUD_CLASSIC);
no_fpm:
	if (EX->hud2 & HUD_VW) {
		if (ret) {
			if ((VVDELAY += st.interval) > VVPERIOD)
				VVDELAY = VVPERIOD;
		} else if (VVDELAY > 0) {
			if ((VVDELAY -= st.interval) < 0)
				VVDELAY = 0;
		}
	} else
		VVDELAY = 0;

	if ((EX->hud2 & HUD_DIRECTOR) &&
	    (EX->misc[12] || EX->misc[13])) {
		x = rx;
		rx = tx;
		tx = x*8;

		y = ry;
		ry = ty;
		ty = y*8;

		sa = SIN(EX->misc[13]);		/* roll */
		ca = COS(EX->misc[13]);

		y = EX->misc[12];		/* pitch */
		if (y > DEG(30))
			y = DEG(30);
		else if (y < -DEG(30))
			y = -DEG(30);
		x = muldiv (h->sx/2, y, DEG(30));
		y = muldiv (h->sy/2, y, DEG(30));
		x = fmul (x, sa);
		y = fmul (y, ca);

		x += h->VV[X];
		y += h->VV[Y];
#if 0
		if (x > h->clipr-tx)
			x = h->clipr-tx;
		else if (x < h->clipl+tx)
			x = h->clipl+tx;
		if (y > h->clipt-ty)
			y = h->clipt-ty;
		else if (y < h->clipb+ty)
			y = h->clipb+ty;
#endif
		x = h->orgx + x;
		y = h->orgy - y;
		show_dir1 (x, y, rx, ry, tx, ty, sa, ca, h->fgi,
			h->orgx, h->orgy, h->sx, h->sy, h->shifty);
	}
}

/* Show the waterline mark.
*/
static void NEAR
show_wl (HUD *h, OBJECT *p, int color)
{
	int	tx, ty;

	if (EX->hud & HUD_PLUS)
		return;

	if (VVDELAY > 0 ||
	    ((HUD_FA18 == (EX->hud1 & HUD_TYPES)) && (EX->equip & EQ_GEAR))) {
		tx = fmul (h->sx, SVV);
		ty = fmul (h->sy, SVV);
		show_w (h->orgx, h->orgy, tx, ty, color);
	}
}

extern void FAR
show_num (int x, int y, long t, int s, int c, int orgx, int orgy, int maxx,
	int maxy, int shifty)
{
	int	dxs, dxc, dys, dyc, l, h;

	num_extent (t, s, &dxs, &dxc, &dys, &dyc);

	--maxx;			/* fight truncation errors */
	--maxy;

	l = orgx-maxx-x;
	h = orgx+maxx-x;
	if (0 > h || 0 < l)
		return;
	if (dxc > h || dxc < l)
		return;
	if (-dys > h || -dys < l)
		return;
	if (dxc-dys > h || dxc-dys < l)
		return;

	l = orgy+shifty-maxy-y;
	h = orgy+shifty+maxy-y;
	if (0 > h || 0 < l)
		return;
	if (-dxs > h || -dxs < l)
		return;
	if (-dyc > h || -dyc < l)
		return;
	if (-dxs-dyc > h || -dxs-dyc < l)
		return;

	stroke_num (x, y, t, s, c);
}

extern void FAR
add_segment (int x1, int y1, int x2, int y2, int c, int orgx, int orgy,
	int sx, int sy, int shifty)
{
	int	i, z1, z2, xl, xh, yl, yh;

/* Not quite midpoint clipping, if both ends are out then we reject the
 * segment which is mostly ok.
*/
	xh = orgx+sx;
	xl = orgx-sx;
	yh = orgy+sy+shifty;
	yl = orgy-sy+shifty;

	z1 = x1>xh || x1<xl || y1>yh || y1<yl;
	z2 = x2>xh || x2<xl || y2>yh || y2<yl;

	if (z1) {
		if (z2)
			return;
		i = x1; x1 = x2; x2 = i;
		i = y1; y1 = y2; y2 = i;
	} else if (!z2) {
		add_line (x1, y1, T_MOVE);
		add_line (x2, y2, c);
		return;
	}

	add_line (x1, y1, T_MOVE);

	i = iabs(x2-x1);
	z1 = iabs(y2-y1);
	if (i < z1)
		i = z1;
	for (; i > 1; i >>= 1) {
		z1 = (x1 + x2)/2;
		z2 = (y1 + y2)/2;
		if (z1>xh || z1<xl || z2>yh || z2<yl) {
			x2 = z1;
			y2 = z2;
		} else {
			x1 = z1;
			y1 = z2;
		}
	}

	add_line (x1, y1, c);
}

extern void FAR
add_dash (int x1, int y1, int x2, int y2, int ndash, int ratio, int c,
	int orgx, int orgy, int sx, int sy)
{
	register int	dx, dy;
	int		i, rx, ry, xl, xh, yl, yh;

	if (!ndash)
		return;

	xl = orgx - sx;
	xh = orgx + sx;
	yl = orgy - sy;
	yh = orgy + sy;

	ratio /= ndash;
	x2 -= x1;
	y2 -= y1;

/* Do symmetric truncation.
*/
	if (x2 < 0)
		rx = -fmul (-x2, ratio);
	else
		rx = fmul (x2, ratio);
	if (y2 < 0)
		ry = -fmul (-y2, ratio);
	else
		ry = fmul (y2, ratio);

	for (i = 0; i < ndash; ++i) {
		dx = x1 + muldiv (x2, i, ndash);
		dy = y1 + muldiv (y2, i, ndash);
		if (dx < xl || dx > xh || dy < yl || dy > yh)
			continue;
		add_line (dx, dy, T_MOVE);
		dx += rx;
		dy += ry;
		if (dx < xl || dx > xh || dy < yl || dy > yh)
			continue;
		add_line (dx, dy, c);
	}
}

extern void FAR
screen_coords (VIEW *view, VECT RR)
{
	int	s;

	s = VP->z;				/* get minimum */
	if (s > VP->maxx)
		s = VP->maxx;
	if (s > VP->maxy)
		s = VP->maxy;
	RR[X] = muldiv (RR[X], s, VP->maxx);
	RR[Z] = muldiv (RR[Z], s, VP->maxy);
	RR[Y] = muldiv (RR[Y], s, VP->z);
}

/* clip the point in R into the screen point D. Note that is the point is
 * inside the screen then a simple projection is done. Otherwise, a point
 * on the edge is returned. If the depth is negative then still clip to
 * the edge.
*/
extern int FAR
clip_to_screen (int D[2], VECT R, int maxx, int maxy, int clipx, int clipy,
	int shifty)
{
	int	off_screen, clip, x, y, ry, t;

	off_screen = 0;			/* some classic clipping */

	if (R[Y] <= 0) {
		ry = -R[Y];
		clip = 0;
	} else
		clip = ry = R[Y];

/* Establish position relative to the clipping pyramid for the screen.
*/
	if (R[X] >= clip)
		off_screen |= 1;	/* right */
	else if (-R[X] >= clip)
		off_screen |= 2;	/* left */

	if (R[Z] >= clip)
		off_screen |= 4;	/* top */
	else if (-R[Z] >= clip)
		off_screen |= 8;	/* bottom */

/* Resolve the corner areas into the correct clipping edge.
*/
	if (off_screen == 5)		/* top right */
		if (R[X] > R[Z])
			off_screen = 1;
		else
			off_screen = 4;
	else if (off_screen == 9)	/* bottom right */
		if (R[X] > -R[Z])
			off_screen = 1;
		else
			off_screen = 8;
	else if (off_screen == 6)	/* top left */
		if (-R[X] > R[Z])
			off_screen = 2;
		else
			off_screen = 4;
	else if (off_screen == 10)	/* bottom left */
		if (-R[X] > -R[Z])
			off_screen = 2;
		else
			off_screen = 8;
	else
		{}

/* Now do the projection and clipping together.
*/
	switch (off_screen) {
	default:
	case 0:
		if (ry == 0)
			x = y = 0;
		else {
			x = muldiv (maxx, R[X], ry);
			y = muldiv (maxy, R[Z], ry);
		}
		break;
	case 1:						/* right */
		x = maxx;
		if (R[X] == 0)
			y = 0;
		else
			y = muldiv (maxy, R[Z], R[X]);
		break;
	case 2:						/* left */
		x = -maxx;
		if (R[X] == 0)
			y = 0;
		else
			y = -muldiv (maxy, R[Z], R[X]);
		break;
	case 4:						/* top */
		if (R[Z] == 0)
			x = 0;
		else
			x = muldiv (maxx, R[X], R[Z]);
		y = maxy;
		break;
	case 8:						/* bottom */
		if (R[Z] == 0)
			x = 0;
		else
			x = -muldiv (maxx, R[X], R[Z]);
		y = -maxy;
		break;
	}
	if (off_screen)
		off_screen = clip ? 1 : 2;

/* Finally check for 2D clipping (for the window) and do it.
*/
	if (x >= clipx) {
		y = muldiv (clipx, y, x);
		x = clipx;
		if (!off_screen)
			off_screen = 1;
	} else if (x <= -clipx) {
		y = -muldiv (clipx, y, x);
		x = -clipx;
		if (!off_screen)
			off_screen = 1;
	}
	if (y >= (t = clipy-shifty)) {
		x = muldiv (t, x, y);
		y = t;
		if (!off_screen)
			off_screen = 1;
	} else if (y <= (t = -clipy-shifty)) {
		x = muldiv (t, x, y);
		y = t;
		if (!off_screen)
			off_screen = 1;
	}

	D[X] = x;
	D[Y] = y;
	return (off_screen);
}

extern void FAR
clip_to_ether (HUD *h, int D[2], int x, int y)
{
	if (D[X] >= x)
		D[X] = h->etherx;
	else if (D[X] <= -x)
		D[X] = -h->etherx;
	if (D[Y] >= y)
		D[Y] = h->ethery;
	else if (D[Y] <= -y)
		D[Y] = -h->ethery;
}

extern void FAR
keep_inside (int *x, int *y, int xl, int xh, int yl, int yh, int orgx,
	int orgy, int clipx, int clipy, int shifty)
{
	int	t;

	if (xl > xh)
		(t = xl, xl = xh, xh = t);
	if (*x > (t = orgx+clipx-xh))
		*x = t;
	else if (*x < (t = orgx-clipx-xl))
		*x = t;

	if (yl > yh)
		(t = yl, yl = yh, yh = t);
	orgy += shifty;
	if (*y > (t = orgy+clipy-yh))
		*y = t;
	else if (*y < (t = orgy-clipy-yl))
		*y = t;
}

extern void FAR
hud_setup (OBJECT *p)
{
	int	font;

	EX->hud  &= HUD_ON|HUD_CURSOR|HUD_ROSS;
	EX->hud1 &= HUD_TYPES|HUD_INAME|HUD_IDIST|HUD_PANEL;
	EX->hud2 &= HUD_ILS|HUD_XGRID;
	EX->ladder = 0;

	EX->hud  |= HUD_DEFAULT|HUD_LADDER;
	EX->hud1 |= HUD_VALARM|HUD_AALARM|HUD_LIMIT|HUD_BORDER;
	EX->hud2 |= HUD_HEADING|HUD_ALTITUDE|HUD_SPEED|HUD_XBREAK;

	switch (EX->hud1 & HUD_TYPES) {
	case HUD_F15:
		EX->hud  |= HUD_FINE|HUD_PLUS|HUD_DATA;
		EX->hud1 |= HUD_KNOTS|HUD_ACCVECT;
		EX->hud2 |= HUD_HIDETGT;
		EX->ladder |= LD_NEGTIP;
		EX->hudarea = 10;
		EX->hudshift = FCON (0.40);
		EX->ldgap = FCON(0.077);
		EX->ldstep = FCON(0.238);
		EX->ldstep0 = FCON(0.391);
		EX->ldstepg = FCON(0.61);	/* ??? */
		EX->ldtip = FCON (0.03);
		EX->ldndash = 5;
		font = 1;
		EX->hudFontSize = FCON(0.06);
		break;
	case HUD_F16:
		EX->hud  |= HUD_PLUS|HUD_DATA;
		EX->hud1 |= HUD_KNOTS|HUD_ACCVECT|HUD_PENDULUM|HUD_CORNER;
		EX->hud2 |= HUD_HISTORY|HUD_TPOINTER;
		EX->ladder |= LD_NEGTIP;
		EX->hudarea = 11;
		EX->hudshift = FCON (0.50);
		EX->ldgap = FCON(0.085);
		EX->ldstep = FCON(0.220);
		EX->ldstep0 = FCON(0.58);
		EX->ldstepg = FCON(0.58);
		EX->ldtip = FCON (0.03);
		EX->ldndash = 4;
		font = 1;
		EX->hudFontSize = FCON(0.045);
		break;
	case HUD_FA18:
		EX->hud  |= HUD_FULLHEADING|HUD_DATA;
		EX->hud1 |= HUD_KNOTS|HUD_PENDULUM;
		EX->hud2 |= HUD_VW;
		EX->ladder |= LD_SLANT|LD_ZENITH|LD_UNDER|LD_TIP0|LD_NEGTIP;
		EX->hudarea = 10;
		EX->hudshift = FCON (0.4);
		EX->ldgap = FCON(0.11);
		EX->ldstep = FCON(0.23);
		EX->ldstep0 = FCON(0.34);
		EX->ldstepg = FCON(0.96);
		EX->ldtip = FCON (0.06);
		EX->ldndash = 5;
		font = 1;
		EX->hudFontSize = FCON(0.06);
		break;
	case HUD_ETHER:
		EX->hud  |= HUD_BIG|HUD_XFINE|HUD_DATA|HUD_PLUS;
		EX->hud1 &= ~HUD_BORDER;
		EX->hud1 |= HUD_KNOTS|HUD_TOP;
		EX->hud2 |= HUD_VW;
		EX->ladder |= LD_NEGTIP;
		EX->hudarea = 15;
		EX->tapelen = 16;		/* scales length */
		EX->hudshift = 0;
		EX->ldgap = FONE/32*5;
		EX->ldstep = FONE/32*12;
		EX->ldstep0 = FONE/32*18;
		EX->ldstepg= FONE/32*18;
		EX->ldtip = FCON (0.03);
		EX->ldndash = 5;
		font = 1;
		EX->hudFontSize = FCON(0.06);
		break;
	default:
		EX->hud  |= HUD_BIG|HUD_XFINE|HUD_PLUS|HUD_DATA;
		EX->hud1 |= HUD_KNOTS|HUD_TOP;
		EX->hud2 |= HUD_VW;
		EX->ladder |= LD_ERECT|LD_NEGTIP;
		EX->hudarea = 11;
		EX->tapelen = 16;		/* scales length */
		EX->hudshift = FCON (0.3);
		EX->ldgap = FONE/32*5;
		EX->ldstep = FONE/32*12;
		EX->ldstep0 = FONE/32*18;
		EX->ldstepg= FONE/32*18;
		EX->ldtip = FCON (0.03);
		EX->ldndash = 5;
		font = 0;
		EX->hudFontSize = FCON(0.05);
		break;
	}
	if (MODEL_CLASSIC != EP->opt[0])
		EX->hud  |= HUD_VV;

	if (!CC || CC == p)
		font_set (font);
}

extern char * FAR
get_wname (int w)
{
	switch (w) {
	case WE_M61:
		return ("GUN");
	case WE_MK82:
		return ("MK82");
	default:
		return ("XXX");
	}
}

static void NEAR
alarm_off (void)
{
	if (-1 != TnAlarm[0])
		TnAlarm[0] = -1;
	if (-1 != TnWarn[0])
		TnWarn[0] = -1;
}

static void NEAR
show_xbreak (HUD *h, OBJECT *p, int color)
{
	int	dx, dy, x, y, t;

	dy = h->sy/2;
	dx = h->sx/2;
	y = h->orgy + h->shifty - h->sy + dy;

	if (p->R[Z] <= 0L || (t = -p->V[Z]/VONE*5) == 0)
		x = 0;
	else
		x = muldiv (h->sx/2, (int)(p->R[Z]/VONE), t);

	add_line (h->orgx-x-dx, y-dy, T_MOVE);
	add_line (h->orgx-x,    y,    color);
	add_line (h->orgx-x-dx, y+dy, color);

	add_line (h->orgx+x+dx, y-dy, T_MOVE);
	add_line (h->orgx+x,    y,    color);
	add_line (h->orgx+x+dx, y+dy, color);
}

static void NEAR
hud_alarm (HUD *h, OBJECT *p, int color, int mode, int hon)
{
	int	alarm, valarm, balarm, blink, ss2, ss3, y;
	long	t;

/* Put high priority alarms last to activate the most urgent aural warning.
*/
	blink = ((int)st.present)&0x0080;	/* 128ms period */
	alarm = 0;
	ss2 = h->ss*2;
	ss3 = h->ss*3;
	y = h->orgy + h->shifty;

	if (IS_PLANE(p) && (p->gpflags & GPF_PILOT)) {
		valarm = ((EX->hud & HUD_ON) && (EX->hud1 & HUD_VALARM))
			 || HDT_HUD == mode;
		balarm = valarm && blink;
		if (EX->flags & PF_GLIMIT) {
			alarm = 2;
			if (balarm)
				stroke_str (h->orgx-h->dd*6, y-h->ss*5,
					"GLIMIT", ss2, color);
		}
		if (EX->flags & PF_STALL) {
			alarm = 2;
			if (balarm)
				stroke_str (h->orgx-h->dd*5, y-ss3, "STALL",
					ss2, color);
		}
		if (EX->fuel <= 0)
			alarm = 2;
		if (valarm) {
			t = 1000L * EX->fuel / EP->fuel_capacity;
			t = (t/1000)*1000+1;
			if (t < 10000L && (st.present%t) < 200)
				stroke_str (h->orgx-h->dd*4, y+h->ss*5, "FUEL",
					ss2, color);
		}
		CL->next->flags &= ~F_VISIBLE;
		if (!(EX->flags & PF_ONGROUND) && !(EX->equip & EQ_GEAR)) {
			if (p->R[Z] > 0 && p->R[Z]+p->V[Z]*5L <= 0) {
				if (hon && (EX->hud2 & HUD_XBREAK))
					show_xbreak (h, p, color);
				if (EX->hud2 & HUD_XGRID)
					CL->next->flags |= F_VISIBLE;
				if (p->R[Z]+p->V[Z]*2L < 0) {
					alarm = 1;
					if (balarm)
						stroke_str (h->orgx-h->dd*10,
							y+ss3,
							"PULL UP", ss3, color);
				}
			}
		}
		if (p->damage <= 0) {
			alarm = 1;
			if (balarm)
				stroke_str (h->orgx-h->dd*8, y, "EJECT",
					ss3, color);
		}
		if (!(EX->hud1 & HUD_AALARM))
			alarm = 0;
	}

	switch (alarm) {
	default:
	case 0:
		alarm_off ();
		break;
	case 1:
		if (st.quiet && -1 == TnAlarm[0]) {
			TnAlarm[0] = 0;
			Snd->List (TnAlarm, 0);
		}
		break;
	case 2:
		if (st.quiet && -1 == TnWarn[0]) {
			TnWarn[0] = 0;
			Snd->List (TnWarn, 0);
		}
		break;
	}
}

static int NEAR
is_in (HUD *h, int x, int y, int dx, int dy)
{
	return (x < h->right-dx && x > h->left+dx &&
		y > h->top+dy   && y < h->bottom-dy);
}

static void NEAR
show_waypoint (HUD *h, VIEW *view, OBJECT *p)
{
	LVECT	*R;
	OBJECT	*target;
	VECT	RR;
	int	D[2];
	int	ind, out, x, y, tx, ty, rx, ry;
	long	hx, hy, t;
	ANGLE	a;

	if (!(EX->hud2 & HUD_WAYPOINT))
		return;

	if (T(target = EX->target) && target->id == EX->tid) {
		R = &target->R;
		ind = 0;
	} else {
		ind = EX->ils;
		if (ind < 1)
			ind = 1;
		R = &ils[ind-1].R;
		ind = 1;
	}

/* Object highlighting diamond.
*/
	if (ind) {
		tx = (h->dd+1)/2;
		ty = (muldiv (h->dd, h->sy, h->sx)+1)/2;

		objects_show (1, view, p, 0, *R, RR);
		if (h->flags & HF_ETHER) {
			x = h->maxx;
			y = h->maxy;
		} else {
			x = h->sx;
			y = h->sy;
		}
		out = clip_to_screen (D, RR, h->maxx, h->maxy, x-tx,
				y-ty, h->shifty);
		if (out && h->flags & HF_ETHER) {
			clip_to_ether (h, D, x-tx, y-ty);
			tx = h->ethertx;
			ty = h->etherty;
			out = 0;
		}
		D[X] = h->orgx + D[X];
		D[Y] = h->orgy - D[Y];
		show_diamond (D[X], D[Y], tx, ty, st.hfg, out);
	}

/* Object position pointer.
*/
	hx = ((*R)[X] - p->R[X])/VONE;
	hy = ((*R)[Y] - p->R[Y])/VONE;
	t = labs(hx) + labs(hy);
	while (t > 0x7fffL) {
		hx >>= 1;
		hy >>= 1;
		t  >>= 1;
	}
	a = ATAN ((int)hx, (int)hy) + p->a[Z];

	tx = EX->ldstep + 2*EX->hudFontSize;
	x = fmul (h->sx, tx);			/* pointer position */
	y = fmul (h->sy, tx);
	if (a < -DEG(5)) {
		x = -x;
		y = -y;
	} else if (a < DEG(5)) {
		x = muldiv (x, a, DEG(5));
		y = muldiv (y, a, DEG(5));
	}
	x = fmul (x, p->cosy);
	y = fmul (y, p->siny);
	x = h->orgx + h->VV[X] + x;
	y = h->orgy - h->VV[Y] - y;

	tx = 4*h->tx;				/* pointer size */
	ty = 4*h->ty;
	rx = tx/4;				/* circle radius */
	ry = ty/4;
	tx = fmul (tx, SIN(a));
	ty = fmul (ty, COS(a));

	if (is_in (h, x, y, rx, ry) && is_in (h, x+tx, y-ty, 0, 0))
		show_ptr (x, y, rx, ry, tx, ty, st.hfg,
			(EX->hud1 & HUD_TYPES) != HUD_CLASSIC);
}

/* Show bullets trail history.
*/

#define BULSEG	(int)(750/3.28)		/* segment size (meters) */
#define BULSTEP	(2*BULSEG)		/* marked step size */
#define BULMIN	(BULSEG/2)		/* nearest point shown */
#define BULMAX	(3*BULSTEP)		/* farthest point shown */

static void NEAR
show_history (HUD *h, VIEW *view, OBJECT *p)
{
	OBJECT	*target;
	HISTORY	*h1, *h2;
	VECT	P, Q, R, V;
	int	D1[2], D2[2];
	int	d, n, d0, d1, dd, dd0, dd1, ddd;
	int	off, tx, ty, first;

	if (!(EX->hud2 & HUD_HISTORY))
		return;

	if (T(target = EX->target)) {
		ddd = (int)(ldist3d (p->R, target->R)/VONE);
		if (ddd > BULMAX)
			ddd = BULMAX;
	} else
		ddd = BULMAX;
	tx = fmul (h->sx, F16RBUL);
	ty = fmul (h->sy, F16RBUL);
	D2[X] = h->orgx;
	D2[Y] = h->orgy;
	off = 0;
	if (ddd < BULSEG) {
		n = BULSEG;
		d = ddd;
	} else {
		d = BULSEG;
		n = d + BULSEG;
	}
	first = 1;
	for (h2 = 0, h1 = EX->history; h1; h2 = h1, h1 = h1->next) {
		if ((d1 = (int)(ldist3d (p->R, h1->R)/VONE)) < d)
			continue;
		D1[X] = D2[X];
		D1[Y] = D2[Y];
		off = 1 + d1/1000;
		if (h2) {
			d0 = (int)(ldist3d (p->R, h2->R)/VONE);
			P[X] = (int)((h2->R[X] - p->R[X])/off);
			P[Y] = (int)((h2->R[Y] - p->R[Y])/off);
			P[Z] = (int)((h2->R[Z] - p->R[Z])/off);
		} else {
			d0 = 0;
			P[X] = P[Y] = P[Z] = 0;
		}
		Q[X] = (int)((h1->R[X] - p->R[X])/off);
		Q[Y] = (int)((h1->R[Y] - p->R[Y])/off);
		Q[Z] = (int)((h1->R[Z] - p->R[Z])/off);
		dd  = d1 - d0;			/* segment size */
		dd0 = d  - d0;			/* leading part */
		dd1 = d1 - d;			/* trailing part */
		R[X] = muldiv (Q[X], dd0, dd) + muldiv (P[X], dd1, dd);
		R[Y] = muldiv (Q[Y], dd0, dd) + muldiv (P[Y], dd1, dd);
		R[Z] = muldiv (Q[Z], dd0, dd) + muldiv (P[Z], dd1, dd);

		VxMmul (V, R, p->T);
		screen_coords (view, V);
		off = clip_to_screen (D2, V, h->maxx, h->maxy, h->clipx-tx,
			h->clipy-ty, h->shifty);
		if (first && d > BULMIN) {
			D1[X] = muldiv (D2[X], BULMIN, d);
			D1[Y] = muldiv (D2[Y], BULMIN, d);
			D1[X] = h->orgx + D1[X];
			D1[Y] = h->orgy - D1[Y];
		}
		D2[X] = h->orgx + D2[X];
		D2[Y] = h->orgy - D2[Y];
		add_segment (D1[X], D1[Y], D2[X], D2[Y], st.hfg,
			h->orgx, h->orgy,h->clipx, h->clipy, h->shifty);
		if (off)
			break;
		if (ddd == d)
			add_5op (T_ELLIPSE, D2[X], D2[Y], tx, ty, st.hfg);
		else if (!(d%BULSTEP)) {
			add_line (D2[X]-tx, D2[Y], T_MOVE);
			add_line (D2[X]+tx, D2[Y], st.hfg);
		}

		if (d >= BULMAX)
			break;

		if (d < ddd && ddd < n)
			d = ddd;
		else {
			d = n;
			n += BULSEG;
		}
		first = 0;
	}
}
#undef DOGRAPH
#undef BULSEG
#undef BULSTEP
#undef BULMIN
#undef BULMAX
