/* -------------------------------- land.c ---------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Look after the landscape. These objects do not move or get simulated!
*/

#include "fly.h"
#include "colors.h"


extern int FAR	paddoc_init (BODY *b);

static int NEAR	local_read (FILE *vxfile, char *line, int name);

#define RANGE	20

static struct {
	short	name;
	short	color;
	LVECT	R;
	AVECT	a;
} Tlnd;

#define READFLD		Tlnd

static struct FldTab FAR LndTbl[] = {
	READI (name),
	READI (color),
	READI (R[X]),
	READI (R[Y]),
	READI (R[Z]),
	READI (a[X]),
	READI (a[Y]),
	READI (a[Z]),
{0, 0}};

#undef READFLD

static int NEAR
land_read (void)
{
	int	i, n;
	int	nlocals;
	OBJECT	*p;
	FILE	*ifile;
	char	line[256];

	if (!st.lndname)
		st.lndname = xstrdup ("fly");

	Sys->BuildFileName (st.filename, st.fdir, st.lndname, LND_EXT);
	if (!(ifile = fopen (st.filename, RTMODE))) {
		LogPrintf ("%s: missing land file\n", st.filename);
		return (-1);
	}
	LogPrintf ("Land     %s\n", st.filename);

	st.lineno = 0;
	nlocals = 0;

	for (n = 0;; ++n) {
		if (field_read (ifile, &LndTbl[0], line) < 0)
			break;
		if (O_EOF == Tlnd.name)
			break;
		else if (Tlnd.name & O_DEFINE) {
			Tlnd.name = O_LOCAL + (Tlnd.name & ~O_DEFINE);
			if (Tlnd.name < O_LOCAL || Tlnd.name >= O_INT+O_EXT) {
				LogPrintf ("%s %ld: bad object name\n",
					st.filename, st.lineno);
				goto badret;
			}
			if (local_read (ifile, line, Tlnd.name))
				goto badret;
			++nlocals;
			continue;
		}
		if (Tlnd.name < 0 || Tlnd.name >= O_INT+O_EXT ||
		    !st.bodies[Tlnd.name]) {
			LogPrintf ("%s %ld: bad object name\n",
				st.filename, st.lineno);
			goto badret;
		}
		for (i = 1; LndTbl[i].type > 0; ++i) {
			if (field_read (ifile, &LndTbl[i], line) < 0)
				goto badret;
		}
		Tlnd.a[Z] = -Tlnd.a[Z];
		p = create_land (Tlnd.name);
		if (!p) {
			LogPrintf ("%s %ld: create land failed\n",
				st.filename, st.lineno);
			goto badret;
		}
		switch (Tlnd.color) {
		case CC_BLACK:
			p->color = st.black;
			break;
		case CC_WHITE:
			p->color = st.white;
			break;
		case CC_RED:
			p->color = st.red;
			break;
		case CC_BLUE:
			p->color = st.blue;
			break;
		case CC_GREEN:
			p->color = st.green;
			break;
		case CC_MAGENTA:
			p->color = st.magenta;
			break;
		case CC_BROWN:
			p->color = st.brown;
			break;
		case CC_GRAY:
			p->color = st.gray;
			break;
		case CC_LBLUE:
			p->color = st.lblue;
			break;
		case CC_LRED:
			p->color = st.lred;
			break;
		case CC_HUDLOW:
			p->color = st.hudlow;
			break;
		case CC_HUDHIGH:
			p->color = st.hudhigh;
			break;
		case CC_LGRAY:
			p->color = st.lgray;
			break;
		case CC_SKYBLUE:
			p->color = st.skyblue;
			break;
		case CC_GROUND:
			p->color = st.ground;
			break;
		default:
			break;		/* keep original color */
		}
		LVcopy (p->R, Tlnd.R);
		LVcopy (p->a, Tlnd.a);
		Mobj (p);
	}
	LogPrintf ("      created %d objects\n", n);
	LogPrintf ("      defined %d locals\n", nlocals);

	fclose (ifile);
	return (0);
badret:
	fclose (ifile);
	return (-1);
}

extern int FAR
land_init (void)
{
	BODY	*b;

	CL = CLT = 0;

	if (!create_land (O_GROUND)		/* must be first! */
	    || !create_land (O_LOW))		/* must be second! */
		return (1);

	if (land_read () < 0)
		return (1);

	if (!(st.flags&SF_LANDSCAPE))
		return (0);

	b = bodies_new (-1);
	if (!b)
		return (1);
	b->init = paddoc_init;
	if ((*b->init) (b)) {
		bodies_del (b->name);
		return (1);
	}

	st.landx = 0x8000;		/* force landscaping */
	st.landy = 0x8000;

	return (0);
}

extern void FAR
land_term (void)
{
	list_clear (&CL);
}

static int NEAR
land_add (ONAME lname, int xx, int yy)
{
	OBJECT	*p;
	int	cloud;

	Fsrand ((xx^(yy<<4))^yy);
	if (Frand() % 128)
		return (0);

	cloud = (Frand()%10)/5;

	p = create_land (lname);

	if (p) {
		p->flags |= F_LAND;
		p->R[X] = xx * 1000L * VONE;
		p->R[Y] = yy * 1000L * VONE;
		if (cloud) {
			p->R[Z] = (Frand()%2000 + 2000)*(long)VONE;
			p->color = st.white;
		} else
			p->R[Z] = 0;
	} else
		return (0);		/* ignore lost paddoc */
	return (0);
}

extern int FAR
land_update (OBJECT *pov)
{
	long	minx, miny, maxx, maxy;
	int	x, y, xl, xh, yl, yh, xxl, xxh, xx, yy;
	OBJECT	*p, *prev;
	ONAME	lname;

	if (F(p = CL) || O_GROUND != p->name) {
		MsgPrintf (-100, "Missing GROUND object!");
		return (1);
	}
	st.bodies[p->name]->dynamics (p, st.interval);

	if (F(p = p->next) || O_LOW != p->name) {
		MsgPrintf (-100, "Missing LOW object!");
		return (1);
	}
	st.bodies[p->name]->dynamics (p, st.interval);

	if (!(st.flags&SF_LANDSCAPE))
		return (0);

	x = (int)(pov->R[X] / VONE / 1000);		/* get square */
	y = (int)(pov->R[Y] / VONE / 1000);

	if (x == st.landx && y == st.landy)
		return (0);

	minx = (x-RANGE)*1000L*VONE;
	maxx = (x+RANGE)*1000L*VONE;
	miny = (y-RANGE)*1000L*VONE;
	maxy = (y+RANGE)*1000L*VONE;
	for (prev = 0, p = CL; p;) {			/* delete old */
		if ((p->flags & F_LAND) &&
		    (p->R[X] < minx || p->R[X] > maxx || 
		     p->R[Y] < miny || p->R[Y] > maxy))
		    	p = delete_land (p);
		else
			p = p->next;
	}

	if (st.landx < x) {
		xxl = x-RANGE;
		xh  = x+RANGE;
		xxh = st.landx+RANGE;
		xl  = xxh+1;
		if (xl < xxl)
			xl = xxl;
	} else {
		xl  = x-RANGE;
		xxh = x+RANGE;
		xxl = st.landx-RANGE;
		xh  = xxl-1;
		if (xh > xxh)
			xh = xxh;
	}

	if (st.landy < y) {
		yh = y+RANGE;
		yl = st.landy+RANGE+1;
		if (yl < y-RANGE)
			yl = y-RANGE;
	} else {
		yl = y-RANGE;
		yh = st.landy-RANGE-1;
		if (yh > y+RANGE)
			yh = y+RANGE;
	}

	st.landx = x;
	st.landy = y;

	if (-1 == (lname = body_name ("PADDOC")))
		return (1);

	for (xx = xl; xx <= xh; ++xx) {
		for (yy = y-RANGE; yy <= y+RANGE; ++yy) {
			if (land_add (lname, xx, yy))
				return (0);
		}
	}

	for (xx = xxl; xx <= xxh; ++xx) {
		for (yy = yl; yy <= yh; ++yy) {
			if (land_add (lname, xx, yy))
				return (0);
		}
	}

	return (0);
}

extern void FAR
local_term (BODY *b)
{
	if (!b->shape)
		return;
	b->shape->v = xfree (b->shape->v);
	DEL0 (b->shape);
	b->term = 0;
	b->create = 0;
	b->delete = 0;
	b->dynamics = 0;
}

static int FAR
local_create (OBJECT *p)
{
	p->color = st.white;
	p->time = FOREVER;
	p->flags |= F_VISIBLE;
	Mident (p->T);
	return (0);
}

static void FAR
local_dynamics (OBJECT *p, int interval)
{}

static void FAR
local_hit (OBJECT *obj, int speed, int extent, int damaging)
{}

static short NEAR
get_xyz (long l, int fine)
{
	if (fine & V_METERS)
		l /= VONE;
	if (l > VMAX || l < -VMAX)
		LogPrintf ("%s %ld: coordinate truncated\n",
			st.filename, st.lineno);
	return ((short)l);
}

static int NEAR
local_read (FILE *vxfile, char *line, int name)
{
	int	i, n, fine;
	long	l;
	VERTEX	*vx;
	BODY	*b;

	vx = 0;
	b = 0;

	if (st.bodies[Tlnd.name]) {
		LogPrintf ("%s %ld: object already defined\n",
			st.filename, st.lineno);
		goto badret;
	}

	if (field_long (vxfile, line, &l))
		goto badret;
	if (l < 1 || l > 2) {
		LogPrintf ("%s %ld: bad detail level\n",
			st.filename, st.lineno);
		goto badret;
	}
	fine = (int)l;

	if (field_long (vxfile, line, &l))
		goto badret;
	if (l <= 0) {
		LogPrintf ("%s %ld: bad shape size\n",
			st.filename, st.lineno);
		goto badret;
	}
	n = (int)l;

	vx = (VERTEX *)xcalloc (sizeof (*vx), n+1);
	if (!vx) {
		LogPrintf ("%s %ld: no shape memory\n",
			st.filename, st.lineno);
		goto badret;
	}

	for (i = 0; i < n; ++i) {
		if (field_long (vxfile, line, &l))
			goto badret;
		vx[i].V[X] = get_xyz (l, fine);
		if (field_long (vxfile, line, &l))
			goto badret;
		vx[i].V[Y] = get_xyz (l, fine);
		if (field_long (vxfile, line, &l))
			goto badret;
		vx[i].V[Z] = get_xyz (l, fine);
		if (field_long (vxfile, line, &l))
			goto badret;
		vx[i].flags = (short)l;
	}

	vx[i].flags = V_EOF;	/* end of list */

	if (F(b = bodies_new (Tlnd.name))) {
		LogPrintf ("%s %ld: no shape 'body'\n",
			st.filename, st.lineno);
		goto badret;
	}

	if (!NEW (b->shape)) {
		LogPrintf ("%s %ld: no shape 'shape'\n",
			st.filename, st.lineno);
		goto badret;
	}

	b->shape->v = vx;
	if (fine & V_FINE)
		b->shape->flags |= SH_FINE;

	b->title = "LOCAL";
	b->shape->weight = 1L;
	b->shape->drag = 0;

	b->init = body_init;
	b->term = local_term;
	b->create = local_create;
	b->delete = body_delete;
	b->dynamics = local_dynamics;
	b->hit = local_hit;

	bodies_extent (b->name);

	LogPrintf ("      local %d size %d\n", Tlnd.name-O_LOCAL, n);

	return (0);
badret:
	if (b)
		bodies_del (b->name);
	if (vx)
		shape_free (vx);
	return (-1);
}
#undef RANGE
