/* --------------------------------- max.c ---------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Read a text macro definition. Directly replaces mac_read() of macros.c.
*/

#include "fly.h"
#include "keyname.h"


static Ulong lineno = 1L, column = 0L, errs = 0L;
static char tabCTRL[] = "@abcdefghijklmnopqrstuvwxyz[\\]^_";
static Ushort buf_str[1024] = {0};


static Ulong NEAR
keyval (Ulong c)
{
	if (c < 32)
		return (tabCTRL[c]| K_CTRL);
	return (c);
}

static Ulong NEAR
eatline (FILE *f)
{
	int	c;

	do {
		c = fgetc (f);
	} while (EOF != c && '\n' != c);
	++lineno;
	column = 0;
	return (EOF == c ? VK_EOF : (Ulong)c);
}

static int NEAR
getstring (FILE *f)
{
	int	c, n, escape;

	for (n = 0, escape = 0;;) {
		c = fgetc (f);
		if (EOF == c) {
			++errs;
			MsgEPrintf (-50, "max: EOF inside string at %ld:%ld",
				lineno, column);
			return (-1);
		}
		if (escape) {
			switch (c) {
			default:
				goto raw;
			case 'a':
				c = 'g';
				break;
			case 'b':
				c = 'h';
				break;
			case 'e':
				c = '[';
				break;
			case 'f':
				c = 'l';
				break;
			case 'n':
				c = 'm';
				break;
			case 'r':
				c = 'j';
				break;
			case 't':
				c = 'i';
				break;
			case 'v':
				c = 'k';
				break;
			case '\n':
				continue;	/* traditional line break */
			}
			c |= K_CTRL;
raw:			escape = 0;
		} else if ('\\' == c) {
			escape = 1;
			continue;
		} else if ('"' == c)
			return (n);
		else if ('\n' == c)		/* allow line split */
			continue;
		if (n >= rangeof (buf_str)) {
			++errs;
			MsgEPrintf (-50, "max: string too long at %ld:%ld",
				lineno, column);
			continue;
		}
		buf_str[n++] = (Ushort)c;
	}
	/* never reached */
}

static Ulong NEAR
getterm (FILE *f)
{
	char		keyname[32+1], *p;
	int		n, i, c;
	Ulong		l;

retry:
	for (n = 0;;) {
		c = fgetc (f);
		++column;
		if (EOF == c) {
			if (0 == n)
				return (VK_EOF);
			break;
		} else if ('\n' == c) {
			++lineno;
			column = 0;
			if (n)
				break;
		} else if (isspace (c)) {
			if (n)
				break;
		} else if (n < sizeof (keyname) - 1) {	/* truncate */
			if (1 == n && '#' == keyname[0]) {
				if ('#' == c)
					return (VK_COM);
				if ('"' == c)
					return (VK_STR);
			}
			keyname[n++] = (char)c;
		}
	}

/* Look word up
*/
	keyname[n] = '\0';
	for (i = 0; k_name[i].name; ++i) {
		if (!stricmp (k_name[i].name, keyname))
			return (k_name[i].value);
	}

	if (1 == n && isprint (keyname[0]))
		return ((Uchar)keyname[0]);

	if (n > 1 && '\\' == keyname[0]) {
		if ('0' == keyname[1]) {
			if (2 == n)
				return (keyval (0));
			l = strtol (keyname+2, &p, 8);
			if (l >= 256L) {
				MsgEPrintf (-50, "\\0%lo too big at %ld:%ld\n",
					l, lineno, column);
				l &= 0x00ffL;
			}
			if ('\0' == *p)
				return (keyval (l));
		} else if ('x' == keyname[1]) {
			if (n > 2) {
				l = strtol (keyname+2, &p, 16);
				if (l >= 256L) {
					MsgEPrintf (-50,
						"\\x%lx too big at %ld:%ld\n",
						l, lineno, column);
					l &= 0x00ffL;
				}
				if ('\0' == *p)
					return (keyval (l));
			}
		} else {
			l = strtol (keyname+1, &p, 10);
			if (l >= 256L) {
				MsgEPrintf (-50, "\\%lu too big at %ld:%ld\n",
					l, lineno, column);
				l &= 0x00ffL;
			}
			if ('\0' == *p)
				return (keyval (l));
		}
	}

	++errs;
	MsgEPrintf (-50, "max: bad token '%s' at %ld:%ld", keyname,
		lineno, column);
	if (EOF == c)
		return (VK_EOF);
	goto retry;
}

static Ulong NEAR
getkey (FILE *f)
{
	Ulong	c, key;

	for (key = 0L;;) {
		c = getterm (f);
		if (VK_EOF == c) {
			if (key) {
				++errs;
				MsgEPrintf (-50,
"max: unexpected end of file at %ld:%ld",
					lineno, column);
			}
			return (VK_EOF);
		} else if (VK_DEF == c) {
			if (key) {
				++errs;
				MsgEPrintf (-50,
"max: bad key preceding 'def' at %ld:%ld",
					lineno, column);
			}
			key = VK_DEF;
		} else if (VK_COM == c) {
			if (key) {
				++errs;
				MsgEPrintf (-50,
"max: bad key preceding '##' %ld:%ld",
					lineno, column);
			}
			key = 0L;
			c = eatline (f);
			if (VK_EOF == c)
				return (VK_EOF);
		} else if (VK_STR == c) {
			if (key) {
				++errs;
				MsgEPrintf (-50,
"max: bad key preceding '#\"' %ld:%ld",
					lineno, column);
			}
			return (VK_STR);
		} else {
			key |= c;
			if (c & K_RAW)
				return (key);
		}
	}
	/* never reached */
}

#define MAXMACLEN	1024

extern int FAR
max_read (MACRO *Macros)
{
	FILE	*max;
	int	i, n, nMacros;
	Ulong	c;
	MACRO	*m;
	Ushort	*macbody;

	macbody = (Ushort *)xcalloc (MAXMACLEN, sizeof (Ushort));

	Sys->BuildFileName (st.filename, st.fdir, st.mname, MAX_EXT);
	max = fopen (st.filename, RTMODE);
	if (!max) {
		MsgEPrintf (-50, "max: open '%s' failed",
			st.filename);
		return (-1);
	}
	LogPrintf ("Max      %s\n", st.filename);

	for (m = Macros, nMacros = 0;;) {
		c = getkey (max);
		if (VK_EOF == c)
			break;
		if (!(VK_DEF & c)) {
			++errs;
			MsgEPrintf (-50, "max: 'def' not found at %ld:%ld",
				lineno, column);
			continue;
		}
def:
		if (nMacros > st.nMacros) {
			++errs;
			MsgEPrintf (-50, "max: too many macros at %ld:%ld",
				lineno, column);
			break;
		}
		m->name = (Ushort)c;
		for (n = 0;;) {
			c = getkey (max);
			if (VK_EOF == c || (VK_DEF & c))
				break;
			if (VK_STR == c) {
				i = getstring (max);
				if (i <= 0)
					continue;
				if (n+i > MAXMACLEN) {
					++errs;
					MsgEPrintf (-50,
"max: macro too long at %ld:%ld",
						lineno, column);
				} else {
					memcpy (macbody+n, buf_str,
						i * sizeof (Ushort));
					n += i;
				}
				continue;
			}
			if (n >= MAXMACLEN) {
				++errs;
				MsgEPrintf (-50,
					"max: macro too long at %ld:%ld",
					lineno, column);
				continue;
			}
			macbody[n++] = (Ushort)c;
		}
		if (0 == n) {
			++errs;
			MsgEPrintf (-50, "max: empty macro at %ld:%ld",
				lineno, column);
		}
		m->len = (Ushort)n;
		m->def = (Ushort *)xcalloc (m->len, sizeof (Ushort));
		if (!m->def) {
			++errs;
			MsgEPrintf (-50, "max: no mem");
			m->name = KEYUNUSED;
			goto ret;
		}
		memcpy (m->def, macbody, m->len * sizeof (Ushort));
		++m;
		++nMacros;
		if (VK_DEF & c)
			goto def;
		if (VK_EOF == c)
			break;
	}
ret:
	fclose (max);
	xfree (macbody);
	return (0 != errs);
}

#undef MAXMACLEN

