/* --------------------------------- grfast.c ------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Vga graphics driver, uses low level graphics primitives, 256 colors only.
*/

#include "fly.h"
#include "colors.h"
#include "gr.h"

#include <conio.h>


static int	width = 0, height = 0;	/* used for debug */

static void FAR
SetActive (int page)
{
	GrSetActive (page * (long)GrSizeY() * (long)GrSizeX());
}

#define DOSYNC		0x0001
#define MAX_SYNC_WAIT	1000L	/* 1 second is long enough */

static void FAR
SetVisual (int page)		/* done */
{
	Ulong	lasttime;

	if (CS->device->flags & DOSYNC) {
		lasttime = st.lasttime;
		while (inp (0x3da) & 0x01) {	/* wait for Display Enabled */
			sys_poll ();
			if (st.lasttime - lasttime > MAX_SYNC_WAIT) {
				LogPrintf ("grfast: sync timed out\n");
				die ();
			}
		}
	}
	GrSetVisual ((long)page * (long)GrSizeX() * (long)GrSizeY());
	if (CS->device->flags & DOSYNC) {
		while (inp (0x3da) & 0x08) {	/* wait for Vert Sync*/
			sys_poll ();
			if (st.lasttime - lasttime > MAX_SYNC_WAIT) {
				LogPrintf ("grfast: sync timed out\n");
				die ();
			}
		}
		while (!(inp (0x3da) & 0x08)) {	/* wait for Vert Sync end */
			sys_poll ();
			if (st.lasttime - lasttime > MAX_SYNC_WAIT) {
				LogPrintf ("grfast: sync timed out\n");
				die ();
			}
		}
	}
}

#undef MAX_SYNC_WAIT

static int FAR
CursorMode (int mode)
{
	GrCursorMode (mode);
	return (0);
}

static void FAR
TextColor (Uint fg, Uint bg)			/* done */
{
	GrTextColor (fg, bg);
}

static void FAR
FontSet (DEVICE *dev, char *fname)		/* done */
{
	GrFontSet (fname);
	dev->FontWidth  = GrTextCol (1);
	dev->FontHeight = GrTextRow (1);
}

static void FAR
TextChar (int c)				/* done */
{
	if (st.flags1 & SF_DBUFFERING) {
		GrTextPush ();
		SetActive (st.which_buffer);
		GrTextChar (c);
		GrTextPop ();
		SetActive (1-st.which_buffer);
	}
	GrTextChar (c);
}

static void FAR
TextPut (int c)					/* done */
{
	if (st.flags1 & SF_DBUFFERING) {
		GrTextPush ();
		SetActive (st.which_buffer);
		GrTextPut (c);
		GrTextPop ();
		SetActive (1-st.which_buffer);
	}
	GrTextPut (c);
}

static void FAR
SetTextPos (int row, int col)			/* done */
{
	GrTextGoto (row, col);
}

static void FAR
PushTextPos (void)
{
	GrTextPush ();
}

static void FAR
PopTextPos (void)
{
	GrTextPop ();
}

static void FAR
TextClear (void)
{
	if (st.flags1 & SF_DBUFFERING) {
		SetActive (st.which_buffer);
		GrTextClear ();
		SetActive (1-st.which_buffer);
	}
	GrTextClear ();
	GrTextGoto (1, 1);
}

static void FAR
WriteMode (int mode)
{
	switch (mode) {
	default:
	case T_MSET:
		mode = 0;
		break;
	case T_MOR:
		mode = GrOR;
		break;
	case T_MXOR:
		mode = GrXOR;
		break;
	}
	GrSetWriteMode (mode);
}

static void FAR
SetPalette (int n, long c)
{
	int	r, g, b;

	r = 0x0ff & (int)(c      );
	g = 0x0ff & (int)(c >>  8); 
	b = 0x0ff & (int)(c >> 16);
	GrSetColor (n, r, g, b);
}

static void FAR
OpenTextWindow (SCREEN *scr)
{
	if (st.flags1 & SF_DBUFFERING) {
		GrTextPush ();
		SetActive (st.which_buffer);
		GrTextWin (scr->minx, scr->minx+scr->sizex, scr->miny,
			scr->miny+scr->sizey, scr->BoColor, scr->BgColor);
		GrTextPop ();
		SetActive (1-st.which_buffer);
	}
	GrTextWin (scr->minx, scr->minx+scr->sizex, scr->miny,
		scr->miny+scr->sizey, scr->BoColor, scr->BgColor);
	TextColor (scr->FgColor, scr->BgColor);
	GrScrollOn ();
}

static void FAR
CloseTextWindow (SCREEN *scr)
{}

struct chip {
	char	*name;
	int	type;
} chips[] = {
	{"VGA",		GRT_NONE},
	{"VESA",	GRT_VESA},
	{"ATI",		GRT_ATI},
	{"AHEADA",	GRT_AHEADA},
	{"AHEADB",	GRT_AHEADB},
	{"CHIPS",	GRT_CHIPS},
	{"EVEREX",	GRT_EVEREX},
	{"GENOA",	GRT_GENOA},
	{"OAK",		GRT_OAK},
	{"PARADISE",	GRT_PARADISE},
	{"T88",		GRT_TRIDENT88},
	{"T89",		GRT_TRIDENT89},
	{"VIDEO7",	GRT_VIDEO7},
	{"ET3K",	GRT_ET3000},
	{"ET4K",	GRT_ET4000},
	{"NCR",		GRT_NCR},
	{"S3",		GRT_S3},
	{"ACUMOS",	GRT_ACUMOS},
	{"AL2101",	GRT_AL2101},
	{"MXIC",	GRT_MXIC},
	{"P2000",	GRT_P2000},
	{"RT3106",	GRT_RT3106},
{0,0}};

static int FAR
get_options (char *options)
{
	char	*p;
	struct chip	*c;

	if (F(p = get_iarg (options, 1)))
		return (1);
	for (c = chips;; ++c) {
		if (!c->name) {
			LogPrintf ("Bad video type %s\n", p);
			return (1);
		}
		if (!strnicmp (p, c->name, sizeof (c->name))) {
			Gr->flags |= c->type;
			break;
		}
	}

	return (0);
}

static int FAR
Init (DEVICE *dev, char *options)
{
	int	i;

	if (get_options (options))
		return (1);

	if (dev->sizex == 0 || dev->sizey == 0) {
		LogPrintf ("Bad WxH in .vmd file\n");
		return (1);
	}
	width = dev->sizex-1;
	height = dev->sizey-1;

	GrSetType (Gr->flags,
		(int)(dev->npages*(long)dev->sizex*(long)dev->sizey/1024L),
		width);

#if 0
	GrSetMode (GR_width_height_graphics, dev->sizex, dev->sizey);
#else
	if (!dev->mode) {
		LogPrintf ("Must have video mode in .vmd file\n");
		return (1);
	}
	GrSetXY (dev->sizex, dev->sizey);
	GrSetBiosMode (dev->mode);
#endif

#if 0
	dev->npages = (int)((1024L*1024L)/(dev->sizex*(long)dev->sizey));
#endif
	while (GrAllocCell () >= 0)		/* get 2-255 */
		;

	st.black   = 0;	/* cannot change */
	st.white   = 1;	/* cannot change */
	st.red     = 2;	/* do not change! */
	st.blue    = 4;	/* do not change! */
	st.magenta = 6;	/* do not change! */
	st.green   = 3;
	st.brown   = 5;
	st.gray    = 7;
	st.hudlow  = 8;
	st.hudhigh = 9;
	st.skyblue = 10;
	st.lred    = 11;
	st.lgray   = 12;
	st.ground  = 13;
	st.lblue   = 14;
/*	st.white   = 15;	keep 15 for OR'ed white */

	SetPalette (st.black,   C_BLACK);
	SetPalette (st.white,   C_WHITE);
	SetPalette (st.red,     C_RED);
	SetPalette (st.blue,    C_BLUE);
	SetPalette (st.magenta, C_MAGENTA);
	SetPalette (st.green,   C_GREEN);
	SetPalette (st.brown,   C_BROWN);
	SetPalette (st.gray,    C_GRAY);
	SetPalette (st.hudlow,  C_LYELLOW);
	SetPalette (st.hudhigh, C_YELLOW);
	SetPalette (st.lred,    C_LIGHTRED);
	SetPalette (st.skyblue, C_SKYBLUE);
	SetPalette (st.lgray,   C_LIGHTGRAY);
	SetPalette (st.ground,  C_GRAY);
	SetPalette (st.lblue,   C_LIGHTBLUE);

	for (i = 0; i < dev->npages; ++i) {
		SetActive (i);
		GrClear (0, 0, dev->sizex, dev->sizey, st.black);
	}

	SetVisual (0);
	SetActive (0);

	if (st.fname == 0)
		st.fname = strdup ("6x8.fnt");
	GrFontSet (st.fname);
	dev->FontWidth  = GrTextCol (1);
	dev->FontHeight = GrTextRow (1);

	return (0);
}

static void FAR
Term (DEVICE *dev)
{
#if 0
	GrSetMode (GR_default_text, 0, 0);
#else
	GrSetBiosMode (0x03);		/* text 80x25 */
#endif
}

#ifdef LOCALELLIPSE
static void FAR
Ellipse (register int x1, register int y1, int rx, int ry, register Uint color)
{
	int	ax, bx, cx, dx, ay, by, cy, dy;

	ax = fmul ( 3196, rx);		/* sin (pi/16) */
	ay = fmul ( 3196, ry);
	bx = fmul ( 9102, rx);		/* sin (3*pi/16) */
	by = fmul ( 9102, ry);
	cx = fmul (13623, rx);		/* sin (5*pi/16) */
	cy = fmul (13623, ry);
	dx = fmul (16069, rx);		/* sin (7*pi/16) */
	dy = fmul (16069, ry);

	GrMoveTo (x1+dx, y1-ay);
	GrDrawTo (x1+cx, y1-by, color);
	GrDrawTo (x1+bx, y1-cy, color);
	GrDrawTo (x1+ax, y1-dy, color);
	GrDrawTo (x1-ax, y1-dy, color);
	GrDrawTo (x1-bx, y1-cy, color);
	GrDrawTo (x1-cx, y1-by, color);
	GrDrawTo (x1-dx, y1-ay, color);
	GrDrawTo (x1-dx, y1+ay, color);
	GrDrawTo (x1-cx, y1+by, color);
	GrDrawTo (x1-bx, y1+cy, color);
	GrDrawTo (x1-ax, y1+dy, color);
	GrDrawTo (x1+ax, y1+dy, color);
	GrDrawTo (x1+bx, y1+cy, color);
	GrDrawTo (x1+cx, y1+by, color);
	GrDrawTo (x1+dx, y1+ay, color);
	GrDrawTo (x1+dx, y1-ay, color);
}
#endif

static void FAR
Flush (void)
{}

#if 0
static void FAR
Clear (SCREEN *scr)
{
	GrClear (scr->minx, scr->miny, scr->sizex, scr->sizey, scr->BgColor);
}
#endif

static void FAR
Shutters (int eye)
{
	if (st.misc[7]) {
		if (eye >= 0)
			outp (st.misc[7]+4, 1+2*eye);
		else if (-1 == eye)
			outp (st.misc[7]+4, 1);		/* on */
		else if (-2 == eye)
			outp (st.misc[7]+4, 0);		/* off */
	}
}

#undef LOCALDRAW

#ifdef LOCALDRAW
static void FAR
Move (int x1, int y1)
{
	if (x1 < CS->minx || x1 > CS->minx+CS->sizex-1 ||
	    y1 < CS->miny || y1 > CS->miny+CS->sizey-1) {
	    	LogPrintf ("Move(%d,%d)\n", x1, y1);
	    	die ();
	}
	GrMoveTo (x1, y1);
}

static void FAR
Draw (int x1, int y1, Uint c)
{
	if (x1 < CS->minx || x1 > CS->minx+CS->sizex-1 ||
	    y1 < CS->miny || y1 > CS->miny+CS->sizey-1) {
	    	LogPrintf ("Draw(%d,%d)\n", x1, y1);
	    	die ();
	}
	GrDrawTo (x1, y1, c);
}
#endif

struct GrDriver NEAR GrFast = {
	"GrFast",
	0,
	0,
	Init,
	Term,
	OpenTextWindow,
	CloseTextWindow,
	FontSet,
	TextPut,
	TextChar,
	TextColor,
	CursorMode,
#ifdef LOCALDRAW
	Move,
	Draw,
#else
	GrMoveTo,
	GrDrawTo,
#endif
	SetVisual,
	SetActive,
	0,	/* Clear() too slow */
	SetTextPos,
	PushTextPos,
	PopTextPos,
	TextClear,
	WriteMode,
	SetPalette,
#ifdef LOCALELLIPSE
	Ellipse,
#else
	GrEllipse,
#endif
	Flush,
	Shutters
};
#undef DOSYNC
#undef LOCALDRAW
#undef LOCALELLIPSE
