/* --------------------------------- speaker.c ------------------------------ */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Sound device driver for the PC speaker.
*/

#include "fly.h"
#include "pc8254.h"

#include <conio.h>


#define	QSIZE	32

struct beep {
	int	freq;
	int	milli;
	int	*list;
	int	id;
};

typedef struct beep BEEP;

static BEEP	*beeps = 0;
static Ulong	lasttime = 0L;
static Ulong	nexttime = 0L;
static int	playing = 0, nbeeps = 0;


static void NEAR
sound_start (int n)
{
	int	i;
	Ulong	flags;

	flags = Sys->Disable ();

	i = inp (PORT_B);	/* get 8255 port B */
	i |= 3;			/* turn on speaker */
	outp (PORT_B, i);	/* set 8255 port B */

	if (n < (int)(XTAL/0x0ffffL))
		n = 0xffff;
	else
		n = (int)(XTAL / n);

	outp (COMMAND_REG, WRITE_CH2);
	outp (CHANNEL_2, n & 0x00ff);		/* LSB */
	outp (CHANNEL_2, (n >> 8) & 0x00ff);	/* MSB */

	Sys->Enable (flags);
}

static void NEAR
sound_stop (void)
{
	int	i;
	Ulong	flags;

	flags = Sys->Disable ();

	i = inp (PORT_B);	/* get 8255 port B */
	i &= ~3;		/* turn off speaker */
	outp (PORT_B, i);	/* set 8255 port B */
	Sys->Enable (flags);
}

static int NEAR
sound_command (BEEP *b)
{
	int	l, ret;

again:
	for (ret = 1; ret && (l = b->list[0]) < 0;) {
		switch (l) {
		default:
		case -1:			/* END */
			ret = 0;
			break;
		case -2:			/* GOTO,address */
			b->list += 2*b->list[1];
			break;
		case -3:			/* REPEAT,address,n,cnt */
			if (b->list[3]) {
				if (--b->list[3])	/* Nth time */
					b->list += 2*b->list[1];
				else			/* done */
					b->list += 2*2;
			} else {			/* start */
				b->list[3] = b->list[2];
				b->list += 2*b->list[1];
			}
			break;
		}
	}
	if (ret) {
		b->freq = *b->list++;
		b->milli = *b->list++/4;
		if (!b->milli)
			goto again;
	}

	return (ret);
}

static void FAR
SndPoll (int force)
{
	int		i, diff, highest, next;
	BEEP		*b;

	if (!beeps)
		return;

	if (!nbeeps) {
		lasttime = st.lasttime;
		return;
	}
	if (!force && nexttime > st.lasttime)
		return;

	diff = (int)(st.lasttime - lasttime);
	lasttime = st.lasttime;
	
	highest = 0;
	next = 0x7fff;
	for (i = 0; i < QSIZE; ++i) {
		b = beeps+i;
		if (!b->milli)
			continue;
		if (b->milli <= diff) {
			if (b->list && sound_command (b))
				goto use_it;
			b->milli = 0;
			--nbeeps;
			continue;
		}
		b->milli -= diff;
use_it:
		if (b->freq > highest)
			highest = b->freq;
		if (next > b->milli)
			next = b->milli;
	}
	if (highest) {
		if (highest != playing) {
			if (playing)
				sound_stop ();
			playing = highest;
			sound_start (playing);
		}
	} else if (playing) {
		sound_stop ();
		playing = 0;
	}
	nexttime = st.lasttime + next;
}

static int NEAR
AllocBeep (int f, int milli, int *list, int id)
{
	int	i;
	BEEP	*b;

	if (!beeps)
		return (-1);

	if (f <= 0)
		f = 440;

	SndPoll (0);
	for (i = QSIZE; --i >= 0;) {
		b = beeps+i;
		if (b->milli)
			continue;
		b->freq = f;
		b->milli = milli;
		b->list = list;
		b->id = id;
		++nbeeps;
		break;
	}
	if (i >= 0)
		SndPoll (1);
	return (i);
}

static int FAR
SndBeep (int f, int milli)
{
	if (!beeps)
		return (-1);
	return (AllocBeep (f, milli, 0, -1));
}

static int FAR
SndList (int *list, int id)
{
	if (!beeps)
		return (-1);
	if (!list || *list < 0 || !(list[1]/4))
		return (-1);
	return (AllocBeep (list[0], list[1]/4, list+2, id));
}

static int FAR
SndEffect (int eff)
{return (0);}

static int FAR
SndInit (char *options)
{
	if (beeps)
		beeps = xfree (beeps);
	beeps = (BEEP *)xcalloc (QSIZE, sizeof (*beeps));
	if (!beeps)
		return (-1);
	memset (beeps, 0, sizeof (beeps));
	playing = 0;
	nbeeps = 0;
	lasttime = nexttime = st.lasttime;

	return (0);
}

static void FAR
SndTerm (void)
{
	beeps = xfree (beeps);
	sound_stop ();
};

#undef QSIZE

struct SndDriver NEAR SndSpeaker = {
	"SPEAKER",
	0,
	SndInit,
	SndTerm,
	SndPoll,
	SndBeep,
	SndEffect,
	SndList
};
