/* --------------------------------- stick.c -------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Handler for the PC joystick as a pointing device. On port A It will
 * read the analog hat if it is present and will read all four buttons.
 * Use of options:
 *
 *  0	analog 1 direction		p,n
 *  1	what analog 1 controls		0,1
 *  2	analog 2 direction		p,n
 *  3	what analog 2 controls		0,1
 *  4	number of times to read		0-36
 *  5	delay between reads		0-36
 *  6	analog 1 base			0-10
 *  7	analog 2 base			0-10
 *  8	analog 3 base			0-10
 *  9	analog 4 base			0-10 (hat, throttle etc.)
*/

#include "fly.h"

#include <dos.h>
#include <conio.h>


#define HAT		0x0001		/* only on A */
#define THROTTLE	0x0002		/* only on A */
#define THTDOWN		0x0004		/* only on A */
#define USETIMER	0x0008
#define FOURBUTTONS	0x0010		/* only on A */
#define READA3		0x0020		/* only on A */
#define USELOG		0x0040		/* log scale on x/y (default) */

#define PO		p->opt
#define FA1D		PO[0]
#define FA1F		PO[1]
#define FA2D		PO[2]
#define FA2F		PO[3]
#define FNREAD		PO[4]
#define FDELAY		PO[5]
#define FA1BASE		PO[6]
#define FA2BASE		PO[7]
#define FA3BASE		PO[8]
#define FA4BASE		PO[9]
#define FA1IDLE		PO[10]
#define FA2IDLE		PO[11]
#define FA3DETENT	PO[12]		/* stick actual AB detent position */
#define FOPTS		PO[13]

#define	REF		100		/* expected full range */
#define	REFDETENT	75		/* Fly8 AB detent position */

struct stick {
	Ushort	a[4];
	Ushort	b[4];
};
typedef struct stick	STICK;

#define JS_PORT		0x201
#define JS_TIMEOUT	32000
#define JS_READ		inp (JS_PORT)
#define JS_READING	(mode ? Tm->Hires () : JS_TIMEOUT-i)

static int NEAR
js_input (STICK *s, int mode, Ushort mask, Ushort nread, Ushort delay)
{
	register int	i;
	register Uint	m;
	unsigned int	t, x1, y1, x2, y2, minx1, miny1, minx2, miny2, ntimes;
	int		js, tt;

	minx1 = miny1 = minx2 = miny2 = 0xffffU;
	memset (s->a, 0, sizeof (s->a));

	for (ntimes = 0;;) {
		i = JS_TIMEOUT;
		t = JS_READING;
		x1 = y1 = x2 = y2 = t;
		outp (JS_PORT, 0);		/* set trigger */
		for (m = mask; m;) {
			while (!(~JS_READ & m) && --i)
				;
			if (!i)
				break;
			tt = JS_READING;
			js = ~JS_READ & m;
			if (js & 0x01) {
				x1 = tt;
				m &= ~0x01;
			}
			if (js & 0x02) {
				y1 = tt;
				m &= ~0x02;
			}
			if (js & 0x04) {
				x2 = tt;
				m &= ~0x04;
			}
			if (js & 0x08) {
				y2 = tt;
				m &= ~0x08;
			}
		}
		if (minx1 > (x1 -= t))
			minx1 = x1;
		if (miny1 > (y1 -= t))
			miny1 = y1;
		if (minx2 > (x2 -= t))
			minx2 = x2;
		if (miny2 > (y2 -= t))
			miny2 = y2;

		if (++ntimes >= nread)	/* read more? */
			break;

		if (T(i = delay)) {		/* delay? */
			tt = 1234;
			for (i *= 10; i-- > 0;)
				tt *= 19;
		}
	}

	js = m | ~mask;
	s->a[0] = (js & 0x01) ? 0 : minx1;	/* analog 1 */
	s->a[1] = (js & 0x02) ? 0 : miny1;	/* analog 2 */
	s->a[2] = (js & 0x04) ? 0 : minx2;	/* analog 3 */
	s->a[3] = (js & 0x08) ? 0 : miny2;	/* analog 4 */

	js = ~JS_READ;
	s->b[0] = !!(js & 0x10);		/* button 1 */
	s->b[1] = !!(js & 0x20);		/* button 2 */
	s->b[2] = !!(js & 0x40);		/* button 3 */
	s->b[3] = !!(js & 0x80);		/* button 4 */

	return (m);
}
#undef JS_PORT
#undef JS_TIMEOUT
#undef JS_READ
#undef JS_READING

/* Interpret the joystick reading as symm_etric -100...+100.
 * The 'base' parameter is interpreted as a fraction of the center reading.
*/
static void NEAR
symm_input (POINTER *p, int channel, int reading, int sign, int base, int idle,
	int transfer)
{
	int	center;

	center = p->c[channel];
	reading -= center;				/* center */
	reading *= sign;				/* orientation */
	center -= muldiv (center, base, 100);		/* clip range */
	if (reading > center)				/* clip + */
		reading = REF;
	else if (reading < -center)			/* clip - */
		reading = -REF;
	else {
		reading = muldiv (REF+idle, reading, center);
		if (reading > idle)
			reading -= idle;		/* active + */
		else if (reading > -idle)
			reading = 0;			/* idle center */
		else
			reading += idle;		/* active - */
	}
	p->a[channel] = reading;
	if (transfer)
		p->l[channel] = (FOPTS & USELOG)
				? lin2log (reading) : reading;
}

/* Interpret the joystick reading as 'hat' position.
 * The 'base' parameter is interpreted as a fraction of the center reading.
 * the reading is spread evenly from 0 to 100 and has the following states
 * in order: 0=up, 25=right, 50=down, 75=left, 100=center.
*/
static void NEAR
hat_input (POINTER *p, int reading, int base, char *btn)
{
	int	center;

	center = p->c[4];

	if (reading >= center)
		reading = REF;
	else
		reading = muldiv (REF+base, reading, center) - base;

	btn[4 + (reading + REF/8)/(REF/4)] = 1;
}

/* Interpret the joystick reading as a 0-100.
 * The 'base' parameter is interpreted as a fraction of the center reading.
*/
static void NEAR
pos_input (POINTER *p, int channel, int reading, int sign, int base,
	int transfer)
{
	int	top, bot;

	top = p->c[channel];
	if (reading >= top)
		reading = REF;
	else {
		bot = muldiv (top, base, 100);
		top -= bot;
		reading -= bot;
		if (reading <= 0 || top <= 0)
			reading = 0;
		else
			reading = muldiv (REF, reading, top);
	}
	if (sign)
		reading = REF - reading;
	p->a[channel] = reading;
	if (transfer)
		p->l[channel] = reading;
}

static int FAR
jsa_read (POINTER *p, int transfer)
{
	STICK	s[1];
	int	i, n;
	char	btn[10];

	if (js_input (s, FOPTS & USETIMER, (FOPTS & READA3) ? 0x0b : 0x03,
			FNREAD, FDELAY))
		return (1);			/* reading failed */

	memset (btn, 0, sizeof (btn));

	symm_input (p, FA1F, s->a[0], -FA1D, FA1BASE, FA1IDLE, transfer);
	symm_input (p, FA2F, s->a[1],  FA2D, FA2BASE, FA2IDLE, transfer);
	if (FOPTS & READA3) {
		if (FOPTS & HAT)
			hat_input (p, s->a[3], FA4BASE, btn);
		else if (FOPTS & THROTTLE) {
			pos_input (p, 3, s->a[3], FOPTS & THTDOWN,
				FA4BASE, transfer);
			if (!transfer)
				;
			else if (p->l[3] > FA3DETENT)
				p->l[3] = REFDETENT + muldiv (p->l[3]-FA3DETENT,
						REF-REFDETENT, REF-FA3DETENT);
			else
				p->l[3] = muldiv (p->l[3],
						REFDETENT, FA3DETENT);
		}
	}

	n = (FOPTS & FOURBUTTONS) ? 4 : 2;

	for (i = 0; i < n; ++i)
		btn[i] = s->b[i];

	do_btns (p, btn, rangeof (btn));

	return (0);
}

/* Calibrate joy-stick. Paddle and 'hat' must be at center!
*/
static int FAR
jsa_cal (POINTER *p)
{
	STICK	s[1];
	int	m, px, py;

	m = 0x03;
	if (FOPTS & (HAT|THROTTLE)) {
		FOPTS |= READA3;
		m |= 0x08;
	}
	m = js_input (s, FOPTS & USETIMER, m, 10, 10);

	if (m & 0x03 || !s->a[0] || !s->a[1])
		return (1);

	px = FA1F;
	py = FA2F;

	p->c[px] = s->a[0];
	p->a[px] = p->l[px] = 0;

	p->c[py] = s->a[1];
	p->a[py] = p->l[py] = 0;

	if (FOPTS & READA3) {
		if (m || 0 == s->a[3])
			FOPTS &= ~READA3;	/* failed */
		else if (FOPTS & HAT) {
			MsgWPrintf (50, "analog hat");
			p->c[4] = p->c[5] = s->a[3];
			p->a[4] = p->l[4] = 0;
			p->a[5] = p->l[5] = 0;
		} else if (FOPTS & THROTTLE) {
			MsgWPrintf (50, "analog throttle");
			p->c[3] = s->a[3];
			p->a[3] = p->l[3] = 0;
		}
	}

	return (0);
}

static void NEAR
js_args (POINTER *p, char *options)
{
	long	l;

	if (get_arg (options, "count"))
		FOPTS &= ~USETIMER;

	if (get_arg (options, "linear"))
		FOPTS &= ~USELOG;

	if (get_narg (options, "rd=", &l))
		FNREAD = 1;
	else
		FNREAD = (int)l;

	if (get_narg (options, "dly=", &l))
		FDELAY = 1;
	else
		FDELAY = (int)l;

	if (get_narg (options, "bx1=", &l))
		FA1BASE = 1;
	else
		FA1BASE = (int)l;
	if (get_narg (options, "by1=", &l))
		FA2BASE = 1;
	else
		FA2BASE = (int)l;

	if (get_narg (options, "ix1=", &l))
		FA1IDLE = 20;
	else
		FA1IDLE = (int)l;
	if (get_narg (options, "iy1=", &l))
		FA2IDLE = 20;
	else
		FA2IDLE = (int)l;

	get_btn (p, options);
}

static int FAR
jsa_init (POINTER *p, char *options)
{
	long	l;

	FOPTS &= ~(HAT|THROTTLE|FOURBUTTONS);
	FOPTS |= USETIMER|USELOG;

	js_args (p, options);

	if (get_arg (options, "hat"))
		FOPTS |= HAT;
	else if (!get_narg (options, "ttl=", &l)) {
		FOPTS |= THROTTLE;
		if (l < 0) {

/* If I move this negation to AFTER the flags setting then micro&soft c8
 * generated bad code and 'l' ends up as 0x10000L!.
*/
			l = -l;
			FOPTS |= THTDOWN;
		}
		if (l > 0 && l < 100)
			FA3DETENT = (int)l;
		else
			FA3DETENT = REFDETENT;
	}

	if (get_arg (options, "four"))
		FOPTS |= FOURBUTTONS;

	if (get_narg (options, "bx2=", &l))
		FA3BASE = 1;
	else
		FA3BASE = (int)l;
	if (get_narg (options, "by2=", &l))
		FA4BASE = 1;
	else
		FA4BASE = (int)l;

	return (jsa_cal (p));
}

static void FAR
js_term (POINTER *p)
{}

static int FAR
js_center (POINTER *p)
{
	p->a[FA1F] = p->a[FA2F] = 0;
	p->l[FA1F] = p->l[FA2F] = 0;

	return (0);
}

static int FAR
jsb_read (POINTER *p, int transfer)
{
	STICK	s[1];
	char	btn[2];

	if (js_input (s, FOPTS & USETIMER, 0x0c, FNREAD, FDELAY))
		return (1);			/* reading failed */

	memset (btn, 0, sizeof (btn));

	symm_input (p, FA1F, s->a[2], -FA1D, FA1BASE, FA1IDLE, transfer);
	symm_input (p, FA2F, s->a[3],  FA2D, FA2BASE, FA2IDLE, transfer);

	btn[0] = s->b[2];
	btn[1] = s->b[3];
	do_btns (p, btn, rangeof (btn));

	return (0);
}

static int FAR
jsb_cal (POINTER *p)
{
	STICK	s[1];
	int	m, px, py;

	m = 0x0c;
	m = js_input (s, FOPTS & USETIMER, m, 10, 10);

	if (m & 0x0c || !s->a[2] || !s->a[3])
		return (1);

	px = FA1F;
	py = FA2F;

	p->c[px] = s->a[2];
	p->a[px] = p->l[px] = 0;

	p->c[py] = s->a[3];
	p->a[py] = p->l[py] = 0;

	return (0);
}

static int FAR
jsb_init (POINTER *p, char *options)
{
	FOPTS |= USETIMER;

	js_args (p, options);

	return (jsb_cal (p));
}

struct PtrDriver NEAR PtrAstick = {
	"ASTICK",
	0,
	jsa_init,
	js_term,
	jsa_cal,
	js_center,
	jsa_read,
	std_key
};

struct PtrDriver NEAR PtrBstick = {
	"BSTICK",
	0,
	jsb_init,
	js_term,
	jsb_cal,
	js_center,
	jsb_read,
	std_key
};

#undef HAT
#undef THROTTLE
#undef THTDOWN
#undef USETIMER
#undef FOURBUTTONS
#undef READA3
#undef USELOG
#undef PO
#undef FA1D
#undef FA1F
#undef FA2D
#undef FA2F
#undef FNREAD
#undef FDELAY
#undef FA1BASE
#undef FA2BASE
#undef FA3BASE
#undef FA4BASE
#undef FA1IDLE
#undef FA2IDLE
#undef REF
#undef FOPTS
