// HexEditView.cpp : implementation of the CHexEditView class
//
// Copyright (c) 1999 by Andrew W. Phillips.
//
// No restrictions are placed on the noncommercial use of this code,
// as long as this text (from the above copyright notice to the
// disclaimer below) is preserved.
//
// This code may be redistributed as long as it remains unmodified
// and is not sold for profit without the author's written consent.
//
// This code, or any part of it, may not be used in any software that
// is sold for profit, without the author's written consent.
//
// DISCLAIMER: This file is provided "as is" with no expressed or
// implied warranty. The author accepts no liability for any damage
// or loss of business that this product may cause.
//

#include "stdafx.h"
#include "HexEdit.h"
#include "HexFileList.h"

#include "HexEditDoc.h"
#include "HexEditView.h"
#include "MainFrm.h"

#include "Boyer.h"
#include "Misc.h"

#ifdef _DEBUG
#include "timer.h"
#endif

#define MAX_CLIPBOARD 16000000L  // Putting more than this on the clipboard (Win 95/98) causes crash

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CHexEditView

IMPLEMENT_DYNCREATE(CHexEditView, CScrView)

BEGIN_MESSAGE_MAP(CHexEditView, CScrView)
        //{{AFX_MSG_MAP(CHexEditView)
        ON_WM_SIZE()
        ON_COMMAND(ID_ADDR_TOGGLE, OnAddrToggle)
        ON_UPDATE_COMMAND_UI(ID_ADDR_TOGGLE, OnUpdateAddrToggle)
        ON_COMMAND(ID_GRAPHIC_TOGGLE, OnGraphicToggle)
        ON_UPDATE_COMMAND_UI(ID_GRAPHIC_TOGGLE, OnUpdateGraphicToggle)
        ON_COMMAND(ID_CHAR_TOGGLE, OnCharToggle)
        ON_UPDATE_COMMAND_UI(ID_CHAR_TOGGLE, OnUpdateCharToggle)
        ON_COMMAND(ID_FONT, OnFont)
        ON_COMMAND(ID_AUTOFIT, OnAutoFit)
        ON_UPDATE_COMMAND_UI(ID_AUTOFIT, OnUpdateAutofit)
        ON_COMMAND(ID_ASC_EBC, OnAscEbc)
        ON_UPDATE_COMMAND_UI(ID_ASC_EBC, OnUpdateAscEbc)
        ON_COMMAND(ID_CONTROL, OnControl)
        ON_UPDATE_COMMAND_UI(ID_CONTROL, OnUpdateControl)
        ON_WM_CHAR()
        ON_WM_SETFOCUS()
        ON_WM_KILLFOCUS()
        ON_WM_DESTROY()
        ON_WM_LBUTTONDOWN()
        ON_COMMAND(ID_MARK, OnMark)
        ON_COMMAND(ID_GOTO_MARK, OnGotoMark)
        ON_WM_LBUTTONDBLCLK()
        ON_WM_KEYDOWN()
        ON_COMMAND(ID_EDIT_UNDO, OnEditUndo)
        ON_COMMAND(ID_SEARCH_HEX, OnSearchHex)
        ON_COMMAND(ID_SEARCH_ASCII, OnSearchAscii)
        ON_UPDATE_COMMAND_UI(ID_EDIT_UNDO, OnUpdateEditUndo)
        ON_COMMAND(ID_SEARCH_FORW, OnSearchForw)
        ON_COMMAND(ID_SEARCH_BACK, OnSearchBack)
        ON_COMMAND(ID_ALLOW_MODS, OnAllowMods)
        ON_UPDATE_COMMAND_UI(ID_ALLOW_MODS, OnUpdateAllowMods)
        ON_COMMAND(ID_CONTROL_TOGGLE, OnControlToggle)
        ON_COMMAND(ID_INSERT, OnInsert)
        ON_UPDATE_COMMAND_UI(ID_INSERT, OnUpdateInsert)
        ON_COMMAND(ID_SEARCH_ICASE, OnSearchIcase)
        ON_UPDATE_COMMAND_UI(ID_SEARCH_FORW, OnUpdateSearch)
        ON_COMMAND(ID_EDIT_COMPARE, OnEditCompare)
        ON_COMMAND(ID_WINDOW_NEXT, OnWindowNext)
        ON_UPDATE_COMMAND_UI(ID_EDIT_COMPARE, OnUpdateEditCompare)
        ON_WM_CONTEXTMENU()
        ON_WM_RBUTTONDOWN()
        ON_COMMAND(ID_INC_BYTE, OnIncByte)
        ON_COMMAND(ID_INC_16BIT, OnInc16bit)
        ON_COMMAND(ID_INC_32BIT, OnInc32bit)
        ON_COMMAND(ID_INC_64BIT, OnInc64bit)
        ON_COMMAND(ID_DEC_BYTE, OnDecByte)
        ON_COMMAND(ID_DEC_16BIT, OnDec16bit)
        ON_COMMAND(ID_DEC_32BIT, OnDec32bit)
        ON_COMMAND(ID_DEC_64BIT, OnDec64bit)
        ON_COMMAND(ID_FLIP_16BIT, OnFlip16bit)
        ON_COMMAND(ID_FLIP_32BIT, OnFlip32bit)
        ON_COMMAND(ID_FLIP_64BIT, OnFlip64bit)
        ON_UPDATE_COMMAND_UI(ID_INC_BYTE, OnUpdateByte)
        ON_UPDATE_COMMAND_UI(ID_INC_16BIT, OnUpdate16bit)
        ON_UPDATE_COMMAND_UI(ID_INC_32BIT, OnUpdate32bit)
        ON_UPDATE_COMMAND_UI(ID_INC_64BIT, OnUpdate64bit)
        ON_WM_LBUTTONUP()
        ON_COMMAND(ID_SELECT_ALL, OnSelectAll)
        ON_COMMAND(ID_EDIT_COPY, OnEditCopy)
        ON_COMMAND(ID_EDIT_CUT, OnEditCut)
        ON_COMMAND(ID_EDIT_PASTE, OnEditPaste)
        ON_UPDATE_COMMAND_UI(ID_EDIT_PASTE, OnUpdateTextPaste)
        ON_UPDATE_COMMAND_UI(ID_EDIT_COPY, OnUpdateClipboard)
        ON_COMMAND(ID_SEARCH_SEL, OnSearchSel)
        ON_UPDATE_COMMAND_UI(ID_PASTE_UNICODE, OnUpdateUnicodePaste)
        ON_WM_SETCURSOR()
        ON_COMMAND(ID_FONT_DEC, OnFontDec)
        ON_COMMAND(ID_FONT_INC, OnFontInc)
        ON_UPDATE_COMMAND_UI(ID_FONT_DEC, OnUpdateFontDec)
        ON_UPDATE_COMMAND_UI(ID_FONT_INC, OnUpdateFontInc)
        ON_UPDATE_COMMAND_UI(ID_EDIT_CUT, OnUpdateEditCut)
        ON_COMMAND(ID_PASTE_ASCII, OnPasteAscii)
        ON_COMMAND(ID_PASTE_EBCDIC, OnPasteEbcdic)
        ON_COMMAND(ID_PASTE_UNICODE, OnPasteUnicode)
        ON_COMMAND(ID_COPY_CCHAR, OnCopyCchar)
        ON_COMMAND(ID_COPY_HEX, OnCopyHex)
        ON_COMMAND(ID_EDIT_WRITEFILE, OnEditWriteFile)
        ON_UPDATE_COMMAND_UI(ID_EDIT_READFILE, OnUpdateReadFile)
        ON_COMMAND(ID_EDIT_READFILE, OnReadFile)
        ON_WM_HSCROLL()
        ON_WM_VSCROLL()
        ON_COMMAND(ID_EXTENDTO_MARK, OnExtendToMark)
        ON_COMMAND(ID_SWAP_MARK, OnSwapMark)
        ON_COMMAND(ID_REDRAW, OnRedraw)
        ON_COMMAND(ID_SCROLL_DOWN, OnScrollDown)
        ON_COMMAND(ID_SCROLL_UP, OnScrollUp)
        ON_COMMAND(ID_SWAP, OnSwap)
        ON_COMMAND(ID_START_LINE, OnStartLine)
        ON_COMMAND(ID_DEL, OnDel)
        ON_UPDATE_COMMAND_UI(ID_SWAP, OnUpdateSwap)
        ON_COMMAND(ID_OEM_TOGGLE, OnOemToggle)
        ON_UPDATE_COMMAND_UI(ID_OEM_TOGGLE, OnUpdateOemToggle)
        ON_UPDATE_COMMAND_UI(ID_CONTROL_TOGGLE, OnUpdateControl)
        ON_UPDATE_COMMAND_UI(ID_SEARCH_BACK, OnUpdateSearch)
        ON_UPDATE_COMMAND_UI(ID_DEC_BYTE, OnUpdateByte)
        ON_UPDATE_COMMAND_UI(ID_DEC_16BIT, OnUpdate16bit)
        ON_UPDATE_COMMAND_UI(ID_DEC_32BIT, OnUpdate32bit)
        ON_UPDATE_COMMAND_UI(ID_DEC_64BIT, OnUpdate64bit)
        ON_UPDATE_COMMAND_UI(ID_FLIP_16BIT, OnUpdate16bit)
        ON_UPDATE_COMMAND_UI(ID_FLIP_32BIT, OnUpdate32bit)
        ON_UPDATE_COMMAND_UI(ID_FLIP_64BIT, OnUpdate64bit)
        ON_UPDATE_COMMAND_UI(ID_SEARCH_SEL, OnUpdateClipboard)
        ON_UPDATE_COMMAND_UI(ID_PASTE_ASCII, OnUpdateTextPaste)
        ON_UPDATE_COMMAND_UI(ID_PASTE_EBCDIC, OnUpdateTextPaste)
        ON_UPDATE_COMMAND_UI(ID_COPY_CCHAR, OnUpdateClipboard)
        ON_UPDATE_COMMAND_UI(ID_COPY_HEX, OnUpdateClipboard)
        ON_UPDATE_COMMAND_UI(ID_EDIT_WRITEFILE, OnUpdateClipboard)
        ON_WM_MOUSEWHEEL()
        //}}AFX_MSG_MAP
        // Standard printing commands
        ON_COMMAND(ID_FILE_PRINT, OnFilePrint)
        ON_COMMAND(ID_FILE_PRINT_DIRECT, OnFilePrint)
        ON_COMMAND(ID_FILE_PRINT_PREVIEW, OnFilePrintPreview)
        ON_COMMAND(ID_JUMP_HEX_ADDR, OnJumpHexAddr)
#ifdef ENCOM
        ON_COMMAND(ID_RS_FORW, OnRsForw)
        ON_UPDATE_COMMAND_UI(ID_RS_FORW, OnUpdateRsForw)
        ON_COMMAND(ID_RS_BACK, OnRsBack)
        ON_UPDATE_COMMAND_UI(ID_RS_BACK, OnUpdateRsBack)
#endif
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CHexEditView construction/destruction
CHexEditView::CHexEditView()
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    // We have to calculate these here since messages such as OnSize depend
    // on them and can be received before OnInitialUpdate() is called.
    win_size_ = null_size;
    pfont_ = NULL;
    pbrush_ = NULL;
    print_font_ = NULL;

    // Set up opening display based on global options
    rowsize_ = aa->open_rowsize_;
    real_offset_ = offset_ = aa->open_offset_;
    if (real_offset_ >= rowsize_)
        offset_ = rowsize_ - 1;         // In case soemone fiddled with the settings
    group_by_ = aa->open_group_by_;

    autofit_ = aa->open_autofit_;
    dec_addr_ = aa->open_dec_addr_;
    display_char_ = aa->open_display_char_;
    ebcdic_ = aa->open_ebcdic_;
    display_control_ = aa->open_control_ ? 1 : 0;
    graphic_ = aa->open_graphic_;
    oem_ = aa->open_oem_;

    set_colours(aa->open_partitions_);

    readonly_ = !aa->open_allow_mods_;
    overtype_ = !aa->open_insert_;

    mark_ = 0L;                         // Mark initially at start of file
    edit_char_ = mark_char_ = FALSE;    // Caret init. in hex not char area

    mouse_down_ = false;
    needs_refresh_ = false;
    needs_hscroll_ = false;
    needs_vscroll_ = false;

    memset((void *)&lf_, '\0', sizeof(lf_));
    _tcscpy(lf_.lfFaceName, _T("Courier")); // A nice fixed (no-proportional) font
    lf_.lfHeight = 16;
    lf_.lfCharSet = ANSI_CHARSET;           // Only allow ANSI character set fonts
    oem_lf_ = lf_;
    _tcscpy(oem_lf_.lfFaceName, _T("Terminal")); // The only certain OEM font?
    oem_lf_.lfHeight = 18;
    oem_lf_.lfCharSet = OEM_CHARSET;         // Only allow OEM/IBM character set fonts

    search_length_ = 0;

#ifndef NDEBUG
    // Make default capacity for undo_ vector small to force reallocation sooner.
    // This increases likelihood of catching bugs related to reallocation.
    undo_.reserve(4);
#else
    undo_.reserve(1024);
#endif
}

CHexEditView::~CHexEditView()
{
    if (pfont_ != NULL)
    {
        delete pfont_;
        pfont_ = NULL;
    }
    if (pbrush_ != NULL)
    {
        delete pbrush_;
        pbrush_ = NULL;
    }
}

BOOL CHexEditView::PreCreateWindow(CREATESTRUCT& cs) 
{
	BOOL retval = CScrView::PreCreateWindow(cs);

    // Get the create context so we can find out if this window is
    // being created via Window/New and the frame/view it's cloned from
    CCreateContext *pContext = (CCreateContext *)cs.lpCreateParams;

    if (pContext->m_pCurrentFrame != NULL)
    {
        // Must have been created via Window/New (ID_WINDOW_NEW)
        ASSERT_KINDOF(CMDIChildWnd, pContext->m_pCurrentFrame);

        // We have the frame so get the view within it
        CHexEditView *pView = (CHexEditView *)pContext->m_pCurrentFrame->GetActiveView();
        ASSERT_KINDOF(CHexEditView, pView);

        // Make this view's undo stack the same as clone view
        undo_ = pView->undo_;

        // Flag any changes as not being made in this view
        std::vector <view_undo>::iterator pu;
        for (pu = undo_.begin(); pu != undo_.end(); ++pu)
        {
            if (pu->utype == undo_change)
                pu->flag = FALSE;
        }
    }

    return retval;
}

void CHexEditView::OnInitialUpdate() 
{
    long start_addr = 0;
    long end_addr = 0;
    CHexEditDoc *pDoc = GetDocument();
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    print_map_mode_ = MM_LOENGLISH;

    // Get options for the window from file settings in CHexFileList
    CHexFileList *pfl = aa->GetFileList();
    int recent_file_index = -1;
    if (pDoc->file_.m_hFile != CFile::hFileNull)
        recent_file_index = pfl->GetIndex(pDoc->file_.GetFilePath());
    if (recent_file_index != -1)
    {
        autofit_ = pfl->autofit_[recent_file_index];
        dec_addr_ = pfl->dec_addr_[recent_file_index];
        display_char_ = pfl->display_char_[recent_file_index];
        ebcdic_ = pfl->ebcdic_[recent_file_index];
        oem_ = pfl->oem_[recent_file_index];
        graphic_ = pfl->graphic_[recent_file_index];
        display_control_ = pfl->control_[recent_file_index] ? 1 : 0;

        if (pfl->ppart_[recent_file_index] != NULL)
            set_colours(*pfl->ppart_[recent_file_index]);

        readonly_ = !pfl->allow_mods_[recent_file_index];
        overtype_ = !pfl->insert_[recent_file_index];
            
        rowsize_ = pfl->rowsize_[recent_file_index];
        real_offset_ = offset_ = pfl->offset_[recent_file_index];
        if (real_offset_ >= rowsize_)
            offset_ = rowsize_ - 1;             // In case soemone fiddled with the settings
        group_by_ = pfl->group_by_[recent_file_index];

        start_addr = pfl->start_addr_[recent_file_index];
        end_addr = pfl->end_addr_[recent_file_index];
        if (start_addr < 0 || start_addr > pDoc->length())
            start_addr = 0;
        if (end_addr < start_addr || end_addr > pDoc->length())
            end_addr = start_addr;

        mark_ = pfl->mark_[recent_file_index];
        if (mark_ < 0 || mark_ > pDoc->length())
            mark_ = 0;
        edit_char_ = pfl->edit_char_[recent_file_index];

        // Get saved font info
        strncpy(lf_.lfFaceName, pfl->face_[recent_file_index], LF_FACESIZE-1);
        lf_.lfFaceName[LF_FACESIZE-1] = '\0';
        lf_.lfHeight = pfl->height_[recent_file_index];
        strncpy(oem_lf_.lfFaceName, pfl->oem_face_[recent_file_index], LF_FACESIZE-1);
        oem_lf_.lfFaceName[LF_FACESIZE-1] = '\0';
        oem_lf_.lfHeight = pfl->oem_height_[recent_file_index];

        // Make sure that window size seems reasonable
        if (pfl->top_[recent_file_index] != -30000 &&
            pfl->right_[recent_file_index] != -30000 &&
            pfl->top_[recent_file_index] < pfl->bottom_[recent_file_index] &&
            pfl->left_[recent_file_index] < pfl->right_[recent_file_index] )
        {
            WINDOWPLACEMENT wp;
            wp.length = sizeof(wp);
            wp.flags = 0;
            wp.showCmd = pfl->cmd_[recent_file_index];
            wp.ptMinPosition = CPoint(-1,-1);
            wp.ptMaxPosition = CPoint(-1,-1);
            CRect newpos(pfl->left_[recent_file_index], pfl->top_[recent_file_index], pfl->right_[recent_file_index], pfl->bottom_[recent_file_index]);

            // If this window has sibling view this window was presumably created
            // using Window/New - make sure its not in the same place as its sibling(s).
            POSITION pos = pDoc->GetFirstViewPosition();
            while (pos != NULL)
            {
                WINDOWPLACEMENT parent_wp;
                parent_wp.length = sizeof(parent_wp);
                CView *pv = pDoc->GetNextView(pos);
                ASSERT(pv != NULL && pv->GetParent() != NULL);
                if (pv != this && pv->GetParent()->GetWindowPlacement(&parent_wp))
                {
                    // If the top left corners are about the same move the
                    // new window down and right a bit.
                    if (abs(newpos.top - parent_wp.rcNormalPosition.top) < 20 &&
                        abs(newpos.left - parent_wp.rcNormalPosition.left) < 20)
                    {
                        newpos += CSize(30, 30);
                    }
                }
            }

            // Check that new position is not completely off the left, right or bottom,
            // and that the top title bar is still visible to allow dragging.
            CRect rr;
            ASSERT(GetParent() != NULL && GetParent()->GetParent() != NULL);
            GetParent()->GetParent()->GetClientRect(&rr);
            if (newpos.left > rr.right-20)
            {
                newpos.left = rr.right - (newpos.right - newpos.left);
                newpos.right = rr.right;
            }
            if (newpos.right < 20)
            {
                newpos.right -= newpos.left;
                newpos.left = 0;
            }
            if (newpos.top > rr.bottom-20)
            {
                newpos.top = rr.bottom - (newpos.bottom - newpos.top);
                newpos.bottom = rr.bottom;
            }
            if (newpos.top < -20)
            {
                newpos.bottom -= newpos.top;
                newpos.top =0;
            }

            wp.rcNormalPosition = newpos;
            GetParent()->SetWindowPlacement(&wp);
        }
        else if (pfl->cmd_[recent_file_index] == SW_SHOWMAXIMIZED)
        {
            WINDOWPLACEMENT wp;
            ASSERT(GetParent() != NULL);
            GetParent()->GetWindowPlacement(&wp);
            wp.showCmd = SW_SHOWMAXIMIZED;
            GetParent()->SetWindowPlacement(&wp);
        }

        // Set size so that then set scroll posn
        if (display_char_)
            SetSize(CSize(char_pos(rowsize_), -1));
        else
            SetSize(CSize(hex_pos(rowsize_), -1));
        SetTSize(CSize(-1, (GetDocument()->length() + offset_)/rowsize_ + 1));
        SetScroll(CPoint(0,pfl->scroll_[recent_file_index]));
    }
    else
    {
        if (aa->open_max_)
        {
            WINDOWPLACEMENT wp;
            ASSERT(GetParent() != NULL);
            GetParent()->GetWindowPlacement(&wp);
            wp.showCmd = SW_SHOWMAXIMIZED;
            GetParent()->SetWindowPlacement(&wp);
        }

        // Set the normal and OEM graphic font
        if (aa->open_plf_ != NULL)
        {
            lf_ = *aa->open_plf_;
        }

        if (aa->open_oem_plf_ != NULL)
        {
            oem_lf_ = *aa->open_oem_plf_;
        }
    }

    // Convert font sizes to logical units
    {
        CPoint convert(0, lf_.lfHeight);
        CClientDC dc(this);
        OnPrepareDC(&dc);
        dc.DPtoLP(&convert);                    // Get screen font size in logical units
        lf_.lfHeight = convert.y;
    }
    {
        CPoint convert(0, oem_lf_.lfHeight);
        CClientDC dc(this);
        OnPrepareDC(&dc);
        dc.DPtoLP(&convert);                    // Get screen font size in logical units
        oem_lf_.lfHeight = convert.y;
    }

    // This can't be done till document available (ie. not in constructor)
    if (GetDocument()->read_only())
        readonly_ = TRUE;

    // Set control bar buttons to display state of current options
    load_bitmaps();

    CScrView::SetMapMode(MM_TEXT);

    ASSERT(pfont_ == NULL);
    pfont_ = new CFont;
    pfont_->CreateFontIndirect(display_char_ && !ebcdic_ && graphic_ && oem_ ? &oem_lf_ : &lf_);
    SetFont(pfont_);

    // Create brush that is XORed with selection when focus lost.  (This
    // gives a more subtle grey selection) when window does not have focus.)
    pbrush_ = new CBrush(RGB(192,192,192));
    SetBrush(pbrush_);

    CScrView::OnInitialUpdate();

    if (recent_file_index != -1)
        SetScroll(CPoint(0,pfl->scroll_[recent_file_index]));

#if 0
    // What's the point of an empty read-only file
    if (GetDocument()->length() == 0 && !GetDocument()->read_only())
        readonly_ = overtype_ = FALSE;
#endif

    if (aa->large_cursor_)
        BlockCaret();
    else
        LineCaret();
    if (!overtype_ || GetDocument()->length() > 0)
    {
        CaretMode();
//      SetCaret(addr2pos(0));
        SetSel(addr2pos(start_addr), addr2pos(end_addr));
        show_prop();
        show_calc();
        show_pos();
    }

    // Since we now have a view in which we can do jumps and searches,
    // make sure the control bar combo controls are enabled.
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
    CWnd *pcombo;           // Ptr to combo control in dialog bar
    pcombo = mm->m_wndEditBar.GetDlgItem(IDC_JUMP_HEX);
    ASSERT(pcombo != NULL);
    if (pcombo != NULL)
        pcombo->EnableWindow(TRUE);
    pcombo = mm->m_wndEditBar.GetDlgItem(IDC_JUMP_DEC);
    ASSERT(pcombo != NULL);
    if (pcombo != NULL)
        pcombo->EnableWindow(TRUE);
    pcombo = mm->m_wndEditBar.GetDlgItem(IDC_SEARCH);
    ASSERT(pcombo != NULL);
    if (pcombo != NULL)
        pcombo->EnableWindow(TRUE);

    ValidateScroll(GetScroll());
}

// Update our options to the CHexFileList 
void CHexEditView::StoreOptions()
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    CHexFileList *pfl = ((CHexEditApp *)AfxGetApp())->GetFileList();
    if (GetDocument()->file_.m_hFile == CFile::hFileNull)
        return;                                 // Not in the list if there's no file
    int ii = pfl->GetIndex(GetDocument()->file_.GetFilePath());
    if (ii != -1)
    {
        // Found in the list so update all the values;
        pfl->autofit_[ii] = autofit_;
        pfl->dec_addr_[ii] = dec_addr_;
        pfl->display_char_[ii] = display_char_;
        pfl->ebcdic_[ii] = ebcdic_;
        pfl->oem_[ii] = oem_;
        pfl->graphic_[ii] = graphic_;
        pfl->control_[ii] = display_control_ > 0 ? TRUE : FALSE;

        // Save the colours if they have been changed from the default
        if (partitions_ != aa->open_partitions_)
        {
            if (pfl->ppart_[ii] == NULL)
                pfl->ppart_[ii] = new std::vector<partn>;
            *pfl->ppart_[ii] = partitions_;
        }
        else if (pfl->ppart_[ii] != NULL)
        {
            delete pfl->ppart_[ii];
            pfl->ppart_[ii] = NULL;
        }

        pfl->allow_mods_[ii] = readonly_ ? FALSE : TRUE;
        pfl->insert_[ii] = overtype_ ? FALSE : TRUE;
            
        pfl->rowsize_[ii] = rowsize_;
        pfl->offset_[ii] = offset_;
        pfl->group_by_[ii] = group_by_;

        long start_addr, end_addr;
        GetSelAddr(start_addr, end_addr);
        pfl->start_addr_[ii] = start_addr;
        pfl->end_addr_[ii] = end_addr;
        pfl->mark_[ii] = mark_;
        pfl->edit_char_[ii] = edit_char_;
        pfl->scroll_[ii] = GetScroll().y;

        pfl->face_[ii] = lf_.lfFaceName;
        pfl->height_[ii] = lf_.lfHeight;
        pfl->oem_face_[ii] = oem_lf_.lfFaceName;
        pfl->oem_height_[ii] = oem_lf_.lfHeight;

        WINDOWPLACEMENT wp;
        ASSERT(GetParent() != NULL);
        if (GetParent()->GetWindowPlacement(&wp))
        {
            pfl->cmd_[ii] = wp.showCmd;
            pfl->top_[ii] = wp.rcNormalPosition.top;
            pfl->left_[ii] = wp.rcNormalPosition.left;
            pfl->bottom_[ii] = wp.rcNormalPosition.bottom;
            pfl->right_[ii] = wp.rcNormalPosition.right;
        }
        else
        {
            pfl->cmd_[ii] = SW_SHOWNORMAL;
            pfl->top_[ii] = -30000;
        }
    }
}

// Stores the colours that the hex values are displayed in.  For efficiency it builds the
// "kala" array which contains a COLORREF for every byte value.
void CHexEditView::set_colours(const std::vector<partn> &c)
{
    int ii;                             // Loop var = byte value from 0 to 255

    partitions_ = c;                       // Store colour ranges so user can change them later

    // Default colours to white (makes unspecified colours invisible)
    for (ii = 0; ii < 256; ++ii)
        kala[ii] = RGB(255, 255, 255);

    // For each group: get its colours and assign it to all byte in the range
    for (std::vector<partn>::reverse_iterator pp = partitions_.rbegin();
                                         pp != partitions_.rend(); ++pp)
        for (range_set<int>::iterator rr = (*pp).range.begin();
             rr != (*pp).range.end(); ++rr)
        {
            kala[*rr] = (*pp).col;
        }
}

CFont *CHexEditView::SetFont(CFont *ff)
{
    CFont *retval = CScrView::SetFont(ff);

    // Work out size of printer font based on the new screen font
    TEXTMETRIC tm;
    CClientDC dc(this);
    OnPrepareDC(&dc);                   // Get screen map mode/font
    dc.GetTextMetrics(&tm);
    CPoint convert(0, tm.tmHeight);
    dc.LPtoDP(&convert);                // Convert screen font size to device units
    dc.SetMapMode(print_map_mode_);
    dc.DPtoLP(&convert);                // Get printer font size in logical units
    print_lfHeight_ = convert.y;

    return retval;
}

void CHexEditView::OnUpdate(CView* pSender, LPARAM lHint, CObject* pHint) 
{
    // Check if this is a change caused by a view
    if (pHint != NULL && pHint->IsKindOf(RUNTIME_CLASS(CRemoveHint)))
    {
        // Remove undo info (as opposed to actually undoing everything as below)
        num_entered_ = num_del_ = num_bs_ = 0;  // Changes now gone from doc undo
        CRemoveHint *prh = dynamic_cast<CRemoveHint *>(pHint);
        std::vector <view_undo, allocator<view_undo> >::iterator pu;
        for (pu = undo_.end(); pu != undo_.begin(); pu--)
            if ((pu-1)->utype == undo_change)
            {
                // Remove everything up to & including the last doc change undo
                ASSERT((pu-1)->index == prh->index);
                undo_.erase(undo_.begin(), pu);
                break;
            }
    }
    else if (pHint != NULL && pHint->IsKindOf(RUNTIME_CLASS(CUndoHint)))
    {
        num_entered_ = num_del_ = num_bs_ = 0;

        // Undo all view changes up to last doc change (no window invalidate)
        CUndoHint *puh = dynamic_cast<CUndoHint *>(pHint);

        // If this is not the view that is undoing
        if (puh->pview != this)
        {
            // Undo all view moves etc up to this change
            ASSERT(undo_.size() > 0);
            while (undo_.size() > 0 && undo_.back().utype != undo_change)
                do_undo();
            ASSERT(undo_.size() > 0);
            ASSERT(undo_.back().index == puh->index);
        }
    }
    else if (pHint != NULL && pHint->IsKindOf(RUNTIME_CLASS(CHexHint)))
    {
        // This hint informs the view that the document has changed with
        // an insertion, deletion or overtype.  This may be as a result of a
        // document undo (if is_undo is true)

        // Get current selection before recalc_display changes row offsets
        long start_addr, end_addr;
        GetSelAddr(start_addr, end_addr);
        int prev_addr_width = addr_width_;

        // Make sure CScrView knows about any size changes
        SetTSize(CSize(-1, (GetDocument()->length() + offset_)/rowsize_ + 1));

        recalc_display();  // file length, hence addr_width_, may have changed

        CHexHint *phh = dynamic_cast<CHexHint *>(pHint);
        ASSERT(phh->address >= 0 && phh->address <= GetDocument()->length());

        // Is this the start of a doc modification?
        // (phh->index == -1 if this a continued modification)
        if (!phh->is_undo && phh->index > -1)
        {
            // New change - save mark, caret + change on undo stack
            // xxx This could be optimized to only save mark and caret
            // if they are moved by the change.
            undo_.push_back(view_undo(undo_setmark));
            undo_.back().address = mark_;
            undo_.push_back(view_undo(undo_move, TRUE));
            undo_.back().address = start_addr;
            if (start_addr != end_addr)
            {
                undo_.push_back(view_undo(undo_sel, TRUE));
                undo_.back().address = end_addr;
            }
            undo_.push_back(view_undo(undo_change, TRUE));
            if (phh->pview == this)
                undo_.back().flag = TRUE;
            else
                undo_.back().flag = FALSE;
#ifndef NDEBUG
            undo_.back().index = phh->index;
#endif
        }

        // Move positions of the caret and the mark if address shifted
        if (!phh->is_undo && phh->utype == mod_insert)
        {
            if (end_addr > phh->address)
            {
//              SetCaret(addr2pos(address + phh->len));
                end_addr += phh->len;
                if (start_addr >= phh->address)
                    start_addr += phh->len;
                SetSel(addr2pos(start_addr), addr2pos(end_addr));
            }

            if (mark_ >= phh->address)
                mark_ += phh->len;
        }
        else if (!phh->is_undo &&
                 (phh->utype == mod_delback || phh->utype == mod_delforw))
        {
            // Check if current selection and the deletion intersect
            if (start_addr < phh->address + phh->len && end_addr > phh->address)
            {
                if (start_addr >= phh->address)     // If sel start within deletion ...
                    start_addr = phh->address;      // ... move it to where chars deleted
                if (end_addr <= phh->address + phh->len) // If sel end within deletion ...
                    end_addr = phh->address;        // ... move it to where chars deleted
                else
                    end_addr -= phh->len;           // past deletion so just move it back
                SetSel(addr2pos(start_addr), addr2pos(end_addr));
            }
            else if (phh->address + phh->len <= start_addr)
            {
                // Deletion is before selection - just move selection backwards
                SetSel(addr2pos(start_addr - phh->len), addr2pos(end_addr - phh->len));
            }

            if (mark_ > phh->address + phh->len)
                mark_ -= phh->len;
            else if (mark_ > phh->address)
                mark_ = phh->address;
        }

        // Work out the addresses of the first and last line displayed
        CRect clip_rect;                // rectangle for calcs/clipping

        // Calculate where the display in document
        GetClientRect(&clip_rect);      // First: get client rectangle
        ConvertFromDP(clip_rect);

        // Calculate the address of the first byte of the top row of the
        // display and the first byte of the row just past bottom
        long addr_top = (clip_rect.top/text_height_)*rowsize_ - offset_;
        long addr_bot = (clip_rect.bottom/text_height_ + 1)*rowsize_ - offset_;

        if (addr_width_ != prev_addr_width)
        {
            // Addresses on left side are now different width so redraw everything
            DoInvalidate();
        }
        else if (phh->address >= addr_bot ||
            (phh->address + (signed)phh->len < addr_top &&
             (phh->utype == mod_replace || phh->utype == mod_repback)))
        {
            // Do nothing - changes after display OR before but with no address shift
            ; /* null statement */
        }
        else if (phh->address < addr_top + rowsize_ &&
                 phh->utype != mod_replace && phh->utype != mod_repback)
        {
            // Whole display changes, due to address shift
            DoInvalidate();
        }
        else if (phh->utype == mod_replace || phh->utype == mod_repback)
        {
            // Replace starting within display - just invalidate changed area
            invalidate_addr_range(phh->address, phh->address + (phh->len<1?1:phh->len));
        }
        else if (GetDocument()->length() +
                    (phh->utype == mod_insert ? -phh->len : +phh->len) < addr_bot)
        {
            // Insertion/deletion and previous end was within display
            // Just invalidate from address to bottom of display
            CRect inv(0, ((phh->address + offset_)/rowsize_) * text_height_,
                      char_pos(rowsize_), ((addr_bot + offset_)/rowsize_) * text_height_);
            ConvertToDP(inv);
            DoInvalidateRect(&inv);
        }
        else
        {
            // Must be insert/delete starting within the displayed area
            // - invalidate from first changed line to end of display
            invalidate_addr_range(phh->address, addr_bot);
        }

        // Remove doc change undo from undo array & do any assoc undos
        if (phh->is_undo && phh->pview != this)
        {
            ASSERT(undo_.size() > 0);
            if (undo_.size() > 0)
            {
                BOOL ptoo = undo_.back().previous_too;

                undo_.pop_back();       // Change already made to doc
                while (ptoo)
                {
                    if (undo_.size() == 0)
                        break;
                    // Undo anything associated to change to doc
                    ptoo = undo_.back().previous_too;
                    do_undo();
                }
            }
        }
        show_prop();
        show_calc();
    }
    else if (pHint != NULL && pHint->IsKindOf(RUNTIME_CLASS(CBGSearchHint)))
    {
        CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
        CBGSearchHint *ph = dynamic_cast<CBGSearchHint *>(pHint);
        if (ph->finished_ == -1)
        {
            // Get current set of search occurrences
            ValidateScroll(GetScroll());

            invalidate_addr_range(ph->start_, ph->end_ + search_length_ - 1);
        }
        else if (ph->finished_)
        {
            // Get new search strings
            ValidateScroll(GetScroll());

            // Invalidate any areas where new search string was found
            std::vector<FILE_ADDRESS>::const_iterator pp, pend;
            for (pp = search_found_.begin(), pend = search_found_.end(); pp != pend; ++pp)
            {
                invalidate_addr_range(*pp, *pp + search_length_);
            }
        }
        else
        {
            // Remove all displayed search strings (but save a copy in tmp for invalidating)
            std::vector<FILE_ADDRESS> tmp;
            int len = search_length_;
            tmp.swap(search_found_);    // Save search_found_ in tmp (and make it empty)

            // Invalidate any areas where search string is currently displayed
            std::vector<FILE_ADDRESS>::const_iterator pp, pend;
            for (pp = tmp.begin(), pend = tmp.end(); pp != pend; ++pp)
            {
                ASSERT(len > 0);
                invalidate_addr_range(*pp, *pp + len);
            }
        }
    }
    else
    {
        recalc_display();
        CScrView::OnUpdate(pSender, lHint, pHint);
    }
}

void CHexEditView::OnActivateView(BOOL bActivate, CView* pActivateView, CView* pDeactiveView) 
{
    if (bActivate)
        GetDocument()->BGThreadPriority(THREAD_PRIORITY_LOWEST);
    else
        GetDocument()->BGThreadPriority(THREAD_PRIORITY_IDLE);

	CScrView::OnActivateView(bActivate, pActivateView, pDeactiveView);
}

// Checks if the document (file) or view (window) is read only.
// If the document is read only it just displays a message and returns.
// If the view is read only it gives the user the option to turn off RO.
// Returns TRUE if the file/view is read only, FALSE if modifiable.
BOOL CHexEditView::check_ro(const char *desc)
{
    CString ss;
    if (readonly_ && GetDocument()->read_only())
    {
        ss.Format("This file cannot be modified.\r"
                  "(You can't %s.)", desc);
        ::HMessageBox(ss);
        CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
        aa->mac_error_ = 10;
        return TRUE;
    }
    else if (readonly_)
    {
        ss.Format("You can't %s since this window is read only.\r"
                  "Do you want to turn off read only mode?", desc);
        if (::HMessageBox(ss, MB_OKCANCEL) != IDOK)
        {
            CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
            aa->mac_error_ = 10;
            return TRUE;
        }
        else
            allow_mods();
    }
    return FALSE;
}

/////////////////////////////////////////////////////////////////////////////
// CHexEditView drawing

void CHexEditView::OnDraw(CDC* pDC)
{
    if (pfont_ == NULL) return;

    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    CHexEditDoc* pDoc = GetDocument();
    ASSERT_VALID(pDoc);

    bool printing = pDC->IsPrinting() != 0 ? true : false;

    // Are we in overtype mode and file is empty?
    if (overtype_ && pDoc->length() == 0)
    {
        // Get the reason that there's no data from the document & display it
        CRect cli;
        GetClientRect(&cli);
        pDC->DPtoLP(&cli);
        if (printing)
        {
            if (cli.bottom < 0)
                cli.top = -2 * print_text_height_;
            else
                cli.top = 2 * print_text_height_;
        }
        const char *mm = pDoc->why0();
        pDC->SetTextColor(RGB(0,0,0));
        pDC->DrawText(mm, -1, &cli, DT_CENTER | DT_TOP | DT_NOPREFIX | DT_SINGLELINE);
        return;
    }

    const char *hex;
    if (aa->hex_ucase_)
        hex = "0123456789ABCDEF?";
    else
        hex = "0123456789abcdef?";

    long first_addr = 0;                // First address to actually display
    long last_addr = pDoc->length();    // One past last address actaully displayed
    long first_line;                    // First line that needs displaying
    long last_line;                     // One past last line to display
    long line_inc;                      // 1 or -1 depending on draw dirn (up/down)
    CRect line_rect;                    // Where to draw current text line
    CRect disp_rect;                    // Display area relative to whole document
    CSize rect_inc;                     // How much to move line_rect each time
    long start_addr, end_addr;          // Address of current selection
    bool neg_x(false), neg_y(false);    // Does map mode have -ve to left or down
    bool has_focus;                     // Does this window have focus?
    int bitspixel = pDC->GetDeviceCaps(BITSPIXEL);
    if (bitspixel > 24) bitspixel = 24;         // 32 is really only 24 bits of colour
    long num_colours = 1L << (bitspixel*pDC->GetDeviceCaps(PLANES));

    ASSERT(offset_ >= 0 && offset_ < rowsize_);
    ASSERT(rowsize_ > 0 && rowsize_ <= max_buf);
    ASSERT(group_by_ > 0);

    if (printing)
    {
        // Work out "client" rect of a printer page
        line_rect.top = line_rect.left = 0;
        line_rect.bottom = pDC->GetDeviceCaps(VERTRES);
        line_rect.right  = pDC->GetDeviceCaps(HORZRES);

        // Convert to logical units but with origin at top left of window
        pDC->DPtoLP(&line_rect);
        if (line_rect.right < 0)
        {
            neg_x = true;
            line_rect.left  = -line_rect.left;
            line_rect.right = -line_rect.right;
        }
        if (line_rect.bottom < 0)
        {
            neg_y = true;
            line_rect.top    = -line_rect.top;
            line_rect.bottom = -line_rect.bottom;
        }

        // Work out which part of document to display (leave 2 blank lines at top)
        disp_rect = line_rect + CSize(0, (curpage_ * lines_per_page_ - 2) * print_text_height_);
    }
    else
    {
        has_focus = (GetFocus() == this);
        HideCaret();
        neg_x = negx(); neg_y = negy();

        // Get display rect in logical units but with origin at top left of window
        GetClientRect(&line_rect);
        ConvertFromDP(line_rect);

        // Display = client rectangle translated to posn in document
        disp_rect = line_rect;
        line_rect -= GetScroll();

        // Get the current selection so that we can display it in reverse video
        GetSelAddr(start_addr, end_addr);

        pDC->SetBkMode(TRANSPARENT);
    }

    // Work out which lines could possibly be in the display area
    if (printing && print_sel_)
    {
        GetSelAddr(first_addr, last_addr);

        // Start drawing from start of selection (+ allow for subsequent pages)
        first_line = (first_addr+offset_)/rowsize_ + curpage_*lines_per_page_;
        if ((last_line = first_line + lines_per_page_) > (last_addr - 1 + offset_)/rowsize_ + 1)
            last_line = (last_addr - 1 + offset_)/rowsize_ + 1;
        line_inc = 1L;
        rect_inc = CSize(0, print_text_height_);

        /* Work out where to display the 1st line */
        line_rect.top += 2 * print_text_height_;
        line_rect.bottom = line_rect.top + print_text_height_;
        line_rect.left -= disp_rect.left;
    }
    else if (printing)
    {
        // Draw just the lines on this page
        first_line = curpage_ * lines_per_page_;
        last_line = first_line + lines_per_page_;
        line_inc = 1L;
        rect_inc = CSize(0, print_text_height_);

        /* Work out where to display the 1st line */
//      line_rect.top -= (disp_rect.top - first_line*print_text_height_);
        line_rect.top += 2 * print_text_height_;
        line_rect.bottom = line_rect.top + print_text_height_;
        line_rect.left -= disp_rect.left;
    }
    else if (ScrollUp())
    {
        // Draw from bottom of window up since we're scrolling up (looks better)
        first_line = disp_rect.bottom/text_height_;
        last_line = disp_rect.top/text_height_ - 1;
        line_inc = -1L;
        rect_inc = CSize(0, -text_height_);

        /* Work out where to display the 1st line */
        line_rect.top -= (disp_rect.top - first_line*text_height_);
        line_rect.bottom = line_rect.top + text_height_;
        line_rect.left -= disp_rect.left;
    }
    else
    {
        // Draw from top of window down
        first_line = disp_rect.top/text_height_;
        last_line = disp_rect.bottom/text_height_ + 1;
        line_inc = 1L;
        rect_inc = CSize(0, text_height_);

        /* Work out where to display the 1st line */
        line_rect.top -= (disp_rect.top - first_line*text_height_);
        line_rect.bottom = line_rect.top + text_height_;
        line_rect.left -= disp_rect.left;
    }

    // Draw search string occurrences
    std::vector<FILE_ADDRESS>::const_iterator pp, pend;
    for (pp = search_found_.begin(), pend = search_found_.end(); pp != pend; ++pp)
    {
        FILE_ADDRESS start_addr = max(*pp, (disp_rect.top/text_height_)*rowsize_ - offset_);
        FILE_ADDRESS end_addr = min(*pp + search_length_,
                                    (disp_rect.bottom/text_height_ + 1)*rowsize_ - offset_);
        FILE_ADDRESS start_line = (start_addr + offset_)/rowsize_;
        FILE_ADDRESS end_line = (end_addr + offset_)/rowsize_;

        CRect rct;
        CBrush bb(RGB(128,255,128));      // Brush to draw the block (green)

        if (start_line == end_line)
        {
            // Draw the block (all on one line)
            rct.top = start_line * text_height_;
            rct.bottom = rct.top + text_height_;
            rct.left = hex_pos((start_addr+offset_)%rowsize_);
            rct.right = hex_pos((end_addr+offset_)%rowsize_ - 1) + 2*text_width_;
            ConvertToDP(rct);
            if (rct.left < rct.right)
                pDC->FillRect(&rct, &bb);

            if (display_char_)
            {
                rct.top = start_line * text_height_;
                rct.bottom = rct.top + text_height_;
                rct.left = char_pos((start_addr+offset_)%rowsize_);
                rct.right = char_pos((end_addr+offset_)%rowsize_);
                ConvertToDP(rct);
                pDC->FillRect(&rct, &bb);
            }

            continue;
        }

        // Block extends over (at least) 2 lines so draw the partial lines at each end
        rct.top = start_line * text_height_;
        rct.bottom = rct.top + text_height_;
        rct.left = hex_pos((start_addr+offset_)%rowsize_);
        rct.right = hex_pos(rowsize_ - 1) + 2*text_width_;
        ConvertToDP(rct);
        ASSERT(rct.left < rct.right);
        pDC->FillRect(&rct, &bb);

        if (display_char_)
        {
            rct.top = start_line * text_height_;
            rct.bottom = rct.top + text_height_;
            rct.left = char_pos((start_addr+offset_)%rowsize_);
            rct.right = char_pos(rowsize_);
            ConvertToDP(rct);
            pDC->FillRect(&rct, &bb);
        }

        rct.top = end_line * text_height_;
        rct.bottom = rct.top + text_height_;
        rct.left = hex_pos(0);
        rct.right = hex_pos((end_addr+offset_)%rowsize_ - 1) + 2*text_width_;
        ConvertToDP(rct);
        if (rct.left < rct.right)
            pDC->FillRect(&rct, &bb);

        if (display_char_)
        {
            rct.top = end_line * text_height_;
            rct.bottom = rct.top + text_height_;
            rct.left = char_pos(0);
            rct.right = char_pos((end_addr+offset_)%rowsize_);
            ConvertToDP(rct);
            pDC->FillRect(&rct, &bb);
        }

        // If more than one line between start and end then draw that block too
        if (start_line + 1 < end_line)
        {
            rct.top = (start_line + 1) * text_height_;
            rct.bottom = end_line * text_height_;
            rct.left = hex_pos(0);
            rct.right = hex_pos(rowsize_ - 1) + 2*text_width_;
            ConvertToDP(rct);
            ASSERT(rct.left < rct.right);
            pDC->FillRect(&rct, &bb);

            if (display_char_)
            {
                rct.top = (start_line + 1) * text_height_;
                rct.bottom = end_line * text_height_;
                rct.left = char_pos(0);
                rct.right = char_pos(rowsize_);
                ConvertToDP(rct);
                pDC->FillRect(&rct, &bb);
            }
        }
    }

    // Draw mark if within display area
    if (mark_ >= (disp_rect.top/text_height_)*rowsize_ - offset_ &&
        mark_ < (disp_rect.bottom/text_height_ + 1)*rowsize_ - offset_)
    {
        CRect mark_rect;                // Where mark is drawn in window coords
        CBrush bb(RGB(0,255,255));      // Cyan brush to draw the mark

        mark_rect.top = ((mark_ + offset_)/rowsize_) * text_height_ - disp_rect.top;
        mark_rect.bottom = mark_rect.top + text_height_;

        mark_rect.left = hex_pos((mark_ + offset_)%rowsize_) - disp_rect.left;
        mark_rect.right = mark_rect.left + 2*text_width_;

        pDC->FillRect(&mark_rect, &bb);

        if (display_char_)
        {
            mark_rect.left = char_pos((mark_ + offset_)%rowsize_) - disp_rect.left;
            mark_rect.right = mark_rect.left + text_width_;
            pDC->FillRect(&mark_rect, &bb);
        }
    }

    for (long line = first_line; line != last_line;
                            line += line_inc, line_rect += rect_inc)
    {
        // Work out where to display line in logical coords (correct sign)
        CRect tt(line_rect);
        if (neg_x)
        {
            tt.left = -tt.left;
            tt.right = -tt.right;
        }
        if (neg_y)
        {
            tt.top = -tt.top;
            tt.bottom = -tt.bottom;
        }

        // No display needed if outside display area or past end of doc
        if (!pDC->RectVisible(&tt) || line*rowsize_ - offset_ > pDoc->length())
            continue;

        CString ss;                     // Temp string for formatting

        // Draw the address into the window
        ss.Format(addr_format_, line*rowsize_ - offset_ > first_addr ?
                                line*rowsize_ - offset_ : first_addr);
        if (printing)
            pDC->SetTextColor(RGB(0,0,0));      // Draw addresses in black on printer
        else
            pDC->SetTextColor(RGB(127,127,127)); // Draw addresses in grey on screen
        if (aa->nice_addr_)
        {
            if (dec_addr_)
            {
                CRect addr_rect(tt);            // tt with right margin where addresses end

                addr_rect.right = addr_rect.left + (addr_width_ - 1)*text_width_;
                AddCommas(ss);
                pDC->DrawText(ss, &addr_rect, DT_TOP | DT_RIGHT | DT_NOPREFIX | DT_SINGLELINE);
            }
            else
            {
                AddSpaces(ss);
                pDC->DrawText(ss, &tt, DT_TOP | DT_NOPREFIX | DT_SINGLELINE);
            }
        }
        else
        {
            pDC->DrawText(ss, &tt, DT_TOP | DT_NOPREFIX | DT_SINGLELINE);
        }

        // Get the bytes to display
        size_t ii;                      // Column of first byte
        size_t last_col;

        if (line*rowsize_ - offset_ < first_addr)
        {
            last_col = pDoc->GetData(buf_ + offset_, rowsize_ - offset_, line*rowsize_) +
                        offset_;
            ii = first_addr - (line*rowsize_ - offset_);
            ASSERT(ii >= 0 && ii <rowsize_);
        }
        else
        {
            last_col = pDoc->GetData(buf_, rowsize_, line*rowsize_ - offset_);
            ii = 0;
        }

        if (line*rowsize_ - offset_ + last_col > last_addr)
        {
            ASSERT(line*rowsize_ - offset_ + last_col - last_addr < rowsize_);
            last_col = last_addr - (line*rowsize_ - offset_);
        }

        // Display each byte as hex (and char if nec.)
        for ( ; ii < last_col; ++ii)
        {
            char hh[2];
            int posx, posc;

            if (printing)
            {
                posx = tt.left + hex_pos(ii, print_text_width_);
                posc = tt.left + char_pos(ii, print_text_width_);
            }
            else
            {
                posx = tt.left + hex_pos(ii);
                posc = tt.left + char_pos(ii);
            }
            
            // Create hex digits and display them
            hh[0] = hex[(buf_[ii]>>4)&0xF];
            hh[1] = hex[buf_[ii]&0xF];

            // This actually displays the bytes (in hex)!
            pDC->SetTextColor(kala[buf_[ii]]);
            pDC->TextOut(posx, tt.top, hh, 2);

            // Display byte in char display area (as ASCII, EBCDIC etc)
            if (display_char_ && !ebcdic_)
            {
                if ((buf_[ii] >= 32 && buf_[ii] < 127) ||
                    (graphic_ && buf_[ii] >= first_char_ && buf_[ii] <= last_char_) )
                {
                    // Display normal char or graphic char if in font
//                    pDC->SetTextColor(RGB(0,0,0));
                    pDC->TextOut(posc, tt.top, (char *)&buf_[ii], 1);
                }
                else if (display_control_ == 0 || buf_[ii] > 31)
                {
                    // Display control char and other chars as red '.'
//                    pDC->SetTextColor(RGB(255,0,0));
                    pDC->TextOut(posc, tt.top, ".", 1);
                }
                else if (display_control_ == 1)
                {
                    // Display control chars as red uppercase equiv.
                    char cc = buf_[ii] + 0x40;
//                    pDC->SetTextColor(RGB(255,0,0));
                    pDC->TextOut(posc, tt.top, &cc, 1);
                }
                else if (display_control_ == 2)
                {
                    // Display control chars as C escape code (in red)
                    const char *check = "\a\b\f\n\r\t\v\0";
                    const char *display = "abfnrtv0";
                    const char *pp;
                    if (/*buf_[ii] != '\0' && */(pp = strchr(check, buf_[ii])) != NULL)
                        pp = display + (pp-check);
                    else
                        pp = ".";
//                    pDC->SetTextColor(RGB(255,0,0));
                    pDC->TextOut(posc, tt.top, pp, 1);
                }
            }
            else if (display_char_)
            {
                // Display EBCDIC (or red dot if not valid EBCDIC char)
                if (e2a_tab[buf_[ii]] == '\0')
                {
//                    pDC->SetTextColor(RGB(255,0,0));
                    pDC->TextOut(posc, tt.top, ".", 1);
                }
                else
                {
//                    pDC->SetTextColor(RGB(0,0,0));
                    pDC->TextOut(posc, tt.top, (char *)&e2a_tab[buf_[ii]], 1);
                }
            }
        }

        // If any part of the line is within the current selection
        if (!printing && start_addr < end_addr && 
            end_addr > line*rowsize_ - offset_ && start_addr < (line+1)*rowsize_ - offset_)
        {
            long start = max(start_addr, line*rowsize_ - offset_);
            long end   = min(end_addr, (line+1)*rowsize_ - offset_);
            ASSERT(end > start);

            CRect rev(line_rect);
            rev.right = rev.left + hex_pos(end - (line*rowsize_ - offset_) - 1) + 2*text_width_;
            rev.left += hex_pos(start - (line*rowsize_ - offset_));
            if (neg_x)
            {
                rev.left = -rev.left;
                rev.right = -rev.right;
            }
            if (neg_y)
            {
                rev.top = -rev.top;
                rev.bottom = -rev.bottom;
            }
            if (has_focus && !edit_char_ || num_colours <= 256)
                pDC->InvertRect(&rev);
            else
                pDC->PatBlt(rev.left, rev.top,
                            rev.right-rev.left, rev.bottom-rev.top,
                            PATINVERT);

            if (display_char_)
            {
                // Draw char selection in reverse
                rev = line_rect;
                rev.right = rev.left + char_pos(end - (line*rowsize_ - offset_) - 1) + text_width_;
                rev.left += char_pos(start - (line*rowsize_ - offset_));
                if (neg_x)
                {
                    rev.left = -rev.left;
                    rev.right = -rev.right;
                }
                if (neg_y)
                {
                    rev.top = -rev.top;
                    rev.bottom = -rev.bottom;
                }
                if (has_focus && edit_char_ || num_colours <= 256)
                    pDC->InvertRect(&rev);
                else
                    pDC->PatBlt(rev.left, rev.top,
                                rev.right-rev.left, rev.bottom-rev.top,
                                PATINVERT);
            }
        }
    } /* for */

    if (!printing)
    {
        ShowCaret();
    }
    move_dlgs();
}

// recalc_display() - recalculates everything to do with the display
// (and redraws it) if anything about how the window is drawn changes.
// This includes font changed, window resized, document changed, display
// options changed (address display, char display, autofit turned on etc).
void CHexEditView::recalc_display()
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    // Save info on the current font
    {
        TEXTMETRIC tm;
        CClientDC dc(this);
        OnPrepareDC(&dc);
        dc.GetTextMetrics(&tm);
        text_height_ = tm.tmHeight + tm.tmExternalLeading; // Height of font
        first_char_ = tm.tmFirstChar;           // 1st valid char of font
        last_char_ = tm.tmLastChar;                     // last valid char of font
//      text_width_ = tm.tmMaxCharWidth;                // width of widest char in font
        CSize size;
        ::GetTextExtentPoint32(dc.m_hDC, "W", 1, &size);
        text_width_ = size.cx;                  // width of "W"
    }

    long length = GetDocument()->length();

    // Work out how much room we need to display addresses on left of window
    if (dec_addr_)
    {
        if (length < 10000)
        {
            addr_width_ = 7;
            addr_format_ = "%4ld:";
        }
        else if (length < 100000)
        {
            addr_width_ = 8;
            addr_format_ = "%5ld:";
        }
        else if (length < 1000000)
        {
            addr_width_ = 9;
            addr_format_ = "%6ld:";
        }
        else if (length < 10000000)
        {
            addr_width_ = 11;
            addr_format_ = "%7ld:";
        }
        else if (length < 100000000)
        {
            addr_width_ = 12;
            addr_format_ = "%8ld:";
        }
        else if (length < 1000000000)
        {
            addr_width_ = 13;
            addr_format_ = "%9ld:";
        }
        else
        {
            addr_width_ = 15;
            addr_format_ = "%10ld:";
        }
    }
    else if(aa->hex_ucase_)
    {
        if (length < 0x10000)
        {
            addr_width_ = 6;
            addr_format_ = "%04lX:";
        }
//      else if (length < 0x100000)
//      {
//          addr_width_ = 8;
//          addr_format_ = "%05lX:";
//      }
        else if (length < 0x1000000)
        {
            addr_width_ = 9;
            addr_format_ = "%06lX:";
        }
//      else if (length < 0x10000000)
//      {
//          addr_width_ = 10;
//          addr_format_ = "%07lX:";
//      }
        else
        {
            addr_width_ = 11;
            addr_format_ = "%08lX:";
        }
    }
    else
    {
        if (length < 0x10000)
        {
            addr_width_ = 6;
            addr_format_ = "%04lx:";
        }
//      else if (length < 0x100000)
//      {
//          addr_width_ = 8;
//          addr_format_ = "%05lx:";
//      }
        else if (length < 0x1000000)
        {
            addr_width_ = 9;
            addr_format_ = "%06lx:";
        }
//      else if (length < 0x10000000)
//      {
//          addr_width_ = 10;
//          addr_format_ = "%07lx:";
//      }
        else
        {
            addr_width_ = 11;
            addr_format_ = "%08lx:";
        }
    }

    // Fit columns to window width?
    if (autofit_)
    {
        CRect cli;              /* Client rectangle */
        GetClientRect(&cli);
        ConvertFromDP(cli);
        cli -= GetScroll();
        ASSERT(cli.left == 0 && cli.top == 0);

        // Work out the scroll pos (address of byte in top left of display)
        CPoint pp = GetScroll();
        long topleft = (pp.y/text_height_) * rowsize_;

        // Work out the current address of the caret/selection
//      long address = pos2addr(GetCaret());
        long start_addr, end_addr;
        GetSelAddr(start_addr, end_addr);

        // Work out how many columns we can display across the window
        // NOTE: These calcs are directly related to the calcs in hex_pos and char_pos
        if (display_char_)
            rowsize_ = (cli.right - addr_width_*text_width_ - (cli.right - addr_width_*text_width_)/(4*group_by_))/
                        (4*text_width_);
        else
            rowsize_ = (cli.right - addr_width_*text_width_ - (cli.right - addr_width_*text_width_)/(3*group_by_))/
                        (3*text_width_);

//      if (display_char_)
//          rowsize_ = (cli.right - addr_width_*text_width_ - (cli.right - addr_width_*text_width_)/16)/
//                      (4*text_width_);
//      else
//          rowsize_ = (cli.right - addr_width_*text_width_ - (cli.right - addr_width_*text_width_)/12)/
//                      (3*text_width_);

        // Must display at least 4 columns & no more than buffer can hold
        if (rowsize_ < 4)
            rowsize_ = 4;
        else if (rowsize_ > max_buf)
            rowsize_ = max_buf;
        if (real_offset_ >= rowsize_)
            offset_ = rowsize_ - 1;
        else
            offset_ = real_offset_;

        // Set size before setting scroll/caret to avoid them being moved to "valid" pos
        SetTSize(CSize(-1, (GetDocument()->length() + offset_)/rowsize_ + 1));

        // Move the scroll position to be roughly the same place
        SetScroll(CPoint(0, topleft/rowsize_ * text_height_));

        // Move the caret/selection to the where the same byte(s) as before
//      SetCaret(addr2pos(address));
        SetSel(addr2pos(start_addr), addr2pos(end_addr));
    }
    else
        SetTSize(CSize(-1, (GetDocument()->length() + offset_)/rowsize_ + 1));

    // Make sure we know the width of the display area
    if (display_char_)
        SetSize(CSize(char_pos(rowsize_), -1));
    else
        SetSize(CSize(hex_pos(rowsize_), -1));
} /* recalc_display() */

// Return display position given a hex area column number
// int CHexEditView::hex_pos(int column, int width) const;

// Return closest hex area column given y display position
// If inside==TRUE then will return a value from 0 to rowsize_ - 1
// else may return rowsize to indicate being past last column.
int CHexEditView::pos_hex(int pos, BOOL inside) const
{
    int col = pos - addr_width_*text_width_;
    col -= (col/(text_width_*(group_by_*3+1)))*text_width_;
    col = col/(3*text_width_);

    // Make sure col is within valid range
    if (col < 0) col = 0;
    else if (inside && col >= rowsize_) col = rowsize_ - 1;
    else if (col > rowsize_) col = rowsize_;

    return col;
}

// Return display position given a char area column number
// int CHexEditView::char_pos(int column, int width /* = 0 */) const;

// Return closest char area column given display (Y) coord
// If inside==TRUE then will return a value from 0 to rowsize_ - 1
// else may return rowsize_ to indicate being past the last column.
int CHexEditView::pos_char(int pos, BOOL inside) const
{
    int col =  (pos - addr_width_*text_width_ - rowsize_*3*text_width_ -
                (rowsize_/group_by_)*text_width_) / text_width_;

    // Make sure col is within valid range
    if (col < 0) col = 0;
    else if (inside && col >= rowsize_) col = rowsize_ - 1;
    else if (col > rowsize_) col = rowsize_;

    return col;
}

void CHexEditView::DoInvalidate()
{
    if (((CHexEditApp *)AfxGetApp())->refresh_off_)
        needs_refresh_ = true;
    else
        CScrView::DoInvalidate();
}

void CHexEditView::DoInvalidateRect(LPCRECT lpRect)
{
    if (((CHexEditApp *)AfxGetApp())->refresh_off_)
        needs_refresh_ = true;
    else
        CScrView::DoInvalidateRect(lpRect);
}

void CHexEditView::DoInvalidateRgn(CRgn* pRgn)
{
    if (((CHexEditApp *)AfxGetApp())->refresh_off_)
        needs_refresh_ = true;
    else
        CScrView::DoInvalidateRgn(pRgn);
}

void CHexEditView::DoScrollWindow(int xx, int yy)
{
    if (((CHexEditApp *)AfxGetApp())->refresh_off_)
        needs_refresh_ = true;
    else
        CScrView::DoScrollWindow(xx, yy);
}

void CHexEditView::DoUpdateWindow()
{
    if (!((CHexEditApp *)AfxGetApp())->refresh_off_)
        CScrView::DoUpdateWindow();
}

void CHexEditView::DoHScroll(int total, int page, int pos)
{
    if (((CHexEditApp *)AfxGetApp())->refresh_off_)
    {
        needs_hscroll_ = true;
        h_total_ = total;
        h_page_ = page;
        h_pos_ = pos;
    }
    else
        CScrView::DoHScroll(total, page, pos);
}

void CHexEditView::DoVScroll(int total, int page, int pos)
{
    if (((CHexEditApp *)AfxGetApp())->refresh_off_)
    {
        needs_vscroll_ = true;
        v_total_ = total;
        v_page_ = page;
        v_pos_ = pos;
    }
    else
        CScrView::DoVScroll(total, page, pos);
}

void CHexEditView::DoUpdate()
{
    if (needs_refresh_)
    {
        CScrView::DoInvalidate();
        needs_refresh_ = false;
        CScrView::DoUpdateWindow();
    }

    if (needs_hscroll_)
    {
        CScrView::DoHScroll(h_total_, h_page_, h_pos_);
        needs_hscroll_ = false;
    }

    if (needs_vscroll_)
    {
        CScrView::DoVScroll(v_total_, v_page_, v_pos_);
        needs_vscroll_ = false;
    }
}

void CHexEditView::InvalidateRange(CPoint start, CPoint end)
{
    if (start == end) return;

#if 0 // not needed since handled by invalidate_range call below
    if (((CHexEditApp *)AfxGetApp())->refresh_off_)
    {
        needs_refresh_ = true;
        return;
    }
#endif

    BOOL saved_edit_char = edit_char_;          // Saved value of edit_char_
    long start_addr, end_addr;                  // Range of addresses to invalidate

    // Work out what we are invalidating (hex or char area)
    if (display_char_ && pos_hex(start.x) == rowsize_ && pos_hex(end.x) == rowsize_)
        edit_char_ = TRUE;                      // Change edit_char_ so pos2addr() works
    else
        edit_char_ = FALSE;

    // Work out range to invalidate
    start_addr = pos2addr(start);
    end_addr = pos2addr(end);

    edit_char_ = saved_edit_char;               // Restore edit_char_

    // Note: We need to invalidate an extra char backwards because in the hex
    // display area the white space to the right of the last byte selected is
    // not shown in reverse video.  When the selection is extended towards the
    // end of file (causing InvalidateRange to be called) not only the newly
    // selected bytes need to be invalidated but also the previous one so that
    // the white area after the character is then drawn in reverse video.
    invalidate_addr_range(start_addr-1, end_addr);
}

void CHexEditView::invalidate_addr_range(long start_addr, long end_addr)
{
#if 0 // not needed since handled by DoInvalidateRect calls below
    if (((CHexEditApp *)AfxGetApp())->refresh_off_)
    {
        needs_refresh_ = true;
        return;
    }
#endif

    CRect inv;                          // The rectangle to actually invalidate
    CRect disp_rect;                    // Rectangle of display in our coords
    GetClientRect(&disp_rect);
    ConvertFromDP(disp_rect);

    // Work out the addresses of the first and (one past) the last byte in display
    long start_disp = (disp_rect.top/text_height_)*rowsize_ - offset_;
    long end_disp = (disp_rect.bottom/text_height_ + 1)*rowsize_ - offset_;

    // Reduce address range to relevant (displayed) area
    if (start_addr < start_disp) start_addr = start_disp;
    if (end_addr > end_disp) end_addr = end_disp;

    if (start_addr >= end_addr)
        return;                         // Nothing to invalidate or all outside display

    // Work out line range that needs invalidating
    long start_line = (start_addr + offset_)/rowsize_;
    long end_line = (end_addr + offset_)/rowsize_;

    // If start and end on the same line just invalidate between them
    if (start_line == end_line)
    {
        inv = CRect(hex_pos((start_addr+offset_)%rowsize_), start_line * text_height_,
                    hex_pos((end_addr+offset_)%rowsize_), (start_line + 1) * text_height_);
        ConvertToDP(inv);
        DoInvalidateRect(&inv);
        if (display_char_)
        {
            inv = CRect(char_pos((start_addr+offset_)%rowsize_), start_line * text_height_,
                        char_pos((end_addr+offset_)%rowsize_), (start_line + 1) * text_height_);
            ConvertToDP(inv);
            DoInvalidateRect(&inv);
        }
        return;
    }

    // Start line before end line so invalidate partial lines at each end
    inv = CRect(hex_pos((start_addr+offset_)%rowsize_), start_line * text_height_,
                hex_pos(rowsize_), (start_line + 1) * text_height_);
    ConvertToDP(inv);
    DoInvalidateRect(&inv);
    if (display_char_)
    {
        inv = CRect(char_pos((start_addr+offset_)%rowsize_), start_line * text_height_,
                    char_pos(rowsize_), (start_line + 1) * text_height_);
        ConvertToDP(inv);
        DoInvalidateRect(&inv);
    }

    inv = CRect(hex_pos(0), end_line * text_height_,
                hex_pos((end_addr+offset_)%rowsize_), (end_line + 1) * text_height_);
    ConvertToDP(inv);
    DoInvalidateRect(&inv);
    if (display_char_)
    {
        inv = CRect(char_pos(0), end_line * text_height_,
                    char_pos((end_addr+offset_)%rowsize_), (end_line + 1) * text_height_);
        ConvertToDP(inv);
        DoInvalidateRect(&inv);
    }

    // If more than one line between start and end then invalidate that block too
    if (start_line + 1 < end_line)
    {
        inv = CRect(hex_pos(0), (start_line + 1) * text_height_,
                    hex_pos(rowsize_), (end_line) * text_height_);
        ConvertToDP(inv);
        DoInvalidateRect(&inv);
        if (display_char_)
        {
            inv = CRect(char_pos(0), (start_line + 1) * text_height_,
                        char_pos(rowsize_), (end_line) * text_height_);
            ConvertToDP(inv);
            DoInvalidateRect(&inv);
        }
    }
}

// Override CScrView::ValidateScroll()
void CHexEditView::ValidateScroll(CPoint &pos, BOOL strict /* =FALSE */)
{
    CScrView::ValidateScroll(pos, strict);

    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (aa->bg_search_ && aa->pboyer_ != NULL)
    {
        CHexEditDoc *pdoc = GetDocument();
        CRect rct;
        GetClientRect(&rct);
        ConvertFromDP(rct);
        FILE_ADDRESS start, end;            // range of file addresses within display

        // Get all occurrences that are within the display - this includes those
        // that have an address before the start of display but extend into it.
        start = (pos.y/text_height_)*rowsize_ - offset_     // First addr in display
                - (aa->pboyer_->length() - 1);              // Length of current search string - 1
        if (start < 0) start = 0;                           // Just in case (prob not nec.)
        end = ((pos.y+rct.Height())/text_height_ + 1)*rowsize_ - offset_;
        search_found_ = pdoc->SearchAddresses(start, end);
        search_length_ = aa->pboyer_->length();
    }
    else
        search_found_.clear();
}

// Override CScrView::ValidateCaret()
void CHexEditView::ValidateCaret(CPoint &pos, BOOL inside /*=true*/)
{
    // Ensure pos is a valid caret position or move it to the closest such one
    long address = pos2addr(pos, inside);
    if (address < 0)
        address = 0;
    else if (address > GetDocument()->length())
        address = GetDocument()->length();
    pos = addr2pos(address);
}

void CHexEditView::DisplayCaret()
{
    CScrView::DisplayCaret();
    move_dlgs();
}

// Move the modeless dialogs if visible and it/they obscure the caret
void CHexEditView::move_dlgs()
{
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
    if (!CaretDisplayed() ||
        (mm->pfind_ == NULL && mm->pprop_ == NULL && !mm->pcalc_->visible_))
    {
        return;                         // No sel or no dlg
    }

    // First get the rectangle containing the selection (in entire doc)
    CSize tt, pp, ll;                   // Size of document total,page,line
    CPoint start, end;                  // Points of start/end of selection
    CRect selrect;                      // Rect enclosing selection + a bit
    CRect c_rect = GetCaretRect();      // Rect enclosing caret
    CSize caret_size(c_rect.right - c_rect.left, c_rect.bottom - c_rect.top);

    GetSize(tt, pp, ll);
    GetSel(start, end);
    if (start.y == end.y)
        selrect = CRect(start.x - caret_size.cx, start.y - caret_size.cy,
                        end.x + 2*caret_size.cx, end.y + 2*caret_size.cy);
    else
        selrect = CRect(0, start.y - caret_size.cy, tt.cx,
                        end.y + 2*caret_size.cy);

    // Get display rectangle (also in doc coords)
    CRect cli;
    GetClientRect(&cli);
    ConvertFromDP(cli);

    // See if any of the selection is visible (intersect selection & display)
    CRect sd_rect;                          // selection within display
    if (!sd_rect.IntersectRect(&cli, selrect))
        return;                             // Selection not in display

    // Convert result (selection visible in display) to device coords
    HideCaret();
    ConvertToDP(sd_rect);
    ClientToScreen(&sd_rect);

    // Get rectangle of find dialogue and property sheet
    CRect find_rect(0,0,0,0), prop_rect(0,0,0,0), calc_rect(0,0,0,0);
    if (mm->pfind_ != NULL)  mm->pfind_->GetWindowRect(&find_rect);
    if (mm->pprop_ != NULL)  mm->pprop_->GetWindowRect(&prop_rect);
    if (mm->pcalc_->visible_)mm->pcalc_->GetWindowRect(&calc_rect);

    CRect intersect_rect;               // Intersection of dialog & selection
    if (mm->pfind_ != NULL && intersect_rect.IntersectRect(&find_rect, &sd_rect))
    {
        // Find dialog overlaps selection so move it (the dialog not the selection)
        CRect new_rect(find_rect);      // New position of find dialog
        int scr_height;
        {
            CClientDC dc(this);
            OnPrepareDC(&dc);
            scr_height = dc.GetDeviceCaps(VERTRES);
        }

        // Move up or down depending on whether middle of selection
        // is above middle of dialogue?
        if (sd_rect.top + sd_rect.bottom < new_rect.top + new_rect.bottom)
        {
            new_rect.OffsetRect(0, sd_rect.bottom - new_rect.top);
            // Is find dialog now over property or calc dialog?
            if (intersect_rect.IntersectRect(&new_rect, prop_rect))
                new_rect.OffsetRect(0, prop_rect.bottom - new_rect.top + 1);
            if (intersect_rect.IntersectRect(&new_rect, calc_rect))
                new_rect.OffsetRect(0, calc_rect.bottom - new_rect.top + 1);
        }
        else
        {
            new_rect.OffsetRect(0, sd_rect.top - new_rect.bottom);
            // Is find dialog now over property or calc dialog?
            if (intersect_rect.IntersectRect(&new_rect, prop_rect))
                new_rect.OffsetRect(0, prop_rect.top - new_rect.bottom - 1);
            if (intersect_rect.IntersectRect(&new_rect, calc_rect))
                new_rect.OffsetRect(0, calc_rect.top - new_rect.bottom - 1);
        }

        // If dialogue is now below the bottom of screen
        if (new_rect.bottom > scr_height - 40)
            new_rect.OffsetRect(0, sd_rect.top - new_rect.bottom);
        if (new_rect.top < 40)
            new_rect.OffsetRect(0, sd_rect.bottom - new_rect.top);

        // If it now (or again) overlaps prop dlg move it right
        if (intersect_rect.IntersectRect(&new_rect, prop_rect))
        {
            if (prop_rect.right + prop_rect.left < new_rect.right + new_rect.left)
                new_rect.OffsetRect(prop_rect.right - new_rect.left, 0);
            else
                new_rect.OffsetRect(prop_rect.left - new_rect.right, 0);
        }
        // If it now (or again) overlaps calc dlg move it right
        if (intersect_rect.IntersectRect(&new_rect, calc_rect))
        {
            if (calc_rect.right + calc_rect.left < new_rect.right + new_rect.left)
                new_rect.OffsetRect(calc_rect.right - new_rect.left, 0);
            else
                new_rect.OffsetRect(calc_rect.left - new_rect.right, 0);
        }

        // Now move it where we (finally) decided
        mm->pfind_->MoveWindow(&new_rect);

        // If the mouse ptr (cursor) was over the dialog move it too
        CPoint mouse_pt;
        ::GetCursorPos(&mouse_pt);
        if (find_rect.PtInRect(mouse_pt))
        {
            // Mouse was over find dialog (probably over "Find Next" button)
            mouse_pt.x += new_rect.left - find_rect.left;
            mouse_pt.y += new_rect.top - find_rect.top;
            ::SetCursorPos(mouse_pt.x, mouse_pt.y);
        }
    }

    if (mm->pprop_ != NULL && intersect_rect.IntersectRect(&prop_rect, &sd_rect))
    {
        // Prop sheet overlaps selection so move it (the dialog not the selection)

        CRect new_rect(prop_rect);      // New position of prop dialog
        int scr_height;
        {
            CClientDC dc(this);
            OnPrepareDC(&dc);
            scr_height = dc.GetDeviceCaps(VERTRES);
        }

        // Move up or down depending on whether middle of selection
        // is above middle of dialogue?
        if (sd_rect.top + sd_rect.bottom < new_rect.top + new_rect.bottom)
        {
            new_rect.OffsetRect(0, sd_rect.bottom - new_rect.top);
            // Is prop dialog now over find or calc dialog?
            if (intersect_rect.IntersectRect(&new_rect, find_rect))
                new_rect.OffsetRect(0, find_rect.bottom - new_rect.top + 1);
            if (intersect_rect.IntersectRect(&new_rect, calc_rect))
                new_rect.OffsetRect(0, calc_rect.bottom - new_rect.top + 1);
        }
        else
        {
            new_rect.OffsetRect(0, sd_rect.top - new_rect.bottom);
            // Is prop dialog now over find or calc dialog?
            if (intersect_rect.IntersectRect(&new_rect, find_rect))
                new_rect.OffsetRect(0, find_rect.top - new_rect.bottom - 1);
            if (intersect_rect.IntersectRect(&new_rect, calc_rect))
                new_rect.OffsetRect(0, calc_rect.top - new_rect.bottom - 1);
        }

        // If dialogue is now below the bottom of screen
        if (new_rect.bottom > scr_height - 40)
            new_rect.OffsetRect(0, sd_rect.top - new_rect.bottom);
        if (new_rect.top < 40)
            new_rect.OffsetRect(0, sd_rect.bottom - new_rect.top);

        // If it now (or again) overlaps find dlg move it right
        if (intersect_rect.IntersectRect(&new_rect, find_rect))
        {
            if (find_rect.right + find_rect.left < new_rect.right + new_rect.left)
                new_rect.OffsetRect(find_rect.right - new_rect.left, 0);
            else
                new_rect.OffsetRect(find_rect.left - new_rect.right, 0);
        }

        // If it now (or again) overlaps calc dlg move it right
        if (intersect_rect.IntersectRect(&new_rect, calc_rect))
        {
            if (calc_rect.right + calc_rect.left < new_rect.right + new_rect.left)
                new_rect.OffsetRect(calc_rect.right - new_rect.left, 0);
            else
                new_rect.OffsetRect(calc_rect.left - new_rect.right, 0);
        }

        // Now move it where we (finally) decided it should go
        mm->pprop_->MoveWindow(&new_rect);
        mm->pprop_->UpdateWindow(); // This avoids getting 2 dlg images in macros

        // If the mouse ptr (cursor) was over the dialog move it too
        CPoint mouse_pt;
        ::GetCursorPos(&mouse_pt);
        if (prop_rect.PtInRect(mouse_pt))
        {
            // Mouse was over prop dialog
            mouse_pt.x += new_rect.left - prop_rect.left;
            mouse_pt.y += new_rect.top - prop_rect.top;
            ::SetCursorPos(mouse_pt.x, mouse_pt.y);
        }
    }

    if (intersect_rect.IntersectRect(&calc_rect, &sd_rect))
    {
        // Calc dlg overlaps selection so move it (the dialog not the selection)

        CRect new_rect(calc_rect);      // New position of calc dialog
        int scr_height;
        {
            CClientDC dc(this);
            OnPrepareDC(&dc);
            scr_height = dc.GetDeviceCaps(VERTRES);
        }

        // Move up or down depending on whether middle of selection
        // is above middle of dialogue?
        if (sd_rect.top + sd_rect.bottom < new_rect.top + new_rect.bottom)
        {
            new_rect.OffsetRect(0, sd_rect.bottom - new_rect.top);
            // Is calc dialog now over find or prop dialog?
            if (intersect_rect.IntersectRect(&new_rect, find_rect))
                new_rect.OffsetRect(0, find_rect.bottom - new_rect.top + 1);
            if (intersect_rect.IntersectRect(&new_rect, prop_rect))
                new_rect.OffsetRect(0, prop_rect.bottom - new_rect.top + 1);
        }
        else
        {
            new_rect.OffsetRect(0, sd_rect.top - new_rect.bottom);
            // Is calc dialog now over find or prop dialog?
            if (intersect_rect.IntersectRect(&new_rect, find_rect))
                new_rect.OffsetRect(0, find_rect.top - new_rect.bottom - 1);
            if (intersect_rect.IntersectRect(&new_rect, prop_rect))
                new_rect.OffsetRect(0, prop_rect.top - new_rect.bottom - 1);
        }

        // If dialogue is now below the bottom of screen
        if (new_rect.bottom > scr_height - 40)
            new_rect.OffsetRect(0, sd_rect.top - new_rect.bottom);
        if (new_rect.top < 40)
            new_rect.OffsetRect(0, sd_rect.bottom - new_rect.top);

        // If it now (or again) overlaps find dlg move it right
        if (intersect_rect.IntersectRect(&new_rect, find_rect))
        {
            if (find_rect.right + find_rect.left < new_rect.right + new_rect.left)
                new_rect.OffsetRect(find_rect.right - new_rect.left, 0);
            else
                new_rect.OffsetRect(find_rect.left - new_rect.right, 0);
        }

        // If it now (or again) overlaps prop dlg move it right
        if (intersect_rect.IntersectRect(&new_rect, prop_rect))
        {
            if (prop_rect.right + prop_rect.left < new_rect.right + new_rect.left)
                new_rect.OffsetRect(prop_rect.right - new_rect.left, 0);
            else
                new_rect.OffsetRect(prop_rect.left - new_rect.right, 0);
        }

        // Now move it where we (finally) decided it should go
        mm->pcalc_->MoveWindow(&new_rect);
        mm->pcalc_->UpdateWindow(); // This avoids getting 2 dlg images in macros

        // If the mouse ptr (cursor) was over the dialog move it too
        CPoint mouse_pt;
        ::GetCursorPos(&mouse_pt);
        if (calc_rect.PtInRect(mouse_pt))
        {
            // Mouse was over calc dialog
            mouse_pt.x += new_rect.left - calc_rect.left;
            mouse_pt.y += new_rect.top - calc_rect.top;
            ::SetCursorPos(mouse_pt.x, mouse_pt.y);
        }
    }
    ShowCaret();
}

// Move scroll or caret position in response to a key press.
// Note that this overrides CScrView::MovePos().
BOOL CHexEditView::MovePos(UINT nChar, UINT nRepCnt,
                         BOOL control_down, BOOL shift_down, BOOL caret_on)
{
    // CScrView::MovePos scrolling behaviour is OK
    if (!caret_on)
        return CScrView::MovePos(nChar, nRepCnt, control_down, shift_down, caret_on);

    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (aa->recording_)
    {
        long vv = nChar;
        if (control_down) vv |= 0x10000;
        if (shift_down)   vv |= 0x20000;
        for (int ii = 0; ii < nRepCnt; ++ii)
            aa->SaveToMacro(km_key, vv);
    }

    long start_addr, end_addr;
    BOOL end_base = GetSelAddr(start_addr, end_addr);   // Is selection base at end of selection?
    long new_address;

    // Start with start of (or end of, if moving forwards) current selection
    if (shift_down)
    {
        // Work out which end of selection is being extended
        if (end_base)
            new_address = start_addr;
        else
            new_address = end_addr;
    }
    else if (start_addr == end_addr )
        new_address = start_addr;                       // No current selection
    else if (nChar == VK_DOWN || nChar == VK_NEXT)
        new_address = end_addr;                         // Move from char after selection
    else if (nChar == VK_RIGHT || nChar == VK_END)
        new_address = end_addr - 1;                     // Move from last char of selection
    else
        new_address = start_addr;                       // Move from start of selection

    CSize tt, pp, ll;                   // Size of document total,page,line

    switch (nChar)
    {
    case VK_LEFT:
        if (control_down)
        {
            // Work out how many groups there are to start of file
            long gpr = (rowsize_ - 1)/group_by_ + 1;    // groups per row
            long groups = ((new_address+offset_)/rowsize_) * gpr +
                          ((new_address+offset_)%rowsize_ + group_by_ - 1)/group_by_;
            // Calculate the group to move to and address of 1st byte
            groups -= nRepCnt;
            new_address = (groups/gpr) * rowsize_ - offset_ + (groups%gpr) * group_by_;
        }
        else
            new_address -= nRepCnt;
        break;
    case VK_RIGHT:
        if (control_down)
        {
            // First work out how many groups there are to start of file
            long gpr = (rowsize_ - 1)/group_by_ + 1;    // groups per row
            long groups = ((new_address+offset_)/rowsize_) * gpr +
                          ((new_address+offset_)%rowsize_)/group_by_;
            // Calculate the group to move to
            groups += nRepCnt;
            new_address = (groups/gpr) * rowsize_ - offset_ + (groups%gpr) * group_by_;
        }
        else
            new_address += nRepCnt;
        break;
    case VK_UP:
        new_address -= rowsize_ * nRepCnt;
        break;
    case VK_DOWN:
        new_address += rowsize_ * nRepCnt;
        break;
    case VK_HOME:
        if (control_down)
            new_address = 0;
        else
            new_address = ((new_address+offset_)/rowsize_) * rowsize_ - offset_;
        break;
    case VK_END:
        if (control_down)
            new_address = GetDocument()->length();
        else
            new_address = ((new_address+offset_)/rowsize_ + 1) * rowsize_ - offset_ - 
                                (shift_down ? 0 : 1);
        break;
    case VK_PRIOR:
        GetSize(tt, pp, ll);
        new_address -= rowsize_ * (pp.cy/text_height_) * nRepCnt;
        break;
    case VK_NEXT:
        GetSize(tt, pp, ll);
        new_address += rowsize_ * (pp.cy/text_height_) * nRepCnt;
        break;
    default:
        return CScrView::MovePos(nChar, nRepCnt, control_down, shift_down, caret_on);
    }

    if (new_address < 0)
    {
        new_address = 0;
        aa->mac_error_ = 2;
    }
    else if (new_address > GetDocument()->length())
    {
        new_address = GetDocument()->length();
        aa->mac_error_ = 2;
    }

    if ((new_address + offset_) % rowsize_ == 0 && !edit_char_)
        SetScroll(CPoint(0,-1));

    if (shift_down && end_base)
        MoveToAddress(end_addr, new_address);
    else if (shift_down)
        MoveToAddress(start_addr, new_address);
    else
        MoveToAddress(new_address);
    return TRUE;                // Indicate that keystroke used
}

// Move the caret to new position (or as close as possible).
// Returns the new position which may be diff to the requested position if
// the requested position was invalid (past EOF).
// Note: Does not update address in tool bar combo (use show_pos(GetPos()),
//       or make sure the caret within display, and does not save undo info.
long CHexEditView::GoAddress(long start_addr, long end_addr /*=-1*/)
{
    if (end_addr < 0 || end_addr > GetDocument()->length())
        end_addr = start_addr;
    if (start_addr < 0 || start_addr > GetDocument()->length())
        start_addr = end_addr = GetDocument()->length();

    SetSel(addr2pos(start_addr), addr2pos(end_addr));

    return start_addr;
}

// Move the caret to new position updating everything:
//  - saved previous position in undo array (if caret is actually moved)
//  - updates the display address in the tool bar address edit controls
//  - Makes sure the caret is visible within display
//  - astart/aend = new selection (if aend = -1 or aend=astart then just sets caret)
//  - pstart/pend = previous selection to be saved in undo list (if they are 
//    -1 it uses the current selection/caret)
void CHexEditView::MoveToAddress(long astart, long aend, long pstart, long pend)
{
    num_entered_ = num_del_ = num_bs_ = 0;      // Stop any editing since caret moved

    if (astart < 0 || astart > GetDocument()->length())
    {
        CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
        CString ss;
        ss.Format("Attempt to jump to invalid address %ld\r"
                  "Jump to EOF instead and continue?", astart);
        if (::HMessageBox(ss, MB_OKCANCEL) != IDOK)
        {
            aa->mac_error_ = 10;
            return;
        }
        else
            aa->mac_error_ = 1;

        astart = GetDocument()->length();
    }
    if (aend < 0 || aend > GetDocument()->length())
        aend = astart;
    ASSERT(pstart >= -1 && pstart <= GetDocument()->length());
    ASSERT(pend >= pstart && pend <= GetDocument()->length());
    if (pstart < 0 || pstart > GetDocument()->length() ||
        pend < pstart || pend > GetDocument()->length())
    {
        GetSelAddr(pstart, pend);
    }

    // Is the caret/selection now in a different position
    if (astart != pstart || aend != pend)
    {
        SetSel(addr2pos(astart), addr2pos(aend), true);
        undo_.push_back(view_undo(undo_move));  // Save move in undo array
        undo_.back().address = pstart;
        if (pstart != pend)
        {
            undo_.push_back(view_undo(undo_sel)); // Save selection end in undo array
            undo_.back().address = pend;
        }
        show_prop();                            // Update prop modeless dlg
        show_calc();
        show_pos();                             // Update tool bar
    }
    DisplayCaret();                             // Make sure caret in display
}

// This saves a move in the undo array for the caret position when focus
// was last lost.  This is used by the tool bar address edit controls to
// add to the undo list when they move the address of the caret.
void CHexEditView::SaveMove()
{
    undo_.push_back(view_undo(undo_move));
    undo_.back().address = saved_start_;
    if (saved_start_ != saved_end_)
    {
        // Save the previous selection
        undo_.push_back(view_undo(undo_sel));
        undo_.back().address = saved_end_;
    }
}

// Convert an address to a display position
CPoint CHexEditView::addr2pos(long address) const
{
    address += offset_;
    if (edit_char_)
        return CPoint(char_pos(address%rowsize_), address/rowsize_ * text_height_);
    else if ((num_entered_ % 2) == 0)
        return CPoint(hex_pos(address%rowsize_), address/rowsize_ * text_height_);
    else
        return CPoint(hex_pos(address%rowsize_) + 2*text_width_,
                      address/rowsize_ * text_height_);
}

long CHexEditView::pos2addr(CPoint pos, BOOL inside /*=true*/) const
{
    long address = (pos.y/text_height_)*rowsize_ - offset_;
    if (edit_char_)
        address += pos_char(pos.x, inside);
    else
        address += pos_hex(pos.x + text_width_/2, inside);
    return address;
}

// Update property display if visible
// address = address of byte(s) to show properties of
//         = -1 to use the current cursor address
//         = -2 to update even when refresh off (using current cursor address)
void CHexEditView::show_prop(long address /*=-1*/)
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (address != -2 && aa->refresh_off_) return;

    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
    if (address < 0) address = GetPos();
    if (mm->pprop_ != NULL)
        mm->pprop_->Update(this, address);
}

void CHexEditView::show_calc()
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
    ASSERT(mm->pcalc_ != NULL);
    if (!aa->refresh_off_ && mm->pcalc_->m_hWnd != 0 && mm->pcalc_->visible_)
        mm->pcalc_->FixFileButtons();
}

// Update hex and decimal address tools in edit bar
// address = address to show in the address tools
//         = -1 to use the current cursor address
//         = -2 to update even when refresh off (using current cursor address)
void CHexEditView::show_pos(long address /*=-1*/)
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (address != -2 && aa->refresh_off_) return;

    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
    CString ss;         // String to hold text of edit box

    if (address < 0) address = GetPos();

    // Set the hex address edit box
    if (aa->hex_ucase_)
        ss.Format("%lX", address);
    else
        ss.Format("%lx", address);
    mm->hec_hex_addr_.SetWindowText(ss);
    mm->hec_hex_addr_.add_spaces();
    mm->hec_hex_addr_.UpdateWindow();           // Force immed. redraw

    // Set the decimal address edit box 
    ss.Format("%lu", address);
    mm->dec_dec_addr_.SetWindowText(ss);
    mm->dec_dec_addr_.add_commas();
    mm->dec_dec_addr_.UpdateWindow();           // Force immed. redraw
}

/////////////////////////////////////////////////////////////////////////////
// CHexEditView printing

// Handles the File/Print command
void CHexEditView::OnFilePrint()
{
    CScrView::OnFilePrint();
    ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_print);
}

BOOL CHexEditView::OnPreparePrinting(CPrintInfo* pInfo)
{
    long start_addr, end_addr;
    GetSelAddr(start_addr, end_addr);
    pInfo->m_pPD->m_pd.Flags |= PD_SHOWHELP;
    if (!pInfo->m_bPreview && start_addr != end_addr)
    {
        pInfo->m_pPD->m_pd.Flags &= ~PD_NOSELECTION;    // Enable selection radio button
        pInfo->m_pPD->m_pd.Flags |= PD_SELECTION;       // Select selection radio button
    }

    BOOL retval = DoPreparePrinting(pInfo);
    if (!retval)
    {
        CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
        aa->mac_error_ = 2;
    }
    return retval;
}

void CHexEditView::OnBeginPrinting(CDC* pDC, CPrintInfo* pInfo)
{
//    CScrView::OnPrepareDC(pDC, pInfo);  why does this call base class version?
    OnPrepareDC(pDC, pInfo);

    pDC->SetMapMode(print_map_mode_);

    // Create font based on screen font but scaled for printer
    LOGFONT print_lf = display_char_ && !ebcdic_ && graphic_ && oem_ ? oem_lf_ : lf_;
    print_lf.lfHeight = print_lfHeight_;
    print_lf.lfWidth = 0;                   // Width calced based on height

    ASSERT(print_font_ == NULL);
    print_font_ = new CFont;
    print_font_->CreateFontIndirect(&print_lf);
    pDC->SelectObject(print_font_);

    // Work out text height of the (printer) font
    TEXTMETRIC tm;
    pDC->GetTextMetrics(&tm);
    print_text_height_ = tm.tmHeight + tm.tmExternalLeading;
    CSize size;
    ::GetTextExtentPoint32(pDC->m_hDC, "W", 1, &size);
    print_text_width_ = size.cx;                        // width of a "W"

    // Work out the page size, number of pages, etc
    page_size_ = CSize(pDC->GetDeviceCaps(HORZRES), pDC->GetDeviceCaps(VERTRES));
    pDC->DPtoLP(&page_size_);               // LPtoDP always returns +ve coords (for CSize)
    lines_per_page_ =  page_size_.cy / print_text_height_ - 3;
    if (lines_per_page_ < 1) lines_per_page_ = 1;

    if (pInfo->m_pPD->PrintSelection())
    {
        print_sel_ = true;                              // Printing current selection

        // Work out the first and last line to be printed and hence the number of pages
        long start_addr, end_addr;
        GetSelAddr(start_addr, end_addr);
        start_addr = ((start_addr + offset_)/rowsize_)*rowsize_ - offset_;
        end_addr = ((end_addr + offset_ - 1)/rowsize_ + 1)*rowsize_ - offset_;
        ASSERT(start_addr < end_addr);
        pInfo->SetMaxPage((end_addr - start_addr - 1)/(rowsize_ * lines_per_page_) + 1);
    }
    else
    {
        print_sel_ = false;                             // NOT printing selection

        // Work out number of pages (so MFC knows when to stop trying to print)
        pInfo->SetMaxPage((GetDocument()->length() + offset_ - 1)/(rowsize_ * lines_per_page_) + 1);

        // If doing print preview then start on the page that contains the caret
        if (pInfo->m_bPreview)
            pInfo->m_nCurPage = (GetPos() + offset_ - 1)/(rowsize_ * lines_per_page_) + 1;
    }
}

void CHexEditView::OnEndPrinting(CDC* pDC, CPrintInfo* pInfo)
{
    pDC->SelectObject(pfont_);
    delete print_font_;
    print_font_ = NULL;

    // BG search finished message may be lost if modeless dlg running
    ((CHexEditApp *)AfxGetApp())->CheckBGSearchFinished();
}

void CHexEditView::OnPrepareDC(CDC* pDC, CPrintInfo* pInfo) 
{
    CScrView::OnPrepareDC(pDC, pInfo);

    // Store page number for OnPrint & OnDraw (called by OnPrint)
    if (pDC->IsPrinting())
    {
        curpage_ = pInfo->m_nCurPage - 1;   // Store page number (zero offset)
    }
}

void CHexEditView::OnPrint(CDC* pDC, CPrintInfo* pInfo) 
{
    pDC->SetMapMode(print_map_mode_);

    CRect rct(0,0,0,0);                     // Rectangle of printed page
    CString ss;                             // String to display at top of page
    rct.bottom = pDC->GetDeviceCaps(VERTRES);
    rct.right  = pDC->GetDeviceCaps(HORZRES);
    pDC->DPtoLP(&rct);

    pDC->SelectObject(print_font_);
    pDC->SetTextColor(RGB(0,0,0));          // Display header in black

    // Print file name (top left)
    CFile *pf = &dynamic_cast<CHexEditDoc *>(GetDocument())->file_;
    if (pf->m_hFile != CFile::hFileNull)
        pDC->DrawText(pf->GetFileName(), &rct, DT_LEFT | DT_TOP | DT_NOPREFIX | DT_SINGLELINE);

    // Print date and time (top right)
    time_t tt;                              // Current (standard C) time
    time(&tt);
    ss = ctime(&tt);
    ss = ss.Left(24);
    pDC->DrawText(ss, &rct, DT_RIGHT | DT_TOP | DT_NOPREFIX | DT_SINGLELINE);

    // Print page number (centred)
    ss.Format("- %ld -", long(pInfo->m_nCurPage));
    pDC->DrawText(ss, &rct, DT_CENTER | DT_TOP | DT_NOPREFIX | DT_SINGLELINE);

    CScrView::OnPrint(pDC, pInfo);          // Calls OnDraw to print rest of page
}

void CHexEditView::OnFilePrintPreview()
{
    CScrView::OnFilePrintPreview();
    ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_preview);

    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
    if (mm->pprop_ != NULL)
        mm->pprop_->Update(NULL, -1);                   // Blank out property dlg fields

    // Disable the dialog bar combo controls for print preview view
    CWnd *pcombo;           // Ptr to combo control in dialog bar
    pcombo = mm->m_wndEditBar.GetDlgItem(IDC_JUMP_HEX);
    ASSERT(pcombo != NULL);
    if (pcombo != NULL)
        pcombo->EnableWindow(FALSE);
    pcombo = mm->m_wndEditBar.GetDlgItem(IDC_JUMP_DEC);
    ASSERT(pcombo != NULL);
    if (pcombo != NULL)
        pcombo->EnableWindow(FALSE);
    pcombo = mm->m_wndEditBar.GetDlgItem(IDC_SEARCH);
    ASSERT(pcombo != NULL);
    if (pcombo != NULL)
        pcombo->EnableWindow(FALSE);
}

void CHexEditView::OnEndPrintPreview(CDC* pDC, CPrintInfo* pInfo, POINT point, CPreviewView* pView) 
{
    CScrView::OnEndPrintPreview(pDC, pInfo, point, pView);

    // Re-enable the control bar combo controls (disabled in print preview)
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
    CWnd *pcombo;           // Ptr to combo control in dialog bar
    pcombo = mm->m_wndEditBar.GetDlgItem(IDC_JUMP_HEX);
    ASSERT(pcombo != NULL);
    if (pcombo != NULL)
        pcombo->EnableWindow(TRUE);
    pcombo = mm->m_wndEditBar.GetDlgItem(IDC_JUMP_DEC);
    ASSERT(pcombo != NULL);
    if (pcombo != NULL)
        pcombo->EnableWindow(TRUE);
    pcombo = mm->m_wndEditBar.GetDlgItem(IDC_SEARCH);
    ASSERT(pcombo != NULL);
    if (pcombo != NULL)
        pcombo->EnableWindow(TRUE);
}

/////////////////////////////////////////////////////////////////////////////
// CHexEditView diagnostics

#ifdef _DEBUG
void CHexEditView::AssertValid() const
{
        CScrView::AssertValid();
}

void CHexEditView::Dump(CDumpContext& dc) const
{
    dc << "\nwin_size_  = " << win_size_;
    dc << "\nrowsize_   = " << rowsize_;
    dc << "\ntext_height_ = " << text_height_;
    dc << "\ntext_width_  = " << text_width_;
    dc << "\nmark_      = " << mark_;
    dc << "  mark_char_ = " << mark_char_;
    dc << "\nedit_char_ = " << edit_char_;
    dc << "\novertype_  = " << overtype_;
    dc << "\nreadonly_  = " << readonly_;
    dc << "\nsaved_start_/end_ = " << saved_start_ << " " << saved_end_;

    std::vector <view_undo, allocator<view_undo> >::const_iterator pu;
    for (pu = undo_.begin(); pu != undo_.end(); ++pu)
    {
        dc << "\nutype = " << (*pu).utype;
        switch (pu->utype)
        {
        case undo_move:
            dc << " MOVE from " << (*pu).address;
            break;
        case undo_sel:
            dc << " SELECTION to " << (*pu).address;
            break;
        case undo_change:
            dc << " DOC CHANGE this view = " << (*pu).flag;
            dc << " index = " << (*pu).index;
            break;
        case undo_swap:
            dc << " CHAR AREA = " << (*pu).flag;
            break;
        case undo_disp:
            dc << " CHAR DISPLAY = " << (*pu).flag;
            break;
        case undo_ebcdic:
            dc << " EBCDIC = " << (*pu).flag;
            break;
        case undo_font:
            dc << " FONT CHANGE ";
            break;
        case undo_autofit:
            dc << " AUTOFIT ";
            break;
        case undo_readonly:
            dc << " READ ONLY " << (*pu).flag;
            break;
        case undo_overtype:
            dc << " OVER TYPE " << (*pu).flag;
            break;
        case undo_setmark:
            dc << " MARK SET address = " << (*pu).address;
            break;
        case undo_markchar:
            dc << " MARK SET char = " << (*pu).flag;
            break;
        case undo_rowsize:
            dc << " ROW SIZE = " << (*pu).rowsize;
            break;
        case undo_group_by:
            dc << " GROUP BY = " << (*pu).rowsize;
            break;
        case undo_offset:
            dc << " OFFSET   = " << (*pu).rowsize;
            break;
        case undo_unknown:
            dc << " UNKNOWN";
            break;
        default:
            dc << " NOT RECOGNISED";
            break;
        }
      }

    CScrView::Dump(dc);
}

CHexEditDoc* CHexEditView::GetDocument() // non-debug version is inline
{
        ASSERT(m_pDocument->IsKindOf(RUNTIME_CLASS(CHexEditDoc)));
        return (CHexEditDoc*)m_pDocument;
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CHexEditView message handlers

void CHexEditView::OnSize(UINT nType, int cx, int cy) 
{
    if (cx == 0 && cy == 0)
    {
        // Not a "real" resize event
        CScrView::OnSize(nType, cx, cy);
    }
    num_entered_ = num_del_ = num_bs_ = 0;      // Can't be editing while mousing

    BOOL caret_displayed = CaretDisplayed(win_size_);
    recalc_display();
    if (caret_displayed)
        DisplayCaret(); // Keep caret within display
    CScrView::OnSize(nType, cx, cy);

    // Make sure we have all the search occurrences for the new window size
    ValidateScroll(GetScroll());

    // Keep track of current window size
    win_size_.cx = cx;
    win_size_.cy = cy;
    {
        CClientDC dc(this);
        OnPrepareDC(&dc);
        dc.DPtoLP(&win_size_);
    }
}

void CHexEditView::OnHScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar) 
{
    CScrView::OnHScroll(nSBCode, nPos, pScrollBar);
    if (nSBCode != SB_THUMBTRACK)
        ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_hscroll, (nSBCode << 16) | nPos);
}

void CHexEditView::OnVScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar) 
{
    CScrView::OnVScroll(nSBCode, nPos, pScrollBar);
    if (nSBCode != SB_THUMBTRACK)
        ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_vscroll, (nSBCode << 16) | nPos);
}

BOOL CHexEditView::OnMouseWheel(UINT nFlags, short zDelta, CPoint pt) 
{
    if ((nFlags & MK_CONTROL) != 0)
    {
        if (zDelta < 0)
            OnFontInc();
        else if (zDelta > 0)
            OnFontDec();
        return TRUE;
    }
    else
        return CScrView::OnMouseWheel(nFlags, zDelta, pt);
}

void CHexEditView::OnChar(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
    if ((nFlags & 0x2100) == 0)
    {
        for (int ii = 0; ii < nRepCnt; ++ii)
            do_char(nChar);
    }
    else
        CScrView::OnChar(nChar, nRepCnt, nFlags);
}

void CHexEditView::do_char(UINT nChar)
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    long start_addr, end_addr;
    GetSelAddr(start_addr, end_addr);

    // Test if edit key (not special key and not BS, tab, ^U, ^D, ^L, return, ^\ or nul)
    // Note: nul byte is just ignored and ^\ is only used in debug version
    if (strchr("\b\034\0", nChar) == NULL)
    {
        unsigned char cc = '\0';
        num_del_ = num_bs_ = 0;         // We're not deleting

        // Warn of a number of different problems
        if (!edit_char_ && !isxdigit(nChar))
        {
            // Non hex digit typed in hex area
            ::Beep(5000,200);
            aa->mac_error_ = 5;
            return;
        }
        if (edit_char_ && ebcdic_ &&
            (nChar > 128 || a2e_tab[nChar] == 0))
        {
            ::HMessageBox("Key pressed is not a valid EBCDIC character");
            aa->mac_error_ = 10;
            return;
        }

        if (check_ro("edit"))
            return;

        if (overtype_ && start_addr != end_addr)
        {
            if (::HMessageBox("This will delete the current selection\r"
                              "which is not allowed in overtype mode.\r"
                              "Do you want to turn off overtype mode?",
                              MB_OKCANCEL) == IDCANCEL)
            {
                aa->mac_error_ = 10;
                return;
            }
            else
                do_insert();
        }
        else if (overtype_ && start_addr == GetDocument()->length())
        {
            if (::HMessageBox("You can't extend this file while in overtype mode.\r"
                              "Do you want to turn off overtype mode?",
                              MB_OKCANCEL) == IDCANCEL)
            {
                aa->mac_error_ = 5;
                return;
            }
            else
                do_insert();
        }

        DisplayCaret();                 // Make sure start is visible

        // If there is a selection then delete it
        if (start_addr != end_addr)
        {
            ASSERT(start_addr >= 0 && end_addr <= GetDocument()->length() && start_addr < end_addr);
            GetDocument()->Change(mod_delforw, start_addr, end_addr - start_addr,
                                  NULL, 0, this);
        }

        if (edit_char_)
        {
            // In char area so insert/replace char at this curr locn
            if (ebcdic_ && nChar < 128)
                cc = a2e_tab[nChar];
            else if (ebcdic_)
                cc = '\0';
            else
                cc = (unsigned char)nChar;
            GetDocument()->Change(
                overtype_ ? mod_replace : mod_insert,
                start_addr, 1, &cc, num_entered_, this);
            // Move the caret to the next char
            ++start_addr;
            num_entered_ += 2;  // One char == 2 nybbles
        }
        else
        {
            // Must be hex digit entered in hex area
            if ((num_entered_ % 2) == 0)
            {
                //  First nybble entered - convert hex to low nybble
                cc = isdigit(nChar) ? nChar - '0' : toupper(nChar) - 'A' + 10;
                GetDocument()->Change(
                    overtype_ ? mod_replace : mod_insert,
                    start_addr, 1, &cc, num_entered_, this);
                ++num_entered_;
            }
            else
            {
                // 2nd nybble - shift over 1st nybble and add this nybble
                cc = '\0';
                size_t got;
                if (start_addr < GetDocument()->length())
                {
                    // If not at end of doc then get nybble to shift
                    got = GetDocument()->GetData(&cc, 1, start_addr);
                    ASSERT(got == 1);
                }
                cc = (cc << 4) + 
                    (isdigit(nChar) ? nChar - '0' : toupper(nChar) - 'A' + 10);
                GetDocument()->Change(
                    overtype_ ? mod_replace : mod_insert,
                    start_addr, 1, &cc, num_entered_, this);
                // Move the caret to the next byte
                ++start_addr;
                ++num_entered_;
            }
        }
        SetCaret(addr2pos(start_addr));
        show_prop();                            // Current char under caret may have changed so update prop dlg
        show_calc();
        show_pos();                             // Update tool bar
        DisplayCaret();                         // Make sure end is visible
    }
#if 0
    else if (nChar == '\f')                             // ^L - centre caret and redraw
    {
        CRect cli;

        // Work out window height in logical units
        GetClientRect(&cli);
        ConvertFromDP(cli);

        // Move display so that caret is centred vertically
        CPoint pp = GetCaret();
        pp.x = 0;                       // Scroll to left side (but see DisplayCaret() below)
        if ((pp.y -= (cli.bottom-cli.top)/2) < 0)
            pp.y = 0;
        SetScroll(pp);

        DoInvalidate();                 // Also redraw display
        DisplayCaret();                 // Make sure caret not off right side
    }
    else if (nChar == '\t' && display_char_)
    {
        num_entered_ = num_del_ = num_bs_ = 0;  // Turn off any consec edits

        // TAB key swaps between hex and char areas
        long start_addr, end_addr;
        BOOL end_base = GetSelAddr(start_addr, end_addr);
        undo_.push_back(view_undo(undo_swap));
        undo_.back().flag = edit_char_;
        edit_char_ = !edit_char_;
        if (end_base)
            SetSel(addr2pos(end_addr), addr2pos(start_addr), true);
        else
            SetSel(addr2pos(start_addr), addr2pos(end_addr));
        DisplayCaret();
    }
    else if (nChar == '\025')                   // ^U - scroll up
    {
        CPoint pp = GetScroll();
        pp.y -= text_height_;
        SetScroll(pp);
    }
    else if (nChar == '\004')                   // ^D - scroll down
    {
        CPoint pp = GetScroll();
        pp.y += text_height_;
        SetScroll(pp);
    }
    else if (nChar == '\r')
    {
        num_entered_ = num_del_ = num_bs_ = 0;  // Turn off any consec edits
        long start_addr, end_addr;
        GetSelAddr(start_addr, end_addr);

        // Check if offset has actually changed
        if (real_offset_ != (rowsize_ - start_addr%rowsize_)%rowsize_)
        {
            undo_.push_back(view_undo(undo_offset));
            undo_.back().rowsize = real_offset_;        // Save previous offset for undo
            real_offset_ = offset_ = (rowsize_ - start_addr%rowsize_)%rowsize_;
            SetSel(addr2pos(start_addr), addr2pos(end_addr));
            recalc_display();
            DoInvalidate();
        }
        else
        {
            ((CMainFrame *)AfxGetMainWnd())->
                StatusBarText("Warning: offset not changed");
            aa->mac_error_ = 1;
        }
    }
#endif
    else if (nChar == '\b')                     // Back space
    {
        // Warn if view is read only
        if (check_ro("backspace"))
            return;

        if (start_addr != end_addr)
        {
            // There is a selection - back space just deletes it
            if (overtype_)
            {
                if (::HMessageBox("This will delete the current selection\r"
                                  "which is not allowed in overtype mode.\r"
                                  "Do you want to turn off overtype mode?",
                                  MB_OKCANCEL) == IDCANCEL)
                {
                    aa->mac_error_ = 5;
                    return;
                }
                else
                    do_insert();
            }

            ASSERT(start_addr >= 0 && end_addr <= GetDocument()->length() && start_addr < end_addr);
            GetDocument()->Change(mod_delforw, start_addr, end_addr - start_addr,
                                  NULL, 0, this);
        }
        else
        {
            // Back space deletes/replaces previous byte
            if ((num_entered_ % 2) != 0)        // Check if in middle of byte entry
                ++start_addr;
            if (start_addr < 1)                 // We can't back space at the start of file
            {
                aa->mac_error_ = 2;
                return;
            }
            start_addr--;                       // Work out the address of the byte to delete
            if (!overtype_)
            {
                // Delete previous char
                GetDocument()->Change(mod_delback, start_addr, 1,
                                  NULL, num_bs_, this);
                num_bs_ += 2;
            }
            else
            {
                // Replace previous char in OVR mode with space or nul byte
                unsigned char cc;

                if (edit_char_)
                {
                    // Use EBCDIC or ASCII space
                    cc = ebcdic_ ? 0x40 : 0x20;
                    GetDocument()->Change(mod_repback, start_addr, 1,
                                          &cc, num_bs_, this);
                }
                else
                {
                    cc = '\0';
                    GetDocument()->Change(mod_repback, start_addr, 1,
                                          &cc, num_bs_, this);
                }
                ++num_bs_;
            }
        }
        num_del_ = num_entered_ = 0;                    // turn off any byte entry

        // Move the caret
        SetCaret(addr2pos(start_addr));
        show_prop();                            // New char - check props
        show_calc();
        show_pos();                             // Update tool bar
        DisplayCaret();                         // Make sure caret is visible
    }
#ifdef _DEBUG
    else if (nChar == '\034')
    {
        // Just delay for 3 seconds
        timer tt;
        // This loop is (hopefully) not optimised away in debug mode (optimisations off)
        while (tt.elapsed () < 2)
            ;
        ((CMainFrame *)AfxGetMainWnd())->StatusBarText("Ctrl+\\");
    }
#endif
    aa->SaveToMacro(km_char, nChar);
}

void CHexEditView::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
    if (nChar >= VK_PRIOR && nChar <= VK_DELETE)
    {
        num_entered_ = 0;
        CPoint start, end;
        if (GetSel(start, end))
            SetSel(end, start, true);
        else
            SetSel(start, end);     // Calls ValidateCaret - causing caret selection to be moved to valid locations
    }
#if 0
    switch(nChar)
    {
    case VK_DELETE:
        // Can't delete if view is read only or in overtype mode
        if (check_ro("delete"))
            return;

        if (overtype_)
        {
            if (::HMessageBox("You can't delete while in overtype mode.\r"
                              "Do you want to turn off overtype mode?",
                              MB_OKCANCEL) == IDCANCEL)
                return;
            else
                do_insert();
        }

        // Handle deletion of chars at caret or selection
        long start, end;
        GetSelAddr(start, end);
//      address = pos2addr(GetCaret());
        if (start == end)
        {
            if ((long)nRepCnt > GetDocument()->length() - start)
                nRepCnt = (UINT)(GetDocument()->length() - start);
            if (nRepCnt > 0)
                GetDocument()->Change(mod_delforw, start, nRepCnt,
                                      NULL, num_del_, this);
            num_del_ += nRepCnt*2;
        }
        else
        {
            GetDocument()->Change(mod_delforw, start, end-start,
                                      NULL, 0, this);
            num_del_ = 0;               // Make sure this is separate deletion
        }
        DisplayCaret();                 // Make sure caret is visible
        return;
    }
#endif

    // If cursor movement reset consec. entered count
    if (nChar >= VK_PRIOR && nChar <= VK_DELETE)
        num_entered_ = num_del_ = num_bs_ = 0;

    CScrView::OnKeyDown(nChar, nRepCnt, nFlags);
}

void CHexEditView::OnKillFocus(CWnd* pNewWnd) 
{
    // Save current address in case edit controls move caret
    GetSelAddr(saved_start_, saved_end_);

    CScrView::OnKillFocus(pNewWnd);

    if (GetView() == NULL)
    {
        CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();

        // Last view now gone so disable the dialog bar combo controls
        CWnd *pcombo;       // Ptr to combo control in dialog bar
        pcombo = mm->m_wndEditBar.GetDlgItem(IDC_JUMP_HEX);
        ASSERT(pcombo != NULL);
        if (pcombo != NULL)
            pcombo->EnableWindow(FALSE);
        pcombo = mm->m_wndEditBar.GetDlgItem(IDC_JUMP_DEC);
        ASSERT(pcombo != NULL);
        if (pcombo != NULL)
            pcombo->EnableWindow(FALSE);
        pcombo = mm->m_wndEditBar.GetDlgItem(IDC_SEARCH);
        ASSERT(pcombo != NULL);
        if (pcombo != NULL)
            pcombo->EnableWindow(FALSE);

        if (mm->pprop_ != NULL)
            mm->pprop_->Update(NULL, -1);
    }

    // Invalidate the current selection so its drawn lighter in inactive window
    long start_addr, end_addr;
    GetSelAddr(start_addr, end_addr);
    if (start_addr != end_addr)
        invalidate_addr_range(start_addr, end_addr);
    num_entered_ = num_del_ = num_bs_ = 0;      // Stop any editing
}

void CHexEditView::OnSetFocus(CWnd* pOldWnd) 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    num_entered_ = num_del_ = num_bs_ = 0;      // Stop any editing
    CScrView::OnSetFocus(pOldWnd);

    // Load bitmaps for this view in case some buttons have different
    // selection to those in view that focus just came from
    load_bitmaps();

    show_prop();
    show_calc();
    show_pos();
    move_dlgs();

    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
    CWnd *pcombo;           // Ptr to combo control in dialog bar
    pcombo = mm->m_wndEditBar.GetDlgItem(IDC_JUMP_HEX);
    ASSERT(pcombo != NULL);
    if (pcombo != NULL)
        pcombo->EnableWindow(TRUE);
    pcombo = mm->m_wndEditBar.GetDlgItem(IDC_JUMP_DEC);
    ASSERT(pcombo != NULL);
    if (pcombo != NULL)
        pcombo->EnableWindow(TRUE);
    pcombo = mm->m_wndEditBar.GetDlgItem(IDC_SEARCH);
    ASSERT(pcombo != NULL);
    if (pcombo != NULL)
        pcombo->EnableWindow(TRUE);

    long start_addr, end_addr;
    GetSelAddr(start_addr, end_addr);
    if (start_addr != end_addr)
    {
        invalidate_addr_range(start_addr, end_addr);
    }

    if (aa->pview_ != this)
    {
        // We only want to store focus change in a macro, if the last CHexEditView that
        // got focus is different to this one.  This allows us to ignore focus changes
        // when other things (find dialog, toolbar buttons etc) temp. get focus.
        // Also this macro entry may be deleted if followed by km_new, km_open,
        // km_win_new, km_childsys (next/prev win or win close) or km_win_next
        // since these implicitly change the focus, and setting it explicitly before
        // to a certain window may ruin their effect.
        ASSERT(GetParent() != NULL);
        CString ss;
        GetParent()->GetWindowText(ss);
        aa->SaveToMacro(km_focus, ss);
    }
    aa->pview_ = this;
}

void CHexEditView::load_bitmaps()
{
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
    mm->bb_addr_toggle_.LoadBitmaps(dec_addr_ ? IDB_ADDR_TOGGLES : IDB_ADDR_TOGGLEU,
                                    IDB_ADDR_TOGGLED,0,IDB_ADDR_TOGGLEX);
    mm->bb_addr_toggle_.Invalidate();
    mm->bb_char_toggle_.LoadBitmaps(display_char_ ? IDB_CHAR_TOGGLES : IDB_CHAR_TOGGLEU,
                                    IDB_CHAR_TOGGLED,0,IDB_CHAR_TOGGLEX);
    mm->bb_char_toggle_.Invalidate();
    mm->bb_asc_ebc_.LoadBitmaps(ebcdic_ ? IDB_ASC_EBCS : IDB_ASC_EBCU,
                                IDB_ASC_EBCD,0,IDB_ASC_EBCX);
    mm->bb_asc_ebc_.Invalidate();
    mm->bb_autofit_.LoadBitmaps(autofit_ ? IDB_AUTOFITS : IDB_AUTOFITU,
                                IDB_AUTOFITD,0,IDB_AUTOFITX);
    mm->bb_autofit_.Invalidate();
    mm->bb_graphic_toggle_.LoadBitmaps(graphic_ ? IDB_GRAPHIC_TOGGLES : IDB_GRAPHIC_TOGGLEU,
                                        IDB_GRAPHIC_TOGGLED,0,IDB_GRAPHIC_TOGGLEX);
    mm->bb_graphic_toggle_.Invalidate();
    if (display_control_ == 1)
        mm->bb_control_.LoadBitmaps(IDB_CONTROL1S,IDB_CONTROLD,0,IDB_CONTROLX);
    else if (display_control_ == 2)
        mm->bb_control_.LoadBitmaps(IDB_CONTROL2S,IDB_CONTROLD,0,IDB_CONTROLX);
    else
        mm->bb_control_.LoadBitmaps(IDB_CONTROLU,IDB_CONTROLD,0,IDB_CONTROLX);
    mm->bb_control_.Invalidate();
    mm->bb_allow_mods_.LoadBitmaps(readonly_ ? IDB_MODSU : IDB_MODSS,
                                    IDB_MODSD,0,IDB_MODSX);
    mm->bb_allow_mods_.Invalidate();
}

void CHexEditView::OnDestroy() 
{
    num_entered_ = num_del_ = num_bs_ = 0;      // Stop any editing
    CScrView::OnDestroy();

    // If there are no more views active ...
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
    if (GetView() == NULL)
    {
        // Last view now gone so disable the dialog bar combo controls
        CWnd *pcombo;       // Ptr to combo control in dialog bar
        pcombo = mm->m_wndEditBar.GetDlgItem(IDC_JUMP_HEX);
        ASSERT(pcombo != NULL);
        if (pcombo != NULL)
            pcombo->EnableWindow(FALSE);
        pcombo = mm->m_wndEditBar.GetDlgItem(IDC_JUMP_DEC);
        ASSERT(pcombo != NULL);
        if (pcombo != NULL)
            pcombo->EnableWindow(FALSE);
        pcombo = mm->m_wndEditBar.GetDlgItem(IDC_SEARCH);
        ASSERT(pcombo != NULL);
        if (pcombo != NULL)
            pcombo->EnableWindow(FALSE);

//      // Also delete the Properties modeless dialog if there
//      if (mm->pprop_ != NULL)
//          mm->pprop_->DestroyWindow();

        // If this is the last window we need to make sure that all file
        // access buttons are disabled in the calculator (if visible)
        show_calc();
    }
}

void CHexEditView::OnLButtonDblClk(UINT nFlags, CPoint point) 
{
    // Mark position of double click (should be caret position)
    OnMark();
    CScrView::OnLButtonDblClk(nFlags, point);
}

void CHexEditView::OnLButtonDown(UINT nFlags, CPoint point) 
{
    CPoint pp(point);                           // Point in our coord system
    ConvertFromDP(pp);

    if (pp.x <= (addr_width_ - 1)*text_width_ ||
        ( display_char_ && pp.x >= char_pos(rowsize_)+text_width_ ||
         !display_char_ && pp.x >= hex_pos(rowsize_) )  )
    {
        // Don't do anything if in address area on left or off to right
        return;
    }

    mouse_down_ = true;                         // We saw left button down event

    // Save some info that may be needed for macro recording
    dev_down_ = point;                          // Mouse down posn (device coords)
    doc_down_ = pp;                             // Mouse down posn (doc coords)
    shift_down_ = shift_down();

    GetSelAddr(prev_start_, prev_end_);
    area_swap_ = FALSE;

    if (!shift_down_)
    {
        num_entered_ = num_del_ = num_bs_ = 0;  // Can't be editing while mousing
        if (display_char_)
        {
            // Allow user to move caret between hex and char areas

            // Which area did the user click in?
            // Note edit_char_ must be set before ValidateCaret() is called while the
            // mouse button is held down so that the dragged selection is drawn correctly.
            BOOL saved_area = edit_char_;
            edit_char_ = pos_hex(pp.x, FALSE) >= rowsize_;
            area_swap_ = edit_char_ != saved_area;      // Remember for undo (button up)
        }
    }

    CScrView::OnLButtonDown(nFlags, point);

    show_prop();
    show_calc();
    show_pos();
}

void CHexEditView::OnLButtonUp(UINT nFlags, CPoint point) 
{
    if (mouse_down_)
    {
        num_entered_ = num_del_ = num_bs_ = 0;  // Can't be editing while mousing

        CScrView::OnLButtonUp(nFlags, point);
        long start_addr, end_addr;
        GetSelAddr(start_addr, end_addr);

        // Save in undo array if position moved
        if (prev_start_ != start_addr || prev_end_ != end_addr || area_swap_)
        {
            // Save the caret position (or start of selection)
            undo_.push_back(view_undo(undo_move));
            undo_.back().address = prev_start_;
            if (prev_start_ != prev_end_)
            {
                // Save the previous selection
                undo_.push_back(view_undo(undo_sel));
                undo_.back().address = prev_end_;
            }
            if (area_swap_)
            {
                // Save the fact that we swapped between areas
                undo_.push_back(view_undo(undo_swap, TRUE));
                undo_.back().flag = !edit_char_;
            }

            // Save info to keystroke macro (if recording)
            CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
            if (aa->recording_)
            {
                mouse_sel ms;                   // Info for km_mouse/km_shift_mouse
                ms.dev_down = dev_down_;        // Point in window where selection started
                CPoint doc_up(point);           // Point where selection ended
                ConvertFromDP(doc_up);          // - converted from device to doc coords

//              ms.doc_dist = doc_up - doc_down_;
                // To avoid problems where a different number of rows are selected on play
                // for slightly different scroll positions we round up to mult. of text_height_.
                ms.doc_dist.cy = (doc_up.y/text_height_ - doc_down_.y/text_height_) * text_height_;
                ms.doc_dist.cx = (doc_up.x/text_width_ - doc_down_.x/text_width_) * text_width_;
                if (shift_down_)
                    aa->SaveToMacro(km_shift_mouse, &ms);
                else
                    aa->SaveToMacro(km_mouse, &ms);
            }
        }
        show_prop();
        show_calc();
        show_pos();
        mouse_down_ = false;
    }
}

// This is here to implement the macro command km_mouse
void CHexEditView::do_mouse(CPoint dev_down, CSize doc_dist) 
{
    CPoint doc_down(dev_down);                  // Point where selection starts in window
    ConvertFromDP(doc_down);                    // ... converted from device to doc coords
    CPoint doc_up(doc_down);                    // Point where selection ends = start +
    doc_up += doc_dist;                         // ... distance of selection

    // Convert selection (doc coords) to addresses and make sure in valid range
    long start_addr, end_addr;
    long length = GetDocument()->length();
    start_addr = pos2addr(doc_down);
    end_addr = pos2addr(doc_up);

    if (start_addr < 0) start_addr = 0; else if (start_addr > length) start_addr = length;
    if (end_addr < 0) end_addr = 0; else if (end_addr > length) end_addr = length;

    // Save undo information
    long prev_start, prev_end;
    GetSelAddr(prev_start, prev_end);
    if (start_addr == prev_start && end_addr == prev_end)
        return;

    BOOL saved_area = edit_char_;
    if (display_char_)                                          // If char area displayed ...
        edit_char_ = pos_hex(doc_down.x, FALSE) >= rowsize_;    // sel may swap between areas
    undo_.push_back(view_undo(undo_move));
    undo_.back().address = prev_start;
    if (prev_start != prev_end)
    {
        // Save the previous selection
        undo_.push_back(view_undo(undo_sel));
        undo_.back().address = prev_end;
    }
    if (edit_char_ != saved_area)
    {
        // Save the fact that we swapped between areas
        undo_.push_back(view_undo(undo_swap, TRUE));
        undo_.back().flag = !edit_char_;
    }
    SetSel(addr2pos(start_addr), addr2pos(end_addr), true);
}

// This is here to implement the macro command km_shift_mouse
void CHexEditView::do_shift_mouse(CPoint dev_down, CSize doc_dist) 
{
    CPoint doc_down(dev_down);                  // Point of left button down in window
    ConvertFromDP(doc_down);                    // ... converted from device to doc coords
    CPoint doc_up(doc_down);                    // Point of left button up (doc coords) = ...
    doc_up += doc_dist;                         // ... doc_down + length of selection
    // Convert selection (doc coords) to addresses and make sure in valid range
    long start_addr, end_addr;                  // New selection range
    long prev_start, prev_end;                  // Previous selection range
    long length = GetDocument()->length();
//    GetSelAddr(start_addr, prev_end);
    if (GetSelAddr(prev_start, prev_end))
        start_addr = prev_end;                  // Base of previous selection was end
    else
        start_addr = prev_start;
    end_addr = pos2addr(doc_up);

    if (start_addr < 0) start_addr = 0; else if (start_addr > length) start_addr = length;
    if (end_addr < 0) end_addr = 0; else if (end_addr > length) end_addr = length;

    if (end_addr != prev_end)
    {
        // Save the previous selection for undo
        undo_.push_back(view_undo(undo_move));
        undo_.back().address = start_addr;
        undo_.push_back(view_undo(undo_sel));
        undo_.back().address = prev_end;

        // Extend the selection
        SetSel(addr2pos(start_addr), addr2pos(end_addr), true);
    }
}

void CHexEditView::OnRButtonDown(UINT nFlags, CPoint point) 
{
    view_context(point);
//    CScrView::OnRButtonDown(nFlags, point);
}

void CHexEditView::OnContextMenu(CWnd* pWnd, CPoint point) 
{
    view_context(point);
}

BOOL CHexEditView::OnSetCursor(CWnd* pWnd, UINT nHitTest, UINT message) 
{
    // Only respond if in client area
    if (nHitTest == HTCLIENT)
    {
        // Work out location of cursor in window and make sure it's
        // not over address area (on left) or past last chars (on right)
        CPoint point;
        ::GetCursorPos(&point);
        ScreenToClient(&point);
        ConvertFromDP(point);
        if (point.x > (addr_width_ - 1)*text_width_ &&
            ( display_char_ && point.x < char_pos(rowsize_)+text_width_ ||
             !display_char_ && point.x < hex_pos(rowsize_) )  )
        {
            // Over hex or char area
            SetCursor(AfxGetApp()->LoadStandardCursor(IDC_IBEAM));
            return TRUE;
        }
    }
        
    return CScrView::OnSetCursor(pWnd, nHitTest, message);
}

// Display views context menu
void CHexEditView::view_context(CPoint point) 
{
    // Get the top level menu that contains the submenus used as popup menus
    CMenu top;
    BOOL ok = top.LoadMenu(IDR_CONTEXT);
    ASSERT(ok);
    if (!ok) return;

    CMenu *ppop = top.GetSubMenu(0);
    ASSERT(ppop != NULL);
    if (ppop != NULL)
    {
        // Convert window coords to the required screen coords
        ClientToScreen(&point);
        VERIFY(ppop->TrackPopupMenu(TPM_LEFTALIGN | TPM_RIGHTBUTTON,
                                    point.x, point.y, GetParent()));
    }

    top.DestroyMenu();
}

/////////////////////////////////////////////////////////////////////////////
// CHexEditView command handlers

void CHexEditView::OnRedraw() 
{
    CRect cli;

    // Work out window height in logical units
    GetClientRect(&cli);
    ConvertFromDP(cli);

    // Move display so that caret is centred vertically
    CPoint pp = GetCaret();
    pp.x = 0;                       // Scroll to left side (but see DisplayCaret() below)
    if ((pp.y -= (cli.bottom-cli.top)/2) < 0)
        pp.y = 0;
    SetScroll(pp);

    DoInvalidate();                 // Also redraw display
    DisplayCaret();                 // Make sure caret not off right side

    ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_redraw);
}

void CHexEditView::OnScrollDown() 
{
    CPoint pp = GetScroll();
    pp.y += text_height_;
    SetScroll(pp);
    ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_scroll_down);
}

void CHexEditView::OnScrollUp() 
{
    CPoint pp = GetScroll();
    pp.y -= text_height_;
    SetScroll(pp);
    ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_scroll_up);
}

// Move current position to start of display line
void CHexEditView::OnStartLine() 
{
    num_entered_ = num_del_ = num_bs_ = 0;  // Turn off any consec edits
    long start_addr, end_addr;
    GetSelAddr(start_addr, end_addr);

    // Check if offset has actually changed
    if (real_offset_ != (rowsize_ - start_addr%rowsize_)%rowsize_)
    {
        undo_.push_back(view_undo(undo_offset));
        undo_.back().rowsize = real_offset_;        // Save previous offset for undo
        real_offset_ = offset_ = (rowsize_ - start_addr%rowsize_)%rowsize_;
        SetSel(addr2pos(start_addr), addr2pos(end_addr));
        recalc_display();
        DoInvalidate();
        ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_start_line);
    }
    else
    {
        ((CMainFrame *)AfxGetMainWnd())->StatusBarText("Warning: offset not changed");
        ((CHexEditApp *)AfxGetApp())->mac_error_ = 1;
    }
}

void CHexEditView::OnSwap() 
{
    if (display_char_)
    {
        num_entered_ = num_del_ = num_bs_ = 0;  // Turn off any consec edits

        // Swap between hex and char areas
        long start_addr, end_addr;
        BOOL end_base = GetSelAddr(start_addr, end_addr);
        undo_.push_back(view_undo(undo_swap));
        undo_.back().flag = edit_char_;
        edit_char_ = !edit_char_;
        if (end_base)
            SetSel(addr2pos(end_addr), addr2pos(start_addr), true);
        else
            SetSel(addr2pos(start_addr), addr2pos(end_addr));
        DisplayCaret();

        ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_swap_areas);
    }
    else
    {
        ::HMessageBox("Cannot swap to character area - not displayed");
        ((CHexEditApp *)AfxGetApp())->mac_error_ = 5;
    }
}

void CHexEditView::OnUpdateSwap(CCmdUI* pCmdUI) 
{
    pCmdUI->Enable(display_char_);      // disallow swap if char area not displayed
}

void CHexEditView::OnDel() 
{
    num_entered_ = 0;           // Not entering but deleting

    CPoint pt_start, pt_end;
    if (GetSel(pt_start, pt_end))
        SetSel(pt_end, pt_start, true);
    else
        SetSel(pt_start, pt_end);     // Calls ValidateCaret - causing caret selection to be moved to valid locations

    if (check_ro("delete"))
        return;

    if (overtype_)
    {
        if (::HMessageBox("You can't delete while in overtype mode.\r"
                          "Do you want to turn off overtype mode?",
                          MB_OKCANCEL) == IDCANCEL)
            return;
        else
            do_insert();
    }

    // Handle deletion of chars at caret or selection
    long start, end;
    GetSelAddr(start, end);
//      address = pos2addr(GetCaret());

    if (start == end)
    {
//        if ((long)nRepCnt > GetDocument()->length() - start)
//            nRepCnt = (UINT)(GetDocument()->length() - start);
//        if (nRepCnt > 0)
//            GetDocument()->Change(mod_delforw, start, nRepCnt,
//                                  NULL, num_del_, this);
//        num_del_ += nRepCnt*2;
        if (start >= GetDocument()->length())
        {
            ((CHexEditApp *)AfxGetApp())->mac_error_ = 2;
            return;
        }

        GetDocument()->Change(mod_delforw, start, 1, NULL, num_del_, this);
        num_del_ += 2;
    }
    else
    {
        GetDocument()->Change(mod_delforw, start, end-start,
                                  NULL, 0, this);
        num_del_ = 0;               // Make sure this is separate deletion
    }
    DisplayCaret();                 // Make sure caret is visible

    ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_del);
}

void CHexEditView::OnSelectAll() 
{
    num_entered_ = num_del_ = num_bs_ = 0;      // Can't be editing while mousing
    GetSelAddr(prev_start_, prev_end_);
    SetSel(addr2pos(0), addr2pos(GetDocument()->length()));
    show_prop();
    show_calc();
    show_pos();

    // Save in undo array if position moved
    if (prev_start_ != 0 || prev_end_ != GetDocument()->length())
    {
        // Save the caret position (or start of selection)
        undo_.push_back(view_undo(undo_move));
        undo_.back().address = prev_start_;
        if (prev_start_ != prev_end_)
        {
            // Save the previous selection
            undo_.push_back(view_undo(undo_sel));
            undo_.back().address = prev_end_;
        }
    }
    ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_sel_all);
}

// Get file name, insert into document and select inserted bytes
void CHexEditView::OnReadFile() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    // Get name of file to read
    CFileDialog dlgFile(TRUE);

    // Set up the title of the dialog
    dlgFile.m_ofn.lpstrTitle = "Insert File";

    // Get the name of the all files filter from resource file
    CString allFilter;
    VERIFY(allFilter.LoadString(AFX_IDS_ALLFILTER));

    // Set up a single file for "All files"
    CString strFilter;
    strFilter += allFilter;
    strFilter += (TCHAR)'\0';   // next string please
    strFilter += _T("*.*");
    strFilter += (TCHAR)'\0';   // last string
    dlgFile.m_ofn.lpstrFilter = strFilter;

    if (!aa->dir_read_.IsEmpty())
        dlgFile.m_ofn.lpstrInitialDir = aa->dir_read_;

    // Run File Open dialog (set flag to say file must exist)
    dlgFile.m_ofn.Flags |= (OFN_FILEMUSTEXIST | OFN_SHOWHELP);
    if (dlgFile.DoModal() == IDOK)
    {
        do_read(dlgFile.GetPathName());

        // Save where the file came from (assumes OFN_NOCHANGEDIR not used)
        char buffer[256];
        if (GetCurrentDirectory(256, buffer))
            aa->dir_read_ = buffer;
    }

    // BG search finished message may be lost if modeless dlg running
    aa->CheckBGSearchFinished();
}

// Actually open and read the file and select the inserted bytes
void CHexEditView::do_read(CString file_name)
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    // Can't modify the file if view is read only or in overtype mode

    if (check_ro("insert file"))
        return;

    if (overtype_)
    {
        if (::HMessageBox("You can't insert a file while in overtype mode.\r"
                          "Do you want to turn off overtype mode?",
                          MB_OKCANCEL) == IDCANCEL)
        {
            aa->mac_error_ = 5;
            return;
        }
        else
            do_insert();
    }

    num_entered_ = num_del_ = num_bs_ = 0;      // Stop any editing

    // Open the file for reading
    CFile ff;                   // The MFC file
    CFileException fe;          // Used to receive file error information

    if (!ff.Open(file_name,
        CFile::modeRead|CFile::shareDenyWrite|CFile::typeBinary, &fe) )
    {
        // Display info about why the open failed
        CString mess(file_name);
        CFileStatus fs;

        switch (fe.m_cause)
        {
        case CFileException::fileNotFound:
            mess += "\rdoes not exist";
            break;
        case CFileException::badPath:
            mess += "\ris an invalid file name or the drive/path does not exist";
            break;
        case CFileException::tooManyOpenFiles:
            mess += "\r- too many files already open";
            break;
        case CFileException::accessDenied:
            if (!CFile::GetStatus(file_name, fs))
                mess += "\rdoes not exist";
            else
            {
                if (fs.m_attribute & CFile::directory)
                    mess += "\ris a directory";
                else if (fs.m_attribute & (CFile::volume|CFile::hidden|CFile::system))
                    mess += "\ris a special file";
                else
                    mess += "\rcannot be used (reason unknown)";
            }
            break;
        case CFileException::sharingViolation:
        case CFileException::lockViolation:
            mess += "\ris in use";
            break;
        case CFileException::hardIO:
            mess += "\r- hardware error";
            break;
        default:
            mess += "\rcould not be opened (reason unknown)";
            break;
        }
        ::HMessageBox(mess);

        aa->mac_error_ = 10;
        return;
    }

    long addr = GetPos();                       // Current caret (address to insert file)
    long data_len = ff.GetLength();             // Length of new data = file len
    if (data_len == 0)
    {
        // No need to insert a zero length file
        ((CMainFrame *)AfxGetMainWnd())->StatusBarText("File is empty: nothing inserted");
        aa->mac_error_ = 1;
        return;
    }

    CWaitCursor wait;                           // Turn on wait cursor (hourglass)

    // If the user tries to insert a huge file we might run out of memory so catch this
    try
    {
        // Get memory to read all of the file
        unsigned char *file_data = new unsigned char[data_len];

        // Read the file into memory
        if (ff.Read((void *)file_data, (UINT)data_len) != data_len)
        {
            ::HMessageBox("Not all of file could be read");
            aa->mac_error_ = 10;
            return;
        }
        ff.Close();

        // Make the memory part of the document
        GetDocument()->Change(mod_insert, addr, data_len, file_data, 0, this);
        delete[] file_data;
    }
    catch(std::bad_alloc)
    {
        ::HMessageBox("Insufficient memory to read the file");
        aa->mac_error_ = 10;
        return;
    }
    SetSel(addr2pos(addr), addr2pos(addr+data_len));
    DisplayCaret();
    aa->SaveToMacro(km_read_file, file_name);
}

void CHexEditView::OnUpdateReadFile(CCmdUI* pCmdUI) 
{
    pCmdUI->Enable(!GetDocument()->read_only()); // disallow insert if file is read only
}

void CHexEditView::OnEditWriteFile() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    num_entered_ = num_del_ = num_bs_ = 0;      // Stop any editing

    // Get the selection
    long start, end;
    GetSelAddr(start, end);
    ASSERT(start >= 0 && start <= end && end <= GetDocument()->length());
    if (start == end)
    {
        // Nothing selected, presumably in macro playback
        ASSERT(aa->playing_);
        ::HMessageBox("Nothing selected to write to file!");
        aa->mac_error_ = 10;
        return;
    }

    // Get the file name to write the selection to
    CFileDialog dlgFile(FALSE);

    // Get the name of the all files filter from resource file
    CString allFilter;
    VERIFY(allFilter.LoadString(AFX_IDS_ALLFILTER));

    // Set up a single filter for "All files"
    CString strFilter;
    strFilter += allFilter;
    strFilter += (TCHAR)'\0';   // next string please
    strFilter += _T("*.*");
    strFilter += (TCHAR)'\0';   // last string
    dlgFile.m_ofn.lpstrFilter = strFilter;

    // Set up the title of the dialog
    dlgFile.m_ofn.lpstrTitle = "Save Selection As";

    if (!aa->dir_write_.IsEmpty())
        dlgFile.m_ofn.lpstrInitialDir = aa->dir_write_;

    dlgFile.m_ofn.Flags |= OFN_SHOWHELP;
    if (dlgFile.DoModal() != IDOK)
    {
        aa->mac_error_ = 2;
        return;
    }

    // Save where the file is to be written to (assumes OFN_NOCHANGEDIR not used)
    char buffer[256];
    if (GetCurrentDirectory(256, buffer))
        aa->dir_write_ = buffer;

    // Write to the file
    CWaitCursor wait;                           // Turn on wait cursor (hourglass)
    if (GetDocument()->WriteData(dlgFile.GetPathName(), start, end))
        aa->SaveToMacro(km_write_file);

    // BG search finished message may be lost if modeless dlg running
    aa->CheckBGSearchFinished();
}

void CHexEditView::OnEditCut() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (!aa->playing_ && GetFocus() != this) SetFocus(); // Ensure focus does not stay in DlgBar

    // Can't delete if view is read only or in overtype mode
    if (check_ro("cut to the clipboard"))
        return;

    if (overtype_)
    {
        if (::HMessageBox("You can't cut while in overtype mode.\r"
                          "Do you want to turn off overtype mode?",
                          MB_OKCANCEL) == IDCANCEL)
        {
            aa->mac_error_ = 5;
            return;
        }
        else
            do_insert();
    }

    // Copy the selection to the clipboard and then delete it
    if (!CopyToClipboard())
    {
        ASSERT(aa->mac_error_ > 0);             // There must have been an error
        return;
    }

    // Handle deletion of chars at caret or selection
    long start, end;
    GetSelAddr(start, end);
    ASSERT(start >= 0 && start < end && end <= GetDocument()->length());
    GetDocument()->Change(mod_delforw, start, end-start,
                              NULL, 0, this);
    DisplayCaret();                     // Make sure caret is visible

    aa->SaveToMacro(km_cut);
}

void CHexEditView::OnUpdateEditCut(CCmdUI* pCmdUI) 
{
    if (GetDocument()->read_only())
        pCmdUI->Enable(FALSE);          // disallow cut if file is read only
    else
        OnUpdateClipboard(pCmdUI);
}

void CHexEditView::OnEditCopy()
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (!aa->playing_ && GetFocus() != this) SetFocus(); // Ensure focus does not stay in DlgBar

    if (CopyToClipboard())
        aa->SaveToMacro(km_copy);
}

// Update handler that turns on certain user interface options (Copy etc) if there
// is a selection -- ie. there is something available to be placed on the clipboard
void CHexEditView::OnUpdateClipboard(CCmdUI* pCmdUI) 
{
    // Is there any text selected?
    CPoint start, end;
    GetSel(start, end);
    pCmdUI->Enable(start != end);
}

const char *CHexEditView::bin_format_name = "BinaryData";

bool CHexEditView::CopyToClipboard()
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    num_entered_ = num_del_ = num_bs_ = 0;      // Stop any editing

    // Get the addresses of the selection
    long start, end;
    GetSelAddr(start, end);
    ASSERT(start >= 0 && start <= end && end <= GetDocument()->length());
    if (start == end)
    {
        ASSERT(aa->playing_);
        // Copy to clipboard while nothing selected, presumably in macro playback
        ::HMessageBox("Nothing selected to place on clipboard!");
        aa->mac_error_ = 10;
        return false;
    }

    if (!aa->is_nt_ && end - start > MAX_CLIPBOARD)
    {
        ::HMessageBox("Too much for clipboard!");
        aa->mac_error_ = 2;
        return false;
    }

    if (!OpenClipboard())
    {
        ::HMessageBox("The clipboard is in use!");
        aa->mac_error_ = 10;
        return false;
    }
    if (!::EmptyClipboard())
    {
        ::HMessageBox("Could not delete previous contents of the clipboard!");
        ::CloseClipboard();
        aa->mac_error_ = 10;
        return false;
    }

    CWaitCursor wait;                           // Turn on wait cursor (hourglass)

    // Copy to the clipboard as text
    {
        // Get windows memory to allow data to be put on clipboard
        HANDLE hh;                                      // Windows handle for memory
        unsigned char *p_cb, *pp;                       // Ptrs to + within the clipboard mem
        if ((hh = ::GlobalAlloc(GMEM_MOVEABLE|GMEM_DDESHARE, end-start+1)) == NULL ||
            (p_cb = reinterpret_cast<unsigned char *>(::GlobalLock(hh))) == NULL)
        {
            ::HMessageBox("Insufficient memory for clipboard data");
            ::CloseClipboard();
            aa->mac_error_ = 10;
            return false;
        }

        // Copy the data from the document to the global memory
        unsigned char *buf = new unsigned char[clipboard_buf_len];
        size_t len;
        pp = p_cb;
        for (long curr = start; curr < end; curr += len)
        {
            len = min(clipboard_buf_len, end - curr);
            VERIFY(GetDocument()->GetData(buf, len, curr) == len);
            // Copy all characters in buffer to clipboard memory (unless nul)
            unsigned char *end_buf = buf + len;
            if (!ebcdic_)
            {
                for (unsigned char *ss = buf; ss < end_buf; ++ss)
                    if (*ss != '\0')
                        *pp++ = *ss;
            }
            else
            {
                for (unsigned char *ss = buf; ss < end_buf; ++ss)
                    if (e2a_tab[*ss] != '\0')
                        *pp++ = e2a_tab[*ss];
            }
        }
        // If pp has not been incremented then no valid characters were copied
        if (pp == p_cb)
        {
            aa->mac_error_ = 1;
            ((CMainFrame *)AfxGetMainWnd())->
                StatusBarText("Warning: no valid text bytes - no text placed on clipboard");
        }
        *pp ='\0';
        delete[] buf;
        ::GlobalUnlock(hh);

        if (::SetClipboardData(CF_TEXT, hh) == NULL)
        {
            ::GlobalFree(hh);
            ::HMessageBox("Could not place text data on clipboard");
            ::CloseClipboard();
            aa->mac_error_ = 10;
            return false;
        }
        aa->ClipBoardAdd(pp - p_cb);
        // Note: the clipboard now owns the memory so ::GlobalFree(hh) should not be called
    }

    // Create the "BinaryData" clipboard format (or get it if it already exists)
    UINT bin_format;
    if ((bin_format = ::RegisterClipboardFormat(bin_format_name)) != 0)
    {
        // Get windows memory to allow binary data to be put on clipboard
        HANDLE hh;                                      // Windows handle for memory
        unsigned char *pp;                              // Actual pointer to the memory
        if ((hh = ::GlobalAlloc(GMEM_MOVEABLE|GMEM_DDESHARE, end-start+4)) == NULL ||
            (pp = reinterpret_cast<unsigned char *>(::GlobalLock(hh))) == NULL)
        {
            ::HMessageBox("Insufficient memory: stored to clipboard as text only");
            ::CloseClipboard();
            aa->mac_error_ = 10;
            return false;
        }

        // Add the binary data length to first 4 bytes of BinaryData clipboard memory
        long *pl = reinterpret_cast<long *>(pp);
        *pl = end - start;

        // Copy the data from the document to the global memory
        VERIFY(GetDocument()->GetData(pp+4, end - start, start) == end - start);

        ::GlobalUnlock(hh);

        if (::SetClipboardData(bin_format, hh) == NULL)
        {
            ::GlobalFree(hh);
            ::HMessageBox("Could not place binary data on clipboard: stored as text only");
            ::CloseClipboard();
            aa->mac_error_ = 10;
            return false;
        }
        // Note: the clipboard now owns the memory so ::GlobalFree(hh) should not be called
    }

    if (!::CloseClipboard())
    {
        aa->mac_error_ = 20;
        return false;
    }
    return true;
}

// Copy to clipboard as hex text
void CHexEditView::OnCopyHex() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    num_entered_ = num_del_ = num_bs_ = 0;      // Stop any editing

    // Get the addresses of the selection
    long start, end;
    GetSelAddr(start, end);
    ASSERT(start >= 0 && start <= end && end <= GetDocument()->length());
    if (start == end)
    {
        // Copy to clipboard while nothing selected, presumably in macro playback
        ASSERT(aa->playing_);
        ::HMessageBox("Nothing selected to place on clipboard!");
        aa->mac_error_ = 10;
        return;
    }

    // Work out the amount of memory needed (may be slightly more than needed).
    // Allow 3 chars for every byte (2 hex digits + a space), plus
    // 2 chars per line (CR+LF), + 1 trailing nul byte.
    long mem_needed = (end-start)*3+((end-start)/rowsize_+2)*2+1;

    if (!aa->is_nt_ && mem_needed > MAX_CLIPBOARD)
    {
        ::HMessageBox("Too much for clipboard!");
        aa->mac_error_ = 2;
        return;
    }

    if (mem_needed > 500000L)
    {
        CString ss;
        ss.Format("Do you really want to put %sbytes on\n"
                  "the clipboard?  This may take some time.",
                  NumScale(double(mem_needed)));
        if (::HMessageBox(ss, MB_YESNO) != IDYES)
        {
            aa->mac_error_ = 5;
            return;
        }
    }

    if (!OpenClipboard())
    {
        ::HMessageBox("The clipboard is in use!");
        aa->mac_error_ = 10;
        return;
    }
    if (!::EmptyClipboard())
    {
        ::HMessageBox("Could delete previous contents of the clipboard!");
        ::CloseClipboard();
        aa->mac_error_ = 10;
        return;
    }

    CWaitCursor wait;                           // Turn on wait cursor (hourglass)

    {
        // Get windows memory to allow data to be put on clipboard
        HANDLE hh;                              // Windows handle for memory
        char *p_cb, *pp;                        // Ptr to start and within the clipboard text

        if ((hh = ::GlobalAlloc(GMEM_MOVEABLE|GMEM_DDESHARE, mem_needed)) == NULL ||
            (p_cb = reinterpret_cast<char *>(::GlobalLock(hh))) == NULL)
        {
            ::HMessageBox("Insufficient memory for clipboard data");
            ::CloseClipboard();
            aa->mac_error_ = 10;
            return;
        }

        unsigned char cc;
        const char *hex;
        if (aa->hex_ucase_)
            hex = "0123456789ABCDEF?";
        else
            hex = "0123456789abcdef?";

        pp = p_cb;

        for (long curr = start; curr < end; )
        {
            VERIFY(GetDocument()->GetData(&cc, 1, curr) == 1);
            *pp++ = hex[(cc>>4)&0xF];
            *pp++ = hex[cc&0xF];
            *pp++ = ' ';
            if ((++curr + offset_)%rowsize_ == 0)
            {
                *pp++ = '\r';
                *pp++ = '\n';
            }
        }
        if ((curr + offset_)%rowsize_ != 0)
        {
            // Add line termination at end
            *pp++ = '\r';
            *pp++ = '\n';
        }
        *pp ='\0';
        ::GlobalUnlock(hh);

        if (::SetClipboardData(CF_TEXT, hh) == NULL)
        {
            ::GlobalFree(hh);
            ::HMessageBox("Could not place data on clipboard");
            ::CloseClipboard();
            aa->mac_error_ = 10;
            return;
        }
        aa->ClipBoardAdd(pp - p_cb);
        // Note: the clipboard now owns the memory so ::GlobalFree(hh) should not be called
    }

    if (!::CloseClipboard())
        aa->mac_error_ = 20;
    else
        aa->SaveToMacro(km_copy_hex);
}

// Copy to cipboard as C source (characters stored as hex ints)
void CHexEditView::OnCopyCchar() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    num_entered_ = num_del_ = num_bs_ = 0;      // Stop any editing

    // Get the addresses of the selection
    long start, end;
    GetSelAddr(start, end);
    ASSERT(start >= 0 && start <= end && end <= GetDocument()->length());
    if (start == end)
    {
        // Copy to clipboard while nothing selected, presumably in macro playback
        ASSERT(aa->playing_);
        ::HMessageBox("Nothing selected to place on clipboard!");
        ::CloseClipboard();
        aa->mac_error_ = 10;
        return;
    }

    // Work out the amount of memory needed (may be slightly more than needed).
    // Allow 6 chars for every byte ("0x" + 2 hex digits + comma + space), plus
    // 7 ("/**/ "+CR+LF) + addr_width_ chars per line, + 1 trailing nul byte.
    long mem_needed = (end-start)*6+((end-start)/rowsize_+2)*(7+addr_width_)+1;

    if (!aa->is_nt_ && mem_needed > MAX_CLIPBOARD)
    {
        ::HMessageBox("Too much for clipboard!");
        aa->mac_error_ = 2;
        return;
    }

    if (mem_needed > 500000L)
    {
        CString ss;
        ss.Format("Do you really want to put %sbytes on\n"
                  "the clipboard?  This may take some time.",
                  NumScale(double(mem_needed)));
        if (::HMessageBox(ss, MB_YESNO) != IDYES)
        {
            aa->mac_error_ = 5;
            return;
        }
    }

    if (!OpenClipboard())
    {
        ::HMessageBox("The clipboard is in use!");
        aa->mac_error_ = 10;
        return;
    }
    if (!::EmptyClipboard())
    {
        ::HMessageBox("Could delete previous contents of the clipboard!");
        ::CloseClipboard();
        aa->mac_error_ = 10;
        return;
    }

    CWaitCursor wait;                           // Turn on wait cursor (hourglass)

    {
        // Get windows memory to allow data to be put on clipboard
        HANDLE hh;                              // Windows handle for memory
        char *p_cb, *pp;                        // Ptr to start and within the clipboard text

        if ((hh = ::GlobalAlloc(GMEM_MOVEABLE|GMEM_DDESHARE, mem_needed)) == NULL ||
            (p_cb = reinterpret_cast<char *>(::GlobalLock(hh))) == NULL)
        {
            ::HMessageBox("Insufficient memory for clipboard data");
            ::CloseClipboard();
            aa->mac_error_ = 10;
            return;
        }

        unsigned char cc;
        const char *hex;
        if (aa->hex_ucase_)
            hex = "0123456789ABCDEF?";
        else
            hex = "0123456789abcdef?";

        pp = p_cb;

        // Add initial address if it would not be added in the loop
        if ((start + offset_)%rowsize_ != 0)
        {
            *pp++ = '/';
            *pp++ = '*';
            sprintf(pp, addr_format_, start);
            pp += strlen(pp);
            *pp++ = '*';
            *pp++ = '/';
            *pp++ = ' ';
        }

        for (long curr = start; curr < end; )
        {
            if ((curr + offset_)%rowsize_ == 0)
            {
                *pp++ = '/';
                *pp++ = '*';
                sprintf(pp, addr_format_, curr);
                pp += strlen(pp);
                *pp++ = '*';
                *pp++ = '/';
                *pp++ = ' ';
            }

            VERIFY(GetDocument()->GetData(&cc, 1, curr) == 1);
            *pp++ = '0';
            *pp++ = 'x';
            *pp++ = hex[(cc>>4)&0xF];
            *pp++ = hex[cc&0xF];
            *pp++ = ',';
            *pp++ = ' ';
            if ((++curr + offset_)%rowsize_ == 0)
            {
                *pp++ = '\r';
                *pp++ = '\n';
            }
        }
        if ((curr + offset_)%rowsize_ != 0)
        {
            // Add line termination at end if it has not been terminated in the loop
            *pp++ = '\r';
            *pp++ = '\n';
        }
        *pp ='\0';
        ::GlobalUnlock(hh);

        if (::SetClipboardData(CF_TEXT, hh) == NULL)
        {
            ::GlobalFree(hh);
            ::HMessageBox("Could not place data on clipboard");
            ::CloseClipboard();
            aa->mac_error_ = 10;
            return;
        }
        aa->ClipBoardAdd(pp - p_cb);
        // Note: the clipboard now owns the memory so ::GlobalFree(hh) should not be called
    }

    if (!::CloseClipboard())
        aa->mac_error_ = 20;
    else
        aa->SaveToMacro(km_copy_cchar);
}

void CHexEditView::OnEditPaste() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (!aa->playing_ && GetFocus() != this) SetFocus(); // Ensure focus does not stay in DlgBar

    // Can't paste if view is read only or in overtype mode
    if (check_ro("paste"))
        return;

    if (overtype_)
    {
        switch (::HMessageBox("Pasting in overtype mode will overwrite data!\r"
                          "Do you want to turn on insert mode?",
                          MB_YESNOCANCEL))
        {
        case IDYES:
            do_insert();
            break;
        case IDNO:
            break; /* do nothing */
        default:
            aa->mac_error_ = 5;
            return;
        }
    }

    num_entered_ = num_del_ = num_bs_ = 0;      // Stop any editing

    UINT ff = 0;                                // Clipboard format number
    HANDLE hh;                                  // Handle to clipboard memory
    unsigned char *pp;                          // Pointer to actual data

    if (!OpenClipboard())
    {
        ::HMessageBox("The clipboard is in use!");
        aa->mac_error_ = 10;
        return;
    }

    CWaitCursor wait;                           // Turn on wait cursor (hourglass)

    // Check if there is a "BinaryData" format (added by us or DevStudio)
    while ((ff = EnumClipboardFormats(ff)) != 0)
    {
        CString tt;
        char name[16];
        size_t nlen = ::GetClipboardFormatName(ff, name, 15);
        name[nlen] = '\0';
        if (strcmp(name, bin_format_name) == 0)
        {
            if ((hh = ::GetClipboardData(ff)) != NULL &&
                (pp = reinterpret_cast<unsigned char *>(::GlobalLock(hh))) != NULL)
            {
                long *pl = reinterpret_cast<long *>(pp);
                long addr = GetPos();
                if (overtype_)
                    GetDocument()->Change(mod_replace, addr, *pl, pp+4, 0, this);
                else
                    GetDocument()->Change(mod_insert, addr, *pl, pp+4, 0, this);
                SetSel(addr2pos(addr), addr2pos(addr+*pl));
                DisplayCaret();
                ::CloseClipboard();
                aa->SaveToMacro(km_paste);
            }
            else
                aa->mac_error_ = 20;    // It's there but couldn't get it
            return;
        }
    }

    // BinaryData format not found so just use text format
    if ((hh = ::GetClipboardData(CF_TEXT)) != NULL &&
        (pp = reinterpret_cast<unsigned char *>(::GlobalLock(hh))) != NULL)
    {
        size_t len = strlen(reinterpret_cast<char *>(pp));
        if (len > 0 && !ebcdic_)        // Bugs in other apps (eg Hedit) might cause len == 0
        {
            long addr = GetPos();
            if (overtype_)
                GetDocument()->Change(mod_replace, addr, len, pp, 0, this);
            else
                GetDocument()->Change(mod_insert, addr, len, pp, 0, this);
            SetSel(addr2pos(addr), addr2pos(addr+len));
            DisplayCaret();
        }
        else if (len > 0)
        {
            // Copy from clipboard to temp buffer converting to EBCDIC
            unsigned char *buf = new unsigned char[len];
            size_t newlen = 0;
            for (size_t ii = 0; ii < len; ++ii)
                if (a2e_tab[pp[ii]] != '\0')
                    buf[newlen++] = a2e_tab[pp[ii]];
            if (newlen > 0)
            {
                // Insert the EBCDIC characters
                long addr = GetPos();
                if (overtype_)
                    GetDocument()->Change(mod_replace, addr, newlen, buf, 0, this);
                else
                    GetDocument()->Change(mod_insert, addr, newlen, buf, 0, this);
                SetSel(addr2pos(addr), addr2pos(addr+newlen));
                DisplayCaret();
            }
            else
            {
                ::HMessageBox("No valid EBCDIC characters to paste");
                aa->mac_error_ = 2;
            }
            delete[] buf;
        }
        else
        {
            ::HMessageBox("Text on clipboard is not valid ASCII text!");
            aa->mac_error_ = 10;    // Invalid text on clipboard?
        }
    }
    else
    {
        // Paste when nothing to paste, presumably in macro
        ::HMessageBox("There is nothing on the clipboard to paste");
        aa->mac_error_ = 10;
    }

    ::CloseClipboard();
    // This actually records even when there were some errors & probably shouldn't
    aa->SaveToMacro(km_paste);
}

void CHexEditView::OnPasteAscii() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    // Can't insert text if view is read only or in overtype mode
    if (check_ro("paste (ASCII)"))
        return;

    if (overtype_)
    {
        switch (::HMessageBox("Pasting in overtype mode will overwrite data!\r"
                          "Do you want to turn on insert mode?",
                          MB_YESNOCANCEL))
        {
        case IDYES:
            do_insert();
            break;
        case IDNO:
            break; /* do nothing */
        default:
            aa->mac_error_ = 5;
            return;
        }
    }

    num_entered_ = num_del_ = num_bs_ = 0;      // Stop any editing

    HANDLE hh;                                  // Handle to clipboard memory
    unsigned char *pp;                          // Pointer to actual data

    if (!OpenClipboard())
    {
        ::HMessageBox("The clipboard is in use!");
        aa->mac_error_ = 10;
        return;
    }

    CWaitCursor wait;                           // Turn on wait cursor (hourglass)

    if ((hh = ::GetClipboardData(CF_TEXT)) != NULL &&
        (pp = reinterpret_cast<unsigned char *>(::GlobalLock(hh))) != NULL)
    {
        size_t len = strlen(reinterpret_cast<char *>(pp));
        if (len > 0)                    // Bugs in other apps (eg Hedit) might cause len == 0
        {
            long addr = GetPos();
            if (overtype_)
                GetDocument()->Change(mod_replace, addr, len, pp, 0, this);
            else
                GetDocument()->Change(mod_insert, addr, len, pp, 0, this);
            SetSel(addr2pos(addr), addr2pos(addr+len));
            DisplayCaret();
        }
        else
            aa->mac_error_ = 20;
    }
    else
    {
        // Paste when nothing to paste, presumably in macro
        ::HMessageBox("There is nothing on the clipboard to paste");
        aa->mac_error_ = 10;
    }

    ::CloseClipboard();
  // This actually records even when there were some errors & probably shouldn't
    aa->SaveToMacro(km_paste_ascii);
}

void CHexEditView::OnPasteEbcdic() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    // Can't insert text if view is read only or in overtype mode
    if (check_ro("paste (EBCDIC)"))
        return;

    if (overtype_)
    {
        switch (::HMessageBox("Pasting in overtype mode will overwrite data!\r"
                          "Do you want to turn on insert mode?",
                          MB_YESNOCANCEL))
        {
        case IDYES:
            do_insert();
            break;
        case IDNO:
            break; /* do nothing */
        default:
            aa->mac_error_ = 5;
            return;
        }
    }

    num_entered_ = num_del_ = num_bs_ = 0;      // Stop any editing

    HANDLE hh;                                  // Handle to clipboard memory
    unsigned char *pp;                          // Pointer to actual data

    if (!OpenClipboard())
    {
        ::HMessageBox("The clipboard is in use!");
        aa->mac_error_ = 10;
        return;
    }

    CWaitCursor wait;                           // Turn on wait cursor (hourglass)

    if ((hh = ::GetClipboardData(CF_TEXT)) != NULL &&
        (pp = reinterpret_cast<unsigned char *>(::GlobalLock(hh))) != NULL)
    {
        size_t len = strlen(reinterpret_cast<char *>(pp));
        if (len > 0)                    // Bugs in other apps (eg Hedit) might cause len == 0
        {
            // Copy from clipboard to temp buffer converting to EBCDIC
            unsigned char *buf = new unsigned char[len];
            size_t newlen = 0;
            for (size_t ii = 0; ii < len; ++ii)
                if (pp[ii] < 128 && a2e_tab[pp[ii]] != '\0')
                    buf[newlen++] = a2e_tab[pp[ii]];
            if (newlen > 0)
            {
                // Insert the EBCDIC characters
                long addr = GetPos();
                if (overtype_)
                    GetDocument()->Change(mod_replace, addr, newlen, buf, 0, this);
                else
                    GetDocument()->Change(mod_insert, addr, newlen, buf, 0, this);
                SetSel(addr2pos(addr), addr2pos(addr+newlen));
                DisplayCaret();
            }
            delete[] buf;
        }
        else
            aa->mac_error_ = 20;
    }
    else
    {
        // Paste when nothing to paste, presumably in macro
        ::HMessageBox("There is nothing on the clipboard to paste");
        aa->mac_error_ = 10;
    }

    ::CloseClipboard();
    aa->SaveToMacro(km_paste_ebcdic);  // xxx don't record if error (mac_error_ > 3)???
}

// Update handler that turns on user interface options (Paste etc) if there is
// text on the clipboard -- ie. there is text that can be pasted into the document
void CHexEditView::OnUpdateTextPaste(CCmdUI* pCmdUI) 
{
    if (GetDocument()->read_only())
        pCmdUI->Enable(FALSE);                  // Disallow paste if file is read only
    else
        pCmdUI->Enable(::IsClipboardFormatAvailable(CF_TEXT));
}

void CHexEditView::OnPasteUnicode() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    // Can't insert text if view is read only or in overtype mode
    if (check_ro("paste (Unicode)"))
        return;

    if (overtype_)
    {
        switch (::HMessageBox("Pasting in overtype mode will overwrite data!\r"
                          "Do you want to turn on insert mode?",
                          MB_YESNOCANCEL))
        {
        case IDYES:
            do_insert();
            break;
        case IDNO:
            break; /* do nothing */
        default:
            aa->mac_error_ = 5;
            return;
        }
    }

    num_entered_ = num_del_ = num_bs_ = 0;      // Stop any editing

    HANDLE hh;                                  // Handle to clipboard memory
    wchar_t *pp;                                // Pointer to actual data

    if (!OpenClipboard())
    {
        ::HMessageBox("The clipboard is in use!");
        aa->mac_error_ = 10;
        return;
    }

    CWaitCursor wait;                           // Turn on wait cursor (hourglass)

    if ((hh = ::GetClipboardData(CF_UNICODETEXT)) != NULL &&
        (pp = reinterpret_cast<wchar_t *>(::GlobalLock(hh))) != NULL)
    {
        size_t len = wcslen(reinterpret_cast<wchar_t *>(pp));
        if (len > 0)                    // Bugs in other apps (eg Hedit) might cause len == 0
        {
            long addr = GetPos();
            if (overtype_)
                GetDocument()->Change(mod_replace, addr, 2*len, (unsigned char *)pp, 0, this);
            else
                GetDocument()->Change(mod_insert, addr, 2*len, (unsigned char *)pp, 0, this);
            SetSel(addr2pos(addr), addr2pos(addr + 2*len));
            DisplayCaret();
        }
        else
            aa->mac_error_ = 20;
    }
    else
    {
        // Paste when nothing to paste, presumably in macro
        ::HMessageBox("There is nothing on the clipboard to paste");
        aa->mac_error_ = 10;
    }

    ::CloseClipboard();
      // This actually records even when there were some errors & probably shouldn't
    aa->SaveToMacro(km_paste_unicode);
}

void CHexEditView::OnUpdateUnicodePaste(CCmdUI* pCmdUI) 
{
    if (GetDocument()->read_only())
        pCmdUI->Enable(FALSE);
    else
        pCmdUI->Enable(::IsClipboardFormatAvailable(CF_UNICODETEXT));
}

void CHexEditView::OnEditUndo() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    CWaitCursor wait;                           // Turn on wait cursor (hourglass)
    if (!aa->playing_ && GetFocus() != this) SetFocus(); // Ensure focus does not stay in DlgBar

    num_entered_ = num_del_ = num_bs_ = 0;      // Stop any editing

    BOOL more = TRUE;

    while (more)
    {
        ASSERT(undo_.size() > 0 || aa->playing_);
        more = undo_.back().previous_too;
        if (!do_undo())
        {
            aa->mac_error_ = 10;
            return;
        }
    }

    aa->SaveToMacro(km_undo);
}

BOOL CHexEditView::do_undo()
{
    // The string 'mess' is used to display a message in the status
    // bar if it is not obvious what has been undone.
    CString mess;
    CMainFrame *mm = dynamic_cast<CMainFrame *>(AfxGetMainWnd());
    CFont *tf;                          // Temp to hold ptr to previous font

    BOOL caret_displayed = CaretDisplayed();
    long start_addr, end_addr;
    BOOL end_base = GetSelAddr(start_addr, end_addr);

    if (undo_.size() == 0)
    {
        // This can only (presumably) happen during a macro
        ::HMessageBox("There is nothing to undo");
        return FALSE;
    }

    switch (undo_.back().utype)
    {
    case undo_move:
        GoAddress(undo_.back().address);
        show_prop();
        show_calc();
        show_pos();
        DisplayCaret();         // Make sure move visible
        break;
    case undo_sel:
        end_addr = undo_.back().address;

        // Now go back to previous undo (which should be undo_move)
        undo_.pop_back();
        ASSERT(undo_.size() > 0);
        ASSERT(undo_.back().utype == undo_move);
        GoAddress(undo_.back().address, end_addr);
        show_prop();
        show_calc();
        show_pos();
        DisplayCaret();         // Make sure move visible
        break;
    case undo_change:
#if 0
        if (readonly_)
        {
            if (::HMessageBox("You can't undo changes while the window is read only.\r"
                              "Do you want to turn off read only mode?",
                              MB_OKCANCEL) == IDCANCEL)
                return FALSE;
            else
                allow_mods();
        }
#endif
        // Note: flag == TRUE if this view originally made the change.
#ifndef NDEBUG
        if (!GetDocument()->Undo(this, undo_.back().index,
                                 undo_.back().flag))
            return FALSE;
#else
        if (!GetDocument()->Undo(this, 0, undo_.back().flag))
            return FALSE;
#endif
        if (!undo_.back().flag)
            mess += "Undo: changes made in different window undone ";
        break;
    case undo_swap:
        ASSERT(display_char_);
        edit_char_ = undo_.back().flag;
        SetSel(addr2pos(start_addr), addr2pos(end_addr));
        DisplayCaret();
        break;
    case undo_disp:
        display_char_ = undo_.back().flag;
        ASSERT(display_char_ || !edit_char_);

        tf = pfont_;
        pfont_ = new CFont;
        pfont_->CreateFontIndirect(display_char_ && !ebcdic_ && graphic_ && oem_ ? &oem_lf_ : &lf_);
        SetFont(pfont_);
        if (tf != NULL)
            delete tf;                  // Delete old font after it's deselected

        recalc_display();
        if (end_base)
            SetSel(addr2pos(end_addr), addr2pos(start_addr), true);
        else
            SetSel(addr2pos(start_addr), addr2pos(end_addr));
        if (caret_displayed)
            DisplayCaret();                     // Keep caret within display
        DoInvalidate();

        mm->bb_char_toggle_.LoadBitmaps(display_char_ ? IDB_CHAR_TOGGLES : IDB_CHAR_TOGGLEU,
                                        IDB_CHAR_TOGGLED,0,IDB_CHAR_TOGGLEX);
        mm->bb_char_toggle_.Invalidate();
        break;
    case undo_ebcdic:
        ebcdic_ = undo_.back().flag;

        tf = pfont_;
        pfont_ = new CFont;
        pfont_->CreateFontIndirect(display_char_ && !ebcdic_ && graphic_ && oem_ ? &oem_lf_ : &lf_);
        SetFont(pfont_);
        if (tf != NULL)
            delete tf;                  // Delete old font after it's deselected

        recalc_display();
        if (end_base)
            SetSel(addr2pos(end_addr), addr2pos(start_addr), true);
        else
            SetSel(addr2pos(start_addr), addr2pos(end_addr));
        if (caret_displayed)
            DisplayCaret();                     // Keep caret within display
        DoInvalidate();

        mm->bb_asc_ebc_.LoadBitmaps(ebcdic_ ? IDB_ASC_EBCS : IDB_ASC_EBCU,
                                    IDB_ASC_EBCD,0,IDB_ASC_EBCX);
        mm->bb_asc_ebc_.Invalidate();
        break;
    case undo_graphic:
        graphic_ = undo_.back().flag;

        tf = pfont_;
        pfont_ = new CFont;
        pfont_->CreateFontIndirect(display_char_ && !ebcdic_ && graphic_ && oem_ ? &oem_lf_ : &lf_);
        SetFont(pfont_);
        if (tf != NULL)
            delete tf;                  // Delete old font after it's deselected

        recalc_display();
        if (end_base)
            SetSel(addr2pos(end_addr), addr2pos(start_addr), true);
        else
            SetSel(addr2pos(start_addr), addr2pos(end_addr));
        if (caret_displayed)
            DisplayCaret();                     // Keep caret within display
        DoInvalidate();

        mm->bb_graphic_toggle_.LoadBitmaps(graphic_ ? IDB_GRAPHIC_TOGGLES : IDB_GRAPHIC_TOGGLEU,
                                            IDB_GRAPHIC_TOGGLED,0,IDB_GRAPHIC_TOGGLEX);
        mm->bb_graphic_toggle_.Invalidate();
        break;
    case undo_oem:
        oem_ = undo_.back().flag;

        tf = pfont_;
        pfont_ = new CFont;
        pfont_->CreateFontIndirect(display_char_ && !ebcdic_ && graphic_ && oem_ ? &oem_lf_ : &lf_);
        SetFont(pfont_);
        if (tf != NULL)
            delete tf;                  // Delete old font after it's deselected

        recalc_display();
        if (end_base)
            SetSel(addr2pos(end_addr), addr2pos(start_addr), true);
        else
            SetSel(addr2pos(start_addr), addr2pos(end_addr));
        if (caret_displayed)
            DisplayCaret();                     // Keep caret within display
        DoInvalidate();
        break;
    case undo_font:
        if (display_char_ && !ebcdic_ && graphic_ && oem_)
            oem_lf_ = *(undo_.back().plf);
        else
            lf_ = *(undo_.back().plf);

        tf = pfont_;
        pfont_ = new CFont;
        pfont_->CreateFontIndirect(display_char_ && !ebcdic_ && graphic_ && oem_ ? &oem_lf_ : &lf_);
        SetFont(pfont_);
        if (tf != NULL)
            delete tf;                  // Delete old font after it's deselected

        // Calculate new position based on new font size
        recalc_display();
        if (end_base)
            SetSel(addr2pos(end_addr), addr2pos(start_addr), true);
        else
            SetSel(addr2pos(start_addr), addr2pos(end_addr));
        if (caret_displayed)
            DisplayCaret();                     // Keep caret within display
        DoInvalidate();
        mess += "Undo: font restored ";
        break;
    case undo_rowsize:
        rowsize_ = undo_.back().rowsize;
        if (offset_ >= rowsize_)
            offset_ = rowsize_ - 1;
        recalc_display();
        if (end_base)
            SetSel(addr2pos(end_addr), addr2pos(start_addr), true);
        else
            SetSel(addr2pos(start_addr), addr2pos(end_addr));
        if (caret_displayed)
            DisplayCaret();     // Keep caret within display
        DoInvalidate();
        break;
    case undo_group_by:
        group_by_ = undo_.back().rowsize;
        recalc_display();
        if (end_base)
            SetSel(addr2pos(end_addr), addr2pos(start_addr), true);
        else
            SetSel(addr2pos(start_addr), addr2pos(end_addr));
        if (caret_displayed)
            DisplayCaret();     // Keep caret within display
        DoInvalidate();
        break;
    case undo_offset:
        real_offset_ = offset_ = undo_.back().rowsize;
        if (real_offset_ >= rowsize_)
            offset_ = rowsize_ - 1;
        recalc_display();
        if (end_base)
            SetSel(addr2pos(end_addr), addr2pos(start_addr), true);
        else
            SetSel(addr2pos(start_addr), addr2pos(end_addr));
        if (caret_displayed)
            DisplayCaret();     // Keep caret within display
        DoInvalidate();
        break;
    case undo_autofit:
        // If rowsize has been specified then autofit is now off (undo turn on)
        autofit_ = undo_.back().rowsize == 0;
//      autofit_ = !autofit_;
        if (!autofit_)
        {
            mess += "Undo: auto fit now OFF ";
            rowsize_ = undo_.back().rowsize;
            if (offset_ >= rowsize_)
                offset_ = rowsize_ - 1;
        }
        recalc_display();
        if (end_base)
            SetSel(addr2pos(end_addr), addr2pos(start_addr), true);
        else
            SetSel(addr2pos(start_addr), addr2pos(end_addr));
        if (caret_displayed)
            DisplayCaret();     // Keep caret within display
        DoInvalidate();

        mm->bb_autofit_.LoadBitmaps(autofit_ ? IDB_AUTOFITS : IDB_AUTOFITU,
                                    IDB_AUTOFITD,0,IDB_AUTOFITX);
        mm->bb_autofit_.Invalidate();
        break;
    case undo_setmark:
        invalidate_addr_range(mark_, mark_ + 1);
        mark_ = undo_.back().address;
        invalidate_addr_range(mark_, mark_ + 1);
        show_calc();                    // Status of some buttons may have changed when mark_ moves
        break;
    case undo_markchar:
        mark_char_ = undo_.back().flag;
        break;
    case undo_overtype:
        overtype_ = undo_.back().flag;
        if (GetDocument()->length() == 0)
            DoInvalidate();
        if (overtype_)
            mess += "Undo: overtype now ON ";
        else
            mess += "Undo: overtype now OFF ";
        break;
    case undo_readonly:
        readonly_ = undo_.back().flag;
        mm->bb_allow_mods_.LoadBitmaps(readonly_ ? IDB_MODSU : IDB_MODSS,
                                        IDB_MODSD,0,IDB_MODSX);
        mm->bb_allow_mods_.Invalidate();
        if (readonly_)
            mess += "Undo: read only now ON ";
        else
            mess += "Undo: read only now OFF ";
        break;
    case undo_unknown:
    default:
        ASSERT(0);
        mess += " Unknown undo! ";
    }
    undo_.pop_back();
    mm->StatusBarText(mess);
    return TRUE;
}

void CHexEditView::OnUpdateEditUndo(CCmdUI* pCmdUI) 
{
    pCmdUI->Enable(undo_.size() > 0);
}

void CHexEditView::OnAddrToggle()
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (!aa->playing_ && GetFocus() != this) SetFocus(); // Ensure focus does not stay in DlgBar

    num_entered_ = num_del_ = num_bs_ = 0;      // Stop any editing

    BOOL caret_displayed = CaretDisplayed();
//    long address = pos2addr(GetCaret());
    long start_addr, end_addr;
    GetSelAddr(start_addr, end_addr);

    dec_addr_ = !dec_addr_;
    recalc_display();
//    SetCaret(addr2pos(address));
    SetSel(addr2pos(start_addr), addr2pos(end_addr));
    if (caret_displayed)
        DisplayCaret();                 // Keep caret within display
    DoInvalidate();

    // Change the toolbar button
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
    mm->bb_addr_toggle_.LoadBitmaps(dec_addr_ ? IDB_ADDR_TOGGLES : IDB_ADDR_TOGGLEU,
                                    IDB_ADDR_TOGGLED,0,IDB_ADDR_TOGGLEX);
    mm->bb_addr_toggle_.Invalidate();
    aa->SaveToMacro(km_addr);
}

void CHexEditView::OnUpdateAddrToggle(CCmdUI* pCmdUI) 
{
    pCmdUI->SetCheck(dec_addr_);
}

void CHexEditView::OnGraphicToggle() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (!aa->playing_ && GetFocus() != this) SetFocus(); // Ensure focus does not stay in DlgBar
    if (!display_char_ || ebcdic_)
    {
        // Can't toggle graphic chars, presumably in macro playback
        ASSERT(aa->playing_);
        if (!display_char_)
            ::HMessageBox("Can't display graphic characters without the char area");
        else
            ::HMessageBox("Graphic characters not supported for EBCDIC");
        aa->mac_error_ = 2;
        return;
    }
    num_entered_ = num_del_ = num_bs_ = 0;      // Stop any editing

    graphic_ = !graphic_;

    BOOL caret_displayed = CaretDisplayed();
    long start_addr, end_addr;
    BOOL end_base = GetSelAddr(start_addr, end_addr);

    CFont *tf = pfont_;
    pfont_ = new CFont;
    pfont_->CreateFontIndirect(graphic_ && oem_ ? &oem_lf_ : &lf_);
    SetFont(pfont_);
    if (tf != NULL)
        delete tf;                      // Delete old font after it's deselected

    // Calculate new position (and new total size) based on change in font size
    recalc_display();
    if (end_base)
        SetSel(addr2pos(end_addr), addr2pos(start_addr), true);
    else
        SetSel(addr2pos(start_addr), addr2pos(end_addr));
    if (caret_displayed)
        DisplayCaret();                 // Keep caret within display
    DoInvalidate();

    // Change the toolbar button
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
    mm->bb_graphic_toggle_.LoadBitmaps(graphic_ ? IDB_GRAPHIC_TOGGLES : IDB_GRAPHIC_TOGGLEU,
                                        IDB_GRAPHIC_TOGGLED,0,IDB_GRAPHIC_TOGGLEX);
    mm->bb_graphic_toggle_.Invalidate();
    undo_.push_back(view_undo(undo_graphic));
    aa->SaveToMacro(km_graphic);
}

void CHexEditView::OnUpdateGraphicToggle(CCmdUI* pCmdUI) 
{
    pCmdUI->Enable(display_char_ && !ebcdic_);
    pCmdUI->SetCheck(graphic_);
}

void CHexEditView::OnCharToggle() 
{
    do_chartoggle();
}

void CHexEditView::do_chartoggle(int state /*=-1*/) 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (!aa->playing_ && GetFocus() != this) SetFocus(); // Ensure focus does not stay in DlgBar
    num_entered_ = num_del_ = num_bs_ = 0;      // Stop any editing

    BOOL caret_displayed = CaretDisplayed();
//    long address = pos2addr(GetCaret());
    long start_addr, end_addr;
    BOOL end_base = GetSelAddr(start_addr, end_addr);

    if (state == -1)
        display_char_ = !display_char_;
    else if ((BOOL)state == display_char_)
        return;                                 // No change - do nothing
    else
        display_char_ = (BOOL)state;
    if (!display_char_ && edit_char_)
    {
        // If we were displaying chars and caret was in char area
        undo_.push_back(view_undo(undo_swap));
        undo_.back().flag = TRUE;
        undo_.push_back(view_undo(undo_disp, TRUE));
        undo_.back().flag = TRUE;
        edit_char_ = FALSE;
    }
    else
    {
        undo_.push_back(view_undo(undo_disp));
        undo_.back().flag = !display_char_;
    }

    CFont *tf = pfont_;
    pfont_ = new CFont;
    pfont_->CreateFontIndirect(display_char_ && !ebcdic_ && graphic_ && oem_ ? &oem_lf_ : &lf_);
    SetFont(pfont_);
    if (tf != NULL)
        delete tf;                      // Delete old font after it's deselected

    // Calculate new position (and new total size) based on change in font size
    recalc_display();
    if (end_base)
        SetSel(addr2pos(end_addr), addr2pos(start_addr), true);
    else
        SetSel(addr2pos(start_addr), addr2pos(end_addr));
    if (caret_displayed)
        DisplayCaret();                 // Keep caret within display
    DoInvalidate();

    // Change the toolbar button
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
    mm->bb_char_toggle_.LoadBitmaps(display_char_ ? IDB_CHAR_TOGGLES : IDB_CHAR_TOGGLEU,
                                    IDB_CHAR_TOGGLED,0,IDB_CHAR_TOGGLEX);
    mm->bb_char_toggle_.Invalidate();
    aa->SaveToMacro(km_char_area, state);
}

void CHexEditView::OnUpdateCharToggle(CCmdUI* pCmdUI) 
{
    pCmdUI->SetCheck(display_char_);
}

void CHexEditView::OnOemToggle() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (!aa->playing_ && GetFocus() != this) SetFocus(); // Ensure focus does not stay in DlgBar
    if (!display_char_ || ebcdic_ || !graphic_)
    {
        // Can't toggle OEM chars, presumably in macro playback
        ASSERT(aa->playing_);
        if (!display_char_)
            ::HMessageBox("Can't display OEM/ANSI graphic characters without the char area");
        else if (ebcdic_)
            ::HMessageBox("Graphic characters not supported for EBCDIC");
        else
            ::HMessageBox("Can't toggle OEM/IBM chars when graphics chars not displayed");
        aa->mac_error_ = 2;
        return;
    }
    num_entered_ = num_del_ = num_bs_ = 0;      // Stop any editing

    oem_ = !oem_;

    BOOL caret_displayed = CaretDisplayed();
    long start_addr, end_addr;
    BOOL end_base = GetSelAddr(start_addr, end_addr);

    CFont *tf = pfont_;
    pfont_ = new CFont;
    pfont_->CreateFontIndirect(oem_ ? &oem_lf_ : &lf_);
    SetFont(pfont_);
    if (tf != NULL)
        delete tf;                      // Delete old font after it's deselected

    // Calculate new position (and new total size) based on change in font size
    recalc_display();
    if (end_base)
        SetSel(addr2pos(end_addr), addr2pos(start_addr), true);
    else
        SetSel(addr2pos(start_addr), addr2pos(end_addr));
    if (caret_displayed)
        DisplayCaret();                 // Keep caret within display
    DoInvalidate();

    undo_.push_back(view_undo(undo_oem));      // Allow undo of graphic display change
    aa->SaveToMacro(km_oem);
}

void CHexEditView::OnUpdateOemToggle(CCmdUI* pCmdUI) 
{
    pCmdUI->Enable(display_char_ && !ebcdic_ && graphic_);
    pCmdUI->SetCheck(oem_);
}

void CHexEditView::OnFontInc() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (!aa->playing_ && GetFocus() != this) SetFocus(); // Ensure focus does not stay in DlgBar
    num_entered_ = num_del_ = num_bs_ = 0;      // Stop any editing

    // Save font for undo
    // xxx this could be done with a size arg rather than a whole LOGFONT structure
    LOGFONT *plf = new LOGFONT;
    *plf = display_char_ && !ebcdic_ && graphic_ && oem_ ? oem_lf_ : lf_;
    for ( ; ; )
    {
        // Increase font size by one pixel
        CPoint convert(plf->lfWidth, plf->lfHeight);
        CClientDC dc(this);
        OnPrepareDC(&dc);
        dc.LPtoDP(&convert);
        if (convert.y < max_font_size)
            ++convert.y;
        else
        {
            ((CMainFrame *)AfxGetMainWnd())->
                StatusBarText("Warning: Font size too big - not increased");
            aa->mac_error_ = 2;
            return;
        }
        dc.DPtoLP(&convert);
        plf->lfHeight = convert.y;
        plf->lfWidth = 0;            // Calced from height

        CFont font;
        font.CreateFontIndirect(plf);
//        if (display_char_ && !ebcdic_ && graphic_ && oem_)
//        {
//            oem_lf_.lfHeight = convert.y;
//            oem_lf_.lfWidth = 0;                        // Calced from height
//            font.CreateFontIndirect(&oem_lf_);
//        }
//        else
//        {
//            lf_.lfHeight = convert.y;
//            lf_.lfWidth = 0;                        // Calced from height
//            font.CreateFontIndirect(&lf_);
//        }
        TEXTMETRIC tm;
        CFont *tf = dc.SelectObject(&font);
        dc.GetTextMetrics(&tm);
        dc.SelectObject(tf);                        // Deselect font before it is destroyed
        if (tm.tmHeight + tm.tmExternalLeading > text_height_)
        {
            if (display_char_ && !ebcdic_ && graphic_ && oem_)
                oem_lf_.lfHeight = convert.y;
            else
                lf_.lfHeight = convert.y;
            break;
        }
    }

    BOOL caret_displayed = CaretDisplayed();
//      long address = pos2addr(GetCaret());
    long start_addr, end_addr;
    BOOL end_base = GetSelAddr(start_addr, end_addr);

    // Create and install the new font
    CFont *tf = pfont_;
    pfont_ = new CFont;
    pfont_->CreateFontIndirect(display_char_ && !ebcdic_ && graphic_ && oem_ ? &oem_lf_ : &lf_);
    SetFont(pfont_);
    if (tf != NULL)
        delete tf;                      // Delete old font after it's deselected

    // Calculate new position (and new total size) based on change in font size
    recalc_display();
    if (end_base)
        SetSel(addr2pos(end_addr), addr2pos(start_addr), true);
    else
        SetSel(addr2pos(start_addr), addr2pos(end_addr));
    if (caret_displayed)
        DisplayCaret();                 // Keep caret within display
    DoInvalidate();
    undo_.push_back(view_undo(undo_font));      // Allow undo of font change
    undo_.back().plf = plf;
    aa->SaveToMacro(km_inc_font);
}

void CHexEditView::OnUpdateFontInc(CCmdUI* pCmdUI) 
{
    // Create a large (max_font_size) font and see if the current font is
    // displayed at the same size on screen.  If so we can't increase the size
    LOGFONT logfont;
    logfont = display_char_ && !ebcdic_ && graphic_ && oem_ ? oem_lf_ : lf_;              // Get font the same as current ...
    {
        CPoint convert(0, max_font_size);       // ... but very big
        CClientDC dc(this);
        OnPrepareDC(&dc);
        dc.DPtoLP(&convert);
        logfont.lfHeight = convert.y;
    }
    logfont.lfWidth = 0;                        // Width calced from height

    // Create font, put into DC, and see what size it would be on screen
    CFont font;
    font.CreateFontIndirect(&logfont);
    {
        TEXTMETRIC tm;
        CClientDC dc(this);
        OnPrepareDC(&dc);
        CFont *tf = dc.SelectObject(&font);
        dc.GetTextMetrics(&tm);
        dc.SelectObject(tf);
        if (tm.tmHeight + tm.tmExternalLeading == text_height_)
            pCmdUI->Enable(FALSE);              // Already at smallest displayable font
    }
}

void CHexEditView::OnFontDec() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (!aa->playing_ && GetFocus() != this) SetFocus(); // Ensure focus does not stay in DlgBar
    num_entered_ = num_del_ = num_bs_ = 0;      // Stop any editing

    // Save font for undo
    // xxx this could be done with a size arg rather than a whole LOGFONT structure
    LOGFONT *plf = new LOGFONT;
    *plf = display_char_ && !ebcdic_ && graphic_ && oem_ ? oem_lf_ : lf_;
    for ( ; ; )
    {
        // Decrease font size by one pixel
        CPoint convert(plf->lfWidth, plf->lfHeight);
        CClientDC dc(this);
        OnPrepareDC(&dc);
        dc.LPtoDP(&convert);
        if (convert.y > 1)
            convert.y--;
        else
        {
            ((CMainFrame *)AfxGetMainWnd())->
                StatusBarText("Warning: Font size already at minimum - not decreased");
            aa->mac_error_ = 2;
            return;
        }
        dc.DPtoLP(&convert);
        plf->lfHeight = convert.y;
        plf->lfWidth = 0;            // Calced from height

        CFont font;
        font.CreateFontIndirect(plf);
//        if (display_char_ && !ebcdic_ && graphic_ && oem_)
//        {
//            oem_lf_.lfHeight = convert.y;
//            oem_lf_.lfWidth = 0;                        // Calced from height
//            font.CreateFontIndirect(&oem_lf_);
//        }
//        else
//        {
//            lf_.lfHeight = convert.y;
//            lf_.lfWidth = 0;                        // Calced from height
//            font.CreateFontIndirect(&lf_);
//        }
        TEXTMETRIC tm;
        CFont *tf = dc.SelectObject(&font);
        dc.GetTextMetrics(&tm);
        dc.SelectObject(tf);                        // Deselect font before it is destroyed
        if (tm.tmHeight + tm.tmExternalLeading < text_height_)
        {
            if (display_char_ && !ebcdic_ && graphic_ && oem_)
                oem_lf_.lfHeight = convert.y;
            else
                lf_.lfHeight = convert.y;
            break;
        }
    }

    BOOL caret_displayed = CaretDisplayed();
//      long address = pos2addr(GetCaret());
    long start_addr, end_addr;
    BOOL end_base = GetSelAddr(start_addr, end_addr);

    // Create and install the new font
    CFont *tf = pfont_;
    pfont_ = new CFont;
    pfont_->CreateFontIndirect(display_char_ && !ebcdic_ && graphic_ && oem_ ? &oem_lf_ : &lf_);
    SetFont(pfont_);
    if (tf != NULL)
        delete tf;                      // Delete old font after it's deselected

    // Calculate new position (and new total size) based on change in font size
    recalc_display();
    if (end_base)
        SetSel(addr2pos(end_addr), addr2pos(start_addr), true);
    else
        SetSel(addr2pos(start_addr), addr2pos(end_addr));
    if (caret_displayed)
        DisplayCaret();                 // Keep caret within display
    DoInvalidate();
    undo_.push_back(view_undo(undo_font));      // Allow undo of font change
    undo_.back().plf = plf;
    aa->SaveToMacro(km_dec_font);
}

void CHexEditView::OnUpdateFontDec(CCmdUI* pCmdUI) 
{
    // If we create a very small font then see what the text height is when that
    // font is used.  If this height is the same as the current text height then the
    // font size can not be decreased any more.
    LOGFONT logfont;
    logfont = display_char_ && !ebcdic_ && graphic_ && oem_ ? oem_lf_ : lf_;              // Get font the same as current ...
    {
        CPoint convert(0, 1);                   // ... but just one pixel high
        CClientDC dc(this);
        OnPrepareDC(&dc);
        dc.DPtoLP(&convert);
        logfont.lfHeight = convert.y;
    }
    logfont.lfWidth = 0;                        // Width calced from height

    // Create font, put into DC, and see what size it would be on screen
    CFont font;
    font.CreateFontIndirect(&logfont);
    {
        TEXTMETRIC tm;
        CClientDC dc(this);
        OnPrepareDC(&dc);
        CFont *tf = dc.SelectObject(&font);
        dc.GetTextMetrics(&tm);
        dc.SelectObject(tf);
        if (tm.tmHeight + tm.tmExternalLeading == text_height_)
            pCmdUI->Enable(FALSE);              // Already at smallest displayable font
    }
}

void CHexEditView::OnFont() 
{
    num_entered_ = num_del_ = num_bs_ = 0;      // Stop any editing

    LOGFONT logfont = display_char_ && !ebcdic_ && graphic_ && oem_ ? oem_lf_ : lf_;

    // Convert font size to units that user can relate to
    {
        CPoint convert(logfont.lfWidth, logfont.lfHeight);
        CClientDC dc(this);
        OnPrepareDC(&dc);
        dc.LPtoDP(&convert);
        logfont.lfWidth = convert.x;
        logfont.lfHeight = convert.y;
    }

    CFontDialog dlg;
    dlg.m_cf.lpLogFont = &logfont;
    dlg.m_cf.Flags |= CF_INITTOLOGFONTSTRUCT |  CF_SELECTSCRIPT | CF_SHOWHELP;
    dlg.m_cf.Flags &= ~(CF_EFFECTS);              // Disable selection of strikethrough, colours etc
    if (dlg.DoModal() == IDOK)
    {
        // Convert font size back to logical units
        dlg.GetCurrentFont(&logfont);
        if (logfont.lfHeight < 0) logfont.lfHeight = -logfont.lfHeight;
        {
            CPoint convert(logfont.lfWidth, logfont.lfHeight);
            CClientDC dc(this);
            OnPrepareDC(&dc);
            dc.DPtoLP(&convert);
            logfont.lfWidth = convert.x;
            logfont.lfHeight = convert.y;
        }

        do_font(&logfont);
    }
    else
    {
        ((CHexEditApp *)AfxGetApp())->mac_error_ = 2;
    }

    // BG search finished message may be lost if modeless dlg running
    ((CHexEditApp *)AfxGetApp())->CheckBGSearchFinished();
}

void CHexEditView::do_font(LOGFONT *plf)
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();

    // Save current LOGFONT for undo
    LOGFONT *prev_lf = new LOGFONT;
    if (display_char_ && !ebcdic_ && graphic_ && oem_)
    {
        // We can't switch to an ANSI char set because we are displaying OEM graphics
        if (plf->lfCharSet == ANSI_CHARSET)
        {
            mm->StatusBarText("Can't switch to ANSI font when displaying IBM/OEM graphics chars");
            aa->mac_error_ = 2;
            return;
        }
        *prev_lf = oem_lf_;
        oem_lf_ = *plf;
    }
    else
    {
        // We can't switch to an OEM char set
        if (plf->lfCharSet == OEM_CHARSET)
        {
            mm->StatusBarText("Can't switch to this font unless displaying IBM/OEM graphics chars");
            aa->mac_error_ = 2;
            return;
        }
        *prev_lf = lf_;
        lf_ = *plf;
    }

    // Set new LOGFONT

    BOOL caret_displayed = CaretDisplayed();
    long start_addr, end_addr;
    BOOL end_base = GetSelAddr(start_addr, end_addr);

    // Create and install the new font
    CFont *tf = pfont_;
    pfont_ = new CFont;
    pfont_->CreateFontIndirect(display_char_ && !ebcdic_ && graphic_ && oem_ ? &oem_lf_ : &lf_);
    SetFont(pfont_);
    if (tf != NULL)
        delete tf;                      // Delete old font after it's deselected

    // Calculate new position (and new total size) based on change in font size
    recalc_display();
    if (end_base)
        SetSel(addr2pos(end_addr), addr2pos(start_addr), true);
    else
        SetSel(addr2pos(start_addr), addr2pos(end_addr));
    if (caret_displayed)
        DisplayCaret();                 // Keep caret within display
    DoInvalidate();
    undo_.push_back(view_undo(undo_font));      // Allow undo of font change
    undo_.back().plf = prev_lf;
    ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_font, 
                                display_char_ && !ebcdic_ && graphic_ && oem_ ? &oem_lf_ : &lf_);
}

void CHexEditView::change_rowsize(int rowsize)
{
    if (rowsize != rowsize_)
    {
        long start_addr, end_addr;
        BOOL end_base = GetSelAddr(start_addr, end_addr);

        undo_.push_back(view_undo(undo_rowsize));
        undo_.back().rowsize = rowsize_;            // Save previous rowsize for undo
        rowsize_ = rowsize;
        recalc_display();

        if (end_base)
            SetSel(addr2pos(end_addr), addr2pos(start_addr), true);
        else
            SetSel(addr2pos(start_addr), addr2pos(end_addr));
        DoInvalidate();
    }
    ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_rowsize, rowsize);
}

void CHexEditView::change_group_by(int group_by)
{
    if (group_by != group_by_)
    {
        long start_addr, end_addr;
        BOOL end_base = GetSelAddr(start_addr, end_addr);

        undo_.push_back(view_undo(undo_group_by));
        undo_.back().rowsize = group_by_;           // Save previous group_by for undo
        group_by_ = group_by;
        recalc_display();

        if (end_base)
            SetSel(addr2pos(end_addr), addr2pos(start_addr), true);
        else
            SetSel(addr2pos(start_addr), addr2pos(end_addr));
        DoInvalidate();
    }
    ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_group_by, group_by);
}

void CHexEditView::change_offset(int offset)
{
    if (offset != real_offset_)
    {
        long start_addr, end_addr;
        BOOL end_base = GetSelAddr(start_addr, end_addr);

        undo_.push_back(view_undo(undo_offset));
        undo_.back().rowsize = real_offset_;        // Save previous offset for undo
        real_offset_ = offset_ = offset;
        recalc_display();

        if (end_base)
            SetSel(addr2pos(end_addr), addr2pos(start_addr), true);
        else
            SetSel(addr2pos(start_addr), addr2pos(end_addr));
        DoInvalidate();
    }
    ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_offset, offset);
}

void CHexEditView::OnAutoFit() 
{
    do_autofit();
}

// Change autofit mode to state (0 or 1).  If state is -1 then toggle autofit.
void CHexEditView::do_autofit(int state /*=-1*/) 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();

    if (!aa->playing_ && GetFocus() != this) SetFocus(); // Ensure focus does not stay in DlgBar
    num_entered_ = num_del_ = num_bs_ = 0;      // Stop any editing

    BOOL caret_displayed = CaretDisplayed();
//    long address = pos2addr(GetCaret());
    long start_addr, end_addr;
    BOOL end_base = GetSelAddr(start_addr, end_addr);

    if (state == -1)
        autofit_ = !autofit_;
    else if ((BOOL)state == autofit_)
        return;                                 // No change - do nothing
    else
        autofit_ = (BOOL)state;
    undo_.push_back(view_undo(undo_autofit));
    if (autofit_)
        undo_.back().rowsize = rowsize_;
    else
        undo_.back().rowsize = 0;
    if (!autofit_ && real_offset_ != offset_)
    {
        // If autofit turned off but offset has been squeezed then save so it's undone
        undo_.push_back(view_undo(undo_offset, TRUE));
        undo_.back().rowsize = real_offset_;    // Save previous offset for undo
    }
    recalc_display();
//    SetCaret(addr2pos(address));
    if (end_base)
        SetSel(addr2pos(end_addr), addr2pos(start_addr), true);
    else
        SetSel(addr2pos(start_addr), addr2pos(end_addr));
    if (caret_displayed)
        DisplayCaret(); // Keep caret within display
    DoInvalidate();

    // Change the toolbar button
    mm->bb_autofit_.LoadBitmaps(autofit_ ? IDB_AUTOFITS : IDB_AUTOFITU,
                                IDB_AUTOFITD,0,IDB_AUTOFITX);
    mm->bb_autofit_.Invalidate();
    aa->SaveToMacro(km_autofit, state);
}

void CHexEditView::OnUpdateAutofit(CCmdUI* pCmdUI) 
{
    pCmdUI->SetCheck(autofit_);
}

void CHexEditView::OnAscEbc() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (!aa->playing_ && GetFocus() != this) SetFocus(); // Ensure focus does not stay in DlgBar
    if (!display_char_)
    {
        // Can't display EBCDIC, presumably in macro playback
        ASSERT(aa->playing_);
        ::HMessageBox("Can't display EBCDIC without the char area");
        aa->mac_error_ = 2;
        return;
    }
    num_entered_ = num_del_ = num_bs_ = 0;      // Stop any editing

    ebcdic_ = !ebcdic_;
    undo_.push_back(view_undo(undo_ebcdic));
    undo_.back().flag = !ebcdic_;

    BOOL caret_displayed = CaretDisplayed();
    long start_addr, end_addr;
    BOOL end_base = GetSelAddr(start_addr, end_addr);

    CFont *tf = pfont_;
    pfont_ = new CFont;
    pfont_->CreateFontIndirect(!ebcdic_ && graphic_ && oem_ ? &oem_lf_ : &lf_);
    SetFont(pfont_);
    if (tf != NULL)
        delete tf;                      // Delete old font after it's deselected

    // Calculate new position (and new total size) based on change in font size
    recalc_display();
    if (end_base)
        SetSel(addr2pos(end_addr), addr2pos(start_addr), true);
    else
        SetSel(addr2pos(start_addr), addr2pos(end_addr));
    if (caret_displayed)
        DisplayCaret();                 // Keep caret within display
    DoInvalidate();

    // Change the toolbar button
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
    mm->bb_asc_ebc_.LoadBitmaps(ebcdic_ ? IDB_ASC_EBCS : IDB_ASC_EBCU,
                                IDB_ASC_EBCD,0,IDB_ASC_EBCX);
    mm->bb_asc_ebc_.Invalidate();

    // Change search type in find modeless dlg if open
    if (mm->pfind_ != NULL)
        mm->pfind_->update_controls(-1, 1);
    aa->SaveToMacro(km_ebcdic);
}

void CHexEditView::OnUpdateAscEbc(CCmdUI* pCmdUI) 
{
    pCmdUI->Enable(display_char_);
    pCmdUI->SetCheck(ebcdic_);
}

void CHexEditView::OnControl() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (!aa->playing_ && GetFocus() != this) SetFocus(); // Ensure focus does not stay in DlgBar
    if (!display_char_ || ebcdic_)
    {
        // Can't toggle control chars, presumably in macro playback
        ASSERT(aa->playing_);
        if (!display_char_)
            ::HMessageBox("Can't display control characters without the char area");
        else
            ::HMessageBox("Control character display not supported for EBCDIC");
        aa->mac_error_ = 2;
        return;
    }
    num_entered_ = num_del_ = num_bs_ = 0;      // Stop any editing

    display_control_ = (display_control_ + 1)%3;
    recalc_display();
    DoInvalidate();

    // Change the toolbar button
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
    if (display_control_ == 1)
        mm->bb_control_.LoadBitmaps(IDB_CONTROL1S,IDB_CONTROLD,0,IDB_CONTROLX);
    else if (display_control_ == 2)
        mm->bb_control_.LoadBitmaps(IDB_CONTROL2S,IDB_CONTROLD,0,IDB_CONTROLX);
    else
        mm->bb_control_.LoadBitmaps(IDB_CONTROLU,IDB_CONTROLD,0,IDB_CONTROLX);
    mm->bb_control_.Invalidate();
    aa->SaveToMacro(km_control);
}

void CHexEditView::OnControlToggle() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (!display_char_ || ebcdic_)
    {
        // Can't toggle control chars, presumably in macro playback
        ASSERT(aa->playing_);
        if (!display_char_)
            ::HMessageBox("Can't display control characters without the char area");
        else
            ::HMessageBox("Control character display not supported for EBCDIC");
        aa->mac_error_ = 2;
        return;
    }

    // This is called as a result of menu item which has only 2 states (unlike
    // dialog bar button handled by OnControl() above which has 3)
    if (display_control_ > 0)
        display_control_ = 0;
    else
        display_control_ = 1;
    recalc_display();
    DoInvalidate();

    // Change the toolbar button to match
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
    if (display_control_ == 1)
        mm->bb_control_.LoadBitmaps(IDB_CONTROL1S,IDB_CONTROLD,0,IDB_CONTROLX);
    else
        mm->bb_control_.LoadBitmaps(IDB_CONTROLU,IDB_CONTROLD,0,IDB_CONTROLX);
    mm->bb_control_.Invalidate();
    aa->SaveToMacro(km_control2);
}

void CHexEditView::OnUpdateControl(CCmdUI* pCmdUI) 
{
    pCmdUI->Enable(display_char_ && !ebcdic_);
    pCmdUI->SetCheck(display_control_ != 0);
}

void CHexEditView::OnMark() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (!aa->playing_ && GetFocus() != this) SetFocus(); // Ensure focus does not stay in DlgBar

    SetMark(GetPos());
    aa->SaveToMacro(km_mark_pos);
}

void CHexEditView::SetMark(long new_mark)
{
    num_entered_ = num_del_ = num_bs_ = 0;      // Stop any editing

    // Invalidate where mark was to change colours
    invalidate_addr_range(mark_, mark_ + 1);

    // Save current mark and move mark_
    undo_.push_back(view_undo(undo_setmark));
    undo_.back().address = mark_;
    undo_.push_back(view_undo(undo_markchar, TRUE));
    undo_.back().flag = mark_char_;
    mark_ = new_mark;

    // Invalidate where mark now is to change background colour
    invalidate_addr_range(mark_, mark_ + 1);

    // Keep track whether mark is in hex or char area (if char area on)
    if (display_char_)
        mark_char_ = edit_char_;

    show_calc();                        // Some button enablement depends on mark_ position (eg. @ Mark)
}

void CHexEditView::OnGotoMark() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (!aa->playing_ && GetFocus() != this) SetFocus(); // Ensure focus does not stay in DlgBar
    num_entered_ = num_del_ = num_bs_ = 0;      // Stop any editing

    if (display_char_ && edit_char_ != mark_char_)
    {
        long start_addr, end_addr;
        GetSelAddr(start_addr, end_addr);
        edit_char_ = mark_char_;
        // We need to specify the previous address since otherwise MoveToAddress
        // will use the current selection which is now in the wrong area.
        MoveToAddress(mark_, mark_, start_addr, end_addr);
        undo_.push_back(view_undo(undo_swap, TRUE));
        undo_.back().flag = !edit_char_;
    }
    else
        MoveToAddress(mark_);
    aa->SaveToMacro(km_goto_mark);
}

void CHexEditView::OnExtendToMark() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (!aa->playing_ && GetFocus() != this) SetFocus(); // Ensure focus does not stay in DlgBar
    num_entered_ = num_del_ = num_bs_ = 0;      // Stop any editing

    long start_addr, end_addr;
    BOOL end_base = GetSelAddr(start_addr, end_addr);
    if ((end_base && mark_ == end_addr) ||
        (!end_base && mark_ == start_addr))
    {
        // There is nothing to do
        aa->mac_error_ = 1;
        return;
    }

    // Move the non-base end of the selection to the mark (MoveToAddress saves undo info)
    if (end_base)
        MoveToAddress(mark_, end_addr);
    else
        MoveToAddress(start_addr, mark_);

    aa->SaveToMacro(km_extendto_mark);
}

// Swap the current caret position with the mark
void CHexEditView::OnSwapMark() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (!aa->playing_ && GetFocus() != this) SetFocus(); // Ensure focus does not stay in DlgBar
    num_entered_ = num_del_ = num_bs_ = 0;      // Stop any editing

    long start_addr, end_addr;
    BOOL end_base = GetSelAddr(start_addr, end_addr);
    if (mark_ == start_addr && start_addr == end_addr)
    {
        // There is nothing to do
        aa->mac_error_ = 1;
        return;
    }

    // If the caret and the mark are in different areas we have to swap them
    if (display_char_ && edit_char_ != mark_char_)
    {
        edit_char_ = !edit_char_;
        mark_char_ = !mark_char_;

        // We need to specify the previous address since otherwise MoveToAddress
        // will use the current selection which is now in the wrong area since
        // edit_char_ has changed.
        MoveToAddress(mark_, mark_, start_addr, end_addr);
        undo_.push_back(view_undo(undo_swap, TRUE));
        undo_.back().flag = !edit_char_;
        undo_.push_back(view_undo(undo_markchar, TRUE));
        undo_.back().flag = !mark_char_;
    }
    else
        MoveToAddress(mark_);

    // Move the mark
    undo_.push_back(view_undo(undo_setmark, TRUE));     // save undo for move mark
    undo_.back().address = mark_;
    invalidate_addr_range(mark_, mark_ + 1);            // force undraw of mark
    mark_ = start_addr;
    invalidate_addr_range(mark_, mark_ + 1);            // force draw of new mark
    show_calc();

    aa->SaveToMacro(km_swap_mark);
}

void CHexEditView::OnJumpHexAddr()      // Alt-J
{
    num_entered_ = num_del_ = num_bs_ = 0;      // Stop any editing
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
    if ((mm->m_wndEditBar.GetStyle() & WS_VISIBLE) != 0)
        mm->m_wndEditBar.GetDlgItem(IDC_JUMP_HEX)->SetFocus();
}

void CHexEditView::OnInsert()
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (!aa->playing_ && GetFocus() != this) SetFocus(); // Ensure focus does not stay in DlgBar
    num_entered_ = num_del_ = num_bs_ = 0;

    do_insert();

    aa->SaveToMacro(km_ovr_ins);
}

void CHexEditView::do_insert()
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    if (readonly_)
    {
        ::HMessageBox("Attempt to toggle OVR/INS in read only mode");
        aa->mac_error_ = 10;
        return;
    }
    overtype_ = !overtype_;
    undo_.push_back(view_undo(undo_overtype));
    undo_.back().flag = !overtype_;
    if (GetDocument()->length() == 0)
    {
        DoInvalidate();
        if (overtype_)
            ScrollMode();
        else
        {
            CaretMode();
            SetCaret(addr2pos(0));
        }
    }
}

void CHexEditView::OnUpdateInsert(CCmdUI* pCmdUI) 
{
    pCmdUI->Enable(!readonly_);
    pCmdUI->SetCheck(!overtype_);
}

void CHexEditView::OnAllowMods() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (!aa->playing_ && GetFocus() != this) SetFocus(); // Ensure focus does not stay in DlgBar
    num_entered_ = num_del_ = num_bs_ = 0;

    allow_mods();

    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();

    // Make sure calc buttons reflect modifiability of file
    ASSERT(mm->pcalc_ != NULL);
    if (!aa->refresh_off_ && mm->pcalc_->m_hWnd != 0 && mm->pcalc_->visible_)
        mm->pcalc_->FixFileButtons();

    // Change the toolbar button
    mm->bb_allow_mods_.LoadBitmaps(readonly_ ? IDB_MODSU : IDB_MODSS,
                                    IDB_MODSD,0,IDB_MODSX);
    mm->bb_allow_mods_.Invalidate();
    aa->SaveToMacro(km_ro_rw);
}

void CHexEditView::allow_mods() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    if (readonly_ && GetDocument()->read_only())
    {
        ::HMessageBox("This file cannot be modified");
        aa->mac_error_ = 10;
        return;
    }
    readonly_ = !readonly_;
    undo_.push_back(view_undo(undo_readonly));
    undo_.back().flag = !readonly_;
}

void CHexEditView::OnUpdateAllowMods(CCmdUI* pCmdUI) 
{
    pCmdUI->Enable(!GetDocument()->read_only());
    pCmdUI->SetCheck(!readonly_);
}

void CHexEditView::OnSearchHex()        // Alt-L, F6
{
//    num_entered_ = num_del_ = num_bs_ = 0;    // Stop any editing
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();

    if ((mm->m_wndEditBar.GetStyle() & WS_VISIBLE) != 0)
    {
        mm->sec_search_.SetMode(CSearchEditControl::mode_hex);
        mm->m_wndEditBar.GetDlgItem(IDC_SEARCH)->SetFocus();
    }
}

void CHexEditView::OnSearchAscii()      // F5
{
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
    if ((mm->m_wndEditBar.GetStyle() & WS_VISIBLE) != 0)
    {
        mm->sec_search_.SetMode(CSearchEditControl::mode_char);
        mm->m_wndEditBar.GetDlgItem(IDC_SEARCH)->SetFocus();
    }
}


void CHexEditView::OnSearchIcase()      // F4
{
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
    if ((mm->m_wndEditBar.GetStyle() & WS_VISIBLE) != 0)
    {
        mm->sec_search_.SetMode(CSearchEditControl::mode_icase);
        mm->m_wndEditBar.GetDlgItem(IDC_SEARCH)->SetFocus();
    }
}

void CHexEditView::OnSearchForw() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (!aa->playing_ && GetFocus() != this) SetFocus(); // Ensure focus does not stay in DlgBar

    if (do_search())
        aa->SaveToMacro(km_find_forw);
}

void CHexEditView::OnSearchBack() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (!aa->playing_ && GetFocus() != this) SetFocus(); // Ensure focus does not stay in DlgBar

    if (do_search(FALSE))
        aa->SaveToMacro(km_find_back);
}

// Search for the next occurrence of the current selection
void CHexEditView::OnSearchSel() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    // Load the current selection into buf.  Allow for each byte to be stored 
    // in buf plus a leading '=' and trailing '\0', but there may be less bytes
    // if non-EBCDIC chars are found (EBCDIC mode) or chars < 13 (ASCII mode)
    long start, end;
    GetSelAddr(start, end);
    ASSERT(start >= 0 && start <= end && end <= GetDocument()->length());
    if (start == end)
    {
        // Nothing selected, presumably in macro playback
        ASSERT(aa->playing_);
        ::HMessageBox("Nothing selected to search for!");
        aa->mac_error_ = 10;
        return;
    }

#if 0
    if (end - start > GetDocument()->length() - (start + 1))
    {
        ::HMessageBox(  "Search failed: Current selection is\r"
                        "larger than distance to end of file.");
        aa->mac_error_ = 10;
        return;
    }
#endif

    unsigned char *buf = new unsigned char[end - start + 2];
    VERIFY(GetDocument()->GetData(buf+1, end - start, start) == end - start);

    buf[0] = '\0';                      // Default to hex search
    if (edit_char_ && ebcdic_)
    {
        unsigned char *pp;              // Pointer into buf
        buf[0] = '=';
        buf[end-start+1] = '\0';
        // Check if all chars are valid EBCDIC and set search text if so
        for (pp = buf+1; pp < buf+end-start+1; ++pp)
            if (e2a_tab[*pp] == '\0')
                buf[0] = '\0';          // Not all hex (use hex search)
        // Convert the EBCDIC characters to ASCII
        for (pp = buf+1; pp < buf+end-start+1; ++pp)
            *pp = e2a_tab[*pp];
        ASSERT(buf[end-start+1] == '\0');       // Check for buffer overrun
    }
    else if (edit_char_)
    {
        buf[0] = '=';
        buf[end-start+1] = '\0';
        // Check if all chars are normal ASCII text and set search text
        for (unsigned char *pp = buf+1; pp < buf+end-start+1; ++pp)
            if (*pp <= '\r')
                buf[0] = '\0';          // Not all ASCII (use hex search)
        ASSERT(buf[end-start+1] == '\0');       // Check for buffer overrun
    }

    // buf[0] == '=' if buf now contains a valid character search string
    // buf[0] == '\0' if an invalid character was detected so do hex search
    if (buf[0] == '\0')
    {
        delete[] buf;
        // Change buf so that there's room for 2 hex digits + space per byte
        buf = new unsigned char[(end - start)*3];
        unsigned char cc;               // One character from document
        long address;                   // Current byte address in document
        unsigned char *pp;              // Current position in search string being built

        const char *hex;
        CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
        if (aa->hex_ucase_)
            hex = "0123456789ABCDEF?";
        else
            hex = "0123456789abcdef?";

        ASSERT(start < end);            // Make sure we don't put '\0' at buf[-1]
        for (address = start, pp = buf; address < end; ++address)
        {
            GetDocument()->GetData(&cc, 1, address);
            *pp++ = hex[(cc>>4)&0xF];
            *pp++ = hex[cc&0xF];
            *pp++ = ' ';
        }
        *(pp-1) = '\0';                 // Terminate string (overwrite last space)
        ASSERT(pp == buf + (end-start)*3);
        ASSERT(address == end);
    }

    ASSERT(buf[0] != '\0');

    // Add the current selection to the search edit control (where OnSearchForw looks)
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();

    mm->sec_search_.SetWindowText((char *)buf);
    delete[] buf;

    if (do_search())
        aa->SaveToMacro(km_find_sel);
}

BOOL CHexEditView::do_search(BOOL forward /*=TRUE*/)
{
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
    CString ss;         // String to hold text of edit box

    mm->sec_search_.GetWindowText(ss);

    // Error if there is nothing to search for:
    // no hex digits AND (length < 2 OR not a text search)
    if ((ss.GetLength() < 2 || 
         ss[0] != CSearchEditControl::sflag_char && ss[0] != CSearchEditControl::iflag_char) &&
        ss.FindOneOf("0123456789ABCDEFabcdef") == -1)
    {
        ::HMessageBox("There is no search text");
        ((CHexEditApp *)AfxGetApp())->mac_error_ = 10;
        return FALSE;
    }
    // Check if user was entering text and pressed Find Forward/Back button
    if (mm->sec_search_.in_edit_)
    {
        // Flag that the text is to be kept when focus lost.
        mm->sec_search_.in_edit_ = FALSE;
        // Save entered search bytes if recording a macro
        ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_find_text, ss);
    }
    mm->AddSearchHistory(ss);
    BOOL retval = Search(ss, ebcdic_ ? 3 : 1, forward);

    // Change search type in find modeless dlg if open
    if (mm->pfind_ != NULL)
        mm->pfind_->update_controls(forward);
    return retval;
}

void CHexEditView::OnUpdateSearch(CCmdUI* pCmdUI) 
{
    CMainFrame *mm = (CMainFrame *)AfxGetMainWnd();
    CString ss;         // String to hold text of edit box

    mm->sec_search_.GetWindowText(ss);
    // Enable if there is something to search for (some chars but
    // not just a single = or ~ OR some hex digits)
//    pCmdUI->Enable( ss.GetLength() > 0 &&
//                  (((ss[0] == CSearchEditControl::sflag_char ||
//                     ss[0] == CSearchEditControl::iflag_char) &&
//                    ss.GetLength() > 1) ||
//                   ss.FindOneOf("0123456789ABCDEFabcdef") > -1) );
    pCmdUI->Enable(ss.GetLength() > 1 &&
                    (ss[0] == CSearchEditControl::sflag_char ||
                     ss[0] == CSearchEditControl::iflag_char   ) ||
                   ss.FindOneOf("0123456789ABCDEFabcdef") > -1);
}

// Search for sequence of bytes from the current position
// 'entered' points to an ASCII string containing hex digits or text
//   - if first char is sflag_char (=) search for following characters
//   - if first char is iflag_char (~) do case-insensitive search
//   - otherwise assume hex digits - convert to bytes and search for that
// 'tt' indicates (for text searches) the character set to search using
//   - 0 for EBCDIC
//   - 1 for ASCII
//   - 2 for Unicode
//   - 3 for EBCDIC
// 'down' is true to search forwards, false to search backwards
// Returns TRUE if the search succeeds, FALSE on failure or error (like empty search sequence)
BOOL CHexEditView::Search(const char *entered, int tt, BOOL down)
{
    const unsigned char *ss;            // Ptr to actual search bytes
    unsigned char *tmp_ss = NULL;       // Search bytes converted from hex or EBCDIC
    size_t length = strlen(entered);
    BOOL icase;                         // Ignore case in search?
    BOOL bad_chars = FALSE;             // Invalid EBCDIC chars in search string?

    icase = (entered[0] == CSearchEditControl::iflag_char);

    if (tt == 3 && (entered[0] == CSearchEditControl::sflag_char ||
                    entered[0] == CSearchEditControl::iflag_char) )
    {
        // EBCDIC search
        const char *pp;
        unsigned char *dd;
        char *aa;

        // Convert string to EBCDIC bytes and store in tmp_ss
        tmp_ss = new unsigned char[length];             // hex bytes to search for
        char *tmp_ascii = new char[length];     // actual string as ASCII
        for (pp = &entered[1], dd = tmp_ss, aa = tmp_ascii; *pp != '\0'; ++pp)
        {
            if (*pp < 128 && *pp >= 0 && a2e_tab[*pp] != '\0')
            {
                *dd++ = a2e_tab[*pp];
                *aa++ = *pp;
            }
            else
                bad_chars = TRUE;
        }
        *dd = '\0';
        *aa = '\0';
        length = dd - tmp_ss;

        ss = tmp_ss;                    // Search using these bytes

        if (bad_chars)
        {
            CString mess;
            mess.Format("Invalid EBCDIC search characters ignored\r"
                        "searching for \"%s\"", tmp_ascii);
            ::HMessageBox(mess);
            ((CHexEditApp *)AfxGetApp())->mac_error_ = 1;
        }
        delete[] tmp_ascii;
    }
    else if (tt == 2 && (entered[0] == CSearchEditControl::sflag_char ||
                         entered[0] == CSearchEditControl::iflag_char) )
    {
        // Unicode search
        const char *pp;                 // Pointer to source ASCI text string
        unsigned char *dd;              // Pointer into search bytes created
        char *aa;                       // Copy of actual search bytes as ASCII

        // Convert string to Unicode bytes and store in tmp_ss
        tmp_ss = new unsigned char[length*2];   // hex bytes to search for
        char *tmp_ascii = new char[length];     // actual string as ASCII
        for (pp = &entered[1], dd = tmp_ss, aa = tmp_ascii; *pp != '\0'; ++pp)
        {
            wchar_t ww;                 // One Unicode (16 bit) character

            if (mbtowc(&ww, pp, 1) == 1)
            {
                *dd++ = ww & 0xFF;
                *dd++ = (ww>>8) & 0xFF;
                *aa++ = *pp;
            }
            else
                bad_chars = TRUE;
        }
        *dd = '\0';
        *aa = '\0';
        length = dd - tmp_ss;

        ss = tmp_ss;                    // Search using these bytes

        if (bad_chars)
        {
            CString mess;
            mess.Format("Invalid Unicode search characters ignored\r"
                        "searching for \"%s\"", tmp_ascii);
            ::HMessageBox(mess);
            ((CHexEditApp *)AfxGetApp())->mac_error_ = 1;
        }
        delete[] tmp_ascii;
    }
    else if (entered[0] == CSearchEditControl::sflag_char ||
             entered[0] == CSearchEditControl::iflag_char)
    {
        // Start search using the ASCII string after iflag/sflag_char
        ss = reinterpret_cast<const unsigned char *>(&entered[1]);
        --length;
    }
    else
    {
        // Convert hex characters to bytes and store in tmp_ss
        tmp_ss = new unsigned char[length/2+2];

        const char *pp;
        unsigned char *dd;
        for (pp = entered, dd = tmp_ss, length = 0, *dd = '\0'; *pp != '\0'; ++pp)
        {
            // Ignore spaces (and anything else)
            if (!isxdigit(*pp))
                continue;

            *dd = (*dd<<4) + (isdigit(*pp) ? *pp - '0' : toupper(*pp) - 'A' + 10);

            if ((++length % 2) == 0)
                *(++dd) = '\0';         // Move to next byte and zero it
        }

        ss = tmp_ss;                    // Start search using these bytes
        if (length > 0)
            length = (length-1)/2 + 1;  // Convert nybble count to byte count

        icase = FALSE;
    }

    BOOL success;
    {
        CWaitCursor wait;               // Cursor restored by destructor (end of block)
        if (down)
            success = search_forw(ss, length, icase, tt);
        else
            success = search_back(ss, length, icase, tt);

        if (tmp_ss != NULL)
            delete[] tmp_ss;
    }

    if (!success)
    {
        ((CHexEditApp *)AfxGetApp())->mac_error_ = 10;
        return FALSE;
    }

    // Cursor has moved to correct address but not nec. to correct area
    if (entered[0] != CSearchEditControl::sflag_char &&
        entered[0] != CSearchEditControl::iflag_char)
    {
        // If currently in char area ...
        if (edit_char_)
        {
            // Move to hex area (since hex search done) and allow undo of this
            undo_.push_back(view_undo(undo_swap, TRUE));
            undo_.back().flag = TRUE;

            // Swap to hex area
            long start_addr, end_addr;
            GetSelAddr(start_addr, end_addr);
            edit_char_ = FALSE;
            SetSel(addr2pos(start_addr), addr2pos(end_addr));
            DisplayCaret();
        }
    }
    else if (display_char_ && !edit_char_)
    {
        // char search so move to char area (if displayed and currently in hex area)
        undo_.push_back(view_undo(undo_swap, TRUE));
        undo_.back().flag = FALSE;

        // Swap to char area
        long start_addr, end_addr;
        GetSelAddr(start_addr, end_addr);
        edit_char_ = TRUE;
        SetSel(addr2pos(start_addr), addr2pos(end_addr));
        DisplayCaret();
    }
    return TRUE;
}

BOOL CHexEditView::search_forw(const unsigned char *ss, size_t length,
                               BOOL icase, int tt)
{
    CHexEditDoc *pdoc = GetDocument();

    // Work out where to start the search
    FILE_ADDRESS start_addr, end_addr;
    GetSelAddr(start_addr, end_addr);
    if (start_addr < end_addr)
        ++start_addr;                   // Change to start search at byte after start of selection

    FILE_ADDRESS bg_next = pdoc->GetNextFound(ss, length, icase, tt, start_addr);
    if (bg_next > -1)
    {
        // Found
        MoveToAddress(bg_next, bg_next + length);
        return TRUE;
    }
    else if (bg_next == -1)
    {
        // There are none
        if (icase && tt == 3)
            ::HMessageBox("Search text not found (case-insensitive EBCDIC search)");
        else if (tt == 3)
            ::HMessageBox("Search text not found (forward EBCDIC search)");
        else if (icase && tt == 2)
            ::HMessageBox("Search text not found (case insensitive Unicode search)");
        else if (tt == 2)
            ::HMessageBox("Search text not found (forward Unicode search)");
        else if (icase)
            ::HMessageBox("Search text not found (case insensitive search)");
        else
            ::HMessageBox("Search text not found (forward search)");

        return FALSE;
    }

    size_t buf_len = min(pdoc->length(), search_buf_len + length - 1);

    // Warn of simple problems
    if (length == 0)
    {
        ::HMessageBox("Empty search sequence");
        return FALSE;
    }
    else if (length > pdoc->length())
    {
        ::HMessageBox("Search byte sequence is longer than the file");
        return FALSE;
    }
    else if (length > buf_len)
    {
        ::HMessageBox("Search byte sequence is too long");
        return FALSE;
    }

    if (bg_next == -3)
    {
        // New search so signal to stop any current bg search
        pdoc->StopSearch();
    }

    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    unsigned char *buf = new unsigned char[buf_len];
    boyer bb(ss, length);       // Boyer-Moore searcher
    FILE_ADDRESS addr_buf = start_addr; // Current location in doc of start of buf
    long show_inc = 0x20000;    // How far between showing addresses
    long next_show;             // Next address to show to user
    long slow_show;             // Slow show update rate when we get here
    size_t got;                 // How many bytes just read

    // Are there enough bytes before EOF for a search?
    if (addr_buf + length <= pdoc->length())
    {
        got = pdoc->GetData(buf, length - 1, addr_buf);
        ASSERT(got == length - 1);      // xxx file length may change?
        next_show = (addr_buf/show_inc + 1)*show_inc;
        slow_show = ((addr_buf+0x800000)/0x800000 + 1)*0x800000;
        while (1)
        {
            unsigned char *pp;  // Where search string's found (or NULL if not)

            if (addr_buf > next_show)
            {
                MSG msg;

                // Do any redrawing, but nothing else
                while (::PeekMessage(&msg, NULL, WM_PAINT, WM_PAINT, PM_NOREMOVE))
                {
                    if (::GetMessage(&msg, NULL, 0, 0))
                    {
                        // Process any events for the window so that we can tell if
                        // the timer has finished or if the user clicks the Stop button
                        ::TranslateMessage(&msg);
                        ::DispatchMessage(&msg);
                    }
                }

                // Check if Escape has been pressed
                if (::PeekMessage(&msg, NULL, WM_KEYDOWN, WM_KEYDOWN, PM_NOREMOVE))
                {
                    VERIFY(GetMessage(&msg, NULL, WM_KEYDOWN, WM_KEYDOWN) > 0);
                    if (msg.wParam == 27 &&
                        ::HMessageBox("Abort search?", MB_YESNO) == IDYES)
                    {
                        delete[] buf;
                        // Start bg search anyway
                        aa->NewSearch(ss, length, icase, tt);
                        if (bg_next == -3)
                        {
                            pdoc->StartSearch(start_addr, addr_buf);
                            aa->StartSearches(pdoc);
                        }
                        ((CMainFrame *)AfxGetMainWnd())
                            ->StatusBarText("Search aborted");
                        aa->mac_error_ = 10;
                        show_pos();
                        return FALSE;
                    }
                }

                // Show search progress
                show_pos(next_show);
                if (next_show >= slow_show)
                    show_inc = 0x200000;
                next_show += show_inc;
            }

            // Get the next buffer full and search it
            got = pdoc->GetData(buf + length - 1, buf_len - (length - 1),
                                addr_buf + length - 1);
            if ((pp = bb.find(buf, length - 1 + got, icase, tt)) != NULL)
            {
                delete[] buf;
                // Start bg search to search the rest of the file
                aa->NewSearch(ss, length, icase, tt);
                if (bg_next == -3)
                {
                    pdoc->StartSearch(start_addr, addr_buf + (pp - buf));
                    aa->StartSearches(pdoc);
                }
                MoveToAddress(addr_buf + (pp - buf), addr_buf + (pp - buf) + length);
                return TRUE;
            }
            addr_buf += got;

            // Check if we're at EOF yet
            if (addr_buf + length - 1 >= pdoc->length())
            {
                ASSERT(addr_buf + length - 1 == pdoc->length());
                break;
            }

            // Move a little bit from the end to the start of the buffer
            // so that we don't miss sequences that overlap the pieces read
            memmove(buf, buf + buf_len - (length - 1), length - 1);
        }
    }
    delete[] buf;

    // Start bg search to search the rest of the file
    aa->NewSearch(ss, length, icase, tt);
    if (bg_next == -3)
    {
        pdoc->StartSearch(start_addr, -1);  // Start bg search (start_addr->EOF already done)
        aa->StartSearches(pdoc);
    }

    show_pos(pdoc->length());
    if (icase && tt == 3)
        ::HMessageBox("Search text not found (case-insensitive EBCDIC search)");
    else if (tt == 3)
        ::HMessageBox("Search text not found (forward EBCDIC search)");
    else if (icase && tt == 2)
        ::HMessageBox("Search text not found (case insensitive Unicode search)");
    else if (tt == 2)
        ::HMessageBox("Search text not found (forward Unicode search)");
    else if (icase)
        ::HMessageBox("Search text not found (case insensitive search)");
    else
        ::HMessageBox("Search text not found (forward search)");

    show_pos();
    return FALSE;
}

BOOL CHexEditView::search_back(const unsigned char *ss, size_t length,
                               BOOL icase, int tt)
{
    CHexEditDoc *pdoc = GetDocument();

    // Work out where to start the search
    FILE_ADDRESS start_addr, end_addr;
    GetSelAddr(start_addr, end_addr);
    if (start_addr < end_addr)
        end_addr--;                   // Change to start search at byte before end of selection

    FILE_ADDRESS bg_next = pdoc->GetPrevFound(ss, length, icase, tt, end_addr - length);
    if (bg_next > -1)
    {
        // Found
        MoveToAddress(bg_next, bg_next + length);
        return TRUE;
    }
    else if (bg_next == -1)
    {
        // There are none
        if (icase && tt == 3)
            ::HMessageBox("Search text not found (insensitive backward EBCDIC search)");
        else if (tt == 3)
            ::HMessageBox("Search text not found (backward EBCDIC search)");
        else if (icase && tt == 2)
            ::HMessageBox("Search text not found (insensitive backward Unicode search)");
        else if (tt == 2)
            ::HMessageBox("Search text not found (backward Unicode search)");
        else if (icase)
            ::HMessageBox("Search text not found (case insensitive backward search)");
        else
            ::HMessageBox("Search text not found (backward search)");

        return FALSE;
    }

    size_t buf_len = min(pdoc->length(), search_buf_len + length - 1);

    // Warn of simple problems
    if (length == 0)
    {
        ::HMessageBox("Empty search sequence");
        return FALSE;
    }
    else if (length > pdoc->length())
    {
        ::HMessageBox("Search byte sequence is longer than the file");
        return FALSE;
    }
    else if (length > buf_len)
    {
        ::HMessageBox("Search byte sequence is too long");
        return FALSE;
    }

    if (bg_next == -3)
    {
        // New search so signal to stop any current bg search
        pdoc->StopSearch();
    }

    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    unsigned char *buf = new unsigned char[buf_len];
    boyer bb(ss, length);       // Boyer-Moore searcher
    FILE_ADDRESS addr_buf = end_addr; // Current location in doc of end of buf
    long next_show;             // Next address to show to user
    const long show_inc = 0x200000; // How far between showing addresses
    size_t got;                 // How many bytes just read

    // If dist. to start of file > length of search bytes
    if (addr_buf >= length)
    {
        next_show = (addr_buf/show_inc)*show_inc;
        while (1)
        {
            unsigned char *pp;  // Where search string's found (or NULL if not)

            if (addr_buf < next_show)
            {
                MSG msg;

                // Do any redrawing, but nothing else
                while (::PeekMessage(&msg, NULL, WM_PAINT, WM_PAINT, PM_NOREMOVE))
                {
                    if (::GetMessage(&msg, NULL, 0, 0))
                    {
                        // Process any events for the window so that we can tell if
                        // the timer has finished or if the user clicks the Stop button
                        ::TranslateMessage(&msg);
                        ::DispatchMessage(&msg);
                    }
                }

                // Check if Escape has been pressed
                if (::PeekMessage(&msg, NULL, WM_KEYDOWN, WM_KEYDOWN, PM_NOREMOVE))
                {
                    VERIFY(GetMessage(&msg, NULL, WM_KEYDOWN, WM_KEYDOWN) > 0);
                    if (msg.wParam == 27 &&
                        ::HMessageBox("Abort search?", MB_YESNO) == IDYES)
                    {
                        delete[] buf;
                        aa->NewSearch(ss, length, icase, tt);
                        if (bg_next == -3)
                        {
                            pdoc->StartSearch(addr_buf - (length-1), end_addr - (length-1));
                            aa->StartSearches(pdoc);
                        }
                        ((CMainFrame *)AfxGetMainWnd())
                            ->StatusBarText("Search aborted");
                        aa->mac_error_ = 10;
                        show_pos();
                        return FALSE;
                    }
                }

                // It's time to display our progress
                show_pos(next_show);
                next_show -= show_inc;
            }

            // Note that search_back uses a slightly different algorithm to search_forw,
            // in that for search_forw the buffer overlap area (so that search strings
            // are not missed that overlap reads) is kept using memmove() whereas
            // search_back reads the same bit of the file twice.  This is simpler
            // esp. for backward searches but may sometimes be slower.

            // Check if less than a buffer full before start of file
            if (addr_buf < buf_len)
            {
                // Just get what's left
                got = pdoc->GetData(buf, addr_buf, 0L);
                ASSERT(got == addr_buf);
            }
            else
            {
                // Get a buffer full (including overlap with previous get)
                got = pdoc->GetData(buf, buf_len, addr_buf - buf_len);
                ASSERT(got == buf_len);
            }
            if ((pp = bb.findback(buf, got, icase, tt)) != NULL)
            {
                delete[] buf;
                // Start bg search to search the rest of the file
                aa->NewSearch(ss, length, icase, tt);
                if (bg_next == -3)
                {
                    pdoc->StartSearch(addr_buf - got + (pp - buf) + 1, end_addr - (length-1));
                    aa->StartSearches(pdoc);
                }
                MoveToAddress(addr_buf - got + (pp - buf),
                              addr_buf - got + (pp - buf) + length);
                return TRUE;
            }
            addr_buf -= got - (length - 1);

            // Check if we're at start of file yet
            if (addr_buf < length)
            {
                ASSERT(addr_buf == length - 1);
                break;
            }
        }
    }
    delete[] buf;

    // Start bg search to search the rest of the file
    aa->NewSearch(ss, length, icase, tt);
    if (bg_next == -3)
    {
        pdoc->StartSearch(0, end_addr - (length-1));
        aa->StartSearches(pdoc);
    }

    show_pos(0L);
    if (icase && tt == 3)
        ::HMessageBox("Search text not found (insensitive backward EBCDIC search)");
    else if (tt == 3)
        ::HMessageBox("Search text not found (backward EBCDIC search)");
    else if (icase && tt == 2)
        ::HMessageBox("Search text not found (insensitive backward Unicode search)");
    else if (tt == 2)
        ::HMessageBox("Search text not found (backward Unicode search)");
    else if (icase)
        ::HMessageBox("Search text not found (case insensitive backward search)");
    else
        ::HMessageBox("Search text not found (backward search)");

    show_pos();
    return FALSE;
}

// 
CMDIChildWnd *CHexEditView::comp_window()
{
    CMainFrame *mm = dynamic_cast<CMainFrame *>(AfxGetMainWnd());
    CMDIChildWnd *nextc;        // Loops through all MDI child frames
    CMDIChildWnd *compc = NULL; // Frame of view available to compare with
    BOOL got_one;               // Have we gound an appropriate view?

    // Get the currently active child MDI window
    nextc = dynamic_cast<CMDIChildWnd *>(mm->MDIGetActive());
    ASSERT(nextc != NULL);
    ASSERT(nextc->GetActiveView() == this);

    // Search for another (non-iconized) window
    for (got_one = FALSE, nextc = dynamic_cast<CMDIChildWnd *>(nextc->GetWindow(GW_HWNDNEXT));
         nextc != NULL; nextc = dynamic_cast<CMDIChildWnd *>(nextc->GetWindow(GW_HWNDNEXT)) )
    {
        if (!nextc->IsIconic())
        {
            // More than one found - use which one?
            if (got_one)
            {
// Can't display message when used in OnUpdateEditCompare call
//              ::HMessageBox("Comparison not performed\r"
//                              "- more than two (non-iconized) windows");
                return NULL;
            }
            got_one = TRUE;
            compc = nextc;
        }
    }

    // If we didn't find a non-iconized window search for iconized one
    if (compc == NULL)
    {
        nextc = dynamic_cast<CMDIChildWnd *>(mm->MDIGetActive());

        // Search for an iconized window
        for (got_one = FALSE, nextc = dynamic_cast<CMDIChildWnd *>(nextc->GetWindow(GW_HWNDNEXT));
             nextc != NULL; nextc = dynamic_cast<CMDIChildWnd *>(nextc->GetWindow(GW_HWNDNEXT)) )
        {
            ASSERT(nextc->IsIconic());          // else compc != NULL
            // If more than one window - use which one?
            if (got_one)
            {
// Can't display message when used in OnUpdateEditCompare call
//              ::HMessageBox("Comparison not performed\r"
//                              "- more than two windows found");
                return NULL;
            }
            got_one = TRUE;
            compc = nextc;
        }
    }
    return compc;
}

void CHexEditView::OnEditCompare() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    CMainFrame *mm = dynamic_cast<CMainFrame *>(AfxGetMainWnd());
    CMDIChildWnd *origc;        // Pointer to MDI child frame of current view

    // Save the currently active child MDI window
    origc = dynamic_cast<CMDIChildWnd *>(mm->MDIGetActive());
    ASSERT(origc != NULL);
    ASSERT(origc->GetActiveView() == this);

    // Restore current window if iconized
    if (origc->IsIconic())
    {
        WINDOWPLACEMENT wp;
        origc->GetWindowPlacement(&wp);
        wp.showCmd = SW_RESTORE;
        origc->SetWindowPlacement(&wp);
    }

    // Get MDI child frame of compare window
    CMDIChildWnd *compc = comp_window();

    // If we found nothing to compare with, display message and return
    if (compc == NULL)
    {
        ::HMessageBox("Comparison not performed\r"
                        "- two non-minimized windows required");
        aa->mac_error_ = 10;
        return;
    }

    // Get view to compare with - active view of compc.  (Actually the
    // active view should be the only view since we don't have splitters.)
    CHexEditView *compv = dynamic_cast<CHexEditView *>(compc->GetActiveView());
    if (compv == NULL)
    {
        ::HMessageBox("Cannot compare with window in print preview mode");
        aa->mac_error_ = 10;
        return;
    }
    ASSERT_KINDOF(CHexEditView, compv);

    CString orig_title, comp_title;     // Title of the windows to be compared
    origc->GetWindowText(orig_title);
    compc->GetWindowText(comp_title);

    CHexEditDoc *origd, *compd;         // Documents of the compared views
    origd = GetDocument();
    compd = compv->GetDocument();

    // Now compare the data from each view starting at END of current selection
    CString mess;                       // Message for user when problem encountered

    long dummy;                         // Not used - start address of selection
    long start_addr;                    // Start address in current view
    long orig_addr, comp_addr;          // Current comp location in both views
    GetSelAddr(dummy, start_addr);
    orig_addr = start_addr;
    compv->GetSelAddr(dummy, comp_addr);
//    start_addr = orig_addr = GetPos();
//    comp_addr = compv->pos2addr(compv->GetCaret());

#ifndef _DEBUG  // Allow self-compare for testing purposes
    // If same doc and same address then we aren't doing anything useful
    if (origd == compd && orig_addr == comp_addr)
    {
        mess.Format("Comparing data with itself in windows\r%s and %s",
                        (const char *)orig_title, (const char *)comp_title);
        ::HMessageBox(mess);
        aa->mac_error_ = 10;
        return;
    }
#endif

    size_t orig_got, comp_got;          // How many bytes obtained from docs

    long show_inc = 0x20000;            // How far between showing addresses
    long next_show = (orig_addr/show_inc + 1)*show_inc; // Next address to show
    long slow_show = ((orig_addr+0x800000)/0x800000 + 1)*0x800000;      // When we slow showing

    // Get memory for compare buffers
    unsigned char *orig_buf = new unsigned char[compare_buf_len];
    unsigned char *comp_buf = new unsigned char[compare_buf_len];

    CWaitCursor wait;
    aa->SaveToMacro(km_compare);

    while (1)
    {
        if (orig_addr > next_show)
        {
            MSG msg;

            // Do any redrawing, but nothing else
            while (::PeekMessage(&msg, NULL, WM_PAINT, WM_PAINT, PM_NOREMOVE))
            {
                if (::GetMessage(&msg, NULL, 0, 0))
                {
                    // Process any events for the window so that we can tell if
                    // the timer has finished or if the user clicks the Stop button
                    ::TranslateMessage(&msg);
                    ::DispatchMessage(&msg);
                }
            }

            // Check if Escape has been pressed
            if (::PeekMessage(&msg, NULL, WM_KEYDOWN, WM_KEYDOWN, PM_NOREMOVE))
            {
                VERIFY(GetMessage(&msg, NULL, WM_KEYDOWN, WM_KEYDOWN) > 0);
                if (msg.wParam == 27 &&
                    ::HMessageBox("Abort comparison?", MB_YESNO) == IDYES)
                {
                    delete[] orig_buf;
                    delete[] comp_buf;
                    ((CMainFrame *)AfxGetMainWnd())
                        ->StatusBarText("Comparison aborted");
                    aa->mac_error_ = 10;
                    show_pos();
                    return;
                }
            }

            show_pos(next_show);
            // If we've been showing the current address for awhile then show
            // less often to avoid slowing down the actual compare
            if (next_show >= slow_show)
                show_inc = 0x200000;
            next_show += show_inc;
        }

        orig_got = origd->GetData(orig_buf, compare_buf_len, orig_addr);
        comp_got = compd->GetData(comp_buf, compare_buf_len, comp_addr);

        size_t comp_len = min(orig_got, comp_got);
        if (comp_len == 0)              // EOF of one or both files
            break;

        if (memcmp(orig_buf, comp_buf, comp_len) != 0)
        {
            // Difference found
            for (size_t pos = 0; pos < comp_len; ++pos)
                if (orig_buf[pos] != comp_buf[pos])
                    break;
            ASSERT(pos < comp_len);

//          mess.Format("Difference found after $%lX (decimal %ld) bytes\r"
//                      "%s at address $%lX (decimal %ld)\r%s at address $%lX (decimal %ld)",
//                      orig_addr + pos - start_addr, orig_addr + pos - start_addr,
//                      orig_title, orig_addr + pos, orig_addr + pos,
//                      comp_title, comp_addr + pos, comp_addr + pos);
//          ::HMessageBox(mess);
            mess.Format("Difference found after %ld bytes", orig_addr + pos - start_addr);
            mm->StatusBarText(mess);

            // Move to where diff found and select that byte in both views
            // (Selecting the byte allows the differences to be seen in both
            // windows without flipping between them, and allows another
            // compare immediately starting at the byte after.)
            compv->MoveToAddress(comp_addr + pos, comp_addr + pos + 1);
            MoveToAddress(orig_addr + pos, orig_addr + pos + 1);
            delete[] orig_buf;
            delete[] comp_buf;
            return;
        }

        if (orig_got != comp_got)
            break;

        orig_addr += orig_got;
        comp_addr += comp_got;
    }

    // If we got here then we hit EOF on one or both files
    if (orig_got == comp_got)
    {
        // EOF (both files) encountered
        ASSERT(orig_got == 0);
        show_pos(orig_addr);

        // Display message box when no differences found so that user sees that
        // something happened -- the caret is not moved and they may not notice
        // a message in the status bar (or the status bar may be invisible).
        // (On the other hand if a difference is found then the caret of both views
        // is moved and we just display a message in the status bar -- putting up 
        // a dialog would just be annoying.)

        mess.Format("No differences found\r"
                    "after %lX (hex) or\r"
                    "%ld (decimal) bytes.",
                    orig_addr + orig_got - start_addr,
                    orig_addr + orig_got - start_addr);
        if (aa->playing_)
            mm->StatusBarText(mess);
        else
            ::HMessageBox(mess);
        aa->mac_error_ = 1;
    }
    else if (orig_got < comp_got)
    {
        // EOF on orig file before EOF on comp file
//      mess.Format("Difference found after $%lX (decimal %ld) bytes\r"
//                  "%s at EOF - address $%lX (decimal %ld)\r%s at address $%lX (decimal %ld)",
//                  orig_addr + orig_got - start_addr, orig_addr + orig_got - start_addr,
//                  orig_title, orig_addr + orig_got, orig_addr + orig_got,
//                  comp_title, comp_addr + orig_got, comp_addr + orig_got);
//      ::HMessageBox(mess);
        mess.Format("EOF on \"%s\" after %ld bytes", orig_title, orig_addr + orig_got - start_addr);
        mm->StatusBarText(mess);

        compv->MoveToAddress(comp_addr + orig_got, comp_addr + orig_got + 1);
        MoveToAddress(orig_addr + orig_got);
    }
    else
    {
        // EOF on comp file before EOF on orig file
//      mess.Format("Difference found after $%lX (decimal %ld) bytes\r"
//                  "%s at address $%lX (decimal %ld)\r%s at EOF - address $%lX (decimal %ld)",
//                  orig_addr + comp_got - start_addr, orig_addr + comp_got - start_addr,
//                  orig_title, orig_addr + comp_got, orig_addr + comp_got,
//                  comp_title, comp_addr + comp_got, comp_addr + comp_got);
//      ::HMessageBox(mess);
        mess.Format("EOF on \"%s\" after %ld bytes", comp_title, orig_addr + comp_got - start_addr);
        mm->StatusBarText(mess);

        compv->MoveToAddress(comp_addr + comp_got);
        MoveToAddress(orig_addr + comp_got, orig_addr + comp_got + 1);
    }
    delete[] orig_buf;
    delete[] comp_buf;
}

void CHexEditView::OnUpdateEditCompare(CCmdUI* pCmdUI) 
{
    pCmdUI->Enable(comp_window() != NULL);
}

// Activate next window
void CHexEditView::OnWindowNext() 
{
    // Activate next window that is not a print preview window and is not minimized
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    CMainFrame *mm = dynamic_cast<CMainFrame *>(::AfxGetMainWnd());
    CMDIChildWnd *currc;        // Currently active MDI child frame
    CMDIChildWnd *nextc;        // Loops through all MDI child frames

    // Save the currently active child MDI window
    currc = dynamic_cast<CMDIChildWnd *>(mm->MDIGetActive());
    ASSERT(currc != NULL);
    ASSERT(currc->GetActiveView() == this);

    while (mm->MDINext(), (nextc = dynamic_cast<CMDIChildWnd *>(mm->MDIGetActive())) != currc)
    {
        // Don't change to iconized windows
        if (!nextc->IsIconic())
        {
            // Make sure it's a CHexEditView (don't change to print preview windows)
            CHexEditView *pview = dynamic_cast<CHexEditView *>(nextc->GetActiveView());
            if (pview != NULL && pview->IsKindOf(RUNTIME_CLASS(CHexEditView)))
            {
                if (aa->recording_ && aa->mac_.size() > 0 && (aa->mac_.back()).ktype == km_focus)
                {
                    // We don't want focus change recorded (see CHexEditView::OnSetFocus)
                    aa->mac_.pop_back();
                }
                aa->SaveToMacro(km_win_next);
                return;
            }
        }
    }
    ((CMainFrame *)AfxGetMainWnd())->
        StatusBarText("Warning: no other non-minimized windows found");
    aa->mac_error_ = 2;
}

// The following functions are used as update handlers
// They enable their corresponding UI handlers if there are
// enough bytes between the current address and the end of file
void CHexEditView::OnUpdateByte(CCmdUI* pCmdUI) 
{
    pCmdUI->Enable(GetPos() < GetDocument()->length());
}

void CHexEditView::OnUpdate16bit(CCmdUI* pCmdUI) 
{
    long start_addr, end_addr;              // Current selection
    GetSelAddr(start_addr, end_addr);
    if (start_addr == end_addr)
        // No selection so only enable if enough before EOF (2 bytes)
        pCmdUI->Enable(start_addr + 1 < GetDocument()->length());
    else
        // Else make sure selection is multiple of 2
        pCmdUI->Enable((end_addr - start_addr)%2 == 0);
}

void CHexEditView::OnUpdate32bit(CCmdUI* pCmdUI) 
{
//    pCmdUI->Enable(GetPos() + 3 < GetDocument()->length());
    long start_addr, end_addr;              // Current selection
    GetSelAddr(start_addr, end_addr);
    if (start_addr == end_addr)
        // No selection so only enable if enough before EOF (4 bytes)
        pCmdUI->Enable(start_addr + 3 < GetDocument()->length());
    else
        // Else make sure selection is multiple of 4
        pCmdUI->Enable((end_addr - start_addr)%4 == 0);
}

void CHexEditView::OnUpdate64bit(CCmdUI* pCmdUI) 
{
//    pCmdUI->Enable(GetPos() + 7 < GetDocument()->length());
    long start_addr, end_addr;              // Current selection
    GetSelAddr(start_addr, end_addr);
    if (start_addr == end_addr)
        // No selection so only enable if enough before EOF (4 bytes)
        pCmdUI->Enable(start_addr + 7 < GetDocument()->length());
    else
        // Else make sure selection is multiple of 8
        pCmdUI->Enable((end_addr - start_addr)%8 == 0);
}

void CHexEditView::OnIncByte() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (check_ro("increment bytes"))
        return;

    // Get current address or selection
    long start_addr, end_addr;          // Start and end of selection
    size_t len;                         // Length of selection
    GetSelAddr(start_addr, end_addr);
    len = size_t(end_addr - start_addr);

    if (start_addr >= GetDocument()->length())
    {
        // At EOF, presumably in macro playback
        ASSERT(aa->playing_);
        ::HMessageBox("Can't increment byte while at end of file");
        aa->mac_error_ = 10;
        return;
    }

    unsigned char cc;                           // Current byte value
    ASSERT(start_addr + len <= GetDocument()->length());
    if (len == 0)
    {
        size_t got = GetDocument()->GetData(&cc, 1, start_addr);
        ASSERT(got == 1);

        // Increment byte value and make change
        ++cc;
        if (cc == 0)
        {
            ((CMainFrame *)AfxGetMainWnd())->StatusBarText("Warning: Increment wrapped around to zero");
            aa->mac_error_ = 1;            // Signal warning on wrap
        }
        GetDocument()->Change(mod_replace, start_addr, 1, &cc, 0, this);
    }
    else
    {
        // Get memory for selection and read it from the document
        unsigned char *buf = new unsigned char[len];
        size_t got = GetDocument()->GetData(buf, len, start_addr);
        ASSERT(got == len);

        // Flip bytes around and make change
        for (size_t ii = 0; ii < len; ++ii)
        {
            ++buf[ii];
        }
        GetDocument()->Change(mod_replace, start_addr, len, buf, 0, this);
        delete[] buf;
    }
    DisplayCaret();
    aa->SaveToMacro(km_inc8);
}

void CHexEditView::OnInc16bit() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (check_ro("increment words"))
        return;

    // Get current address or selection
    long start_addr, end_addr;          // Start and end of selection
    size_t len;                         // Length of selection
    GetSelAddr(start_addr, end_addr);
    len = size_t(end_addr - start_addr);

    if (len == 0 && start_addr + 1 >= GetDocument()->length())
    {
        // Not enough bytes before EOF, presumably in macro playback
        ASSERT(aa->playing_);
        ::HMessageBox("Insufficient bytes before EOF to increment word");
        aa->mac_error_ = 10;
        return;
    }
    else if (len > 0 && len%2 != 0)
    {
        // Selection not a multiple of 2
        ASSERT(aa->playing_);
        ::HMessageBox("Selection must have an even number of bytes to increment words");
        aa->mac_error_ = 10;
        return;
    }

    // If no selection just flip one word
    if (len == 0)
    {
        end_addr = start_addr + 2;
        len = 2;
    }
    ASSERT(end_addr - start_addr == len);
    ASSERT(start_addr + len <= GetDocument()->length());

    // Get memory for bytes to be flipped and read them from the document
    unsigned short *buf = new unsigned short[len/2];
    size_t got = GetDocument()->GetData((unsigned char *)buf, len, start_addr);
    ASSERT(got == len);

    // Flip bytes around and make change
    for (size_t ii = 0; ii < len/2; ++ii)
    {
        // Increment the word
        ++buf[ii];
    }
    if (len == 2 && buf[0] == 0)
    {
        ((CMainFrame *)AfxGetMainWnd())->StatusBarText("Warning: Increment wrapped around to zero");
        aa->mac_error_ = 1;            // Signal warning on wrap
    }
    GetDocument()->Change(mod_replace, start_addr, len, (unsigned char *)buf, 0, this);
    delete[] buf;

    DisplayCaret();
    aa->SaveToMacro(km_inc16);
}

void CHexEditView::OnInc32bit() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (check_ro("increment double words"))
        return;

    // Get current address or selection
    long start_addr, end_addr;          // Start and end of selection
    size_t len;                         // Length of selection
    GetSelAddr(start_addr, end_addr);
    len = size_t(end_addr - start_addr);

    if (len == 0 && start_addr + 3 >= GetDocument()->length())
    {
        // Not enough bytes before EOF, presumably in macro playback
        ASSERT(aa->playing_);
        ::HMessageBox("Insufficient bytes before EOF to increment double word");
        aa->mac_error_ = 10;
        return;
    }
    else if (len > 0 && len%4 != 0)
    {
        // Selection not a multiple of 4
        ASSERT(aa->playing_);
        ::HMessageBox("Selection must be multiple of 4 for increment double words");
        aa->mac_error_ = 10;
        return;
    }

    // If no selection just flip one word
    if (len == 0)
    {
        end_addr = start_addr + 4;
        len = 4;
    }
    ASSERT(end_addr - start_addr == len);
    ASSERT(start_addr + len <= GetDocument()->length());

    // Get memory for bytes to be flipped and read them from the document
    unsigned long *buf = new unsigned long[len/4];
    size_t got = GetDocument()->GetData((unsigned char *)buf, len, start_addr);
    ASSERT(got == len);

    // Flip bytes around and make change
    for (size_t ii = 0; ii < len/4; ++ii)
    {
        // Increment the word
        ++buf[ii];
    }
    if (len == 4 && buf[0] == 0)
    {
        ((CMainFrame *)AfxGetMainWnd())->StatusBarText("Warning: Increment wrapped around to zero");
        aa->mac_error_ = 1;            // Signal warning on wrap
    }
    GetDocument()->Change(mod_replace, start_addr, len, (unsigned char *)buf, 0, this);
    delete[] buf;

    DisplayCaret();
    aa->SaveToMacro(km_inc32);
}

void CHexEditView::OnInc64bit() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (check_ro("increment quad words"))
        return;

    // Get current address or selection
    long start_addr, end_addr;          // Start and end of selection
    size_t len;                         // Length of selection
    GetSelAddr(start_addr, end_addr);
    len = size_t(end_addr - start_addr);

    if (len == 0 && start_addr + 7 >= GetDocument()->length())
    {
        // Not enough bytes before EOF, presumably in macro playback
        ASSERT(aa->playing_);
        ::HMessageBox("Insufficient bytes before EOF to increment quad word");
        aa->mac_error_ = 10;
        return;
    }
    else if (len > 0 && len%8 != 0)
    {
        // Selection not a multiple of 8
        ASSERT(aa->playing_);
        ::HMessageBox("Selection must be multiple of 8 for increment quad words");
        aa->mac_error_ = 10;
        return;
    }

    // If no selection just flip one word
    if (len == 0)
    {
        end_addr = start_addr + 8;
        len = 8;
    }
    ASSERT(end_addr - start_addr == len);
    ASSERT(start_addr + len <= GetDocument()->length());

    // Get memory for bytes to be flipped and read them from the document
    unsigned __int64 *buf = new unsigned __int64[len/8];
    size_t got = GetDocument()->GetData((unsigned char *)buf, len, start_addr);
    ASSERT(got == len);

    // Flip bytes around and make change
    for (size_t ii = 0; ii < len/8; ++ii)
    {
        // Increment the quad word
        ++buf[ii];
    }
    if (len == 8 && buf[0] == 0)
    {
        ((CMainFrame *)AfxGetMainWnd())->StatusBarText("Warning: Increment wrapped around to zero");
        aa->mac_error_ = 1;            // Signal warning on wrap
    }
    GetDocument()->Change(mod_replace, start_addr, len, (unsigned char *)buf, 0, this);
    delete[] buf;

    DisplayCaret();
    aa->SaveToMacro(km_inc64);
}

void CHexEditView::OnDecByte() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (check_ro("decrement bytes"))
        return;
    // Get current address or selection
    long start_addr, end_addr;          // Start and end of selection
    size_t len;                         // Length of selection
    GetSelAddr(start_addr, end_addr);
    len = size_t(end_addr - start_addr);

    if (start_addr >= GetDocument()->length())
    {
        // At EOF, presumably in macro playback
        ASSERT(aa->playing_);
        ::HMessageBox("Can't decrement byte while at end of file"); ////
        aa->mac_error_ = 10;
        return;
    }

    unsigned char cc;                           // Current byte value
    ASSERT(start_addr + len <= GetDocument()->length());
    if (len == 0)
    {
        size_t got = GetDocument()->GetData(&cc, 1, start_addr);
        ASSERT(got == 1);

        // Decrement byte value and make change
        if (cc == 0)
        {
            ((CMainFrame *)AfxGetMainWnd())->StatusBarText("Warning: Decrement wrapped around past zero");
            aa->mac_error_ = 1;            // Signal warning on wrap
        }
        cc--;
        GetDocument()->Change(mod_replace, start_addr, 1, &cc, 0, this);
    }
    else
    {
        // Get memory for selection and read it from the document
        unsigned char *buf = new unsigned char[len];
        size_t got = GetDocument()->GetData(buf, len, start_addr);
        ASSERT(got == len);

        // Flip bytes around and make change
        for (size_t ii = 0; ii < len; ++ii)
        {
            buf[ii]--;
        }
        GetDocument()->Change(mod_replace, start_addr, len, buf, 0, this);
        delete[] buf;
    }

    DisplayCaret();
    aa->SaveToMacro(km_dec8);
}

void CHexEditView::OnDec16bit() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (check_ro("decrement words"))
        return;
    // Get current address or selection
    long start_addr, end_addr;          // Start and end of selection
    size_t len;                         // Length of selection
    GetSelAddr(start_addr, end_addr);
    len = size_t(end_addr - start_addr);

    if (len == 0 && start_addr + 1 >= GetDocument()->length())
    {
        // Not enough bytes before EOF, presumably in macro playback
        ASSERT(aa->playing_);
        ::HMessageBox("Insufficient bytes before EOF to decrement word");
        aa->mac_error_ = 10;
        return;
    }
    else if (len > 0 && len%2 != 0)
    {
        // Selection not a multiple of 2
        ASSERT(aa->playing_);
        ::HMessageBox("Selection must have an even number of bytes to decrement words");
        aa->mac_error_ = 10;
        return;
    }

    // If no selection just flip one word
    if (len == 0)
    {
        end_addr = start_addr + 2;
        len = 2;
    }
    ASSERT(end_addr - start_addr == len);
    ASSERT(start_addr + len <= GetDocument()->length());

    // Get memory for bytes to be flipped and read them from the document
    unsigned short *buf = new unsigned short[len/2];
    size_t got = GetDocument()->GetData((unsigned char *)buf, len, start_addr);
    ASSERT(got == len);

    if (len == 2 && buf[0] == 0)
    {
        ((CMainFrame *)AfxGetMainWnd())->StatusBarText("Warning: Decrement wrapped around past zero");
        aa->mac_error_ = 1;            // Signal warning on wrap
    }

    // Flip bytes around and make change
    for (size_t ii = 0; ii < len/2; ++ii)
    {
        // Decrement the word
        buf[ii]--;
    }
    GetDocument()->Change(mod_replace, start_addr, len, (unsigned char *)buf, 0, this);
    delete[] buf;

    DisplayCaret();
    aa->SaveToMacro(km_dec16);
}

void CHexEditView::OnDec32bit() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (check_ro("decrement double words"))
        return;
    // Get current address or selection
    long start_addr, end_addr;          // Start and end of selection
    size_t len;                         // Length of selection
    GetSelAddr(start_addr, end_addr);
    len = size_t(end_addr - start_addr);

    if (len == 0 && start_addr + 3 >= GetDocument()->length())
    {
        // Not enough bytes before EOF, presumably in macro playback
        ASSERT(aa->playing_);
        ::HMessageBox("Insufficient bytes before EOF to decrement double word");
        aa->mac_error_ = 10;
        return;
    }
    else if (len > 0 && len%4 != 0)
    {
        // Selection not a multiple of 4
        ASSERT(aa->playing_);
        ::HMessageBox("Selection must be multiple of 4 for decrement double words");
        aa->mac_error_ = 10;
        return;
    }

    // If no selection just flip one word
    if (len == 0)
    {
        end_addr = start_addr + 4;
        len = 4;
    }
    ASSERT(end_addr - start_addr == len);
    ASSERT(start_addr + len <= GetDocument()->length());

    // Get memory for bytes to be flipped and read them from the document
    unsigned long *buf = new unsigned long[len/4];
    size_t got = GetDocument()->GetData((unsigned char *)buf, len, start_addr);
    ASSERT(got == len);

    if (len == 4 && buf[0] == 0)
    {
        ((CMainFrame *)AfxGetMainWnd())->StatusBarText("Warning: Decrement wrapped around past zero");
        aa->mac_error_ = 1;            // Signal warning on wrap
    }

    // Flip bytes around and make change
    for (size_t ii = 0; ii < len/4; ++ii)
    {
        // Decrement the word
        buf[ii]--;
    }
    GetDocument()->Change(mod_replace, start_addr, len, (unsigned char *)buf, 0, this);
    delete[] buf;

    DisplayCaret();
    aa->SaveToMacro(km_dec32);
}

void CHexEditView::OnDec64bit() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (check_ro("decrement quad words"))
        return;
    // Get current address or selection
    long start_addr, end_addr;          // Start and end of selection
    size_t len;                         // Length of selection
    GetSelAddr(start_addr, end_addr);
    len = size_t(end_addr - start_addr);

    if (len == 0 && start_addr + 7 >= GetDocument()->length())
    {
        // Not enough bytes before EOF, presumably in macro playback
        ASSERT(aa->playing_);
        ::HMessageBox("Insufficient bytes before EOF to decrement quad word");
        aa->mac_error_ = 10;
        return;
    }
    else if (len > 0 && len%8 != 0)
    {
        // Selection not a multiple of 8
        ASSERT(aa->playing_);
        ::HMessageBox("Selection must be multiple of 8 for decrement quad words");
        aa->mac_error_ = 10;
        return;
    }

    // If no selection just flip one word
    if (len == 0)
    {
        end_addr = start_addr + 8;
        len = 8;
    }
    ASSERT(end_addr - start_addr == len);
    ASSERT(start_addr + len <= GetDocument()->length());

    // Get memory for bytes to be flipped and read them from the document
    unsigned __int64 *buf = new unsigned __int64[len/8];
    size_t got = GetDocument()->GetData((unsigned char *)buf, len, start_addr);
    ASSERT(got == len);

    if (len == 8 && buf[0] == 0)
    {
        ((CMainFrame *)AfxGetMainWnd())->StatusBarText("Warning: Decrement wrapped around past zero");
        aa->mac_error_ = 1;            // Signal warning on wrap
    }

    // Flip bytes around and make change
    for (size_t ii = 0; ii < len/8; ++ii)
    {
        // Decrement the quad word
        buf[ii]--;
    }
    GetDocument()->Change(mod_replace, start_addr, len, (unsigned char *)buf, 0, this);
    delete[] buf;

    DisplayCaret();
    aa->SaveToMacro(km_dec64);
}

void CHexEditView::OnFlip16bit() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (check_ro("flip bytes"))
        return;

    // Get current address or selection
    long start_addr, end_addr;          // Start and end of selection
    size_t len;                         // Length of selection
    GetSelAddr(start_addr, end_addr);
    len = size_t(end_addr - start_addr);

    // Do validation which is normally protected by OnUpdate16bit
    if (len == 0 && start_addr + 1 >= GetDocument()->length())
    {
        // Not enough bytes before EOF, presumably in macro playback
        ASSERT(aa->playing_);
        ::HMessageBox("Insufficient bytes before EOF to flip word");
        aa->mac_error_ = 10;
        return;
    }
    else if (len > 0 && len%2 != 0)
    {
        // Selection not a multiple of 2
        ASSERT(aa->playing_);
        ::HMessageBox("Selection must have an even number of bytes to flip words");
        aa->mac_error_ = 10;
        return;
    }

    // If no selection just flip one word
    if (len == 0)
    {
        end_addr = start_addr + 2;
        len = 2;
    }
    ASSERT(end_addr - start_addr == len);
    ASSERT(start_addr + len <= GetDocument()->length());

    // Get memory for bytes to be flipped and read them from the document
    unsigned char *buf = new unsigned char[len];
    size_t got = GetDocument()->GetData(buf, len, start_addr);
    ASSERT(got == len);

    // Flip bytes around and make change
//    unsigned char cc = buf[0]; buf[0] = buf[1]; buf[1] = cc;
    for (size_t ii = 0; ii < len; ii += 2)
    {
        unsigned char cc = buf[ii];
        buf[ii] = buf[ii+1];
        buf[ii+1] = cc;
    }
    GetDocument()->Change(mod_replace, start_addr, len, buf, 0, this);
    delete[] buf;

    DisplayCaret();
    aa->SaveToMacro(km_flip16);
}

void CHexEditView::OnFlip32bit() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (check_ro("flip bytes"))
        return;

    // Get current address or selection
    long start_addr, end_addr;          // Start and end of selection
    size_t len;                         // Length of selection
    GetSelAddr(start_addr, end_addr);
    len = size_t(end_addr - start_addr);

    if (len == 0 && start_addr + 3 >= GetDocument()->length())
    {
        // Not enough bytes before EOF, presumably in macro playback
        ASSERT(aa->playing_);
        ::HMessageBox("Insufficient bytes before EOF to flip double word");
        aa->mac_error_ = 10;
        return;
    }
    else if (len > 0 && len%4 != 0)
    {
        // Selection not a multiple of 4
        ASSERT(aa->playing_);
        ::HMessageBox("Selection size must be multiple of 4 to flip double words");
        aa->mac_error_ = 10;
        return;
    }

    // If no selection just flip one double word
    if (len == 0)
    {
        end_addr = start_addr + 4;
        len = 4;
    }
    ASSERT(len == end_addr - start_addr);
    ASSERT(start_addr + len <= GetDocument()->length());

    // Get memory for bytes to be flipped and read them from the document
    unsigned char *buf = new unsigned char[len];
    size_t got = GetDocument()->GetData(buf, len, start_addr);
    ASSERT(got == len);

    // Flip bytes around and make change to doc
    for (size_t ii = 0; ii < len; ii += 4)
    {
        unsigned char cc = buf[ii];
        buf[ii] = buf[ii+3];
        buf[ii+3] = cc;
        cc = buf[ii+1];
        buf[ii+1] = buf[ii+2];
        buf[ii+2] = cc;
    }
    GetDocument()->Change(mod_replace, start_addr, len, buf, 0, this);
    delete[] buf;

    DisplayCaret();
    aa->SaveToMacro(km_flip32);
}

void CHexEditView::OnFlip64bit() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (check_ro("flip bytes"))
        return;

    // Get current address or selection
    long start_addr, end_addr;          // Start and end of selection
    size_t len;                         // Length of selection
    GetSelAddr(start_addr, end_addr);
    len = size_t(end_addr - start_addr);

    // Do validation which is normally protected by OnUpdate16bit
    if (len == 0 && start_addr + 7 >= GetDocument()->length())
    {
        // Not enough bytes before EOF, presumably in macro playback
        ASSERT(aa->playing_);
        ::HMessageBox("Insufficient bytes before EOF to flip quad word");
        aa->mac_error_ = 10;
        return;
    }
    else if (len > 0 && len%8 != 0)
    {
        // Selection not a multiple of 8
        ASSERT(aa->playing_);
        ::HMessageBox("Selection must be a multiple of 8 to flip quad words");
        aa->mac_error_ = 10;
        return;
    }

    // If no selection just flip one quad word
    if (len == 0)
    {
        end_addr = start_addr + 8;
        len = 8;
    }
    ASSERT(end_addr - start_addr == len);
    ASSERT(start_addr + len <= GetDocument()->length());

    // Get memory for bytes to be flipped and read them from the document
    unsigned char *buf = new unsigned char[len];
    size_t got = GetDocument()->GetData(buf, len, start_addr);
    ASSERT(got == len);

    // Flip bytes around and make change to doc
    for (size_t ii = 0; ii < len; ii += 8)
    {
        unsigned char cc;
        cc = buf[ii];   buf[ii]   = buf[ii+7]; buf[ii+7] = cc;
        cc = buf[ii+1]; buf[ii+1] = buf[ii+6]; buf[ii+6] = cc;
        cc = buf[ii+2]; buf[ii+2] = buf[ii+5]; buf[ii+5] = cc;
        cc = buf[ii+3]; buf[ii+3] = buf[ii+4]; buf[ii+4] = cc;
    }
    GetDocument()->Change(mod_replace, start_addr, len, buf, 0, this);
    delete[] buf;

    DisplayCaret();
    aa->SaveToMacro(km_flip64);
}

#ifdef ENCOM
void CHexEditView::OnRsForw() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (!aa->playing_ && GetFocus() != this) SetFocus(); // Ensure focus does not stay in DlgBar
    CMainFrame *mm = dynamic_cast<CMainFrame *>(AfxGetMainWnd());

    CString ss;                                 // Message string
    unsigned long length;                       // Length of RS block
    unsigned long end_length;                   // length obtained from RS tail
    unsigned long address = GetPos();           // Current address in file
    unsigned char buf[6];                       // Temp to hold bits of file
    size_t got;                                 // # of bytes received from doc

    got = GetDocument()->GetData(buf, 2, address);
    if (got < 2 || ((buf[0] != 'F' || buf[1] != 'H') && (buf[0] != 'R' || buf[1] != 'S')))
    {
        // There should an "FH" or an "RS" unless we are in keystroke macro
        ASSERT(aa->playing_);
        ::HMessageBox("Can't move to next RS record - no FH or RS found");
        aa->mac_error_ = 10;
        return;
    }

    // First check if it's an FH structure
    if (buf[0] == 'F' && buf[1] == 'H')
    {
        got = GetDocument()->GetData(buf, 2, address + 2);
        if (got == 2 && buf[0] == '\0' && buf[1] == '\0')
        {
            // Appears to be a valid FH structure but check that it's
            // followed by an RS or another FH (but not EOF?)
            // since it's esy to get a bogus FH
            got = GetDocument()->GetData(buf, 2, address + 90);
            if (got == 2 &&
                (buf[0] == 'R' && buf[1] == 'S' ||
                 buf[0] == 'F' && buf[1] == 'H') )
            {
                mm->StatusBarText("Valid FH structure");
                MoveToAddress(address + 90);
                aa->SaveToMacro(km_rs_forw);
                return;
            }
            if (address + 90 > GetDocument()->length())
                ::HMessageBox("Incomplete FH structure");
            else
                ::HMessageBox("No RS or FH following FH structure --\r"
                              "implies this is not an FH structure");
            aa->mac_error_ = 10;
            return;
        }
        ::HMessageBox("Invalid FH structure");
        aa->mac_error_ = 10;
        return;
    }

    // Not FH so must be an RS record
#ifndef NDEBUG
    got = GetDocument()->GetData(buf, 2, address);
    ASSERT(got == 2 && buf[0] == 'R' && buf[1] == 'S');
#endif
    // Get length of record
    got = GetDocument()->GetData(buf, 6, address + 2);
    length = (buf[3]<<24) + (buf[2]<<16) + (buf[1]<<8) + buf[0];

    if (address + length >= address  &&         // No overflow
        address + length + 18L < LONG_MAX)      // GetData only handles 2^31
    {
        // Check that tail length is the same as length in header
        got = GetDocument()->GetData(buf, 4, address + 14L + length);
        end_length = (buf[3]<<24) + (buf[2]<<16) + (buf[1]<<8) + buf[0];

        if (got == 4 && length == end_length)
        {
            // Seems to be valid (new-style) RS block
            ss.Format("RS length:%ld%s%s%s", length,
                buf[5] != '\0' ? ", Continued" : "",
                buf[4] & '\x1' ? ", Hard Err" : "",
                buf[4] & '\x2' ? ", Soft Err" : "");
            mm->StatusBarText(ss);
            MoveToAddress(address + length + 18L);
            undo_.push_back(view_undo(undo_setmark, TRUE));
            undo_.back().address = mark_;
            invalidate_addr_range(mark_, mark_+1);
            mark_ = address + 14L;
            invalidate_addr_range(mark_, mark_+1);
            show_calc();
//          DoInvalidate();
            aa->SaveToMacro(km_rs_forw);
            return;
        }
    }

    // Assume that it's an old style RS record
    got = GetDocument()->GetData(buf, 2, address + 2);
    length = (buf[1]<<8) + buf[0];

    // Check that tail length is the same as length in header
    got = GetDocument()->GetData(buf, 2, address + 4L + length);
    end_length = (buf[1]<<8) + buf[0];

    if (got == 2 && length == end_length)
    {
        // Seems to be valid old-style RS block
        ss.Format("OLD RS format: length %ld", length);
        mm->StatusBarText(ss);
        MoveToAddress(address + length + 6L);
        undo_.push_back(view_undo(undo_setmark, TRUE));
        undo_.back().address = mark_;
        invalidate_addr_range(mark_, mark_+1);
        mark_ = address + 4L;
        invalidate_addr_range(mark_, mark_+1);
        show_calc();
        aa->SaveToMacro(km_rs_forw);
        return;
    }

    // Not recognisable - work out what's wrong
    if (address + 18L > GetDocument()->length())
    {
        ::HMessageBox("Incomplete RS structure");
        aa->mac_error_ = 10;
        return;
    }

    got = GetDocument()->GetData(buf, 6, address + 2);
    length = (buf[3]<<24) + (buf[2]<<16) + (buf[1]<<8) + buf[0];
    if (address + length < address  ||          // overflow
        address + length + 18L > GetDocument()->length())
    {
        ss.Format("Invalid RS structure\r"
                  "header length (%lu) indicates\r"
                  "RS block end is past end of file",
                  length);
        ::HMessageBox(ss);
        aa->mac_error_ = 10;
        return;
    }

    // Check that tail length is the same as length in header
    got = GetDocument()->GetData(buf, 4, address + 14L + length);
    end_length = (buf[3]<<24) + (buf[2]<<16) + (buf[1]<<8) + buf[0];
    if (length != end_length)
    {
        ss.Format("Invalid RS structure\r"
                  "header length (%lu)\r"
                  "differs from tail length (%lu)",
                  length, end_length);
        ::HMessageBox(ss);
        aa->mac_error_ = 10;
        return;
    }

    ASSERT(0);
    ::HMessageBox("Invalid RS structure?");
    aa->mac_error_ = 10;
}

void CHexEditView::OnUpdateRsForw(CCmdUI* pCmdUI) 
{
    unsigned char buf[4];
    size_t got = GetDocument()->GetData(buf, 2, GetPos());

    // Enable RS scan forward if "RS" is directly after caret
    pCmdUI->Enable(got == 2 && 
                   (buf[0] == 'R' && buf[1] == 'S' ||
                    buf[0] == 'F' && buf[1] == 'H') );
}

void CHexEditView::OnRsBack() 
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (!aa->playing_ && GetFocus() != this) SetFocus(); // Ensure focus does not stay in DlgBar
    CMainFrame *mm = dynamic_cast<CMainFrame *>(AfxGetMainWnd());

    CStatusBar *psb = &dynamic_cast<CMainFrame *>(AfxGetMainWnd())->m_wndStatusBar;
    ASSERT(psb != NULL);

    CString ss;                                 // Message string
    unsigned long end_length;                   // length obtained from RS tail
    unsigned long start_length;                 // Length of from header
    unsigned long address = GetPos();           // Current address in file
    unsigned char buf[8];                       // Temp to hold bits of file
    size_t got;                                 // # of bytes received from doc

    // Could an RS block fit between here and start of file?
    if (address >= 18)
    {
        // Get RS length from tail
        got = GetDocument()->GetData(buf, 4, address - 4);
        end_length = (buf[3]<<24) + (buf[2]<<16) + (buf[1]<<8) + buf[0];

        // Is start of block after start of file?
        if (end_length <= address - 18UL)
        {
            got = GetDocument()->GetData(buf, 8, address - end_length - 18);
            start_length = (buf[5]<<24) + (buf[4]<<16) + (buf[3]<<8) + buf[2];
            if (got == 8 && buf[0] == 'R' && buf[1] == 'S' &&
                start_length == end_length)
            {
                // Seems to be valid (new-style) RS block
                ss.Format("RS length:%ld%s%s%s", end_length,
                    buf[7] != '\0' ? ", Continued" : "",
                    buf[6] & '\x1' ? ", Hard Err" : "",
                    buf[6] & '\x2' ? ", Soft Err" : "");
                mm->StatusBarText(ss);
                MoveToAddress(address - end_length - 18L);
                undo_.push_back(view_undo(undo_setmark, TRUE));
                undo_.back().address = mark_;
                invalidate_addr_range(mark_, mark_+1);
                mark_ = address - end_length - 4L;
                invalidate_addr_range(mark_, mark_+1);
                show_calc();
//              DoInvalidate();
                aa->SaveToMacro(km_rs_back);
                return;
            }
        }
    }

    // Next check if it's an old style RS block
    if (address >= 6)
    {
        // Get RS length from tail
        got = GetDocument()->GetData(buf, 2, address - 2);
        end_length = (buf[1]<<8) + buf[0];

        // Is start of block after start of file?
        if (end_length <= address - 6UL)
        {
            got = GetDocument()->GetData(buf, 4, address - end_length - 6);
            start_length = (buf[3]<<8) + buf[2];
            if (got == 4 && buf[0] == 'R' && buf[1] == 'S' &&
                start_length == end_length)
            {
                // Seems to be valid old-style RS block
                ss.Format("OLD RS format: length %ld", end_length);
                mm->StatusBarText(ss);
                MoveToAddress(address - end_length - 6);
                undo_.push_back(view_undo(undo_setmark, TRUE));
                undo_.back().address = mark_;
                invalidate_addr_range(mark_, mark_+1);
                mark_ = address - end_length - 2L;
                invalidate_addr_range(mark_, mark_+1);
                show_calc();
                aa->SaveToMacro(km_rs_back);
                return;
            }
        }
    }

    // Could it be an FH structure?
    if (address >= 90)
    {
        got = GetDocument()->GetData(buf, 4, address - 90);
        if (got == 4 && buf[0] == 'F' && buf[1] == 'H' && buf[2] == '\0' && buf[3] == '\0')
        {
            mm->StatusBarText("Valid FH structure");
            MoveToAddress(address - 90);
            aa->SaveToMacro(km_rs_back);
            return;
        }
    }

    if (address < 18)
    {
        ::HMessageBox("Too close to start of file\rfor RS structure");
        aa->mac_error_ = 10;
        return;
    }

    // Some sort of error: get what should be tail RS length
    got = GetDocument()->GetData(buf, 4, address - 4);
    end_length = (buf[3]<<24) + (buf[2]<<16) + (buf[1]<<8) + buf[0];

    if (end_length > address - 18UL)
    {
        ss.Format("Tail length (%lu) indicates\r"
                  "RS block header is before start of file",
                  end_length);
        ::HMessageBox(ss);
        aa->mac_error_ = 10;
        return;
    }

    got = GetDocument()->GetData(buf, 6, address - end_length - 18);
    start_length = (buf[5]<<24) + (buf[4]<<16) + (buf[3]<<8) + buf[2];
    if (got == 6 && (buf[0] != 'R' || buf[1] != 'S'))
    {
        ss.Format("RS header not found at %lu",
                  address - end_length - 18UL);
        ::HMessageBox(ss);
        aa->mac_error_ = 10;
        return;
    }

    if (got == 6 && start_length != end_length)
    {
        ss.Format("Invalid RS structure\r"
                  "tail length (%lu)\r"
                  "differs from header length (%lu)",
                  end_length, start_length);
        ::HMessageBox(ss);
        aa->mac_error_ = 10;
        return;
    }

    ASSERT(0);
    ::HMessageBox("Invalid RS structure?");
    aa->mac_error_ = 10;
}

void CHexEditView::OnUpdateRsBack(CCmdUI* pCmdUI) 
{
    long address = GetPos();
    long file_length = GetDocument()->length();
    unsigned char buf[2];
    size_t got = GetDocument()->GetData(buf, 2, address);

    // Enable RS scan back if we could be after RS block, ie.
    // if we're at end of file OR
    // if next 2 chars are "RS" and we're not at start of file
    pCmdUI->Enable(address == file_length ||
                   address > 5 && got == 2 &&
                    (buf[0] == 'R' && buf[1] == 'S' ||
                     buf[0] == 'F' && buf[1] == 'H') );
}
#endif

