// MainFrm.cpp : implementation of the CMainFrame class
//
// Copyright (c) 1999 by Andrew W. Phillips.
//
// No restrictions are placed on the noncommercial use of this code,
// as long as this text (from the above copyright notice to the
// disclaimer below) is preserved.
//
// This code may be redistributed as long as it remains unmodified
// and is not sold for profit without the author's written consent.
//
// This code, or any part of it, may not be used in any software that
// is sold for profit, without the author's written consent.
//
// DISCLAIMER: This file is provided "as is" with no expressed or
// implied warranty. The author accepts no liability for any damage
// or loss of business that this product may cause.
//

#include "stdafx.h"
#include <MultiMon.h>
#include "HexEdit.h"

#include "HexEditDoc.h"
#include "HexEditView.h"
#include "MainFrm.h"
#include "Misc.h"
#include <afxpriv.h>            // for WM_COMMANDHELP

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CMainFrame

IMPLEMENT_DYNAMIC(CMainFrame, CMDIFrameWnd)

static UINT WM_FINDREPLACE = ::RegisterWindowMessage(FINDMSGSTRING);

BEGIN_MESSAGE_MAP(CMainFrame, CMDIFrameWnd)
        ON_WM_INITMENU()
        //{{AFX_MSG_MAP(CMainFrame)
        ON_WM_CREATE()
        ON_WM_CLOSE()
        ON_COMMAND(ID_EDIT_FIND, OnEditFind)
        ON_UPDATE_COMMAND_UI(ID_EDIT_FIND, OnUpdateEditFind)
        ON_COMMAND(ID_EDIT_FIND2, OnEditFind2)
        ON_COMMAND(ID_EDIT_GOTO, OnEditGoto)
        ON_UPDATE_COMMAND_UI(ID_EDIT_GOTO, OnUpdateEditGoto)
        ON_UPDATE_COMMAND_UI(ID_EDIT_FIND2, OnUpdateEditFind)
        ON_COMMAND(ID_CALCULATOR, OnCalculator)
        //}}AFX_MSG_MAP
        ON_COMMAND(ID_WINDOW_NEW, OnWindowNew)
        ON_COMMAND_EX(ID_WINDOW_ARRANGE, OnMDIWindowCmd)
        ON_COMMAND_EX(ID_WINDOW_CASCADE, OnMDIWindowCmd)
        ON_COMMAND_EX(ID_WINDOW_TILE_HORZ, OnMDIWindowCmd)
        ON_COMMAND_EX(ID_WINDOW_TILE_VERT, OnMDIWindowCmd)
        ON_WM_SYSCOMMAND()
        ON_COMMAND(ID_VIEW_TOOLBAR, OnViewToolbar)
        ON_UPDATE_COMMAND_UI(ID_VIEW_TOOLBAR, OnUpdateViewToolbar)
        ON_COMMAND(ID_VIEW_EDITBAR, OnViewEditbar)
        ON_UPDATE_COMMAND_UI(ID_VIEW_EDITBAR, OnUpdateViewEditbar)

        ON_COMMAND(ID_HELP_FINDER, OnHelpFinder)
        ON_COMMAND(ID_HELP, OnHelp)
        ON_COMMAND(ID_HELP_INDEX, OnHelpIndex)
        ON_MESSAGE(WM_COMMANDHELP, OnCommandHelp)
        ON_COMMAND(ID_CONTEXT_HELP, OnContextHelp)
        ON_COMMAND(ID_DEFAULT_HELP, OnHelpFinder)

        ON_CBN_EDITCHANGE(IDC_JUMP_HEX, OnChangeAddrHex)
        ON_CBN_SELENDOK(IDC_JUMP_HEX, OnSelAddrHex)
        ON_CBN_EDITCHANGE(IDC_JUMP_DEC, OnChangeAddrDec)
        ON_CBN_SELENDOK(IDC_JUMP_DEC, OnSelAddrDec)
        ON_COMMAND_EX(ID_VIEW_STATUS_BAR, OnBarCheck)
        ON_UPDATE_COMMAND_UI(ID_INDICATOR_OCCURRENCES, OnUpdateOccurrences)
        ON_UPDATE_COMMAND_UI(ID_INDICATOR_HEX_ADDR, OnUpdateAddrHex)
        ON_UPDATE_COMMAND_UI(ID_INDICATOR_DEC_ADDR, OnUpdateAddrDec)
        ON_UPDATE_COMMAND_UI(ID_INDICATOR_VALUES, OnUpdateValues)
        ON_UPDATE_COMMAND_UI(ID_INDICATOR_READONLY, OnUpdateReadonly)
        ON_UPDATE_COMMAND_UI(ID_INDICATOR_OVR, OnUpdateOvr)
        ON_UPDATE_COMMAND_UI(ID_INDICATOR_REC, OnUpdateRec)
        ON_REGISTERED_MESSAGE(CHexEditApp::wm_hexedit, OnOpenMsg)

        ON_REGISTERED_MESSAGE(WM_FINDREPLACE, OnFindDlgMess)
        ON_MESSAGE(WM_USER, OnReturn)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CMainFrame construction/destruction

CMainFrame::CMainFrame()
{
    pfind_ = NULL;
    pprop_ = NULL;
}

CMainFrame::~CMainFrame()
{
}

int CMainFrame::OnCreate(LPCREATESTRUCT lpCreateStruct)
{
        CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

        if (CMDIFrameWnd::OnCreate(lpCreateStruct) == -1)
                return -1;

        EnableDocking(CBRS_ALIGN_ANY);

        if (!m_wndToolBar.Create(this, IDW_TBAR,
                CBRS_TOP | CBRS_TOOLTIPS | CBRS_FLYBY, IDW_TBAR) ||
            !m_wndEditBar.Create(this, IDW_EBAR,
                CBRS_TOP | CBRS_TOOLTIPS | CBRS_FLYBY, IDW_EBAR))
        {
                TRACE0("Failed to create a dialog bar\n");
                return -1;      // fail to create
        }
        m_wndToolBar.SetWindowText("Toolbar");
        m_wndEditBar.SetWindowText("Edit");

        if (!bb_file_open_.SubclassDlgItem(ID_FILE_OPEN, &m_wndToolBar) ||
            !bb_file_save_.SubclassDlgItem(ID_FILE_SAVE, &m_wndToolBar) ||
            !bb_file_print_.SubclassDlgItem(ID_FILE_PRINT, &m_wndToolBar) ||
            !bb_edit_cut_.SubclassDlgItem(ID_EDIT_CUT, &m_wndToolBar) ||
            !bb_edit_copy_.SubclassDlgItem(ID_EDIT_COPY, &m_wndToolBar) ||
            !bb_edit_paste_.SubclassDlgItem(ID_EDIT_PASTE, &m_wndToolBar) ||
            !bb_edit_undo_.SubclassDlgItem(ID_EDIT_UNDO, &m_wndToolBar) ||

            !bb_autofit_.SubclassDlgItem(ID_AUTOFIT, &m_wndToolBar) ||
            !bb_font_inc_.SubclassDlgItem(ID_FONT_INC, &m_wndToolBar) ||
            !bb_font_dec_.SubclassDlgItem(ID_FONT_DEC, &m_wndToolBar) ||
            !bb_addr_toggle_.SubclassDlgItem(ID_ADDR_TOGGLE, &m_wndToolBar) ||
            !bb_char_toggle_.SubclassDlgItem(ID_CHAR_TOGGLE, &m_wndToolBar) ||
            !bb_asc_ebc_.SubclassDlgItem(ID_ASC_EBC, &m_wndToolBar) ||
            !bb_control_.SubclassDlgItem(ID_CONTROL, &m_wndToolBar) ||
            !bb_graphic_toggle_.SubclassDlgItem(ID_GRAPHIC_TOGGLE, &m_wndToolBar) ||
            !bb_allow_mods_.SubclassDlgItem(ID_ALLOW_MODS, &m_wndToolBar) ||
            !bb_edit_rec_.SubclassDlgItem(ID_RECORD, &m_wndToolBar) ||
            !bb_edit_play_.SubclassDlgItem(ID_PLAY, &m_wndToolBar) ||
#ifdef ENCOM
            !bb_rs_back_.SubclassDlgItem(ID_RS_BACK, &m_wndEditBar) ||
            !bb_rs_forw_.SubclassDlgItem(ID_RS_FORW, &m_wndEditBar) ||
#endif
            !bb_mark_.SubclassDlgItem(ID_MARK, &m_wndEditBar) ||
            !bb_goto_mark_.SubclassDlgItem(ID_GOTO_MARK, &m_wndEditBar) ||
            !bb_search_back_.SubclassDlgItem(ID_SEARCH_BACK, &m_wndEditBar) ||
            !bb_search_forw_.SubclassDlgItem(ID_SEARCH_FORW, &m_wndEditBar) )
        {
                TRACE0("Failed to find dialog bar button\n");
                return -1;
        }

        CWnd *pcombo;           // Ptr to combo box control
        CWnd *pedit;            // Ptr to edit control of combo box

        if ((pcombo = m_wndEditBar.GetDlgItem(IDC_SEARCH)) == NULL ||
            (pedit = pcombo->ChildWindowFromPoint(CPoint(25,5))) == NULL ||
            pedit == pcombo ||
            !sec_search_.SubclassWindow(pedit->m_hWnd) )
        {
                TRACE0("Failed to find search combo\n");
        }

        if ((pcombo = m_wndEditBar.GetDlgItem(IDC_JUMP_HEX)) == NULL ||
            (pedit = pcombo->ChildWindowFromPoint(CPoint(25,5))) == NULL ||
//CWP_SKIPINVISIBLE |CWP_SKIPDISABLED |CWP_SKIPTRANSPARENT
            pedit == pcombo ||
            !hec_hex_addr_.SubclassWindow(pedit->m_hWnd) )
        {
                TRACE0("Failed to find hex address combo\n");
        }

        if ((pcombo = m_wndEditBar.GetDlgItem(IDC_JUMP_DEC)) == NULL ||
            (pedit = pcombo->ChildWindowFromPoint(CPoint(25,5))) == NULL ||
            pedit == pcombo ||
            !dec_dec_addr_.SubclassWindow(pedit->m_hWnd) )
        {
                TRACE0("Failed to find dec address combo\n");
        }

        if (!bb_file_open_.LoadBitmaps(IDB_FILE_OPENU,IDB_FILE_OPEND) ||
            !bb_file_save_.LoadBitmaps(IDB_FILE_SAVEU,IDB_FILE_SAVED,0,IDB_FILE_SAVEX) ||
            !bb_file_print_.LoadBitmaps(IDB_FILE_PRINTU,IDB_FILE_PRINTD,0,IDB_FILE_PRINTX) ||
            !bb_edit_cut_.LoadBitmaps(IDB_EDIT_CUTU,IDB_EDIT_CUTD,0,IDB_EDIT_CUTX) ||
            !bb_edit_copy_.LoadBitmaps(IDB_EDIT_COPYU,IDB_EDIT_COPYD,0,IDB_EDIT_COPYX) ||
            !bb_edit_paste_.LoadBitmaps(IDB_EDIT_PASTEU,IDB_EDIT_PASTED,0,IDB_EDIT_PASTEX) ||
            !bb_edit_undo_.LoadBitmaps(IDB_EDIT_UNDOU,IDB_EDIT_UNDOD,0,IDB_EDIT_UNDOX) ||

            !bb_autofit_.LoadBitmaps(IDB_AUTOFITU,IDB_AUTOFITD,0,IDB_AUTOFITX) ||
            !bb_font_inc_.LoadBitmaps(IDB_FINCU,IDB_FINCD,0,IDB_FINCX) ||
            !bb_font_dec_.LoadBitmaps(IDB_FDECU,IDB_FDECD,0,IDB_FDECX) ||
            !bb_addr_toggle_.LoadBitmaps(IDB_ADDR_TOGGLEU,IDB_ADDR_TOGGLED,0,IDB_ADDR_TOGGLEX) ||
            !bb_char_toggle_.LoadBitmaps(IDB_CHAR_TOGGLEU,IDB_CHAR_TOGGLED,0,IDB_CHAR_TOGGLEX) ||
            !bb_asc_ebc_.LoadBitmaps(IDB_ASC_EBCU,IDB_ASC_EBCD,0,IDB_ASC_EBCX) ||
            !bb_control_.LoadBitmaps(IDB_CONTROLU,IDB_CONTROLD,0,IDB_CONTROLX) ||
            !bb_graphic_toggle_.LoadBitmaps(IDB_GRAPHIC_TOGGLEU,IDB_GRAPHIC_TOGGLED,0,IDB_GRAPHIC_TOGGLEX) ||
            !bb_allow_mods_.LoadBitmaps(IDB_MODSU,IDB_MODSD,0,IDB_MODSX) ||
            !bb_edit_rec_.LoadBitmaps(IDB_EDIT_RECU,IDB_EDIT_RECD,0,IDB_EDIT_RECX) ||
            !bb_edit_play_.LoadBitmaps(IDB_EDIT_PLAYU,IDB_EDIT_PLAYD,0,IDB_EDIT_PLAYX) ||
#ifdef ENCOM
            !bb_rs_back_.LoadBitmaps(IDB_RS_BACKU,IDB_RS_BACKD,0,IDB_RS_BACKX) ||
            !bb_rs_forw_.LoadBitmaps(IDB_RS_FORWU,IDB_RS_FORWD,0,IDB_RS_FORWX) ||
#endif
            !bb_mark_.LoadBitmaps(IDB_MARKU,IDB_MARKD,0,IDB_MARKX) ||
            !bb_goto_mark_.LoadBitmaps(IDB_GOTO_MARKU,IDB_GOTO_MARKD,0,IDB_GOTO_MARKX) ||
            !bb_search_back_.LoadBitmaps(IDB_EDIT_FINDPREVU,IDB_EDIT_FINDPREVD,0,IDB_EDIT_FINDPREVX) ||
            !bb_search_forw_.LoadBitmaps(IDB_EDIT_FINDNEXTU,IDB_EDIT_FINDNEXTD,0,IDB_EDIT_FINDNEXTX) )
        {
                TRACE0("Failed to find dialog bar bitmap\n");
                return -1;
        }
        hec_hex_addr_.SetLimitText(9);
        dec_dec_addr_.SetLimitText(13);

//      m_wndToolBar.SetBarStyle(m_wndToolBar.GetBarStyle() |
//              CBRS_TOOLTIPS | CBRS_FLYBY);
        m_wndToolBar.EnableDocking(CBRS_ALIGN_TOP | CBRS_ALIGN_BOTTOM);

//      m_wndEditBar.SetBarStyle(m_wndEditBar.GetBarStyle() |
//              CBRS_TOOLTIPS | CBRS_FLYBY);
        m_wndEditBar.EnableDocking(CBRS_ALIGN_TOP | CBRS_ALIGN_BOTTOM);

        DockControlBar(&m_wndToolBar);
        DockControlBar(&m_wndEditBar);

        // Note the dialog bar button bitmaps are also handled by the view
        // (inited in OnInitialUpdate and changed when options changed).
        if (!m_wndStatusBar.Create(this))
        {
                TRACE0("Failed to create status bar\n");
                return -1;      // fail to create
        }

        LoadBarState("DockState");

        // Load search strings into mainframe edit bar
        LoadSearchHistory(aa);

        pcalc_ = new CCalcDlg;

        return 0;
}

BOOL CMainFrame::PreCreateWindow(CREATESTRUCT& cs) 
{
    WNDCLASS wndclass;
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
	HINSTANCE hInst = AfxGetInstanceHandle();

    BOOL retval = CMDIFrameWnd::PreCreateWindow(cs);

    ::GetClassInfo(hInst, cs.lpszClass, &wndclass);
    wndclass.style &= ~(CS_HREDRAW|CS_VREDRAW);
    wndclass.lpszClassName = aa->szHexEditClassName;
	wndclass.hIcon = ::LoadIcon(hInst, MAKEINTRESOURCE(IDR_MAINFRAME));
	ASSERT(wndclass.hIcon != 0);
	if (!AfxRegisterClass(&wndclass))
		AfxThrowResourceException();

	cs.lpszClass = aa->szHexEditClassName;

    if (aa->open_restore_)
    {
        int ss;
        if ((ss = aa->GetProfileInt("MainFrame", "WindowState", -1)) != -1)
            aa->m_nCmdShow = ss;

        // Get the window position/size
        int top, left, bottom, right;
        top = aa->GetProfileInt("MainFrame", "WindowTop", -30000);
        left = aa->GetProfileInt("MainFrame", "WindowLeft", -30000);
        bottom = aa->GetProfileInt("MainFrame", "WindowBottom", -30000);
        right = aa->GetProfileInt("MainFrame", "WindowRight", -30000);

        // If the values look OK change the CREATESTRUCT value correspondingly
        if (top != -30000 && right != -30000 && top < bottom && left < right)
        {
            // Get the work area within the display
            CRect rct;
            if (aa->mult_monitor_)
            {
                CRect rr(left, top, right, bottom);
                HMONITOR hh = MonitorFromRect(&rr, MONITOR_DEFAULTTONEAREST);
                MONITORINFO mi;
                mi.cbSize = sizeof(mi);
                if (hh != 0 && GetMonitorInfo(hh, &mi))
                    rct = mi.rcWork;  // work area of nearest monitor
                else
                {
                    // Shouldn't happen but if it does use the whole virtual screen
                    ASSERT(0);
                    rct = CRect(::GetSystemMetrics(SM_XVIRTUALSCREEN),
                        ::GetSystemMetrics(SM_YVIRTUALSCREEN),
                        ::GetSystemMetrics(SM_XVIRTUALSCREEN) + ::GetSystemMetrics(SM_CXVIRTUALSCREEN),
                        ::GetSystemMetrics(SM_YVIRTUALSCREEN) + ::GetSystemMetrics(SM_CYVIRTUALSCREEN));
                }
            }
            else if (!::SystemParametersInfo(SPI_GETWORKAREA, 0, &rct, 0))
            {
                // I don't know if this will ever happen since the Windows documentation
                // is pathetic and does not say when or why SystemParametersInfo might fail.
                rct = CRect(0, 0, ::GetSystemMetrics(SM_CXFULLSCREEN),
                                  ::GetSystemMetrics(SM_CYFULLSCREEN));
            }

            // Make sure that the window is not off the screen (or just on it).
            // (There might be a different screen resolution since options were saved.)
            if (left > rct.right - 20)
            {
                left = rct.right - (right - left);
                right = rct.right;
            }
            if (right < rct.left + 20)
            {
                right = rct.left + (right - left);
                left = rct.left;
            }
            if (top > rct.bottom - 20)
            {
                top = rct.bottom - (bottom - top);
                bottom = rct.bottom;
            }
            // Make sure top is not more than a little bit off the top of screen
            if (top < rct.top - 15)
            {
                bottom = rct.top + (bottom - top);
                top = rct.top;
            }

            // Set window width and height
            cs.cx = right - left;
            cs.cy = bottom - top;

            cs.x = left;
            cs.y = top;
        }
    }

    return retval;
}

BOOL CMainFrame::PreTranslateMessage(MSG* pMsg) 
{
    // We need to let the edit bar controls see their keys rather than have them intercepted by accelerator
    if (pMsg->hwnd == sec_search_.m_hWnd ||
        pMsg->hwnd == hec_hex_addr_.m_hWnd ||
        pMsg->hwnd == dec_dec_addr_.m_hWnd )
    {
        return FALSE;
    }

    return CMDIFrameWnd::PreTranslateMessage(pMsg);
}

void CMainFrame::OnClose() 
{
    // Save state (posn, docked, hidden etc) of control bars (tool, edit
    // & status bars) if save options on exit is on
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    SaveSearchHistory(aa);

    SaveBarState("DockState");

    if (aa->save_exit_)
        SaveFrameOptions();

    CMDIFrameWnd::OnClose();
}

LONG CMainFrame::OnOpenMsg(UINT, LONG lParam)
{
	char filename[256];
	filename[0] = '\0';
    return ::GlobalGetAtomName((ATOM)lParam, filename, sizeof(filename)) > 0 &&
           AfxGetApp()->OpenDocumentFile(filename) != NULL;
}

void CMainFrame::SaveFrameOptions()
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    // Save info about the main frame window position
    aa->WriteProfileInt("MainFrame", "Restore", aa->open_restore_ ? 1 : 0);
    if (aa->open_restore_)
    {
        WINDOWPLACEMENT wp;
        GetWindowPlacement(&wp);
        aa->WriteProfileInt("MainFrame", "WindowState", wp.showCmd);
        aa->WriteProfileInt("MainFrame", "WindowTop",    wp.rcNormalPosition.top);
        aa->WriteProfileInt("MainFrame", "WindowLeft",   wp.rcNormalPosition.left);
        aa->WriteProfileInt("MainFrame", "WindowBottom", wp.rcNormalPosition.bottom);
        aa->WriteProfileInt("MainFrame", "WindowRight",  wp.rcNormalPosition.right);
    }
    aa->WriteProfileInt("MainFrame", "ShowPropDlg", int(pprop_ != NULL));
    aa->WriteProfileInt("MainFrame", "ShowCalcDlg", int(pcalc_->visible_));
}

/////////////////////////////////////////////////////////////////////////////
// CMainFrame diagnostics

#ifdef _DEBUG
void CMainFrame::AssertValid() const
{
        CMDIFrameWnd::AssertValid();
}

void CMainFrame::Dump(CDumpContext& dc) const
{
        CMDIFrameWnd::Dump(dc);
}

#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CMainFrame message handlers

// Handles control menu commands and system buttons (Minimize etc)
void CMainFrame::OnSysCommand(UINT nID, LONG lParam)
{
    CMDIFrameWnd::OnSysCommand(nID, lParam);

    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    nID &= 0xFFF0;
    if (nID == SC_MINIMIZE || nID == SC_RESTORE || nID == SC_MAXIMIZE)
        aa->SaveToMacro(km_mainsys, nID);
}

void CMainFrame::OnWindowNew()
{
    // Store options for the active view so that the new view will get the same ones
    if (GetView() != NULL)
        GetView()->StoreOptions();

    CMDIFrameWnd::OnWindowNew();

    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (aa->recording_ && aa->mac_.size() > 0 && (aa->mac_.back()).ktype == km_focus)
    {
        // We don't want focus change recorded (see CHexEditView::OnSetFocus)
        aa->mac_.pop_back();
    }
    aa->SaveToMacro(km_win_new);
}

// Handles the window menu commands: cascade, tile, arrange
BOOL CMainFrame::OnMDIWindowCmd(UINT nID)
{
    BOOL retval = CMDIFrameWnd::OnMDIWindowCmd(nID);

    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (retval)
        aa->SaveToMacro(km_win_cmd, nID);
    else
        aa->mac_error_ = 20;
    return retval;
}

void CMainFrame::OnUpdateViewToolbar(CCmdUI* pCmdUI) 
{
    pCmdUI->SetCheck((m_wndToolBar.GetStyle() & WS_VISIBLE) != 0);
}

void CMainFrame::OnViewToolbar() 
{
    ShowControlBar(&m_wndToolBar, (m_wndToolBar.GetStyle() & WS_VISIBLE) == 0, FALSE);
    ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_toolbar);
}

void CMainFrame::OnUpdateViewEditbar(CCmdUI* pCmdUI) 
{
    pCmdUI->SetCheck((m_wndEditBar.GetStyle() & WS_VISIBLE) != 0);
}

void CMainFrame::OnViewEditbar() 
{
    ShowControlBar(&m_wndEditBar, (m_wndEditBar.GetStyle() & WS_VISIBLE) == 0, FALSE);
    ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_editbar);
}

void CMainFrame::OnContextHelp()
{
    CMDIFrameWnd::OnContextHelp();
}

void CMainFrame::OnHelpFinder()
{
    CMDIFrameWnd::OnHelpFinder();
    ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_topics);
}

void CMainFrame::OnHelp()
{
    CMDIFrameWnd::OnHelp();
    ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_help);
}

// This is here just so we can intercept calls in the debugger
LRESULT CMainFrame::OnCommandHelp(WPARAM wParam, LPARAM lParam)
{
    return CMDIFrameWnd::OnCommandHelp(wParam, lParam);
}

void CMainFrame::OnUpdateReadonly(CCmdUI *pCmdUI)
{
    // Get the active view
    CHexEditView *pview = GetView();
    if (pview != NULL)
    {
        pCmdUI->Enable(TRUE);
        if (pview->ReadOnly())
            pCmdUI->SetText("RO");
        else
            pCmdUI->SetText("RW");
    }
    else
        pCmdUI->Enable(FALSE);
}

void CMainFrame::OnUpdateOvr(CCmdUI *pCmdUI)
{
    // Get the active view
    CHexEditView *pview = GetView();
    if (pview != NULL && !pview->ReadOnly())
    {
        pCmdUI->Enable(TRUE);
        if (pview->OverType())
            pCmdUI->SetText("OVR");
        else
            pCmdUI->SetText("INS");
    }
    else
        pCmdUI->Enable(FALSE);
}

void CMainFrame::OnUpdateRec(CCmdUI *pCmdUI)
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (aa->recording_)
    {
        pCmdUI->Enable(TRUE);
        pCmdUI->SetText("REC");
    }
    else
        pCmdUI->Enable(FALSE);
}

// Called on EDITCHANGE message when the contents of the edit control of the
// combo box have been changed.
void CMainFrame::OnChangeAddrHex()
{
    // Make sure this change was made by the user since if the value is changed
    // programmatically this message is also sent.  To check that the user did
    // it we check that the edit control actually has focus.
    if (GetFocus()->m_hWnd == hec_hex_addr_.m_hWnd)
    {
        // Get the address from the current string entered
        CString ss;
        hec_hex_addr_.GetWindowText(ss);

        const char *pp;                         // Ptr into ss
        char buf[13];                           // ss without commas (raw digits)
        char *qq;                               // Ptr into buf

        for (pp = ss.GetBuffer(0), qq = buf; *pp != '\0'; ++pp)
            if (isxdigit(*pp))
                *qq++ = *pp;
        *qq = '\0';

        unsigned long address = strtoul(buf, NULL, 16);

        // Update the decimal edit control with the corresponding address
        ss.Format("%lu", address);
        dec_dec_addr_.SetWindowText(ss);
        dec_dec_addr_.add_commas();

#if 0
        // Move the caret of the current view to the address
        CHexEditView *pview = GetView();
        if (pview != NULL)
        {
            pview->GoAddress(address);
            pview->DisplayCaret();
        }
#endif
    }
}

// Called when an item is selected from the combo box list
void CMainFrame::OnSelAddrHex()
{
    int curr;
    CComboBox *pcombo = (CComboBox *)m_wndEditBar.GetDlgItem(IDC_JUMP_HEX);

    ASSERT(pcombo != NULL);
    if (pcombo != NULL && (curr = pcombo->GetCurSel()) != CB_ERR)
    {
        // Get the address from the string just selected
        CString ss;
        pcombo->GetLBText(curr, ss);

        const char *pp;                         // Ptr into ss
        char buf[13];                           // ss without commas (raw digits)
        char *qq;                               // Ptr into buf

        for (pp = ss.GetBuffer(0), qq = buf; *pp != '\0'; ++pp)
            if (isxdigit(*pp))
                *qq++ = *pp;
        *qq = '\0';

        unsigned long address = strtoul(buf, NULL, 16);

        // Update the decimal edit control with the corresponding address
        ss.Format("%lu", address);
        dec_dec_addr_.SetWindowText(ss);
        dec_dec_addr_.add_commas();

#if 0
        // Move the caret of the current view to the address
        CHexEditView *pview = GetView();
        if (pview != NULL)
        {
            pview->GoAddress(address);
            pview->DisplayCaret();
        }
#endif
    }
}

void CMainFrame::OnChangeAddrDec()
{
    // Make sure this change was made by the user since if the value is changed
    // programmatically this message is also sent.  To check that the user did
    // it we check the edit control has focus.
    if (GetFocus()->m_hWnd == dec_dec_addr_.m_hWnd)
    {
        CString ss;                             // Current address (in decimal)
        dec_dec_addr_.GetWindowText(ss);
        const char *pp;                         // Ptr into ss
        char buf[13];                           // ss without commas (raw digits)
        char *qq;                               // Ptr into buf

        for (pp = ss.GetBuffer(0), qq = buf; *pp != '\0'; ++pp)
            if (isdigit(*pp))
                *qq++ = *pp;
        *qq = '\0';

        errno = 0;
        unsigned long address = strtoul(buf, NULL, 10);

        // Update the hex address field to reflect the same address (in hex)
        if (address == ULONG_MAX && errno == ERANGE)
            hec_hex_addr_.SetWindowText("OVERFLOW");
        else
        {
            CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
            if (aa->hex_ucase_)
                ss.Format("%lX", address);
            else
                ss.Format("%lx", address);
            hec_hex_addr_.SetWindowText(ss);
            hec_hex_addr_.add_spaces();
        }

#if 0
        // Move the caret of the current view to the address
        CHexEditView *pview = GetView();
        if (pview != NULL)
        {
            pview->GoAddress(address);
            pview->DisplayCaret();
        }
#endif
    }
}

// Called when an item is selected from the combo box list
void CMainFrame::OnSelAddrDec()
{
    int curr;
    CComboBox *pcombo = (CComboBox *)m_wndEditBar.GetDlgItem(IDC_JUMP_DEC);

    ASSERT(pcombo != NULL);
    if (pcombo != NULL && (curr = pcombo->GetCurSel()) != CB_ERR)
    {
        // Get the address from the string just selected
        CString ss;
        pcombo->GetLBText(curr, ss);
        const char *pp;                         // Ptr into ss
        char buf[13];                           // ss without commas (raw digits)
        char *qq;                               // Ptr into buf

        for (pp = ss.GetBuffer(0), qq = buf; *pp != '\0'; ++pp)
            if (isdigit(*pp))
                *qq++ = *pp;
        *qq = '\0';

        errno = 0;
        unsigned long address = strtoul(buf, NULL, 10);

        // Update the decimal edit control with the corresponding address
        if (address == ULONG_MAX && errno == ERANGE)
            hec_hex_addr_.SetWindowText("OVERFLOW");
        else
        {
            ss.Format("%lX", address);
            hec_hex_addr_.SetWindowText(ss);
            hec_hex_addr_.add_spaces();
        }

#if 0
        // Move the caret of the current view to the address
        CHexEditView *pview = GetView();
        if (pview != NULL)
        {
            pview->GoAddress(address);
            pview->DisplayCaret();
        }
#endif
    }
}

void CMainFrame::OnUpdateAddrHex(CCmdUI *pCmdUI)
{
    // Get the active view
    CHexEditView *pview = GetView();
    if (pview != NULL)
    {
        // Update pane with Current offset from marked position (hex)
        CString ss;
        CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
        if (aa->hex_ucase_)
            ss.Format("%lX", pview->GetMarkOffset());
        else
            ss.Format("%lx", pview->GetMarkOffset());
        pCmdUI->SetText(ss);
        pCmdUI->Enable();
    }
    else
        pCmdUI->Enable(FALSE);
}

void CMainFrame::OnUpdateAddrDec(CCmdUI *pCmdUI)
{
    // Get the active view
    CHexEditView *pview = GetView();
    if (pview != NULL)
    {
        // Update pane with Current offset from marked position (decimal)
        CString ss;
        ss.Format("%ld", pview->GetMarkOffset());
        pCmdUI->SetText(ss);
        pCmdUI->Enable();
    }
    else
        pCmdUI->Enable(FALSE);
}

void CMainFrame::OnUpdateOccurrences(CCmdUI *pCmdUI)
{
    CHexEditView *pview = GetView();

    int ii;
    if (pview != NULL && (ii = pview->GetDocument()->SearchOccurrences()) > -1)
    {
        CString ss;
        ss.Format("%ld", long(ii));
        if (ss.GetLength() < 7)
            AddCommas(ss);
        else if (ss.GetLength() > 7)
            ss = "TOO BIG";
        pCmdUI->SetText(ss);
        pCmdUI->Enable();
    }
    else
        pCmdUI->Enable(FALSE);
}

void CMainFrame::OnUpdateValues(CCmdUI *pCmdUI)
{
    unsigned char cc;

    CHexEditView *pview = GetView();
    if (pview != NULL && pview->GetDocument()->GetData(&cc, 1, pview->GetPos()) == 1)
    {
        CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

        // Update pane with values of current byte
        char binbuf[9];             // To display binary value
        for (int ii = 0; ii < 8; ++ii)
            binbuf[ii] = (cc & (0x80>>ii)) ? '1' : '0';
        binbuf[8] = '\0';
        CString ss;

        if (aa->hex_ucase_)
        {
            if (!pview->EbcdicMode() && cc < 0x20)      // Control char?
                ss.Format("%02X, %d, %03o, %8s, '^%c'",cc,cc,cc,binbuf,cc+0x40);
            else if (!pview->EbcdicMode())
                ss.Format("%02X, %d, %03o, %8s, '%c'",cc,cc,cc,binbuf,cc);
            else if (e2a_tab[cc] != '\0')
                ss.Format("%02X, %d, %03o, %8s, '%c'",cc,cc,cc,binbuf, e2a_tab[cc]);
            else
                ss.Format("%02X, %d, %03o, %8s, none",cc,cc,cc,binbuf);
        }
        else
        {
            if (!pview->EbcdicMode() && cc < 0x20)      // Control char?
                ss.Format("%02x, %d, %03o, %8s, '^%c'",cc,cc,cc,binbuf,cc+0x40);
            else if (!pview->EbcdicMode())
                ss.Format("%02x, %d, %03o, %8s, '%c'",cc,cc,cc,binbuf,cc);
            else if (e2a_tab[cc] != '\0')
                ss.Format("%02x, %d, %03o, %8s, '%c'",cc,cc,cc,binbuf, e2a_tab[cc]);
            else
                ss.Format("%02x, %d, %03o, %8s, none",cc,cc,cc,binbuf);
        }
        pCmdUI->SetText(ss);
        pCmdUI->Enable();
    }
    else
        pCmdUI->Enable(FALSE);
}

void CMainFrame::StatusBarText(const char *mess /*=NULL*/)
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

    if (mess == NULL)
        m_wndStatusBar.SetPaneText(0, last_mess_);      // Update now
    else if (aa->refresh_off_)
        last_mess_ = mess;                              // Save message for later update
    else
        m_wndStatusBar.SetPaneText(0, mess);            // Display immediately
}

// Add an address to the list of hex address combo
void CMainFrame::AddHexHistory(const CString &ss)
{
    CComboBox *pcombo = (CComboBox *)m_wndEditBar.GetDlgItem(IDC_JUMP_HEX);
    ASSERT(pcombo != NULL);
    if (pcombo != NULL)
    {
        // Search history and remove any string the same (avoids lots of duplicates)
        int numstr = pcombo->GetCount();
        ASSERT(numstr != CB_ERR);
        if (numstr != CB_ERR)
            for (int ii = 0; ii < numstr; ++ii)
            {
                CString lbs;
                pcombo->GetLBText(ii, lbs);
                // See if this list (history) string matches the current one
                if (lbs == ss)
                {
                    pcombo->DeleteString(ii);
                    // Note: Deleting a list item string which has been selected from
                    // the list also deletes the edit control text so we need to set
                    // it back as a workaround to this problem.
                    pcombo->SetWindowText(ss);
                    hec_hex_addr_.add_spaces();
                    break;
                }
            }
        // Now just add at the top of the history list
        pcombo->InsertString(0, ss);
    }
}

// Add an address to the list of dec address combo
void CMainFrame::AddDecHistory(const CString &ss)
{
    CComboBox *pcombo = (CComboBox *)m_wndEditBar.GetDlgItem(IDC_JUMP_DEC);
    ASSERT(pcombo != NULL);
    if (pcombo != NULL)
    {
        // Search history and remove any string the same (avoids lots of duplicates)
        int numstr = pcombo->GetCount();
        ASSERT(numstr != CB_ERR);
        if (numstr != CB_ERR)
            for (int ii = 0; ii < numstr; ++ii)
            {
                CString lbs;
                pcombo->GetLBText(ii, lbs);
                // See if this list (history) string matches the current one
                if (lbs == ss)
                {
                    pcombo->DeleteString(ii);
                    // Note: Deleting a list item string which has been selected from
                    // the list also deletes the edit control text so we need to set
                    // it back as a workaround to this problem.
                    pcombo->SetWindowText(ss);
                    dec_dec_addr_.add_commas();
                    break;
                }
            }
        // Now just add at the top of the history list
        pcombo->InsertString(0, ss);
    }
}

// Add a search string to the list box of the search combo
void CMainFrame::AddSearchHistory(const CString &ss)
{
    CComboBox *p1 = (CComboBox *)m_wndEditBar.GetDlgItem(IDC_SEARCH);
    ASSERT(p1 != NULL);
    CComboBox *p2 = NULL;

    if (pfind_ != NULL)
    {
        p2 = (CComboBox *)pfind_->GetDlgItem(IDC_COMBO);
        ASSERT(p2 != NULL);
    }
    if (!ss.IsEmpty() && p1 != NULL)
    {
        // Search history and remove any string the same (avoids lots of duplicates)
        int numstr = p1->GetCount();
        ASSERT(numstr != CB_ERR);
        if (numstr != CB_ERR)
            for (int ii = 0; ii < numstr; ++ii)
            {
                CString lbs;
                p1->GetLBText(ii, lbs);
                // See if this list (history) string matches the current one
                if (lbs == ss)
                {
                    p1->DeleteString(ii);
                    // Note: Deleting a list item string which has been selected from
                    // the list also deletes the edit control text so we need to set
                    // it back as a workaround to this problem.
                    p1->SetWindowText(ss);

                    // Delete the same string from find dialog history list if there
                    if (p2 != NULL)
                    {
                        p2->GetLBText(ii, lbs);
                        if (ss[0] == CSearchEditControl::sflag_char ||
                            ss[0] == CSearchEditControl::iflag_char)
                        {
                            // Text case sensitive or insensitive
                            if (lbs == ss.Mid(1))
                            {
                                p2->DeleteString(ii);
                                p2->SetWindowText(ss.Mid(1));
                            }
                        }
                        else if (lbs == ss)
                        {
                            // Hex search string
                            p2->DeleteString(ii);
                            p2->SetWindowText(ss);
                        }
                    }
                    break;
                }
            }
        // Now just add this string at the top of the list
        p1->InsertString(0, ss);

		// Increase dropped width if this string is wider than any existing one
		CDC *pDC = p1->GetDC();
		CSize str_size = pDC->GetTextExtent(ss);
		if (str_size.cx*4/5+10 > p1->GetDroppedWidth())
		{
			if (str_size.cx > 500)
				p1->SetDroppedWidth(400);
			else
				p1->SetDroppedWidth(str_size.cx*4/5+10);
		}

        if (p2 != NULL)
        {
            if (ss[0] == CSearchEditControl::sflag_char ||
                ss[0] == CSearchEditControl::iflag_char)
                p2->InsertString(0, ss.Mid(1));
            else
                p2->InsertString(0, ss);
            p2->SetItemData(0, (DWORD)ss[0]);
        }
    }
}

static const int max_search_hist = 10;

// Retrieve search history from .INI file/registry
void CMainFrame::LoadSearchHistory(CHexEditApp *aa)
{
    CString ss;                         // Profile string
    CString entry;                      // "Search1" etc
	int max_len = 0;					// To find the longest string

    CComboBox *pcombo = (CComboBox *)m_wndEditBar.GetDlgItem(IDC_SEARCH);
    ASSERT(pcombo != NULL);

    pcombo->ResetContent();
	CDC *pDC = pcombo->GetDC();
	CSize str_size;

    for (int ii = 0; ii < max_search_hist; ++ii)
    {
        entry.Format("Search%d", ii + 1);
        ss = aa->GetProfileString("History", entry);
        if (!ss.IsEmpty())
        {
            pcombo->AddString(ss);
            str_size = pDC->GetTextExtent(ss);
            if (str_size.cx > max_len) max_len = str_size.cx;
        }
    }
	if (max_len > 500) max_len = 500;
	pcombo->SetDroppedWidth(max_len*4/5+10);
}

// Save search history to .INI file/registry (no more than 10)
void CMainFrame::SaveSearchHistory(CHexEditApp *aa)
{
    CString ss;                         // Profile string
    CString entry;                      // "Search1" etc

    CComboBox *pcombo = (CComboBox *)m_wndEditBar.GetDlgItem(IDC_SEARCH);
    ASSERT(pcombo != NULL);
    int last = min(pcombo->GetCount(), max_search_hist);

    for (int ii = 0; ii < last; ++ii)
    {
        pcombo->GetLBText(ii, ss);
        if (!ss.IsEmpty())
        {
            entry.Format("Search%d", ii + 1);
            aa->WriteProfileString("History", entry, ss);
        }
    }
}

// Create find modeless dialog - if already exists then just activate it
void CMainFrame::OnEditFind() 
{
    if (GetView() != NULL)
    {
        if (pfind_ == NULL)
            pfind_ = new CHexFindDialog;
        else
            pfind_->SetFocus();

        ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_find_dlg);
    }
}

void CMainFrame::OnEditFind2() 
{
    if (GetView() != NULL)
    {
        if (pfind_ == NULL)
            pfind_ = new CHexFindDialog;
        else
            pfind_->SetFocus();
        pfind_->set_hex();              // Set to hex search only

        ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_find_dlg);
    }
}

// We can't do a search unless there is a document open
void CMainFrame::OnUpdateEditFind(CCmdUI* pCmdUI) 
{
    pCmdUI->Enable(GetView() != NULL);
}

// This is called in response to a message generated by the find modeless dlg.
LONG CMainFrame::OnFindDlgMess(WPARAM wParam, LPARAM lParam)
{
    ASSERT(pfind_ != NULL);             // Make sure we know it's there

    FINDREPLACE *pfr = (FINDREPLACE *)lParam;
    ASSERT(pfr != NULL && pfr->lStructSize >= 22 && pfr->hwndOwner != 0);

    if (pfr->Flags & FR_DIALOGTERM)
    {
        // User hit the cancel button
        // Keep track of the fact that the modeless dlg is gone
        pfind_ = NULL;
        ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_find_close);
    }
    else if (pfr->Flags & FR_FINDNEXT)
    {
        // User hit the find next button
        CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());

        // Build the search string
        CString ss;                     // New search bytes from dialog
        CString curr_ss;                // Current search bytes
        if (pfind_->find_type_ != 0)    // If not hex search
        {
            // Set case sensitivity flag depending on control setting
            if (pfr->Flags & FR_MATCHCASE)
                ss = CSearchEditControl::sflag_char;
            else
                ss = CSearchEditControl::iflag_char;
        }
        ss += pfr->lpstrFindWhat;

        // Update the search combo with same string and add to history list
        sec_search_.GetWindowText(curr_ss);
        if (ss != curr_ss)
        {
            // New search bytes
            sec_search_.SetWindowText(ss);      // Update Find Tool
            aa->SaveToMacro(km_find_text, ss);  // Keep track of bytes entered in macro
            AddSearchHistory(ss);               // Add new search to top of history
        }

        // Calculate the current view so that we can access Search() member
        CHexEditView *pview = GetView();
        if (pview != NULL)
        {
            if (pview->Search(ss, pfind_->find_type_, (pfr->Flags & FR_DOWN) != 0))
            {
                aa->SaveToMacro(km_find_forw);
                pview->SetFocus();
            }
            else
            {
                pfind_->edit_.SetFocus();
//              pfind_->SetFocus();
            }
        }
    }
    return 0L;
}


void CMainFrame::OnCalculator() 
{
    ASSERT(pcalc_ != NULL);
    if (pcalc_->m_hWnd == 0)
        pcalc_->Create();

    pcalc_->SetWindowText("Calculator");
    pcalc_->ShowWindow(SW_RESTORE);
    pcalc_->visible_ = TRUE;

    // Make sure controls and displayed calc value are up to date
    pcalc_->UpdateData(FALSE);
    pcalc_->edit_.Put();
    pcalc_->ShowBinop();
    pcalc_->FixFileButtons();

    pcalc_->SetFocus();

    ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_calc_dlg);
}

void CMainFrame::OnEditGoto() 
{
    CHexEditView *pview = GetView();
    if (pview != NULL)
    {
        ASSERT(pcalc_ != NULL);
        if (pcalc_->m_hWnd == 0)
            pcalc_->Create();

        pcalc_->SetWindowText("Go To");
        pcalc_->ShowWindow(SW_RESTORE);
        pcalc_->visible_ = TRUE;

        // Make sure controls are up to date and put current address into it
        pcalc_->UpdateData(FALSE);

        long start, end;
        pview->GetSelAddr(start, end);
        pcalc_->Set(start);

        pcalc_->ShowBinop();
        pcalc_->FixFileButtons();

        pcalc_->SetFocus();

        ((CHexEditApp *)AfxGetApp())->SaveToMacro(km_goto);
    }
}

void CMainFrame::OnUpdateEditGoto(CCmdUI* pCmdUI) 
{
    pCmdUI->Enable(GetView() != NULL);
}

// OnReturn is called in response to an edit control (of a combo control)
// sending a WM_USER message to the parent (CMainFrame).  This happens
// when the user presses the Enter key.
// lparam is a pointer to the edit control that generated the message.
LRESULT CMainFrame::OnReturn(WPARAM, LPARAM lparam)
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    CString ss;         // Temp string for getting/setting edit text

    if ((CWnd *)lparam == &sec_search_)
    {
        // WM_USER message generated by the search edit control
        CHexEditView *pview = GetView();
        if (pview != NULL)
        {
            // Get the search text
            sec_search_.GetWindowText(ss);

            // Save in search history (combo) list
            aa->SaveToMacro(km_find_text, ss);
            AddSearchHistory(ss);

            // Do search
            if (pview->Search(ss, pview->EbcdicMode() ? 3 : 1))
                aa->SaveToMacro(km_find_forw);

            // Change find modeless dlg controls to match this search
            if (pfind_ != NULL)
                pfind_->update_controls(1);
        }
    }
    else if ((CWnd *)lparam == &hec_hex_addr_ ||
             (CWnd *)lparam == &dec_dec_addr_)
    {
        // WM_USER message generated by an address edit control
        unsigned long address;          // Address (converted from string)

        if ((CWnd *)lparam == &hec_hex_addr_)
        {
            // Get address from hex edit control
            hec_hex_addr_.GetWindowText(ss);

            const char *pp;                             // Ptr into ss
            char buf[13];                               // ss without commas (raw digits)
            char *qq;                           // Ptr into buf

            for (pp = ss.GetBuffer(0), qq = buf; *pp != '\0'; ++pp)
                if (isxdigit(*pp))
                    *qq++ = *pp;
            *qq = '\0';

            address = strtoul(buf, NULL, 16);
            AddHexHistory(ss);
        }
        else if ((CWnd *)lparam == &dec_dec_addr_)
        {
            // Get address from decimal edit control
            dec_dec_addr_.GetWindowText(ss);
            const char *pp;                     // Ptr into ss
            char buf[13];                       // ss without commas (raw digits)
            char *qq;                           // Ptr into buf
            for (pp = ss.GetBuffer(0), qq = buf; *pp != '\0'; ++pp)
                if (isdigit(*pp))
                    *qq++ = *pp;
            *qq = '\0';

            errno = 0;
            address = strtoul(buf, NULL, 10);
            if (!(address == ULONG_MAX && errno == ERANGE))
                AddDecHistory(ss);
        }
        // If recording macro indicate jump & save address jumped to
        aa->SaveToMacro(km_address_tool, address);

        // Get active view and go to new address in it
        CHexEditView *pview = GetView();
        if (pview != NULL)
        {
            // Try to go to the address requested
            long actual;
            if ((actual = pview->GoAddress(address)) != address)
            {
                // Could not go to the requested address - tell user
                if (aa->hex_ucase_)
                    ss.Format("Invalid address entered. Address set to EOF \"%lX\"", actual);
                else
                    ss.Format("Invalid address entered. Address set to EOF \"%lx\"", actual);
                ::HMessageBox(ss);
            }
            pview->SaveMove();          // Save prev pos in undo array
            pview->DisplayCaret();

            // Save the actual current address in the both edit controls
            if (aa->hex_ucase_)
                ss.Format("%lX", actual);
            else
                ss.Format("%lx", actual);
            hec_hex_addr_.SetWindowText(ss);
            hec_hex_addr_.add_spaces();

            ss.Format("%lu", actual);
            dec_dec_addr_.SetWindowText(ss);
            dec_dec_addr_.add_commas();
        }
    }
    else
        ASSERT(0);              // Where did this message come from?

    return 0L;
}

// Display control bars context menu
void CMainFrame::bar_context(CPoint point)
{
    // Get the top level menu that contains the submenus used as popup menus
    CMenu top;
    BOOL ok = top.LoadMenu(IDR_CONTEXT);
    ASSERT(ok);
    if (!ok) return;

    CMenu *ppop = top.GetSubMenu(1);
    ASSERT(ppop != NULL);
    if (ppop != NULL)
        VERIFY(ppop->TrackPopupMenu(TPM_LEFTALIGN | TPM_RIGHTBUTTON,
                             point.x, point.y, this));

    top.DestroyMenu();
}

void CMainFrame::OnInitMenu(CMenu* pMenu)
{
    // Remove this?
    CMDIFrameWnd::OnInitMenu(pMenu);
}

// Toggle display of status bar
BOOL CMainFrame::OnBarCheck(UINT nID)
{
    CHexEditApp *aa = dynamic_cast<CHexEditApp *>(AfxGetApp());
    if (CMDIFrameWnd::OnBarCheck(nID))
    {
        aa->SaveToMacro(km_bar, nID);
        return TRUE;
    }
    else
    {
        aa->mac_error_ = 2;
        return FALSE;
    }
}
