/* 
FILE: 			ExampleGUI.java

Graphical query tool for use with UnityJDBC driver.

Search for "TODO" to find out where you can easily add your own databases and queries to this application.
*/

package Tests;

import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;

import javax.swing.BorderFactory;
import javax.swing.BoxLayout;
import javax.swing.DefaultCellEditor;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTable;
import javax.swing.JTextArea;
import javax.swing.JTree;
import javax.swing.border.Border;
import javax.swing.border.EtchedBorder;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumnModel;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.TreePath;

import unity.annotation.AnnotatedSourceDatabase;
import unity.annotation.AnnotatedSourceField;
import unity.annotation.AnnotatedSourceTable;
import unity.io.FileManager;

public class ExampleGUI extends JFrame 
{
	private static final long serialVersionUID = 1L;
	private static int FrameWidth = 1000, FrameHeight = 900;
    private ViewTree schemaTree;
    private DataTable dataTable;
    private JTextArea queryBox, messageBox;
    private JButton execButton, clearButton;
    private MyDataModel dataSource;
    private boolean queryExecuting;
    private QueryThread qt;
    private String urlUnity = "jdbc:Unity://Tests/xspec/UnityDemo.xml";
    private static JScrollPane treeView;
    private JSplitPane splitPane;
    private static ExampleGUI frame;
    private JScrollPane dataView;
    private Border boxBorder;
    private JPanel dataPanel;

    public static void main(String[] args) throws IOException 
    {
        frame = new ExampleGUI("jdbc:Unity://Tests/xspec/UnityDemo.xml");
        frame.setSize(FrameWidth, FrameHeight);
        frame.setTitle("Unity Query Application");
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        frame.setBackground(Color.blue);
        frame.setVisible(true);
        frame.focusQueryBox();
    }

    public ExampleGUI(String xspec) throws IOException 
    {
        addWindowListener(new WindowCloser()); 
        Container contentPane = getContentPane();
        boxBorder = BorderFactory.createEtchedBorder(EtchedBorder.LOWERED);
        Font myFont = new Font("SansSerif", Font.PLAIN, 12);

        // Connect to databases to get information
        dataSource = new MyDataModel();

        String u = FileManager.getXMLFile(xspec);
        if(u.indexOf("jdbc:Unity://")<0)
            u = "jdbc:Unity://" + u;
        setUnityURL(u);
        
        dataSource.makeDBConnection(urlUnity);

        schemaTree = buildTree();
        dataTable = new DataTable(dataSource);

        // Right panel contains query box, message box, and results table
        // Left panel contains tree of attributes/tables in all databases
        JPanel rightPanel = new JPanel();       
        rightPanel.setLayout(new BoxLayout(rightPanel, BoxLayout.Y_AXIS));
        rightPanel.add(new JLabel("Query: "));
        queryBox = new JTextArea(5, 60);
        queryBox.setLineWrap(true);
        queryBox.setBorder(boxBorder);
        queryBox.setFont(myFont);
        queryBox.setText("SELECT  FROM  ;");
        queryBox.setCaretPosition(7);
        rightPanel.add(queryBox);
        clearButton = new JButton("Clear");
        execButton = new JButton("EXECUTE");
        execButton.setForeground(Color.green);
        execButton.addActionListener((ActionListener) new ButtonListener());
        clearButton.addActionListener((ActionListener) new ButtonListener());
        JPanel tmp2 = new JPanel();
        tmp2.add(execButton);    
        tmp2.add(clearButton);            
        rightPanel.add(tmp2);

        rightPanel.add(new JLabel("Messages:"));
        messageBox = new JTextArea(10, 50);
        messageBox.setLineWrap(true);
        messageBox.setEditable(false);
        messageBox.setBorder(boxBorder);
        messageBox.setFont(myFont);
        rightPanel.add(messageBox);        
        dataPanel = new JPanel();
        dataPanel.setLayout(new BoxLayout(dataPanel, BoxLayout.Y_AXIS));
        dataPanel.add(new JLabel("Results: "));
        dataView = new JScrollPane(dataTable);
        dataView.setBorder(boxBorder);
        dataPanel.add(dataView);
        rightPanel.add(dataPanel);

        // Main center panel with schema tree (left) and query builder (right)
        treeView = new JScrollPane(schemaTree);
        splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
        splitPane.setLeftComponent(treeView);
        splitPane.setRightComponent(rightPanel);
        splitPane.setDividerLocation(225);
        contentPane.add(splitPane); // Add the split pane to frame
        queryExecuting = false;

        this.setJMenuBar(new Menu());
    }

    public void setUnityURL(String url) {
        urlUnity = url;
    }

    public void reInit(String url) throws IOException {
        
    	url = url.substring(url.indexOf("jdbc:Unity://")+13);
    	String u = FileManager.getXMLFile(url);
        if(u.indexOf("jdbc:Unity://")<0)
            u = "jdbc:Unity://" + u;
        setUnityURL(u);

        // Connect to databases to get information
        dataSource = new MyDataModel();

        dataSource.makeDBConnection(urlUnity);

        schemaTree = buildTree();
        dataView.remove(dataTable);
        dataPanel.remove(dataView);
        dataTable = new DataTable(dataSource);
        dataView = new JScrollPane(dataTable);
        dataView.setBorder(boxBorder);
        dataPanel.add(dataView);

        splitPane.remove(treeView);

        treeView = new JScrollPane(schemaTree);
        splitPane.setLeftComponent(treeView);

        splitPane.setDividerLocation(225);
    }

    public void focusQueryBox() {
        queryBox.grabFocus();
    }

    // Window Adapter for this Frame
    private class WindowCloser extends WindowAdapter {
        public void windowClosing(WindowEvent event) {
            dataSource.closeDBConnection();
            System.exit(0);
        }
    }

    private ViewTree buildTree() 
    {
        DefaultMutableTreeNode root = new DefaultMutableTreeNode("Databases");
        DefaultMutableTreeNode dbNode, tblNode, fldNode;

        // Get schema information from UnityConnection (in dataSource)
        ArrayList dbs = dataSource.getDatabases();
        for (int i = 0; i < dbs.size(); i++) {
            AnnotatedSourceDatabase asd = (AnnotatedSourceDatabase) dbs.get(i);
            dbNode = new DefaultMutableTreeNode(asd.getDatabaseName());
            root.add(dbNode);

            Iterator tables = (asd.getSourceTables().values()).iterator();
            while (tables.hasNext()) {
                AnnotatedSourceTable ast = (AnnotatedSourceTable) tables.next();
                tblNode = new DefaultMutableTreeNode(ast.getTableName());
                dbNode.add(tblNode);

                Iterator fields = (ast.getSourceFields().values()).iterator();
                while (fields.hasNext()) {
                    AnnotatedSourceField asf = (AnnotatedSourceField) fields.next();
                    fldNode = new DefaultMutableTreeNode(asf.getName());
                    tblNode.add(fldNode);
                }
            }
        }
        
        return new ViewTree(root);
    }

    public class MyComboBoxRenderer extends JComboBox implements TableCellRenderer 
    {
		private static final long serialVersionUID = 1L;

		public MyComboBoxRenderer(ArrayList ar) {
            super(ar.toArray());
        }

        public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) 
        {
            if (isSelected) {
                setForeground(table.getSelectionForeground());
                super.setBackground(table.getSelectionBackground());
            } else {
                setForeground(Color.blue);
                setBackground(table.getBackground());
            }
            return this;
        }
    }

    public class MyComboBoxEditor extends DefaultCellEditor 
    {   
		private static final long serialVersionUID = 1L;

		public MyComboBoxEditor(JComboBox cb) {
            super(cb);
        }
    }

    private class QueryThread extends Thread {
        private String query;

        public QueryThread(String qry) {
            query = qry;
        }

        public void run() {
            try {
                queryExecuting = true;
                long startTime = System.currentTimeMillis();

                dataSource.doQuery(query, startTime, messageBox);
                long endTime = System.currentTimeMillis();
 
                messageBox.append("Number of results: "+ dataSource.getRowCount() + "\n");
                messageBox.append("Query complete.\n");
                long execTime = endTime - startTime;
                messageBox.append("Query Exection time is " + execTime + " ms.");
            } catch (Exception e) {
                messageBox.append("ERROR: " + e + "\n");
            }
            queryExecuting = false;
            execButton.setText("EXECUTE");
            execButton.setForeground(Color.green);
        }
    }

    private class ButtonListener implements ActionListener {
        public void actionPerformed(ActionEvent event) {
            Object source = event.getSource();
            if (source == execButton) {
                if (queryExecuting) {
                    messageBox.setText("User stopped query.\n");
                    execButton.setText("EXECUTE");
                    execButton.setForeground(Color.green);
                    queryExecuting = false;                    
                    qt.stop();
                } else {
                    messageBox.setText("Executing query...\n");
                    execButton.setText("STOP");
                    execButton.setForeground(Color.red);
                    qt = new QueryThread(queryBox.getText());
                    qt.start();
                }
                execButton.repaint();
            } else if (source == clearButton) {
                messageBox.setText("");
                queryBox.setText("SELECT  FROM  ;");
                queryBox.setCaretPosition(7);
                queryBox.grabFocus();
            }
        }
    }

    private class DataTable extends JTable 
    {        
		private static final long serialVersionUID = 1L;
		
		MyDataModel dataModel;

        public DataTable(MyDataModel dm) {
            super(dm);
            getTableHeader().addMouseListener( (MouseListener) new HeaderSortListener());
            dataModel = (MyDataModel) dm;
            setRowHeight(20);
        }

        private class HeaderSortListener extends MouseAdapter {
            public void mouseClicked(MouseEvent e) {
                TableColumnModel columnModel = getColumnModel();
                int column = columnModel.getColumnIndexAtX(e.getX());
                if (e.getClickCount() == 1 && column != -1)
                    dataModel.sortColumn(column);
            }
        }

        public TableCellRenderer getCellRenderer(int row, int column) {
            Object obj = dataModel.getValueAt(row, column);
            if (obj == null)
                return super.getCellRenderer(row, column);            

            if (obj.getClass() == ArrayList.class) {
                return new MyComboBoxRenderer((ArrayList) obj);
            }
            try {
            	return super.getCellRenderer(row, column);
            }
            catch (Exception e)
            {	// Discard errors that occur due to synchronization problems (query updating table and event queue trying to display it)	
				return null;
			}
            
        }

        public TableCellEditor getCellEditor(int row, int column) {
            Object obj = dataModel.getValueAt(row, column);

            if (obj == null)
                return super.getCellEditor(row, column);

            if (obj.getClass() == ArrayList.class) {
                JComboBox cb = new JComboBox(((ArrayList) obj).toArray());
                cb.setForeground(Color.blue);
                cb.setBackground(Color.white);
                return new MyComboBoxEditor(cb);
            }
            return super.getCellEditor(row, column);
        }
    }

    private class ViewTree extends JTree 
    {
       	private static final long serialVersionUID = 1L;

		public ViewTree(DefaultMutableTreeNode parent) {
            super(parent);
            addTreeSelectionListener(new MyTreeListener());
            setRootVisible(true);
            expandTree();
        }

        public void expandTree() 
        {
            // Expands all nodes
            // Enumeration e = ((DefaultMutableTreeNode)
            // getModel().getRoot()).depthFirstEnumeration();
            // Expand only first nodes
            Enumeration e = ((DefaultMutableTreeNode) getModel().getRoot()).children();
            for (; e.hasMoreElements();) {
                TreePath t = new TreePath(((DefaultMutableTreeNode) e.nextElement()).getPath());
                setExpandedState(t, true);
            }
        }

        private class MyTreeListener implements TreeSelectionListener {
            public void valueChanged(TreeSelectionEvent e) {
                DefaultMutableTreeNode node = (DefaultMutableTreeNode) getLastSelectedPathComponent();

                if (node == null)
                    return;                
                if (node.getDepth() > 1)
                    return; // Only insert leaf nodes (fields) or tables

                String st = node.toString();

                node = (DefaultMutableTreeNode) node.getParent();
                while (node.getParent() != null) // Do not add root node
                {
                    st = node.toString() + "." + st;
                    node = (DefaultMutableTreeNode) node.getParent();
                }

                queryBox.insert(st.toLowerCase() + " ", queryBox.getCaretPosition());
                queryBox.grabFocus();
            }
        }
    }

    private class Menu extends JMenuBar 
    {
		private static final long serialVersionUID = 1L;

		String[] fileItems = new String[] { "Exit" };

		// TODO: Add your own query names here.
        String[] menuLabel = new String[] { 
        /* 1 */	"Cross-Database Join Test",                 
        /* 2 */ "ORDER BY Test", 
        /* 3 */ "Optimizer Test (Distributed Join)",
        /* 4 */ "Expression Test", 
        /* 5 */ "GROUP BY test", 
        /* 6 */ "Performance Benchmark Test",
        /* 7 */ "MERGE and MATCH Test"
        	}; 

        // TODO: Add your own query strings here.
        String[] queries = new String[] {
        /* 1 */ "SELECT L.l_extendedprice, P.p_retailprice, L.l_partkey\n FROM OrderDB.Lineitem L, PartDB.Part P\n WHERE P.p_partkey = L.l_partkey AND L.l_partkey < 10;",
        /* 2 */	"SELECT PartDB.Part.p_name, OrderDB.LineItem.l_orderkey\n FROM PartDB.Part, OrderDB.Lineitem\n WHERE PartDB.Part.p_partkey = OrderDB.Lineitem.l_partkey AND OrderDB.Lineitem.l_linenumber > 6\n ORDER BY PartDB.Part.p_name;",        
        /* 3 */ "SELECT P.p_name, L.l_orderkey, S.s_name\n FROM PartDB.PART P, OrderDB.LINEITEM L, PartDB.SUPPLIER S\n WHERE S.s_suppkey = 3 AND P.p_partkey = L.l_partkey AND S.s_suppkey = L.l_suppkey;",        
        /* 4 */ "SELECT N.N_nationkey+5 AS MyAttrName\n FROM orderDB.nation as N, partDB.region as R\n WHERE N.n_nationkey +10 < 20 and R.r_regionkey = N.n_regionkey;",
        /* 5 */ "SELECT COUNT(N.n_nationkey)\n FROM orderDB.nation N, partDB.region R\n WHERE R.r_regionkey = N.n_regionkey\n GROUP BY N.n_regionkey;",
        /* 6 */ "SELECT L.L_ExtendedPrice, P.p_retailprice, L.l_partkey\n FROM OrderDB.LINEITEM L, PartDB.Part P, OrderDB.Orders O\n WHERE P.p_partkey = L.l_partkey and L.l_orderkey = O.o_orderkey and L.l_orderkey < 1000000;", 
        /* 7 */ "SELECT N.n_nationkey, N.n_regionkey, N.n_name FROM OrderDB.Nation as N\n MERGE ON Q1.C1=Q2.C2 EXTRACT ALL GROUPREF(Q1.C2*3,Q2.C1+5) AS M\n SELECT N.n_regionkey, N.n_nationkey AS N2 FROM PartDB.Nation as N; "		
        	};
        
        // TODO: Put your English description of your queries here.
        String[] english = new String[] {
        /* 1 */ "A Unity test containing a large client-side join.", 
        /* 2 */ "This query demonstrates the capability to order the results of a query on the client-side.",
        /* 3 */ "Demonstrates Optimizer Selecting a Distributed Query and Pushing Selection Down to Source.\nPushing down the selection S.s_suppkey = 3 allows the system to execute the joins of\n SUPPLIER, LINEITEM, and PART in a staged fashion.\nThis increases performance as the entire LINEITEM or PART table does not have to be retrieved from the source.\nThis results in up to 10 times speed improvement!",        
        /* 4 */ "Shows expressions in SELECT clause executed on client-side.",
        /* 5 */ "Shows GROUP BY clause executed on client-side.",
        /* 6 */ "Performance test requiring large client-side joins on 100 MB TPC-H database.\nShows benefit of early response of Early Hash Join compared to standard hybrid hash join.",
        /* 7 */ "Shows MERGE operator to relate attributes across data sources.\nNote the values in blue (pop-down lists) show attributes that have different values between sources."
                };

        // TODO: Each query needs to specify its sources XML file.  In this case, all are the same, but they do not have to be.
        String[] sources = new String[] {
        /* 1 */ "jdbc:Unity://Tests/xspec/UnityDemo.xml",
        /* 2 */ "jdbc:Unity://Tests/xspec/UnityDemo.xml",
        /* 3 */ "jdbc:Unity://Tests/xspec/UnityDemo.xml",
        /* 4 */ "jdbc:Unity://Tests/xspec/UnityDemo.xml",
        /* 5 */ "jdbc:Unity://Tests/xspec/UnityDemo.xml",
        /* 6 */ "jdbc:Unity://Tests/xspec/UnityDemo.xml",
        /* 7 */ "jdbc:Unity://Tests/xspec/UnityDemo.xml"    
        		};

        // TODO: Use this menu to select a different sources XML file.  This is used to update the left-hand query tree.
        // In this case, there is only one but you can add more.
        String[] menuSource = new String[] { "Unity Demo" };
        String[] menuSources = new String[] { "jdbc:Unity://Tests/xspec/UnityDemo.xml"};

        HashMap query = new HashMap();
        HashMap eng = new HashMap();
        HashMap source = new HashMap();
        HashMap integrations = new HashMap();

        public Menu() 
        {
            JMenu fileMenu = new JMenu("File");
            JMenu editMenu = new JMenu("Demonstration");
            JMenu subMenu = new JMenu("Queries");
            JMenu sourceMenu = new JMenu("Sources");

            for (int i = 0; i < menuSource.length; i++) 
                integrations.put(menuSource[i], menuSources[i]);

            for (int i = 0; i < menuLabel.length; i++) {
                query.put(menuLabel[i], queries[i]);
                eng.put(menuLabel[i], english[i]);
                source.put(menuLabel[i], sources[i]);
            }

            ActionListener printListener = new ActionListener() {
                public void actionPerformed(ActionEvent event) {
                    String command = event.getActionCommand();
                    try
                    {
                    	if (command.toLowerCase().equals("exit")) 
                    		System.exit(0);                    
                    	else if (integrations.get(command) != null) 
                    		reInit((String) integrations.get(command));
                    	else 
                    	{
                    		String qBox = (String) query.get(command);
                    		if (qBox == null) 
                    		{
                    			messageBox.setText("This query is not defined in this application.");
                    			return;
                    		}
                    		reInit((String) source.get(command));
                    		queryBox.setText(qBox);
                    		String desc = (String) eng.get(command);
                    		if (desc == null)
                    			messageBox.setText("");
                    		else
                    			messageBox.setText(desc);
                    	}
                    }
                    catch (IOException e)
                    {	messageBox.setText(e.toString());
                    }
                }
            };

            for (int i = 0; i < fileItems.length; i++) {
                JMenuItem item = new JMenuItem(fileItems[i]);
                item.addActionListener(printListener);
                fileMenu.add(item);
            }

            // Assemble the File menus with keyboard accelerators.
            for (int i = 0; i < menuLabel.length; i++) {
                JMenuItem item = new JMenuItem(menuLabel[i]);
                item.addActionListener(printListener);
                subMenu.add(item);
            }
            for (int i = 0; i < menuSource.length; i++) {
                JMenuItem item = new JMenuItem(menuSource[i]);
                item.addActionListener(printListener);
                sourceMenu.add(item);
            }

            editMenu.add(subMenu);
            editMenu.add(sourceMenu);
            add(fileMenu);
            add(editMenu);
        }
    }
}