--
--   Installation script for "Schema Version Control for Oracle" (SVCO)
--   http://www.sumsoftsolutions.com
--
--   File Name: install.sql
--   SVCO Version: 1.0.0
--   DB: Oracle 10g and 11g
--   OS: non-specific
--   Comments: For details please see readme.txt
--

SET SERVEROUTPUT ON SIZE 1000000
SET PAGESIZE 0
SET VERIFY OFF
WHENEVER OSERROR EXIT
WHENEVER SQLERROR EXIT

PROMPT .............................................................
PROMPT ... Installing "Schema Version Control for Oracle" (SVCO) ...
PROMPT ...         http://www.sumsoftsolutions.com               ...
PROMPT ... PLEASE READ LICENSE AGREEMENT CAREFULLY               ...
PROMPT ...                 (file license.txt) BEFORE INSTALLING  ...
PROMPT .............................................................
PROMPT
PROMPT ... Checking installation requirements ......................

-- Check block
VARIABLE o_ver CHAR(2)

declare
    v_ver   varchar2(64);
    v_comp  varchar2(64);
    v_count number;
begin
	if  '&_PRIVILEGE' <> 'AS SYSDBA' or '&_PRIVILEGE' is null then
		RAISE_APPLICATION_ERROR(-20700, '!!! Please login AS SYSDBA !!!');
	end if;	
	if upper('&_USER') <> 'SYS' then
		RAISE_APPLICATION_ERROR(-20700, '!!! Please login as SYS user !!!');
	end if;
	sys.dbms_utility.db_version(version => v_ver, compatibility => v_comp);
	if to_number(replace(v_ver, '.', null)) < 101020 then
		RAISE_APPLICATION_ERROR(-20700, '!!! Oracle version should be 10.1.0.2.0 or greater !!!');
  end if;
  select count(1)
  into v_count
  from all_users
  where username = 'SVCO';
  if v_count > 0 then
		RAISE_APPLICATION_ERROR(-20700, '!!! SVCO user allready exists !!!');
  end if;
  :o_ver := substr(v_ver, 1, 2);
end;
/
spool data/ver.sql
select 'DEFINE v_ver = '||:o_ver from dual;
spool off
@data/ver
PROMPT ... Done ....................................................
PROMPT
ACCEPT tabtbs DEFAULT 'USERS' PROMPT 'Please enter TABLESPACE name for SVCO TABLES (Default is USERS) : '
ACCEPT idxtbs DEFAULT 'INDEXES' PROMPT 'Please enter TABLESPACE name for SVCO INDEXES (Default is INDEXES) : '
ACCEPT pwd DEFAULT 'SVCO' PROMPT 'Please enter PASSWORD for SVCO USER (Default is SVCO) : ' HIDE
ACCEPT tmptbs DEFAULT 'TEMP' PROMPT 'Please enter TEMPORARY TABLESPACE name for SVCO USER (Default is TEMP) : '
PROMPT
PROMPT ... Creating required Roles .................................
CREATE ROLE SVCO_REP_ADMIN NOT IDENTIFIED
/
CREATE ROLE SVCO_REP_OPERATOR NOT IDENTIFIED
/
GRANT SVCO_REP_OPERATOR TO SVCO_REP_ADMIN
/
GRANT SELECT_CATALOG_ROLE TO SVCO_REP_ADMIN
/
PROMPT ... Done ....................................................
PROMPT
PROMPT ... Creating SVCO USER ......................................
CREATE USER SVCO IDENTIFIED BY "&pwd"
    DEFAULT TABLESPACE &tabtbs
    TEMPORARY TABLESPACE &tmptbs
    PROFILE DEFAULT
    ACCOUNT UNLOCK
/
GRANT CONNECT TO SVCO
/
GRANT SVCO_REP_ADMIN TO SVCO WITH ADMIN OPTION
/
GRANT SVCO_REP_OPERATOR TO SVCO WITH ADMIN OPTION
/
ALTER USER SVCO DEFAULT ROLE CONNECT,
                             SVCO_REP_ADMIN,
                             SVCO_REP_OPERATOR
/
GRANT UNLIMITED TABLESPACE TO SVCO
/
GRANT EXECUTE ON SYS.DBMS_ALERT TO SVCO
/
GRANT EXECUTE ON SYS.DBMS_CRYPTO TO SVCO
/
GRANT EXECUTE ON SYS.UTL_XML TO SVCO
/
PROMPT ... Done ....................................................
PROMPT
PROMPT ... Installing SVCO objects .................................
@data/sequences
@data/tables
@data/triggers
@data/&v_ver/packages
@data/views
@data/grants
PROMPT ... Done ....................................................
PROMPT
PROMPT ... Installing SVCO data ....................................
@data/data
PROMPT ... Done ....................................................
PROMPT
PROMPT ... Testing SVCO ............................................
declare
  rev  number;
	test varchar2(5);
	ddl  clob;
begin
	 SVCO.repository.createrevision(Revision => rev,
	                                RevisionCreated => test,
	                                SchemaName => 'SVCO', 
	                                Version => SVCO.repository_util.CurrentSVCOVersion, 
	                                Comments => 'Installation test');
	 if test = 'FALSE' then
	 	RAISE_APPLICATION_ERROR(-20700, '!!! SVCO test not working !!!');
	 end if;	 
	 SVCO.repository_admin.createddlfromrevision(schemaname => 'SVCO',
                                               ddl => ddl);   
end;
/
PROMPT ... Done ....................................................
PROMPT
PROMPT ... All done ...
PROMPT
exit
