using System;
using System.Data;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Web.UI.HtmlControls;
using System.ComponentModel;
using System.ComponentModel.Design;
using System.Collections;
using System.Collections.Specialized;
using System.Web.UI.Design;
using System.Drawing;
using System.Drawing.Design;
using ComputerBuff.Controls.XPanel.Design;
using System.Text;
using System.IO;

namespace ComputerBuff.Controls.XPanel
{
	/// <summary>
	///		State Enumeration
	/// </summary>
	public enum State
	{
		/// <summary>
		///		Collapsed Visible State
		/// </summary>
		Collapsed,
		/// <summary>
		///		Expanded Visible State
		/// </summary>
		Expanded
	}

	/// <summary>
	///		XPanel menu control class
	/// </summary>
	//ToolboxBitmap(typeof(Bitmap), "NeptuneTab"),
	[DefaultProperty("HeaderText")]
	[ToolboxBitmap(typeof(XPanel))]
	[ToolboxData("<{0}:XPanel runat=server></{0}:XPanel>")]
	[ParseChildren(false)]
	[PersistChildren(false)]
	[ControlBuilder(typeof(Design.MenuItemBuilder))]
	[Designer(typeof(Design.XPanelDesigner))]
	public class XPanel : WebControl, INamingContainer
	{
		#region Constants
		const string menuName = "RightNavigationMenu";
		const string javaScriptVoid = "javascript:void(0)";
		#endregion Constants

		#region Fields
		private string hideImageUrl = null;
		private string showImageUrl = null;
		private string newImageUrl = null;
		private string headerText;
		private Unit width = 184;
		private MenuItems menuItems;
		private Color headerBackColor = ColorTranslator.FromHtml("#6699cc");
		private Color headerForeColor = Color.White;
		private Color borderColor = Color.Black;
		private FontInfo headerFont;
		private BorderStyle borderStyle;
		private Unit borderWidth;
		private string groupName = null;
		private State state = State.Expanded;
		#endregion Fields

		#region Properties
		/// <summary>
		///		Gets all a collection of all MenuItems contained within the menu.
		/// </summary>
		[Category("Item"),
		DefaultValue("Items to display within the menu"),
		PersistenceMode(PersistenceMode.InnerDefaultProperty),
		DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
		public MenuItems MenuItems
		{
			get
			{
				return this.menuItems;
			}
		}

		/// <summary>
		///		Header text of the menu.
		/// </summary>
		[Bindable(true),
		Category("Appearance"),
		Description("Menu header text")]
		public string HeaderText
		{
			get
			{
				return headerText;
			}

			set
			{
				headerText = value;
			}
		}

		/// <summary>
		///		Gets/Sets the visible state of the menu.
		/// </summary>
		[Bindable(true),
		Category("Appearance"),
		Description("Current visual state of the menu")]
		public State State
		{
			get
			{
				return this.state;
			}

			set
			{
				state = value;
			}
		}

		/// <summary>
		///		Width of the menu.
		/// </summary>
		public override Unit Width
		{
			get
			{
				return this.width;
			}

			set
			{
				this.width = value;
			}
		}

		/// <summary>
		///		Image to display for the Hide link of the menu.
		/// </summary>
		[Bindable(true),
		Category("Appearance"),
		Description("Url for the hide image"),
		Editor(typeof(ImageUrlEditor), typeof(UITypeEditor))]
		public string HideImageUrl
		{
			get
			{
				return this.hideImageUrl;
			}

			set
			{
				this.hideImageUrl = value;
			}
		}

		/// <summary>
		///		Image to display for the Show link of the menu.
		/// </summary>
		[Bindable(true),
		Category("Appearance"),
		Description("Url for the show image"),
		Editor(typeof(ImageUrlEditor), typeof(UITypeEditor))]
		public string ShowImageUrl
		{
			get
			{
				return this.showImageUrl;
			}

			set
			{
				this.showImageUrl = value;
			}
		}

		/// <summary>
		///		Image to use for a "New" menu item.
		/// </summary>
		[Bindable(true),
		Category("Appearance"),
		Description("Url for the image to display for menu items designated as 'New'"),
		Editor(typeof(ImageUrlEditor), typeof(UITypeEditor))]
		public string NewImageUrl
		{
			get
			{
				return this.newImageUrl;
			}

			set
			{
				this.newImageUrl = value;
			}
		}

		/// <summary>
		///		Height property
		/// </summary>
		/// <remarks>Hide from propery explorer</remarks>
		[Browsable(false)]
		public override Unit Height
		{
			get
			{
				return base.Height;
			}
			set
			{
				base.Height = value;
			}
		}

		/// <summary>
		///		CssClass property
		/// </summary>
		/// <remarks>Hide from propery explorer</remarks>
		[Browsable(false)]
		public override string CssClass
		{
			get
			{
				return base.CssClass;
			}
			set
			{
				base.CssClass = value;
			}
		}


		/// <summary>
		///		ForeColor property
		/// </summary>
		/// <remarks>Hide from property explorer</remarks>
		[Browsable(false)]
		public override Color ForeColor
		{
			get
			{
				return base.ForeColor;
			}
			set
			{
				base.ForeColor = value;
			}
		}

		/// <summary>
		///		Font propery
		/// </summary>
		/// <remarks>Hide from property explorer</remarks>
		[Browsable(false)]
		public override FontInfo Font
		{
			get
			{
				return base.Font;
			}
		}

		/// <summary>
		///		BackColor property
		/// </summary>
		/// <remarks>Hide from property explorer</remarks>
		[Browsable(false)]
		public override Color BackColor
		{
			get
			{
				return base.BackColor;
			}
			set
			{
				base.BackColor = value;
			}
		}

		/// <summary>
		///		Header background color.
		/// </summary>
		[Bindable(true),
		Category("Appearance"),
		Description("Color used for the menu header background"),
		DefaultValue("#6699cc")]
		public Color HeaderBackColor
		{
			get
			{
				return this.headerBackColor;
			}

			set
			{
				this.headerBackColor = value;
			}
		}

		/// <summary>
		///		Font used for the header text.
		/// </summary>
		[Bindable(true),
		Category("Appearance"),
		Description("Font used for the menu header"),
		PersistenceMode(PersistenceMode.Attribute),
		DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
		public FontInfo HeaderFont
		{
			get
			{
				if (this.headerFont == null)
					this.headerFont = base.Font;

				return this.headerFont;
			}

			set
			{
				this.headerFont = value;
			}
		}

		/// <summary>
		///		Header color of the menu.
		/// </summary>
		[Bindable(true),
		Category("Appearance"),
		Description("Color used for the menu header text"),
		DefaultValue(typeof(Color), "Color.White")]
		public Color HeaderForeColor
		{
			get
			{
				return this.headerForeColor;
			}

			set
			{
				this.headerForeColor = value;
			}
		}

		/// <summary>
		///		Border color of the menu.
		/// </summary>
		[Browsable(true),
		Category("Appearance"),
		Description("Color of the border of the menu"),
		DefaultValue(typeof(Color), "Color.Black")]
		public override Color BorderColor
		{
			get
			{
				return this.borderColor;
			}

			set
			{
				this.borderColor = value;
			}
		}	

		/// <summary>
		///		Border style of the menu.
		/// </summary>
		[Browsable(true),
		Category("Appearance"),
		Description("Style of the menu border")]
		public override BorderStyle BorderStyle
		{
			get
			{
				return this.borderStyle;
			}

			set
			{
				this.borderStyle = value;
			}
		}

		/// <summary>
		///		Border width of the menu.
		/// </summary>
		[Browsable(true),
		Category("Appearance"),
		Description("Width of the menu border"),
		DefaultValue(typeof(Unit),"Unit.Empty")]
		public override Unit BorderWidth
		{
			get
			{
				return this.borderWidth;
			}

			set
			{
				this.borderWidth = value;
			}
		}

		/// <summary>
		///		Group to which this menu belongs to.  The makes all panels in the same group mutually exclusive.
		/// </summary>
		[Browsable(true),
		Category("Behavior"),
		Description("Group that this XPanel belongs to"),
		DefaultValue(typeof(Unit),"Unit.Empty")]
		public string GroupName
		{
			get
			{
				return this.groupName;
			}

			set
			{
				this.groupName = value;
			}
		}
		#endregion Properties

		#region Methods
		/// <summary>
		///		Creates a new XPanel object
		/// </summary>
		public XPanel()
		{
			menuItems = new MenuItems();
		}

		/// <summary>
		/// This member overrides Control.Render.
		/// </summary>
		protected override void Render(System.Web.UI.HtmlTextWriter writer)
		{
			// these two call's are needed to show the control at design time
			ChildControlsCreated = false;
			
			EnsureChildControls();
			
			// delegate the actual rendering to baseclass
			base.Render(writer); 
		}

		/// <summary>
		///		Adds a new menu item to the menu
		/// </summary>
		/// <param name="text">Text of the menu item</param>
		/// <param name="url">Url of the menu item</param>
		/// <param name="urlTarget">Target of the menu item</param>
		public void AddMenuItem(string text, string url, TargetType urlTarget)
		{
			this.MenuItems.Add(new MenuItem(text, url, urlTarget));
		}

		/// <summary>
		/// This member overrides Control.AddParsedSubObject.
		/// </summary>
		protected override void AddParsedSubObject(object obj)
		{
			MenuItem menuItem = obj as MenuItem;
			if (menuItem != null)
			{
				this.MenuItems.Add(menuItem);
				return;
			}

			base.AddParsedSubObject(obj); 
		}

		/// <summary>
		///		Overrides the CreateChildControls function of the WebControl base class
		/// </summary>
		protected override void CreateChildControls()
		{	
			// call base method
			base.CreateChildControls();

			HtmlTableRow tr = null;
			HtmlTableCell tc = null;
			HtmlTable menuItemTable = null;
			HtmlTable navTable = null;
			string panelID = "SectionPanel;" + this.GroupName + ";" + this.ShowImageUrl;
			string headerText = this.HeaderText;

			if ((headerText == null || headerText.Length == 0) && IsDesignTime)
				headerText = "[" + this.ID + "]";

			navTable = new HtmlTable();
			navTable.CellPadding = 0;
			navTable.CellSpacing = 0;
			navTable.Width = this.Width.ToString();
			navTable.Border = 0;
			navTable.ID = "MenuSection";

			if (this.BorderStyle != BorderStyle.None && this.BorderStyle != BorderStyle.NotSet)
			{
				navTable.Attributes.Add("style", "BORDER-RIGHT: " + this.BorderWidth.Value + "px " + this.BorderStyle.ToString() + 
					"; BORDER-TOP: " + this.BorderWidth.Value + "px " + this.BorderStyle.ToString() +
					"; BORDER-LEFT: " + this.BorderWidth.Value + "px " + this.BorderStyle.ToString() +
					"; BORDER-BOTTOM: " + this.BorderWidth.Value + "px " + this.BorderStyle.ToString());
				navTable.BorderColor = ColorTranslator.ToHtml(this.BorderColor);
			}

			//Start Header Row
			tr = new HtmlTableRow();
			navTable.Rows.Add(tr);

			tc = new HtmlTableCell();
			tc.Width = "95%";
			tc.Height = "19";
			tc.BgColor = ColorTranslator.ToHtml(this.HeaderBackColor);

			//Create menu item table
			menuItemTable = new HtmlTable();
			menuItemTable.ID = panelID;
			menuItemTable.Border = 0;
			menuItemTable.CellPadding = 0;
			menuItemTable.CellSpacing = 0;
			menuItemTable.Width = "100%";

			HyperLink lnkHeader = new HyperLink();
			lnkHeader.Width = Unit.Percentage(100);
			lnkHeader.EnableViewState = false;
			lnkHeader.Text = "&nbsp;" + headerText;
			lnkHeader.ForeColor = this.HeaderForeColor;
			lnkHeader.Font.Bold  = this.HeaderFont.Bold;
			lnkHeader.Font.Italic = this.HeaderFont.Italic;
			lnkHeader.Font.Name = this.HeaderFont.Name;
			lnkHeader.Font.Size = this.HeaderFont.Size;
			lnkHeader.Font.Strikeout = this.HeaderFont.Strikeout;
			lnkHeader.Font.Underline = this.HeaderFont.Underline;
			lnkHeader.Attributes["href"] = XPanel.javaScriptVoid;
			lnkHeader.Style.Add("text-decoration", "none");
			lnkHeader.Attributes["onclick"] = "expandit('" + this.ClientID + "_" + menuItemTable.ClientID + "', '" + this.HideImageUrl + 
												"', '" + this.ShowImageUrl + "', '" + this.GroupName + "')";
			tc.Controls.Add(lnkHeader);
			tr.Cells.Add(tc);

			tc = new HtmlTableCell();
			tc.Width = "5%";
			tc.Height = "19";
			tc.Align = "Right";
			tc.BgColor = ColorTranslator.ToHtml(this.HeaderBackColor);
			tc.NoWrap = true;

			if (this.HideImageUrl != null && this.HideImageUrl.Length > 0)
			{
				System.Web.UI.WebControls.Image sectionHeaderArrowImage = new System.Web.UI.WebControls.Image();
				sectionHeaderArrowImage.ID = "ArrowImage";

				if (this.State == State.Expanded)
					sectionHeaderArrowImage.ImageUrl = this.HideImageUrl;
				else
					sectionHeaderArrowImage.ImageUrl = this.ShowImageUrl;

				HyperLink sectionHeaderLink = new HyperLink();
				sectionHeaderLink.Attributes["href"] = XPanel.javaScriptVoid;
				sectionHeaderLink.Attributes["onClick"] = "expandit('" + this.ClientID + "_" + menuItemTable.ClientID + "', '" + this.HideImageUrl + "', '" + this.ShowImageUrl + "', '" + this.GroupName + "')";
				sectionHeaderLink.Text = "Hello";
			
				//add image to link
				sectionHeaderLink.Controls.Add(sectionHeaderArrowImage);
				tc.Controls.Add(sectionHeaderLink);
			}

			tr.Cells.Add(tc);
			//End Header Row

			//Start Menu Items Row
			tr = new HtmlTableRow();
			navTable.Rows.Add(tr);

			tc = new HtmlTableCell();
			tc.ColSpan = 3;
			tc.BgColor = "#ffffff";
	
			tc.Height = "1px";
			tc.Width = "100%";

			if (this.State == State.Expanded)
				menuItemTable.Attributes["style"] = "display:inline";
			else
				menuItemTable.Attributes["style"] = "display:none";

			tc.Controls.Add(menuItemTable);
			tr.Cells.Add(tc);

			//Menu Items
			if (this.menuItems != null)
			{
				foreach(MenuItem menuItem in this.menuItems)
				{
					tr = new HtmlTableRow();
					menuItemTable.Rows.Add(tr);
			
					//Image cell
					tc = new HtmlTableCell();
					tc.Width = "2%";
					tc.BgColor = ColorTranslator.ToHtml(menuItem.BackColor);

					if (menuItem.CssClass != null && menuItem.CssClass.Length > 0)
						tc.Attributes.Add("class", menuItem.CssClass);

					if (menuItem.ImageUrl != null && menuItem.ImageUrl.Length > 0)
					{
						System.Web.UI.WebControls.Image image = new System.Web.UI.WebControls.Image();
						image.ImageUrl = menuItem.ImageUrl;
						image.ImageAlign = ImageAlign.Middle;
						
						tc.Controls.Add(image);
					}

					tc.Controls.Add(new LiteralControl("&nbsp;"));

					tr.Controls.Add(tc);
					
					//Menu Item Cell
					tc = new HtmlTableCell();
					tc.Width = "98%";
					tc.BgColor = ColorTranslator.ToHtml(menuItem.BackColor);
					
					if (menuItem.CssClass != null && menuItem.CssClass.Length > 0)
						tc.Attributes.Add("class", menuItem.CssClass);

					HyperLink lnkMenuItem = new HyperLink();
					lnkMenuItem.NavigateUrl = menuItem.NavigateUrl;
					lnkMenuItem.Text = menuItem.Text;
					if (menuItem.CssClass == String.Empty || menuItem.CssClass.Length == 0 || menuItem.CssClass == null)
					{
						lnkMenuItem.ForeColor = menuItem.ForeColor;
						lnkMenuItem.Font.Bold  = menuItem.Font.Bold;
						lnkMenuItem.Font.Italic = menuItem.Font.Italic;
						lnkMenuItem.Font.Name = menuItem.Font.Name;
						lnkMenuItem.Font.Size = menuItem.Font.Size;
						lnkMenuItem.Font.Strikeout = menuItem.Font.Strikeout;
						lnkMenuItem.Font.Underline = menuItem.Font.Underline;
					}
					else
					{
						lnkMenuItem.CssClass = menuItem.CssClass;
					}

					switch (menuItem.Target)
					{
						case TargetType._blank:
							lnkMenuItem.Target = "_blank";
							break;
						case TargetType._parent :
							lnkMenuItem.Target = "_parent";
							break;
						case TargetType._self:
							lnkMenuItem.Target = "_self";
							break;

						case TargetType._top:
							lnkMenuItem.Target = "_top";
							break;

						default:
							lnkMenuItem.Target = "_blank";
							break;
					}

					tc.Controls.Add(lnkMenuItem);

					if (this.NewImageUrl != null && menuItem.ShowNewImage)
					{
						System.Web.UI.WebControls.Image image = new System.Web.UI.WebControls.Image();
						image.ImageUrl = this.NewImageUrl;
						image.ImageAlign = ImageAlign.AbsMiddle;
						tc.Controls.Add(new LiteralControl("&nbsp;"));
						tc.Controls.Add(image);
					}

					tr.Cells.Add(tc);
				}
			}

			//Add table to the page
			this.Controls.Add(navTable);

			if (!this.IsDesignTime)
			{
				if (!this.Page.IsClientScriptBlockRegistered("RuntimeScript"))
					this.Page.RegisterClientScriptBlock("RuntimeScript", Script.GetRuntimeScript(this.ID));
			}
		}

		/// <summary>
		///		IsDesignTime property
		/// </summary>
		protected virtual bool IsDesignTime
		{
			get 
			{
				return (Context == null);
			}
		}
		#endregion Methods
	}
}

namespace ComputerBuff.Controls.XPanel.Design
{
	using ComputerBuff.Controls;

	/// <summary>
	/// Interacts with the parser to build a PanelBar control.
	/// </summary>
	public class MenuItemBuilder : ControlBuilder
	{
		/// <summary>
		/// This member overrides ControlBuilder.GetChildControlType.
		/// </summary>
		public override Type GetChildControlType(string tagName, IDictionary attributes)
		{
			// check is the tag is an PanelBarItem tag
			if (tagName.ToLower().IndexOf("menuitem") >= 0)
			{
				// yes, so return PanelBarItem type
				return typeof(MenuItem);
			}
			return null;
		}
	}

	/// <summary>
	///		This class is used for design-time support only.
	/// </summary>
	/// <remarks>This class cannot be inherited</remarks>
	public sealed class XPanelDesigner : ControlDesigner 
	{
		private XPanel xPanel;
		private DesignerVerbCollection m_Verbs;

		/// <summary>
		///		DesignerVerbCollection is overridden from ComponentDesigner
		/// </summary>
		public override DesignerVerbCollection Verbs
		{
			get 
			{
				if (m_Verbs == null) 
				{
					// Create and initialize the collection of verbs
					m_Verbs = new DesignerVerbCollection();
            
					m_Verbs.Add( new DesignerVerb("About XPanel...", new 
						EventHandler(OnAboutSelected)) );
				}
				return m_Verbs;
			}
		}

		private void OnAboutSelected(object sender, EventArgs args) 
		{
			About about = new About();

			about.ShowDialog();
		}

		/// <summary>
		///		Gets the HTML that is used to represent the control at design time.
		/// </summary>
		/// <returns>string - Design-Time HTML</returns>
		public override string GetDesignTimeHtml()
		{
			// get PanelBar control
			xPanel = (XPanel)Component;
			// return designtime html. Render the control to a stringwriter and return the result
			using(StringWriter stringWriter = new StringWriter()) 
			using(HtmlTextWriter htmlTextWriter = new HtmlTextWriter(stringWriter))
			{
				xPanel.RenderControl(htmlTextWriter);
				return stringWriter.ToString(); 
			}
		}

		/// <summary>
		///		Gets the HTML that provides information about the specified exception.
		///		This method is typically called after an error has been encountered at design time.
		/// </summary>
		/// <param name="e">The exception that occurred. </param>
		/// <returns>string - The HTML for the specified exception.</returns>
		protected override string GetErrorDesignTimeHtml(Exception e)
		{
			return e.Message;
		}
	}
}
