{ This unit contains routines for string parsing and evaluation.  }
{ Written by Jari Karjala.  Bug reports to t33869n@saha.hut.FI    }
{ The original inspiration is from OtaData's Pascal-book.         }

{ String is first transformed into a tree structure and then it   }
{ can be evaluated by using a separate function. This is very     }
{ useful, when you are, for example, plotting a graph.            }
{ Currently only one variable is allowed.  There is no limit in   }
{ nesting parentheses. Procedures can handle most basic functions }
{ like trigonometric, hyperbolic and logarithmic functions. Error }
{ handling in quite extensive, and error messages are returned to }
{ caller in strings and therefore they can be handled by the      }
{ calling routine.  Some overflow errors may pass the checks.     }

Unit StrEval;

{$IFDEF CPU87}
{$N+}
{ This should declare Reals in this unit as Doubles. Not tested.  }
Type Real = Double;
{$ENDIF}

Interface

{ This procedure parses the expression in the string ST and assigns the    }
{ tree root to pointer E.  User must supply a single character variable-   }
{ identifier. The string variable ERR returns possible error message.      }

Procedure ProcessSTR(st:string; var e:pointer; varname:char; var err:string);

{ This function evaluates the expression tree created by ProcessSTR().     }
{ Pointer E points to tree root. X is the current value of user variable.  }
{ The string ERR returns possible error message. If ERR<>'' then the value }
{ of the Eval() is undefined.                                              }

Function  Eval(e:pointer; x:real; var err:string):real;

Implementation

type
  symbol=(fin,constant,plus,minus,times,divide,
          fabs,fsin,fcos,ftan,fasin,facos,fatan,
          fsinh,fcosh,ftanh,fartanh,
          fsqr,fsqrt,flog,fexp,fln,
          unaryminus,variable,leftpar,rightpar);
  operator=constant..variable;
  ExprPtr=^expr;
  expr=record
    case oprt:operator of
      constant:(value:real);
      plus,minus,times,divide:(opnd1,opnd2:ExprPtr);
      variable:();
      unaryminus:(opnd:ExprPtr);
      fsin:(argum:ExprPtr)
    end;

Procedure ProcessSTR(st:string; var e:pointer; varname:char; var err:string);
const
  EndChar   = '@';
var
  sy           : symbol;
  leadingminus : boolean;
  input        : char;
  strptr       : integer;
  v            : real;

Procedure GetSymbol;
var
  fn:string[4];

Procedure get(var c:char);
begin
  c:=upcase(st[strptr]);
  strptr:=succ(strptr);
  if strptr>length(st) then c:=EndChar;
end;

Function value(st:string):real;
var er1,er2:integer; c:real;
begin
  val(copy(st,pred(strptr),length(st)), c, er1);
  if er1<>0 then val(copy(st,pred(strptr),pred(er1)), c, er2);
  strptr:=strptr+er1-2;
  value:=c;
end;

begin { GetSymbol }
  while input=' ' do get(input);
  if input in ['0'..'9'] then begin
    sy:=constant;
    v:=value(st);
    get(input)
  end
  else if input in ['+','-','*','/','(',')'] then begin
    case input of
      '+':sy:=plus;
      '-':sy:=minus;
      '*':sy:=times;
      '/':sy:=divide;
      '(':sy:=leftpar;
      ')':sy:=rightpar
    end;
    get(input);
    if sy in [plus..divide] then
      if input in ['+','-','*','/'] then begin
        e:=nil; err:='Too Many Operators'; exit
    end;
  end
  else if input=EndChar then sy:=fin
  else if input in ['A'..'Z'] then begin
	fn:=input;
	get(input);
	if (fn=varname) and not (input in ['A'..'Z']) then sy:=variable
	else begin
          while input in ['A'..'Z'] do begin
            if length(fn)<5 then
              if input in ['A'..'Z'] then fn:=fn+input else fn:=fn+' ';
            get(input);
          end;
          fn:=fn+'    ';
          if fn='ABS ' then sy:=fabs else
          if fn='SIN ' then sy:=fsin else
          if fn='COS ' then sy:=fcos else
          if fn='TAN ' then sy:=ftan else
          if fn='ARCS' then sy:=fasin else
          if fn='ARCC' then sy:=facos else
          if fn='ARCT' then sy:=fatan else
          if fn='SINH' then sy:=fsinh else
          if fn='COSH' then sy:=fcosh else
          if fn='TANH' then sy:=ftanh else
          if fn='ARTA' then sy:=fartanh else
          if fn='SQR ' then sy:=fsqr else
          if fn='SQRT' then sy:=fsqrt else
          if fn='EXP ' then sy:=fexp else
          if fn='LOG ' then sy:=flog else
          if fn='LN ' then sy:=fln else begin
            e:=nil; err:='Unknown Identifier '+fn;
          exit
         end
       end
     end
  else begin e:=nil; err:='Syntax Error'; exit end
end; { GetSymbol }

Procedure GetExpression(var e:ExprPtr);
var
  y:ExprPtr;
  op:operator;

Procedure GetTerm(var e:ExprPtr);
var
  y:ExprPtr;
  op:operator;

Procedure GetFactor(var e:ExprPtr);
begin
  case sy of
    constant:begin
               new(e);
               e^.oprt:=constant;
               e^.value:=v
             end;
    variable:begin
               new(e);
               e^.oprt:=variable
             end;
    leftpar :begin
               GetSymbol; if err<>'' then exit;
               GetExpression(e);
               if sy<>rightpar then begin
                 e:=nil; err:='Missing )';
                 exit
               end;
             end;
    fabs..fln:
             begin
               new(e);
               e^.oprt:=sy;
               GetSymbol; if err<>'' then exit;
               if sy=leftpar then begin
                 GetSymbol; if err<>'' then exit;
                 GetExpression(e^.argum);
                 if sy<>rightpar then begin
                   e:=nil; err:='Missing )';
                   exit
                 end
               end
               else begin
                 e:=nil; err:='Missing (';
                 exit
               end;
             end;
    fin,plus,minus,times,divide,rightpar:begin
               e:=nil; err:='Error in Factor';
               exit
             end;
  end;
  GetSymbol;
end;

begin { GetTerm }
  GetFactor(e);
  while sy in [times,divide] do
  begin
    new(y);
    with y^ do begin
      oprt:=sy;
      opnd1:=e;
      GetSymbol; if err<>'' then exit;
      GetFactor(opnd2); if err<>'' then exit;
    end;
    e:=y;
    if err<>'' then exit;
  end
end;

begin { GetExpression }
  leadingminus:=(sy=minus);
  if sy in [plus,minus] then GetSymbol; if err<>'' then exit;
  GetTerm(e); if err<>'' then exit;
  if leadingminus then begin
    new(y);
    with y^ do
    begin
      oprt:=unaryminus;
      opnd:=e
    end;
    if err<>'' then exit;
    e:=y
  end;
  while sy in [plus,minus] do begin
    new(y);
    with y^ do
    begin
      oprt:=sy;
      opnd1:=e;
      GetSymbol; if err<>'' then exit;
      GetTerm(opnd2); if err<>'' then exit;
    end;
    if err<>'' then exit;
    e:=y
  end
end; { GetExpression }

begin { ProcessSTR }
  st:=st+EndChar;
  input:=' '; strptr:=1; err:='';
  GetSymbol; if err<>'' then exit;
  GetExpression(ExprPtr(e));
  if err='' then if sy<>fin then begin e:=nil; err:='Unexpected end' end;
end; { ProcessSTR }

Function Eval(e:pointer; x:real; var err:string):real;
  Function log(x:real):real;
    begin
      if x>0 then log:=ln(x)/2.302585093  { ln(10) }
             else err:='LOG error'
    end;
  Function arcsin(x:real):real;
    begin
      if abs(x)<=1 then arcsin:=arctan(x/sqrt(-x*x+1))
                   else err:='ARCSIN error'
    end;
  Function arccos(x:real):real;
    begin
      if abs(x)<=1 then arccos:=1.5707963268-arctan(x/sqrt(-x*x+1))
                   else err:='ARCCOS error'
    end;
  Function sinh(x:real):real;
    begin
      sinh:=0.5*(exp(x)-exp(-x))
    end;
  Function cosh(x:real):real;
    begin
      cosh:=0.5*(exp(x)+exp(-x))
    end;
  Function tanh(x:real):real;
    begin
      tanh:=(exp(2*x)-1)/(exp(2*x)+1)
    end;
  Function artanh(x:real):real;
    begin
      if abs(x)<1 then artanh:=0.5*ln((1+x)/(1-x))
                  else err:='ARTANH error'
    end;

  Function ev(e:ExprPtr):real;
  var a:real;
  begin
    with e^ do
      case oprt of
        constant   : ev:=value;
        plus       : ev:=ev(opnd1)+ev(opnd2);
        minus      : ev:=ev(opnd1)-ev(opnd2);
        times      : ev:=ev(opnd1)*ev(opnd2);
        divide     : begin a:=ev(opnd2);
                       if a<>0 then ev:=ev(opnd1)/a else err:='Division by 0'
                     end;
        unaryminus : ev:=-ev(opnd);
        variable   : ev:=x;
        fabs       : ev:=abs(ev(argum));
        fsin       : ev:=sin(ev(argum));
        fcos       : ev:=cos(ev(argum));
        ftan       : begin a:=cos(ev(argum));
                       if a<>0 then ev:=sin(ev(opnd1))/a else err:='TAN error'
                     end;
        fasin      : ev:=arcsin(ev(argum));
        facos      : ev:=arccos(ev(argum));
        fatan      : ev:=arctan(ev(argum));
        fsinh      : ev:=sinh(ev(argum));
        fcosh      : ev:=cosh(ev(argum));
        ftanh      : ev:=tanh(ev(argum));
        fartanh    : ev:=artanh(ev(argum));
        fsqr       : ev:=sqr(ev(argum));
        fsqrt      : begin a:=ev(argum);
                       if a>=0 then ev:=sqrt(a) else err:='SQRT error'
                     end;
        fexp       : begin a:=ev(argum);
                       if a<70 then ev:=exp(a) else err:='EXP overflow'
                     end;
        fln        : begin a:=ev(argum);
                       if a>0 then ev:=ln(a) else err:='LN error'
                     end;
        flog       : ev:=log(ev(argum))
      end;
  end;
begin { eval }
  err:='';
  eval:=ev(e)
end; { eval }
end. { evalunit }