/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * FTP level specific code
 */

#include <general.h>
#include <protospec.h>
#include <arpa/ftp.h>

char *index();

/*  */

ftp_conn_request(addr, a_ch, errind)
    struct address_t *addr;
    channel_t *a_ch;
    error_t   *errind;
{
    int s;
    
    pprintf("ftp_conn_request(0x%x, 0x%x, 0x%x)\n",
	    addr, a_ch, errind);
    
    if ((s = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "socket call", "ftp_conn_request",
	       getsyserr());
	*errind = NOTOK;
	return;
    }

    if (connect(s, addr, sizeof(*addr)) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "connect call", "ftp_conn_request",
	       getsyserr());
	*errind = NOTOK;
	return;
    }

    a_ch->control_in = fdopen(s, "r");
    if (a_ch->control_in == NULL) {
	eprintf(EF_SYSCALL, COMMUNICATION, "fdopen call", "ftp_conn_request",
	       getsyserr());
	(void)close(s);
	*errind = NOTOK;
	return;
    }
    setbuf(a_ch->control_in, NULL);

#ifdef SYSV
    s = dup(s);
    if (s == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "dup call", "ftp_conn_request",
	       getsyserr());
	(void)close(s);
	*errind = NOTOK;
	return;
    }
#endif SYSV	
    a_ch->control_out = fdopen(s, "w");
    if (a_ch->control_out == NULL) {
	eprintf(EF_SYSCALL, COMMUNICATION, "fdopen call", "ftp_conn_request",
	       getsyserr());
	(void)close(s);
	*errind = NOTOK;
	return;
    }
	
    a_ch->data = NULL;
    *errind = OK;
} /* ftp_conn_request */

/*  */

ftp_await_conn_ind(server, a_ch, errind)
    struct server_t *server;	
    channel_t *a_ch;
    error_t *errind;
{
    struct sockaddr_in otheraddr;
    int len = sizeof(otheraddr), s;
    
    pprintf("ftp_await_conn_ind(server %d)\n", server->sock);

    if ((s = accept(server->sock, &otheraddr, &len)) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "accept call", "ftp_await_conn_ind",
	       getsyserr());
	*errind = NOTOK;
	return;
    }

    a_ch->control_in = fdopen(s, "r");
    if (a_ch->control_in == NULL) {
	eprintf(EF_SYSCALL, COMMUNICATION, "fdopen call", "ftp_await_conn_ind",
	       getsyserr());
	(void)close(s);
	*errind = NOTOK;
	return;
    }
    setbuf(a_ch->control_in, NULL);

#ifdef SYSV
    s = dup(s);
    if (s == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "dup call", "ftp_await_conn_ind",
	       getsyserr());
	(void)close(s);
	*errind = NOTOK;
	return;
    }
#endif SYSV	
    a_ch->control_out = fdopen(s, "w");
    if (a_ch->control_out == NULL) {
	eprintf(EF_SYSCALL, COMMUNICATION, "fdopen call", "ftp_await_conn_ind",
	       getsyserr());
	(void)close(s);
	*errind = NOTOK;
	return;
    }
    
    a_ch->data = NULL;
    *errind = OK;
} /* ftp_await_conn_ind */

/*  */

ftp_disc_request(ch, errind)
    channel_t ch;
    error_t   *errind;
{
    int s;
    
    pprintf("ftp_disc_request(%d, 0x%x)\n", fileno(ch.control_in), errind);
    
    if (fclose(ch.control_in)== NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "fclose()", "ftp_disc_request",
	       getsyserr());
    }
    (void)fclose(ch.control_out);

    if (ch.data != NULL) {
	if (fclose(ch.data)== NOTOK) {
	    pprintf(EF_IN4X, COMMUNICATION, "closing data connection",
		    "ftp_disc_request",
		    getsyserr());
	}
    }

    *errind = OK;
} /* ftp_disc_request */

/*  */

ftp_await_disc_ind(ch, errind)
    channel_t ch;
    error_t *errind;
{
    char buf;	/* very small buffer */
    int cc;	/* Not char to work with int constant EOF */

    
    pprintf("ftp_await_disc_ind(ch %d, 0x%x)\n",
	    fileno(ch.control_in), errind);

    cc = read(fileno(ch.control_in), &buf, 1);
    if (cc == NOTOK) {
	if (errno == ECONNRESET)
	    cc = 0;		/* Other end was fast to close */
	else {
	    int tmp = errno;
	    eprintf(EF_IN3, COMMUNICATION, "read on a tcp socket",
		    "ftp_await_disc_ind");
	    (void)fclose(ch.control_in);
	    (void)fclose(ch.control_out);
	    errno = tmp;
	    *errind = NOTOK;
	    return;
	}
    }
    
    if (cc != 0) {
	eprintf(EF_IN4, COMMUNICATION, "receiving data",
	       "ftp_await_disc_ind",
	       "\nShould get a 0 i.e. EOF = DiscIndication");
    }

    if (fclose(ch.control_in)== NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "fclose call", "ftp_await_disc_ind",
	       getsyserr());
    }
    (void)fclose(ch.control_out);

    if (ch.data != NULL) {
	if (fclose(ch.data)== NOTOK) {
	    pprintf(EF_IN4X, COMMUNICATION, "closing data connection",
		    "ftp_await_disc_ind",
		    getsyserr());
	}
    }

    *errind = OK;
} /* ftp_await_disc_ind */

/*  */

ftp_create_server(aa_server, aa_addr, errind)
    struct server_t **aa_server;
    struct address_t **aa_addr;
    error_t   *errind;
{

    int s;
    struct sockaddr_in myaddr;
    int len = sizeof(myaddr);
    
    pprintf("ftp_create_server(0x%x, 0x%x, 0x%x)\n",
		aa_server, aa_addr, errind);

    if ((s = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "socket call", "ftp_create_server",
	       getsyserr());
	*errind = NOTOK;
	return;
    }

    myaddr.sin_family = AF_INET;
    myaddr.sin_port = 0;
    myaddr.sin_addr.s_addr = INADDR_ANY;

    /* BUG FIX */
    bzero(myaddr.sin_zero, sizeof (myaddr.sin_zero));

    if (bind(s, &myaddr, sizeof(myaddr)) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "bind call", "ftp_create_server",
	       getsyserr());
	return NOTOK;
    }
    if (listen(s, 5) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "listen call", "ftp_create_server",
	       getsyserr());
	*errind = NOTOK;
	return;
    }

    if (getsockname(s, &myaddr, &len) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "getsockname", "ftp_create_server",
	       getsyserr());
	*errind = NOTOK;
	return;
    }
    if (len != sizeof(myaddr)) {
	eprintf(EF_SYSCALL, COMMUNICATION, "getsockname", "ftp_create_server",
	       "\nBad length on returned address");
	*errind = NOTOK;
	return;
    }
    if (get_myipaddr(&myaddr.sin_addr) == NOTOK) {
	eprintf(EF_IN3, INTERNAL, "init_myipaddr", "CreateServer");
	*errind = NOTOK;
	return;
    }

    pprintf("Created socket %d with address: 0x%x:%d\n",
		s, myaddr.sin_addr.s_addr, myaddr.sin_port);

    *aa_server = (struct server_t *)malloc(sizeof(struct server_t));
    if (*aa_server == NULL) {
	eprintf(EF_IN3, INTERNAL, RESOURCE, "CreateServer");
	*errind = NOTOK;
	return;
    }
    *aa_addr = (struct address_t *)malloc(sizeof(struct address_t));
    if (*aa_addr == NULL) {
	eprintf(EF_IN3, INTERNAL, RESOURCE, "CreateServer");
	*errind = NOTOK;
	free((char *)*aa_server);
	return;
    }
    (*aa_server)->sock = s;
    **aa_addr = myaddr;		/* struct copy */
    *errind = OK;
} /* ftp_create_server */

/*  */

ftp_destroy_server(a_server, errind)
    struct server_t *a_server;
    error_t   *errind;
{
    
    pprintf("ftp_destroy_server(0x%x, 0x%x)\n", a_server, errind);

    if (close(a_server->sock) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "close", "ftp_destroy_server",
	       getsyserr());
	free((char *)a_server);
	*errind = NOTOK;
	return;
    }

    free((char *)a_server);
	
    *errind = OK;
} /* ftp_destroy_server */

/*  */

ftp_start_bulk_get(ch, errind)
    channel_t *ch;
    error_t *errind;
{
    pprintf("ftp_start_bulk_get(%d:0x%x, 0x%x)\n",
	    fileno(ch->control_in), (ch->data), errind);

    if (ch->data != NULL) {
	pprintf("closing data connection in ftp_start_bulk_get\n");
	(void)fclose(ch->data);
	ch->data = NULL;
    }
    
    if (client_init(ch, "RETR", "r") == NOTOK) {
	pprintf(EF_IN3, INTERNAL, "client_init", "ftam_start_bulk_get");
	(void)fclose(ch->data);
	ch->data = NULL;
	*errind = NOTOK;
        return;
    }

    *errind = OK;
} /* ftp_start_bulk_get */

/*  */

ftp_await_start_bulk_get(ch, errind)
    channel_t *ch;
    error_t *errind;
{
    pprintf("ftp_await_start_bulk_get(%d:0x%x, 0x%x)\n",
	    fileno(ch->control_in), ch->data, errind);

    if (ch->data != NULL) {
	pprintf("closing data connection in ftp_await_start_bulk_get\n");
	(void)fclose(ch->data);
	ch->data = NULL;
    }
    
    if (server_init(ch, "RETR", "w") == NOTOK) {
	pprintf(EF_IN3, INTERNAL, "server_init", "ftam_await_start_bulk_get");
	(void)fclose(ch->data);
	ch->data = NULL;
	*errind = NOTOK;
        return;
    }

    *errind = OK;
} /* ftp_await_start_bulk_get */

/*  */

ftp_start_bulk_put(ch, errind)
    channel_t *ch;
    error_t *errind;
{
    pprintf("ftp_start_bulk_put(%d:0x%x, 0x%x)\n",
	    fileno(ch->control_in), (ch->data), errind);

    if (ch->data != NULL) {
	pprintf("closing data connection in ftp_start_bulk_put\n");
	(void)fclose(ch->data);
	ch->data = NULL;
    }
    
    if (client_init(ch, "STOR", "w") == NOTOK) {
	pprintf(EF_IN3, INTERNAL, "client_init", "ftam_start_bulk_put");
	(void)fclose(ch->data);
	ch->data = NULL;
	*errind = NOTOK;
        return;
    }

    *errind = OK;
} /* ftp_start_bulk_put */

/*  */
    
ftp_await_start_bulk_put(ch, errind)
    channel_t *ch;
    error_t *errind;
{
    pprintf("ftp_await_start_bulk_put(%d:0x%X, 0x%x)\n",
	    fileno(ch->control_in), ch->data, errind);

    if (ch->data != NULL) {
	pprintf("closing data connection in ftp_await_start_bulk_put\n");
	(void)fclose(ch->data);
	ch->data = NULL;
    }
    
    if (server_init(ch, "STOR", "r") == NOTOK) {
	pprintf(EF_IN3, INTERNAL, "server_init", "ftam_await_start_bulk_put");
	(void)fclose(ch->data);
	ch->data = NULL;
	*errind = NOTOK;
        return;
    }

    *errind = OK;
} /* ftp_await_start_bulk_put */

/*  */

ftp_await_bulk_stop(ch, errind)
    channel_t *ch;
    error_t *errind;
{
    pprintf("ftp_await_bulk_stop(%d:%d, 0x%x)\n",
	    fileno(ch->control_in), fileno(ch->data), errind);

    reply(ch->control_out, 226, "Transfer complete.");

    (void)fclose(ch->data);
    ch->data = NULL;
    *errind = OK;
} /* ftam_await_bulk_stop */

/*  */

int client_init(ch, cmd, mode)
    channel_t *ch;
    char *cmd, *mode;
{
    struct sockaddr_in data_addr;
    struct sockaddr_in from;
    register char *p, *a;
    int result, len;
    int s, so;
    
    pprintf("client_init(%d:0x%x, %s, %s)\n",
	    fileno(ch->control_in), ch->data, cmd, mode);
    
    s = socket(AF_INET, SOCK_STREAM, 0);
    if (s == NOTOK) {
	eprintf(EF_SYSCALL, INTERNAL, "socket", "client_init", getsyserr());
	return NOTOK;
    }

    data_addr.sin_family = AF_INET;
    data_addr.sin_port = 0;
    data_addr.sin_addr.s_addr = INADDR_ANY;

    /* BUG FIX */
    bzero(data_addr.sin_zero, sizeof (data_addr.sin_zero));

    if (bind(s, &data_addr, sizeof(data_addr)) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "bind call", "tcp_create_server",
	       getsyserr());
	return NOTOK;
    }
    if (listen(s, 1) == NOTOK) {
	eprintf(EF_SYSCALL, INTERNAL, "listen", "client_init", getsyserr());
	return NOTOK;
    }
    
    
    len = sizeof (data_addr);
    if (getsockname(s, (char *)&data_addr, &len) < 0) {
	eprintf(EF_SYSCALL, INTERNAL, "getsockname", "client_init",
		getsyserr());
	return NOTOK;
    }
    if (get_myipaddr(&data_addr.sin_addr) == NOTOK) {
	eprintf(EF_IN3, INTERNAL, "get_myipaddr", "client_init");
	return NOTOK;
    }

    a = (char *)&data_addr.sin_addr;
    p = (char *)&data_addr.sin_port;
#define	UC(b)	(((int)b)&0xff)
    (void)command(ch->control_out, "PORT %d,%d,%d,%d,%d,%d",
		UC(a[0]), UC(a[1]), UC(a[2]), UC(a[3]),
		UC(p[0]), UC(p[1]));
    
    if (getreply(ch->control_in, 0) != COMPLETE) {
	pprintf(EF_IN3, INTERNAL, "command", "client_init");
	return NOTOK;
    }

    (void)command(ch->control_out, "%s", cmd);
    if (getreply(ch->control_in, 0) != PRELIM) {
	pprintf(EF_IN3, INTERNAL, "command", "client_init");
        return NOTOK;
    }

    len = sizeof(struct sockaddr);
    so = accept(s, (struct sockaddr *) &from, &len);
    if (so == NOTOK) {
	eprintf(EF_SYSCALL, INTERNAL, "accept", "client_init", getsyserr());
	(void)close(s);
	return NOTOK;
    }

    (void)close(s);
    ch->data = fdopen(so, mode);
    if (ch->data == NULL) {
	eprintf(EF_SYSCALL, INTERNAL, "fdopen", "client_init",
		getsyserr());
	(void)close(so);
	return NOTOK;
    }

    return OK;
} /* client_init */

/*  */


ftp_report_error(err, str)
    error_t *err;
    char *str;
{
    
    pprintf("ftp_report_error(0x%x)\n", err);
    
    if (err == NULL) {
	eprintf(EF_SYSCALL, COMMUNICATION, "Error", str,
	       getsyserr());
    } else {
	eprintf(EF_SYSCALL, COMMUNICATION, "FTP protocol", str,
	       getsyserr());
    }
} /* ftp_report_error */

/*  */

/*
 * The rest of the code is based on the UCB source code fot ftp and ftpd
 * having this copyright notice:
 *
 * Copyright (c) 1985 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * specifies the terms and conditions for redistribution.
 */

static code = 0;

/*  */

/*VARARGS2*/
command(f, fmt, args)
    FILE *f;
    char *fmt;
{
    int res;
    
    pprintf("command(%d, %s)\n", fileno(f), fmt);
    
    if (f == NULL) {
	eprintf(EF_IN3, INTERNAL, "No control connection for command",
		"command in FTP protocol");
	return NOTOK;
    }

    if (ProtoDebug) {
	_doprnt(fmt, &args, stderr);
	fprintf(stderr, "\r\n");
	fflush(stderr);
    }

    _doprnt(fmt, &args, f);
    if (fprintf(f, "\r\n") == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "fprintf", "command", getsyserr());
    }
    if (fflush(f) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "fflush", "command", getsyserr());
    }
    return OK;
} /* command */


/*  */

#include <ctype.h>

getreply(f, expecteof)
    FILE *f;
    int expecteof;
{
    register int c, n;
    register int dig;
    int originalcode = 0, continuation = 0;

    pprintf("getreply(%d, %d)\n", fileno(f), expecteof);

    for (;;) {
	dig = n = code = 0;
	while ((c = getc(f)) != '\n') {
	    dig++;
	    if (c == EOF) {
		if (expecteof) {
		    code = 221;
		    return (0);
		}
		pprintf("getreply: Lost connection.\n");
		(void) fflush(stdout);
		code = 421;
		return(4);
	    }
	    if (dig < 4 && isdigit(c))
		code = code * 10 + (c - '0');
	    if (dig == 4 && c == '-') {
		if (continuation)
		    code = 0;
		continuation++;
	    }
	    if (n == 0)
		n = c;
	}
	if (continuation && code != originalcode) {
	    if (originalcode == 0)
		originalcode = code;
	    continue;
	}
	if (code == 421 || originalcode == 421) {
	    pprintf("getreply: Lost connection.\n");
	    (void) fflush(stdout);
	}
	return (n - '0');
    }
} /* getreply */

/*  */

/* FTPD */

static struct sockaddr_in data_dest;

int server_init(ch, cmd, mode)
    channel_t *ch;
    char *cmd, *mode;
{
    int s;
    
    pprintf("server_init(%d:0x%x, %s, %s)\n",
	    fileno(ch->control_in), ch->data, cmd, mode);
    
    if (getcommand(ch->control_in, "PORT") == NOTOK) {
	pprintf(EF_IN3, INTERNAL, "getcommand(PORT)",
		"server_init");
	return NOTOK;
    }

    reply(ch->control_out, 200, "PORT command successful.");

    if (getcommand(ch->control_in, cmd) == NOTOK){ 
	pprintf(EF_IN3, INTERNAL, "getcommand()",
		"server_init");
	return NOTOK;
    }
    
    s = socket(AF_INET, SOCK_STREAM, 0);
    if (s == NOTOK) {
	eprintf(EF_SYSCALL, INTERNAL, "socket", "server_init", getsyserr());
	return NOTOK;
    }
    
    ch->data = fdopen(s, mode);
    
    if (ch->data == NULL) {
	reply(ch->control_out, 425, "Can't create data socket (%s,%d): %s.",
	      inet_ntoa(data_dest.sin_addr),
	      ntohs(data_dest.sin_port),
	      sys_errlist[errno]);
	(void)close(s);
	return NOTOK;
    }

    pprintf("server_init: s %d, ch->data 0x%x\n", s, ch->data);
    pprintf("\tdata_dest: %d:0x%x:%d\n", data_dest.sin_family,
		data_dest.sin_addr.s_addr, data_dest.sin_port);
    
    reply(ch->control_out, 150, "Opening data connection (%s,%d).",
	  inet_ntoa(data_dest.sin_addr),
	  ntohs(data_dest.sin_port));

    if (connect(s, &data_dest, sizeof (data_dest)) == NOTOK) {
	eprintf(EF_SYSCALL, INTERNAL, "connect", "server_init",
		getsyserr());
	eprintf("\tdata_dest: %d:0x%x:%d\n", data_dest.sin_family,
		data_dest.sin_addr.s_addr, data_dest.sin_port);
	reply(ch->control_out, 425, "Can't build data connection: %s.",
	      sys_errlist[errno]);
	(void) fclose(ch->data);
	ch->data = NULL;
	return NOTOK;
    }
    
    return OK;
} /* server_init */

/*  */

/*
 * Handle PORT, STOR, RETR commands. Returns Ok if the expected command
 * was found, else NOTOK. PORT parameters go into data_dest - other
 * parameters are ignored
 */

getcommand(f, command)
    FILE *f;
    char *command;
{
#define LINESIZE 128     
    char line[LINESIZE];
    int cpos;
    char c;

    pprintf("getcommand(%d, %s)\n", fileno(f), command);

    if (fgets(line, LINESIZE, f) == NULL) {
	eprintf(EF_IN3, INTERNAL, "fgets", "getcommand");
	return NOTOK;
    }
    
    pprintf("getcommand: %s\n", line);

    if (index(line, '\r')) {
	char *cp;
	cp = index(line, '\r');
	cp[0] = '\n';
	cp[1] = '\0';
    }

    if (index(line, ' '))
	cpos = index(line, ' ') - line;
    else
	cpos = index(line, '\n') - line;
    if (cpos == 0)
	cpos = 4;

    c = line[cpos];
    line[cpos] = '\0';
    upper(line);
    if (strcmp(line, command) != 0) {
	pprintf("ftp command match failed: %s, %s\n", line, command);
	return NOTOK;
    }
    if (strcmp(line, "PORT") == 0) {
	register char *a, *p;
	int nread;
	int a0,a1,a2,a3,p0,p1;
	
	line[cpos] = c;
	nread = sscanf(line+cpos, "%d,%d,%d,%d,%d,%d",
		       &a0, &a1, &a2, &a3, &p0, &p1);
	
	if (nread == EOF) {
	    eprintf(EF_IN3, INTERNAL, "sscanf", "getcomman - EOF");
	    pprintf("\tline = %s\n", line);
	    return NOTOK;
	}
	if (nread != 6) {
	    eprintf(EF_IN3, INTERNAL, "sscanf", "getcomman - too few");
	    pprintf("\tnread %d, line = %s\n", nread, line);
	    return NOTOK;
	}
	a = (char *)&data_dest.sin_addr;
	a[0] = a0; a[1] = a1; a[2] = a2; a[3] = a3;
	p = (char *)&data_dest.sin_port;
	p[0] = p0; p[1] = p1;

	data_dest.sin_family = AF_INET;
	pprintf("getcommand: line: %s\n", line+cpos);
	pprintf("\tdata_dest: %d:0x%x:%d\n", data_dest.sin_family,
		data_dest.sin_addr.s_addr, data_dest.sin_port);
	
    } else if (strcmp(line, "STOR") == 0) {
	line[cpos] = c;
	pprintf("storing to file %s\n", line+cpos);
    } else if (strcmp(line, "RETR") == 0){ 
	line[cpos] = c;
	pprintf("retrieveing from file %s\n", line+cpos);
    } else {
	line[cpos] = c;
	eprintf(EF_IN4, INTERNAL, PARAMETER, "getcommand",
		line);
	return NOTOK;
    }
    
    return OK;
} /* getcommand */

/*  */

/*VARARGS3*/
reply(f, n, s, args)
    FILE *f;
    int n;
    char *s;
{
    pprintf("reply(0x%x, %d, %s)\n", f, n, s);

    if (ProtoDebug) {
	fprintf(stderr, "%d ", n);
	_doprnt(s, &args, stderr);
	fprintf(stderr, "\r\n");
	fflush(stderr);
    }

    if (fprintf(f, "%d ", n) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "fprintf", "reply", getsyserr());
    }
    _doprnt(s, &args, f);
    if (fprintf(f, "\r\n") == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "fprintf", "reply", getsyserr());
    }
    if (fflush(f) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "fflush", "reply", getsyserr());
    }
} /* reply */

/*  */

static upper(str)
    char *str;
{
    while (*str != '\0') {
	if (islower(*str))
	    *str = toupper(*str);
	str++;
    }
}

